package Dependency::Resolver;
$Dependency::Resolver::VERSION = '0.03';
use utf8;
use Moose;

sub dep_resolv {
    my ($self, $node, $resolved, $seen) = @_;

    $resolved ||= [];
    $seen     ||= [];

    #print $node->name . "\n";
    push( @$seen, $node);

    for my $edge ( @{$node->edges} ) {
        # if edge is not in resolved
        if ( ! grep { $_ eq $edge} @$resolved ) {

            die sprintf "Circular reference detected: %s -> %s", $node->name, $edge->name
                if ( grep { $_ eq $edge} @$seen);

            $self->dep_resolv( $edge, $resolved, $seen );
        }
    }
    push( @$resolved, $node);
    return $resolved;
}




=head1 NAME

Dependency::Resolver - Dependency resolving algorithm

=head1 VERSION

version 0.03

based on http://www.electricmonk.nl/log/2008/08/07/dependency-resolving-algorithm/

=head1 SYNOPSIS

my $a = Dependency::Resolver::Node->new( name => 'a' );
my $b = Dependency::Resolver::Node->new( name => 'b' );
my $c = Dependency::Resolver::Node->new( name => 'c' );
my $d = Dependency::Resolver::Node->new( name => 'd' );
my $e = Dependency::Resolver::Node->new( name => 'e' );

$a->addEdge($b);    # a depends on b
$a->addEdge($d);    # a depends on d
$b->addEdge($c);    # b depends on c
$b->addEdge($e);    # b depends on e
$c->addEdge($d);    # c depends on d
$c->addEdge($e);    # c depends on e

my $resolver = Dependency::Resolver->new;

my $resolved = $resolver->dep_resolv($a);


=head1 SUBROUTINES/METHODS

=head2 dep_resolv($node)

returns an arrayref of nodes resolved

=head1 AUTHOR

Daniel Brosseau, C<< <dab at catapulse.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-dependency-resolver at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Dependency-Resolver>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Dependency::Resolver


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Dependency-Resolver>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Dependency-Resolver>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Dependency-Resolver>

=item * Search CPAN

L<http://search.cpan.org/dist/Dependency-Resolver/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2015 Daniel Brosseau.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Dependency::Resolver
