use strict;
use warnings;
use Test::More tests => 24;
use Markdown::Simple;

# Test basic functionality
{
    my $result = markdown_to_html("Hello world");
    is($result, "Hello world", "Plain text passes through");
}

# Test headers
{
    my $result = markdown_to_html("# Header 1");
    is($result, "<h1>Header 1</h1>\n", "H1 header");
    
    $result = markdown_to_html("## Header 2");
    is($result, "<h2>Header 2</h2>\n", "H2 header");
    
    $result = markdown_to_html("### Header 3");
    is($result, "<h3>Header 3</h3>\n", "H3 header");
}

# Test bold
{
    my $result = markdown_to_html("**bold text**");
    is($result, "<strong>bold text</strong>", "Bold text");
}

# Test italic
{
    my $result = markdown_to_html("*italic text*");
    is($result, "<em>italic text</em>", "Italic text");
}

# Test inline code
{
    my $result = markdown_to_html("`code`");
    is($result, "<code>code</code>", "Inline code");
}

# Test fenced code blocks
{
    my $result = markdown_to_html("```\ncode block\n```");
    is($result, "<pre><code>code block\n</code></pre>\n", "Fenced code block");
    
    $result = markdown_to_html("```javascript\nvar x = 1;\n```");
    is($result, "<pre><code class=\"language-javascript\">var x = 1;\n</code></pre>\n", "Fenced code block with language");
}

# Test links
{
    my $result = markdown_to_html("[link text](http://example.com)");
    is($result, "<a href=\"http://example.com\">link text</a>", "Link");
}

# Test images
{
    my $result = markdown_to_html("![alt text](image.jpg)");
    is($result, "<img alt=\"alt text\" src=\"image.jpg\" />", "Image");
}

# Test strikethrough
{
    my $result = markdown_to_html("~~strikethrough~~");
    is($result, "<del>strikethrough</del>", "Strikethrough");
}

# Test task lists
{
    my $result = markdown_to_html("- [ ] unchecked task");
    is($result, "<li><input type=\"checkbox\" disabled /> unchecked task</li>\n", "Unchecked task");
    
    $result = markdown_to_html("- [x] checked task");
    is($result, "<li><input type=\"checkbox\" checked disabled /> checked task</li>\n", "Checked task");
}

# Test unordered lists
{
    my $result = markdown_to_html("- list item");
    is($result, "<ul><li>list item</li></ul>\n", "Unordered list with dash");
    
    $result = markdown_to_html("* list item");
    is($result, "<ul><li>list item</li></ul>\n", "Unordered list with asterisk");
}

# Test ordered lists
{
    my $result = markdown_to_html("1. first item");
    is($result, "<ol><li>first item</li></ol>\n", "Ordered list");
}

# Test tables
{
    my $markdown = "| Header 1 | Header 2 |\n|----------|----------|\n| Cell 1   | Cell 2   |";
    my $result = markdown_to_html($markdown);
    like($result, qr/<table>/, "Table contains table tag");
    like($result, qr/<th>Header 1<\/th>/, "Table header 1");
    like($result, qr/<td>Cell 1<\/td>/, "Table cell 1");
}

# Test options - disable features
{
    my $result = markdown_to_html("**bold**", { bold => 0, italic => 0, unordered_lists => 0 });
    is($result, "**bold**", "Bold disabled");
    
    $result = markdown_to_html("*italic*", { italic => 0, unordered_lists => 0 });
    is($result, "*italic*", "Italic disabled");
    
    $result = markdown_to_html("# Header", { headers => 0 });
    is($result, "# Header", "Headers disabled");
}

# Test complex combinations
{
    my $result = markdown_to_html("**bold** and *italic* with `code`");
    is($result, "<strong>bold</strong> and <em>italic</em> with <code>code</code>", "Multiple formatting");
}

done_testing();
