; Events to include at the start of the stuff generated by a transformation
;
; Copyright (C) 2014-2021 Kestrel Institute
;
; License: A 3-clause BSD license. See the file books/3BSD-mod.txt.
;
; Author: Eric Smith (eric.smith@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ACL2")

(include-book "misc/install-not-normalized" :dir :system)

;; Returns a (possibly-empty) list of events, setting the default-defun-mode to
;; :logic if needed.  Note that the effect is local to the enclosing book or
;; encapsulate.
(defun maybe-switch-to-logic-mode (wrld)
  (declare (xargs :guard (and (plist-worldp wrld)
                              (alistp (table-alist 'acl2-defaults-table wrld)))))
  (if (eq (default-defun-mode wrld) :logic)
      nil ; already in logic mode, so do nothing
    `((logic))))

;; Returns a (possibly-empty) list of events, clearing any default-hints.  Note
;; that the effect is local to the enclosing book or encapsulate.
(defun maybe-clear-default-hints (wrld)
  (declare (xargs :guard (and (plist-worldp wrld)
                              (alistp (table-alist 'default-hints-table wrld)))))
  (if (default-hints wrld)
      `((set-default-hints nil)) ;this macro-expands to a call of local
    ;; No default-hints, so do nothing:
    nil))

;; Returns a (possibly-empty) list of events, clearing any override-hints.
;; Note that the effect is local to the enclosing book or encapsulate.
(defun maybe-clear-override-hints (wrld)
  (declare (xargs :guard (and (plist-worldp wrld)
                              (alistp (table-alist 'default-hints-table wrld)))))
  (if (override-hints wrld)
      `((set-override-hints nil)) ;this macro-expands to a call of local
    ;; No override-hints, so do nothing:
    nil))

;; Returns a (possibly-empty) list of events that ensure that the
;; verify-guards-eagerness is EAGERNESS.  Note that the effect is local to the
;; enclosing book or encapsulate.
(defun maybe-set-verify-guards-eagerness (eagerness wrld)
  (declare (xargs :guard (and (member eagerness '(0 1 2 3))
                              (plist-worldp wrld)
                              (alistp (table-alist 'acl2-defaults-table wrld)))))
  (if (not (equal eagerness (default-verify-guards-eagerness wrld)))
      `((set-verify-guards-eagerness ,eagerness))
    ;; verify-guards-eagerness is already as desired, so do nothing:
    nil))

;; Returns a (possibly-empty) list of events that ensure that the "double
;; whammy property" (aka, dwp) is nil.
(defun maybe-clear-dwp (wrld)
  (declare (xargs :guard (and (plist-worldp wrld)
                              (alistp (table-alist 'dwp-table wrld)))))
  (if (get-dwp nil wrld)
      `((set-dwp nil))
    ;; dwp is already nil, so do nothing:
    nil))

;; Returns a list of standard events for transforming FN.  These go at the
;; beginning of the encapsulate.  Everything in this is local to the
;; surrounding encapsulate.
(defun transformation-prologue (fn wrld)
  (declare (xargs :guard (and (symbolp fn)
                              (plist-worldp wrld)
                              (alistp (table-alist 'acl2-defaults-table wrld))
                              (alistp (table-alist 'default-hints-table wrld))
                              (alistp (table-alist 'dwp-table wrld)))))
  (append (maybe-switch-to-logic-mode wrld)
          (maybe-set-verify-guards-eagerness 1 wrld)
          ;; todo: consider install-not-normalized-event:
          `((local (install-not-normalized ,fn))) ;todo: omit if not necessary (because the normalized and unnormalized bodies are the same)?
          ;; TODO: We may not want to do this until after applicability conditions (if any) are proved:
          (maybe-clear-default-hints wrld)
          (maybe-clear-override-hints wrld)
          (maybe-clear-dwp wrld)
          ;; todo: what else?
          ))
