// Copyright (c) Dropbox, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

// Package team_log : has no documentation (yet)
package team_log

import (
	"encoding/json"
	"time"

	"github.com/dropbox/dropbox-sdk-go-unofficial/v6/dropbox"
	"github.com/dropbox/dropbox-sdk-go-unofficial/v6/dropbox/files"
	"github.com/dropbox/dropbox-sdk-go-unofficial/v6/dropbox/sharing"
	"github.com/dropbox/dropbox-sdk-go-unofficial/v6/dropbox/team"
	"github.com/dropbox/dropbox-sdk-go-unofficial/v6/dropbox/team_common"
	"github.com/dropbox/dropbox-sdk-go-unofficial/v6/dropbox/team_policies"
)

// AccessMethodLogInfo : Indicates the method in which the action was performed.
type AccessMethodLogInfo struct {
	dropbox.Tagged
	// AdminConsole : Admin console session details.
	AdminConsole *WebSessionLogInfo `json:"admin_console,omitempty"`
	// Api : Api session details.
	Api *ApiSessionLogInfo `json:"api,omitempty"`
	// ContentManager : Content manager session details.
	ContentManager *WebSessionLogInfo `json:"content_manager,omitempty"`
	// EndUser : End user session details.
	EndUser IsSessionLogInfo `json:"end_user,omitempty"`
	// EnterpriseConsole : Enterprise console session details.
	EnterpriseConsole *WebSessionLogInfo `json:"enterprise_console,omitempty"`
	// SignInAs : Sign in as session details.
	SignInAs *WebSessionLogInfo `json:"sign_in_as,omitempty"`
}

// Valid tag values for AccessMethodLogInfo
const (
	AccessMethodLogInfoAdminConsole      = "admin_console"
	AccessMethodLogInfoApi               = "api"
	AccessMethodLogInfoContentManager    = "content_manager"
	AccessMethodLogInfoEndUser           = "end_user"
	AccessMethodLogInfoEnterpriseConsole = "enterprise_console"
	AccessMethodLogInfoSignInAs          = "sign_in_as"
	AccessMethodLogInfoOther             = "other"
)

// UnmarshalJSON deserializes into a AccessMethodLogInfo instance
func (u *AccessMethodLogInfo) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
		// EndUser : End user session details.
		EndUser json.RawMessage `json:"end_user,omitempty"`
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "admin_console":
		if err = json.Unmarshal(body, &u.AdminConsole); err != nil {
			return err
		}

	case "api":
		if err = json.Unmarshal(body, &u.Api); err != nil {
			return err
		}

	case "content_manager":
		if err = json.Unmarshal(body, &u.ContentManager); err != nil {
			return err
		}

	case "end_user":
		if u.EndUser, err = IsSessionLogInfoFromJSON(w.EndUser); err != nil {
			return err
		}

	case "enterprise_console":
		if err = json.Unmarshal(body, &u.EnterpriseConsole); err != nil {
			return err
		}

	case "sign_in_as":
		if err = json.Unmarshal(body, &u.SignInAs); err != nil {
			return err
		}

	}
	return nil
}

// AccountCaptureAvailability : has no documentation (yet)
type AccountCaptureAvailability struct {
	dropbox.Tagged
}

// Valid tag values for AccountCaptureAvailability
const (
	AccountCaptureAvailabilityAvailable   = "available"
	AccountCaptureAvailabilityUnavailable = "unavailable"
	AccountCaptureAvailabilityOther       = "other"
)

// AccountCaptureChangeAvailabilityDetails : Granted/revoked option to enable
// account capture on team domains.
type AccountCaptureChangeAvailabilityDetails struct {
	// NewValue : New account capture availabilty value.
	NewValue *AccountCaptureAvailability `json:"new_value"`
	// PreviousValue : Previous account capture availabilty value. Might be
	// missing due to historical data gap.
	PreviousValue *AccountCaptureAvailability `json:"previous_value,omitempty"`
}

// NewAccountCaptureChangeAvailabilityDetails returns a new AccountCaptureChangeAvailabilityDetails instance
func NewAccountCaptureChangeAvailabilityDetails(NewValue *AccountCaptureAvailability) *AccountCaptureChangeAvailabilityDetails {
	s := new(AccountCaptureChangeAvailabilityDetails)
	s.NewValue = NewValue
	return s
}

// AccountCaptureChangeAvailabilityType : has no documentation (yet)
type AccountCaptureChangeAvailabilityType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAccountCaptureChangeAvailabilityType returns a new AccountCaptureChangeAvailabilityType instance
func NewAccountCaptureChangeAvailabilityType(Description string) *AccountCaptureChangeAvailabilityType {
	s := new(AccountCaptureChangeAvailabilityType)
	s.Description = Description
	return s
}

// AccountCaptureChangePolicyDetails : Changed account capture setting on team
// domain.
type AccountCaptureChangePolicyDetails struct {
	// NewValue : New account capture policy.
	NewValue *AccountCapturePolicy `json:"new_value"`
	// PreviousValue : Previous account capture policy. Might be missing due to
	// historical data gap.
	PreviousValue *AccountCapturePolicy `json:"previous_value,omitempty"`
}

// NewAccountCaptureChangePolicyDetails returns a new AccountCaptureChangePolicyDetails instance
func NewAccountCaptureChangePolicyDetails(NewValue *AccountCapturePolicy) *AccountCaptureChangePolicyDetails {
	s := new(AccountCaptureChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// AccountCaptureChangePolicyType : has no documentation (yet)
type AccountCaptureChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAccountCaptureChangePolicyType returns a new AccountCaptureChangePolicyType instance
func NewAccountCaptureChangePolicyType(Description string) *AccountCaptureChangePolicyType {
	s := new(AccountCaptureChangePolicyType)
	s.Description = Description
	return s
}

// AccountCaptureMigrateAccountDetails : Account-captured user migrated account
// to team.
type AccountCaptureMigrateAccountDetails struct {
	// DomainName : Domain name.
	DomainName string `json:"domain_name"`
}

// NewAccountCaptureMigrateAccountDetails returns a new AccountCaptureMigrateAccountDetails instance
func NewAccountCaptureMigrateAccountDetails(DomainName string) *AccountCaptureMigrateAccountDetails {
	s := new(AccountCaptureMigrateAccountDetails)
	s.DomainName = DomainName
	return s
}

// AccountCaptureMigrateAccountType : has no documentation (yet)
type AccountCaptureMigrateAccountType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAccountCaptureMigrateAccountType returns a new AccountCaptureMigrateAccountType instance
func NewAccountCaptureMigrateAccountType(Description string) *AccountCaptureMigrateAccountType {
	s := new(AccountCaptureMigrateAccountType)
	s.Description = Description
	return s
}

// AccountCaptureNotificationEmailsSentDetails : Sent account capture email to
// all unmanaged members.
type AccountCaptureNotificationEmailsSentDetails struct {
	// DomainName : Domain name.
	DomainName string `json:"domain_name"`
	// NotificationType : Account-capture email notification type.
	NotificationType *AccountCaptureNotificationType `json:"notification_type,omitempty"`
}

// NewAccountCaptureNotificationEmailsSentDetails returns a new AccountCaptureNotificationEmailsSentDetails instance
func NewAccountCaptureNotificationEmailsSentDetails(DomainName string) *AccountCaptureNotificationEmailsSentDetails {
	s := new(AccountCaptureNotificationEmailsSentDetails)
	s.DomainName = DomainName
	return s
}

// AccountCaptureNotificationEmailsSentType : has no documentation (yet)
type AccountCaptureNotificationEmailsSentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAccountCaptureNotificationEmailsSentType returns a new AccountCaptureNotificationEmailsSentType instance
func NewAccountCaptureNotificationEmailsSentType(Description string) *AccountCaptureNotificationEmailsSentType {
	s := new(AccountCaptureNotificationEmailsSentType)
	s.Description = Description
	return s
}

// AccountCaptureNotificationType : has no documentation (yet)
type AccountCaptureNotificationType struct {
	dropbox.Tagged
}

// Valid tag values for AccountCaptureNotificationType
const (
	AccountCaptureNotificationTypeActionableNotification       = "actionable_notification"
	AccountCaptureNotificationTypeProactiveWarningNotification = "proactive_warning_notification"
	AccountCaptureNotificationTypeOther                        = "other"
)

// AccountCapturePolicy : has no documentation (yet)
type AccountCapturePolicy struct {
	dropbox.Tagged
}

// Valid tag values for AccountCapturePolicy
const (
	AccountCapturePolicyAllUsers     = "all_users"
	AccountCapturePolicyDisabled     = "disabled"
	AccountCapturePolicyInvitedUsers = "invited_users"
	AccountCapturePolicyOther        = "other"
)

// AccountCaptureRelinquishAccountDetails : Account-captured user changed
// account email to personal email.
type AccountCaptureRelinquishAccountDetails struct {
	// DomainName : Domain name.
	DomainName string `json:"domain_name"`
}

// NewAccountCaptureRelinquishAccountDetails returns a new AccountCaptureRelinquishAccountDetails instance
func NewAccountCaptureRelinquishAccountDetails(DomainName string) *AccountCaptureRelinquishAccountDetails {
	s := new(AccountCaptureRelinquishAccountDetails)
	s.DomainName = DomainName
	return s
}

// AccountCaptureRelinquishAccountType : has no documentation (yet)
type AccountCaptureRelinquishAccountType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAccountCaptureRelinquishAccountType returns a new AccountCaptureRelinquishAccountType instance
func NewAccountCaptureRelinquishAccountType(Description string) *AccountCaptureRelinquishAccountType {
	s := new(AccountCaptureRelinquishAccountType)
	s.Description = Description
	return s
}

// AccountLockOrUnlockedDetails : Unlocked/locked account after failed sign in
// attempts.
type AccountLockOrUnlockedDetails struct {
	// PreviousValue : The previous account status.
	PreviousValue *AccountState `json:"previous_value"`
	// NewValue : The new account status.
	NewValue *AccountState `json:"new_value"`
}

// NewAccountLockOrUnlockedDetails returns a new AccountLockOrUnlockedDetails instance
func NewAccountLockOrUnlockedDetails(PreviousValue *AccountState, NewValue *AccountState) *AccountLockOrUnlockedDetails {
	s := new(AccountLockOrUnlockedDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// AccountLockOrUnlockedType : has no documentation (yet)
type AccountLockOrUnlockedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAccountLockOrUnlockedType returns a new AccountLockOrUnlockedType instance
func NewAccountLockOrUnlockedType(Description string) *AccountLockOrUnlockedType {
	s := new(AccountLockOrUnlockedType)
	s.Description = Description
	return s
}

// AccountState : has no documentation (yet)
type AccountState struct {
	dropbox.Tagged
}

// Valid tag values for AccountState
const (
	AccountStateLocked   = "locked"
	AccountStateUnlocked = "unlocked"
	AccountStateOther    = "other"
)

// ActionDetails : Additional information indicating the action taken that
// caused status change.
type ActionDetails struct {
	dropbox.Tagged
	// RemoveAction : Define how the user was removed from the team.
	RemoveAction *MemberRemoveActionType `json:"remove_action,omitempty"`
	// TeamInviteDetails : Additional information relevant when someone is
	// invited to the team.
	TeamInviteDetails *TeamInviteDetails `json:"team_invite_details,omitempty"`
	// TeamJoinDetails : Additional information relevant when a new member joins
	// the team.
	TeamJoinDetails *JoinTeamDetails `json:"team_join_details,omitempty"`
}

// Valid tag values for ActionDetails
const (
	ActionDetailsRemoveAction      = "remove_action"
	ActionDetailsTeamInviteDetails = "team_invite_details"
	ActionDetailsTeamJoinDetails   = "team_join_details"
	ActionDetailsOther             = "other"
)

// UnmarshalJSON deserializes into a ActionDetails instance
func (u *ActionDetails) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
		// RemoveAction : Define how the user was removed from the team.
		RemoveAction *MemberRemoveActionType `json:"remove_action,omitempty"`
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "remove_action":
		u.RemoveAction = w.RemoveAction

	case "team_invite_details":
		if err = json.Unmarshal(body, &u.TeamInviteDetails); err != nil {
			return err
		}

	case "team_join_details":
		if err = json.Unmarshal(body, &u.TeamJoinDetails); err != nil {
			return err
		}

	}
	return nil
}

// ActorLogInfo : The entity who performed the action.
type ActorLogInfo struct {
	dropbox.Tagged
	// Admin : The admin who did the action.
	Admin IsUserLogInfo `json:"admin,omitempty"`
	// App : The application who did the action.
	App IsAppLogInfo `json:"app,omitempty"`
	// Reseller : Action done by reseller.
	Reseller *ResellerLogInfo `json:"reseller,omitempty"`
	// User : The user who did the action.
	User IsUserLogInfo `json:"user,omitempty"`
}

// Valid tag values for ActorLogInfo
const (
	ActorLogInfoAdmin     = "admin"
	ActorLogInfoAnonymous = "anonymous"
	ActorLogInfoApp       = "app"
	ActorLogInfoDropbox   = "dropbox"
	ActorLogInfoReseller  = "reseller"
	ActorLogInfoUser      = "user"
	ActorLogInfoOther     = "other"
)

// UnmarshalJSON deserializes into a ActorLogInfo instance
func (u *ActorLogInfo) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
		// Admin : The admin who did the action.
		Admin json.RawMessage `json:"admin,omitempty"`
		// App : The application who did the action.
		App json.RawMessage `json:"app,omitempty"`
		// User : The user who did the action.
		User json.RawMessage `json:"user,omitempty"`
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "admin":
		if u.Admin, err = IsUserLogInfoFromJSON(w.Admin); err != nil {
			return err
		}

	case "app":
		if u.App, err = IsAppLogInfoFromJSON(w.App); err != nil {
			return err
		}

	case "reseller":
		if err = json.Unmarshal(body, &u.Reseller); err != nil {
			return err
		}

	case "user":
		if u.User, err = IsUserLogInfoFromJSON(w.User); err != nil {
			return err
		}

	}
	return nil
}

// AdminAlertCategoryEnum : Alert category
type AdminAlertCategoryEnum struct {
	dropbox.Tagged
}

// Valid tag values for AdminAlertCategoryEnum
const (
	AdminAlertCategoryEnumAccountTakeover       = "account_takeover"
	AdminAlertCategoryEnumDataLossProtection    = "data_loss_protection"
	AdminAlertCategoryEnumInformationGovernance = "information_governance"
	AdminAlertCategoryEnumMalwareSharing        = "malware_sharing"
	AdminAlertCategoryEnumMassiveFileOperation  = "massive_file_operation"
	AdminAlertCategoryEnumNa                    = "na"
	AdminAlertCategoryEnumThreatManagement      = "threat_management"
	AdminAlertCategoryEnumOther                 = "other"
)

// AdminAlertGeneralStateEnum : Alert state
type AdminAlertGeneralStateEnum struct {
	dropbox.Tagged
}

// Valid tag values for AdminAlertGeneralStateEnum
const (
	AdminAlertGeneralStateEnumActive     = "active"
	AdminAlertGeneralStateEnumDismissed  = "dismissed"
	AdminAlertGeneralStateEnumInProgress = "in_progress"
	AdminAlertGeneralStateEnumNa         = "na"
	AdminAlertGeneralStateEnumResolved   = "resolved"
	AdminAlertGeneralStateEnumOther      = "other"
)

// AdminAlertSeverityEnum : Alert severity
type AdminAlertSeverityEnum struct {
	dropbox.Tagged
}

// Valid tag values for AdminAlertSeverityEnum
const (
	AdminAlertSeverityEnumHigh   = "high"
	AdminAlertSeverityEnumInfo   = "info"
	AdminAlertSeverityEnumLow    = "low"
	AdminAlertSeverityEnumMedium = "medium"
	AdminAlertSeverityEnumNa     = "na"
	AdminAlertSeverityEnumOther  = "other"
)

// AdminAlertingAlertConfiguration : Alert configurations
type AdminAlertingAlertConfiguration struct {
	// AlertState : Alert state.
	AlertState *AdminAlertingAlertStatePolicy `json:"alert_state,omitempty"`
	// SensitivityLevel : Sensitivity level.
	SensitivityLevel *AdminAlertingAlertSensitivity `json:"sensitivity_level,omitempty"`
	// RecipientsSettings : Recipient settings.
	RecipientsSettings *RecipientsConfiguration `json:"recipients_settings,omitempty"`
	// Text : Text.
	Text string `json:"text,omitempty"`
	// ExcludedFileExtensions : Excluded file extensions.
	ExcludedFileExtensions string `json:"excluded_file_extensions,omitempty"`
}

// NewAdminAlertingAlertConfiguration returns a new AdminAlertingAlertConfiguration instance
func NewAdminAlertingAlertConfiguration() *AdminAlertingAlertConfiguration {
	s := new(AdminAlertingAlertConfiguration)
	return s
}

// AdminAlertingAlertSensitivity : Alert sensitivity
type AdminAlertingAlertSensitivity struct {
	dropbox.Tagged
}

// Valid tag values for AdminAlertingAlertSensitivity
const (
	AdminAlertingAlertSensitivityHigh    = "high"
	AdminAlertingAlertSensitivityHighest = "highest"
	AdminAlertingAlertSensitivityInvalid = "invalid"
	AdminAlertingAlertSensitivityLow     = "low"
	AdminAlertingAlertSensitivityLowest  = "lowest"
	AdminAlertingAlertSensitivityMedium  = "medium"
	AdminAlertingAlertSensitivityOther   = "other"
)

// AdminAlertingAlertStateChangedDetails : Changed an alert state.
type AdminAlertingAlertStateChangedDetails struct {
	// AlertName : Alert name.
	AlertName string `json:"alert_name"`
	// AlertSeverity : Alert severity.
	AlertSeverity *AdminAlertSeverityEnum `json:"alert_severity"`
	// AlertCategory : Alert category.
	AlertCategory *AdminAlertCategoryEnum `json:"alert_category"`
	// AlertInstanceId : Alert ID.
	AlertInstanceId string `json:"alert_instance_id"`
	// PreviousValue : Alert state before the change.
	PreviousValue *AdminAlertGeneralStateEnum `json:"previous_value"`
	// NewValue : Alert state after the change.
	NewValue *AdminAlertGeneralStateEnum `json:"new_value"`
}

// NewAdminAlertingAlertStateChangedDetails returns a new AdminAlertingAlertStateChangedDetails instance
func NewAdminAlertingAlertStateChangedDetails(AlertName string, AlertSeverity *AdminAlertSeverityEnum, AlertCategory *AdminAlertCategoryEnum, AlertInstanceId string, PreviousValue *AdminAlertGeneralStateEnum, NewValue *AdminAlertGeneralStateEnum) *AdminAlertingAlertStateChangedDetails {
	s := new(AdminAlertingAlertStateChangedDetails)
	s.AlertName = AlertName
	s.AlertSeverity = AlertSeverity
	s.AlertCategory = AlertCategory
	s.AlertInstanceId = AlertInstanceId
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// AdminAlertingAlertStateChangedType : has no documentation (yet)
type AdminAlertingAlertStateChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAdminAlertingAlertStateChangedType returns a new AdminAlertingAlertStateChangedType instance
func NewAdminAlertingAlertStateChangedType(Description string) *AdminAlertingAlertStateChangedType {
	s := new(AdminAlertingAlertStateChangedType)
	s.Description = Description
	return s
}

// AdminAlertingAlertStatePolicy : Policy for controlling whether an alert can
// be triggered or not
type AdminAlertingAlertStatePolicy struct {
	dropbox.Tagged
}

// Valid tag values for AdminAlertingAlertStatePolicy
const (
	AdminAlertingAlertStatePolicyOff   = "off"
	AdminAlertingAlertStatePolicyOn    = "on"
	AdminAlertingAlertStatePolicyOther = "other"
)

// AdminAlertingChangedAlertConfigDetails : Changed an alert setting.
type AdminAlertingChangedAlertConfigDetails struct {
	// AlertName : Alert Name.
	AlertName string `json:"alert_name"`
	// PreviousAlertConfig : Previous alert configuration.
	PreviousAlertConfig *AdminAlertingAlertConfiguration `json:"previous_alert_config"`
	// NewAlertConfig : New alert configuration.
	NewAlertConfig *AdminAlertingAlertConfiguration `json:"new_alert_config"`
}

// NewAdminAlertingChangedAlertConfigDetails returns a new AdminAlertingChangedAlertConfigDetails instance
func NewAdminAlertingChangedAlertConfigDetails(AlertName string, PreviousAlertConfig *AdminAlertingAlertConfiguration, NewAlertConfig *AdminAlertingAlertConfiguration) *AdminAlertingChangedAlertConfigDetails {
	s := new(AdminAlertingChangedAlertConfigDetails)
	s.AlertName = AlertName
	s.PreviousAlertConfig = PreviousAlertConfig
	s.NewAlertConfig = NewAlertConfig
	return s
}

// AdminAlertingChangedAlertConfigType : has no documentation (yet)
type AdminAlertingChangedAlertConfigType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAdminAlertingChangedAlertConfigType returns a new AdminAlertingChangedAlertConfigType instance
func NewAdminAlertingChangedAlertConfigType(Description string) *AdminAlertingChangedAlertConfigType {
	s := new(AdminAlertingChangedAlertConfigType)
	s.Description = Description
	return s
}

// AdminAlertingTriggeredAlertDetails : Triggered security alert.
type AdminAlertingTriggeredAlertDetails struct {
	// AlertName : Alert name.
	AlertName string `json:"alert_name"`
	// AlertSeverity : Alert severity.
	AlertSeverity *AdminAlertSeverityEnum `json:"alert_severity"`
	// AlertCategory : Alert category.
	AlertCategory *AdminAlertCategoryEnum `json:"alert_category"`
	// AlertInstanceId : Alert ID.
	AlertInstanceId string `json:"alert_instance_id"`
}

// NewAdminAlertingTriggeredAlertDetails returns a new AdminAlertingTriggeredAlertDetails instance
func NewAdminAlertingTriggeredAlertDetails(AlertName string, AlertSeverity *AdminAlertSeverityEnum, AlertCategory *AdminAlertCategoryEnum, AlertInstanceId string) *AdminAlertingTriggeredAlertDetails {
	s := new(AdminAlertingTriggeredAlertDetails)
	s.AlertName = AlertName
	s.AlertSeverity = AlertSeverity
	s.AlertCategory = AlertCategory
	s.AlertInstanceId = AlertInstanceId
	return s
}

// AdminAlertingTriggeredAlertType : has no documentation (yet)
type AdminAlertingTriggeredAlertType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAdminAlertingTriggeredAlertType returns a new AdminAlertingTriggeredAlertType instance
func NewAdminAlertingTriggeredAlertType(Description string) *AdminAlertingTriggeredAlertType {
	s := new(AdminAlertingTriggeredAlertType)
	s.Description = Description
	return s
}

// AdminConsoleAppPermission : has no documentation (yet)
type AdminConsoleAppPermission struct {
	dropbox.Tagged
}

// Valid tag values for AdminConsoleAppPermission
const (
	AdminConsoleAppPermissionDefaultForListedApps   = "default_for_listed_apps"
	AdminConsoleAppPermissionDefaultForUnlistedApps = "default_for_unlisted_apps"
	AdminConsoleAppPermissionOther                  = "other"
)

// AdminConsoleAppPolicy : has no documentation (yet)
type AdminConsoleAppPolicy struct {
	dropbox.Tagged
}

// Valid tag values for AdminConsoleAppPolicy
const (
	AdminConsoleAppPolicyAllow   = "allow"
	AdminConsoleAppPolicyBlock   = "block"
	AdminConsoleAppPolicyDefault = "default"
	AdminConsoleAppPolicyOther   = "other"
)

// AdminEmailRemindersChangedDetails : Changed admin reminder settings for
// requests to join the team.
type AdminEmailRemindersChangedDetails struct {
	// NewValue : To.
	NewValue *AdminEmailRemindersPolicy `json:"new_value"`
	// PreviousValue : From.
	PreviousValue *AdminEmailRemindersPolicy `json:"previous_value"`
}

// NewAdminEmailRemindersChangedDetails returns a new AdminEmailRemindersChangedDetails instance
func NewAdminEmailRemindersChangedDetails(NewValue *AdminEmailRemindersPolicy, PreviousValue *AdminEmailRemindersPolicy) *AdminEmailRemindersChangedDetails {
	s := new(AdminEmailRemindersChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// AdminEmailRemindersChangedType : has no documentation (yet)
type AdminEmailRemindersChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAdminEmailRemindersChangedType returns a new AdminEmailRemindersChangedType instance
func NewAdminEmailRemindersChangedType(Description string) *AdminEmailRemindersChangedType {
	s := new(AdminEmailRemindersChangedType)
	s.Description = Description
	return s
}

// AdminEmailRemindersPolicy : Policy for deciding whether team admins receive
// reminder emails for requests to join the team
type AdminEmailRemindersPolicy struct {
	dropbox.Tagged
}

// Valid tag values for AdminEmailRemindersPolicy
const (
	AdminEmailRemindersPolicyDefault  = "default"
	AdminEmailRemindersPolicyDisabled = "disabled"
	AdminEmailRemindersPolicyEnabled  = "enabled"
	AdminEmailRemindersPolicyOther    = "other"
)

// AdminRole : has no documentation (yet)
type AdminRole struct {
	dropbox.Tagged
}

// Valid tag values for AdminRole
const (
	AdminRoleBillingAdmin        = "billing_admin"
	AdminRoleComplianceAdmin     = "compliance_admin"
	AdminRoleContentAdmin        = "content_admin"
	AdminRoleLimitedAdmin        = "limited_admin"
	AdminRoleMemberOnly          = "member_only"
	AdminRoleReportingAdmin      = "reporting_admin"
	AdminRoleSecurityAdmin       = "security_admin"
	AdminRoleSupportAdmin        = "support_admin"
	AdminRoleTeamAdmin           = "team_admin"
	AdminRoleUserManagementAdmin = "user_management_admin"
	AdminRoleOther               = "other"
)

// AlertRecipientsSettingType : Alert recipients setting type
type AlertRecipientsSettingType struct {
	dropbox.Tagged
}

// Valid tag values for AlertRecipientsSettingType
const (
	AlertRecipientsSettingTypeCustomList = "custom_list"
	AlertRecipientsSettingTypeInvalid    = "invalid"
	AlertRecipientsSettingTypeNone       = "none"
	AlertRecipientsSettingTypeTeamAdmins = "team_admins"
	AlertRecipientsSettingTypeOther      = "other"
)

// AllowDownloadDisabledDetails : Disabled downloads.
type AllowDownloadDisabledDetails struct {
}

// NewAllowDownloadDisabledDetails returns a new AllowDownloadDisabledDetails instance
func NewAllowDownloadDisabledDetails() *AllowDownloadDisabledDetails {
	s := new(AllowDownloadDisabledDetails)
	return s
}

// AllowDownloadDisabledType : has no documentation (yet)
type AllowDownloadDisabledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAllowDownloadDisabledType returns a new AllowDownloadDisabledType instance
func NewAllowDownloadDisabledType(Description string) *AllowDownloadDisabledType {
	s := new(AllowDownloadDisabledType)
	s.Description = Description
	return s
}

// AllowDownloadEnabledDetails : Enabled downloads.
type AllowDownloadEnabledDetails struct {
}

// NewAllowDownloadEnabledDetails returns a new AllowDownloadEnabledDetails instance
func NewAllowDownloadEnabledDetails() *AllowDownloadEnabledDetails {
	s := new(AllowDownloadEnabledDetails)
	return s
}

// AllowDownloadEnabledType : has no documentation (yet)
type AllowDownloadEnabledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAllowDownloadEnabledType returns a new AllowDownloadEnabledType instance
func NewAllowDownloadEnabledType(Description string) *AllowDownloadEnabledType {
	s := new(AllowDownloadEnabledType)
	s.Description = Description
	return s
}

// ApiSessionLogInfo : Api session.
type ApiSessionLogInfo struct {
	// RequestId : Api request ID.
	RequestId string `json:"request_id"`
}

// NewApiSessionLogInfo returns a new ApiSessionLogInfo instance
func NewApiSessionLogInfo(RequestId string) *ApiSessionLogInfo {
	s := new(ApiSessionLogInfo)
	s.RequestId = RequestId
	return s
}

// AppBlockedByPermissionsDetails : Failed to connect app for member.
type AppBlockedByPermissionsDetails struct {
	// AppInfo : Relevant application details.
	AppInfo IsAppLogInfo `json:"app_info"`
}

// NewAppBlockedByPermissionsDetails returns a new AppBlockedByPermissionsDetails instance
func NewAppBlockedByPermissionsDetails(AppInfo IsAppLogInfo) *AppBlockedByPermissionsDetails {
	s := new(AppBlockedByPermissionsDetails)
	s.AppInfo = AppInfo
	return s
}

// UnmarshalJSON deserializes into a AppBlockedByPermissionsDetails instance
func (u *AppBlockedByPermissionsDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// AppInfo : Relevant application details.
		AppInfo json.RawMessage `json:"app_info"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	AppInfo, err := IsAppLogInfoFromJSON(w.AppInfo)
	if err != nil {
		return err
	}
	u.AppInfo = AppInfo
	return nil
}

// AppBlockedByPermissionsType : has no documentation (yet)
type AppBlockedByPermissionsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAppBlockedByPermissionsType returns a new AppBlockedByPermissionsType instance
func NewAppBlockedByPermissionsType(Description string) *AppBlockedByPermissionsType {
	s := new(AppBlockedByPermissionsType)
	s.Description = Description
	return s
}

// AppLinkTeamDetails : Linked app for team.
type AppLinkTeamDetails struct {
	// AppInfo : Relevant application details.
	AppInfo IsAppLogInfo `json:"app_info"`
}

// NewAppLinkTeamDetails returns a new AppLinkTeamDetails instance
func NewAppLinkTeamDetails(AppInfo IsAppLogInfo) *AppLinkTeamDetails {
	s := new(AppLinkTeamDetails)
	s.AppInfo = AppInfo
	return s
}

// UnmarshalJSON deserializes into a AppLinkTeamDetails instance
func (u *AppLinkTeamDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// AppInfo : Relevant application details.
		AppInfo json.RawMessage `json:"app_info"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	AppInfo, err := IsAppLogInfoFromJSON(w.AppInfo)
	if err != nil {
		return err
	}
	u.AppInfo = AppInfo
	return nil
}

// AppLinkTeamType : has no documentation (yet)
type AppLinkTeamType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAppLinkTeamType returns a new AppLinkTeamType instance
func NewAppLinkTeamType(Description string) *AppLinkTeamType {
	s := new(AppLinkTeamType)
	s.Description = Description
	return s
}

// AppLinkUserDetails : Linked app for member.
type AppLinkUserDetails struct {
	// AppInfo : Relevant application details.
	AppInfo IsAppLogInfo `json:"app_info"`
}

// NewAppLinkUserDetails returns a new AppLinkUserDetails instance
func NewAppLinkUserDetails(AppInfo IsAppLogInfo) *AppLinkUserDetails {
	s := new(AppLinkUserDetails)
	s.AppInfo = AppInfo
	return s
}

// UnmarshalJSON deserializes into a AppLinkUserDetails instance
func (u *AppLinkUserDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// AppInfo : Relevant application details.
		AppInfo json.RawMessage `json:"app_info"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	AppInfo, err := IsAppLogInfoFromJSON(w.AppInfo)
	if err != nil {
		return err
	}
	u.AppInfo = AppInfo
	return nil
}

// AppLinkUserType : has no documentation (yet)
type AppLinkUserType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAppLinkUserType returns a new AppLinkUserType instance
func NewAppLinkUserType(Description string) *AppLinkUserType {
	s := new(AppLinkUserType)
	s.Description = Description
	return s
}

// AppLogInfo : App's logged information.
type AppLogInfo struct {
	// AppId : App unique ID.
	AppId string `json:"app_id,omitempty"`
	// DisplayName : App display name.
	DisplayName string `json:"display_name,omitempty"`
}

// NewAppLogInfo returns a new AppLogInfo instance
func NewAppLogInfo() *AppLogInfo {
	s := new(AppLogInfo)
	return s
}

// IsAppLogInfo is the interface type for AppLogInfo and its subtypes
type IsAppLogInfo interface {
	IsAppLogInfo()
}

// IsAppLogInfo implements the IsAppLogInfo interface
func (u *AppLogInfo) IsAppLogInfo() {}

type appLogInfoUnion struct {
	dropbox.Tagged
	// UserOrTeamLinkedApp : has no documentation (yet)
	UserOrTeamLinkedApp *UserOrTeamLinkedAppLogInfo `json:"user_or_team_linked_app,omitempty"`
	// UserLinkedApp : has no documentation (yet)
	UserLinkedApp *UserLinkedAppLogInfo `json:"user_linked_app,omitempty"`
	// TeamLinkedApp : has no documentation (yet)
	TeamLinkedApp *TeamLinkedAppLogInfo `json:"team_linked_app,omitempty"`
}

// Valid tag values for AppLogInfo
const (
	AppLogInfoUserOrTeamLinkedApp = "user_or_team_linked_app"
	AppLogInfoUserLinkedApp       = "user_linked_app"
	AppLogInfoTeamLinkedApp       = "team_linked_app"
)

// UnmarshalJSON deserializes into a appLogInfoUnion instance
func (u *appLogInfoUnion) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "user_or_team_linked_app":
		if err = json.Unmarshal(body, &u.UserOrTeamLinkedApp); err != nil {
			return err
		}

	case "user_linked_app":
		if err = json.Unmarshal(body, &u.UserLinkedApp); err != nil {
			return err
		}

	case "team_linked_app":
		if err = json.Unmarshal(body, &u.TeamLinkedApp); err != nil {
			return err
		}

	}
	return nil
}

// IsAppLogInfoFromJSON converts JSON to a concrete IsAppLogInfo instance
func IsAppLogInfoFromJSON(data []byte) (IsAppLogInfo, error) {
	var t appLogInfoUnion
	if err := json.Unmarshal(data, &t); err != nil {
		return nil, err
	}
	switch t.Tag {
	case "user_or_team_linked_app":
		return t.UserOrTeamLinkedApp, nil

	case "user_linked_app":
		return t.UserLinkedApp, nil

	case "team_linked_app":
		return t.TeamLinkedApp, nil

	}
	return nil, nil
}

// AppPermissionsChangedDetails : Changed app permissions.
type AppPermissionsChangedDetails struct {
	// AppName : Name of the app.
	AppName string `json:"app_name,omitempty"`
	// Permission : Permission that was changed.
	Permission *AdminConsoleAppPermission `json:"permission,omitempty"`
	// PreviousValue : Previous policy.
	PreviousValue *AdminConsoleAppPolicy `json:"previous_value"`
	// NewValue : New policy.
	NewValue *AdminConsoleAppPolicy `json:"new_value"`
}

// NewAppPermissionsChangedDetails returns a new AppPermissionsChangedDetails instance
func NewAppPermissionsChangedDetails(PreviousValue *AdminConsoleAppPolicy, NewValue *AdminConsoleAppPolicy) *AppPermissionsChangedDetails {
	s := new(AppPermissionsChangedDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// AppPermissionsChangedType : has no documentation (yet)
type AppPermissionsChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAppPermissionsChangedType returns a new AppPermissionsChangedType instance
func NewAppPermissionsChangedType(Description string) *AppPermissionsChangedType {
	s := new(AppPermissionsChangedType)
	s.Description = Description
	return s
}

// AppUnlinkTeamDetails : Unlinked app for team.
type AppUnlinkTeamDetails struct {
	// AppInfo : Relevant application details.
	AppInfo IsAppLogInfo `json:"app_info"`
}

// NewAppUnlinkTeamDetails returns a new AppUnlinkTeamDetails instance
func NewAppUnlinkTeamDetails(AppInfo IsAppLogInfo) *AppUnlinkTeamDetails {
	s := new(AppUnlinkTeamDetails)
	s.AppInfo = AppInfo
	return s
}

// UnmarshalJSON deserializes into a AppUnlinkTeamDetails instance
func (u *AppUnlinkTeamDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// AppInfo : Relevant application details.
		AppInfo json.RawMessage `json:"app_info"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	AppInfo, err := IsAppLogInfoFromJSON(w.AppInfo)
	if err != nil {
		return err
	}
	u.AppInfo = AppInfo
	return nil
}

// AppUnlinkTeamType : has no documentation (yet)
type AppUnlinkTeamType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAppUnlinkTeamType returns a new AppUnlinkTeamType instance
func NewAppUnlinkTeamType(Description string) *AppUnlinkTeamType {
	s := new(AppUnlinkTeamType)
	s.Description = Description
	return s
}

// AppUnlinkUserDetails : Unlinked app for member.
type AppUnlinkUserDetails struct {
	// AppInfo : Relevant application details.
	AppInfo IsAppLogInfo `json:"app_info"`
}

// NewAppUnlinkUserDetails returns a new AppUnlinkUserDetails instance
func NewAppUnlinkUserDetails(AppInfo IsAppLogInfo) *AppUnlinkUserDetails {
	s := new(AppUnlinkUserDetails)
	s.AppInfo = AppInfo
	return s
}

// UnmarshalJSON deserializes into a AppUnlinkUserDetails instance
func (u *AppUnlinkUserDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// AppInfo : Relevant application details.
		AppInfo json.RawMessage `json:"app_info"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	AppInfo, err := IsAppLogInfoFromJSON(w.AppInfo)
	if err != nil {
		return err
	}
	u.AppInfo = AppInfo
	return nil
}

// AppUnlinkUserType : has no documentation (yet)
type AppUnlinkUserType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewAppUnlinkUserType returns a new AppUnlinkUserType instance
func NewAppUnlinkUserType(Description string) *AppUnlinkUserType {
	s := new(AppUnlinkUserType)
	s.Description = Description
	return s
}

// ApplyNamingConventionDetails : Applied naming convention.
type ApplyNamingConventionDetails struct {
}

// NewApplyNamingConventionDetails returns a new ApplyNamingConventionDetails instance
func NewApplyNamingConventionDetails() *ApplyNamingConventionDetails {
	s := new(ApplyNamingConventionDetails)
	return s
}

// ApplyNamingConventionType : has no documentation (yet)
type ApplyNamingConventionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewApplyNamingConventionType returns a new ApplyNamingConventionType instance
func NewApplyNamingConventionType(Description string) *ApplyNamingConventionType {
	s := new(ApplyNamingConventionType)
	s.Description = Description
	return s
}

// AssetLogInfo : Asset details.
type AssetLogInfo struct {
	dropbox.Tagged
	// File : File's details.
	File *FileLogInfo `json:"file,omitempty"`
	// Folder : Folder's details.
	Folder *FolderLogInfo `json:"folder,omitempty"`
	// PaperDocument : Paper document's details.
	PaperDocument *PaperDocumentLogInfo `json:"paper_document,omitempty"`
	// PaperFolder : Paper folder's details.
	PaperFolder *PaperFolderLogInfo `json:"paper_folder,omitempty"`
	// ShowcaseDocument : Showcase document's details.
	ShowcaseDocument *ShowcaseDocumentLogInfo `json:"showcase_document,omitempty"`
}

// Valid tag values for AssetLogInfo
const (
	AssetLogInfoFile             = "file"
	AssetLogInfoFolder           = "folder"
	AssetLogInfoPaperDocument    = "paper_document"
	AssetLogInfoPaperFolder      = "paper_folder"
	AssetLogInfoShowcaseDocument = "showcase_document"
	AssetLogInfoOther            = "other"
)

// UnmarshalJSON deserializes into a AssetLogInfo instance
func (u *AssetLogInfo) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "file":
		if err = json.Unmarshal(body, &u.File); err != nil {
			return err
		}

	case "folder":
		if err = json.Unmarshal(body, &u.Folder); err != nil {
			return err
		}

	case "paper_document":
		if err = json.Unmarshal(body, &u.PaperDocument); err != nil {
			return err
		}

	case "paper_folder":
		if err = json.Unmarshal(body, &u.PaperFolder); err != nil {
			return err
		}

	case "showcase_document":
		if err = json.Unmarshal(body, &u.ShowcaseDocument); err != nil {
			return err
		}

	}
	return nil
}

// BackupStatus : Backup status
type BackupStatus struct {
	dropbox.Tagged
}

// Valid tag values for BackupStatus
const (
	BackupStatusDisabled = "disabled"
	BackupStatusEnabled  = "enabled"
	BackupStatusOther    = "other"
)

// BinderAddPageDetails : Added Binder page.
type BinderAddPageDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// DocTitle : Title of the Binder doc.
	DocTitle string `json:"doc_title"`
	// BinderItemName : Name of the Binder page/section.
	BinderItemName string `json:"binder_item_name"`
}

// NewBinderAddPageDetails returns a new BinderAddPageDetails instance
func NewBinderAddPageDetails(EventUuid string, DocTitle string, BinderItemName string) *BinderAddPageDetails {
	s := new(BinderAddPageDetails)
	s.EventUuid = EventUuid
	s.DocTitle = DocTitle
	s.BinderItemName = BinderItemName
	return s
}

// BinderAddPageType : has no documentation (yet)
type BinderAddPageType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewBinderAddPageType returns a new BinderAddPageType instance
func NewBinderAddPageType(Description string) *BinderAddPageType {
	s := new(BinderAddPageType)
	s.Description = Description
	return s
}

// BinderAddSectionDetails : Added Binder section.
type BinderAddSectionDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// DocTitle : Title of the Binder doc.
	DocTitle string `json:"doc_title"`
	// BinderItemName : Name of the Binder page/section.
	BinderItemName string `json:"binder_item_name"`
}

// NewBinderAddSectionDetails returns a new BinderAddSectionDetails instance
func NewBinderAddSectionDetails(EventUuid string, DocTitle string, BinderItemName string) *BinderAddSectionDetails {
	s := new(BinderAddSectionDetails)
	s.EventUuid = EventUuid
	s.DocTitle = DocTitle
	s.BinderItemName = BinderItemName
	return s
}

// BinderAddSectionType : has no documentation (yet)
type BinderAddSectionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewBinderAddSectionType returns a new BinderAddSectionType instance
func NewBinderAddSectionType(Description string) *BinderAddSectionType {
	s := new(BinderAddSectionType)
	s.Description = Description
	return s
}

// BinderRemovePageDetails : Removed Binder page.
type BinderRemovePageDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// DocTitle : Title of the Binder doc.
	DocTitle string `json:"doc_title"`
	// BinderItemName : Name of the Binder page/section.
	BinderItemName string `json:"binder_item_name"`
}

// NewBinderRemovePageDetails returns a new BinderRemovePageDetails instance
func NewBinderRemovePageDetails(EventUuid string, DocTitle string, BinderItemName string) *BinderRemovePageDetails {
	s := new(BinderRemovePageDetails)
	s.EventUuid = EventUuid
	s.DocTitle = DocTitle
	s.BinderItemName = BinderItemName
	return s
}

// BinderRemovePageType : has no documentation (yet)
type BinderRemovePageType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewBinderRemovePageType returns a new BinderRemovePageType instance
func NewBinderRemovePageType(Description string) *BinderRemovePageType {
	s := new(BinderRemovePageType)
	s.Description = Description
	return s
}

// BinderRemoveSectionDetails : Removed Binder section.
type BinderRemoveSectionDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// DocTitle : Title of the Binder doc.
	DocTitle string `json:"doc_title"`
	// BinderItemName : Name of the Binder page/section.
	BinderItemName string `json:"binder_item_name"`
}

// NewBinderRemoveSectionDetails returns a new BinderRemoveSectionDetails instance
func NewBinderRemoveSectionDetails(EventUuid string, DocTitle string, BinderItemName string) *BinderRemoveSectionDetails {
	s := new(BinderRemoveSectionDetails)
	s.EventUuid = EventUuid
	s.DocTitle = DocTitle
	s.BinderItemName = BinderItemName
	return s
}

// BinderRemoveSectionType : has no documentation (yet)
type BinderRemoveSectionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewBinderRemoveSectionType returns a new BinderRemoveSectionType instance
func NewBinderRemoveSectionType(Description string) *BinderRemoveSectionType {
	s := new(BinderRemoveSectionType)
	s.Description = Description
	return s
}

// BinderRenamePageDetails : Renamed Binder page.
type BinderRenamePageDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// DocTitle : Title of the Binder doc.
	DocTitle string `json:"doc_title"`
	// BinderItemName : Name of the Binder page/section.
	BinderItemName string `json:"binder_item_name"`
	// PreviousBinderItemName : Previous name of the Binder page/section.
	PreviousBinderItemName string `json:"previous_binder_item_name,omitempty"`
}

// NewBinderRenamePageDetails returns a new BinderRenamePageDetails instance
func NewBinderRenamePageDetails(EventUuid string, DocTitle string, BinderItemName string) *BinderRenamePageDetails {
	s := new(BinderRenamePageDetails)
	s.EventUuid = EventUuid
	s.DocTitle = DocTitle
	s.BinderItemName = BinderItemName
	return s
}

// BinderRenamePageType : has no documentation (yet)
type BinderRenamePageType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewBinderRenamePageType returns a new BinderRenamePageType instance
func NewBinderRenamePageType(Description string) *BinderRenamePageType {
	s := new(BinderRenamePageType)
	s.Description = Description
	return s
}

// BinderRenameSectionDetails : Renamed Binder section.
type BinderRenameSectionDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// DocTitle : Title of the Binder doc.
	DocTitle string `json:"doc_title"`
	// BinderItemName : Name of the Binder page/section.
	BinderItemName string `json:"binder_item_name"`
	// PreviousBinderItemName : Previous name of the Binder page/section.
	PreviousBinderItemName string `json:"previous_binder_item_name,omitempty"`
}

// NewBinderRenameSectionDetails returns a new BinderRenameSectionDetails instance
func NewBinderRenameSectionDetails(EventUuid string, DocTitle string, BinderItemName string) *BinderRenameSectionDetails {
	s := new(BinderRenameSectionDetails)
	s.EventUuid = EventUuid
	s.DocTitle = DocTitle
	s.BinderItemName = BinderItemName
	return s
}

// BinderRenameSectionType : has no documentation (yet)
type BinderRenameSectionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewBinderRenameSectionType returns a new BinderRenameSectionType instance
func NewBinderRenameSectionType(Description string) *BinderRenameSectionType {
	s := new(BinderRenameSectionType)
	s.Description = Description
	return s
}

// BinderReorderPageDetails : Reordered Binder page.
type BinderReorderPageDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// DocTitle : Title of the Binder doc.
	DocTitle string `json:"doc_title"`
	// BinderItemName : Name of the Binder page/section.
	BinderItemName string `json:"binder_item_name"`
}

// NewBinderReorderPageDetails returns a new BinderReorderPageDetails instance
func NewBinderReorderPageDetails(EventUuid string, DocTitle string, BinderItemName string) *BinderReorderPageDetails {
	s := new(BinderReorderPageDetails)
	s.EventUuid = EventUuid
	s.DocTitle = DocTitle
	s.BinderItemName = BinderItemName
	return s
}

// BinderReorderPageType : has no documentation (yet)
type BinderReorderPageType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewBinderReorderPageType returns a new BinderReorderPageType instance
func NewBinderReorderPageType(Description string) *BinderReorderPageType {
	s := new(BinderReorderPageType)
	s.Description = Description
	return s
}

// BinderReorderSectionDetails : Reordered Binder section.
type BinderReorderSectionDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// DocTitle : Title of the Binder doc.
	DocTitle string `json:"doc_title"`
	// BinderItemName : Name of the Binder page/section.
	BinderItemName string `json:"binder_item_name"`
}

// NewBinderReorderSectionDetails returns a new BinderReorderSectionDetails instance
func NewBinderReorderSectionDetails(EventUuid string, DocTitle string, BinderItemName string) *BinderReorderSectionDetails {
	s := new(BinderReorderSectionDetails)
	s.EventUuid = EventUuid
	s.DocTitle = DocTitle
	s.BinderItemName = BinderItemName
	return s
}

// BinderReorderSectionType : has no documentation (yet)
type BinderReorderSectionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewBinderReorderSectionType returns a new BinderReorderSectionType instance
func NewBinderReorderSectionType(Description string) *BinderReorderSectionType {
	s := new(BinderReorderSectionType)
	s.Description = Description
	return s
}

// CameraUploadsPolicy : Policy for controlling if team members can activate
// camera uploads
type CameraUploadsPolicy struct {
	dropbox.Tagged
}

// Valid tag values for CameraUploadsPolicy
const (
	CameraUploadsPolicyDisabled = "disabled"
	CameraUploadsPolicyEnabled  = "enabled"
	CameraUploadsPolicyOther    = "other"
)

// CameraUploadsPolicyChangedDetails : Changed camera uploads setting for team.
type CameraUploadsPolicyChangedDetails struct {
	// NewValue : New camera uploads setting.
	NewValue *CameraUploadsPolicy `json:"new_value"`
	// PreviousValue : Previous camera uploads setting.
	PreviousValue *CameraUploadsPolicy `json:"previous_value"`
}

// NewCameraUploadsPolicyChangedDetails returns a new CameraUploadsPolicyChangedDetails instance
func NewCameraUploadsPolicyChangedDetails(NewValue *CameraUploadsPolicy, PreviousValue *CameraUploadsPolicy) *CameraUploadsPolicyChangedDetails {
	s := new(CameraUploadsPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// CameraUploadsPolicyChangedType : has no documentation (yet)
type CameraUploadsPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewCameraUploadsPolicyChangedType returns a new CameraUploadsPolicyChangedType instance
func NewCameraUploadsPolicyChangedType(Description string) *CameraUploadsPolicyChangedType {
	s := new(CameraUploadsPolicyChangedType)
	s.Description = Description
	return s
}

// CaptureTranscriptPolicy : Policy for deciding whether team users can
// transcription in Capture
type CaptureTranscriptPolicy struct {
	dropbox.Tagged
}

// Valid tag values for CaptureTranscriptPolicy
const (
	CaptureTranscriptPolicyDefault  = "default"
	CaptureTranscriptPolicyDisabled = "disabled"
	CaptureTranscriptPolicyEnabled  = "enabled"
	CaptureTranscriptPolicyOther    = "other"
)

// CaptureTranscriptPolicyChangedDetails : Changed Capture transcription policy
// for team.
type CaptureTranscriptPolicyChangedDetails struct {
	// NewValue : To.
	NewValue *CaptureTranscriptPolicy `json:"new_value"`
	// PreviousValue : From.
	PreviousValue *CaptureTranscriptPolicy `json:"previous_value"`
}

// NewCaptureTranscriptPolicyChangedDetails returns a new CaptureTranscriptPolicyChangedDetails instance
func NewCaptureTranscriptPolicyChangedDetails(NewValue *CaptureTranscriptPolicy, PreviousValue *CaptureTranscriptPolicy) *CaptureTranscriptPolicyChangedDetails {
	s := new(CaptureTranscriptPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// CaptureTranscriptPolicyChangedType : has no documentation (yet)
type CaptureTranscriptPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewCaptureTranscriptPolicyChangedType returns a new CaptureTranscriptPolicyChangedType instance
func NewCaptureTranscriptPolicyChangedType(Description string) *CaptureTranscriptPolicyChangedType {
	s := new(CaptureTranscriptPolicyChangedType)
	s.Description = Description
	return s
}

// Certificate : Certificate details.
type Certificate struct {
	// Subject : Certificate subject.
	Subject string `json:"subject"`
	// Issuer : Certificate issuer.
	Issuer string `json:"issuer"`
	// IssueDate : Certificate issue date.
	IssueDate string `json:"issue_date"`
	// ExpirationDate : Certificate expiration date.
	ExpirationDate string `json:"expiration_date"`
	// SerialNumber : Certificate serial number.
	SerialNumber string `json:"serial_number"`
	// Sha1Fingerprint : Certificate sha1 fingerprint.
	Sha1Fingerprint string `json:"sha1_fingerprint"`
	// CommonName : Certificate common name.
	CommonName string `json:"common_name,omitempty"`
}

// NewCertificate returns a new Certificate instance
func NewCertificate(Subject string, Issuer string, IssueDate string, ExpirationDate string, SerialNumber string, Sha1Fingerprint string) *Certificate {
	s := new(Certificate)
	s.Subject = Subject
	s.Issuer = Issuer
	s.IssueDate = IssueDate
	s.ExpirationDate = ExpirationDate
	s.SerialNumber = SerialNumber
	s.Sha1Fingerprint = Sha1Fingerprint
	return s
}

// ChangeLinkExpirationPolicy : Policy for deciding whether the team's default
// expiration days policy must be enforced when an externally shared link is
// updated
type ChangeLinkExpirationPolicy struct {
	dropbox.Tagged
}

// Valid tag values for ChangeLinkExpirationPolicy
const (
	ChangeLinkExpirationPolicyAllowed    = "allowed"
	ChangeLinkExpirationPolicyNotAllowed = "not_allowed"
	ChangeLinkExpirationPolicyOther      = "other"
)

// ChangedEnterpriseAdminRoleDetails : Changed enterprise admin role.
type ChangedEnterpriseAdminRoleDetails struct {
	// PreviousValue : The member&#x2019s previous enterprise admin role.
	PreviousValue *FedAdminRole `json:"previous_value"`
	// NewValue : The member&#x2019s new enterprise admin role.
	NewValue *FedAdminRole `json:"new_value"`
	// TeamName : The name of the member&#x2019s team.
	TeamName string `json:"team_name"`
}

// NewChangedEnterpriseAdminRoleDetails returns a new ChangedEnterpriseAdminRoleDetails instance
func NewChangedEnterpriseAdminRoleDetails(PreviousValue *FedAdminRole, NewValue *FedAdminRole, TeamName string) *ChangedEnterpriseAdminRoleDetails {
	s := new(ChangedEnterpriseAdminRoleDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	s.TeamName = TeamName
	return s
}

// ChangedEnterpriseAdminRoleType : has no documentation (yet)
type ChangedEnterpriseAdminRoleType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewChangedEnterpriseAdminRoleType returns a new ChangedEnterpriseAdminRoleType instance
func NewChangedEnterpriseAdminRoleType(Description string) *ChangedEnterpriseAdminRoleType {
	s := new(ChangedEnterpriseAdminRoleType)
	s.Description = Description
	return s
}

// ChangedEnterpriseConnectedTeamStatusDetails : Changed enterprise-connected
// team status.
type ChangedEnterpriseConnectedTeamStatusDetails struct {
	// Action : The preformed change in the team&#x2019s connection status.
	Action *FedHandshakeAction `json:"action"`
	// AdditionalInfo : Additional information about the organization or team.
	AdditionalInfo *FederationStatusChangeAdditionalInfo `json:"additional_info"`
	// PreviousValue : Previous request state.
	PreviousValue *TrustedTeamsRequestState `json:"previous_value"`
	// NewValue : New request state.
	NewValue *TrustedTeamsRequestState `json:"new_value"`
}

// NewChangedEnterpriseConnectedTeamStatusDetails returns a new ChangedEnterpriseConnectedTeamStatusDetails instance
func NewChangedEnterpriseConnectedTeamStatusDetails(Action *FedHandshakeAction, AdditionalInfo *FederationStatusChangeAdditionalInfo, PreviousValue *TrustedTeamsRequestState, NewValue *TrustedTeamsRequestState) *ChangedEnterpriseConnectedTeamStatusDetails {
	s := new(ChangedEnterpriseConnectedTeamStatusDetails)
	s.Action = Action
	s.AdditionalInfo = AdditionalInfo
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// ChangedEnterpriseConnectedTeamStatusType : has no documentation (yet)
type ChangedEnterpriseConnectedTeamStatusType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewChangedEnterpriseConnectedTeamStatusType returns a new ChangedEnterpriseConnectedTeamStatusType instance
func NewChangedEnterpriseConnectedTeamStatusType(Description string) *ChangedEnterpriseConnectedTeamStatusType {
	s := new(ChangedEnterpriseConnectedTeamStatusType)
	s.Description = Description
	return s
}

// ClassificationChangePolicyDetails : Changed classification policy for team.
type ClassificationChangePolicyDetails struct {
	// PreviousValue : Previous classification policy.
	PreviousValue *ClassificationPolicyEnumWrapper `json:"previous_value"`
	// NewValue : New classification policy.
	NewValue *ClassificationPolicyEnumWrapper `json:"new_value"`
	// ClassificationType : Policy type.
	ClassificationType *ClassificationType `json:"classification_type"`
}

// NewClassificationChangePolicyDetails returns a new ClassificationChangePolicyDetails instance
func NewClassificationChangePolicyDetails(PreviousValue *ClassificationPolicyEnumWrapper, NewValue *ClassificationPolicyEnumWrapper, ClassificationType *ClassificationType) *ClassificationChangePolicyDetails {
	s := new(ClassificationChangePolicyDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	s.ClassificationType = ClassificationType
	return s
}

// ClassificationChangePolicyType : has no documentation (yet)
type ClassificationChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewClassificationChangePolicyType returns a new ClassificationChangePolicyType instance
func NewClassificationChangePolicyType(Description string) *ClassificationChangePolicyType {
	s := new(ClassificationChangePolicyType)
	s.Description = Description
	return s
}

// ClassificationCreateReportDetails : Created Classification report.
type ClassificationCreateReportDetails struct {
}

// NewClassificationCreateReportDetails returns a new ClassificationCreateReportDetails instance
func NewClassificationCreateReportDetails() *ClassificationCreateReportDetails {
	s := new(ClassificationCreateReportDetails)
	return s
}

// ClassificationCreateReportFailDetails : Couldn't create Classification
// report.
type ClassificationCreateReportFailDetails struct {
	// FailureReason : Failure reason.
	FailureReason *team.TeamReportFailureReason `json:"failure_reason"`
}

// NewClassificationCreateReportFailDetails returns a new ClassificationCreateReportFailDetails instance
func NewClassificationCreateReportFailDetails(FailureReason *team.TeamReportFailureReason) *ClassificationCreateReportFailDetails {
	s := new(ClassificationCreateReportFailDetails)
	s.FailureReason = FailureReason
	return s
}

// ClassificationCreateReportFailType : has no documentation (yet)
type ClassificationCreateReportFailType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewClassificationCreateReportFailType returns a new ClassificationCreateReportFailType instance
func NewClassificationCreateReportFailType(Description string) *ClassificationCreateReportFailType {
	s := new(ClassificationCreateReportFailType)
	s.Description = Description
	return s
}

// ClassificationCreateReportType : has no documentation (yet)
type ClassificationCreateReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewClassificationCreateReportType returns a new ClassificationCreateReportType instance
func NewClassificationCreateReportType(Description string) *ClassificationCreateReportType {
	s := new(ClassificationCreateReportType)
	s.Description = Description
	return s
}

// ClassificationPolicyEnumWrapper : Policy for controlling team access to the
// classification feature
type ClassificationPolicyEnumWrapper struct {
	dropbox.Tagged
}

// Valid tag values for ClassificationPolicyEnumWrapper
const (
	ClassificationPolicyEnumWrapperDisabled             = "disabled"
	ClassificationPolicyEnumWrapperEnabled              = "enabled"
	ClassificationPolicyEnumWrapperMemberAndTeamFolders = "member_and_team_folders"
	ClassificationPolicyEnumWrapperTeamFolders          = "team_folders"
	ClassificationPolicyEnumWrapperOther                = "other"
)

// ClassificationType : The type of classification (currently only personal
// information)
type ClassificationType struct {
	dropbox.Tagged
}

// Valid tag values for ClassificationType
const (
	ClassificationTypePersonalInformation = "personal_information"
	ClassificationTypePii                 = "pii"
	ClassificationTypeOther               = "other"
)

// CollectionShareDetails : Shared album.
type CollectionShareDetails struct {
	// AlbumName : Album name.
	AlbumName string `json:"album_name"`
}

// NewCollectionShareDetails returns a new CollectionShareDetails instance
func NewCollectionShareDetails(AlbumName string) *CollectionShareDetails {
	s := new(CollectionShareDetails)
	s.AlbumName = AlbumName
	return s
}

// CollectionShareType : has no documentation (yet)
type CollectionShareType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewCollectionShareType returns a new CollectionShareType instance
func NewCollectionShareType(Description string) *CollectionShareType {
	s := new(CollectionShareType)
	s.Description = Description
	return s
}

// ComputerBackupPolicy : Policy for controlling team access to computer backup
// feature
type ComputerBackupPolicy struct {
	dropbox.Tagged
}

// Valid tag values for ComputerBackupPolicy
const (
	ComputerBackupPolicyDefault  = "default"
	ComputerBackupPolicyDisabled = "disabled"
	ComputerBackupPolicyEnabled  = "enabled"
	ComputerBackupPolicyOther    = "other"
)

// ComputerBackupPolicyChangedDetails : Changed computer backup policy for team.
type ComputerBackupPolicyChangedDetails struct {
	// NewValue : New computer backup policy.
	NewValue *ComputerBackupPolicy `json:"new_value"`
	// PreviousValue : Previous computer backup policy.
	PreviousValue *ComputerBackupPolicy `json:"previous_value"`
}

// NewComputerBackupPolicyChangedDetails returns a new ComputerBackupPolicyChangedDetails instance
func NewComputerBackupPolicyChangedDetails(NewValue *ComputerBackupPolicy, PreviousValue *ComputerBackupPolicy) *ComputerBackupPolicyChangedDetails {
	s := new(ComputerBackupPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// ComputerBackupPolicyChangedType : has no documentation (yet)
type ComputerBackupPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewComputerBackupPolicyChangedType returns a new ComputerBackupPolicyChangedType instance
func NewComputerBackupPolicyChangedType(Description string) *ComputerBackupPolicyChangedType {
	s := new(ComputerBackupPolicyChangedType)
	s.Description = Description
	return s
}

// ConnectedTeamName : The name of the team
type ConnectedTeamName struct {
	// Team : The name of the team.
	Team string `json:"team"`
}

// NewConnectedTeamName returns a new ConnectedTeamName instance
func NewConnectedTeamName(Team string) *ConnectedTeamName {
	s := new(ConnectedTeamName)
	s.Team = Team
	return s
}

// ContentAdministrationPolicyChangedDetails : Changed content management
// setting.
type ContentAdministrationPolicyChangedDetails struct {
	// NewValue : New content administration policy.
	NewValue string `json:"new_value"`
	// PreviousValue : Previous content administration policy.
	PreviousValue string `json:"previous_value"`
}

// NewContentAdministrationPolicyChangedDetails returns a new ContentAdministrationPolicyChangedDetails instance
func NewContentAdministrationPolicyChangedDetails(NewValue string, PreviousValue string) *ContentAdministrationPolicyChangedDetails {
	s := new(ContentAdministrationPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// ContentAdministrationPolicyChangedType : has no documentation (yet)
type ContentAdministrationPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewContentAdministrationPolicyChangedType returns a new ContentAdministrationPolicyChangedType instance
func NewContentAdministrationPolicyChangedType(Description string) *ContentAdministrationPolicyChangedType {
	s := new(ContentAdministrationPolicyChangedType)
	s.Description = Description
	return s
}

// ContentPermanentDeletePolicy : Policy for pemanent content deletion
type ContentPermanentDeletePolicy struct {
	dropbox.Tagged
}

// Valid tag values for ContentPermanentDeletePolicy
const (
	ContentPermanentDeletePolicyDisabled = "disabled"
	ContentPermanentDeletePolicyEnabled  = "enabled"
	ContentPermanentDeletePolicyOther    = "other"
)

// ContextLogInfo : The primary entity on which the action was done.
type ContextLogInfo struct {
	dropbox.Tagged
	// NonTeamMember : Action was done on behalf of a non team member.
	NonTeamMember *NonTeamMemberLogInfo `json:"non_team_member,omitempty"`
	// OrganizationTeam : Action was done on behalf of a team that's part of an
	// organization.
	OrganizationTeam *TeamLogInfo `json:"organization_team,omitempty"`
	// TeamMember : Action was done on behalf of a team member.
	TeamMember *TeamMemberLogInfo `json:"team_member,omitempty"`
	// TrustedNonTeamMember : Action was done on behalf of a trusted non team
	// member.
	TrustedNonTeamMember *TrustedNonTeamMemberLogInfo `json:"trusted_non_team_member,omitempty"`
}

// Valid tag values for ContextLogInfo
const (
	ContextLogInfoAnonymous            = "anonymous"
	ContextLogInfoNonTeamMember        = "non_team_member"
	ContextLogInfoOrganizationTeam     = "organization_team"
	ContextLogInfoTeam                 = "team"
	ContextLogInfoTeamMember           = "team_member"
	ContextLogInfoTrustedNonTeamMember = "trusted_non_team_member"
	ContextLogInfoOther                = "other"
)

// UnmarshalJSON deserializes into a ContextLogInfo instance
func (u *ContextLogInfo) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "non_team_member":
		if err = json.Unmarshal(body, &u.NonTeamMember); err != nil {
			return err
		}

	case "organization_team":
		if err = json.Unmarshal(body, &u.OrganizationTeam); err != nil {
			return err
		}

	case "team_member":
		if err = json.Unmarshal(body, &u.TeamMember); err != nil {
			return err
		}

	case "trusted_non_team_member":
		if err = json.Unmarshal(body, &u.TrustedNonTeamMember); err != nil {
			return err
		}

	}
	return nil
}

// CreateFolderDetails : Created folders.
type CreateFolderDetails struct {
}

// NewCreateFolderDetails returns a new CreateFolderDetails instance
func NewCreateFolderDetails() *CreateFolderDetails {
	s := new(CreateFolderDetails)
	return s
}

// CreateFolderType : has no documentation (yet)
type CreateFolderType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewCreateFolderType returns a new CreateFolderType instance
func NewCreateFolderType(Description string) *CreateFolderType {
	s := new(CreateFolderType)
	s.Description = Description
	return s
}

// CreateTeamInviteLinkDetails : Created team invite link.
type CreateTeamInviteLinkDetails struct {
	// LinkUrl : The invite link url that was created.
	LinkUrl string `json:"link_url"`
	// ExpiryDate : The expiration date of the invite link.
	ExpiryDate string `json:"expiry_date"`
}

// NewCreateTeamInviteLinkDetails returns a new CreateTeamInviteLinkDetails instance
func NewCreateTeamInviteLinkDetails(LinkUrl string, ExpiryDate string) *CreateTeamInviteLinkDetails {
	s := new(CreateTeamInviteLinkDetails)
	s.LinkUrl = LinkUrl
	s.ExpiryDate = ExpiryDate
	return s
}

// CreateTeamInviteLinkType : has no documentation (yet)
type CreateTeamInviteLinkType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewCreateTeamInviteLinkType returns a new CreateTeamInviteLinkType instance
func NewCreateTeamInviteLinkType(Description string) *CreateTeamInviteLinkType {
	s := new(CreateTeamInviteLinkType)
	s.Description = Description
	return s
}

// DataPlacementRestrictionChangePolicyDetails : Set restrictions on data center
// locations where team data resides.
type DataPlacementRestrictionChangePolicyDetails struct {
	// PreviousValue : Previous placement restriction.
	PreviousValue *PlacementRestriction `json:"previous_value"`
	// NewValue : New placement restriction.
	NewValue *PlacementRestriction `json:"new_value"`
}

// NewDataPlacementRestrictionChangePolicyDetails returns a new DataPlacementRestrictionChangePolicyDetails instance
func NewDataPlacementRestrictionChangePolicyDetails(PreviousValue *PlacementRestriction, NewValue *PlacementRestriction) *DataPlacementRestrictionChangePolicyDetails {
	s := new(DataPlacementRestrictionChangePolicyDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// DataPlacementRestrictionChangePolicyType : has no documentation (yet)
type DataPlacementRestrictionChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDataPlacementRestrictionChangePolicyType returns a new DataPlacementRestrictionChangePolicyType instance
func NewDataPlacementRestrictionChangePolicyType(Description string) *DataPlacementRestrictionChangePolicyType {
	s := new(DataPlacementRestrictionChangePolicyType)
	s.Description = Description
	return s
}

// DataPlacementRestrictionSatisfyPolicyDetails : Completed restrictions on data
// center locations where team data resides.
type DataPlacementRestrictionSatisfyPolicyDetails struct {
	// PlacementRestriction : Placement restriction.
	PlacementRestriction *PlacementRestriction `json:"placement_restriction"`
}

// NewDataPlacementRestrictionSatisfyPolicyDetails returns a new DataPlacementRestrictionSatisfyPolicyDetails instance
func NewDataPlacementRestrictionSatisfyPolicyDetails(PlacementRestriction *PlacementRestriction) *DataPlacementRestrictionSatisfyPolicyDetails {
	s := new(DataPlacementRestrictionSatisfyPolicyDetails)
	s.PlacementRestriction = PlacementRestriction
	return s
}

// DataPlacementRestrictionSatisfyPolicyType : has no documentation (yet)
type DataPlacementRestrictionSatisfyPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDataPlacementRestrictionSatisfyPolicyType returns a new DataPlacementRestrictionSatisfyPolicyType instance
func NewDataPlacementRestrictionSatisfyPolicyType(Description string) *DataPlacementRestrictionSatisfyPolicyType {
	s := new(DataPlacementRestrictionSatisfyPolicyType)
	s.Description = Description
	return s
}

// DataResidencyMigrationRequestSuccessfulDetails : Requested data residency
// migration for team data.
type DataResidencyMigrationRequestSuccessfulDetails struct {
}

// NewDataResidencyMigrationRequestSuccessfulDetails returns a new DataResidencyMigrationRequestSuccessfulDetails instance
func NewDataResidencyMigrationRequestSuccessfulDetails() *DataResidencyMigrationRequestSuccessfulDetails {
	s := new(DataResidencyMigrationRequestSuccessfulDetails)
	return s
}

// DataResidencyMigrationRequestSuccessfulType : has no documentation (yet)
type DataResidencyMigrationRequestSuccessfulType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDataResidencyMigrationRequestSuccessfulType returns a new DataResidencyMigrationRequestSuccessfulType instance
func NewDataResidencyMigrationRequestSuccessfulType(Description string) *DataResidencyMigrationRequestSuccessfulType {
	s := new(DataResidencyMigrationRequestSuccessfulType)
	s.Description = Description
	return s
}

// DataResidencyMigrationRequestUnsuccessfulDetails : Request for data residency
// migration for team data has failed.
type DataResidencyMigrationRequestUnsuccessfulDetails struct {
}

// NewDataResidencyMigrationRequestUnsuccessfulDetails returns a new DataResidencyMigrationRequestUnsuccessfulDetails instance
func NewDataResidencyMigrationRequestUnsuccessfulDetails() *DataResidencyMigrationRequestUnsuccessfulDetails {
	s := new(DataResidencyMigrationRequestUnsuccessfulDetails)
	return s
}

// DataResidencyMigrationRequestUnsuccessfulType : has no documentation (yet)
type DataResidencyMigrationRequestUnsuccessfulType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDataResidencyMigrationRequestUnsuccessfulType returns a new DataResidencyMigrationRequestUnsuccessfulType instance
func NewDataResidencyMigrationRequestUnsuccessfulType(Description string) *DataResidencyMigrationRequestUnsuccessfulType {
	s := new(DataResidencyMigrationRequestUnsuccessfulType)
	s.Description = Description
	return s
}

// DefaultLinkExpirationDaysPolicy : Policy for the default number of days until
// an externally shared link expires
type DefaultLinkExpirationDaysPolicy struct {
	dropbox.Tagged
}

// Valid tag values for DefaultLinkExpirationDaysPolicy
const (
	DefaultLinkExpirationDaysPolicyDay1   = "day_1"
	DefaultLinkExpirationDaysPolicyDay180 = "day_180"
	DefaultLinkExpirationDaysPolicyDay3   = "day_3"
	DefaultLinkExpirationDaysPolicyDay30  = "day_30"
	DefaultLinkExpirationDaysPolicyDay7   = "day_7"
	DefaultLinkExpirationDaysPolicyDay90  = "day_90"
	DefaultLinkExpirationDaysPolicyNone   = "none"
	DefaultLinkExpirationDaysPolicyYear1  = "year_1"
	DefaultLinkExpirationDaysPolicyOther  = "other"
)

// DeleteTeamInviteLinkDetails : Deleted team invite link.
type DeleteTeamInviteLinkDetails struct {
	// LinkUrl : The invite link url that was deleted.
	LinkUrl string `json:"link_url"`
}

// NewDeleteTeamInviteLinkDetails returns a new DeleteTeamInviteLinkDetails instance
func NewDeleteTeamInviteLinkDetails(LinkUrl string) *DeleteTeamInviteLinkDetails {
	s := new(DeleteTeamInviteLinkDetails)
	s.LinkUrl = LinkUrl
	return s
}

// DeleteTeamInviteLinkType : has no documentation (yet)
type DeleteTeamInviteLinkType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeleteTeamInviteLinkType returns a new DeleteTeamInviteLinkType instance
func NewDeleteTeamInviteLinkType(Description string) *DeleteTeamInviteLinkType {
	s := new(DeleteTeamInviteLinkType)
	s.Description = Description
	return s
}

// DeviceSessionLogInfo : Device's session logged information.
type DeviceSessionLogInfo struct {
	// IpAddress : The IP address of the last activity from this session.
	IpAddress string `json:"ip_address,omitempty"`
	// Created : The time this session was created.
	Created *time.Time `json:"created,omitempty"`
	// Updated : The time of the last activity from this session.
	Updated *time.Time `json:"updated,omitempty"`
}

// NewDeviceSessionLogInfo returns a new DeviceSessionLogInfo instance
func NewDeviceSessionLogInfo() *DeviceSessionLogInfo {
	s := new(DeviceSessionLogInfo)
	return s
}

// IsDeviceSessionLogInfo is the interface type for DeviceSessionLogInfo and its subtypes
type IsDeviceSessionLogInfo interface {
	IsDeviceSessionLogInfo()
}

// IsDeviceSessionLogInfo implements the IsDeviceSessionLogInfo interface
func (u *DeviceSessionLogInfo) IsDeviceSessionLogInfo() {}

type deviceSessionLogInfoUnion struct {
	dropbox.Tagged
	// DesktopDeviceSession : has no documentation (yet)
	DesktopDeviceSession *DesktopDeviceSessionLogInfo `json:"desktop_device_session,omitempty"`
	// MobileDeviceSession : has no documentation (yet)
	MobileDeviceSession *MobileDeviceSessionLogInfo `json:"mobile_device_session,omitempty"`
	// WebDeviceSession : has no documentation (yet)
	WebDeviceSession *WebDeviceSessionLogInfo `json:"web_device_session,omitempty"`
	// LegacyDeviceSession : has no documentation (yet)
	LegacyDeviceSession *LegacyDeviceSessionLogInfo `json:"legacy_device_session,omitempty"`
}

// Valid tag values for DeviceSessionLogInfo
const (
	DeviceSessionLogInfoDesktopDeviceSession = "desktop_device_session"
	DeviceSessionLogInfoMobileDeviceSession  = "mobile_device_session"
	DeviceSessionLogInfoWebDeviceSession     = "web_device_session"
	DeviceSessionLogInfoLegacyDeviceSession  = "legacy_device_session"
)

// UnmarshalJSON deserializes into a deviceSessionLogInfoUnion instance
func (u *deviceSessionLogInfoUnion) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "desktop_device_session":
		if err = json.Unmarshal(body, &u.DesktopDeviceSession); err != nil {
			return err
		}

	case "mobile_device_session":
		if err = json.Unmarshal(body, &u.MobileDeviceSession); err != nil {
			return err
		}

	case "web_device_session":
		if err = json.Unmarshal(body, &u.WebDeviceSession); err != nil {
			return err
		}

	case "legacy_device_session":
		if err = json.Unmarshal(body, &u.LegacyDeviceSession); err != nil {
			return err
		}

	}
	return nil
}

// IsDeviceSessionLogInfoFromJSON converts JSON to a concrete IsDeviceSessionLogInfo instance
func IsDeviceSessionLogInfoFromJSON(data []byte) (IsDeviceSessionLogInfo, error) {
	var t deviceSessionLogInfoUnion
	if err := json.Unmarshal(data, &t); err != nil {
		return nil, err
	}
	switch t.Tag {
	case "desktop_device_session":
		return t.DesktopDeviceSession, nil

	case "mobile_device_session":
		return t.MobileDeviceSession, nil

	case "web_device_session":
		return t.WebDeviceSession, nil

	case "legacy_device_session":
		return t.LegacyDeviceSession, nil

	}
	return nil, nil
}

// DesktopDeviceSessionLogInfo : Information about linked Dropbox desktop client
// sessions
type DesktopDeviceSessionLogInfo struct {
	DeviceSessionLogInfo
	// SessionInfo : Desktop session unique id.
	SessionInfo *DesktopSessionLogInfo `json:"session_info,omitempty"`
	// HostName : Name of the hosting desktop.
	HostName string `json:"host_name"`
	// ClientType : The Dropbox desktop client type.
	ClientType *team.DesktopPlatform `json:"client_type"`
	// ClientVersion : The Dropbox client version.
	ClientVersion string `json:"client_version,omitempty"`
	// Platform : Information on the hosting platform.
	Platform string `json:"platform"`
	// IsDeleteOnUnlinkSupported : Whether itu2019s possible to delete all of
	// the account files upon unlinking.
	IsDeleteOnUnlinkSupported bool `json:"is_delete_on_unlink_supported"`
}

// NewDesktopDeviceSessionLogInfo returns a new DesktopDeviceSessionLogInfo instance
func NewDesktopDeviceSessionLogInfo(HostName string, ClientType *team.DesktopPlatform, Platform string, IsDeleteOnUnlinkSupported bool) *DesktopDeviceSessionLogInfo {
	s := new(DesktopDeviceSessionLogInfo)
	s.HostName = HostName
	s.ClientType = ClientType
	s.Platform = Platform
	s.IsDeleteOnUnlinkSupported = IsDeleteOnUnlinkSupported
	return s
}

// SessionLogInfo : Session's logged information.
type SessionLogInfo struct {
	// SessionId : Session ID.
	SessionId string `json:"session_id,omitempty"`
}

// NewSessionLogInfo returns a new SessionLogInfo instance
func NewSessionLogInfo() *SessionLogInfo {
	s := new(SessionLogInfo)
	return s
}

// IsSessionLogInfo is the interface type for SessionLogInfo and its subtypes
type IsSessionLogInfo interface {
	IsSessionLogInfo()
}

// IsSessionLogInfo implements the IsSessionLogInfo interface
func (u *SessionLogInfo) IsSessionLogInfo() {}

type sessionLogInfoUnion struct {
	dropbox.Tagged
	// Web : has no documentation (yet)
	Web *WebSessionLogInfo `json:"web,omitempty"`
	// Desktop : has no documentation (yet)
	Desktop *DesktopSessionLogInfo `json:"desktop,omitempty"`
	// Mobile : has no documentation (yet)
	Mobile *MobileSessionLogInfo `json:"mobile,omitempty"`
}

// Valid tag values for SessionLogInfo
const (
	SessionLogInfoWeb     = "web"
	SessionLogInfoDesktop = "desktop"
	SessionLogInfoMobile  = "mobile"
)

// UnmarshalJSON deserializes into a sessionLogInfoUnion instance
func (u *sessionLogInfoUnion) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "web":
		if err = json.Unmarshal(body, &u.Web); err != nil {
			return err
		}

	case "desktop":
		if err = json.Unmarshal(body, &u.Desktop); err != nil {
			return err
		}

	case "mobile":
		if err = json.Unmarshal(body, &u.Mobile); err != nil {
			return err
		}

	}
	return nil
}

// IsSessionLogInfoFromJSON converts JSON to a concrete IsSessionLogInfo instance
func IsSessionLogInfoFromJSON(data []byte) (IsSessionLogInfo, error) {
	var t sessionLogInfoUnion
	if err := json.Unmarshal(data, &t); err != nil {
		return nil, err
	}
	switch t.Tag {
	case "web":
		return t.Web, nil

	case "desktop":
		return t.Desktop, nil

	case "mobile":
		return t.Mobile, nil

	}
	return nil, nil
}

// DesktopSessionLogInfo : Desktop session.
type DesktopSessionLogInfo struct {
	SessionLogInfo
}

// NewDesktopSessionLogInfo returns a new DesktopSessionLogInfo instance
func NewDesktopSessionLogInfo() *DesktopSessionLogInfo {
	s := new(DesktopSessionLogInfo)
	return s
}

// DeviceApprovalsAddExceptionDetails : Added members to device approvals
// exception list.
type DeviceApprovalsAddExceptionDetails struct {
}

// NewDeviceApprovalsAddExceptionDetails returns a new DeviceApprovalsAddExceptionDetails instance
func NewDeviceApprovalsAddExceptionDetails() *DeviceApprovalsAddExceptionDetails {
	s := new(DeviceApprovalsAddExceptionDetails)
	return s
}

// DeviceApprovalsAddExceptionType : has no documentation (yet)
type DeviceApprovalsAddExceptionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceApprovalsAddExceptionType returns a new DeviceApprovalsAddExceptionType instance
func NewDeviceApprovalsAddExceptionType(Description string) *DeviceApprovalsAddExceptionType {
	s := new(DeviceApprovalsAddExceptionType)
	s.Description = Description
	return s
}

// DeviceApprovalsChangeDesktopPolicyDetails : Set/removed limit on number of
// computers member can link to team Dropbox account.
type DeviceApprovalsChangeDesktopPolicyDetails struct {
	// NewValue : New desktop device approvals policy. Might be missing due to
	// historical data gap.
	NewValue *DeviceApprovalsPolicy `json:"new_value,omitempty"`
	// PreviousValue : Previous desktop device approvals policy. Might be
	// missing due to historical data gap.
	PreviousValue *DeviceApprovalsPolicy `json:"previous_value,omitempty"`
}

// NewDeviceApprovalsChangeDesktopPolicyDetails returns a new DeviceApprovalsChangeDesktopPolicyDetails instance
func NewDeviceApprovalsChangeDesktopPolicyDetails() *DeviceApprovalsChangeDesktopPolicyDetails {
	s := new(DeviceApprovalsChangeDesktopPolicyDetails)
	return s
}

// DeviceApprovalsChangeDesktopPolicyType : has no documentation (yet)
type DeviceApprovalsChangeDesktopPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceApprovalsChangeDesktopPolicyType returns a new DeviceApprovalsChangeDesktopPolicyType instance
func NewDeviceApprovalsChangeDesktopPolicyType(Description string) *DeviceApprovalsChangeDesktopPolicyType {
	s := new(DeviceApprovalsChangeDesktopPolicyType)
	s.Description = Description
	return s
}

// DeviceApprovalsChangeMobilePolicyDetails : Set/removed limit on number of
// mobile devices member can link to team Dropbox account.
type DeviceApprovalsChangeMobilePolicyDetails struct {
	// NewValue : New mobile device approvals policy. Might be missing due to
	// historical data gap.
	NewValue *DeviceApprovalsPolicy `json:"new_value,omitempty"`
	// PreviousValue : Previous mobile device approvals policy. Might be missing
	// due to historical data gap.
	PreviousValue *DeviceApprovalsPolicy `json:"previous_value,omitempty"`
}

// NewDeviceApprovalsChangeMobilePolicyDetails returns a new DeviceApprovalsChangeMobilePolicyDetails instance
func NewDeviceApprovalsChangeMobilePolicyDetails() *DeviceApprovalsChangeMobilePolicyDetails {
	s := new(DeviceApprovalsChangeMobilePolicyDetails)
	return s
}

// DeviceApprovalsChangeMobilePolicyType : has no documentation (yet)
type DeviceApprovalsChangeMobilePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceApprovalsChangeMobilePolicyType returns a new DeviceApprovalsChangeMobilePolicyType instance
func NewDeviceApprovalsChangeMobilePolicyType(Description string) *DeviceApprovalsChangeMobilePolicyType {
	s := new(DeviceApprovalsChangeMobilePolicyType)
	s.Description = Description
	return s
}

// DeviceApprovalsChangeOverageActionDetails : Changed device approvals setting
// when member is over limit.
type DeviceApprovalsChangeOverageActionDetails struct {
	// NewValue : New over the limits policy. Might be missing due to historical
	// data gap.
	NewValue *team_policies.RolloutMethod `json:"new_value,omitempty"`
	// PreviousValue : Previous over the limit policy. Might be missing due to
	// historical data gap.
	PreviousValue *team_policies.RolloutMethod `json:"previous_value,omitempty"`
}

// NewDeviceApprovalsChangeOverageActionDetails returns a new DeviceApprovalsChangeOverageActionDetails instance
func NewDeviceApprovalsChangeOverageActionDetails() *DeviceApprovalsChangeOverageActionDetails {
	s := new(DeviceApprovalsChangeOverageActionDetails)
	return s
}

// DeviceApprovalsChangeOverageActionType : has no documentation (yet)
type DeviceApprovalsChangeOverageActionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceApprovalsChangeOverageActionType returns a new DeviceApprovalsChangeOverageActionType instance
func NewDeviceApprovalsChangeOverageActionType(Description string) *DeviceApprovalsChangeOverageActionType {
	s := new(DeviceApprovalsChangeOverageActionType)
	s.Description = Description
	return s
}

// DeviceApprovalsChangeUnlinkActionDetails : Changed device approvals setting
// when member unlinks approved device.
type DeviceApprovalsChangeUnlinkActionDetails struct {
	// NewValue : New device unlink policy. Might be missing due to historical
	// data gap.
	NewValue *DeviceUnlinkPolicy `json:"new_value,omitempty"`
	// PreviousValue : Previous device unlink policy. Might be missing due to
	// historical data gap.
	PreviousValue *DeviceUnlinkPolicy `json:"previous_value,omitempty"`
}

// NewDeviceApprovalsChangeUnlinkActionDetails returns a new DeviceApprovalsChangeUnlinkActionDetails instance
func NewDeviceApprovalsChangeUnlinkActionDetails() *DeviceApprovalsChangeUnlinkActionDetails {
	s := new(DeviceApprovalsChangeUnlinkActionDetails)
	return s
}

// DeviceApprovalsChangeUnlinkActionType : has no documentation (yet)
type DeviceApprovalsChangeUnlinkActionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceApprovalsChangeUnlinkActionType returns a new DeviceApprovalsChangeUnlinkActionType instance
func NewDeviceApprovalsChangeUnlinkActionType(Description string) *DeviceApprovalsChangeUnlinkActionType {
	s := new(DeviceApprovalsChangeUnlinkActionType)
	s.Description = Description
	return s
}

// DeviceApprovalsPolicy : has no documentation (yet)
type DeviceApprovalsPolicy struct {
	dropbox.Tagged
}

// Valid tag values for DeviceApprovalsPolicy
const (
	DeviceApprovalsPolicyLimited   = "limited"
	DeviceApprovalsPolicyUnlimited = "unlimited"
	DeviceApprovalsPolicyOther     = "other"
)

// DeviceApprovalsRemoveExceptionDetails : Removed members from device approvals
// exception list.
type DeviceApprovalsRemoveExceptionDetails struct {
}

// NewDeviceApprovalsRemoveExceptionDetails returns a new DeviceApprovalsRemoveExceptionDetails instance
func NewDeviceApprovalsRemoveExceptionDetails() *DeviceApprovalsRemoveExceptionDetails {
	s := new(DeviceApprovalsRemoveExceptionDetails)
	return s
}

// DeviceApprovalsRemoveExceptionType : has no documentation (yet)
type DeviceApprovalsRemoveExceptionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceApprovalsRemoveExceptionType returns a new DeviceApprovalsRemoveExceptionType instance
func NewDeviceApprovalsRemoveExceptionType(Description string) *DeviceApprovalsRemoveExceptionType {
	s := new(DeviceApprovalsRemoveExceptionType)
	s.Description = Description
	return s
}

// DeviceChangeIpDesktopDetails : Changed IP address associated with active
// desktop session.
type DeviceChangeIpDesktopDetails struct {
	// DeviceSessionInfo : Device's session logged information.
	DeviceSessionInfo IsDeviceSessionLogInfo `json:"device_session_info"`
}

// NewDeviceChangeIpDesktopDetails returns a new DeviceChangeIpDesktopDetails instance
func NewDeviceChangeIpDesktopDetails(DeviceSessionInfo IsDeviceSessionLogInfo) *DeviceChangeIpDesktopDetails {
	s := new(DeviceChangeIpDesktopDetails)
	s.DeviceSessionInfo = DeviceSessionInfo
	return s
}

// UnmarshalJSON deserializes into a DeviceChangeIpDesktopDetails instance
func (u *DeviceChangeIpDesktopDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// DeviceSessionInfo : Device's session logged information.
		DeviceSessionInfo json.RawMessage `json:"device_session_info"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	DeviceSessionInfo, err := IsDeviceSessionLogInfoFromJSON(w.DeviceSessionInfo)
	if err != nil {
		return err
	}
	u.DeviceSessionInfo = DeviceSessionInfo
	return nil
}

// DeviceChangeIpDesktopType : has no documentation (yet)
type DeviceChangeIpDesktopType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceChangeIpDesktopType returns a new DeviceChangeIpDesktopType instance
func NewDeviceChangeIpDesktopType(Description string) *DeviceChangeIpDesktopType {
	s := new(DeviceChangeIpDesktopType)
	s.Description = Description
	return s
}

// DeviceChangeIpMobileDetails : Changed IP address associated with active
// mobile session.
type DeviceChangeIpMobileDetails struct {
	// DeviceSessionInfo : Device's session logged information.
	DeviceSessionInfo IsDeviceSessionLogInfo `json:"device_session_info,omitempty"`
}

// NewDeviceChangeIpMobileDetails returns a new DeviceChangeIpMobileDetails instance
func NewDeviceChangeIpMobileDetails() *DeviceChangeIpMobileDetails {
	s := new(DeviceChangeIpMobileDetails)
	return s
}

// UnmarshalJSON deserializes into a DeviceChangeIpMobileDetails instance
func (u *DeviceChangeIpMobileDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// DeviceSessionInfo : Device's session logged information.
		DeviceSessionInfo json.RawMessage `json:"device_session_info,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	DeviceSessionInfo, err := IsDeviceSessionLogInfoFromJSON(w.DeviceSessionInfo)
	if err != nil {
		return err
	}
	u.DeviceSessionInfo = DeviceSessionInfo
	return nil
}

// DeviceChangeIpMobileType : has no documentation (yet)
type DeviceChangeIpMobileType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceChangeIpMobileType returns a new DeviceChangeIpMobileType instance
func NewDeviceChangeIpMobileType(Description string) *DeviceChangeIpMobileType {
	s := new(DeviceChangeIpMobileType)
	s.Description = Description
	return s
}

// DeviceChangeIpWebDetails : Changed IP address associated with active web
// session.
type DeviceChangeIpWebDetails struct {
	// UserAgent : Web browser name.
	UserAgent string `json:"user_agent"`
}

// NewDeviceChangeIpWebDetails returns a new DeviceChangeIpWebDetails instance
func NewDeviceChangeIpWebDetails(UserAgent string) *DeviceChangeIpWebDetails {
	s := new(DeviceChangeIpWebDetails)
	s.UserAgent = UserAgent
	return s
}

// DeviceChangeIpWebType : has no documentation (yet)
type DeviceChangeIpWebType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceChangeIpWebType returns a new DeviceChangeIpWebType instance
func NewDeviceChangeIpWebType(Description string) *DeviceChangeIpWebType {
	s := new(DeviceChangeIpWebType)
	s.Description = Description
	return s
}

// DeviceDeleteOnUnlinkFailDetails : Failed to delete all files from unlinked
// device.
type DeviceDeleteOnUnlinkFailDetails struct {
	// SessionInfo : Session unique id.
	SessionInfo IsSessionLogInfo `json:"session_info,omitempty"`
	// DisplayName : The device name. Might be missing due to historical data
	// gap.
	DisplayName string `json:"display_name,omitempty"`
	// NumFailures : The number of times that remote file deletion failed.
	NumFailures int64 `json:"num_failures"`
}

// NewDeviceDeleteOnUnlinkFailDetails returns a new DeviceDeleteOnUnlinkFailDetails instance
func NewDeviceDeleteOnUnlinkFailDetails(NumFailures int64) *DeviceDeleteOnUnlinkFailDetails {
	s := new(DeviceDeleteOnUnlinkFailDetails)
	s.NumFailures = NumFailures
	return s
}

// UnmarshalJSON deserializes into a DeviceDeleteOnUnlinkFailDetails instance
func (u *DeviceDeleteOnUnlinkFailDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// NumFailures : The number of times that remote file deletion failed.
		NumFailures int64 `json:"num_failures"`
		// SessionInfo : Session unique id.
		SessionInfo json.RawMessage `json:"session_info,omitempty"`
		// DisplayName : The device name. Might be missing due to historical
		// data gap.
		DisplayName string `json:"display_name,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	u.NumFailures = w.NumFailures
	SessionInfo, err := IsSessionLogInfoFromJSON(w.SessionInfo)
	if err != nil {
		return err
	}
	u.SessionInfo = SessionInfo
	u.DisplayName = w.DisplayName
	return nil
}

// DeviceDeleteOnUnlinkFailType : has no documentation (yet)
type DeviceDeleteOnUnlinkFailType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceDeleteOnUnlinkFailType returns a new DeviceDeleteOnUnlinkFailType instance
func NewDeviceDeleteOnUnlinkFailType(Description string) *DeviceDeleteOnUnlinkFailType {
	s := new(DeviceDeleteOnUnlinkFailType)
	s.Description = Description
	return s
}

// DeviceDeleteOnUnlinkSuccessDetails : Deleted all files from unlinked device.
type DeviceDeleteOnUnlinkSuccessDetails struct {
	// SessionInfo : Session unique id.
	SessionInfo IsSessionLogInfo `json:"session_info,omitempty"`
	// DisplayName : The device name. Might be missing due to historical data
	// gap.
	DisplayName string `json:"display_name,omitempty"`
}

// NewDeviceDeleteOnUnlinkSuccessDetails returns a new DeviceDeleteOnUnlinkSuccessDetails instance
func NewDeviceDeleteOnUnlinkSuccessDetails() *DeviceDeleteOnUnlinkSuccessDetails {
	s := new(DeviceDeleteOnUnlinkSuccessDetails)
	return s
}

// UnmarshalJSON deserializes into a DeviceDeleteOnUnlinkSuccessDetails instance
func (u *DeviceDeleteOnUnlinkSuccessDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SessionInfo : Session unique id.
		SessionInfo json.RawMessage `json:"session_info,omitempty"`
		// DisplayName : The device name. Might be missing due to historical
		// data gap.
		DisplayName string `json:"display_name,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	SessionInfo, err := IsSessionLogInfoFromJSON(w.SessionInfo)
	if err != nil {
		return err
	}
	u.SessionInfo = SessionInfo
	u.DisplayName = w.DisplayName
	return nil
}

// DeviceDeleteOnUnlinkSuccessType : has no documentation (yet)
type DeviceDeleteOnUnlinkSuccessType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceDeleteOnUnlinkSuccessType returns a new DeviceDeleteOnUnlinkSuccessType instance
func NewDeviceDeleteOnUnlinkSuccessType(Description string) *DeviceDeleteOnUnlinkSuccessType {
	s := new(DeviceDeleteOnUnlinkSuccessType)
	s.Description = Description
	return s
}

// DeviceLinkFailDetails : Failed to link device.
type DeviceLinkFailDetails struct {
	// IpAddress : IP address. Might be missing due to historical data gap.
	IpAddress string `json:"ip_address,omitempty"`
	// DeviceType : A description of the device used while user approval
	// blocked.
	DeviceType *DeviceType `json:"device_type"`
}

// NewDeviceLinkFailDetails returns a new DeviceLinkFailDetails instance
func NewDeviceLinkFailDetails(DeviceType *DeviceType) *DeviceLinkFailDetails {
	s := new(DeviceLinkFailDetails)
	s.DeviceType = DeviceType
	return s
}

// DeviceLinkFailType : has no documentation (yet)
type DeviceLinkFailType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceLinkFailType returns a new DeviceLinkFailType instance
func NewDeviceLinkFailType(Description string) *DeviceLinkFailType {
	s := new(DeviceLinkFailType)
	s.Description = Description
	return s
}

// DeviceLinkSuccessDetails : Linked device.
type DeviceLinkSuccessDetails struct {
	// DeviceSessionInfo : Device's session logged information.
	DeviceSessionInfo IsDeviceSessionLogInfo `json:"device_session_info,omitempty"`
}

// NewDeviceLinkSuccessDetails returns a new DeviceLinkSuccessDetails instance
func NewDeviceLinkSuccessDetails() *DeviceLinkSuccessDetails {
	s := new(DeviceLinkSuccessDetails)
	return s
}

// UnmarshalJSON deserializes into a DeviceLinkSuccessDetails instance
func (u *DeviceLinkSuccessDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// DeviceSessionInfo : Device's session logged information.
		DeviceSessionInfo json.RawMessage `json:"device_session_info,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	DeviceSessionInfo, err := IsDeviceSessionLogInfoFromJSON(w.DeviceSessionInfo)
	if err != nil {
		return err
	}
	u.DeviceSessionInfo = DeviceSessionInfo
	return nil
}

// DeviceLinkSuccessType : has no documentation (yet)
type DeviceLinkSuccessType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceLinkSuccessType returns a new DeviceLinkSuccessType instance
func NewDeviceLinkSuccessType(Description string) *DeviceLinkSuccessType {
	s := new(DeviceLinkSuccessType)
	s.Description = Description
	return s
}

// DeviceManagementDisabledDetails : Disabled device management.
type DeviceManagementDisabledDetails struct {
}

// NewDeviceManagementDisabledDetails returns a new DeviceManagementDisabledDetails instance
func NewDeviceManagementDisabledDetails() *DeviceManagementDisabledDetails {
	s := new(DeviceManagementDisabledDetails)
	return s
}

// DeviceManagementDisabledType : has no documentation (yet)
type DeviceManagementDisabledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceManagementDisabledType returns a new DeviceManagementDisabledType instance
func NewDeviceManagementDisabledType(Description string) *DeviceManagementDisabledType {
	s := new(DeviceManagementDisabledType)
	s.Description = Description
	return s
}

// DeviceManagementEnabledDetails : Enabled device management.
type DeviceManagementEnabledDetails struct {
}

// NewDeviceManagementEnabledDetails returns a new DeviceManagementEnabledDetails instance
func NewDeviceManagementEnabledDetails() *DeviceManagementEnabledDetails {
	s := new(DeviceManagementEnabledDetails)
	return s
}

// DeviceManagementEnabledType : has no documentation (yet)
type DeviceManagementEnabledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceManagementEnabledType returns a new DeviceManagementEnabledType instance
func NewDeviceManagementEnabledType(Description string) *DeviceManagementEnabledType {
	s := new(DeviceManagementEnabledType)
	s.Description = Description
	return s
}

// DeviceSyncBackupStatusChangedDetails : Enabled/disabled backup for computer.
type DeviceSyncBackupStatusChangedDetails struct {
	// DesktopDeviceSessionInfo : Device's session logged information.
	DesktopDeviceSessionInfo *DesktopDeviceSessionLogInfo `json:"desktop_device_session_info"`
	// PreviousValue : Previous status of computer backup on the device.
	PreviousValue *BackupStatus `json:"previous_value"`
	// NewValue : Next status of computer backup on the device.
	NewValue *BackupStatus `json:"new_value"`
}

// NewDeviceSyncBackupStatusChangedDetails returns a new DeviceSyncBackupStatusChangedDetails instance
func NewDeviceSyncBackupStatusChangedDetails(DesktopDeviceSessionInfo *DesktopDeviceSessionLogInfo, PreviousValue *BackupStatus, NewValue *BackupStatus) *DeviceSyncBackupStatusChangedDetails {
	s := new(DeviceSyncBackupStatusChangedDetails)
	s.DesktopDeviceSessionInfo = DesktopDeviceSessionInfo
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// DeviceSyncBackupStatusChangedType : has no documentation (yet)
type DeviceSyncBackupStatusChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceSyncBackupStatusChangedType returns a new DeviceSyncBackupStatusChangedType instance
func NewDeviceSyncBackupStatusChangedType(Description string) *DeviceSyncBackupStatusChangedType {
	s := new(DeviceSyncBackupStatusChangedType)
	s.Description = Description
	return s
}

// DeviceType : has no documentation (yet)
type DeviceType struct {
	dropbox.Tagged
}

// Valid tag values for DeviceType
const (
	DeviceTypeDesktop = "desktop"
	DeviceTypeMobile  = "mobile"
	DeviceTypeOther   = "other"
)

// DeviceUnlinkDetails : Disconnected device.
type DeviceUnlinkDetails struct {
	// SessionInfo : Session unique id.
	SessionInfo IsSessionLogInfo `json:"session_info,omitempty"`
	// DisplayName : The device name. Might be missing due to historical data
	// gap.
	DisplayName string `json:"display_name,omitempty"`
	// DeleteData : True if the user requested to delete data after device
	// unlink, false otherwise.
	DeleteData bool `json:"delete_data"`
}

// NewDeviceUnlinkDetails returns a new DeviceUnlinkDetails instance
func NewDeviceUnlinkDetails(DeleteData bool) *DeviceUnlinkDetails {
	s := new(DeviceUnlinkDetails)
	s.DeleteData = DeleteData
	return s
}

// UnmarshalJSON deserializes into a DeviceUnlinkDetails instance
func (u *DeviceUnlinkDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// DeleteData : True if the user requested to delete data after device
		// unlink, false otherwise.
		DeleteData bool `json:"delete_data"`
		// SessionInfo : Session unique id.
		SessionInfo json.RawMessage `json:"session_info,omitempty"`
		// DisplayName : The device name. Might be missing due to historical
		// data gap.
		DisplayName string `json:"display_name,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	u.DeleteData = w.DeleteData
	SessionInfo, err := IsSessionLogInfoFromJSON(w.SessionInfo)
	if err != nil {
		return err
	}
	u.SessionInfo = SessionInfo
	u.DisplayName = w.DisplayName
	return nil
}

// DeviceUnlinkPolicy : has no documentation (yet)
type DeviceUnlinkPolicy struct {
	dropbox.Tagged
}

// Valid tag values for DeviceUnlinkPolicy
const (
	DeviceUnlinkPolicyKeep   = "keep"
	DeviceUnlinkPolicyRemove = "remove"
	DeviceUnlinkPolicyOther  = "other"
)

// DeviceUnlinkType : has no documentation (yet)
type DeviceUnlinkType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDeviceUnlinkType returns a new DeviceUnlinkType instance
func NewDeviceUnlinkType(Description string) *DeviceUnlinkType {
	s := new(DeviceUnlinkType)
	s.Description = Description
	return s
}

// DirectoryRestrictionsAddMembersDetails : Added members to directory
// restrictions list.
type DirectoryRestrictionsAddMembersDetails struct {
}

// NewDirectoryRestrictionsAddMembersDetails returns a new DirectoryRestrictionsAddMembersDetails instance
func NewDirectoryRestrictionsAddMembersDetails() *DirectoryRestrictionsAddMembersDetails {
	s := new(DirectoryRestrictionsAddMembersDetails)
	return s
}

// DirectoryRestrictionsAddMembersType : has no documentation (yet)
type DirectoryRestrictionsAddMembersType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDirectoryRestrictionsAddMembersType returns a new DirectoryRestrictionsAddMembersType instance
func NewDirectoryRestrictionsAddMembersType(Description string) *DirectoryRestrictionsAddMembersType {
	s := new(DirectoryRestrictionsAddMembersType)
	s.Description = Description
	return s
}

// DirectoryRestrictionsRemoveMembersDetails : Removed members from directory
// restrictions list.
type DirectoryRestrictionsRemoveMembersDetails struct {
}

// NewDirectoryRestrictionsRemoveMembersDetails returns a new DirectoryRestrictionsRemoveMembersDetails instance
func NewDirectoryRestrictionsRemoveMembersDetails() *DirectoryRestrictionsRemoveMembersDetails {
	s := new(DirectoryRestrictionsRemoveMembersDetails)
	return s
}

// DirectoryRestrictionsRemoveMembersType : has no documentation (yet)
type DirectoryRestrictionsRemoveMembersType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDirectoryRestrictionsRemoveMembersType returns a new DirectoryRestrictionsRemoveMembersType instance
func NewDirectoryRestrictionsRemoveMembersType(Description string) *DirectoryRestrictionsRemoveMembersType {
	s := new(DirectoryRestrictionsRemoveMembersType)
	s.Description = Description
	return s
}

// DisabledDomainInvitesDetails : Disabled domain invites.
type DisabledDomainInvitesDetails struct {
}

// NewDisabledDomainInvitesDetails returns a new DisabledDomainInvitesDetails instance
func NewDisabledDomainInvitesDetails() *DisabledDomainInvitesDetails {
	s := new(DisabledDomainInvitesDetails)
	return s
}

// DisabledDomainInvitesType : has no documentation (yet)
type DisabledDomainInvitesType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDisabledDomainInvitesType returns a new DisabledDomainInvitesType instance
func NewDisabledDomainInvitesType(Description string) *DisabledDomainInvitesType {
	s := new(DisabledDomainInvitesType)
	s.Description = Description
	return s
}

// DispositionActionType : has no documentation (yet)
type DispositionActionType struct {
	dropbox.Tagged
}

// Valid tag values for DispositionActionType
const (
	DispositionActionTypeAutomaticDelete            = "automatic_delete"
	DispositionActionTypeAutomaticPermanentlyDelete = "automatic_permanently_delete"
	DispositionActionTypeOther                      = "other"
)

// DomainInvitesApproveRequestToJoinTeamDetails : Approved user's request to
// join team.
type DomainInvitesApproveRequestToJoinTeamDetails struct {
}

// NewDomainInvitesApproveRequestToJoinTeamDetails returns a new DomainInvitesApproveRequestToJoinTeamDetails instance
func NewDomainInvitesApproveRequestToJoinTeamDetails() *DomainInvitesApproveRequestToJoinTeamDetails {
	s := new(DomainInvitesApproveRequestToJoinTeamDetails)
	return s
}

// DomainInvitesApproveRequestToJoinTeamType : has no documentation (yet)
type DomainInvitesApproveRequestToJoinTeamType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDomainInvitesApproveRequestToJoinTeamType returns a new DomainInvitesApproveRequestToJoinTeamType instance
func NewDomainInvitesApproveRequestToJoinTeamType(Description string) *DomainInvitesApproveRequestToJoinTeamType {
	s := new(DomainInvitesApproveRequestToJoinTeamType)
	s.Description = Description
	return s
}

// DomainInvitesDeclineRequestToJoinTeamDetails : Declined user's request to
// join team.
type DomainInvitesDeclineRequestToJoinTeamDetails struct {
}

// NewDomainInvitesDeclineRequestToJoinTeamDetails returns a new DomainInvitesDeclineRequestToJoinTeamDetails instance
func NewDomainInvitesDeclineRequestToJoinTeamDetails() *DomainInvitesDeclineRequestToJoinTeamDetails {
	s := new(DomainInvitesDeclineRequestToJoinTeamDetails)
	return s
}

// DomainInvitesDeclineRequestToJoinTeamType : has no documentation (yet)
type DomainInvitesDeclineRequestToJoinTeamType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDomainInvitesDeclineRequestToJoinTeamType returns a new DomainInvitesDeclineRequestToJoinTeamType instance
func NewDomainInvitesDeclineRequestToJoinTeamType(Description string) *DomainInvitesDeclineRequestToJoinTeamType {
	s := new(DomainInvitesDeclineRequestToJoinTeamType)
	s.Description = Description
	return s
}

// DomainInvitesEmailExistingUsersDetails : Sent domain invites to existing
// domain accounts.
type DomainInvitesEmailExistingUsersDetails struct {
	// DomainName : Domain names.
	DomainName string `json:"domain_name"`
	// NumRecipients : Number of recipients.
	NumRecipients uint64 `json:"num_recipients"`
}

// NewDomainInvitesEmailExistingUsersDetails returns a new DomainInvitesEmailExistingUsersDetails instance
func NewDomainInvitesEmailExistingUsersDetails(DomainName string, NumRecipients uint64) *DomainInvitesEmailExistingUsersDetails {
	s := new(DomainInvitesEmailExistingUsersDetails)
	s.DomainName = DomainName
	s.NumRecipients = NumRecipients
	return s
}

// DomainInvitesEmailExistingUsersType : has no documentation (yet)
type DomainInvitesEmailExistingUsersType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDomainInvitesEmailExistingUsersType returns a new DomainInvitesEmailExistingUsersType instance
func NewDomainInvitesEmailExistingUsersType(Description string) *DomainInvitesEmailExistingUsersType {
	s := new(DomainInvitesEmailExistingUsersType)
	s.Description = Description
	return s
}

// DomainInvitesRequestToJoinTeamDetails : Requested to join team.
type DomainInvitesRequestToJoinTeamDetails struct {
}

// NewDomainInvitesRequestToJoinTeamDetails returns a new DomainInvitesRequestToJoinTeamDetails instance
func NewDomainInvitesRequestToJoinTeamDetails() *DomainInvitesRequestToJoinTeamDetails {
	s := new(DomainInvitesRequestToJoinTeamDetails)
	return s
}

// DomainInvitesRequestToJoinTeamType : has no documentation (yet)
type DomainInvitesRequestToJoinTeamType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDomainInvitesRequestToJoinTeamType returns a new DomainInvitesRequestToJoinTeamType instance
func NewDomainInvitesRequestToJoinTeamType(Description string) *DomainInvitesRequestToJoinTeamType {
	s := new(DomainInvitesRequestToJoinTeamType)
	s.Description = Description
	return s
}

// DomainInvitesSetInviteNewUserPrefToNoDetails : Disabled "Automatically invite
// new users".
type DomainInvitesSetInviteNewUserPrefToNoDetails struct {
}

// NewDomainInvitesSetInviteNewUserPrefToNoDetails returns a new DomainInvitesSetInviteNewUserPrefToNoDetails instance
func NewDomainInvitesSetInviteNewUserPrefToNoDetails() *DomainInvitesSetInviteNewUserPrefToNoDetails {
	s := new(DomainInvitesSetInviteNewUserPrefToNoDetails)
	return s
}

// DomainInvitesSetInviteNewUserPrefToNoType : has no documentation (yet)
type DomainInvitesSetInviteNewUserPrefToNoType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDomainInvitesSetInviteNewUserPrefToNoType returns a new DomainInvitesSetInviteNewUserPrefToNoType instance
func NewDomainInvitesSetInviteNewUserPrefToNoType(Description string) *DomainInvitesSetInviteNewUserPrefToNoType {
	s := new(DomainInvitesSetInviteNewUserPrefToNoType)
	s.Description = Description
	return s
}

// DomainInvitesSetInviteNewUserPrefToYesDetails : Enabled "Automatically invite
// new users".
type DomainInvitesSetInviteNewUserPrefToYesDetails struct {
}

// NewDomainInvitesSetInviteNewUserPrefToYesDetails returns a new DomainInvitesSetInviteNewUserPrefToYesDetails instance
func NewDomainInvitesSetInviteNewUserPrefToYesDetails() *DomainInvitesSetInviteNewUserPrefToYesDetails {
	s := new(DomainInvitesSetInviteNewUserPrefToYesDetails)
	return s
}

// DomainInvitesSetInviteNewUserPrefToYesType : has no documentation (yet)
type DomainInvitesSetInviteNewUserPrefToYesType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDomainInvitesSetInviteNewUserPrefToYesType returns a new DomainInvitesSetInviteNewUserPrefToYesType instance
func NewDomainInvitesSetInviteNewUserPrefToYesType(Description string) *DomainInvitesSetInviteNewUserPrefToYesType {
	s := new(DomainInvitesSetInviteNewUserPrefToYesType)
	s.Description = Description
	return s
}

// DomainVerificationAddDomainFailDetails : Failed to verify team domain.
type DomainVerificationAddDomainFailDetails struct {
	// DomainName : Domain name.
	DomainName string `json:"domain_name"`
	// VerificationMethod : Domain name verification method. Might be missing
	// due to historical data gap.
	VerificationMethod string `json:"verification_method,omitempty"`
}

// NewDomainVerificationAddDomainFailDetails returns a new DomainVerificationAddDomainFailDetails instance
func NewDomainVerificationAddDomainFailDetails(DomainName string) *DomainVerificationAddDomainFailDetails {
	s := new(DomainVerificationAddDomainFailDetails)
	s.DomainName = DomainName
	return s
}

// DomainVerificationAddDomainFailType : has no documentation (yet)
type DomainVerificationAddDomainFailType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDomainVerificationAddDomainFailType returns a new DomainVerificationAddDomainFailType instance
func NewDomainVerificationAddDomainFailType(Description string) *DomainVerificationAddDomainFailType {
	s := new(DomainVerificationAddDomainFailType)
	s.Description = Description
	return s
}

// DomainVerificationAddDomainSuccessDetails : Verified team domain.
type DomainVerificationAddDomainSuccessDetails struct {
	// DomainNames : Domain names.
	DomainNames []string `json:"domain_names"`
	// VerificationMethod : Domain name verification method. Might be missing
	// due to historical data gap.
	VerificationMethod string `json:"verification_method,omitempty"`
}

// NewDomainVerificationAddDomainSuccessDetails returns a new DomainVerificationAddDomainSuccessDetails instance
func NewDomainVerificationAddDomainSuccessDetails(DomainNames []string) *DomainVerificationAddDomainSuccessDetails {
	s := new(DomainVerificationAddDomainSuccessDetails)
	s.DomainNames = DomainNames
	return s
}

// DomainVerificationAddDomainSuccessType : has no documentation (yet)
type DomainVerificationAddDomainSuccessType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDomainVerificationAddDomainSuccessType returns a new DomainVerificationAddDomainSuccessType instance
func NewDomainVerificationAddDomainSuccessType(Description string) *DomainVerificationAddDomainSuccessType {
	s := new(DomainVerificationAddDomainSuccessType)
	s.Description = Description
	return s
}

// DomainVerificationRemoveDomainDetails : Removed domain from list of verified
// team domains.
type DomainVerificationRemoveDomainDetails struct {
	// DomainNames : Domain names.
	DomainNames []string `json:"domain_names"`
}

// NewDomainVerificationRemoveDomainDetails returns a new DomainVerificationRemoveDomainDetails instance
func NewDomainVerificationRemoveDomainDetails(DomainNames []string) *DomainVerificationRemoveDomainDetails {
	s := new(DomainVerificationRemoveDomainDetails)
	s.DomainNames = DomainNames
	return s
}

// DomainVerificationRemoveDomainType : has no documentation (yet)
type DomainVerificationRemoveDomainType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDomainVerificationRemoveDomainType returns a new DomainVerificationRemoveDomainType instance
func NewDomainVerificationRemoveDomainType(Description string) *DomainVerificationRemoveDomainType {
	s := new(DomainVerificationRemoveDomainType)
	s.Description = Description
	return s
}

// DownloadPolicyType : Shared content downloads policy
type DownloadPolicyType struct {
	dropbox.Tagged
}

// Valid tag values for DownloadPolicyType
const (
	DownloadPolicyTypeAllow    = "allow"
	DownloadPolicyTypeDisallow = "disallow"
	DownloadPolicyTypeOther    = "other"
)

// DropboxPasswordsExportedDetails : Exported passwords.
type DropboxPasswordsExportedDetails struct {
	// Platform : The platform the device runs export.
	Platform string `json:"platform"`
}

// NewDropboxPasswordsExportedDetails returns a new DropboxPasswordsExportedDetails instance
func NewDropboxPasswordsExportedDetails(Platform string) *DropboxPasswordsExportedDetails {
	s := new(DropboxPasswordsExportedDetails)
	s.Platform = Platform
	return s
}

// DropboxPasswordsExportedType : has no documentation (yet)
type DropboxPasswordsExportedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDropboxPasswordsExportedType returns a new DropboxPasswordsExportedType instance
func NewDropboxPasswordsExportedType(Description string) *DropboxPasswordsExportedType {
	s := new(DropboxPasswordsExportedType)
	s.Description = Description
	return s
}

// DropboxPasswordsNewDeviceEnrolledDetails : Enrolled new Dropbox Passwords
// device.
type DropboxPasswordsNewDeviceEnrolledDetails struct {
	// IsFirstDevice : Whether it's a first device enrolled.
	IsFirstDevice bool `json:"is_first_device"`
	// Platform : The platform the device is enrolled.
	Platform string `json:"platform"`
}

// NewDropboxPasswordsNewDeviceEnrolledDetails returns a new DropboxPasswordsNewDeviceEnrolledDetails instance
func NewDropboxPasswordsNewDeviceEnrolledDetails(IsFirstDevice bool, Platform string) *DropboxPasswordsNewDeviceEnrolledDetails {
	s := new(DropboxPasswordsNewDeviceEnrolledDetails)
	s.IsFirstDevice = IsFirstDevice
	s.Platform = Platform
	return s
}

// DropboxPasswordsNewDeviceEnrolledType : has no documentation (yet)
type DropboxPasswordsNewDeviceEnrolledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDropboxPasswordsNewDeviceEnrolledType returns a new DropboxPasswordsNewDeviceEnrolledType instance
func NewDropboxPasswordsNewDeviceEnrolledType(Description string) *DropboxPasswordsNewDeviceEnrolledType {
	s := new(DropboxPasswordsNewDeviceEnrolledType)
	s.Description = Description
	return s
}

// DropboxPasswordsPolicy : Policy for deciding whether team users can use
// Dropbox Passwords
type DropboxPasswordsPolicy struct {
	dropbox.Tagged
}

// Valid tag values for DropboxPasswordsPolicy
const (
	DropboxPasswordsPolicyDefault  = "default"
	DropboxPasswordsPolicyDisabled = "disabled"
	DropboxPasswordsPolicyEnabled  = "enabled"
	DropboxPasswordsPolicyOther    = "other"
)

// DropboxPasswordsPolicyChangedDetails : Changed Dropbox Passwords policy for
// team.
type DropboxPasswordsPolicyChangedDetails struct {
	// NewValue : To.
	NewValue *DropboxPasswordsPolicy `json:"new_value"`
	// PreviousValue : From.
	PreviousValue *DropboxPasswordsPolicy `json:"previous_value"`
}

// NewDropboxPasswordsPolicyChangedDetails returns a new DropboxPasswordsPolicyChangedDetails instance
func NewDropboxPasswordsPolicyChangedDetails(NewValue *DropboxPasswordsPolicy, PreviousValue *DropboxPasswordsPolicy) *DropboxPasswordsPolicyChangedDetails {
	s := new(DropboxPasswordsPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// DropboxPasswordsPolicyChangedType : has no documentation (yet)
type DropboxPasswordsPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewDropboxPasswordsPolicyChangedType returns a new DropboxPasswordsPolicyChangedType instance
func NewDropboxPasswordsPolicyChangedType(Description string) *DropboxPasswordsPolicyChangedType {
	s := new(DropboxPasswordsPolicyChangedType)
	s.Description = Description
	return s
}

// DurationLogInfo : Represents a time duration: unit and amount
type DurationLogInfo struct {
	// Unit : Time unit.
	Unit *TimeUnit `json:"unit"`
	// Amount : Amount of time.
	Amount uint64 `json:"amount"`
}

// NewDurationLogInfo returns a new DurationLogInfo instance
func NewDurationLogInfo(Unit *TimeUnit, Amount uint64) *DurationLogInfo {
	s := new(DurationLogInfo)
	s.Unit = Unit
	s.Amount = Amount
	return s
}

// EmailIngestPolicy : Policy for deciding whether a team can use Email to
// Dropbox feature
type EmailIngestPolicy struct {
	dropbox.Tagged
}

// Valid tag values for EmailIngestPolicy
const (
	EmailIngestPolicyDisabled = "disabled"
	EmailIngestPolicyEnabled  = "enabled"
	EmailIngestPolicyOther    = "other"
)

// EmailIngestPolicyChangedDetails : Changed email to Dropbox policy for team.
type EmailIngestPolicyChangedDetails struct {
	// NewValue : To.
	NewValue *EmailIngestPolicy `json:"new_value"`
	// PreviousValue : From.
	PreviousValue *EmailIngestPolicy `json:"previous_value"`
}

// NewEmailIngestPolicyChangedDetails returns a new EmailIngestPolicyChangedDetails instance
func NewEmailIngestPolicyChangedDetails(NewValue *EmailIngestPolicy, PreviousValue *EmailIngestPolicy) *EmailIngestPolicyChangedDetails {
	s := new(EmailIngestPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// EmailIngestPolicyChangedType : has no documentation (yet)
type EmailIngestPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEmailIngestPolicyChangedType returns a new EmailIngestPolicyChangedType instance
func NewEmailIngestPolicyChangedType(Description string) *EmailIngestPolicyChangedType {
	s := new(EmailIngestPolicyChangedType)
	s.Description = Description
	return s
}

// EmailIngestReceiveFileDetails : Received files via Email to Dropbox.
type EmailIngestReceiveFileDetails struct {
	// InboxName : Inbox name.
	InboxName string `json:"inbox_name"`
	// AttachmentNames : Submitted file names.
	AttachmentNames []string `json:"attachment_names"`
	// Subject : Subject of the email.
	Subject string `json:"subject,omitempty"`
	// FromName : The name as provided by the submitter.
	FromName string `json:"from_name,omitempty"`
	// FromEmail : The email as provided by the submitter.
	FromEmail string `json:"from_email,omitempty"`
}

// NewEmailIngestReceiveFileDetails returns a new EmailIngestReceiveFileDetails instance
func NewEmailIngestReceiveFileDetails(InboxName string, AttachmentNames []string) *EmailIngestReceiveFileDetails {
	s := new(EmailIngestReceiveFileDetails)
	s.InboxName = InboxName
	s.AttachmentNames = AttachmentNames
	return s
}

// EmailIngestReceiveFileType : has no documentation (yet)
type EmailIngestReceiveFileType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEmailIngestReceiveFileType returns a new EmailIngestReceiveFileType instance
func NewEmailIngestReceiveFileType(Description string) *EmailIngestReceiveFileType {
	s := new(EmailIngestReceiveFileType)
	s.Description = Description
	return s
}

// EmmAddExceptionDetails : Added members to EMM exception list.
type EmmAddExceptionDetails struct {
}

// NewEmmAddExceptionDetails returns a new EmmAddExceptionDetails instance
func NewEmmAddExceptionDetails() *EmmAddExceptionDetails {
	s := new(EmmAddExceptionDetails)
	return s
}

// EmmAddExceptionType : has no documentation (yet)
type EmmAddExceptionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEmmAddExceptionType returns a new EmmAddExceptionType instance
func NewEmmAddExceptionType(Description string) *EmmAddExceptionType {
	s := new(EmmAddExceptionType)
	s.Description = Description
	return s
}

// EmmChangePolicyDetails : Enabled/disabled enterprise mobility management for
// members.
type EmmChangePolicyDetails struct {
	// NewValue : New enterprise mobility management policy.
	NewValue *team_policies.EmmState `json:"new_value"`
	// PreviousValue : Previous enterprise mobility management policy. Might be
	// missing due to historical data gap.
	PreviousValue *team_policies.EmmState `json:"previous_value,omitempty"`
}

// NewEmmChangePolicyDetails returns a new EmmChangePolicyDetails instance
func NewEmmChangePolicyDetails(NewValue *team_policies.EmmState) *EmmChangePolicyDetails {
	s := new(EmmChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// EmmChangePolicyType : has no documentation (yet)
type EmmChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEmmChangePolicyType returns a new EmmChangePolicyType instance
func NewEmmChangePolicyType(Description string) *EmmChangePolicyType {
	s := new(EmmChangePolicyType)
	s.Description = Description
	return s
}

// EmmCreateExceptionsReportDetails : Created EMM-excluded users report.
type EmmCreateExceptionsReportDetails struct {
}

// NewEmmCreateExceptionsReportDetails returns a new EmmCreateExceptionsReportDetails instance
func NewEmmCreateExceptionsReportDetails() *EmmCreateExceptionsReportDetails {
	s := new(EmmCreateExceptionsReportDetails)
	return s
}

// EmmCreateExceptionsReportType : has no documentation (yet)
type EmmCreateExceptionsReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEmmCreateExceptionsReportType returns a new EmmCreateExceptionsReportType instance
func NewEmmCreateExceptionsReportType(Description string) *EmmCreateExceptionsReportType {
	s := new(EmmCreateExceptionsReportType)
	s.Description = Description
	return s
}

// EmmCreateUsageReportDetails : Created EMM mobile app usage report.
type EmmCreateUsageReportDetails struct {
}

// NewEmmCreateUsageReportDetails returns a new EmmCreateUsageReportDetails instance
func NewEmmCreateUsageReportDetails() *EmmCreateUsageReportDetails {
	s := new(EmmCreateUsageReportDetails)
	return s
}

// EmmCreateUsageReportType : has no documentation (yet)
type EmmCreateUsageReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEmmCreateUsageReportType returns a new EmmCreateUsageReportType instance
func NewEmmCreateUsageReportType(Description string) *EmmCreateUsageReportType {
	s := new(EmmCreateUsageReportType)
	s.Description = Description
	return s
}

// EmmErrorDetails : Failed to sign in via EMM.
type EmmErrorDetails struct {
	// ErrorDetails : Error details.
	ErrorDetails *FailureDetailsLogInfo `json:"error_details"`
}

// NewEmmErrorDetails returns a new EmmErrorDetails instance
func NewEmmErrorDetails(ErrorDetails *FailureDetailsLogInfo) *EmmErrorDetails {
	s := new(EmmErrorDetails)
	s.ErrorDetails = ErrorDetails
	return s
}

// EmmErrorType : has no documentation (yet)
type EmmErrorType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEmmErrorType returns a new EmmErrorType instance
func NewEmmErrorType(Description string) *EmmErrorType {
	s := new(EmmErrorType)
	s.Description = Description
	return s
}

// EmmRefreshAuthTokenDetails : Refreshed auth token used for setting up EMM.
type EmmRefreshAuthTokenDetails struct {
}

// NewEmmRefreshAuthTokenDetails returns a new EmmRefreshAuthTokenDetails instance
func NewEmmRefreshAuthTokenDetails() *EmmRefreshAuthTokenDetails {
	s := new(EmmRefreshAuthTokenDetails)
	return s
}

// EmmRefreshAuthTokenType : has no documentation (yet)
type EmmRefreshAuthTokenType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEmmRefreshAuthTokenType returns a new EmmRefreshAuthTokenType instance
func NewEmmRefreshAuthTokenType(Description string) *EmmRefreshAuthTokenType {
	s := new(EmmRefreshAuthTokenType)
	s.Description = Description
	return s
}

// EmmRemoveExceptionDetails : Removed members from EMM exception list.
type EmmRemoveExceptionDetails struct {
}

// NewEmmRemoveExceptionDetails returns a new EmmRemoveExceptionDetails instance
func NewEmmRemoveExceptionDetails() *EmmRemoveExceptionDetails {
	s := new(EmmRemoveExceptionDetails)
	return s
}

// EmmRemoveExceptionType : has no documentation (yet)
type EmmRemoveExceptionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEmmRemoveExceptionType returns a new EmmRemoveExceptionType instance
func NewEmmRemoveExceptionType(Description string) *EmmRemoveExceptionType {
	s := new(EmmRemoveExceptionType)
	s.Description = Description
	return s
}

// EnabledDomainInvitesDetails : Enabled domain invites.
type EnabledDomainInvitesDetails struct {
}

// NewEnabledDomainInvitesDetails returns a new EnabledDomainInvitesDetails instance
func NewEnabledDomainInvitesDetails() *EnabledDomainInvitesDetails {
	s := new(EnabledDomainInvitesDetails)
	return s
}

// EnabledDomainInvitesType : has no documentation (yet)
type EnabledDomainInvitesType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEnabledDomainInvitesType returns a new EnabledDomainInvitesType instance
func NewEnabledDomainInvitesType(Description string) *EnabledDomainInvitesType {
	s := new(EnabledDomainInvitesType)
	s.Description = Description
	return s
}

// EndedEnterpriseAdminSessionDeprecatedDetails : Ended enterprise admin
// session.
type EndedEnterpriseAdminSessionDeprecatedDetails struct {
	// FederationExtraDetails : More information about the organization or team.
	FederationExtraDetails *FedExtraDetails `json:"federation_extra_details"`
}

// NewEndedEnterpriseAdminSessionDeprecatedDetails returns a new EndedEnterpriseAdminSessionDeprecatedDetails instance
func NewEndedEnterpriseAdminSessionDeprecatedDetails(FederationExtraDetails *FedExtraDetails) *EndedEnterpriseAdminSessionDeprecatedDetails {
	s := new(EndedEnterpriseAdminSessionDeprecatedDetails)
	s.FederationExtraDetails = FederationExtraDetails
	return s
}

// EndedEnterpriseAdminSessionDeprecatedType : has no documentation (yet)
type EndedEnterpriseAdminSessionDeprecatedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEndedEnterpriseAdminSessionDeprecatedType returns a new EndedEnterpriseAdminSessionDeprecatedType instance
func NewEndedEnterpriseAdminSessionDeprecatedType(Description string) *EndedEnterpriseAdminSessionDeprecatedType {
	s := new(EndedEnterpriseAdminSessionDeprecatedType)
	s.Description = Description
	return s
}

// EndedEnterpriseAdminSessionDetails : Ended enterprise admin session.
type EndedEnterpriseAdminSessionDetails struct {
}

// NewEndedEnterpriseAdminSessionDetails returns a new EndedEnterpriseAdminSessionDetails instance
func NewEndedEnterpriseAdminSessionDetails() *EndedEnterpriseAdminSessionDetails {
	s := new(EndedEnterpriseAdminSessionDetails)
	return s
}

// EndedEnterpriseAdminSessionType : has no documentation (yet)
type EndedEnterpriseAdminSessionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEndedEnterpriseAdminSessionType returns a new EndedEnterpriseAdminSessionType instance
func NewEndedEnterpriseAdminSessionType(Description string) *EndedEnterpriseAdminSessionType {
	s := new(EndedEnterpriseAdminSessionType)
	s.Description = Description
	return s
}

// EnforceLinkPasswordPolicy : Policy for deciding whether password must be
// enforced when an externally shared link is updated
type EnforceLinkPasswordPolicy struct {
	dropbox.Tagged
}

// Valid tag values for EnforceLinkPasswordPolicy
const (
	EnforceLinkPasswordPolicyOptional = "optional"
	EnforceLinkPasswordPolicyRequired = "required"
	EnforceLinkPasswordPolicyOther    = "other"
)

// EnterpriseSettingsLockingDetails : Changed who can update a setting.
type EnterpriseSettingsLockingDetails struct {
	// TeamName : The secondary team name.
	TeamName string `json:"team_name"`
	// SettingsPageName : Settings page name.
	SettingsPageName string `json:"settings_page_name"`
	// PreviousSettingsPageLockingState : Previous locked settings page state.
	PreviousSettingsPageLockingState string `json:"previous_settings_page_locking_state"`
	// NewSettingsPageLockingState : New locked settings page state.
	NewSettingsPageLockingState string `json:"new_settings_page_locking_state"`
}

// NewEnterpriseSettingsLockingDetails returns a new EnterpriseSettingsLockingDetails instance
func NewEnterpriseSettingsLockingDetails(TeamName string, SettingsPageName string, PreviousSettingsPageLockingState string, NewSettingsPageLockingState string) *EnterpriseSettingsLockingDetails {
	s := new(EnterpriseSettingsLockingDetails)
	s.TeamName = TeamName
	s.SettingsPageName = SettingsPageName
	s.PreviousSettingsPageLockingState = PreviousSettingsPageLockingState
	s.NewSettingsPageLockingState = NewSettingsPageLockingState
	return s
}

// EnterpriseSettingsLockingType : has no documentation (yet)
type EnterpriseSettingsLockingType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewEnterpriseSettingsLockingType returns a new EnterpriseSettingsLockingType instance
func NewEnterpriseSettingsLockingType(Description string) *EnterpriseSettingsLockingType {
	s := new(EnterpriseSettingsLockingType)
	s.Description = Description
	return s
}

// EventCategory : Category of events in event audit log.
type EventCategory struct {
	dropbox.Tagged
}

// Valid tag values for EventCategory
const (
	EventCategoryAdminAlerting  = "admin_alerting"
	EventCategoryApps           = "apps"
	EventCategoryComments       = "comments"
	EventCategoryDataGovernance = "data_governance"
	EventCategoryDevices        = "devices"
	EventCategoryDomains        = "domains"
	EventCategoryFileOperations = "file_operations"
	EventCategoryFileRequests   = "file_requests"
	EventCategoryGroups         = "groups"
	EventCategoryLogins         = "logins"
	EventCategoryMembers        = "members"
	EventCategoryPaper          = "paper"
	EventCategoryPasswords      = "passwords"
	EventCategoryReports        = "reports"
	EventCategorySharing        = "sharing"
	EventCategoryShowcase       = "showcase"
	EventCategorySso            = "sso"
	EventCategoryTeamFolders    = "team_folders"
	EventCategoryTeamPolicies   = "team_policies"
	EventCategoryTeamProfile    = "team_profile"
	EventCategoryTfa            = "tfa"
	EventCategoryTrustedTeams   = "trusted_teams"
	EventCategoryOther          = "other"
)

// EventDetails : Additional fields depending on the event type.
type EventDetails struct {
	dropbox.Tagged
	// AdminAlertingAlertStateChangedDetails : has no documentation (yet)
	AdminAlertingAlertStateChangedDetails *AdminAlertingAlertStateChangedDetails `json:"admin_alerting_alert_state_changed_details,omitempty"`
	// AdminAlertingChangedAlertConfigDetails : has no documentation (yet)
	AdminAlertingChangedAlertConfigDetails *AdminAlertingChangedAlertConfigDetails `json:"admin_alerting_changed_alert_config_details,omitempty"`
	// AdminAlertingTriggeredAlertDetails : has no documentation (yet)
	AdminAlertingTriggeredAlertDetails *AdminAlertingTriggeredAlertDetails `json:"admin_alerting_triggered_alert_details,omitempty"`
	// AppBlockedByPermissionsDetails : has no documentation (yet)
	AppBlockedByPermissionsDetails *AppBlockedByPermissionsDetails `json:"app_blocked_by_permissions_details,omitempty"`
	// AppLinkTeamDetails : has no documentation (yet)
	AppLinkTeamDetails *AppLinkTeamDetails `json:"app_link_team_details,omitempty"`
	// AppLinkUserDetails : has no documentation (yet)
	AppLinkUserDetails *AppLinkUserDetails `json:"app_link_user_details,omitempty"`
	// AppUnlinkTeamDetails : has no documentation (yet)
	AppUnlinkTeamDetails *AppUnlinkTeamDetails `json:"app_unlink_team_details,omitempty"`
	// AppUnlinkUserDetails : has no documentation (yet)
	AppUnlinkUserDetails *AppUnlinkUserDetails `json:"app_unlink_user_details,omitempty"`
	// IntegrationConnectedDetails : has no documentation (yet)
	IntegrationConnectedDetails *IntegrationConnectedDetails `json:"integration_connected_details,omitempty"`
	// IntegrationDisconnectedDetails : has no documentation (yet)
	IntegrationDisconnectedDetails *IntegrationDisconnectedDetails `json:"integration_disconnected_details,omitempty"`
	// FileAddCommentDetails : has no documentation (yet)
	FileAddCommentDetails *FileAddCommentDetails `json:"file_add_comment_details,omitempty"`
	// FileChangeCommentSubscriptionDetails : has no documentation (yet)
	FileChangeCommentSubscriptionDetails *FileChangeCommentSubscriptionDetails `json:"file_change_comment_subscription_details,omitempty"`
	// FileDeleteCommentDetails : has no documentation (yet)
	FileDeleteCommentDetails *FileDeleteCommentDetails `json:"file_delete_comment_details,omitempty"`
	// FileEditCommentDetails : has no documentation (yet)
	FileEditCommentDetails *FileEditCommentDetails `json:"file_edit_comment_details,omitempty"`
	// FileLikeCommentDetails : has no documentation (yet)
	FileLikeCommentDetails *FileLikeCommentDetails `json:"file_like_comment_details,omitempty"`
	// FileResolveCommentDetails : has no documentation (yet)
	FileResolveCommentDetails *FileResolveCommentDetails `json:"file_resolve_comment_details,omitempty"`
	// FileUnlikeCommentDetails : has no documentation (yet)
	FileUnlikeCommentDetails *FileUnlikeCommentDetails `json:"file_unlike_comment_details,omitempty"`
	// FileUnresolveCommentDetails : has no documentation (yet)
	FileUnresolveCommentDetails *FileUnresolveCommentDetails `json:"file_unresolve_comment_details,omitempty"`
	// GovernancePolicyAddFoldersDetails : has no documentation (yet)
	GovernancePolicyAddFoldersDetails *GovernancePolicyAddFoldersDetails `json:"governance_policy_add_folders_details,omitempty"`
	// GovernancePolicyAddFolderFailedDetails : has no documentation (yet)
	GovernancePolicyAddFolderFailedDetails *GovernancePolicyAddFolderFailedDetails `json:"governance_policy_add_folder_failed_details,omitempty"`
	// GovernancePolicyContentDisposedDetails : has no documentation (yet)
	GovernancePolicyContentDisposedDetails *GovernancePolicyContentDisposedDetails `json:"governance_policy_content_disposed_details,omitempty"`
	// GovernancePolicyCreateDetails : has no documentation (yet)
	GovernancePolicyCreateDetails *GovernancePolicyCreateDetails `json:"governance_policy_create_details,omitempty"`
	// GovernancePolicyDeleteDetails : has no documentation (yet)
	GovernancePolicyDeleteDetails *GovernancePolicyDeleteDetails `json:"governance_policy_delete_details,omitempty"`
	// GovernancePolicyEditDetailsDetails : has no documentation (yet)
	GovernancePolicyEditDetailsDetails *GovernancePolicyEditDetailsDetails `json:"governance_policy_edit_details_details,omitempty"`
	// GovernancePolicyEditDurationDetails : has no documentation (yet)
	GovernancePolicyEditDurationDetails *GovernancePolicyEditDurationDetails `json:"governance_policy_edit_duration_details,omitempty"`
	// GovernancePolicyExportCreatedDetails : has no documentation (yet)
	GovernancePolicyExportCreatedDetails *GovernancePolicyExportCreatedDetails `json:"governance_policy_export_created_details,omitempty"`
	// GovernancePolicyExportRemovedDetails : has no documentation (yet)
	GovernancePolicyExportRemovedDetails *GovernancePolicyExportRemovedDetails `json:"governance_policy_export_removed_details,omitempty"`
	// GovernancePolicyRemoveFoldersDetails : has no documentation (yet)
	GovernancePolicyRemoveFoldersDetails *GovernancePolicyRemoveFoldersDetails `json:"governance_policy_remove_folders_details,omitempty"`
	// GovernancePolicyReportCreatedDetails : has no documentation (yet)
	GovernancePolicyReportCreatedDetails *GovernancePolicyReportCreatedDetails `json:"governance_policy_report_created_details,omitempty"`
	// GovernancePolicyZipPartDownloadedDetails : has no documentation (yet)
	GovernancePolicyZipPartDownloadedDetails *GovernancePolicyZipPartDownloadedDetails `json:"governance_policy_zip_part_downloaded_details,omitempty"`
	// LegalHoldsActivateAHoldDetails : has no documentation (yet)
	LegalHoldsActivateAHoldDetails *LegalHoldsActivateAHoldDetails `json:"legal_holds_activate_a_hold_details,omitempty"`
	// LegalHoldsAddMembersDetails : has no documentation (yet)
	LegalHoldsAddMembersDetails *LegalHoldsAddMembersDetails `json:"legal_holds_add_members_details,omitempty"`
	// LegalHoldsChangeHoldDetailsDetails : has no documentation (yet)
	LegalHoldsChangeHoldDetailsDetails *LegalHoldsChangeHoldDetailsDetails `json:"legal_holds_change_hold_details_details,omitempty"`
	// LegalHoldsChangeHoldNameDetails : has no documentation (yet)
	LegalHoldsChangeHoldNameDetails *LegalHoldsChangeHoldNameDetails `json:"legal_holds_change_hold_name_details,omitempty"`
	// LegalHoldsExportAHoldDetails : has no documentation (yet)
	LegalHoldsExportAHoldDetails *LegalHoldsExportAHoldDetails `json:"legal_holds_export_a_hold_details,omitempty"`
	// LegalHoldsExportCancelledDetails : has no documentation (yet)
	LegalHoldsExportCancelledDetails *LegalHoldsExportCancelledDetails `json:"legal_holds_export_cancelled_details,omitempty"`
	// LegalHoldsExportDownloadedDetails : has no documentation (yet)
	LegalHoldsExportDownloadedDetails *LegalHoldsExportDownloadedDetails `json:"legal_holds_export_downloaded_details,omitempty"`
	// LegalHoldsExportRemovedDetails : has no documentation (yet)
	LegalHoldsExportRemovedDetails *LegalHoldsExportRemovedDetails `json:"legal_holds_export_removed_details,omitempty"`
	// LegalHoldsReleaseAHoldDetails : has no documentation (yet)
	LegalHoldsReleaseAHoldDetails *LegalHoldsReleaseAHoldDetails `json:"legal_holds_release_a_hold_details,omitempty"`
	// LegalHoldsRemoveMembersDetails : has no documentation (yet)
	LegalHoldsRemoveMembersDetails *LegalHoldsRemoveMembersDetails `json:"legal_holds_remove_members_details,omitempty"`
	// LegalHoldsReportAHoldDetails : has no documentation (yet)
	LegalHoldsReportAHoldDetails *LegalHoldsReportAHoldDetails `json:"legal_holds_report_a_hold_details,omitempty"`
	// DeviceChangeIpDesktopDetails : has no documentation (yet)
	DeviceChangeIpDesktopDetails *DeviceChangeIpDesktopDetails `json:"device_change_ip_desktop_details,omitempty"`
	// DeviceChangeIpMobileDetails : has no documentation (yet)
	DeviceChangeIpMobileDetails *DeviceChangeIpMobileDetails `json:"device_change_ip_mobile_details,omitempty"`
	// DeviceChangeIpWebDetails : has no documentation (yet)
	DeviceChangeIpWebDetails *DeviceChangeIpWebDetails `json:"device_change_ip_web_details,omitempty"`
	// DeviceDeleteOnUnlinkFailDetails : has no documentation (yet)
	DeviceDeleteOnUnlinkFailDetails *DeviceDeleteOnUnlinkFailDetails `json:"device_delete_on_unlink_fail_details,omitempty"`
	// DeviceDeleteOnUnlinkSuccessDetails : has no documentation (yet)
	DeviceDeleteOnUnlinkSuccessDetails *DeviceDeleteOnUnlinkSuccessDetails `json:"device_delete_on_unlink_success_details,omitempty"`
	// DeviceLinkFailDetails : has no documentation (yet)
	DeviceLinkFailDetails *DeviceLinkFailDetails `json:"device_link_fail_details,omitempty"`
	// DeviceLinkSuccessDetails : has no documentation (yet)
	DeviceLinkSuccessDetails *DeviceLinkSuccessDetails `json:"device_link_success_details,omitempty"`
	// DeviceManagementDisabledDetails : has no documentation (yet)
	DeviceManagementDisabledDetails *DeviceManagementDisabledDetails `json:"device_management_disabled_details,omitempty"`
	// DeviceManagementEnabledDetails : has no documentation (yet)
	DeviceManagementEnabledDetails *DeviceManagementEnabledDetails `json:"device_management_enabled_details,omitempty"`
	// DeviceSyncBackupStatusChangedDetails : has no documentation (yet)
	DeviceSyncBackupStatusChangedDetails *DeviceSyncBackupStatusChangedDetails `json:"device_sync_backup_status_changed_details,omitempty"`
	// DeviceUnlinkDetails : has no documentation (yet)
	DeviceUnlinkDetails *DeviceUnlinkDetails `json:"device_unlink_details,omitempty"`
	// DropboxPasswordsExportedDetails : has no documentation (yet)
	DropboxPasswordsExportedDetails *DropboxPasswordsExportedDetails `json:"dropbox_passwords_exported_details,omitempty"`
	// DropboxPasswordsNewDeviceEnrolledDetails : has no documentation (yet)
	DropboxPasswordsNewDeviceEnrolledDetails *DropboxPasswordsNewDeviceEnrolledDetails `json:"dropbox_passwords_new_device_enrolled_details,omitempty"`
	// EmmRefreshAuthTokenDetails : has no documentation (yet)
	EmmRefreshAuthTokenDetails *EmmRefreshAuthTokenDetails `json:"emm_refresh_auth_token_details,omitempty"`
	// ExternalDriveBackupEligibilityStatusCheckedDetails : has no documentation
	// (yet)
	ExternalDriveBackupEligibilityStatusCheckedDetails *ExternalDriveBackupEligibilityStatusCheckedDetails `json:"external_drive_backup_eligibility_status_checked_details,omitempty"`
	// ExternalDriveBackupStatusChangedDetails : has no documentation (yet)
	ExternalDriveBackupStatusChangedDetails *ExternalDriveBackupStatusChangedDetails `json:"external_drive_backup_status_changed_details,omitempty"`
	// AccountCaptureChangeAvailabilityDetails : has no documentation (yet)
	AccountCaptureChangeAvailabilityDetails *AccountCaptureChangeAvailabilityDetails `json:"account_capture_change_availability_details,omitempty"`
	// AccountCaptureMigrateAccountDetails : has no documentation (yet)
	AccountCaptureMigrateAccountDetails *AccountCaptureMigrateAccountDetails `json:"account_capture_migrate_account_details,omitempty"`
	// AccountCaptureNotificationEmailsSentDetails : has no documentation (yet)
	AccountCaptureNotificationEmailsSentDetails *AccountCaptureNotificationEmailsSentDetails `json:"account_capture_notification_emails_sent_details,omitempty"`
	// AccountCaptureRelinquishAccountDetails : has no documentation (yet)
	AccountCaptureRelinquishAccountDetails *AccountCaptureRelinquishAccountDetails `json:"account_capture_relinquish_account_details,omitempty"`
	// DisabledDomainInvitesDetails : has no documentation (yet)
	DisabledDomainInvitesDetails *DisabledDomainInvitesDetails `json:"disabled_domain_invites_details,omitempty"`
	// DomainInvitesApproveRequestToJoinTeamDetails : has no documentation (yet)
	DomainInvitesApproveRequestToJoinTeamDetails *DomainInvitesApproveRequestToJoinTeamDetails `json:"domain_invites_approve_request_to_join_team_details,omitempty"`
	// DomainInvitesDeclineRequestToJoinTeamDetails : has no documentation (yet)
	DomainInvitesDeclineRequestToJoinTeamDetails *DomainInvitesDeclineRequestToJoinTeamDetails `json:"domain_invites_decline_request_to_join_team_details,omitempty"`
	// DomainInvitesEmailExistingUsersDetails : has no documentation (yet)
	DomainInvitesEmailExistingUsersDetails *DomainInvitesEmailExistingUsersDetails `json:"domain_invites_email_existing_users_details,omitempty"`
	// DomainInvitesRequestToJoinTeamDetails : has no documentation (yet)
	DomainInvitesRequestToJoinTeamDetails *DomainInvitesRequestToJoinTeamDetails `json:"domain_invites_request_to_join_team_details,omitempty"`
	// DomainInvitesSetInviteNewUserPrefToNoDetails : has no documentation (yet)
	DomainInvitesSetInviteNewUserPrefToNoDetails *DomainInvitesSetInviteNewUserPrefToNoDetails `json:"domain_invites_set_invite_new_user_pref_to_no_details,omitempty"`
	// DomainInvitesSetInviteNewUserPrefToYesDetails : has no documentation
	// (yet)
	DomainInvitesSetInviteNewUserPrefToYesDetails *DomainInvitesSetInviteNewUserPrefToYesDetails `json:"domain_invites_set_invite_new_user_pref_to_yes_details,omitempty"`
	// DomainVerificationAddDomainFailDetails : has no documentation (yet)
	DomainVerificationAddDomainFailDetails *DomainVerificationAddDomainFailDetails `json:"domain_verification_add_domain_fail_details,omitempty"`
	// DomainVerificationAddDomainSuccessDetails : has no documentation (yet)
	DomainVerificationAddDomainSuccessDetails *DomainVerificationAddDomainSuccessDetails `json:"domain_verification_add_domain_success_details,omitempty"`
	// DomainVerificationRemoveDomainDetails : has no documentation (yet)
	DomainVerificationRemoveDomainDetails *DomainVerificationRemoveDomainDetails `json:"domain_verification_remove_domain_details,omitempty"`
	// EnabledDomainInvitesDetails : has no documentation (yet)
	EnabledDomainInvitesDetails *EnabledDomainInvitesDetails `json:"enabled_domain_invites_details,omitempty"`
	// ApplyNamingConventionDetails : has no documentation (yet)
	ApplyNamingConventionDetails *ApplyNamingConventionDetails `json:"apply_naming_convention_details,omitempty"`
	// CreateFolderDetails : has no documentation (yet)
	CreateFolderDetails *CreateFolderDetails `json:"create_folder_details,omitempty"`
	// FileAddDetails : has no documentation (yet)
	FileAddDetails *FileAddDetails `json:"file_add_details,omitempty"`
	// FileCopyDetails : has no documentation (yet)
	FileCopyDetails *FileCopyDetails `json:"file_copy_details,omitempty"`
	// FileDeleteDetails : has no documentation (yet)
	FileDeleteDetails *FileDeleteDetails `json:"file_delete_details,omitempty"`
	// FileDownloadDetails : has no documentation (yet)
	FileDownloadDetails *FileDownloadDetails `json:"file_download_details,omitempty"`
	// FileEditDetails : has no documentation (yet)
	FileEditDetails *FileEditDetails `json:"file_edit_details,omitempty"`
	// FileGetCopyReferenceDetails : has no documentation (yet)
	FileGetCopyReferenceDetails *FileGetCopyReferenceDetails `json:"file_get_copy_reference_details,omitempty"`
	// FileLockingLockStatusChangedDetails : has no documentation (yet)
	FileLockingLockStatusChangedDetails *FileLockingLockStatusChangedDetails `json:"file_locking_lock_status_changed_details,omitempty"`
	// FileMoveDetails : has no documentation (yet)
	FileMoveDetails *FileMoveDetails `json:"file_move_details,omitempty"`
	// FilePermanentlyDeleteDetails : has no documentation (yet)
	FilePermanentlyDeleteDetails *FilePermanentlyDeleteDetails `json:"file_permanently_delete_details,omitempty"`
	// FilePreviewDetails : has no documentation (yet)
	FilePreviewDetails *FilePreviewDetails `json:"file_preview_details,omitempty"`
	// FileRenameDetails : has no documentation (yet)
	FileRenameDetails *FileRenameDetails `json:"file_rename_details,omitempty"`
	// FileRestoreDetails : has no documentation (yet)
	FileRestoreDetails *FileRestoreDetails `json:"file_restore_details,omitempty"`
	// FileRevertDetails : has no documentation (yet)
	FileRevertDetails *FileRevertDetails `json:"file_revert_details,omitempty"`
	// FileRollbackChangesDetails : has no documentation (yet)
	FileRollbackChangesDetails *FileRollbackChangesDetails `json:"file_rollback_changes_details,omitempty"`
	// FileSaveCopyReferenceDetails : has no documentation (yet)
	FileSaveCopyReferenceDetails *FileSaveCopyReferenceDetails `json:"file_save_copy_reference_details,omitempty"`
	// FolderOverviewDescriptionChangedDetails : has no documentation (yet)
	FolderOverviewDescriptionChangedDetails *FolderOverviewDescriptionChangedDetails `json:"folder_overview_description_changed_details,omitempty"`
	// FolderOverviewItemPinnedDetails : has no documentation (yet)
	FolderOverviewItemPinnedDetails *FolderOverviewItemPinnedDetails `json:"folder_overview_item_pinned_details,omitempty"`
	// FolderOverviewItemUnpinnedDetails : has no documentation (yet)
	FolderOverviewItemUnpinnedDetails *FolderOverviewItemUnpinnedDetails `json:"folder_overview_item_unpinned_details,omitempty"`
	// ObjectLabelAddedDetails : has no documentation (yet)
	ObjectLabelAddedDetails *ObjectLabelAddedDetails `json:"object_label_added_details,omitempty"`
	// ObjectLabelRemovedDetails : has no documentation (yet)
	ObjectLabelRemovedDetails *ObjectLabelRemovedDetails `json:"object_label_removed_details,omitempty"`
	// ObjectLabelUpdatedValueDetails : has no documentation (yet)
	ObjectLabelUpdatedValueDetails *ObjectLabelUpdatedValueDetails `json:"object_label_updated_value_details,omitempty"`
	// OrganizeFolderWithTidyDetails : has no documentation (yet)
	OrganizeFolderWithTidyDetails *OrganizeFolderWithTidyDetails `json:"organize_folder_with_tidy_details,omitempty"`
	// RewindFolderDetails : has no documentation (yet)
	RewindFolderDetails *RewindFolderDetails `json:"rewind_folder_details,omitempty"`
	// UndoNamingConventionDetails : has no documentation (yet)
	UndoNamingConventionDetails *UndoNamingConventionDetails `json:"undo_naming_convention_details,omitempty"`
	// UndoOrganizeFolderWithTidyDetails : has no documentation (yet)
	UndoOrganizeFolderWithTidyDetails *UndoOrganizeFolderWithTidyDetails `json:"undo_organize_folder_with_tidy_details,omitempty"`
	// UserTagsAddedDetails : has no documentation (yet)
	UserTagsAddedDetails *UserTagsAddedDetails `json:"user_tags_added_details,omitempty"`
	// UserTagsRemovedDetails : has no documentation (yet)
	UserTagsRemovedDetails *UserTagsRemovedDetails `json:"user_tags_removed_details,omitempty"`
	// EmailIngestReceiveFileDetails : has no documentation (yet)
	EmailIngestReceiveFileDetails *EmailIngestReceiveFileDetails `json:"email_ingest_receive_file_details,omitempty"`
	// FileRequestChangeDetails : has no documentation (yet)
	FileRequestChangeDetails *FileRequestChangeDetails `json:"file_request_change_details,omitempty"`
	// FileRequestCloseDetails : has no documentation (yet)
	FileRequestCloseDetails *FileRequestCloseDetails `json:"file_request_close_details,omitempty"`
	// FileRequestCreateDetails : has no documentation (yet)
	FileRequestCreateDetails *FileRequestCreateDetails `json:"file_request_create_details,omitempty"`
	// FileRequestDeleteDetails : has no documentation (yet)
	FileRequestDeleteDetails *FileRequestDeleteDetails `json:"file_request_delete_details,omitempty"`
	// FileRequestReceiveFileDetails : has no documentation (yet)
	FileRequestReceiveFileDetails *FileRequestReceiveFileDetails `json:"file_request_receive_file_details,omitempty"`
	// GroupAddExternalIdDetails : has no documentation (yet)
	GroupAddExternalIdDetails *GroupAddExternalIdDetails `json:"group_add_external_id_details,omitempty"`
	// GroupAddMemberDetails : has no documentation (yet)
	GroupAddMemberDetails *GroupAddMemberDetails `json:"group_add_member_details,omitempty"`
	// GroupChangeExternalIdDetails : has no documentation (yet)
	GroupChangeExternalIdDetails *GroupChangeExternalIdDetails `json:"group_change_external_id_details,omitempty"`
	// GroupChangeManagementTypeDetails : has no documentation (yet)
	GroupChangeManagementTypeDetails *GroupChangeManagementTypeDetails `json:"group_change_management_type_details,omitempty"`
	// GroupChangeMemberRoleDetails : has no documentation (yet)
	GroupChangeMemberRoleDetails *GroupChangeMemberRoleDetails `json:"group_change_member_role_details,omitempty"`
	// GroupCreateDetails : has no documentation (yet)
	GroupCreateDetails *GroupCreateDetails `json:"group_create_details,omitempty"`
	// GroupDeleteDetails : has no documentation (yet)
	GroupDeleteDetails *GroupDeleteDetails `json:"group_delete_details,omitempty"`
	// GroupDescriptionUpdatedDetails : has no documentation (yet)
	GroupDescriptionUpdatedDetails *GroupDescriptionUpdatedDetails `json:"group_description_updated_details,omitempty"`
	// GroupJoinPolicyUpdatedDetails : has no documentation (yet)
	GroupJoinPolicyUpdatedDetails *GroupJoinPolicyUpdatedDetails `json:"group_join_policy_updated_details,omitempty"`
	// GroupMovedDetails : has no documentation (yet)
	GroupMovedDetails *GroupMovedDetails `json:"group_moved_details,omitempty"`
	// GroupRemoveExternalIdDetails : has no documentation (yet)
	GroupRemoveExternalIdDetails *GroupRemoveExternalIdDetails `json:"group_remove_external_id_details,omitempty"`
	// GroupRemoveMemberDetails : has no documentation (yet)
	GroupRemoveMemberDetails *GroupRemoveMemberDetails `json:"group_remove_member_details,omitempty"`
	// GroupRenameDetails : has no documentation (yet)
	GroupRenameDetails *GroupRenameDetails `json:"group_rename_details,omitempty"`
	// AccountLockOrUnlockedDetails : has no documentation (yet)
	AccountLockOrUnlockedDetails *AccountLockOrUnlockedDetails `json:"account_lock_or_unlocked_details,omitempty"`
	// EmmErrorDetails : has no documentation (yet)
	EmmErrorDetails *EmmErrorDetails `json:"emm_error_details,omitempty"`
	// GuestAdminSignedInViaTrustedTeamsDetails : has no documentation (yet)
	GuestAdminSignedInViaTrustedTeamsDetails *GuestAdminSignedInViaTrustedTeamsDetails `json:"guest_admin_signed_in_via_trusted_teams_details,omitempty"`
	// GuestAdminSignedOutViaTrustedTeamsDetails : has no documentation (yet)
	GuestAdminSignedOutViaTrustedTeamsDetails *GuestAdminSignedOutViaTrustedTeamsDetails `json:"guest_admin_signed_out_via_trusted_teams_details,omitempty"`
	// LoginFailDetails : has no documentation (yet)
	LoginFailDetails *LoginFailDetails `json:"login_fail_details,omitempty"`
	// LoginSuccessDetails : has no documentation (yet)
	LoginSuccessDetails *LoginSuccessDetails `json:"login_success_details,omitempty"`
	// LogoutDetails : has no documentation (yet)
	LogoutDetails *LogoutDetails `json:"logout_details,omitempty"`
	// ResellerSupportSessionEndDetails : has no documentation (yet)
	ResellerSupportSessionEndDetails *ResellerSupportSessionEndDetails `json:"reseller_support_session_end_details,omitempty"`
	// ResellerSupportSessionStartDetails : has no documentation (yet)
	ResellerSupportSessionStartDetails *ResellerSupportSessionStartDetails `json:"reseller_support_session_start_details,omitempty"`
	// SignInAsSessionEndDetails : has no documentation (yet)
	SignInAsSessionEndDetails *SignInAsSessionEndDetails `json:"sign_in_as_session_end_details,omitempty"`
	// SignInAsSessionStartDetails : has no documentation (yet)
	SignInAsSessionStartDetails *SignInAsSessionStartDetails `json:"sign_in_as_session_start_details,omitempty"`
	// SsoErrorDetails : has no documentation (yet)
	SsoErrorDetails *SsoErrorDetails `json:"sso_error_details,omitempty"`
	// CreateTeamInviteLinkDetails : has no documentation (yet)
	CreateTeamInviteLinkDetails *CreateTeamInviteLinkDetails `json:"create_team_invite_link_details,omitempty"`
	// DeleteTeamInviteLinkDetails : has no documentation (yet)
	DeleteTeamInviteLinkDetails *DeleteTeamInviteLinkDetails `json:"delete_team_invite_link_details,omitempty"`
	// MemberAddExternalIdDetails : has no documentation (yet)
	MemberAddExternalIdDetails *MemberAddExternalIdDetails `json:"member_add_external_id_details,omitempty"`
	// MemberAddNameDetails : has no documentation (yet)
	MemberAddNameDetails *MemberAddNameDetails `json:"member_add_name_details,omitempty"`
	// MemberChangeAdminRoleDetails : has no documentation (yet)
	MemberChangeAdminRoleDetails *MemberChangeAdminRoleDetails `json:"member_change_admin_role_details,omitempty"`
	// MemberChangeEmailDetails : has no documentation (yet)
	MemberChangeEmailDetails *MemberChangeEmailDetails `json:"member_change_email_details,omitempty"`
	// MemberChangeExternalIdDetails : has no documentation (yet)
	MemberChangeExternalIdDetails *MemberChangeExternalIdDetails `json:"member_change_external_id_details,omitempty"`
	// MemberChangeMembershipTypeDetails : has no documentation (yet)
	MemberChangeMembershipTypeDetails *MemberChangeMembershipTypeDetails `json:"member_change_membership_type_details,omitempty"`
	// MemberChangeNameDetails : has no documentation (yet)
	MemberChangeNameDetails *MemberChangeNameDetails `json:"member_change_name_details,omitempty"`
	// MemberChangeResellerRoleDetails : has no documentation (yet)
	MemberChangeResellerRoleDetails *MemberChangeResellerRoleDetails `json:"member_change_reseller_role_details,omitempty"`
	// MemberChangeStatusDetails : has no documentation (yet)
	MemberChangeStatusDetails *MemberChangeStatusDetails `json:"member_change_status_details,omitempty"`
	// MemberDeleteManualContactsDetails : has no documentation (yet)
	MemberDeleteManualContactsDetails *MemberDeleteManualContactsDetails `json:"member_delete_manual_contacts_details,omitempty"`
	// MemberDeleteProfilePhotoDetails : has no documentation (yet)
	MemberDeleteProfilePhotoDetails *MemberDeleteProfilePhotoDetails `json:"member_delete_profile_photo_details,omitempty"`
	// MemberPermanentlyDeleteAccountContentsDetails : has no documentation
	// (yet)
	MemberPermanentlyDeleteAccountContentsDetails *MemberPermanentlyDeleteAccountContentsDetails `json:"member_permanently_delete_account_contents_details,omitempty"`
	// MemberRemoveExternalIdDetails : has no documentation (yet)
	MemberRemoveExternalIdDetails *MemberRemoveExternalIdDetails `json:"member_remove_external_id_details,omitempty"`
	// MemberSetProfilePhotoDetails : has no documentation (yet)
	MemberSetProfilePhotoDetails *MemberSetProfilePhotoDetails `json:"member_set_profile_photo_details,omitempty"`
	// MemberSpaceLimitsAddCustomQuotaDetails : has no documentation (yet)
	MemberSpaceLimitsAddCustomQuotaDetails *MemberSpaceLimitsAddCustomQuotaDetails `json:"member_space_limits_add_custom_quota_details,omitempty"`
	// MemberSpaceLimitsChangeCustomQuotaDetails : has no documentation (yet)
	MemberSpaceLimitsChangeCustomQuotaDetails *MemberSpaceLimitsChangeCustomQuotaDetails `json:"member_space_limits_change_custom_quota_details,omitempty"`
	// MemberSpaceLimitsChangeStatusDetails : has no documentation (yet)
	MemberSpaceLimitsChangeStatusDetails *MemberSpaceLimitsChangeStatusDetails `json:"member_space_limits_change_status_details,omitempty"`
	// MemberSpaceLimitsRemoveCustomQuotaDetails : has no documentation (yet)
	MemberSpaceLimitsRemoveCustomQuotaDetails *MemberSpaceLimitsRemoveCustomQuotaDetails `json:"member_space_limits_remove_custom_quota_details,omitempty"`
	// MemberSuggestDetails : has no documentation (yet)
	MemberSuggestDetails *MemberSuggestDetails `json:"member_suggest_details,omitempty"`
	// MemberTransferAccountContentsDetails : has no documentation (yet)
	MemberTransferAccountContentsDetails *MemberTransferAccountContentsDetails `json:"member_transfer_account_contents_details,omitempty"`
	// PendingSecondaryEmailAddedDetails : has no documentation (yet)
	PendingSecondaryEmailAddedDetails *PendingSecondaryEmailAddedDetails `json:"pending_secondary_email_added_details,omitempty"`
	// SecondaryEmailDeletedDetails : has no documentation (yet)
	SecondaryEmailDeletedDetails *SecondaryEmailDeletedDetails `json:"secondary_email_deleted_details,omitempty"`
	// SecondaryEmailVerifiedDetails : has no documentation (yet)
	SecondaryEmailVerifiedDetails *SecondaryEmailVerifiedDetails `json:"secondary_email_verified_details,omitempty"`
	// SecondaryMailsPolicyChangedDetails : has no documentation (yet)
	SecondaryMailsPolicyChangedDetails *SecondaryMailsPolicyChangedDetails `json:"secondary_mails_policy_changed_details,omitempty"`
	// BinderAddPageDetails : has no documentation (yet)
	BinderAddPageDetails *BinderAddPageDetails `json:"binder_add_page_details,omitempty"`
	// BinderAddSectionDetails : has no documentation (yet)
	BinderAddSectionDetails *BinderAddSectionDetails `json:"binder_add_section_details,omitempty"`
	// BinderRemovePageDetails : has no documentation (yet)
	BinderRemovePageDetails *BinderRemovePageDetails `json:"binder_remove_page_details,omitempty"`
	// BinderRemoveSectionDetails : has no documentation (yet)
	BinderRemoveSectionDetails *BinderRemoveSectionDetails `json:"binder_remove_section_details,omitempty"`
	// BinderRenamePageDetails : has no documentation (yet)
	BinderRenamePageDetails *BinderRenamePageDetails `json:"binder_rename_page_details,omitempty"`
	// BinderRenameSectionDetails : has no documentation (yet)
	BinderRenameSectionDetails *BinderRenameSectionDetails `json:"binder_rename_section_details,omitempty"`
	// BinderReorderPageDetails : has no documentation (yet)
	BinderReorderPageDetails *BinderReorderPageDetails `json:"binder_reorder_page_details,omitempty"`
	// BinderReorderSectionDetails : has no documentation (yet)
	BinderReorderSectionDetails *BinderReorderSectionDetails `json:"binder_reorder_section_details,omitempty"`
	// PaperContentAddMemberDetails : has no documentation (yet)
	PaperContentAddMemberDetails *PaperContentAddMemberDetails `json:"paper_content_add_member_details,omitempty"`
	// PaperContentAddToFolderDetails : has no documentation (yet)
	PaperContentAddToFolderDetails *PaperContentAddToFolderDetails `json:"paper_content_add_to_folder_details,omitempty"`
	// PaperContentArchiveDetails : has no documentation (yet)
	PaperContentArchiveDetails *PaperContentArchiveDetails `json:"paper_content_archive_details,omitempty"`
	// PaperContentCreateDetails : has no documentation (yet)
	PaperContentCreateDetails *PaperContentCreateDetails `json:"paper_content_create_details,omitempty"`
	// PaperContentPermanentlyDeleteDetails : has no documentation (yet)
	PaperContentPermanentlyDeleteDetails *PaperContentPermanentlyDeleteDetails `json:"paper_content_permanently_delete_details,omitempty"`
	// PaperContentRemoveFromFolderDetails : has no documentation (yet)
	PaperContentRemoveFromFolderDetails *PaperContentRemoveFromFolderDetails `json:"paper_content_remove_from_folder_details,omitempty"`
	// PaperContentRemoveMemberDetails : has no documentation (yet)
	PaperContentRemoveMemberDetails *PaperContentRemoveMemberDetails `json:"paper_content_remove_member_details,omitempty"`
	// PaperContentRenameDetails : has no documentation (yet)
	PaperContentRenameDetails *PaperContentRenameDetails `json:"paper_content_rename_details,omitempty"`
	// PaperContentRestoreDetails : has no documentation (yet)
	PaperContentRestoreDetails *PaperContentRestoreDetails `json:"paper_content_restore_details,omitempty"`
	// PaperDocAddCommentDetails : has no documentation (yet)
	PaperDocAddCommentDetails *PaperDocAddCommentDetails `json:"paper_doc_add_comment_details,omitempty"`
	// PaperDocChangeMemberRoleDetails : has no documentation (yet)
	PaperDocChangeMemberRoleDetails *PaperDocChangeMemberRoleDetails `json:"paper_doc_change_member_role_details,omitempty"`
	// PaperDocChangeSharingPolicyDetails : has no documentation (yet)
	PaperDocChangeSharingPolicyDetails *PaperDocChangeSharingPolicyDetails `json:"paper_doc_change_sharing_policy_details,omitempty"`
	// PaperDocChangeSubscriptionDetails : has no documentation (yet)
	PaperDocChangeSubscriptionDetails *PaperDocChangeSubscriptionDetails `json:"paper_doc_change_subscription_details,omitempty"`
	// PaperDocDeletedDetails : has no documentation (yet)
	PaperDocDeletedDetails *PaperDocDeletedDetails `json:"paper_doc_deleted_details,omitempty"`
	// PaperDocDeleteCommentDetails : has no documentation (yet)
	PaperDocDeleteCommentDetails *PaperDocDeleteCommentDetails `json:"paper_doc_delete_comment_details,omitempty"`
	// PaperDocDownloadDetails : has no documentation (yet)
	PaperDocDownloadDetails *PaperDocDownloadDetails `json:"paper_doc_download_details,omitempty"`
	// PaperDocEditDetails : has no documentation (yet)
	PaperDocEditDetails *PaperDocEditDetails `json:"paper_doc_edit_details,omitempty"`
	// PaperDocEditCommentDetails : has no documentation (yet)
	PaperDocEditCommentDetails *PaperDocEditCommentDetails `json:"paper_doc_edit_comment_details,omitempty"`
	// PaperDocFollowedDetails : has no documentation (yet)
	PaperDocFollowedDetails *PaperDocFollowedDetails `json:"paper_doc_followed_details,omitempty"`
	// PaperDocMentionDetails : has no documentation (yet)
	PaperDocMentionDetails *PaperDocMentionDetails `json:"paper_doc_mention_details,omitempty"`
	// PaperDocOwnershipChangedDetails : has no documentation (yet)
	PaperDocOwnershipChangedDetails *PaperDocOwnershipChangedDetails `json:"paper_doc_ownership_changed_details,omitempty"`
	// PaperDocRequestAccessDetails : has no documentation (yet)
	PaperDocRequestAccessDetails *PaperDocRequestAccessDetails `json:"paper_doc_request_access_details,omitempty"`
	// PaperDocResolveCommentDetails : has no documentation (yet)
	PaperDocResolveCommentDetails *PaperDocResolveCommentDetails `json:"paper_doc_resolve_comment_details,omitempty"`
	// PaperDocRevertDetails : has no documentation (yet)
	PaperDocRevertDetails *PaperDocRevertDetails `json:"paper_doc_revert_details,omitempty"`
	// PaperDocSlackShareDetails : has no documentation (yet)
	PaperDocSlackShareDetails *PaperDocSlackShareDetails `json:"paper_doc_slack_share_details,omitempty"`
	// PaperDocTeamInviteDetails : has no documentation (yet)
	PaperDocTeamInviteDetails *PaperDocTeamInviteDetails `json:"paper_doc_team_invite_details,omitempty"`
	// PaperDocTrashedDetails : has no documentation (yet)
	PaperDocTrashedDetails *PaperDocTrashedDetails `json:"paper_doc_trashed_details,omitempty"`
	// PaperDocUnresolveCommentDetails : has no documentation (yet)
	PaperDocUnresolveCommentDetails *PaperDocUnresolveCommentDetails `json:"paper_doc_unresolve_comment_details,omitempty"`
	// PaperDocUntrashedDetails : has no documentation (yet)
	PaperDocUntrashedDetails *PaperDocUntrashedDetails `json:"paper_doc_untrashed_details,omitempty"`
	// PaperDocViewDetails : has no documentation (yet)
	PaperDocViewDetails *PaperDocViewDetails `json:"paper_doc_view_details,omitempty"`
	// PaperExternalViewAllowDetails : has no documentation (yet)
	PaperExternalViewAllowDetails *PaperExternalViewAllowDetails `json:"paper_external_view_allow_details,omitempty"`
	// PaperExternalViewDefaultTeamDetails : has no documentation (yet)
	PaperExternalViewDefaultTeamDetails *PaperExternalViewDefaultTeamDetails `json:"paper_external_view_default_team_details,omitempty"`
	// PaperExternalViewForbidDetails : has no documentation (yet)
	PaperExternalViewForbidDetails *PaperExternalViewForbidDetails `json:"paper_external_view_forbid_details,omitempty"`
	// PaperFolderChangeSubscriptionDetails : has no documentation (yet)
	PaperFolderChangeSubscriptionDetails *PaperFolderChangeSubscriptionDetails `json:"paper_folder_change_subscription_details,omitempty"`
	// PaperFolderDeletedDetails : has no documentation (yet)
	PaperFolderDeletedDetails *PaperFolderDeletedDetails `json:"paper_folder_deleted_details,omitempty"`
	// PaperFolderFollowedDetails : has no documentation (yet)
	PaperFolderFollowedDetails *PaperFolderFollowedDetails `json:"paper_folder_followed_details,omitempty"`
	// PaperFolderTeamInviteDetails : has no documentation (yet)
	PaperFolderTeamInviteDetails *PaperFolderTeamInviteDetails `json:"paper_folder_team_invite_details,omitempty"`
	// PaperPublishedLinkChangePermissionDetails : has no documentation (yet)
	PaperPublishedLinkChangePermissionDetails *PaperPublishedLinkChangePermissionDetails `json:"paper_published_link_change_permission_details,omitempty"`
	// PaperPublishedLinkCreateDetails : has no documentation (yet)
	PaperPublishedLinkCreateDetails *PaperPublishedLinkCreateDetails `json:"paper_published_link_create_details,omitempty"`
	// PaperPublishedLinkDisabledDetails : has no documentation (yet)
	PaperPublishedLinkDisabledDetails *PaperPublishedLinkDisabledDetails `json:"paper_published_link_disabled_details,omitempty"`
	// PaperPublishedLinkViewDetails : has no documentation (yet)
	PaperPublishedLinkViewDetails *PaperPublishedLinkViewDetails `json:"paper_published_link_view_details,omitempty"`
	// PasswordChangeDetails : has no documentation (yet)
	PasswordChangeDetails *PasswordChangeDetails `json:"password_change_details,omitempty"`
	// PasswordResetDetails : has no documentation (yet)
	PasswordResetDetails *PasswordResetDetails `json:"password_reset_details,omitempty"`
	// PasswordResetAllDetails : has no documentation (yet)
	PasswordResetAllDetails *PasswordResetAllDetails `json:"password_reset_all_details,omitempty"`
	// ClassificationCreateReportDetails : has no documentation (yet)
	ClassificationCreateReportDetails *ClassificationCreateReportDetails `json:"classification_create_report_details,omitempty"`
	// ClassificationCreateReportFailDetails : has no documentation (yet)
	ClassificationCreateReportFailDetails *ClassificationCreateReportFailDetails `json:"classification_create_report_fail_details,omitempty"`
	// EmmCreateExceptionsReportDetails : has no documentation (yet)
	EmmCreateExceptionsReportDetails *EmmCreateExceptionsReportDetails `json:"emm_create_exceptions_report_details,omitempty"`
	// EmmCreateUsageReportDetails : has no documentation (yet)
	EmmCreateUsageReportDetails *EmmCreateUsageReportDetails `json:"emm_create_usage_report_details,omitempty"`
	// ExportMembersReportDetails : has no documentation (yet)
	ExportMembersReportDetails *ExportMembersReportDetails `json:"export_members_report_details,omitempty"`
	// ExportMembersReportFailDetails : has no documentation (yet)
	ExportMembersReportFailDetails *ExportMembersReportFailDetails `json:"export_members_report_fail_details,omitempty"`
	// ExternalSharingCreateReportDetails : has no documentation (yet)
	ExternalSharingCreateReportDetails *ExternalSharingCreateReportDetails `json:"external_sharing_create_report_details,omitempty"`
	// ExternalSharingReportFailedDetails : has no documentation (yet)
	ExternalSharingReportFailedDetails *ExternalSharingReportFailedDetails `json:"external_sharing_report_failed_details,omitempty"`
	// NoExpirationLinkGenCreateReportDetails : has no documentation (yet)
	NoExpirationLinkGenCreateReportDetails *NoExpirationLinkGenCreateReportDetails `json:"no_expiration_link_gen_create_report_details,omitempty"`
	// NoExpirationLinkGenReportFailedDetails : has no documentation (yet)
	NoExpirationLinkGenReportFailedDetails *NoExpirationLinkGenReportFailedDetails `json:"no_expiration_link_gen_report_failed_details,omitempty"`
	// NoPasswordLinkGenCreateReportDetails : has no documentation (yet)
	NoPasswordLinkGenCreateReportDetails *NoPasswordLinkGenCreateReportDetails `json:"no_password_link_gen_create_report_details,omitempty"`
	// NoPasswordLinkGenReportFailedDetails : has no documentation (yet)
	NoPasswordLinkGenReportFailedDetails *NoPasswordLinkGenReportFailedDetails `json:"no_password_link_gen_report_failed_details,omitempty"`
	// NoPasswordLinkViewCreateReportDetails : has no documentation (yet)
	NoPasswordLinkViewCreateReportDetails *NoPasswordLinkViewCreateReportDetails `json:"no_password_link_view_create_report_details,omitempty"`
	// NoPasswordLinkViewReportFailedDetails : has no documentation (yet)
	NoPasswordLinkViewReportFailedDetails *NoPasswordLinkViewReportFailedDetails `json:"no_password_link_view_report_failed_details,omitempty"`
	// OutdatedLinkViewCreateReportDetails : has no documentation (yet)
	OutdatedLinkViewCreateReportDetails *OutdatedLinkViewCreateReportDetails `json:"outdated_link_view_create_report_details,omitempty"`
	// OutdatedLinkViewReportFailedDetails : has no documentation (yet)
	OutdatedLinkViewReportFailedDetails *OutdatedLinkViewReportFailedDetails `json:"outdated_link_view_report_failed_details,omitempty"`
	// PaperAdminExportStartDetails : has no documentation (yet)
	PaperAdminExportStartDetails *PaperAdminExportStartDetails `json:"paper_admin_export_start_details,omitempty"`
	// SmartSyncCreateAdminPrivilegeReportDetails : has no documentation (yet)
	SmartSyncCreateAdminPrivilegeReportDetails *SmartSyncCreateAdminPrivilegeReportDetails `json:"smart_sync_create_admin_privilege_report_details,omitempty"`
	// TeamActivityCreateReportDetails : has no documentation (yet)
	TeamActivityCreateReportDetails *TeamActivityCreateReportDetails `json:"team_activity_create_report_details,omitempty"`
	// TeamActivityCreateReportFailDetails : has no documentation (yet)
	TeamActivityCreateReportFailDetails *TeamActivityCreateReportFailDetails `json:"team_activity_create_report_fail_details,omitempty"`
	// CollectionShareDetails : has no documentation (yet)
	CollectionShareDetails *CollectionShareDetails `json:"collection_share_details,omitempty"`
	// FileTransfersFileAddDetails : has no documentation (yet)
	FileTransfersFileAddDetails *FileTransfersFileAddDetails `json:"file_transfers_file_add_details,omitempty"`
	// FileTransfersTransferDeleteDetails : has no documentation (yet)
	FileTransfersTransferDeleteDetails *FileTransfersTransferDeleteDetails `json:"file_transfers_transfer_delete_details,omitempty"`
	// FileTransfersTransferDownloadDetails : has no documentation (yet)
	FileTransfersTransferDownloadDetails *FileTransfersTransferDownloadDetails `json:"file_transfers_transfer_download_details,omitempty"`
	// FileTransfersTransferSendDetails : has no documentation (yet)
	FileTransfersTransferSendDetails *FileTransfersTransferSendDetails `json:"file_transfers_transfer_send_details,omitempty"`
	// FileTransfersTransferViewDetails : has no documentation (yet)
	FileTransfersTransferViewDetails *FileTransfersTransferViewDetails `json:"file_transfers_transfer_view_details,omitempty"`
	// NoteAclInviteOnlyDetails : has no documentation (yet)
	NoteAclInviteOnlyDetails *NoteAclInviteOnlyDetails `json:"note_acl_invite_only_details,omitempty"`
	// NoteAclLinkDetails : has no documentation (yet)
	NoteAclLinkDetails *NoteAclLinkDetails `json:"note_acl_link_details,omitempty"`
	// NoteAclTeamLinkDetails : has no documentation (yet)
	NoteAclTeamLinkDetails *NoteAclTeamLinkDetails `json:"note_acl_team_link_details,omitempty"`
	// NoteSharedDetails : has no documentation (yet)
	NoteSharedDetails *NoteSharedDetails `json:"note_shared_details,omitempty"`
	// NoteShareReceiveDetails : has no documentation (yet)
	NoteShareReceiveDetails *NoteShareReceiveDetails `json:"note_share_receive_details,omitempty"`
	// OpenNoteSharedDetails : has no documentation (yet)
	OpenNoteSharedDetails *OpenNoteSharedDetails `json:"open_note_shared_details,omitempty"`
	// SfAddGroupDetails : has no documentation (yet)
	SfAddGroupDetails *SfAddGroupDetails `json:"sf_add_group_details,omitempty"`
	// SfAllowNonMembersToViewSharedLinksDetails : has no documentation (yet)
	SfAllowNonMembersToViewSharedLinksDetails *SfAllowNonMembersToViewSharedLinksDetails `json:"sf_allow_non_members_to_view_shared_links_details,omitempty"`
	// SfExternalInviteWarnDetails : has no documentation (yet)
	SfExternalInviteWarnDetails *SfExternalInviteWarnDetails `json:"sf_external_invite_warn_details,omitempty"`
	// SfFbInviteDetails : has no documentation (yet)
	SfFbInviteDetails *SfFbInviteDetails `json:"sf_fb_invite_details,omitempty"`
	// SfFbInviteChangeRoleDetails : has no documentation (yet)
	SfFbInviteChangeRoleDetails *SfFbInviteChangeRoleDetails `json:"sf_fb_invite_change_role_details,omitempty"`
	// SfFbUninviteDetails : has no documentation (yet)
	SfFbUninviteDetails *SfFbUninviteDetails `json:"sf_fb_uninvite_details,omitempty"`
	// SfInviteGroupDetails : has no documentation (yet)
	SfInviteGroupDetails *SfInviteGroupDetails `json:"sf_invite_group_details,omitempty"`
	// SfTeamGrantAccessDetails : has no documentation (yet)
	SfTeamGrantAccessDetails *SfTeamGrantAccessDetails `json:"sf_team_grant_access_details,omitempty"`
	// SfTeamInviteDetails : has no documentation (yet)
	SfTeamInviteDetails *SfTeamInviteDetails `json:"sf_team_invite_details,omitempty"`
	// SfTeamInviteChangeRoleDetails : has no documentation (yet)
	SfTeamInviteChangeRoleDetails *SfTeamInviteChangeRoleDetails `json:"sf_team_invite_change_role_details,omitempty"`
	// SfTeamJoinDetails : has no documentation (yet)
	SfTeamJoinDetails *SfTeamJoinDetails `json:"sf_team_join_details,omitempty"`
	// SfTeamJoinFromOobLinkDetails : has no documentation (yet)
	SfTeamJoinFromOobLinkDetails *SfTeamJoinFromOobLinkDetails `json:"sf_team_join_from_oob_link_details,omitempty"`
	// SfTeamUninviteDetails : has no documentation (yet)
	SfTeamUninviteDetails *SfTeamUninviteDetails `json:"sf_team_uninvite_details,omitempty"`
	// SharedContentAddInviteesDetails : has no documentation (yet)
	SharedContentAddInviteesDetails *SharedContentAddInviteesDetails `json:"shared_content_add_invitees_details,omitempty"`
	// SharedContentAddLinkExpiryDetails : has no documentation (yet)
	SharedContentAddLinkExpiryDetails *SharedContentAddLinkExpiryDetails `json:"shared_content_add_link_expiry_details,omitempty"`
	// SharedContentAddLinkPasswordDetails : has no documentation (yet)
	SharedContentAddLinkPasswordDetails *SharedContentAddLinkPasswordDetails `json:"shared_content_add_link_password_details,omitempty"`
	// SharedContentAddMemberDetails : has no documentation (yet)
	SharedContentAddMemberDetails *SharedContentAddMemberDetails `json:"shared_content_add_member_details,omitempty"`
	// SharedContentChangeDownloadsPolicyDetails : has no documentation (yet)
	SharedContentChangeDownloadsPolicyDetails *SharedContentChangeDownloadsPolicyDetails `json:"shared_content_change_downloads_policy_details,omitempty"`
	// SharedContentChangeInviteeRoleDetails : has no documentation (yet)
	SharedContentChangeInviteeRoleDetails *SharedContentChangeInviteeRoleDetails `json:"shared_content_change_invitee_role_details,omitempty"`
	// SharedContentChangeLinkAudienceDetails : has no documentation (yet)
	SharedContentChangeLinkAudienceDetails *SharedContentChangeLinkAudienceDetails `json:"shared_content_change_link_audience_details,omitempty"`
	// SharedContentChangeLinkExpiryDetails : has no documentation (yet)
	SharedContentChangeLinkExpiryDetails *SharedContentChangeLinkExpiryDetails `json:"shared_content_change_link_expiry_details,omitempty"`
	// SharedContentChangeLinkPasswordDetails : has no documentation (yet)
	SharedContentChangeLinkPasswordDetails *SharedContentChangeLinkPasswordDetails `json:"shared_content_change_link_password_details,omitempty"`
	// SharedContentChangeMemberRoleDetails : has no documentation (yet)
	SharedContentChangeMemberRoleDetails *SharedContentChangeMemberRoleDetails `json:"shared_content_change_member_role_details,omitempty"`
	// SharedContentChangeViewerInfoPolicyDetails : has no documentation (yet)
	SharedContentChangeViewerInfoPolicyDetails *SharedContentChangeViewerInfoPolicyDetails `json:"shared_content_change_viewer_info_policy_details,omitempty"`
	// SharedContentClaimInvitationDetails : has no documentation (yet)
	SharedContentClaimInvitationDetails *SharedContentClaimInvitationDetails `json:"shared_content_claim_invitation_details,omitempty"`
	// SharedContentCopyDetails : has no documentation (yet)
	SharedContentCopyDetails *SharedContentCopyDetails `json:"shared_content_copy_details,omitempty"`
	// SharedContentDownloadDetails : has no documentation (yet)
	SharedContentDownloadDetails *SharedContentDownloadDetails `json:"shared_content_download_details,omitempty"`
	// SharedContentRelinquishMembershipDetails : has no documentation (yet)
	SharedContentRelinquishMembershipDetails *SharedContentRelinquishMembershipDetails `json:"shared_content_relinquish_membership_details,omitempty"`
	// SharedContentRemoveInviteesDetails : has no documentation (yet)
	SharedContentRemoveInviteesDetails *SharedContentRemoveInviteesDetails `json:"shared_content_remove_invitees_details,omitempty"`
	// SharedContentRemoveLinkExpiryDetails : has no documentation (yet)
	SharedContentRemoveLinkExpiryDetails *SharedContentRemoveLinkExpiryDetails `json:"shared_content_remove_link_expiry_details,omitempty"`
	// SharedContentRemoveLinkPasswordDetails : has no documentation (yet)
	SharedContentRemoveLinkPasswordDetails *SharedContentRemoveLinkPasswordDetails `json:"shared_content_remove_link_password_details,omitempty"`
	// SharedContentRemoveMemberDetails : has no documentation (yet)
	SharedContentRemoveMemberDetails *SharedContentRemoveMemberDetails `json:"shared_content_remove_member_details,omitempty"`
	// SharedContentRequestAccessDetails : has no documentation (yet)
	SharedContentRequestAccessDetails *SharedContentRequestAccessDetails `json:"shared_content_request_access_details,omitempty"`
	// SharedContentRestoreInviteesDetails : has no documentation (yet)
	SharedContentRestoreInviteesDetails *SharedContentRestoreInviteesDetails `json:"shared_content_restore_invitees_details,omitempty"`
	// SharedContentRestoreMemberDetails : has no documentation (yet)
	SharedContentRestoreMemberDetails *SharedContentRestoreMemberDetails `json:"shared_content_restore_member_details,omitempty"`
	// SharedContentUnshareDetails : has no documentation (yet)
	SharedContentUnshareDetails *SharedContentUnshareDetails `json:"shared_content_unshare_details,omitempty"`
	// SharedContentViewDetails : has no documentation (yet)
	SharedContentViewDetails *SharedContentViewDetails `json:"shared_content_view_details,omitempty"`
	// SharedFolderChangeLinkPolicyDetails : has no documentation (yet)
	SharedFolderChangeLinkPolicyDetails *SharedFolderChangeLinkPolicyDetails `json:"shared_folder_change_link_policy_details,omitempty"`
	// SharedFolderChangeMembersInheritancePolicyDetails : has no documentation
	// (yet)
	SharedFolderChangeMembersInheritancePolicyDetails *SharedFolderChangeMembersInheritancePolicyDetails `json:"shared_folder_change_members_inheritance_policy_details,omitempty"`
	// SharedFolderChangeMembersManagementPolicyDetails : has no documentation
	// (yet)
	SharedFolderChangeMembersManagementPolicyDetails *SharedFolderChangeMembersManagementPolicyDetails `json:"shared_folder_change_members_management_policy_details,omitempty"`
	// SharedFolderChangeMembersPolicyDetails : has no documentation (yet)
	SharedFolderChangeMembersPolicyDetails *SharedFolderChangeMembersPolicyDetails `json:"shared_folder_change_members_policy_details,omitempty"`
	// SharedFolderCreateDetails : has no documentation (yet)
	SharedFolderCreateDetails *SharedFolderCreateDetails `json:"shared_folder_create_details,omitempty"`
	// SharedFolderDeclineInvitationDetails : has no documentation (yet)
	SharedFolderDeclineInvitationDetails *SharedFolderDeclineInvitationDetails `json:"shared_folder_decline_invitation_details,omitempty"`
	// SharedFolderMountDetails : has no documentation (yet)
	SharedFolderMountDetails *SharedFolderMountDetails `json:"shared_folder_mount_details,omitempty"`
	// SharedFolderNestDetails : has no documentation (yet)
	SharedFolderNestDetails *SharedFolderNestDetails `json:"shared_folder_nest_details,omitempty"`
	// SharedFolderTransferOwnershipDetails : has no documentation (yet)
	SharedFolderTransferOwnershipDetails *SharedFolderTransferOwnershipDetails `json:"shared_folder_transfer_ownership_details,omitempty"`
	// SharedFolderUnmountDetails : has no documentation (yet)
	SharedFolderUnmountDetails *SharedFolderUnmountDetails `json:"shared_folder_unmount_details,omitempty"`
	// SharedLinkAddExpiryDetails : has no documentation (yet)
	SharedLinkAddExpiryDetails *SharedLinkAddExpiryDetails `json:"shared_link_add_expiry_details,omitempty"`
	// SharedLinkChangeExpiryDetails : has no documentation (yet)
	SharedLinkChangeExpiryDetails *SharedLinkChangeExpiryDetails `json:"shared_link_change_expiry_details,omitempty"`
	// SharedLinkChangeVisibilityDetails : has no documentation (yet)
	SharedLinkChangeVisibilityDetails *SharedLinkChangeVisibilityDetails `json:"shared_link_change_visibility_details,omitempty"`
	// SharedLinkCopyDetails : has no documentation (yet)
	SharedLinkCopyDetails *SharedLinkCopyDetails `json:"shared_link_copy_details,omitempty"`
	// SharedLinkCreateDetails : has no documentation (yet)
	SharedLinkCreateDetails *SharedLinkCreateDetails `json:"shared_link_create_details,omitempty"`
	// SharedLinkDisableDetails : has no documentation (yet)
	SharedLinkDisableDetails *SharedLinkDisableDetails `json:"shared_link_disable_details,omitempty"`
	// SharedLinkDownloadDetails : has no documentation (yet)
	SharedLinkDownloadDetails *SharedLinkDownloadDetails `json:"shared_link_download_details,omitempty"`
	// SharedLinkRemoveExpiryDetails : has no documentation (yet)
	SharedLinkRemoveExpiryDetails *SharedLinkRemoveExpiryDetails `json:"shared_link_remove_expiry_details,omitempty"`
	// SharedLinkSettingsAddExpirationDetails : has no documentation (yet)
	SharedLinkSettingsAddExpirationDetails *SharedLinkSettingsAddExpirationDetails `json:"shared_link_settings_add_expiration_details,omitempty"`
	// SharedLinkSettingsAddPasswordDetails : has no documentation (yet)
	SharedLinkSettingsAddPasswordDetails *SharedLinkSettingsAddPasswordDetails `json:"shared_link_settings_add_password_details,omitempty"`
	// SharedLinkSettingsAllowDownloadDisabledDetails : has no documentation
	// (yet)
	SharedLinkSettingsAllowDownloadDisabledDetails *SharedLinkSettingsAllowDownloadDisabledDetails `json:"shared_link_settings_allow_download_disabled_details,omitempty"`
	// SharedLinkSettingsAllowDownloadEnabledDetails : has no documentation
	// (yet)
	SharedLinkSettingsAllowDownloadEnabledDetails *SharedLinkSettingsAllowDownloadEnabledDetails `json:"shared_link_settings_allow_download_enabled_details,omitempty"`
	// SharedLinkSettingsChangeAudienceDetails : has no documentation (yet)
	SharedLinkSettingsChangeAudienceDetails *SharedLinkSettingsChangeAudienceDetails `json:"shared_link_settings_change_audience_details,omitempty"`
	// SharedLinkSettingsChangeExpirationDetails : has no documentation (yet)
	SharedLinkSettingsChangeExpirationDetails *SharedLinkSettingsChangeExpirationDetails `json:"shared_link_settings_change_expiration_details,omitempty"`
	// SharedLinkSettingsChangePasswordDetails : has no documentation (yet)
	SharedLinkSettingsChangePasswordDetails *SharedLinkSettingsChangePasswordDetails `json:"shared_link_settings_change_password_details,omitempty"`
	// SharedLinkSettingsRemoveExpirationDetails : has no documentation (yet)
	SharedLinkSettingsRemoveExpirationDetails *SharedLinkSettingsRemoveExpirationDetails `json:"shared_link_settings_remove_expiration_details,omitempty"`
	// SharedLinkSettingsRemovePasswordDetails : has no documentation (yet)
	SharedLinkSettingsRemovePasswordDetails *SharedLinkSettingsRemovePasswordDetails `json:"shared_link_settings_remove_password_details,omitempty"`
	// SharedLinkShareDetails : has no documentation (yet)
	SharedLinkShareDetails *SharedLinkShareDetails `json:"shared_link_share_details,omitempty"`
	// SharedLinkViewDetails : has no documentation (yet)
	SharedLinkViewDetails *SharedLinkViewDetails `json:"shared_link_view_details,omitempty"`
	// SharedNoteOpenedDetails : has no documentation (yet)
	SharedNoteOpenedDetails *SharedNoteOpenedDetails `json:"shared_note_opened_details,omitempty"`
	// ShmodelDisableDownloadsDetails : has no documentation (yet)
	ShmodelDisableDownloadsDetails *ShmodelDisableDownloadsDetails `json:"shmodel_disable_downloads_details,omitempty"`
	// ShmodelEnableDownloadsDetails : has no documentation (yet)
	ShmodelEnableDownloadsDetails *ShmodelEnableDownloadsDetails `json:"shmodel_enable_downloads_details,omitempty"`
	// ShmodelGroupShareDetails : has no documentation (yet)
	ShmodelGroupShareDetails *ShmodelGroupShareDetails `json:"shmodel_group_share_details,omitempty"`
	// ShowcaseAccessGrantedDetails : has no documentation (yet)
	ShowcaseAccessGrantedDetails *ShowcaseAccessGrantedDetails `json:"showcase_access_granted_details,omitempty"`
	// ShowcaseAddMemberDetails : has no documentation (yet)
	ShowcaseAddMemberDetails *ShowcaseAddMemberDetails `json:"showcase_add_member_details,omitempty"`
	// ShowcaseArchivedDetails : has no documentation (yet)
	ShowcaseArchivedDetails *ShowcaseArchivedDetails `json:"showcase_archived_details,omitempty"`
	// ShowcaseCreatedDetails : has no documentation (yet)
	ShowcaseCreatedDetails *ShowcaseCreatedDetails `json:"showcase_created_details,omitempty"`
	// ShowcaseDeleteCommentDetails : has no documentation (yet)
	ShowcaseDeleteCommentDetails *ShowcaseDeleteCommentDetails `json:"showcase_delete_comment_details,omitempty"`
	// ShowcaseEditedDetails : has no documentation (yet)
	ShowcaseEditedDetails *ShowcaseEditedDetails `json:"showcase_edited_details,omitempty"`
	// ShowcaseEditCommentDetails : has no documentation (yet)
	ShowcaseEditCommentDetails *ShowcaseEditCommentDetails `json:"showcase_edit_comment_details,omitempty"`
	// ShowcaseFileAddedDetails : has no documentation (yet)
	ShowcaseFileAddedDetails *ShowcaseFileAddedDetails `json:"showcase_file_added_details,omitempty"`
	// ShowcaseFileDownloadDetails : has no documentation (yet)
	ShowcaseFileDownloadDetails *ShowcaseFileDownloadDetails `json:"showcase_file_download_details,omitempty"`
	// ShowcaseFileRemovedDetails : has no documentation (yet)
	ShowcaseFileRemovedDetails *ShowcaseFileRemovedDetails `json:"showcase_file_removed_details,omitempty"`
	// ShowcaseFileViewDetails : has no documentation (yet)
	ShowcaseFileViewDetails *ShowcaseFileViewDetails `json:"showcase_file_view_details,omitempty"`
	// ShowcasePermanentlyDeletedDetails : has no documentation (yet)
	ShowcasePermanentlyDeletedDetails *ShowcasePermanentlyDeletedDetails `json:"showcase_permanently_deleted_details,omitempty"`
	// ShowcasePostCommentDetails : has no documentation (yet)
	ShowcasePostCommentDetails *ShowcasePostCommentDetails `json:"showcase_post_comment_details,omitempty"`
	// ShowcaseRemoveMemberDetails : has no documentation (yet)
	ShowcaseRemoveMemberDetails *ShowcaseRemoveMemberDetails `json:"showcase_remove_member_details,omitempty"`
	// ShowcaseRenamedDetails : has no documentation (yet)
	ShowcaseRenamedDetails *ShowcaseRenamedDetails `json:"showcase_renamed_details,omitempty"`
	// ShowcaseRequestAccessDetails : has no documentation (yet)
	ShowcaseRequestAccessDetails *ShowcaseRequestAccessDetails `json:"showcase_request_access_details,omitempty"`
	// ShowcaseResolveCommentDetails : has no documentation (yet)
	ShowcaseResolveCommentDetails *ShowcaseResolveCommentDetails `json:"showcase_resolve_comment_details,omitempty"`
	// ShowcaseRestoredDetails : has no documentation (yet)
	ShowcaseRestoredDetails *ShowcaseRestoredDetails `json:"showcase_restored_details,omitempty"`
	// ShowcaseTrashedDetails : has no documentation (yet)
	ShowcaseTrashedDetails *ShowcaseTrashedDetails `json:"showcase_trashed_details,omitempty"`
	// ShowcaseTrashedDeprecatedDetails : has no documentation (yet)
	ShowcaseTrashedDeprecatedDetails *ShowcaseTrashedDeprecatedDetails `json:"showcase_trashed_deprecated_details,omitempty"`
	// ShowcaseUnresolveCommentDetails : has no documentation (yet)
	ShowcaseUnresolveCommentDetails *ShowcaseUnresolveCommentDetails `json:"showcase_unresolve_comment_details,omitempty"`
	// ShowcaseUntrashedDetails : has no documentation (yet)
	ShowcaseUntrashedDetails *ShowcaseUntrashedDetails `json:"showcase_untrashed_details,omitempty"`
	// ShowcaseUntrashedDeprecatedDetails : has no documentation (yet)
	ShowcaseUntrashedDeprecatedDetails *ShowcaseUntrashedDeprecatedDetails `json:"showcase_untrashed_deprecated_details,omitempty"`
	// ShowcaseViewDetails : has no documentation (yet)
	ShowcaseViewDetails *ShowcaseViewDetails `json:"showcase_view_details,omitempty"`
	// SsoAddCertDetails : has no documentation (yet)
	SsoAddCertDetails *SsoAddCertDetails `json:"sso_add_cert_details,omitempty"`
	// SsoAddLoginUrlDetails : has no documentation (yet)
	SsoAddLoginUrlDetails *SsoAddLoginUrlDetails `json:"sso_add_login_url_details,omitempty"`
	// SsoAddLogoutUrlDetails : has no documentation (yet)
	SsoAddLogoutUrlDetails *SsoAddLogoutUrlDetails `json:"sso_add_logout_url_details,omitempty"`
	// SsoChangeCertDetails : has no documentation (yet)
	SsoChangeCertDetails *SsoChangeCertDetails `json:"sso_change_cert_details,omitempty"`
	// SsoChangeLoginUrlDetails : has no documentation (yet)
	SsoChangeLoginUrlDetails *SsoChangeLoginUrlDetails `json:"sso_change_login_url_details,omitempty"`
	// SsoChangeLogoutUrlDetails : has no documentation (yet)
	SsoChangeLogoutUrlDetails *SsoChangeLogoutUrlDetails `json:"sso_change_logout_url_details,omitempty"`
	// SsoChangeSamlIdentityModeDetails : has no documentation (yet)
	SsoChangeSamlIdentityModeDetails *SsoChangeSamlIdentityModeDetails `json:"sso_change_saml_identity_mode_details,omitempty"`
	// SsoRemoveCertDetails : has no documentation (yet)
	SsoRemoveCertDetails *SsoRemoveCertDetails `json:"sso_remove_cert_details,omitempty"`
	// SsoRemoveLoginUrlDetails : has no documentation (yet)
	SsoRemoveLoginUrlDetails *SsoRemoveLoginUrlDetails `json:"sso_remove_login_url_details,omitempty"`
	// SsoRemoveLogoutUrlDetails : has no documentation (yet)
	SsoRemoveLogoutUrlDetails *SsoRemoveLogoutUrlDetails `json:"sso_remove_logout_url_details,omitempty"`
	// TeamFolderChangeStatusDetails : has no documentation (yet)
	TeamFolderChangeStatusDetails *TeamFolderChangeStatusDetails `json:"team_folder_change_status_details,omitempty"`
	// TeamFolderCreateDetails : has no documentation (yet)
	TeamFolderCreateDetails *TeamFolderCreateDetails `json:"team_folder_create_details,omitempty"`
	// TeamFolderDowngradeDetails : has no documentation (yet)
	TeamFolderDowngradeDetails *TeamFolderDowngradeDetails `json:"team_folder_downgrade_details,omitempty"`
	// TeamFolderPermanentlyDeleteDetails : has no documentation (yet)
	TeamFolderPermanentlyDeleteDetails *TeamFolderPermanentlyDeleteDetails `json:"team_folder_permanently_delete_details,omitempty"`
	// TeamFolderRenameDetails : has no documentation (yet)
	TeamFolderRenameDetails *TeamFolderRenameDetails `json:"team_folder_rename_details,omitempty"`
	// TeamSelectiveSyncSettingsChangedDetails : has no documentation (yet)
	TeamSelectiveSyncSettingsChangedDetails *TeamSelectiveSyncSettingsChangedDetails `json:"team_selective_sync_settings_changed_details,omitempty"`
	// AccountCaptureChangePolicyDetails : has no documentation (yet)
	AccountCaptureChangePolicyDetails *AccountCaptureChangePolicyDetails `json:"account_capture_change_policy_details,omitempty"`
	// AdminEmailRemindersChangedDetails : has no documentation (yet)
	AdminEmailRemindersChangedDetails *AdminEmailRemindersChangedDetails `json:"admin_email_reminders_changed_details,omitempty"`
	// AllowDownloadDisabledDetails : has no documentation (yet)
	AllowDownloadDisabledDetails *AllowDownloadDisabledDetails `json:"allow_download_disabled_details,omitempty"`
	// AllowDownloadEnabledDetails : has no documentation (yet)
	AllowDownloadEnabledDetails *AllowDownloadEnabledDetails `json:"allow_download_enabled_details,omitempty"`
	// AppPermissionsChangedDetails : has no documentation (yet)
	AppPermissionsChangedDetails *AppPermissionsChangedDetails `json:"app_permissions_changed_details,omitempty"`
	// CameraUploadsPolicyChangedDetails : has no documentation (yet)
	CameraUploadsPolicyChangedDetails *CameraUploadsPolicyChangedDetails `json:"camera_uploads_policy_changed_details,omitempty"`
	// CaptureTranscriptPolicyChangedDetails : has no documentation (yet)
	CaptureTranscriptPolicyChangedDetails *CaptureTranscriptPolicyChangedDetails `json:"capture_transcript_policy_changed_details,omitempty"`
	// ClassificationChangePolicyDetails : has no documentation (yet)
	ClassificationChangePolicyDetails *ClassificationChangePolicyDetails `json:"classification_change_policy_details,omitempty"`
	// ComputerBackupPolicyChangedDetails : has no documentation (yet)
	ComputerBackupPolicyChangedDetails *ComputerBackupPolicyChangedDetails `json:"computer_backup_policy_changed_details,omitempty"`
	// ContentAdministrationPolicyChangedDetails : has no documentation (yet)
	ContentAdministrationPolicyChangedDetails *ContentAdministrationPolicyChangedDetails `json:"content_administration_policy_changed_details,omitempty"`
	// DataPlacementRestrictionChangePolicyDetails : has no documentation (yet)
	DataPlacementRestrictionChangePolicyDetails *DataPlacementRestrictionChangePolicyDetails `json:"data_placement_restriction_change_policy_details,omitempty"`
	// DataPlacementRestrictionSatisfyPolicyDetails : has no documentation (yet)
	DataPlacementRestrictionSatisfyPolicyDetails *DataPlacementRestrictionSatisfyPolicyDetails `json:"data_placement_restriction_satisfy_policy_details,omitempty"`
	// DeviceApprovalsAddExceptionDetails : has no documentation (yet)
	DeviceApprovalsAddExceptionDetails *DeviceApprovalsAddExceptionDetails `json:"device_approvals_add_exception_details,omitempty"`
	// DeviceApprovalsChangeDesktopPolicyDetails : has no documentation (yet)
	DeviceApprovalsChangeDesktopPolicyDetails *DeviceApprovalsChangeDesktopPolicyDetails `json:"device_approvals_change_desktop_policy_details,omitempty"`
	// DeviceApprovalsChangeMobilePolicyDetails : has no documentation (yet)
	DeviceApprovalsChangeMobilePolicyDetails *DeviceApprovalsChangeMobilePolicyDetails `json:"device_approvals_change_mobile_policy_details,omitempty"`
	// DeviceApprovalsChangeOverageActionDetails : has no documentation (yet)
	DeviceApprovalsChangeOverageActionDetails *DeviceApprovalsChangeOverageActionDetails `json:"device_approvals_change_overage_action_details,omitempty"`
	// DeviceApprovalsChangeUnlinkActionDetails : has no documentation (yet)
	DeviceApprovalsChangeUnlinkActionDetails *DeviceApprovalsChangeUnlinkActionDetails `json:"device_approvals_change_unlink_action_details,omitempty"`
	// DeviceApprovalsRemoveExceptionDetails : has no documentation (yet)
	DeviceApprovalsRemoveExceptionDetails *DeviceApprovalsRemoveExceptionDetails `json:"device_approvals_remove_exception_details,omitempty"`
	// DirectoryRestrictionsAddMembersDetails : has no documentation (yet)
	DirectoryRestrictionsAddMembersDetails *DirectoryRestrictionsAddMembersDetails `json:"directory_restrictions_add_members_details,omitempty"`
	// DirectoryRestrictionsRemoveMembersDetails : has no documentation (yet)
	DirectoryRestrictionsRemoveMembersDetails *DirectoryRestrictionsRemoveMembersDetails `json:"directory_restrictions_remove_members_details,omitempty"`
	// DropboxPasswordsPolicyChangedDetails : has no documentation (yet)
	DropboxPasswordsPolicyChangedDetails *DropboxPasswordsPolicyChangedDetails `json:"dropbox_passwords_policy_changed_details,omitempty"`
	// EmailIngestPolicyChangedDetails : has no documentation (yet)
	EmailIngestPolicyChangedDetails *EmailIngestPolicyChangedDetails `json:"email_ingest_policy_changed_details,omitempty"`
	// EmmAddExceptionDetails : has no documentation (yet)
	EmmAddExceptionDetails *EmmAddExceptionDetails `json:"emm_add_exception_details,omitempty"`
	// EmmChangePolicyDetails : has no documentation (yet)
	EmmChangePolicyDetails *EmmChangePolicyDetails `json:"emm_change_policy_details,omitempty"`
	// EmmRemoveExceptionDetails : has no documentation (yet)
	EmmRemoveExceptionDetails *EmmRemoveExceptionDetails `json:"emm_remove_exception_details,omitempty"`
	// ExtendedVersionHistoryChangePolicyDetails : has no documentation (yet)
	ExtendedVersionHistoryChangePolicyDetails *ExtendedVersionHistoryChangePolicyDetails `json:"extended_version_history_change_policy_details,omitempty"`
	// ExternalDriveBackupPolicyChangedDetails : has no documentation (yet)
	ExternalDriveBackupPolicyChangedDetails *ExternalDriveBackupPolicyChangedDetails `json:"external_drive_backup_policy_changed_details,omitempty"`
	// FileCommentsChangePolicyDetails : has no documentation (yet)
	FileCommentsChangePolicyDetails *FileCommentsChangePolicyDetails `json:"file_comments_change_policy_details,omitempty"`
	// FileLockingPolicyChangedDetails : has no documentation (yet)
	FileLockingPolicyChangedDetails *FileLockingPolicyChangedDetails `json:"file_locking_policy_changed_details,omitempty"`
	// FileProviderMigrationPolicyChangedDetails : has no documentation (yet)
	FileProviderMigrationPolicyChangedDetails *FileProviderMigrationPolicyChangedDetails `json:"file_provider_migration_policy_changed_details,omitempty"`
	// FileRequestsChangePolicyDetails : has no documentation (yet)
	FileRequestsChangePolicyDetails *FileRequestsChangePolicyDetails `json:"file_requests_change_policy_details,omitempty"`
	// FileRequestsEmailsEnabledDetails : has no documentation (yet)
	FileRequestsEmailsEnabledDetails *FileRequestsEmailsEnabledDetails `json:"file_requests_emails_enabled_details,omitempty"`
	// FileRequestsEmailsRestrictedToTeamOnlyDetails : has no documentation
	// (yet)
	FileRequestsEmailsRestrictedToTeamOnlyDetails *FileRequestsEmailsRestrictedToTeamOnlyDetails `json:"file_requests_emails_restricted_to_team_only_details,omitempty"`
	// FileTransfersPolicyChangedDetails : has no documentation (yet)
	FileTransfersPolicyChangedDetails *FileTransfersPolicyChangedDetails `json:"file_transfers_policy_changed_details,omitempty"`
	// GoogleSsoChangePolicyDetails : has no documentation (yet)
	GoogleSsoChangePolicyDetails *GoogleSsoChangePolicyDetails `json:"google_sso_change_policy_details,omitempty"`
	// GroupUserManagementChangePolicyDetails : has no documentation (yet)
	GroupUserManagementChangePolicyDetails *GroupUserManagementChangePolicyDetails `json:"group_user_management_change_policy_details,omitempty"`
	// IntegrationPolicyChangedDetails : has no documentation (yet)
	IntegrationPolicyChangedDetails *IntegrationPolicyChangedDetails `json:"integration_policy_changed_details,omitempty"`
	// InviteAcceptanceEmailPolicyChangedDetails : has no documentation (yet)
	InviteAcceptanceEmailPolicyChangedDetails *InviteAcceptanceEmailPolicyChangedDetails `json:"invite_acceptance_email_policy_changed_details,omitempty"`
	// MemberRequestsChangePolicyDetails : has no documentation (yet)
	MemberRequestsChangePolicyDetails *MemberRequestsChangePolicyDetails `json:"member_requests_change_policy_details,omitempty"`
	// MemberSendInvitePolicyChangedDetails : has no documentation (yet)
	MemberSendInvitePolicyChangedDetails *MemberSendInvitePolicyChangedDetails `json:"member_send_invite_policy_changed_details,omitempty"`
	// MemberSpaceLimitsAddExceptionDetails : has no documentation (yet)
	MemberSpaceLimitsAddExceptionDetails *MemberSpaceLimitsAddExceptionDetails `json:"member_space_limits_add_exception_details,omitempty"`
	// MemberSpaceLimitsChangeCapsTypePolicyDetails : has no documentation (yet)
	MemberSpaceLimitsChangeCapsTypePolicyDetails *MemberSpaceLimitsChangeCapsTypePolicyDetails `json:"member_space_limits_change_caps_type_policy_details,omitempty"`
	// MemberSpaceLimitsChangePolicyDetails : has no documentation (yet)
	MemberSpaceLimitsChangePolicyDetails *MemberSpaceLimitsChangePolicyDetails `json:"member_space_limits_change_policy_details,omitempty"`
	// MemberSpaceLimitsRemoveExceptionDetails : has no documentation (yet)
	MemberSpaceLimitsRemoveExceptionDetails *MemberSpaceLimitsRemoveExceptionDetails `json:"member_space_limits_remove_exception_details,omitempty"`
	// MemberSuggestionsChangePolicyDetails : has no documentation (yet)
	MemberSuggestionsChangePolicyDetails *MemberSuggestionsChangePolicyDetails `json:"member_suggestions_change_policy_details,omitempty"`
	// MicrosoftOfficeAddinChangePolicyDetails : has no documentation (yet)
	MicrosoftOfficeAddinChangePolicyDetails *MicrosoftOfficeAddinChangePolicyDetails `json:"microsoft_office_addin_change_policy_details,omitempty"`
	// NetworkControlChangePolicyDetails : has no documentation (yet)
	NetworkControlChangePolicyDetails *NetworkControlChangePolicyDetails `json:"network_control_change_policy_details,omitempty"`
	// PaperChangeDeploymentPolicyDetails : has no documentation (yet)
	PaperChangeDeploymentPolicyDetails *PaperChangeDeploymentPolicyDetails `json:"paper_change_deployment_policy_details,omitempty"`
	// PaperChangeMemberLinkPolicyDetails : has no documentation (yet)
	PaperChangeMemberLinkPolicyDetails *PaperChangeMemberLinkPolicyDetails `json:"paper_change_member_link_policy_details,omitempty"`
	// PaperChangeMemberPolicyDetails : has no documentation (yet)
	PaperChangeMemberPolicyDetails *PaperChangeMemberPolicyDetails `json:"paper_change_member_policy_details,omitempty"`
	// PaperChangePolicyDetails : has no documentation (yet)
	PaperChangePolicyDetails *PaperChangePolicyDetails `json:"paper_change_policy_details,omitempty"`
	// PaperDefaultFolderPolicyChangedDetails : has no documentation (yet)
	PaperDefaultFolderPolicyChangedDetails *PaperDefaultFolderPolicyChangedDetails `json:"paper_default_folder_policy_changed_details,omitempty"`
	// PaperDesktopPolicyChangedDetails : has no documentation (yet)
	PaperDesktopPolicyChangedDetails *PaperDesktopPolicyChangedDetails `json:"paper_desktop_policy_changed_details,omitempty"`
	// PaperEnabledUsersGroupAdditionDetails : has no documentation (yet)
	PaperEnabledUsersGroupAdditionDetails *PaperEnabledUsersGroupAdditionDetails `json:"paper_enabled_users_group_addition_details,omitempty"`
	// PaperEnabledUsersGroupRemovalDetails : has no documentation (yet)
	PaperEnabledUsersGroupRemovalDetails *PaperEnabledUsersGroupRemovalDetails `json:"paper_enabled_users_group_removal_details,omitempty"`
	// PasswordStrengthRequirementsChangePolicyDetails : has no documentation
	// (yet)
	PasswordStrengthRequirementsChangePolicyDetails *PasswordStrengthRequirementsChangePolicyDetails `json:"password_strength_requirements_change_policy_details,omitempty"`
	// PermanentDeleteChangePolicyDetails : has no documentation (yet)
	PermanentDeleteChangePolicyDetails *PermanentDeleteChangePolicyDetails `json:"permanent_delete_change_policy_details,omitempty"`
	// ResellerSupportChangePolicyDetails : has no documentation (yet)
	ResellerSupportChangePolicyDetails *ResellerSupportChangePolicyDetails `json:"reseller_support_change_policy_details,omitempty"`
	// RewindPolicyChangedDetails : has no documentation (yet)
	RewindPolicyChangedDetails *RewindPolicyChangedDetails `json:"rewind_policy_changed_details,omitempty"`
	// SendForSignaturePolicyChangedDetails : has no documentation (yet)
	SendForSignaturePolicyChangedDetails *SendForSignaturePolicyChangedDetails `json:"send_for_signature_policy_changed_details,omitempty"`
	// SharingChangeFolderJoinPolicyDetails : has no documentation (yet)
	SharingChangeFolderJoinPolicyDetails *SharingChangeFolderJoinPolicyDetails `json:"sharing_change_folder_join_policy_details,omitempty"`
	// SharingChangeLinkAllowChangeExpirationPolicyDetails : has no
	// documentation (yet)
	SharingChangeLinkAllowChangeExpirationPolicyDetails *SharingChangeLinkAllowChangeExpirationPolicyDetails `json:"sharing_change_link_allow_change_expiration_policy_details,omitempty"`
	// SharingChangeLinkDefaultExpirationPolicyDetails : has no documentation
	// (yet)
	SharingChangeLinkDefaultExpirationPolicyDetails *SharingChangeLinkDefaultExpirationPolicyDetails `json:"sharing_change_link_default_expiration_policy_details,omitempty"`
	// SharingChangeLinkEnforcePasswordPolicyDetails : has no documentation
	// (yet)
	SharingChangeLinkEnforcePasswordPolicyDetails *SharingChangeLinkEnforcePasswordPolicyDetails `json:"sharing_change_link_enforce_password_policy_details,omitempty"`
	// SharingChangeLinkPolicyDetails : has no documentation (yet)
	SharingChangeLinkPolicyDetails *SharingChangeLinkPolicyDetails `json:"sharing_change_link_policy_details,omitempty"`
	// SharingChangeMemberPolicyDetails : has no documentation (yet)
	SharingChangeMemberPolicyDetails *SharingChangeMemberPolicyDetails `json:"sharing_change_member_policy_details,omitempty"`
	// ShowcaseChangeDownloadPolicyDetails : has no documentation (yet)
	ShowcaseChangeDownloadPolicyDetails *ShowcaseChangeDownloadPolicyDetails `json:"showcase_change_download_policy_details,omitempty"`
	// ShowcaseChangeEnabledPolicyDetails : has no documentation (yet)
	ShowcaseChangeEnabledPolicyDetails *ShowcaseChangeEnabledPolicyDetails `json:"showcase_change_enabled_policy_details,omitempty"`
	// ShowcaseChangeExternalSharingPolicyDetails : has no documentation (yet)
	ShowcaseChangeExternalSharingPolicyDetails *ShowcaseChangeExternalSharingPolicyDetails `json:"showcase_change_external_sharing_policy_details,omitempty"`
	// SmarterSmartSyncPolicyChangedDetails : has no documentation (yet)
	SmarterSmartSyncPolicyChangedDetails *SmarterSmartSyncPolicyChangedDetails `json:"smarter_smart_sync_policy_changed_details,omitempty"`
	// SmartSyncChangePolicyDetails : has no documentation (yet)
	SmartSyncChangePolicyDetails *SmartSyncChangePolicyDetails `json:"smart_sync_change_policy_details,omitempty"`
	// SmartSyncNotOptOutDetails : has no documentation (yet)
	SmartSyncNotOptOutDetails *SmartSyncNotOptOutDetails `json:"smart_sync_not_opt_out_details,omitempty"`
	// SmartSyncOptOutDetails : has no documentation (yet)
	SmartSyncOptOutDetails *SmartSyncOptOutDetails `json:"smart_sync_opt_out_details,omitempty"`
	// SsoChangePolicyDetails : has no documentation (yet)
	SsoChangePolicyDetails *SsoChangePolicyDetails `json:"sso_change_policy_details,omitempty"`
	// TeamBrandingPolicyChangedDetails : has no documentation (yet)
	TeamBrandingPolicyChangedDetails *TeamBrandingPolicyChangedDetails `json:"team_branding_policy_changed_details,omitempty"`
	// TeamExtensionsPolicyChangedDetails : has no documentation (yet)
	TeamExtensionsPolicyChangedDetails *TeamExtensionsPolicyChangedDetails `json:"team_extensions_policy_changed_details,omitempty"`
	// TeamSelectiveSyncPolicyChangedDetails : has no documentation (yet)
	TeamSelectiveSyncPolicyChangedDetails *TeamSelectiveSyncPolicyChangedDetails `json:"team_selective_sync_policy_changed_details,omitempty"`
	// TeamSharingWhitelistSubjectsChangedDetails : has no documentation (yet)
	TeamSharingWhitelistSubjectsChangedDetails *TeamSharingWhitelistSubjectsChangedDetails `json:"team_sharing_whitelist_subjects_changed_details,omitempty"`
	// TfaAddExceptionDetails : has no documentation (yet)
	TfaAddExceptionDetails *TfaAddExceptionDetails `json:"tfa_add_exception_details,omitempty"`
	// TfaChangePolicyDetails : has no documentation (yet)
	TfaChangePolicyDetails *TfaChangePolicyDetails `json:"tfa_change_policy_details,omitempty"`
	// TfaRemoveExceptionDetails : has no documentation (yet)
	TfaRemoveExceptionDetails *TfaRemoveExceptionDetails `json:"tfa_remove_exception_details,omitempty"`
	// TwoAccountChangePolicyDetails : has no documentation (yet)
	TwoAccountChangePolicyDetails *TwoAccountChangePolicyDetails `json:"two_account_change_policy_details,omitempty"`
	// ViewerInfoPolicyChangedDetails : has no documentation (yet)
	ViewerInfoPolicyChangedDetails *ViewerInfoPolicyChangedDetails `json:"viewer_info_policy_changed_details,omitempty"`
	// WatermarkingPolicyChangedDetails : has no documentation (yet)
	WatermarkingPolicyChangedDetails *WatermarkingPolicyChangedDetails `json:"watermarking_policy_changed_details,omitempty"`
	// WebSessionsChangeActiveSessionLimitDetails : has no documentation (yet)
	WebSessionsChangeActiveSessionLimitDetails *WebSessionsChangeActiveSessionLimitDetails `json:"web_sessions_change_active_session_limit_details,omitempty"`
	// WebSessionsChangeFixedLengthPolicyDetails : has no documentation (yet)
	WebSessionsChangeFixedLengthPolicyDetails *WebSessionsChangeFixedLengthPolicyDetails `json:"web_sessions_change_fixed_length_policy_details,omitempty"`
	// WebSessionsChangeIdleLengthPolicyDetails : has no documentation (yet)
	WebSessionsChangeIdleLengthPolicyDetails *WebSessionsChangeIdleLengthPolicyDetails `json:"web_sessions_change_idle_length_policy_details,omitempty"`
	// DataResidencyMigrationRequestSuccessfulDetails : has no documentation
	// (yet)
	DataResidencyMigrationRequestSuccessfulDetails *DataResidencyMigrationRequestSuccessfulDetails `json:"data_residency_migration_request_successful_details,omitempty"`
	// DataResidencyMigrationRequestUnsuccessfulDetails : has no documentation
	// (yet)
	DataResidencyMigrationRequestUnsuccessfulDetails *DataResidencyMigrationRequestUnsuccessfulDetails `json:"data_residency_migration_request_unsuccessful_details,omitempty"`
	// TeamMergeFromDetails : has no documentation (yet)
	TeamMergeFromDetails *TeamMergeFromDetails `json:"team_merge_from_details,omitempty"`
	// TeamMergeToDetails : has no documentation (yet)
	TeamMergeToDetails *TeamMergeToDetails `json:"team_merge_to_details,omitempty"`
	// TeamProfileAddBackgroundDetails : has no documentation (yet)
	TeamProfileAddBackgroundDetails *TeamProfileAddBackgroundDetails `json:"team_profile_add_background_details,omitempty"`
	// TeamProfileAddLogoDetails : has no documentation (yet)
	TeamProfileAddLogoDetails *TeamProfileAddLogoDetails `json:"team_profile_add_logo_details,omitempty"`
	// TeamProfileChangeBackgroundDetails : has no documentation (yet)
	TeamProfileChangeBackgroundDetails *TeamProfileChangeBackgroundDetails `json:"team_profile_change_background_details,omitempty"`
	// TeamProfileChangeDefaultLanguageDetails : has no documentation (yet)
	TeamProfileChangeDefaultLanguageDetails *TeamProfileChangeDefaultLanguageDetails `json:"team_profile_change_default_language_details,omitempty"`
	// TeamProfileChangeLogoDetails : has no documentation (yet)
	TeamProfileChangeLogoDetails *TeamProfileChangeLogoDetails `json:"team_profile_change_logo_details,omitempty"`
	// TeamProfileChangeNameDetails : has no documentation (yet)
	TeamProfileChangeNameDetails *TeamProfileChangeNameDetails `json:"team_profile_change_name_details,omitempty"`
	// TeamProfileRemoveBackgroundDetails : has no documentation (yet)
	TeamProfileRemoveBackgroundDetails *TeamProfileRemoveBackgroundDetails `json:"team_profile_remove_background_details,omitempty"`
	// TeamProfileRemoveLogoDetails : has no documentation (yet)
	TeamProfileRemoveLogoDetails *TeamProfileRemoveLogoDetails `json:"team_profile_remove_logo_details,omitempty"`
	// TfaAddBackupPhoneDetails : has no documentation (yet)
	TfaAddBackupPhoneDetails *TfaAddBackupPhoneDetails `json:"tfa_add_backup_phone_details,omitempty"`
	// TfaAddSecurityKeyDetails : has no documentation (yet)
	TfaAddSecurityKeyDetails *TfaAddSecurityKeyDetails `json:"tfa_add_security_key_details,omitempty"`
	// TfaChangeBackupPhoneDetails : has no documentation (yet)
	TfaChangeBackupPhoneDetails *TfaChangeBackupPhoneDetails `json:"tfa_change_backup_phone_details,omitempty"`
	// TfaChangeStatusDetails : has no documentation (yet)
	TfaChangeStatusDetails *TfaChangeStatusDetails `json:"tfa_change_status_details,omitempty"`
	// TfaRemoveBackupPhoneDetails : has no documentation (yet)
	TfaRemoveBackupPhoneDetails *TfaRemoveBackupPhoneDetails `json:"tfa_remove_backup_phone_details,omitempty"`
	// TfaRemoveSecurityKeyDetails : has no documentation (yet)
	TfaRemoveSecurityKeyDetails *TfaRemoveSecurityKeyDetails `json:"tfa_remove_security_key_details,omitempty"`
	// TfaResetDetails : has no documentation (yet)
	TfaResetDetails *TfaResetDetails `json:"tfa_reset_details,omitempty"`
	// ChangedEnterpriseAdminRoleDetails : has no documentation (yet)
	ChangedEnterpriseAdminRoleDetails *ChangedEnterpriseAdminRoleDetails `json:"changed_enterprise_admin_role_details,omitempty"`
	// ChangedEnterpriseConnectedTeamStatusDetails : has no documentation (yet)
	ChangedEnterpriseConnectedTeamStatusDetails *ChangedEnterpriseConnectedTeamStatusDetails `json:"changed_enterprise_connected_team_status_details,omitempty"`
	// EndedEnterpriseAdminSessionDetails : has no documentation (yet)
	EndedEnterpriseAdminSessionDetails *EndedEnterpriseAdminSessionDetails `json:"ended_enterprise_admin_session_details,omitempty"`
	// EndedEnterpriseAdminSessionDeprecatedDetails : has no documentation (yet)
	EndedEnterpriseAdminSessionDeprecatedDetails *EndedEnterpriseAdminSessionDeprecatedDetails `json:"ended_enterprise_admin_session_deprecated_details,omitempty"`
	// EnterpriseSettingsLockingDetails : has no documentation (yet)
	EnterpriseSettingsLockingDetails *EnterpriseSettingsLockingDetails `json:"enterprise_settings_locking_details,omitempty"`
	// GuestAdminChangeStatusDetails : has no documentation (yet)
	GuestAdminChangeStatusDetails *GuestAdminChangeStatusDetails `json:"guest_admin_change_status_details,omitempty"`
	// StartedEnterpriseAdminSessionDetails : has no documentation (yet)
	StartedEnterpriseAdminSessionDetails *StartedEnterpriseAdminSessionDetails `json:"started_enterprise_admin_session_details,omitempty"`
	// TeamMergeRequestAcceptedDetails : has no documentation (yet)
	TeamMergeRequestAcceptedDetails *TeamMergeRequestAcceptedDetails `json:"team_merge_request_accepted_details,omitempty"`
	// TeamMergeRequestAcceptedShownToPrimaryTeamDetails : has no documentation
	// (yet)
	TeamMergeRequestAcceptedShownToPrimaryTeamDetails *TeamMergeRequestAcceptedShownToPrimaryTeamDetails `json:"team_merge_request_accepted_shown_to_primary_team_details,omitempty"`
	// TeamMergeRequestAcceptedShownToSecondaryTeamDetails : has no
	// documentation (yet)
	TeamMergeRequestAcceptedShownToSecondaryTeamDetails *TeamMergeRequestAcceptedShownToSecondaryTeamDetails `json:"team_merge_request_accepted_shown_to_secondary_team_details,omitempty"`
	// TeamMergeRequestAutoCanceledDetails : has no documentation (yet)
	TeamMergeRequestAutoCanceledDetails *TeamMergeRequestAutoCanceledDetails `json:"team_merge_request_auto_canceled_details,omitempty"`
	// TeamMergeRequestCanceledDetails : has no documentation (yet)
	TeamMergeRequestCanceledDetails *TeamMergeRequestCanceledDetails `json:"team_merge_request_canceled_details,omitempty"`
	// TeamMergeRequestCanceledShownToPrimaryTeamDetails : has no documentation
	// (yet)
	TeamMergeRequestCanceledShownToPrimaryTeamDetails *TeamMergeRequestCanceledShownToPrimaryTeamDetails `json:"team_merge_request_canceled_shown_to_primary_team_details,omitempty"`
	// TeamMergeRequestCanceledShownToSecondaryTeamDetails : has no
	// documentation (yet)
	TeamMergeRequestCanceledShownToSecondaryTeamDetails *TeamMergeRequestCanceledShownToSecondaryTeamDetails `json:"team_merge_request_canceled_shown_to_secondary_team_details,omitempty"`
	// TeamMergeRequestExpiredDetails : has no documentation (yet)
	TeamMergeRequestExpiredDetails *TeamMergeRequestExpiredDetails `json:"team_merge_request_expired_details,omitempty"`
	// TeamMergeRequestExpiredShownToPrimaryTeamDetails : has no documentation
	// (yet)
	TeamMergeRequestExpiredShownToPrimaryTeamDetails *TeamMergeRequestExpiredShownToPrimaryTeamDetails `json:"team_merge_request_expired_shown_to_primary_team_details,omitempty"`
	// TeamMergeRequestExpiredShownToSecondaryTeamDetails : has no documentation
	// (yet)
	TeamMergeRequestExpiredShownToSecondaryTeamDetails *TeamMergeRequestExpiredShownToSecondaryTeamDetails `json:"team_merge_request_expired_shown_to_secondary_team_details,omitempty"`
	// TeamMergeRequestRejectedShownToPrimaryTeamDetails : has no documentation
	// (yet)
	TeamMergeRequestRejectedShownToPrimaryTeamDetails *TeamMergeRequestRejectedShownToPrimaryTeamDetails `json:"team_merge_request_rejected_shown_to_primary_team_details,omitempty"`
	// TeamMergeRequestRejectedShownToSecondaryTeamDetails : has no
	// documentation (yet)
	TeamMergeRequestRejectedShownToSecondaryTeamDetails *TeamMergeRequestRejectedShownToSecondaryTeamDetails `json:"team_merge_request_rejected_shown_to_secondary_team_details,omitempty"`
	// TeamMergeRequestReminderDetails : has no documentation (yet)
	TeamMergeRequestReminderDetails *TeamMergeRequestReminderDetails `json:"team_merge_request_reminder_details,omitempty"`
	// TeamMergeRequestReminderShownToPrimaryTeamDetails : has no documentation
	// (yet)
	TeamMergeRequestReminderShownToPrimaryTeamDetails *TeamMergeRequestReminderShownToPrimaryTeamDetails `json:"team_merge_request_reminder_shown_to_primary_team_details,omitempty"`
	// TeamMergeRequestReminderShownToSecondaryTeamDetails : has no
	// documentation (yet)
	TeamMergeRequestReminderShownToSecondaryTeamDetails *TeamMergeRequestReminderShownToSecondaryTeamDetails `json:"team_merge_request_reminder_shown_to_secondary_team_details,omitempty"`
	// TeamMergeRequestRevokedDetails : has no documentation (yet)
	TeamMergeRequestRevokedDetails *TeamMergeRequestRevokedDetails `json:"team_merge_request_revoked_details,omitempty"`
	// TeamMergeRequestSentShownToPrimaryTeamDetails : has no documentation
	// (yet)
	TeamMergeRequestSentShownToPrimaryTeamDetails *TeamMergeRequestSentShownToPrimaryTeamDetails `json:"team_merge_request_sent_shown_to_primary_team_details,omitempty"`
	// TeamMergeRequestSentShownToSecondaryTeamDetails : has no documentation
	// (yet)
	TeamMergeRequestSentShownToSecondaryTeamDetails *TeamMergeRequestSentShownToSecondaryTeamDetails `json:"team_merge_request_sent_shown_to_secondary_team_details,omitempty"`
	// MissingDetails : Hints that this event was returned with missing details
	// due to an internal error.
	MissingDetails *MissingDetails `json:"missing_details,omitempty"`
}

// Valid tag values for EventDetails
const (
	EventDetailsAdminAlertingAlertStateChangedDetails               = "admin_alerting_alert_state_changed_details"
	EventDetailsAdminAlertingChangedAlertConfigDetails              = "admin_alerting_changed_alert_config_details"
	EventDetailsAdminAlertingTriggeredAlertDetails                  = "admin_alerting_triggered_alert_details"
	EventDetailsAppBlockedByPermissionsDetails                      = "app_blocked_by_permissions_details"
	EventDetailsAppLinkTeamDetails                                  = "app_link_team_details"
	EventDetailsAppLinkUserDetails                                  = "app_link_user_details"
	EventDetailsAppUnlinkTeamDetails                                = "app_unlink_team_details"
	EventDetailsAppUnlinkUserDetails                                = "app_unlink_user_details"
	EventDetailsIntegrationConnectedDetails                         = "integration_connected_details"
	EventDetailsIntegrationDisconnectedDetails                      = "integration_disconnected_details"
	EventDetailsFileAddCommentDetails                               = "file_add_comment_details"
	EventDetailsFileChangeCommentSubscriptionDetails                = "file_change_comment_subscription_details"
	EventDetailsFileDeleteCommentDetails                            = "file_delete_comment_details"
	EventDetailsFileEditCommentDetails                              = "file_edit_comment_details"
	EventDetailsFileLikeCommentDetails                              = "file_like_comment_details"
	EventDetailsFileResolveCommentDetails                           = "file_resolve_comment_details"
	EventDetailsFileUnlikeCommentDetails                            = "file_unlike_comment_details"
	EventDetailsFileUnresolveCommentDetails                         = "file_unresolve_comment_details"
	EventDetailsGovernancePolicyAddFoldersDetails                   = "governance_policy_add_folders_details"
	EventDetailsGovernancePolicyAddFolderFailedDetails              = "governance_policy_add_folder_failed_details"
	EventDetailsGovernancePolicyContentDisposedDetails              = "governance_policy_content_disposed_details"
	EventDetailsGovernancePolicyCreateDetails                       = "governance_policy_create_details"
	EventDetailsGovernancePolicyDeleteDetails                       = "governance_policy_delete_details"
	EventDetailsGovernancePolicyEditDetailsDetails                  = "governance_policy_edit_details_details"
	EventDetailsGovernancePolicyEditDurationDetails                 = "governance_policy_edit_duration_details"
	EventDetailsGovernancePolicyExportCreatedDetails                = "governance_policy_export_created_details"
	EventDetailsGovernancePolicyExportRemovedDetails                = "governance_policy_export_removed_details"
	EventDetailsGovernancePolicyRemoveFoldersDetails                = "governance_policy_remove_folders_details"
	EventDetailsGovernancePolicyReportCreatedDetails                = "governance_policy_report_created_details"
	EventDetailsGovernancePolicyZipPartDownloadedDetails            = "governance_policy_zip_part_downloaded_details"
	EventDetailsLegalHoldsActivateAHoldDetails                      = "legal_holds_activate_a_hold_details"
	EventDetailsLegalHoldsAddMembersDetails                         = "legal_holds_add_members_details"
	EventDetailsLegalHoldsChangeHoldDetailsDetails                  = "legal_holds_change_hold_details_details"
	EventDetailsLegalHoldsChangeHoldNameDetails                     = "legal_holds_change_hold_name_details"
	EventDetailsLegalHoldsExportAHoldDetails                        = "legal_holds_export_a_hold_details"
	EventDetailsLegalHoldsExportCancelledDetails                    = "legal_holds_export_cancelled_details"
	EventDetailsLegalHoldsExportDownloadedDetails                   = "legal_holds_export_downloaded_details"
	EventDetailsLegalHoldsExportRemovedDetails                      = "legal_holds_export_removed_details"
	EventDetailsLegalHoldsReleaseAHoldDetails                       = "legal_holds_release_a_hold_details"
	EventDetailsLegalHoldsRemoveMembersDetails                      = "legal_holds_remove_members_details"
	EventDetailsLegalHoldsReportAHoldDetails                        = "legal_holds_report_a_hold_details"
	EventDetailsDeviceChangeIpDesktopDetails                        = "device_change_ip_desktop_details"
	EventDetailsDeviceChangeIpMobileDetails                         = "device_change_ip_mobile_details"
	EventDetailsDeviceChangeIpWebDetails                            = "device_change_ip_web_details"
	EventDetailsDeviceDeleteOnUnlinkFailDetails                     = "device_delete_on_unlink_fail_details"
	EventDetailsDeviceDeleteOnUnlinkSuccessDetails                  = "device_delete_on_unlink_success_details"
	EventDetailsDeviceLinkFailDetails                               = "device_link_fail_details"
	EventDetailsDeviceLinkSuccessDetails                            = "device_link_success_details"
	EventDetailsDeviceManagementDisabledDetails                     = "device_management_disabled_details"
	EventDetailsDeviceManagementEnabledDetails                      = "device_management_enabled_details"
	EventDetailsDeviceSyncBackupStatusChangedDetails                = "device_sync_backup_status_changed_details"
	EventDetailsDeviceUnlinkDetails                                 = "device_unlink_details"
	EventDetailsDropboxPasswordsExportedDetails                     = "dropbox_passwords_exported_details"
	EventDetailsDropboxPasswordsNewDeviceEnrolledDetails            = "dropbox_passwords_new_device_enrolled_details"
	EventDetailsEmmRefreshAuthTokenDetails                          = "emm_refresh_auth_token_details"
	EventDetailsExternalDriveBackupEligibilityStatusCheckedDetails  = "external_drive_backup_eligibility_status_checked_details"
	EventDetailsExternalDriveBackupStatusChangedDetails             = "external_drive_backup_status_changed_details"
	EventDetailsAccountCaptureChangeAvailabilityDetails             = "account_capture_change_availability_details"
	EventDetailsAccountCaptureMigrateAccountDetails                 = "account_capture_migrate_account_details"
	EventDetailsAccountCaptureNotificationEmailsSentDetails         = "account_capture_notification_emails_sent_details"
	EventDetailsAccountCaptureRelinquishAccountDetails              = "account_capture_relinquish_account_details"
	EventDetailsDisabledDomainInvitesDetails                        = "disabled_domain_invites_details"
	EventDetailsDomainInvitesApproveRequestToJoinTeamDetails        = "domain_invites_approve_request_to_join_team_details"
	EventDetailsDomainInvitesDeclineRequestToJoinTeamDetails        = "domain_invites_decline_request_to_join_team_details"
	EventDetailsDomainInvitesEmailExistingUsersDetails              = "domain_invites_email_existing_users_details"
	EventDetailsDomainInvitesRequestToJoinTeamDetails               = "domain_invites_request_to_join_team_details"
	EventDetailsDomainInvitesSetInviteNewUserPrefToNoDetails        = "domain_invites_set_invite_new_user_pref_to_no_details"
	EventDetailsDomainInvitesSetInviteNewUserPrefToYesDetails       = "domain_invites_set_invite_new_user_pref_to_yes_details"
	EventDetailsDomainVerificationAddDomainFailDetails              = "domain_verification_add_domain_fail_details"
	EventDetailsDomainVerificationAddDomainSuccessDetails           = "domain_verification_add_domain_success_details"
	EventDetailsDomainVerificationRemoveDomainDetails               = "domain_verification_remove_domain_details"
	EventDetailsEnabledDomainInvitesDetails                         = "enabled_domain_invites_details"
	EventDetailsApplyNamingConventionDetails                        = "apply_naming_convention_details"
	EventDetailsCreateFolderDetails                                 = "create_folder_details"
	EventDetailsFileAddDetails                                      = "file_add_details"
	EventDetailsFileCopyDetails                                     = "file_copy_details"
	EventDetailsFileDeleteDetails                                   = "file_delete_details"
	EventDetailsFileDownloadDetails                                 = "file_download_details"
	EventDetailsFileEditDetails                                     = "file_edit_details"
	EventDetailsFileGetCopyReferenceDetails                         = "file_get_copy_reference_details"
	EventDetailsFileLockingLockStatusChangedDetails                 = "file_locking_lock_status_changed_details"
	EventDetailsFileMoveDetails                                     = "file_move_details"
	EventDetailsFilePermanentlyDeleteDetails                        = "file_permanently_delete_details"
	EventDetailsFilePreviewDetails                                  = "file_preview_details"
	EventDetailsFileRenameDetails                                   = "file_rename_details"
	EventDetailsFileRestoreDetails                                  = "file_restore_details"
	EventDetailsFileRevertDetails                                   = "file_revert_details"
	EventDetailsFileRollbackChangesDetails                          = "file_rollback_changes_details"
	EventDetailsFileSaveCopyReferenceDetails                        = "file_save_copy_reference_details"
	EventDetailsFolderOverviewDescriptionChangedDetails             = "folder_overview_description_changed_details"
	EventDetailsFolderOverviewItemPinnedDetails                     = "folder_overview_item_pinned_details"
	EventDetailsFolderOverviewItemUnpinnedDetails                   = "folder_overview_item_unpinned_details"
	EventDetailsObjectLabelAddedDetails                             = "object_label_added_details"
	EventDetailsObjectLabelRemovedDetails                           = "object_label_removed_details"
	EventDetailsObjectLabelUpdatedValueDetails                      = "object_label_updated_value_details"
	EventDetailsOrganizeFolderWithTidyDetails                       = "organize_folder_with_tidy_details"
	EventDetailsRewindFolderDetails                                 = "rewind_folder_details"
	EventDetailsUndoNamingConventionDetails                         = "undo_naming_convention_details"
	EventDetailsUndoOrganizeFolderWithTidyDetails                   = "undo_organize_folder_with_tidy_details"
	EventDetailsUserTagsAddedDetails                                = "user_tags_added_details"
	EventDetailsUserTagsRemovedDetails                              = "user_tags_removed_details"
	EventDetailsEmailIngestReceiveFileDetails                       = "email_ingest_receive_file_details"
	EventDetailsFileRequestChangeDetails                            = "file_request_change_details"
	EventDetailsFileRequestCloseDetails                             = "file_request_close_details"
	EventDetailsFileRequestCreateDetails                            = "file_request_create_details"
	EventDetailsFileRequestDeleteDetails                            = "file_request_delete_details"
	EventDetailsFileRequestReceiveFileDetails                       = "file_request_receive_file_details"
	EventDetailsGroupAddExternalIdDetails                           = "group_add_external_id_details"
	EventDetailsGroupAddMemberDetails                               = "group_add_member_details"
	EventDetailsGroupChangeExternalIdDetails                        = "group_change_external_id_details"
	EventDetailsGroupChangeManagementTypeDetails                    = "group_change_management_type_details"
	EventDetailsGroupChangeMemberRoleDetails                        = "group_change_member_role_details"
	EventDetailsGroupCreateDetails                                  = "group_create_details"
	EventDetailsGroupDeleteDetails                                  = "group_delete_details"
	EventDetailsGroupDescriptionUpdatedDetails                      = "group_description_updated_details"
	EventDetailsGroupJoinPolicyUpdatedDetails                       = "group_join_policy_updated_details"
	EventDetailsGroupMovedDetails                                   = "group_moved_details"
	EventDetailsGroupRemoveExternalIdDetails                        = "group_remove_external_id_details"
	EventDetailsGroupRemoveMemberDetails                            = "group_remove_member_details"
	EventDetailsGroupRenameDetails                                  = "group_rename_details"
	EventDetailsAccountLockOrUnlockedDetails                        = "account_lock_or_unlocked_details"
	EventDetailsEmmErrorDetails                                     = "emm_error_details"
	EventDetailsGuestAdminSignedInViaTrustedTeamsDetails            = "guest_admin_signed_in_via_trusted_teams_details"
	EventDetailsGuestAdminSignedOutViaTrustedTeamsDetails           = "guest_admin_signed_out_via_trusted_teams_details"
	EventDetailsLoginFailDetails                                    = "login_fail_details"
	EventDetailsLoginSuccessDetails                                 = "login_success_details"
	EventDetailsLogoutDetails                                       = "logout_details"
	EventDetailsResellerSupportSessionEndDetails                    = "reseller_support_session_end_details"
	EventDetailsResellerSupportSessionStartDetails                  = "reseller_support_session_start_details"
	EventDetailsSignInAsSessionEndDetails                           = "sign_in_as_session_end_details"
	EventDetailsSignInAsSessionStartDetails                         = "sign_in_as_session_start_details"
	EventDetailsSsoErrorDetails                                     = "sso_error_details"
	EventDetailsCreateTeamInviteLinkDetails                         = "create_team_invite_link_details"
	EventDetailsDeleteTeamInviteLinkDetails                         = "delete_team_invite_link_details"
	EventDetailsMemberAddExternalIdDetails                          = "member_add_external_id_details"
	EventDetailsMemberAddNameDetails                                = "member_add_name_details"
	EventDetailsMemberChangeAdminRoleDetails                        = "member_change_admin_role_details"
	EventDetailsMemberChangeEmailDetails                            = "member_change_email_details"
	EventDetailsMemberChangeExternalIdDetails                       = "member_change_external_id_details"
	EventDetailsMemberChangeMembershipTypeDetails                   = "member_change_membership_type_details"
	EventDetailsMemberChangeNameDetails                             = "member_change_name_details"
	EventDetailsMemberChangeResellerRoleDetails                     = "member_change_reseller_role_details"
	EventDetailsMemberChangeStatusDetails                           = "member_change_status_details"
	EventDetailsMemberDeleteManualContactsDetails                   = "member_delete_manual_contacts_details"
	EventDetailsMemberDeleteProfilePhotoDetails                     = "member_delete_profile_photo_details"
	EventDetailsMemberPermanentlyDeleteAccountContentsDetails       = "member_permanently_delete_account_contents_details"
	EventDetailsMemberRemoveExternalIdDetails                       = "member_remove_external_id_details"
	EventDetailsMemberSetProfilePhotoDetails                        = "member_set_profile_photo_details"
	EventDetailsMemberSpaceLimitsAddCustomQuotaDetails              = "member_space_limits_add_custom_quota_details"
	EventDetailsMemberSpaceLimitsChangeCustomQuotaDetails           = "member_space_limits_change_custom_quota_details"
	EventDetailsMemberSpaceLimitsChangeStatusDetails                = "member_space_limits_change_status_details"
	EventDetailsMemberSpaceLimitsRemoveCustomQuotaDetails           = "member_space_limits_remove_custom_quota_details"
	EventDetailsMemberSuggestDetails                                = "member_suggest_details"
	EventDetailsMemberTransferAccountContentsDetails                = "member_transfer_account_contents_details"
	EventDetailsPendingSecondaryEmailAddedDetails                   = "pending_secondary_email_added_details"
	EventDetailsSecondaryEmailDeletedDetails                        = "secondary_email_deleted_details"
	EventDetailsSecondaryEmailVerifiedDetails                       = "secondary_email_verified_details"
	EventDetailsSecondaryMailsPolicyChangedDetails                  = "secondary_mails_policy_changed_details"
	EventDetailsBinderAddPageDetails                                = "binder_add_page_details"
	EventDetailsBinderAddSectionDetails                             = "binder_add_section_details"
	EventDetailsBinderRemovePageDetails                             = "binder_remove_page_details"
	EventDetailsBinderRemoveSectionDetails                          = "binder_remove_section_details"
	EventDetailsBinderRenamePageDetails                             = "binder_rename_page_details"
	EventDetailsBinderRenameSectionDetails                          = "binder_rename_section_details"
	EventDetailsBinderReorderPageDetails                            = "binder_reorder_page_details"
	EventDetailsBinderReorderSectionDetails                         = "binder_reorder_section_details"
	EventDetailsPaperContentAddMemberDetails                        = "paper_content_add_member_details"
	EventDetailsPaperContentAddToFolderDetails                      = "paper_content_add_to_folder_details"
	EventDetailsPaperContentArchiveDetails                          = "paper_content_archive_details"
	EventDetailsPaperContentCreateDetails                           = "paper_content_create_details"
	EventDetailsPaperContentPermanentlyDeleteDetails                = "paper_content_permanently_delete_details"
	EventDetailsPaperContentRemoveFromFolderDetails                 = "paper_content_remove_from_folder_details"
	EventDetailsPaperContentRemoveMemberDetails                     = "paper_content_remove_member_details"
	EventDetailsPaperContentRenameDetails                           = "paper_content_rename_details"
	EventDetailsPaperContentRestoreDetails                          = "paper_content_restore_details"
	EventDetailsPaperDocAddCommentDetails                           = "paper_doc_add_comment_details"
	EventDetailsPaperDocChangeMemberRoleDetails                     = "paper_doc_change_member_role_details"
	EventDetailsPaperDocChangeSharingPolicyDetails                  = "paper_doc_change_sharing_policy_details"
	EventDetailsPaperDocChangeSubscriptionDetails                   = "paper_doc_change_subscription_details"
	EventDetailsPaperDocDeletedDetails                              = "paper_doc_deleted_details"
	EventDetailsPaperDocDeleteCommentDetails                        = "paper_doc_delete_comment_details"
	EventDetailsPaperDocDownloadDetails                             = "paper_doc_download_details"
	EventDetailsPaperDocEditDetails                                 = "paper_doc_edit_details"
	EventDetailsPaperDocEditCommentDetails                          = "paper_doc_edit_comment_details"
	EventDetailsPaperDocFollowedDetails                             = "paper_doc_followed_details"
	EventDetailsPaperDocMentionDetails                              = "paper_doc_mention_details"
	EventDetailsPaperDocOwnershipChangedDetails                     = "paper_doc_ownership_changed_details"
	EventDetailsPaperDocRequestAccessDetails                        = "paper_doc_request_access_details"
	EventDetailsPaperDocResolveCommentDetails                       = "paper_doc_resolve_comment_details"
	EventDetailsPaperDocRevertDetails                               = "paper_doc_revert_details"
	EventDetailsPaperDocSlackShareDetails                           = "paper_doc_slack_share_details"
	EventDetailsPaperDocTeamInviteDetails                           = "paper_doc_team_invite_details"
	EventDetailsPaperDocTrashedDetails                              = "paper_doc_trashed_details"
	EventDetailsPaperDocUnresolveCommentDetails                     = "paper_doc_unresolve_comment_details"
	EventDetailsPaperDocUntrashedDetails                            = "paper_doc_untrashed_details"
	EventDetailsPaperDocViewDetails                                 = "paper_doc_view_details"
	EventDetailsPaperExternalViewAllowDetails                       = "paper_external_view_allow_details"
	EventDetailsPaperExternalViewDefaultTeamDetails                 = "paper_external_view_default_team_details"
	EventDetailsPaperExternalViewForbidDetails                      = "paper_external_view_forbid_details"
	EventDetailsPaperFolderChangeSubscriptionDetails                = "paper_folder_change_subscription_details"
	EventDetailsPaperFolderDeletedDetails                           = "paper_folder_deleted_details"
	EventDetailsPaperFolderFollowedDetails                          = "paper_folder_followed_details"
	EventDetailsPaperFolderTeamInviteDetails                        = "paper_folder_team_invite_details"
	EventDetailsPaperPublishedLinkChangePermissionDetails           = "paper_published_link_change_permission_details"
	EventDetailsPaperPublishedLinkCreateDetails                     = "paper_published_link_create_details"
	EventDetailsPaperPublishedLinkDisabledDetails                   = "paper_published_link_disabled_details"
	EventDetailsPaperPublishedLinkViewDetails                       = "paper_published_link_view_details"
	EventDetailsPasswordChangeDetails                               = "password_change_details"
	EventDetailsPasswordResetDetails                                = "password_reset_details"
	EventDetailsPasswordResetAllDetails                             = "password_reset_all_details"
	EventDetailsClassificationCreateReportDetails                   = "classification_create_report_details"
	EventDetailsClassificationCreateReportFailDetails               = "classification_create_report_fail_details"
	EventDetailsEmmCreateExceptionsReportDetails                    = "emm_create_exceptions_report_details"
	EventDetailsEmmCreateUsageReportDetails                         = "emm_create_usage_report_details"
	EventDetailsExportMembersReportDetails                          = "export_members_report_details"
	EventDetailsExportMembersReportFailDetails                      = "export_members_report_fail_details"
	EventDetailsExternalSharingCreateReportDetails                  = "external_sharing_create_report_details"
	EventDetailsExternalSharingReportFailedDetails                  = "external_sharing_report_failed_details"
	EventDetailsNoExpirationLinkGenCreateReportDetails              = "no_expiration_link_gen_create_report_details"
	EventDetailsNoExpirationLinkGenReportFailedDetails              = "no_expiration_link_gen_report_failed_details"
	EventDetailsNoPasswordLinkGenCreateReportDetails                = "no_password_link_gen_create_report_details"
	EventDetailsNoPasswordLinkGenReportFailedDetails                = "no_password_link_gen_report_failed_details"
	EventDetailsNoPasswordLinkViewCreateReportDetails               = "no_password_link_view_create_report_details"
	EventDetailsNoPasswordLinkViewReportFailedDetails               = "no_password_link_view_report_failed_details"
	EventDetailsOutdatedLinkViewCreateReportDetails                 = "outdated_link_view_create_report_details"
	EventDetailsOutdatedLinkViewReportFailedDetails                 = "outdated_link_view_report_failed_details"
	EventDetailsPaperAdminExportStartDetails                        = "paper_admin_export_start_details"
	EventDetailsSmartSyncCreateAdminPrivilegeReportDetails          = "smart_sync_create_admin_privilege_report_details"
	EventDetailsTeamActivityCreateReportDetails                     = "team_activity_create_report_details"
	EventDetailsTeamActivityCreateReportFailDetails                 = "team_activity_create_report_fail_details"
	EventDetailsCollectionShareDetails                              = "collection_share_details"
	EventDetailsFileTransfersFileAddDetails                         = "file_transfers_file_add_details"
	EventDetailsFileTransfersTransferDeleteDetails                  = "file_transfers_transfer_delete_details"
	EventDetailsFileTransfersTransferDownloadDetails                = "file_transfers_transfer_download_details"
	EventDetailsFileTransfersTransferSendDetails                    = "file_transfers_transfer_send_details"
	EventDetailsFileTransfersTransferViewDetails                    = "file_transfers_transfer_view_details"
	EventDetailsNoteAclInviteOnlyDetails                            = "note_acl_invite_only_details"
	EventDetailsNoteAclLinkDetails                                  = "note_acl_link_details"
	EventDetailsNoteAclTeamLinkDetails                              = "note_acl_team_link_details"
	EventDetailsNoteSharedDetails                                   = "note_shared_details"
	EventDetailsNoteShareReceiveDetails                             = "note_share_receive_details"
	EventDetailsOpenNoteSharedDetails                               = "open_note_shared_details"
	EventDetailsSfAddGroupDetails                                   = "sf_add_group_details"
	EventDetailsSfAllowNonMembersToViewSharedLinksDetails           = "sf_allow_non_members_to_view_shared_links_details"
	EventDetailsSfExternalInviteWarnDetails                         = "sf_external_invite_warn_details"
	EventDetailsSfFbInviteDetails                                   = "sf_fb_invite_details"
	EventDetailsSfFbInviteChangeRoleDetails                         = "sf_fb_invite_change_role_details"
	EventDetailsSfFbUninviteDetails                                 = "sf_fb_uninvite_details"
	EventDetailsSfInviteGroupDetails                                = "sf_invite_group_details"
	EventDetailsSfTeamGrantAccessDetails                            = "sf_team_grant_access_details"
	EventDetailsSfTeamInviteDetails                                 = "sf_team_invite_details"
	EventDetailsSfTeamInviteChangeRoleDetails                       = "sf_team_invite_change_role_details"
	EventDetailsSfTeamJoinDetails                                   = "sf_team_join_details"
	EventDetailsSfTeamJoinFromOobLinkDetails                        = "sf_team_join_from_oob_link_details"
	EventDetailsSfTeamUninviteDetails                               = "sf_team_uninvite_details"
	EventDetailsSharedContentAddInviteesDetails                     = "shared_content_add_invitees_details"
	EventDetailsSharedContentAddLinkExpiryDetails                   = "shared_content_add_link_expiry_details"
	EventDetailsSharedContentAddLinkPasswordDetails                 = "shared_content_add_link_password_details"
	EventDetailsSharedContentAddMemberDetails                       = "shared_content_add_member_details"
	EventDetailsSharedContentChangeDownloadsPolicyDetails           = "shared_content_change_downloads_policy_details"
	EventDetailsSharedContentChangeInviteeRoleDetails               = "shared_content_change_invitee_role_details"
	EventDetailsSharedContentChangeLinkAudienceDetails              = "shared_content_change_link_audience_details"
	EventDetailsSharedContentChangeLinkExpiryDetails                = "shared_content_change_link_expiry_details"
	EventDetailsSharedContentChangeLinkPasswordDetails              = "shared_content_change_link_password_details"
	EventDetailsSharedContentChangeMemberRoleDetails                = "shared_content_change_member_role_details"
	EventDetailsSharedContentChangeViewerInfoPolicyDetails          = "shared_content_change_viewer_info_policy_details"
	EventDetailsSharedContentClaimInvitationDetails                 = "shared_content_claim_invitation_details"
	EventDetailsSharedContentCopyDetails                            = "shared_content_copy_details"
	EventDetailsSharedContentDownloadDetails                        = "shared_content_download_details"
	EventDetailsSharedContentRelinquishMembershipDetails            = "shared_content_relinquish_membership_details"
	EventDetailsSharedContentRemoveInviteesDetails                  = "shared_content_remove_invitees_details"
	EventDetailsSharedContentRemoveLinkExpiryDetails                = "shared_content_remove_link_expiry_details"
	EventDetailsSharedContentRemoveLinkPasswordDetails              = "shared_content_remove_link_password_details"
	EventDetailsSharedContentRemoveMemberDetails                    = "shared_content_remove_member_details"
	EventDetailsSharedContentRequestAccessDetails                   = "shared_content_request_access_details"
	EventDetailsSharedContentRestoreInviteesDetails                 = "shared_content_restore_invitees_details"
	EventDetailsSharedContentRestoreMemberDetails                   = "shared_content_restore_member_details"
	EventDetailsSharedContentUnshareDetails                         = "shared_content_unshare_details"
	EventDetailsSharedContentViewDetails                            = "shared_content_view_details"
	EventDetailsSharedFolderChangeLinkPolicyDetails                 = "shared_folder_change_link_policy_details"
	EventDetailsSharedFolderChangeMembersInheritancePolicyDetails   = "shared_folder_change_members_inheritance_policy_details"
	EventDetailsSharedFolderChangeMembersManagementPolicyDetails    = "shared_folder_change_members_management_policy_details"
	EventDetailsSharedFolderChangeMembersPolicyDetails              = "shared_folder_change_members_policy_details"
	EventDetailsSharedFolderCreateDetails                           = "shared_folder_create_details"
	EventDetailsSharedFolderDeclineInvitationDetails                = "shared_folder_decline_invitation_details"
	EventDetailsSharedFolderMountDetails                            = "shared_folder_mount_details"
	EventDetailsSharedFolderNestDetails                             = "shared_folder_nest_details"
	EventDetailsSharedFolderTransferOwnershipDetails                = "shared_folder_transfer_ownership_details"
	EventDetailsSharedFolderUnmountDetails                          = "shared_folder_unmount_details"
	EventDetailsSharedLinkAddExpiryDetails                          = "shared_link_add_expiry_details"
	EventDetailsSharedLinkChangeExpiryDetails                       = "shared_link_change_expiry_details"
	EventDetailsSharedLinkChangeVisibilityDetails                   = "shared_link_change_visibility_details"
	EventDetailsSharedLinkCopyDetails                               = "shared_link_copy_details"
	EventDetailsSharedLinkCreateDetails                             = "shared_link_create_details"
	EventDetailsSharedLinkDisableDetails                            = "shared_link_disable_details"
	EventDetailsSharedLinkDownloadDetails                           = "shared_link_download_details"
	EventDetailsSharedLinkRemoveExpiryDetails                       = "shared_link_remove_expiry_details"
	EventDetailsSharedLinkSettingsAddExpirationDetails              = "shared_link_settings_add_expiration_details"
	EventDetailsSharedLinkSettingsAddPasswordDetails                = "shared_link_settings_add_password_details"
	EventDetailsSharedLinkSettingsAllowDownloadDisabledDetails      = "shared_link_settings_allow_download_disabled_details"
	EventDetailsSharedLinkSettingsAllowDownloadEnabledDetails       = "shared_link_settings_allow_download_enabled_details"
	EventDetailsSharedLinkSettingsChangeAudienceDetails             = "shared_link_settings_change_audience_details"
	EventDetailsSharedLinkSettingsChangeExpirationDetails           = "shared_link_settings_change_expiration_details"
	EventDetailsSharedLinkSettingsChangePasswordDetails             = "shared_link_settings_change_password_details"
	EventDetailsSharedLinkSettingsRemoveExpirationDetails           = "shared_link_settings_remove_expiration_details"
	EventDetailsSharedLinkSettingsRemovePasswordDetails             = "shared_link_settings_remove_password_details"
	EventDetailsSharedLinkShareDetails                              = "shared_link_share_details"
	EventDetailsSharedLinkViewDetails                               = "shared_link_view_details"
	EventDetailsSharedNoteOpenedDetails                             = "shared_note_opened_details"
	EventDetailsShmodelDisableDownloadsDetails                      = "shmodel_disable_downloads_details"
	EventDetailsShmodelEnableDownloadsDetails                       = "shmodel_enable_downloads_details"
	EventDetailsShmodelGroupShareDetails                            = "shmodel_group_share_details"
	EventDetailsShowcaseAccessGrantedDetails                        = "showcase_access_granted_details"
	EventDetailsShowcaseAddMemberDetails                            = "showcase_add_member_details"
	EventDetailsShowcaseArchivedDetails                             = "showcase_archived_details"
	EventDetailsShowcaseCreatedDetails                              = "showcase_created_details"
	EventDetailsShowcaseDeleteCommentDetails                        = "showcase_delete_comment_details"
	EventDetailsShowcaseEditedDetails                               = "showcase_edited_details"
	EventDetailsShowcaseEditCommentDetails                          = "showcase_edit_comment_details"
	EventDetailsShowcaseFileAddedDetails                            = "showcase_file_added_details"
	EventDetailsShowcaseFileDownloadDetails                         = "showcase_file_download_details"
	EventDetailsShowcaseFileRemovedDetails                          = "showcase_file_removed_details"
	EventDetailsShowcaseFileViewDetails                             = "showcase_file_view_details"
	EventDetailsShowcasePermanentlyDeletedDetails                   = "showcase_permanently_deleted_details"
	EventDetailsShowcasePostCommentDetails                          = "showcase_post_comment_details"
	EventDetailsShowcaseRemoveMemberDetails                         = "showcase_remove_member_details"
	EventDetailsShowcaseRenamedDetails                              = "showcase_renamed_details"
	EventDetailsShowcaseRequestAccessDetails                        = "showcase_request_access_details"
	EventDetailsShowcaseResolveCommentDetails                       = "showcase_resolve_comment_details"
	EventDetailsShowcaseRestoredDetails                             = "showcase_restored_details"
	EventDetailsShowcaseTrashedDetails                              = "showcase_trashed_details"
	EventDetailsShowcaseTrashedDeprecatedDetails                    = "showcase_trashed_deprecated_details"
	EventDetailsShowcaseUnresolveCommentDetails                     = "showcase_unresolve_comment_details"
	EventDetailsShowcaseUntrashedDetails                            = "showcase_untrashed_details"
	EventDetailsShowcaseUntrashedDeprecatedDetails                  = "showcase_untrashed_deprecated_details"
	EventDetailsShowcaseViewDetails                                 = "showcase_view_details"
	EventDetailsSsoAddCertDetails                                   = "sso_add_cert_details"
	EventDetailsSsoAddLoginUrlDetails                               = "sso_add_login_url_details"
	EventDetailsSsoAddLogoutUrlDetails                              = "sso_add_logout_url_details"
	EventDetailsSsoChangeCertDetails                                = "sso_change_cert_details"
	EventDetailsSsoChangeLoginUrlDetails                            = "sso_change_login_url_details"
	EventDetailsSsoChangeLogoutUrlDetails                           = "sso_change_logout_url_details"
	EventDetailsSsoChangeSamlIdentityModeDetails                    = "sso_change_saml_identity_mode_details"
	EventDetailsSsoRemoveCertDetails                                = "sso_remove_cert_details"
	EventDetailsSsoRemoveLoginUrlDetails                            = "sso_remove_login_url_details"
	EventDetailsSsoRemoveLogoutUrlDetails                           = "sso_remove_logout_url_details"
	EventDetailsTeamFolderChangeStatusDetails                       = "team_folder_change_status_details"
	EventDetailsTeamFolderCreateDetails                             = "team_folder_create_details"
	EventDetailsTeamFolderDowngradeDetails                          = "team_folder_downgrade_details"
	EventDetailsTeamFolderPermanentlyDeleteDetails                  = "team_folder_permanently_delete_details"
	EventDetailsTeamFolderRenameDetails                             = "team_folder_rename_details"
	EventDetailsTeamSelectiveSyncSettingsChangedDetails             = "team_selective_sync_settings_changed_details"
	EventDetailsAccountCaptureChangePolicyDetails                   = "account_capture_change_policy_details"
	EventDetailsAdminEmailRemindersChangedDetails                   = "admin_email_reminders_changed_details"
	EventDetailsAllowDownloadDisabledDetails                        = "allow_download_disabled_details"
	EventDetailsAllowDownloadEnabledDetails                         = "allow_download_enabled_details"
	EventDetailsAppPermissionsChangedDetails                        = "app_permissions_changed_details"
	EventDetailsCameraUploadsPolicyChangedDetails                   = "camera_uploads_policy_changed_details"
	EventDetailsCaptureTranscriptPolicyChangedDetails               = "capture_transcript_policy_changed_details"
	EventDetailsClassificationChangePolicyDetails                   = "classification_change_policy_details"
	EventDetailsComputerBackupPolicyChangedDetails                  = "computer_backup_policy_changed_details"
	EventDetailsContentAdministrationPolicyChangedDetails           = "content_administration_policy_changed_details"
	EventDetailsDataPlacementRestrictionChangePolicyDetails         = "data_placement_restriction_change_policy_details"
	EventDetailsDataPlacementRestrictionSatisfyPolicyDetails        = "data_placement_restriction_satisfy_policy_details"
	EventDetailsDeviceApprovalsAddExceptionDetails                  = "device_approvals_add_exception_details"
	EventDetailsDeviceApprovalsChangeDesktopPolicyDetails           = "device_approvals_change_desktop_policy_details"
	EventDetailsDeviceApprovalsChangeMobilePolicyDetails            = "device_approvals_change_mobile_policy_details"
	EventDetailsDeviceApprovalsChangeOverageActionDetails           = "device_approvals_change_overage_action_details"
	EventDetailsDeviceApprovalsChangeUnlinkActionDetails            = "device_approvals_change_unlink_action_details"
	EventDetailsDeviceApprovalsRemoveExceptionDetails               = "device_approvals_remove_exception_details"
	EventDetailsDirectoryRestrictionsAddMembersDetails              = "directory_restrictions_add_members_details"
	EventDetailsDirectoryRestrictionsRemoveMembersDetails           = "directory_restrictions_remove_members_details"
	EventDetailsDropboxPasswordsPolicyChangedDetails                = "dropbox_passwords_policy_changed_details"
	EventDetailsEmailIngestPolicyChangedDetails                     = "email_ingest_policy_changed_details"
	EventDetailsEmmAddExceptionDetails                              = "emm_add_exception_details"
	EventDetailsEmmChangePolicyDetails                              = "emm_change_policy_details"
	EventDetailsEmmRemoveExceptionDetails                           = "emm_remove_exception_details"
	EventDetailsExtendedVersionHistoryChangePolicyDetails           = "extended_version_history_change_policy_details"
	EventDetailsExternalDriveBackupPolicyChangedDetails             = "external_drive_backup_policy_changed_details"
	EventDetailsFileCommentsChangePolicyDetails                     = "file_comments_change_policy_details"
	EventDetailsFileLockingPolicyChangedDetails                     = "file_locking_policy_changed_details"
	EventDetailsFileProviderMigrationPolicyChangedDetails           = "file_provider_migration_policy_changed_details"
	EventDetailsFileRequestsChangePolicyDetails                     = "file_requests_change_policy_details"
	EventDetailsFileRequestsEmailsEnabledDetails                    = "file_requests_emails_enabled_details"
	EventDetailsFileRequestsEmailsRestrictedToTeamOnlyDetails       = "file_requests_emails_restricted_to_team_only_details"
	EventDetailsFileTransfersPolicyChangedDetails                   = "file_transfers_policy_changed_details"
	EventDetailsGoogleSsoChangePolicyDetails                        = "google_sso_change_policy_details"
	EventDetailsGroupUserManagementChangePolicyDetails              = "group_user_management_change_policy_details"
	EventDetailsIntegrationPolicyChangedDetails                     = "integration_policy_changed_details"
	EventDetailsInviteAcceptanceEmailPolicyChangedDetails           = "invite_acceptance_email_policy_changed_details"
	EventDetailsMemberRequestsChangePolicyDetails                   = "member_requests_change_policy_details"
	EventDetailsMemberSendInvitePolicyChangedDetails                = "member_send_invite_policy_changed_details"
	EventDetailsMemberSpaceLimitsAddExceptionDetails                = "member_space_limits_add_exception_details"
	EventDetailsMemberSpaceLimitsChangeCapsTypePolicyDetails        = "member_space_limits_change_caps_type_policy_details"
	EventDetailsMemberSpaceLimitsChangePolicyDetails                = "member_space_limits_change_policy_details"
	EventDetailsMemberSpaceLimitsRemoveExceptionDetails             = "member_space_limits_remove_exception_details"
	EventDetailsMemberSuggestionsChangePolicyDetails                = "member_suggestions_change_policy_details"
	EventDetailsMicrosoftOfficeAddinChangePolicyDetails             = "microsoft_office_addin_change_policy_details"
	EventDetailsNetworkControlChangePolicyDetails                   = "network_control_change_policy_details"
	EventDetailsPaperChangeDeploymentPolicyDetails                  = "paper_change_deployment_policy_details"
	EventDetailsPaperChangeMemberLinkPolicyDetails                  = "paper_change_member_link_policy_details"
	EventDetailsPaperChangeMemberPolicyDetails                      = "paper_change_member_policy_details"
	EventDetailsPaperChangePolicyDetails                            = "paper_change_policy_details"
	EventDetailsPaperDefaultFolderPolicyChangedDetails              = "paper_default_folder_policy_changed_details"
	EventDetailsPaperDesktopPolicyChangedDetails                    = "paper_desktop_policy_changed_details"
	EventDetailsPaperEnabledUsersGroupAdditionDetails               = "paper_enabled_users_group_addition_details"
	EventDetailsPaperEnabledUsersGroupRemovalDetails                = "paper_enabled_users_group_removal_details"
	EventDetailsPasswordStrengthRequirementsChangePolicyDetails     = "password_strength_requirements_change_policy_details"
	EventDetailsPermanentDeleteChangePolicyDetails                  = "permanent_delete_change_policy_details"
	EventDetailsResellerSupportChangePolicyDetails                  = "reseller_support_change_policy_details"
	EventDetailsRewindPolicyChangedDetails                          = "rewind_policy_changed_details"
	EventDetailsSendForSignaturePolicyChangedDetails                = "send_for_signature_policy_changed_details"
	EventDetailsSharingChangeFolderJoinPolicyDetails                = "sharing_change_folder_join_policy_details"
	EventDetailsSharingChangeLinkAllowChangeExpirationPolicyDetails = "sharing_change_link_allow_change_expiration_policy_details"
	EventDetailsSharingChangeLinkDefaultExpirationPolicyDetails     = "sharing_change_link_default_expiration_policy_details"
	EventDetailsSharingChangeLinkEnforcePasswordPolicyDetails       = "sharing_change_link_enforce_password_policy_details"
	EventDetailsSharingChangeLinkPolicyDetails                      = "sharing_change_link_policy_details"
	EventDetailsSharingChangeMemberPolicyDetails                    = "sharing_change_member_policy_details"
	EventDetailsShowcaseChangeDownloadPolicyDetails                 = "showcase_change_download_policy_details"
	EventDetailsShowcaseChangeEnabledPolicyDetails                  = "showcase_change_enabled_policy_details"
	EventDetailsShowcaseChangeExternalSharingPolicyDetails          = "showcase_change_external_sharing_policy_details"
	EventDetailsSmarterSmartSyncPolicyChangedDetails                = "smarter_smart_sync_policy_changed_details"
	EventDetailsSmartSyncChangePolicyDetails                        = "smart_sync_change_policy_details"
	EventDetailsSmartSyncNotOptOutDetails                           = "smart_sync_not_opt_out_details"
	EventDetailsSmartSyncOptOutDetails                              = "smart_sync_opt_out_details"
	EventDetailsSsoChangePolicyDetails                              = "sso_change_policy_details"
	EventDetailsTeamBrandingPolicyChangedDetails                    = "team_branding_policy_changed_details"
	EventDetailsTeamExtensionsPolicyChangedDetails                  = "team_extensions_policy_changed_details"
	EventDetailsTeamSelectiveSyncPolicyChangedDetails               = "team_selective_sync_policy_changed_details"
	EventDetailsTeamSharingWhitelistSubjectsChangedDetails          = "team_sharing_whitelist_subjects_changed_details"
	EventDetailsTfaAddExceptionDetails                              = "tfa_add_exception_details"
	EventDetailsTfaChangePolicyDetails                              = "tfa_change_policy_details"
	EventDetailsTfaRemoveExceptionDetails                           = "tfa_remove_exception_details"
	EventDetailsTwoAccountChangePolicyDetails                       = "two_account_change_policy_details"
	EventDetailsViewerInfoPolicyChangedDetails                      = "viewer_info_policy_changed_details"
	EventDetailsWatermarkingPolicyChangedDetails                    = "watermarking_policy_changed_details"
	EventDetailsWebSessionsChangeActiveSessionLimitDetails          = "web_sessions_change_active_session_limit_details"
	EventDetailsWebSessionsChangeFixedLengthPolicyDetails           = "web_sessions_change_fixed_length_policy_details"
	EventDetailsWebSessionsChangeIdleLengthPolicyDetails            = "web_sessions_change_idle_length_policy_details"
	EventDetailsDataResidencyMigrationRequestSuccessfulDetails      = "data_residency_migration_request_successful_details"
	EventDetailsDataResidencyMigrationRequestUnsuccessfulDetails    = "data_residency_migration_request_unsuccessful_details"
	EventDetailsTeamMergeFromDetails                                = "team_merge_from_details"
	EventDetailsTeamMergeToDetails                                  = "team_merge_to_details"
	EventDetailsTeamProfileAddBackgroundDetails                     = "team_profile_add_background_details"
	EventDetailsTeamProfileAddLogoDetails                           = "team_profile_add_logo_details"
	EventDetailsTeamProfileChangeBackgroundDetails                  = "team_profile_change_background_details"
	EventDetailsTeamProfileChangeDefaultLanguageDetails             = "team_profile_change_default_language_details"
	EventDetailsTeamProfileChangeLogoDetails                        = "team_profile_change_logo_details"
	EventDetailsTeamProfileChangeNameDetails                        = "team_profile_change_name_details"
	EventDetailsTeamProfileRemoveBackgroundDetails                  = "team_profile_remove_background_details"
	EventDetailsTeamProfileRemoveLogoDetails                        = "team_profile_remove_logo_details"
	EventDetailsTfaAddBackupPhoneDetails                            = "tfa_add_backup_phone_details"
	EventDetailsTfaAddSecurityKeyDetails                            = "tfa_add_security_key_details"
	EventDetailsTfaChangeBackupPhoneDetails                         = "tfa_change_backup_phone_details"
	EventDetailsTfaChangeStatusDetails                              = "tfa_change_status_details"
	EventDetailsTfaRemoveBackupPhoneDetails                         = "tfa_remove_backup_phone_details"
	EventDetailsTfaRemoveSecurityKeyDetails                         = "tfa_remove_security_key_details"
	EventDetailsTfaResetDetails                                     = "tfa_reset_details"
	EventDetailsChangedEnterpriseAdminRoleDetails                   = "changed_enterprise_admin_role_details"
	EventDetailsChangedEnterpriseConnectedTeamStatusDetails         = "changed_enterprise_connected_team_status_details"
	EventDetailsEndedEnterpriseAdminSessionDetails                  = "ended_enterprise_admin_session_details"
	EventDetailsEndedEnterpriseAdminSessionDeprecatedDetails        = "ended_enterprise_admin_session_deprecated_details"
	EventDetailsEnterpriseSettingsLockingDetails                    = "enterprise_settings_locking_details"
	EventDetailsGuestAdminChangeStatusDetails                       = "guest_admin_change_status_details"
	EventDetailsStartedEnterpriseAdminSessionDetails                = "started_enterprise_admin_session_details"
	EventDetailsTeamMergeRequestAcceptedDetails                     = "team_merge_request_accepted_details"
	EventDetailsTeamMergeRequestAcceptedShownToPrimaryTeamDetails   = "team_merge_request_accepted_shown_to_primary_team_details"
	EventDetailsTeamMergeRequestAcceptedShownToSecondaryTeamDetails = "team_merge_request_accepted_shown_to_secondary_team_details"
	EventDetailsTeamMergeRequestAutoCanceledDetails                 = "team_merge_request_auto_canceled_details"
	EventDetailsTeamMergeRequestCanceledDetails                     = "team_merge_request_canceled_details"
	EventDetailsTeamMergeRequestCanceledShownToPrimaryTeamDetails   = "team_merge_request_canceled_shown_to_primary_team_details"
	EventDetailsTeamMergeRequestCanceledShownToSecondaryTeamDetails = "team_merge_request_canceled_shown_to_secondary_team_details"
	EventDetailsTeamMergeRequestExpiredDetails                      = "team_merge_request_expired_details"
	EventDetailsTeamMergeRequestExpiredShownToPrimaryTeamDetails    = "team_merge_request_expired_shown_to_primary_team_details"
	EventDetailsTeamMergeRequestExpiredShownToSecondaryTeamDetails  = "team_merge_request_expired_shown_to_secondary_team_details"
	EventDetailsTeamMergeRequestRejectedShownToPrimaryTeamDetails   = "team_merge_request_rejected_shown_to_primary_team_details"
	EventDetailsTeamMergeRequestRejectedShownToSecondaryTeamDetails = "team_merge_request_rejected_shown_to_secondary_team_details"
	EventDetailsTeamMergeRequestReminderDetails                     = "team_merge_request_reminder_details"
	EventDetailsTeamMergeRequestReminderShownToPrimaryTeamDetails   = "team_merge_request_reminder_shown_to_primary_team_details"
	EventDetailsTeamMergeRequestReminderShownToSecondaryTeamDetails = "team_merge_request_reminder_shown_to_secondary_team_details"
	EventDetailsTeamMergeRequestRevokedDetails                      = "team_merge_request_revoked_details"
	EventDetailsTeamMergeRequestSentShownToPrimaryTeamDetails       = "team_merge_request_sent_shown_to_primary_team_details"
	EventDetailsTeamMergeRequestSentShownToSecondaryTeamDetails     = "team_merge_request_sent_shown_to_secondary_team_details"
	EventDetailsMissingDetails                                      = "missing_details"
	EventDetailsOther                                               = "other"
)

// UnmarshalJSON deserializes into a EventDetails instance
func (u *EventDetails) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "admin_alerting_alert_state_changed_details":
		if err = json.Unmarshal(body, &u.AdminAlertingAlertStateChangedDetails); err != nil {
			return err
		}

	case "admin_alerting_changed_alert_config_details":
		if err = json.Unmarshal(body, &u.AdminAlertingChangedAlertConfigDetails); err != nil {
			return err
		}

	case "admin_alerting_triggered_alert_details":
		if err = json.Unmarshal(body, &u.AdminAlertingTriggeredAlertDetails); err != nil {
			return err
		}

	case "app_blocked_by_permissions_details":
		if err = json.Unmarshal(body, &u.AppBlockedByPermissionsDetails); err != nil {
			return err
		}

	case "app_link_team_details":
		if err = json.Unmarshal(body, &u.AppLinkTeamDetails); err != nil {
			return err
		}

	case "app_link_user_details":
		if err = json.Unmarshal(body, &u.AppLinkUserDetails); err != nil {
			return err
		}

	case "app_unlink_team_details":
		if err = json.Unmarshal(body, &u.AppUnlinkTeamDetails); err != nil {
			return err
		}

	case "app_unlink_user_details":
		if err = json.Unmarshal(body, &u.AppUnlinkUserDetails); err != nil {
			return err
		}

	case "integration_connected_details":
		if err = json.Unmarshal(body, &u.IntegrationConnectedDetails); err != nil {
			return err
		}

	case "integration_disconnected_details":
		if err = json.Unmarshal(body, &u.IntegrationDisconnectedDetails); err != nil {
			return err
		}

	case "file_add_comment_details":
		if err = json.Unmarshal(body, &u.FileAddCommentDetails); err != nil {
			return err
		}

	case "file_change_comment_subscription_details":
		if err = json.Unmarshal(body, &u.FileChangeCommentSubscriptionDetails); err != nil {
			return err
		}

	case "file_delete_comment_details":
		if err = json.Unmarshal(body, &u.FileDeleteCommentDetails); err != nil {
			return err
		}

	case "file_edit_comment_details":
		if err = json.Unmarshal(body, &u.FileEditCommentDetails); err != nil {
			return err
		}

	case "file_like_comment_details":
		if err = json.Unmarshal(body, &u.FileLikeCommentDetails); err != nil {
			return err
		}

	case "file_resolve_comment_details":
		if err = json.Unmarshal(body, &u.FileResolveCommentDetails); err != nil {
			return err
		}

	case "file_unlike_comment_details":
		if err = json.Unmarshal(body, &u.FileUnlikeCommentDetails); err != nil {
			return err
		}

	case "file_unresolve_comment_details":
		if err = json.Unmarshal(body, &u.FileUnresolveCommentDetails); err != nil {
			return err
		}

	case "governance_policy_add_folders_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyAddFoldersDetails); err != nil {
			return err
		}

	case "governance_policy_add_folder_failed_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyAddFolderFailedDetails); err != nil {
			return err
		}

	case "governance_policy_content_disposed_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyContentDisposedDetails); err != nil {
			return err
		}

	case "governance_policy_create_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyCreateDetails); err != nil {
			return err
		}

	case "governance_policy_delete_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyDeleteDetails); err != nil {
			return err
		}

	case "governance_policy_edit_details_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyEditDetailsDetails); err != nil {
			return err
		}

	case "governance_policy_edit_duration_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyEditDurationDetails); err != nil {
			return err
		}

	case "governance_policy_export_created_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyExportCreatedDetails); err != nil {
			return err
		}

	case "governance_policy_export_removed_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyExportRemovedDetails); err != nil {
			return err
		}

	case "governance_policy_remove_folders_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyRemoveFoldersDetails); err != nil {
			return err
		}

	case "governance_policy_report_created_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyReportCreatedDetails); err != nil {
			return err
		}

	case "governance_policy_zip_part_downloaded_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyZipPartDownloadedDetails); err != nil {
			return err
		}

	case "legal_holds_activate_a_hold_details":
		if err = json.Unmarshal(body, &u.LegalHoldsActivateAHoldDetails); err != nil {
			return err
		}

	case "legal_holds_add_members_details":
		if err = json.Unmarshal(body, &u.LegalHoldsAddMembersDetails); err != nil {
			return err
		}

	case "legal_holds_change_hold_details_details":
		if err = json.Unmarshal(body, &u.LegalHoldsChangeHoldDetailsDetails); err != nil {
			return err
		}

	case "legal_holds_change_hold_name_details":
		if err = json.Unmarshal(body, &u.LegalHoldsChangeHoldNameDetails); err != nil {
			return err
		}

	case "legal_holds_export_a_hold_details":
		if err = json.Unmarshal(body, &u.LegalHoldsExportAHoldDetails); err != nil {
			return err
		}

	case "legal_holds_export_cancelled_details":
		if err = json.Unmarshal(body, &u.LegalHoldsExportCancelledDetails); err != nil {
			return err
		}

	case "legal_holds_export_downloaded_details":
		if err = json.Unmarshal(body, &u.LegalHoldsExportDownloadedDetails); err != nil {
			return err
		}

	case "legal_holds_export_removed_details":
		if err = json.Unmarshal(body, &u.LegalHoldsExportRemovedDetails); err != nil {
			return err
		}

	case "legal_holds_release_a_hold_details":
		if err = json.Unmarshal(body, &u.LegalHoldsReleaseAHoldDetails); err != nil {
			return err
		}

	case "legal_holds_remove_members_details":
		if err = json.Unmarshal(body, &u.LegalHoldsRemoveMembersDetails); err != nil {
			return err
		}

	case "legal_holds_report_a_hold_details":
		if err = json.Unmarshal(body, &u.LegalHoldsReportAHoldDetails); err != nil {
			return err
		}

	case "device_change_ip_desktop_details":
		if err = json.Unmarshal(body, &u.DeviceChangeIpDesktopDetails); err != nil {
			return err
		}

	case "device_change_ip_mobile_details":
		if err = json.Unmarshal(body, &u.DeviceChangeIpMobileDetails); err != nil {
			return err
		}

	case "device_change_ip_web_details":
		if err = json.Unmarshal(body, &u.DeviceChangeIpWebDetails); err != nil {
			return err
		}

	case "device_delete_on_unlink_fail_details":
		if err = json.Unmarshal(body, &u.DeviceDeleteOnUnlinkFailDetails); err != nil {
			return err
		}

	case "device_delete_on_unlink_success_details":
		if err = json.Unmarshal(body, &u.DeviceDeleteOnUnlinkSuccessDetails); err != nil {
			return err
		}

	case "device_link_fail_details":
		if err = json.Unmarshal(body, &u.DeviceLinkFailDetails); err != nil {
			return err
		}

	case "device_link_success_details":
		if err = json.Unmarshal(body, &u.DeviceLinkSuccessDetails); err != nil {
			return err
		}

	case "device_management_disabled_details":
		if err = json.Unmarshal(body, &u.DeviceManagementDisabledDetails); err != nil {
			return err
		}

	case "device_management_enabled_details":
		if err = json.Unmarshal(body, &u.DeviceManagementEnabledDetails); err != nil {
			return err
		}

	case "device_sync_backup_status_changed_details":
		if err = json.Unmarshal(body, &u.DeviceSyncBackupStatusChangedDetails); err != nil {
			return err
		}

	case "device_unlink_details":
		if err = json.Unmarshal(body, &u.DeviceUnlinkDetails); err != nil {
			return err
		}

	case "dropbox_passwords_exported_details":
		if err = json.Unmarshal(body, &u.DropboxPasswordsExportedDetails); err != nil {
			return err
		}

	case "dropbox_passwords_new_device_enrolled_details":
		if err = json.Unmarshal(body, &u.DropboxPasswordsNewDeviceEnrolledDetails); err != nil {
			return err
		}

	case "emm_refresh_auth_token_details":
		if err = json.Unmarshal(body, &u.EmmRefreshAuthTokenDetails); err != nil {
			return err
		}

	case "external_drive_backup_eligibility_status_checked_details":
		if err = json.Unmarshal(body, &u.ExternalDriveBackupEligibilityStatusCheckedDetails); err != nil {
			return err
		}

	case "external_drive_backup_status_changed_details":
		if err = json.Unmarshal(body, &u.ExternalDriveBackupStatusChangedDetails); err != nil {
			return err
		}

	case "account_capture_change_availability_details":
		if err = json.Unmarshal(body, &u.AccountCaptureChangeAvailabilityDetails); err != nil {
			return err
		}

	case "account_capture_migrate_account_details":
		if err = json.Unmarshal(body, &u.AccountCaptureMigrateAccountDetails); err != nil {
			return err
		}

	case "account_capture_notification_emails_sent_details":
		if err = json.Unmarshal(body, &u.AccountCaptureNotificationEmailsSentDetails); err != nil {
			return err
		}

	case "account_capture_relinquish_account_details":
		if err = json.Unmarshal(body, &u.AccountCaptureRelinquishAccountDetails); err != nil {
			return err
		}

	case "disabled_domain_invites_details":
		if err = json.Unmarshal(body, &u.DisabledDomainInvitesDetails); err != nil {
			return err
		}

	case "domain_invites_approve_request_to_join_team_details":
		if err = json.Unmarshal(body, &u.DomainInvitesApproveRequestToJoinTeamDetails); err != nil {
			return err
		}

	case "domain_invites_decline_request_to_join_team_details":
		if err = json.Unmarshal(body, &u.DomainInvitesDeclineRequestToJoinTeamDetails); err != nil {
			return err
		}

	case "domain_invites_email_existing_users_details":
		if err = json.Unmarshal(body, &u.DomainInvitesEmailExistingUsersDetails); err != nil {
			return err
		}

	case "domain_invites_request_to_join_team_details":
		if err = json.Unmarshal(body, &u.DomainInvitesRequestToJoinTeamDetails); err != nil {
			return err
		}

	case "domain_invites_set_invite_new_user_pref_to_no_details":
		if err = json.Unmarshal(body, &u.DomainInvitesSetInviteNewUserPrefToNoDetails); err != nil {
			return err
		}

	case "domain_invites_set_invite_new_user_pref_to_yes_details":
		if err = json.Unmarshal(body, &u.DomainInvitesSetInviteNewUserPrefToYesDetails); err != nil {
			return err
		}

	case "domain_verification_add_domain_fail_details":
		if err = json.Unmarshal(body, &u.DomainVerificationAddDomainFailDetails); err != nil {
			return err
		}

	case "domain_verification_add_domain_success_details":
		if err = json.Unmarshal(body, &u.DomainVerificationAddDomainSuccessDetails); err != nil {
			return err
		}

	case "domain_verification_remove_domain_details":
		if err = json.Unmarshal(body, &u.DomainVerificationRemoveDomainDetails); err != nil {
			return err
		}

	case "enabled_domain_invites_details":
		if err = json.Unmarshal(body, &u.EnabledDomainInvitesDetails); err != nil {
			return err
		}

	case "apply_naming_convention_details":
		if err = json.Unmarshal(body, &u.ApplyNamingConventionDetails); err != nil {
			return err
		}

	case "create_folder_details":
		if err = json.Unmarshal(body, &u.CreateFolderDetails); err != nil {
			return err
		}

	case "file_add_details":
		if err = json.Unmarshal(body, &u.FileAddDetails); err != nil {
			return err
		}

	case "file_copy_details":
		if err = json.Unmarshal(body, &u.FileCopyDetails); err != nil {
			return err
		}

	case "file_delete_details":
		if err = json.Unmarshal(body, &u.FileDeleteDetails); err != nil {
			return err
		}

	case "file_download_details":
		if err = json.Unmarshal(body, &u.FileDownloadDetails); err != nil {
			return err
		}

	case "file_edit_details":
		if err = json.Unmarshal(body, &u.FileEditDetails); err != nil {
			return err
		}

	case "file_get_copy_reference_details":
		if err = json.Unmarshal(body, &u.FileGetCopyReferenceDetails); err != nil {
			return err
		}

	case "file_locking_lock_status_changed_details":
		if err = json.Unmarshal(body, &u.FileLockingLockStatusChangedDetails); err != nil {
			return err
		}

	case "file_move_details":
		if err = json.Unmarshal(body, &u.FileMoveDetails); err != nil {
			return err
		}

	case "file_permanently_delete_details":
		if err = json.Unmarshal(body, &u.FilePermanentlyDeleteDetails); err != nil {
			return err
		}

	case "file_preview_details":
		if err = json.Unmarshal(body, &u.FilePreviewDetails); err != nil {
			return err
		}

	case "file_rename_details":
		if err = json.Unmarshal(body, &u.FileRenameDetails); err != nil {
			return err
		}

	case "file_restore_details":
		if err = json.Unmarshal(body, &u.FileRestoreDetails); err != nil {
			return err
		}

	case "file_revert_details":
		if err = json.Unmarshal(body, &u.FileRevertDetails); err != nil {
			return err
		}

	case "file_rollback_changes_details":
		if err = json.Unmarshal(body, &u.FileRollbackChangesDetails); err != nil {
			return err
		}

	case "file_save_copy_reference_details":
		if err = json.Unmarshal(body, &u.FileSaveCopyReferenceDetails); err != nil {
			return err
		}

	case "folder_overview_description_changed_details":
		if err = json.Unmarshal(body, &u.FolderOverviewDescriptionChangedDetails); err != nil {
			return err
		}

	case "folder_overview_item_pinned_details":
		if err = json.Unmarshal(body, &u.FolderOverviewItemPinnedDetails); err != nil {
			return err
		}

	case "folder_overview_item_unpinned_details":
		if err = json.Unmarshal(body, &u.FolderOverviewItemUnpinnedDetails); err != nil {
			return err
		}

	case "object_label_added_details":
		if err = json.Unmarshal(body, &u.ObjectLabelAddedDetails); err != nil {
			return err
		}

	case "object_label_removed_details":
		if err = json.Unmarshal(body, &u.ObjectLabelRemovedDetails); err != nil {
			return err
		}

	case "object_label_updated_value_details":
		if err = json.Unmarshal(body, &u.ObjectLabelUpdatedValueDetails); err != nil {
			return err
		}

	case "organize_folder_with_tidy_details":
		if err = json.Unmarshal(body, &u.OrganizeFolderWithTidyDetails); err != nil {
			return err
		}

	case "rewind_folder_details":
		if err = json.Unmarshal(body, &u.RewindFolderDetails); err != nil {
			return err
		}

	case "undo_naming_convention_details":
		if err = json.Unmarshal(body, &u.UndoNamingConventionDetails); err != nil {
			return err
		}

	case "undo_organize_folder_with_tidy_details":
		if err = json.Unmarshal(body, &u.UndoOrganizeFolderWithTidyDetails); err != nil {
			return err
		}

	case "user_tags_added_details":
		if err = json.Unmarshal(body, &u.UserTagsAddedDetails); err != nil {
			return err
		}

	case "user_tags_removed_details":
		if err = json.Unmarshal(body, &u.UserTagsRemovedDetails); err != nil {
			return err
		}

	case "email_ingest_receive_file_details":
		if err = json.Unmarshal(body, &u.EmailIngestReceiveFileDetails); err != nil {
			return err
		}

	case "file_request_change_details":
		if err = json.Unmarshal(body, &u.FileRequestChangeDetails); err != nil {
			return err
		}

	case "file_request_close_details":
		if err = json.Unmarshal(body, &u.FileRequestCloseDetails); err != nil {
			return err
		}

	case "file_request_create_details":
		if err = json.Unmarshal(body, &u.FileRequestCreateDetails); err != nil {
			return err
		}

	case "file_request_delete_details":
		if err = json.Unmarshal(body, &u.FileRequestDeleteDetails); err != nil {
			return err
		}

	case "file_request_receive_file_details":
		if err = json.Unmarshal(body, &u.FileRequestReceiveFileDetails); err != nil {
			return err
		}

	case "group_add_external_id_details":
		if err = json.Unmarshal(body, &u.GroupAddExternalIdDetails); err != nil {
			return err
		}

	case "group_add_member_details":
		if err = json.Unmarshal(body, &u.GroupAddMemberDetails); err != nil {
			return err
		}

	case "group_change_external_id_details":
		if err = json.Unmarshal(body, &u.GroupChangeExternalIdDetails); err != nil {
			return err
		}

	case "group_change_management_type_details":
		if err = json.Unmarshal(body, &u.GroupChangeManagementTypeDetails); err != nil {
			return err
		}

	case "group_change_member_role_details":
		if err = json.Unmarshal(body, &u.GroupChangeMemberRoleDetails); err != nil {
			return err
		}

	case "group_create_details":
		if err = json.Unmarshal(body, &u.GroupCreateDetails); err != nil {
			return err
		}

	case "group_delete_details":
		if err = json.Unmarshal(body, &u.GroupDeleteDetails); err != nil {
			return err
		}

	case "group_description_updated_details":
		if err = json.Unmarshal(body, &u.GroupDescriptionUpdatedDetails); err != nil {
			return err
		}

	case "group_join_policy_updated_details":
		if err = json.Unmarshal(body, &u.GroupJoinPolicyUpdatedDetails); err != nil {
			return err
		}

	case "group_moved_details":
		if err = json.Unmarshal(body, &u.GroupMovedDetails); err != nil {
			return err
		}

	case "group_remove_external_id_details":
		if err = json.Unmarshal(body, &u.GroupRemoveExternalIdDetails); err != nil {
			return err
		}

	case "group_remove_member_details":
		if err = json.Unmarshal(body, &u.GroupRemoveMemberDetails); err != nil {
			return err
		}

	case "group_rename_details":
		if err = json.Unmarshal(body, &u.GroupRenameDetails); err != nil {
			return err
		}

	case "account_lock_or_unlocked_details":
		if err = json.Unmarshal(body, &u.AccountLockOrUnlockedDetails); err != nil {
			return err
		}

	case "emm_error_details":
		if err = json.Unmarshal(body, &u.EmmErrorDetails); err != nil {
			return err
		}

	case "guest_admin_signed_in_via_trusted_teams_details":
		if err = json.Unmarshal(body, &u.GuestAdminSignedInViaTrustedTeamsDetails); err != nil {
			return err
		}

	case "guest_admin_signed_out_via_trusted_teams_details":
		if err = json.Unmarshal(body, &u.GuestAdminSignedOutViaTrustedTeamsDetails); err != nil {
			return err
		}

	case "login_fail_details":
		if err = json.Unmarshal(body, &u.LoginFailDetails); err != nil {
			return err
		}

	case "login_success_details":
		if err = json.Unmarshal(body, &u.LoginSuccessDetails); err != nil {
			return err
		}

	case "logout_details":
		if err = json.Unmarshal(body, &u.LogoutDetails); err != nil {
			return err
		}

	case "reseller_support_session_end_details":
		if err = json.Unmarshal(body, &u.ResellerSupportSessionEndDetails); err != nil {
			return err
		}

	case "reseller_support_session_start_details":
		if err = json.Unmarshal(body, &u.ResellerSupportSessionStartDetails); err != nil {
			return err
		}

	case "sign_in_as_session_end_details":
		if err = json.Unmarshal(body, &u.SignInAsSessionEndDetails); err != nil {
			return err
		}

	case "sign_in_as_session_start_details":
		if err = json.Unmarshal(body, &u.SignInAsSessionStartDetails); err != nil {
			return err
		}

	case "sso_error_details":
		if err = json.Unmarshal(body, &u.SsoErrorDetails); err != nil {
			return err
		}

	case "create_team_invite_link_details":
		if err = json.Unmarshal(body, &u.CreateTeamInviteLinkDetails); err != nil {
			return err
		}

	case "delete_team_invite_link_details":
		if err = json.Unmarshal(body, &u.DeleteTeamInviteLinkDetails); err != nil {
			return err
		}

	case "member_add_external_id_details":
		if err = json.Unmarshal(body, &u.MemberAddExternalIdDetails); err != nil {
			return err
		}

	case "member_add_name_details":
		if err = json.Unmarshal(body, &u.MemberAddNameDetails); err != nil {
			return err
		}

	case "member_change_admin_role_details":
		if err = json.Unmarshal(body, &u.MemberChangeAdminRoleDetails); err != nil {
			return err
		}

	case "member_change_email_details":
		if err = json.Unmarshal(body, &u.MemberChangeEmailDetails); err != nil {
			return err
		}

	case "member_change_external_id_details":
		if err = json.Unmarshal(body, &u.MemberChangeExternalIdDetails); err != nil {
			return err
		}

	case "member_change_membership_type_details":
		if err = json.Unmarshal(body, &u.MemberChangeMembershipTypeDetails); err != nil {
			return err
		}

	case "member_change_name_details":
		if err = json.Unmarshal(body, &u.MemberChangeNameDetails); err != nil {
			return err
		}

	case "member_change_reseller_role_details":
		if err = json.Unmarshal(body, &u.MemberChangeResellerRoleDetails); err != nil {
			return err
		}

	case "member_change_status_details":
		if err = json.Unmarshal(body, &u.MemberChangeStatusDetails); err != nil {
			return err
		}

	case "member_delete_manual_contacts_details":
		if err = json.Unmarshal(body, &u.MemberDeleteManualContactsDetails); err != nil {
			return err
		}

	case "member_delete_profile_photo_details":
		if err = json.Unmarshal(body, &u.MemberDeleteProfilePhotoDetails); err != nil {
			return err
		}

	case "member_permanently_delete_account_contents_details":
		if err = json.Unmarshal(body, &u.MemberPermanentlyDeleteAccountContentsDetails); err != nil {
			return err
		}

	case "member_remove_external_id_details":
		if err = json.Unmarshal(body, &u.MemberRemoveExternalIdDetails); err != nil {
			return err
		}

	case "member_set_profile_photo_details":
		if err = json.Unmarshal(body, &u.MemberSetProfilePhotoDetails); err != nil {
			return err
		}

	case "member_space_limits_add_custom_quota_details":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsAddCustomQuotaDetails); err != nil {
			return err
		}

	case "member_space_limits_change_custom_quota_details":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsChangeCustomQuotaDetails); err != nil {
			return err
		}

	case "member_space_limits_change_status_details":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsChangeStatusDetails); err != nil {
			return err
		}

	case "member_space_limits_remove_custom_quota_details":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsRemoveCustomQuotaDetails); err != nil {
			return err
		}

	case "member_suggest_details":
		if err = json.Unmarshal(body, &u.MemberSuggestDetails); err != nil {
			return err
		}

	case "member_transfer_account_contents_details":
		if err = json.Unmarshal(body, &u.MemberTransferAccountContentsDetails); err != nil {
			return err
		}

	case "pending_secondary_email_added_details":
		if err = json.Unmarshal(body, &u.PendingSecondaryEmailAddedDetails); err != nil {
			return err
		}

	case "secondary_email_deleted_details":
		if err = json.Unmarshal(body, &u.SecondaryEmailDeletedDetails); err != nil {
			return err
		}

	case "secondary_email_verified_details":
		if err = json.Unmarshal(body, &u.SecondaryEmailVerifiedDetails); err != nil {
			return err
		}

	case "secondary_mails_policy_changed_details":
		if err = json.Unmarshal(body, &u.SecondaryMailsPolicyChangedDetails); err != nil {
			return err
		}

	case "binder_add_page_details":
		if err = json.Unmarshal(body, &u.BinderAddPageDetails); err != nil {
			return err
		}

	case "binder_add_section_details":
		if err = json.Unmarshal(body, &u.BinderAddSectionDetails); err != nil {
			return err
		}

	case "binder_remove_page_details":
		if err = json.Unmarshal(body, &u.BinderRemovePageDetails); err != nil {
			return err
		}

	case "binder_remove_section_details":
		if err = json.Unmarshal(body, &u.BinderRemoveSectionDetails); err != nil {
			return err
		}

	case "binder_rename_page_details":
		if err = json.Unmarshal(body, &u.BinderRenamePageDetails); err != nil {
			return err
		}

	case "binder_rename_section_details":
		if err = json.Unmarshal(body, &u.BinderRenameSectionDetails); err != nil {
			return err
		}

	case "binder_reorder_page_details":
		if err = json.Unmarshal(body, &u.BinderReorderPageDetails); err != nil {
			return err
		}

	case "binder_reorder_section_details":
		if err = json.Unmarshal(body, &u.BinderReorderSectionDetails); err != nil {
			return err
		}

	case "paper_content_add_member_details":
		if err = json.Unmarshal(body, &u.PaperContentAddMemberDetails); err != nil {
			return err
		}

	case "paper_content_add_to_folder_details":
		if err = json.Unmarshal(body, &u.PaperContentAddToFolderDetails); err != nil {
			return err
		}

	case "paper_content_archive_details":
		if err = json.Unmarshal(body, &u.PaperContentArchiveDetails); err != nil {
			return err
		}

	case "paper_content_create_details":
		if err = json.Unmarshal(body, &u.PaperContentCreateDetails); err != nil {
			return err
		}

	case "paper_content_permanently_delete_details":
		if err = json.Unmarshal(body, &u.PaperContentPermanentlyDeleteDetails); err != nil {
			return err
		}

	case "paper_content_remove_from_folder_details":
		if err = json.Unmarshal(body, &u.PaperContentRemoveFromFolderDetails); err != nil {
			return err
		}

	case "paper_content_remove_member_details":
		if err = json.Unmarshal(body, &u.PaperContentRemoveMemberDetails); err != nil {
			return err
		}

	case "paper_content_rename_details":
		if err = json.Unmarshal(body, &u.PaperContentRenameDetails); err != nil {
			return err
		}

	case "paper_content_restore_details":
		if err = json.Unmarshal(body, &u.PaperContentRestoreDetails); err != nil {
			return err
		}

	case "paper_doc_add_comment_details":
		if err = json.Unmarshal(body, &u.PaperDocAddCommentDetails); err != nil {
			return err
		}

	case "paper_doc_change_member_role_details":
		if err = json.Unmarshal(body, &u.PaperDocChangeMemberRoleDetails); err != nil {
			return err
		}

	case "paper_doc_change_sharing_policy_details":
		if err = json.Unmarshal(body, &u.PaperDocChangeSharingPolicyDetails); err != nil {
			return err
		}

	case "paper_doc_change_subscription_details":
		if err = json.Unmarshal(body, &u.PaperDocChangeSubscriptionDetails); err != nil {
			return err
		}

	case "paper_doc_deleted_details":
		if err = json.Unmarshal(body, &u.PaperDocDeletedDetails); err != nil {
			return err
		}

	case "paper_doc_delete_comment_details":
		if err = json.Unmarshal(body, &u.PaperDocDeleteCommentDetails); err != nil {
			return err
		}

	case "paper_doc_download_details":
		if err = json.Unmarshal(body, &u.PaperDocDownloadDetails); err != nil {
			return err
		}

	case "paper_doc_edit_details":
		if err = json.Unmarshal(body, &u.PaperDocEditDetails); err != nil {
			return err
		}

	case "paper_doc_edit_comment_details":
		if err = json.Unmarshal(body, &u.PaperDocEditCommentDetails); err != nil {
			return err
		}

	case "paper_doc_followed_details":
		if err = json.Unmarshal(body, &u.PaperDocFollowedDetails); err != nil {
			return err
		}

	case "paper_doc_mention_details":
		if err = json.Unmarshal(body, &u.PaperDocMentionDetails); err != nil {
			return err
		}

	case "paper_doc_ownership_changed_details":
		if err = json.Unmarshal(body, &u.PaperDocOwnershipChangedDetails); err != nil {
			return err
		}

	case "paper_doc_request_access_details":
		if err = json.Unmarshal(body, &u.PaperDocRequestAccessDetails); err != nil {
			return err
		}

	case "paper_doc_resolve_comment_details":
		if err = json.Unmarshal(body, &u.PaperDocResolveCommentDetails); err != nil {
			return err
		}

	case "paper_doc_revert_details":
		if err = json.Unmarshal(body, &u.PaperDocRevertDetails); err != nil {
			return err
		}

	case "paper_doc_slack_share_details":
		if err = json.Unmarshal(body, &u.PaperDocSlackShareDetails); err != nil {
			return err
		}

	case "paper_doc_team_invite_details":
		if err = json.Unmarshal(body, &u.PaperDocTeamInviteDetails); err != nil {
			return err
		}

	case "paper_doc_trashed_details":
		if err = json.Unmarshal(body, &u.PaperDocTrashedDetails); err != nil {
			return err
		}

	case "paper_doc_unresolve_comment_details":
		if err = json.Unmarshal(body, &u.PaperDocUnresolveCommentDetails); err != nil {
			return err
		}

	case "paper_doc_untrashed_details":
		if err = json.Unmarshal(body, &u.PaperDocUntrashedDetails); err != nil {
			return err
		}

	case "paper_doc_view_details":
		if err = json.Unmarshal(body, &u.PaperDocViewDetails); err != nil {
			return err
		}

	case "paper_external_view_allow_details":
		if err = json.Unmarshal(body, &u.PaperExternalViewAllowDetails); err != nil {
			return err
		}

	case "paper_external_view_default_team_details":
		if err = json.Unmarshal(body, &u.PaperExternalViewDefaultTeamDetails); err != nil {
			return err
		}

	case "paper_external_view_forbid_details":
		if err = json.Unmarshal(body, &u.PaperExternalViewForbidDetails); err != nil {
			return err
		}

	case "paper_folder_change_subscription_details":
		if err = json.Unmarshal(body, &u.PaperFolderChangeSubscriptionDetails); err != nil {
			return err
		}

	case "paper_folder_deleted_details":
		if err = json.Unmarshal(body, &u.PaperFolderDeletedDetails); err != nil {
			return err
		}

	case "paper_folder_followed_details":
		if err = json.Unmarshal(body, &u.PaperFolderFollowedDetails); err != nil {
			return err
		}

	case "paper_folder_team_invite_details":
		if err = json.Unmarshal(body, &u.PaperFolderTeamInviteDetails); err != nil {
			return err
		}

	case "paper_published_link_change_permission_details":
		if err = json.Unmarshal(body, &u.PaperPublishedLinkChangePermissionDetails); err != nil {
			return err
		}

	case "paper_published_link_create_details":
		if err = json.Unmarshal(body, &u.PaperPublishedLinkCreateDetails); err != nil {
			return err
		}

	case "paper_published_link_disabled_details":
		if err = json.Unmarshal(body, &u.PaperPublishedLinkDisabledDetails); err != nil {
			return err
		}

	case "paper_published_link_view_details":
		if err = json.Unmarshal(body, &u.PaperPublishedLinkViewDetails); err != nil {
			return err
		}

	case "password_change_details":
		if err = json.Unmarshal(body, &u.PasswordChangeDetails); err != nil {
			return err
		}

	case "password_reset_details":
		if err = json.Unmarshal(body, &u.PasswordResetDetails); err != nil {
			return err
		}

	case "password_reset_all_details":
		if err = json.Unmarshal(body, &u.PasswordResetAllDetails); err != nil {
			return err
		}

	case "classification_create_report_details":
		if err = json.Unmarshal(body, &u.ClassificationCreateReportDetails); err != nil {
			return err
		}

	case "classification_create_report_fail_details":
		if err = json.Unmarshal(body, &u.ClassificationCreateReportFailDetails); err != nil {
			return err
		}

	case "emm_create_exceptions_report_details":
		if err = json.Unmarshal(body, &u.EmmCreateExceptionsReportDetails); err != nil {
			return err
		}

	case "emm_create_usage_report_details":
		if err = json.Unmarshal(body, &u.EmmCreateUsageReportDetails); err != nil {
			return err
		}

	case "export_members_report_details":
		if err = json.Unmarshal(body, &u.ExportMembersReportDetails); err != nil {
			return err
		}

	case "export_members_report_fail_details":
		if err = json.Unmarshal(body, &u.ExportMembersReportFailDetails); err != nil {
			return err
		}

	case "external_sharing_create_report_details":
		if err = json.Unmarshal(body, &u.ExternalSharingCreateReportDetails); err != nil {
			return err
		}

	case "external_sharing_report_failed_details":
		if err = json.Unmarshal(body, &u.ExternalSharingReportFailedDetails); err != nil {
			return err
		}

	case "no_expiration_link_gen_create_report_details":
		if err = json.Unmarshal(body, &u.NoExpirationLinkGenCreateReportDetails); err != nil {
			return err
		}

	case "no_expiration_link_gen_report_failed_details":
		if err = json.Unmarshal(body, &u.NoExpirationLinkGenReportFailedDetails); err != nil {
			return err
		}

	case "no_password_link_gen_create_report_details":
		if err = json.Unmarshal(body, &u.NoPasswordLinkGenCreateReportDetails); err != nil {
			return err
		}

	case "no_password_link_gen_report_failed_details":
		if err = json.Unmarshal(body, &u.NoPasswordLinkGenReportFailedDetails); err != nil {
			return err
		}

	case "no_password_link_view_create_report_details":
		if err = json.Unmarshal(body, &u.NoPasswordLinkViewCreateReportDetails); err != nil {
			return err
		}

	case "no_password_link_view_report_failed_details":
		if err = json.Unmarshal(body, &u.NoPasswordLinkViewReportFailedDetails); err != nil {
			return err
		}

	case "outdated_link_view_create_report_details":
		if err = json.Unmarshal(body, &u.OutdatedLinkViewCreateReportDetails); err != nil {
			return err
		}

	case "outdated_link_view_report_failed_details":
		if err = json.Unmarshal(body, &u.OutdatedLinkViewReportFailedDetails); err != nil {
			return err
		}

	case "paper_admin_export_start_details":
		if err = json.Unmarshal(body, &u.PaperAdminExportStartDetails); err != nil {
			return err
		}

	case "smart_sync_create_admin_privilege_report_details":
		if err = json.Unmarshal(body, &u.SmartSyncCreateAdminPrivilegeReportDetails); err != nil {
			return err
		}

	case "team_activity_create_report_details":
		if err = json.Unmarshal(body, &u.TeamActivityCreateReportDetails); err != nil {
			return err
		}

	case "team_activity_create_report_fail_details":
		if err = json.Unmarshal(body, &u.TeamActivityCreateReportFailDetails); err != nil {
			return err
		}

	case "collection_share_details":
		if err = json.Unmarshal(body, &u.CollectionShareDetails); err != nil {
			return err
		}

	case "file_transfers_file_add_details":
		if err = json.Unmarshal(body, &u.FileTransfersFileAddDetails); err != nil {
			return err
		}

	case "file_transfers_transfer_delete_details":
		if err = json.Unmarshal(body, &u.FileTransfersTransferDeleteDetails); err != nil {
			return err
		}

	case "file_transfers_transfer_download_details":
		if err = json.Unmarshal(body, &u.FileTransfersTransferDownloadDetails); err != nil {
			return err
		}

	case "file_transfers_transfer_send_details":
		if err = json.Unmarshal(body, &u.FileTransfersTransferSendDetails); err != nil {
			return err
		}

	case "file_transfers_transfer_view_details":
		if err = json.Unmarshal(body, &u.FileTransfersTransferViewDetails); err != nil {
			return err
		}

	case "note_acl_invite_only_details":
		if err = json.Unmarshal(body, &u.NoteAclInviteOnlyDetails); err != nil {
			return err
		}

	case "note_acl_link_details":
		if err = json.Unmarshal(body, &u.NoteAclLinkDetails); err != nil {
			return err
		}

	case "note_acl_team_link_details":
		if err = json.Unmarshal(body, &u.NoteAclTeamLinkDetails); err != nil {
			return err
		}

	case "note_shared_details":
		if err = json.Unmarshal(body, &u.NoteSharedDetails); err != nil {
			return err
		}

	case "note_share_receive_details":
		if err = json.Unmarshal(body, &u.NoteShareReceiveDetails); err != nil {
			return err
		}

	case "open_note_shared_details":
		if err = json.Unmarshal(body, &u.OpenNoteSharedDetails); err != nil {
			return err
		}

	case "sf_add_group_details":
		if err = json.Unmarshal(body, &u.SfAddGroupDetails); err != nil {
			return err
		}

	case "sf_allow_non_members_to_view_shared_links_details":
		if err = json.Unmarshal(body, &u.SfAllowNonMembersToViewSharedLinksDetails); err != nil {
			return err
		}

	case "sf_external_invite_warn_details":
		if err = json.Unmarshal(body, &u.SfExternalInviteWarnDetails); err != nil {
			return err
		}

	case "sf_fb_invite_details":
		if err = json.Unmarshal(body, &u.SfFbInviteDetails); err != nil {
			return err
		}

	case "sf_fb_invite_change_role_details":
		if err = json.Unmarshal(body, &u.SfFbInviteChangeRoleDetails); err != nil {
			return err
		}

	case "sf_fb_uninvite_details":
		if err = json.Unmarshal(body, &u.SfFbUninviteDetails); err != nil {
			return err
		}

	case "sf_invite_group_details":
		if err = json.Unmarshal(body, &u.SfInviteGroupDetails); err != nil {
			return err
		}

	case "sf_team_grant_access_details":
		if err = json.Unmarshal(body, &u.SfTeamGrantAccessDetails); err != nil {
			return err
		}

	case "sf_team_invite_details":
		if err = json.Unmarshal(body, &u.SfTeamInviteDetails); err != nil {
			return err
		}

	case "sf_team_invite_change_role_details":
		if err = json.Unmarshal(body, &u.SfTeamInviteChangeRoleDetails); err != nil {
			return err
		}

	case "sf_team_join_details":
		if err = json.Unmarshal(body, &u.SfTeamJoinDetails); err != nil {
			return err
		}

	case "sf_team_join_from_oob_link_details":
		if err = json.Unmarshal(body, &u.SfTeamJoinFromOobLinkDetails); err != nil {
			return err
		}

	case "sf_team_uninvite_details":
		if err = json.Unmarshal(body, &u.SfTeamUninviteDetails); err != nil {
			return err
		}

	case "shared_content_add_invitees_details":
		if err = json.Unmarshal(body, &u.SharedContentAddInviteesDetails); err != nil {
			return err
		}

	case "shared_content_add_link_expiry_details":
		if err = json.Unmarshal(body, &u.SharedContentAddLinkExpiryDetails); err != nil {
			return err
		}

	case "shared_content_add_link_password_details":
		if err = json.Unmarshal(body, &u.SharedContentAddLinkPasswordDetails); err != nil {
			return err
		}

	case "shared_content_add_member_details":
		if err = json.Unmarshal(body, &u.SharedContentAddMemberDetails); err != nil {
			return err
		}

	case "shared_content_change_downloads_policy_details":
		if err = json.Unmarshal(body, &u.SharedContentChangeDownloadsPolicyDetails); err != nil {
			return err
		}

	case "shared_content_change_invitee_role_details":
		if err = json.Unmarshal(body, &u.SharedContentChangeInviteeRoleDetails); err != nil {
			return err
		}

	case "shared_content_change_link_audience_details":
		if err = json.Unmarshal(body, &u.SharedContentChangeLinkAudienceDetails); err != nil {
			return err
		}

	case "shared_content_change_link_expiry_details":
		if err = json.Unmarshal(body, &u.SharedContentChangeLinkExpiryDetails); err != nil {
			return err
		}

	case "shared_content_change_link_password_details":
		if err = json.Unmarshal(body, &u.SharedContentChangeLinkPasswordDetails); err != nil {
			return err
		}

	case "shared_content_change_member_role_details":
		if err = json.Unmarshal(body, &u.SharedContentChangeMemberRoleDetails); err != nil {
			return err
		}

	case "shared_content_change_viewer_info_policy_details":
		if err = json.Unmarshal(body, &u.SharedContentChangeViewerInfoPolicyDetails); err != nil {
			return err
		}

	case "shared_content_claim_invitation_details":
		if err = json.Unmarshal(body, &u.SharedContentClaimInvitationDetails); err != nil {
			return err
		}

	case "shared_content_copy_details":
		if err = json.Unmarshal(body, &u.SharedContentCopyDetails); err != nil {
			return err
		}

	case "shared_content_download_details":
		if err = json.Unmarshal(body, &u.SharedContentDownloadDetails); err != nil {
			return err
		}

	case "shared_content_relinquish_membership_details":
		if err = json.Unmarshal(body, &u.SharedContentRelinquishMembershipDetails); err != nil {
			return err
		}

	case "shared_content_remove_invitees_details":
		if err = json.Unmarshal(body, &u.SharedContentRemoveInviteesDetails); err != nil {
			return err
		}

	case "shared_content_remove_link_expiry_details":
		if err = json.Unmarshal(body, &u.SharedContentRemoveLinkExpiryDetails); err != nil {
			return err
		}

	case "shared_content_remove_link_password_details":
		if err = json.Unmarshal(body, &u.SharedContentRemoveLinkPasswordDetails); err != nil {
			return err
		}

	case "shared_content_remove_member_details":
		if err = json.Unmarshal(body, &u.SharedContentRemoveMemberDetails); err != nil {
			return err
		}

	case "shared_content_request_access_details":
		if err = json.Unmarshal(body, &u.SharedContentRequestAccessDetails); err != nil {
			return err
		}

	case "shared_content_restore_invitees_details":
		if err = json.Unmarshal(body, &u.SharedContentRestoreInviteesDetails); err != nil {
			return err
		}

	case "shared_content_restore_member_details":
		if err = json.Unmarshal(body, &u.SharedContentRestoreMemberDetails); err != nil {
			return err
		}

	case "shared_content_unshare_details":
		if err = json.Unmarshal(body, &u.SharedContentUnshareDetails); err != nil {
			return err
		}

	case "shared_content_view_details":
		if err = json.Unmarshal(body, &u.SharedContentViewDetails); err != nil {
			return err
		}

	case "shared_folder_change_link_policy_details":
		if err = json.Unmarshal(body, &u.SharedFolderChangeLinkPolicyDetails); err != nil {
			return err
		}

	case "shared_folder_change_members_inheritance_policy_details":
		if err = json.Unmarshal(body, &u.SharedFolderChangeMembersInheritancePolicyDetails); err != nil {
			return err
		}

	case "shared_folder_change_members_management_policy_details":
		if err = json.Unmarshal(body, &u.SharedFolderChangeMembersManagementPolicyDetails); err != nil {
			return err
		}

	case "shared_folder_change_members_policy_details":
		if err = json.Unmarshal(body, &u.SharedFolderChangeMembersPolicyDetails); err != nil {
			return err
		}

	case "shared_folder_create_details":
		if err = json.Unmarshal(body, &u.SharedFolderCreateDetails); err != nil {
			return err
		}

	case "shared_folder_decline_invitation_details":
		if err = json.Unmarshal(body, &u.SharedFolderDeclineInvitationDetails); err != nil {
			return err
		}

	case "shared_folder_mount_details":
		if err = json.Unmarshal(body, &u.SharedFolderMountDetails); err != nil {
			return err
		}

	case "shared_folder_nest_details":
		if err = json.Unmarshal(body, &u.SharedFolderNestDetails); err != nil {
			return err
		}

	case "shared_folder_transfer_ownership_details":
		if err = json.Unmarshal(body, &u.SharedFolderTransferOwnershipDetails); err != nil {
			return err
		}

	case "shared_folder_unmount_details":
		if err = json.Unmarshal(body, &u.SharedFolderUnmountDetails); err != nil {
			return err
		}

	case "shared_link_add_expiry_details":
		if err = json.Unmarshal(body, &u.SharedLinkAddExpiryDetails); err != nil {
			return err
		}

	case "shared_link_change_expiry_details":
		if err = json.Unmarshal(body, &u.SharedLinkChangeExpiryDetails); err != nil {
			return err
		}

	case "shared_link_change_visibility_details":
		if err = json.Unmarshal(body, &u.SharedLinkChangeVisibilityDetails); err != nil {
			return err
		}

	case "shared_link_copy_details":
		if err = json.Unmarshal(body, &u.SharedLinkCopyDetails); err != nil {
			return err
		}

	case "shared_link_create_details":
		if err = json.Unmarshal(body, &u.SharedLinkCreateDetails); err != nil {
			return err
		}

	case "shared_link_disable_details":
		if err = json.Unmarshal(body, &u.SharedLinkDisableDetails); err != nil {
			return err
		}

	case "shared_link_download_details":
		if err = json.Unmarshal(body, &u.SharedLinkDownloadDetails); err != nil {
			return err
		}

	case "shared_link_remove_expiry_details":
		if err = json.Unmarshal(body, &u.SharedLinkRemoveExpiryDetails); err != nil {
			return err
		}

	case "shared_link_settings_add_expiration_details":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsAddExpirationDetails); err != nil {
			return err
		}

	case "shared_link_settings_add_password_details":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsAddPasswordDetails); err != nil {
			return err
		}

	case "shared_link_settings_allow_download_disabled_details":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsAllowDownloadDisabledDetails); err != nil {
			return err
		}

	case "shared_link_settings_allow_download_enabled_details":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsAllowDownloadEnabledDetails); err != nil {
			return err
		}

	case "shared_link_settings_change_audience_details":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsChangeAudienceDetails); err != nil {
			return err
		}

	case "shared_link_settings_change_expiration_details":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsChangeExpirationDetails); err != nil {
			return err
		}

	case "shared_link_settings_change_password_details":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsChangePasswordDetails); err != nil {
			return err
		}

	case "shared_link_settings_remove_expiration_details":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsRemoveExpirationDetails); err != nil {
			return err
		}

	case "shared_link_settings_remove_password_details":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsRemovePasswordDetails); err != nil {
			return err
		}

	case "shared_link_share_details":
		if err = json.Unmarshal(body, &u.SharedLinkShareDetails); err != nil {
			return err
		}

	case "shared_link_view_details":
		if err = json.Unmarshal(body, &u.SharedLinkViewDetails); err != nil {
			return err
		}

	case "shared_note_opened_details":
		if err = json.Unmarshal(body, &u.SharedNoteOpenedDetails); err != nil {
			return err
		}

	case "shmodel_disable_downloads_details":
		if err = json.Unmarshal(body, &u.ShmodelDisableDownloadsDetails); err != nil {
			return err
		}

	case "shmodel_enable_downloads_details":
		if err = json.Unmarshal(body, &u.ShmodelEnableDownloadsDetails); err != nil {
			return err
		}

	case "shmodel_group_share_details":
		if err = json.Unmarshal(body, &u.ShmodelGroupShareDetails); err != nil {
			return err
		}

	case "showcase_access_granted_details":
		if err = json.Unmarshal(body, &u.ShowcaseAccessGrantedDetails); err != nil {
			return err
		}

	case "showcase_add_member_details":
		if err = json.Unmarshal(body, &u.ShowcaseAddMemberDetails); err != nil {
			return err
		}

	case "showcase_archived_details":
		if err = json.Unmarshal(body, &u.ShowcaseArchivedDetails); err != nil {
			return err
		}

	case "showcase_created_details":
		if err = json.Unmarshal(body, &u.ShowcaseCreatedDetails); err != nil {
			return err
		}

	case "showcase_delete_comment_details":
		if err = json.Unmarshal(body, &u.ShowcaseDeleteCommentDetails); err != nil {
			return err
		}

	case "showcase_edited_details":
		if err = json.Unmarshal(body, &u.ShowcaseEditedDetails); err != nil {
			return err
		}

	case "showcase_edit_comment_details":
		if err = json.Unmarshal(body, &u.ShowcaseEditCommentDetails); err != nil {
			return err
		}

	case "showcase_file_added_details":
		if err = json.Unmarshal(body, &u.ShowcaseFileAddedDetails); err != nil {
			return err
		}

	case "showcase_file_download_details":
		if err = json.Unmarshal(body, &u.ShowcaseFileDownloadDetails); err != nil {
			return err
		}

	case "showcase_file_removed_details":
		if err = json.Unmarshal(body, &u.ShowcaseFileRemovedDetails); err != nil {
			return err
		}

	case "showcase_file_view_details":
		if err = json.Unmarshal(body, &u.ShowcaseFileViewDetails); err != nil {
			return err
		}

	case "showcase_permanently_deleted_details":
		if err = json.Unmarshal(body, &u.ShowcasePermanentlyDeletedDetails); err != nil {
			return err
		}

	case "showcase_post_comment_details":
		if err = json.Unmarshal(body, &u.ShowcasePostCommentDetails); err != nil {
			return err
		}

	case "showcase_remove_member_details":
		if err = json.Unmarshal(body, &u.ShowcaseRemoveMemberDetails); err != nil {
			return err
		}

	case "showcase_renamed_details":
		if err = json.Unmarshal(body, &u.ShowcaseRenamedDetails); err != nil {
			return err
		}

	case "showcase_request_access_details":
		if err = json.Unmarshal(body, &u.ShowcaseRequestAccessDetails); err != nil {
			return err
		}

	case "showcase_resolve_comment_details":
		if err = json.Unmarshal(body, &u.ShowcaseResolveCommentDetails); err != nil {
			return err
		}

	case "showcase_restored_details":
		if err = json.Unmarshal(body, &u.ShowcaseRestoredDetails); err != nil {
			return err
		}

	case "showcase_trashed_details":
		if err = json.Unmarshal(body, &u.ShowcaseTrashedDetails); err != nil {
			return err
		}

	case "showcase_trashed_deprecated_details":
		if err = json.Unmarshal(body, &u.ShowcaseTrashedDeprecatedDetails); err != nil {
			return err
		}

	case "showcase_unresolve_comment_details":
		if err = json.Unmarshal(body, &u.ShowcaseUnresolveCommentDetails); err != nil {
			return err
		}

	case "showcase_untrashed_details":
		if err = json.Unmarshal(body, &u.ShowcaseUntrashedDetails); err != nil {
			return err
		}

	case "showcase_untrashed_deprecated_details":
		if err = json.Unmarshal(body, &u.ShowcaseUntrashedDeprecatedDetails); err != nil {
			return err
		}

	case "showcase_view_details":
		if err = json.Unmarshal(body, &u.ShowcaseViewDetails); err != nil {
			return err
		}

	case "sso_add_cert_details":
		if err = json.Unmarshal(body, &u.SsoAddCertDetails); err != nil {
			return err
		}

	case "sso_add_login_url_details":
		if err = json.Unmarshal(body, &u.SsoAddLoginUrlDetails); err != nil {
			return err
		}

	case "sso_add_logout_url_details":
		if err = json.Unmarshal(body, &u.SsoAddLogoutUrlDetails); err != nil {
			return err
		}

	case "sso_change_cert_details":
		if err = json.Unmarshal(body, &u.SsoChangeCertDetails); err != nil {
			return err
		}

	case "sso_change_login_url_details":
		if err = json.Unmarshal(body, &u.SsoChangeLoginUrlDetails); err != nil {
			return err
		}

	case "sso_change_logout_url_details":
		if err = json.Unmarshal(body, &u.SsoChangeLogoutUrlDetails); err != nil {
			return err
		}

	case "sso_change_saml_identity_mode_details":
		if err = json.Unmarshal(body, &u.SsoChangeSamlIdentityModeDetails); err != nil {
			return err
		}

	case "sso_remove_cert_details":
		if err = json.Unmarshal(body, &u.SsoRemoveCertDetails); err != nil {
			return err
		}

	case "sso_remove_login_url_details":
		if err = json.Unmarshal(body, &u.SsoRemoveLoginUrlDetails); err != nil {
			return err
		}

	case "sso_remove_logout_url_details":
		if err = json.Unmarshal(body, &u.SsoRemoveLogoutUrlDetails); err != nil {
			return err
		}

	case "team_folder_change_status_details":
		if err = json.Unmarshal(body, &u.TeamFolderChangeStatusDetails); err != nil {
			return err
		}

	case "team_folder_create_details":
		if err = json.Unmarshal(body, &u.TeamFolderCreateDetails); err != nil {
			return err
		}

	case "team_folder_downgrade_details":
		if err = json.Unmarshal(body, &u.TeamFolderDowngradeDetails); err != nil {
			return err
		}

	case "team_folder_permanently_delete_details":
		if err = json.Unmarshal(body, &u.TeamFolderPermanentlyDeleteDetails); err != nil {
			return err
		}

	case "team_folder_rename_details":
		if err = json.Unmarshal(body, &u.TeamFolderRenameDetails); err != nil {
			return err
		}

	case "team_selective_sync_settings_changed_details":
		if err = json.Unmarshal(body, &u.TeamSelectiveSyncSettingsChangedDetails); err != nil {
			return err
		}

	case "account_capture_change_policy_details":
		if err = json.Unmarshal(body, &u.AccountCaptureChangePolicyDetails); err != nil {
			return err
		}

	case "admin_email_reminders_changed_details":
		if err = json.Unmarshal(body, &u.AdminEmailRemindersChangedDetails); err != nil {
			return err
		}

	case "allow_download_disabled_details":
		if err = json.Unmarshal(body, &u.AllowDownloadDisabledDetails); err != nil {
			return err
		}

	case "allow_download_enabled_details":
		if err = json.Unmarshal(body, &u.AllowDownloadEnabledDetails); err != nil {
			return err
		}

	case "app_permissions_changed_details":
		if err = json.Unmarshal(body, &u.AppPermissionsChangedDetails); err != nil {
			return err
		}

	case "camera_uploads_policy_changed_details":
		if err = json.Unmarshal(body, &u.CameraUploadsPolicyChangedDetails); err != nil {
			return err
		}

	case "capture_transcript_policy_changed_details":
		if err = json.Unmarshal(body, &u.CaptureTranscriptPolicyChangedDetails); err != nil {
			return err
		}

	case "classification_change_policy_details":
		if err = json.Unmarshal(body, &u.ClassificationChangePolicyDetails); err != nil {
			return err
		}

	case "computer_backup_policy_changed_details":
		if err = json.Unmarshal(body, &u.ComputerBackupPolicyChangedDetails); err != nil {
			return err
		}

	case "content_administration_policy_changed_details":
		if err = json.Unmarshal(body, &u.ContentAdministrationPolicyChangedDetails); err != nil {
			return err
		}

	case "data_placement_restriction_change_policy_details":
		if err = json.Unmarshal(body, &u.DataPlacementRestrictionChangePolicyDetails); err != nil {
			return err
		}

	case "data_placement_restriction_satisfy_policy_details":
		if err = json.Unmarshal(body, &u.DataPlacementRestrictionSatisfyPolicyDetails); err != nil {
			return err
		}

	case "device_approvals_add_exception_details":
		if err = json.Unmarshal(body, &u.DeviceApprovalsAddExceptionDetails); err != nil {
			return err
		}

	case "device_approvals_change_desktop_policy_details":
		if err = json.Unmarshal(body, &u.DeviceApprovalsChangeDesktopPolicyDetails); err != nil {
			return err
		}

	case "device_approvals_change_mobile_policy_details":
		if err = json.Unmarshal(body, &u.DeviceApprovalsChangeMobilePolicyDetails); err != nil {
			return err
		}

	case "device_approvals_change_overage_action_details":
		if err = json.Unmarshal(body, &u.DeviceApprovalsChangeOverageActionDetails); err != nil {
			return err
		}

	case "device_approvals_change_unlink_action_details":
		if err = json.Unmarshal(body, &u.DeviceApprovalsChangeUnlinkActionDetails); err != nil {
			return err
		}

	case "device_approvals_remove_exception_details":
		if err = json.Unmarshal(body, &u.DeviceApprovalsRemoveExceptionDetails); err != nil {
			return err
		}

	case "directory_restrictions_add_members_details":
		if err = json.Unmarshal(body, &u.DirectoryRestrictionsAddMembersDetails); err != nil {
			return err
		}

	case "directory_restrictions_remove_members_details":
		if err = json.Unmarshal(body, &u.DirectoryRestrictionsRemoveMembersDetails); err != nil {
			return err
		}

	case "dropbox_passwords_policy_changed_details":
		if err = json.Unmarshal(body, &u.DropboxPasswordsPolicyChangedDetails); err != nil {
			return err
		}

	case "email_ingest_policy_changed_details":
		if err = json.Unmarshal(body, &u.EmailIngestPolicyChangedDetails); err != nil {
			return err
		}

	case "emm_add_exception_details":
		if err = json.Unmarshal(body, &u.EmmAddExceptionDetails); err != nil {
			return err
		}

	case "emm_change_policy_details":
		if err = json.Unmarshal(body, &u.EmmChangePolicyDetails); err != nil {
			return err
		}

	case "emm_remove_exception_details":
		if err = json.Unmarshal(body, &u.EmmRemoveExceptionDetails); err != nil {
			return err
		}

	case "extended_version_history_change_policy_details":
		if err = json.Unmarshal(body, &u.ExtendedVersionHistoryChangePolicyDetails); err != nil {
			return err
		}

	case "external_drive_backup_policy_changed_details":
		if err = json.Unmarshal(body, &u.ExternalDriveBackupPolicyChangedDetails); err != nil {
			return err
		}

	case "file_comments_change_policy_details":
		if err = json.Unmarshal(body, &u.FileCommentsChangePolicyDetails); err != nil {
			return err
		}

	case "file_locking_policy_changed_details":
		if err = json.Unmarshal(body, &u.FileLockingPolicyChangedDetails); err != nil {
			return err
		}

	case "file_provider_migration_policy_changed_details":
		if err = json.Unmarshal(body, &u.FileProviderMigrationPolicyChangedDetails); err != nil {
			return err
		}

	case "file_requests_change_policy_details":
		if err = json.Unmarshal(body, &u.FileRequestsChangePolicyDetails); err != nil {
			return err
		}

	case "file_requests_emails_enabled_details":
		if err = json.Unmarshal(body, &u.FileRequestsEmailsEnabledDetails); err != nil {
			return err
		}

	case "file_requests_emails_restricted_to_team_only_details":
		if err = json.Unmarshal(body, &u.FileRequestsEmailsRestrictedToTeamOnlyDetails); err != nil {
			return err
		}

	case "file_transfers_policy_changed_details":
		if err = json.Unmarshal(body, &u.FileTransfersPolicyChangedDetails); err != nil {
			return err
		}

	case "google_sso_change_policy_details":
		if err = json.Unmarshal(body, &u.GoogleSsoChangePolicyDetails); err != nil {
			return err
		}

	case "group_user_management_change_policy_details":
		if err = json.Unmarshal(body, &u.GroupUserManagementChangePolicyDetails); err != nil {
			return err
		}

	case "integration_policy_changed_details":
		if err = json.Unmarshal(body, &u.IntegrationPolicyChangedDetails); err != nil {
			return err
		}

	case "invite_acceptance_email_policy_changed_details":
		if err = json.Unmarshal(body, &u.InviteAcceptanceEmailPolicyChangedDetails); err != nil {
			return err
		}

	case "member_requests_change_policy_details":
		if err = json.Unmarshal(body, &u.MemberRequestsChangePolicyDetails); err != nil {
			return err
		}

	case "member_send_invite_policy_changed_details":
		if err = json.Unmarshal(body, &u.MemberSendInvitePolicyChangedDetails); err != nil {
			return err
		}

	case "member_space_limits_add_exception_details":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsAddExceptionDetails); err != nil {
			return err
		}

	case "member_space_limits_change_caps_type_policy_details":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsChangeCapsTypePolicyDetails); err != nil {
			return err
		}

	case "member_space_limits_change_policy_details":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsChangePolicyDetails); err != nil {
			return err
		}

	case "member_space_limits_remove_exception_details":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsRemoveExceptionDetails); err != nil {
			return err
		}

	case "member_suggestions_change_policy_details":
		if err = json.Unmarshal(body, &u.MemberSuggestionsChangePolicyDetails); err != nil {
			return err
		}

	case "microsoft_office_addin_change_policy_details":
		if err = json.Unmarshal(body, &u.MicrosoftOfficeAddinChangePolicyDetails); err != nil {
			return err
		}

	case "network_control_change_policy_details":
		if err = json.Unmarshal(body, &u.NetworkControlChangePolicyDetails); err != nil {
			return err
		}

	case "paper_change_deployment_policy_details":
		if err = json.Unmarshal(body, &u.PaperChangeDeploymentPolicyDetails); err != nil {
			return err
		}

	case "paper_change_member_link_policy_details":
		if err = json.Unmarshal(body, &u.PaperChangeMemberLinkPolicyDetails); err != nil {
			return err
		}

	case "paper_change_member_policy_details":
		if err = json.Unmarshal(body, &u.PaperChangeMemberPolicyDetails); err != nil {
			return err
		}

	case "paper_change_policy_details":
		if err = json.Unmarshal(body, &u.PaperChangePolicyDetails); err != nil {
			return err
		}

	case "paper_default_folder_policy_changed_details":
		if err = json.Unmarshal(body, &u.PaperDefaultFolderPolicyChangedDetails); err != nil {
			return err
		}

	case "paper_desktop_policy_changed_details":
		if err = json.Unmarshal(body, &u.PaperDesktopPolicyChangedDetails); err != nil {
			return err
		}

	case "paper_enabled_users_group_addition_details":
		if err = json.Unmarshal(body, &u.PaperEnabledUsersGroupAdditionDetails); err != nil {
			return err
		}

	case "paper_enabled_users_group_removal_details":
		if err = json.Unmarshal(body, &u.PaperEnabledUsersGroupRemovalDetails); err != nil {
			return err
		}

	case "password_strength_requirements_change_policy_details":
		if err = json.Unmarshal(body, &u.PasswordStrengthRequirementsChangePolicyDetails); err != nil {
			return err
		}

	case "permanent_delete_change_policy_details":
		if err = json.Unmarshal(body, &u.PermanentDeleteChangePolicyDetails); err != nil {
			return err
		}

	case "reseller_support_change_policy_details":
		if err = json.Unmarshal(body, &u.ResellerSupportChangePolicyDetails); err != nil {
			return err
		}

	case "rewind_policy_changed_details":
		if err = json.Unmarshal(body, &u.RewindPolicyChangedDetails); err != nil {
			return err
		}

	case "send_for_signature_policy_changed_details":
		if err = json.Unmarshal(body, &u.SendForSignaturePolicyChangedDetails); err != nil {
			return err
		}

	case "sharing_change_folder_join_policy_details":
		if err = json.Unmarshal(body, &u.SharingChangeFolderJoinPolicyDetails); err != nil {
			return err
		}

	case "sharing_change_link_allow_change_expiration_policy_details":
		if err = json.Unmarshal(body, &u.SharingChangeLinkAllowChangeExpirationPolicyDetails); err != nil {
			return err
		}

	case "sharing_change_link_default_expiration_policy_details":
		if err = json.Unmarshal(body, &u.SharingChangeLinkDefaultExpirationPolicyDetails); err != nil {
			return err
		}

	case "sharing_change_link_enforce_password_policy_details":
		if err = json.Unmarshal(body, &u.SharingChangeLinkEnforcePasswordPolicyDetails); err != nil {
			return err
		}

	case "sharing_change_link_policy_details":
		if err = json.Unmarshal(body, &u.SharingChangeLinkPolicyDetails); err != nil {
			return err
		}

	case "sharing_change_member_policy_details":
		if err = json.Unmarshal(body, &u.SharingChangeMemberPolicyDetails); err != nil {
			return err
		}

	case "showcase_change_download_policy_details":
		if err = json.Unmarshal(body, &u.ShowcaseChangeDownloadPolicyDetails); err != nil {
			return err
		}

	case "showcase_change_enabled_policy_details":
		if err = json.Unmarshal(body, &u.ShowcaseChangeEnabledPolicyDetails); err != nil {
			return err
		}

	case "showcase_change_external_sharing_policy_details":
		if err = json.Unmarshal(body, &u.ShowcaseChangeExternalSharingPolicyDetails); err != nil {
			return err
		}

	case "smarter_smart_sync_policy_changed_details":
		if err = json.Unmarshal(body, &u.SmarterSmartSyncPolicyChangedDetails); err != nil {
			return err
		}

	case "smart_sync_change_policy_details":
		if err = json.Unmarshal(body, &u.SmartSyncChangePolicyDetails); err != nil {
			return err
		}

	case "smart_sync_not_opt_out_details":
		if err = json.Unmarshal(body, &u.SmartSyncNotOptOutDetails); err != nil {
			return err
		}

	case "smart_sync_opt_out_details":
		if err = json.Unmarshal(body, &u.SmartSyncOptOutDetails); err != nil {
			return err
		}

	case "sso_change_policy_details":
		if err = json.Unmarshal(body, &u.SsoChangePolicyDetails); err != nil {
			return err
		}

	case "team_branding_policy_changed_details":
		if err = json.Unmarshal(body, &u.TeamBrandingPolicyChangedDetails); err != nil {
			return err
		}

	case "team_extensions_policy_changed_details":
		if err = json.Unmarshal(body, &u.TeamExtensionsPolicyChangedDetails); err != nil {
			return err
		}

	case "team_selective_sync_policy_changed_details":
		if err = json.Unmarshal(body, &u.TeamSelectiveSyncPolicyChangedDetails); err != nil {
			return err
		}

	case "team_sharing_whitelist_subjects_changed_details":
		if err = json.Unmarshal(body, &u.TeamSharingWhitelistSubjectsChangedDetails); err != nil {
			return err
		}

	case "tfa_add_exception_details":
		if err = json.Unmarshal(body, &u.TfaAddExceptionDetails); err != nil {
			return err
		}

	case "tfa_change_policy_details":
		if err = json.Unmarshal(body, &u.TfaChangePolicyDetails); err != nil {
			return err
		}

	case "tfa_remove_exception_details":
		if err = json.Unmarshal(body, &u.TfaRemoveExceptionDetails); err != nil {
			return err
		}

	case "two_account_change_policy_details":
		if err = json.Unmarshal(body, &u.TwoAccountChangePolicyDetails); err != nil {
			return err
		}

	case "viewer_info_policy_changed_details":
		if err = json.Unmarshal(body, &u.ViewerInfoPolicyChangedDetails); err != nil {
			return err
		}

	case "watermarking_policy_changed_details":
		if err = json.Unmarshal(body, &u.WatermarkingPolicyChangedDetails); err != nil {
			return err
		}

	case "web_sessions_change_active_session_limit_details":
		if err = json.Unmarshal(body, &u.WebSessionsChangeActiveSessionLimitDetails); err != nil {
			return err
		}

	case "web_sessions_change_fixed_length_policy_details":
		if err = json.Unmarshal(body, &u.WebSessionsChangeFixedLengthPolicyDetails); err != nil {
			return err
		}

	case "web_sessions_change_idle_length_policy_details":
		if err = json.Unmarshal(body, &u.WebSessionsChangeIdleLengthPolicyDetails); err != nil {
			return err
		}

	case "data_residency_migration_request_successful_details":
		if err = json.Unmarshal(body, &u.DataResidencyMigrationRequestSuccessfulDetails); err != nil {
			return err
		}

	case "data_residency_migration_request_unsuccessful_details":
		if err = json.Unmarshal(body, &u.DataResidencyMigrationRequestUnsuccessfulDetails); err != nil {
			return err
		}

	case "team_merge_from_details":
		if err = json.Unmarshal(body, &u.TeamMergeFromDetails); err != nil {
			return err
		}

	case "team_merge_to_details":
		if err = json.Unmarshal(body, &u.TeamMergeToDetails); err != nil {
			return err
		}

	case "team_profile_add_background_details":
		if err = json.Unmarshal(body, &u.TeamProfileAddBackgroundDetails); err != nil {
			return err
		}

	case "team_profile_add_logo_details":
		if err = json.Unmarshal(body, &u.TeamProfileAddLogoDetails); err != nil {
			return err
		}

	case "team_profile_change_background_details":
		if err = json.Unmarshal(body, &u.TeamProfileChangeBackgroundDetails); err != nil {
			return err
		}

	case "team_profile_change_default_language_details":
		if err = json.Unmarshal(body, &u.TeamProfileChangeDefaultLanguageDetails); err != nil {
			return err
		}

	case "team_profile_change_logo_details":
		if err = json.Unmarshal(body, &u.TeamProfileChangeLogoDetails); err != nil {
			return err
		}

	case "team_profile_change_name_details":
		if err = json.Unmarshal(body, &u.TeamProfileChangeNameDetails); err != nil {
			return err
		}

	case "team_profile_remove_background_details":
		if err = json.Unmarshal(body, &u.TeamProfileRemoveBackgroundDetails); err != nil {
			return err
		}

	case "team_profile_remove_logo_details":
		if err = json.Unmarshal(body, &u.TeamProfileRemoveLogoDetails); err != nil {
			return err
		}

	case "tfa_add_backup_phone_details":
		if err = json.Unmarshal(body, &u.TfaAddBackupPhoneDetails); err != nil {
			return err
		}

	case "tfa_add_security_key_details":
		if err = json.Unmarshal(body, &u.TfaAddSecurityKeyDetails); err != nil {
			return err
		}

	case "tfa_change_backup_phone_details":
		if err = json.Unmarshal(body, &u.TfaChangeBackupPhoneDetails); err != nil {
			return err
		}

	case "tfa_change_status_details":
		if err = json.Unmarshal(body, &u.TfaChangeStatusDetails); err != nil {
			return err
		}

	case "tfa_remove_backup_phone_details":
		if err = json.Unmarshal(body, &u.TfaRemoveBackupPhoneDetails); err != nil {
			return err
		}

	case "tfa_remove_security_key_details":
		if err = json.Unmarshal(body, &u.TfaRemoveSecurityKeyDetails); err != nil {
			return err
		}

	case "tfa_reset_details":
		if err = json.Unmarshal(body, &u.TfaResetDetails); err != nil {
			return err
		}

	case "changed_enterprise_admin_role_details":
		if err = json.Unmarshal(body, &u.ChangedEnterpriseAdminRoleDetails); err != nil {
			return err
		}

	case "changed_enterprise_connected_team_status_details":
		if err = json.Unmarshal(body, &u.ChangedEnterpriseConnectedTeamStatusDetails); err != nil {
			return err
		}

	case "ended_enterprise_admin_session_details":
		if err = json.Unmarshal(body, &u.EndedEnterpriseAdminSessionDetails); err != nil {
			return err
		}

	case "ended_enterprise_admin_session_deprecated_details":
		if err = json.Unmarshal(body, &u.EndedEnterpriseAdminSessionDeprecatedDetails); err != nil {
			return err
		}

	case "enterprise_settings_locking_details":
		if err = json.Unmarshal(body, &u.EnterpriseSettingsLockingDetails); err != nil {
			return err
		}

	case "guest_admin_change_status_details":
		if err = json.Unmarshal(body, &u.GuestAdminChangeStatusDetails); err != nil {
			return err
		}

	case "started_enterprise_admin_session_details":
		if err = json.Unmarshal(body, &u.StartedEnterpriseAdminSessionDetails); err != nil {
			return err
		}

	case "team_merge_request_accepted_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestAcceptedDetails); err != nil {
			return err
		}

	case "team_merge_request_accepted_shown_to_primary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestAcceptedShownToPrimaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_accepted_shown_to_secondary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestAcceptedShownToSecondaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_auto_canceled_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestAutoCanceledDetails); err != nil {
			return err
		}

	case "team_merge_request_canceled_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestCanceledDetails); err != nil {
			return err
		}

	case "team_merge_request_canceled_shown_to_primary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestCanceledShownToPrimaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_canceled_shown_to_secondary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestCanceledShownToSecondaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_expired_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestExpiredDetails); err != nil {
			return err
		}

	case "team_merge_request_expired_shown_to_primary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestExpiredShownToPrimaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_expired_shown_to_secondary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestExpiredShownToSecondaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_rejected_shown_to_primary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestRejectedShownToPrimaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_rejected_shown_to_secondary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestRejectedShownToSecondaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_reminder_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestReminderDetails); err != nil {
			return err
		}

	case "team_merge_request_reminder_shown_to_primary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestReminderShownToPrimaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_reminder_shown_to_secondary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestReminderShownToSecondaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_revoked_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestRevokedDetails); err != nil {
			return err
		}

	case "team_merge_request_sent_shown_to_primary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestSentShownToPrimaryTeamDetails); err != nil {
			return err
		}

	case "team_merge_request_sent_shown_to_secondary_team_details":
		if err = json.Unmarshal(body, &u.TeamMergeRequestSentShownToSecondaryTeamDetails); err != nil {
			return err
		}

	case "missing_details":
		if err = json.Unmarshal(body, &u.MissingDetails); err != nil {
			return err
		}

	}
	return nil
}

// EventType : The type of the event with description.
type EventType struct {
	dropbox.Tagged
	// AdminAlertingAlertStateChanged : (admin_alerting) Changed an alert state
	AdminAlertingAlertStateChanged *AdminAlertingAlertStateChangedType `json:"admin_alerting_alert_state_changed,omitempty"`
	// AdminAlertingChangedAlertConfig : (admin_alerting) Changed an alert
	// setting
	AdminAlertingChangedAlertConfig *AdminAlertingChangedAlertConfigType `json:"admin_alerting_changed_alert_config,omitempty"`
	// AdminAlertingTriggeredAlert : (admin_alerting) Triggered security alert
	AdminAlertingTriggeredAlert *AdminAlertingTriggeredAlertType `json:"admin_alerting_triggered_alert,omitempty"`
	// AppBlockedByPermissions : (apps) Failed to connect app for member
	AppBlockedByPermissions *AppBlockedByPermissionsType `json:"app_blocked_by_permissions,omitempty"`
	// AppLinkTeam : (apps) Linked app for team
	AppLinkTeam *AppLinkTeamType `json:"app_link_team,omitempty"`
	// AppLinkUser : (apps) Linked app for member
	AppLinkUser *AppLinkUserType `json:"app_link_user,omitempty"`
	// AppUnlinkTeam : (apps) Unlinked app for team
	AppUnlinkTeam *AppUnlinkTeamType `json:"app_unlink_team,omitempty"`
	// AppUnlinkUser : (apps) Unlinked app for member
	AppUnlinkUser *AppUnlinkUserType `json:"app_unlink_user,omitempty"`
	// IntegrationConnected : (apps) Connected integration for member
	IntegrationConnected *IntegrationConnectedType `json:"integration_connected,omitempty"`
	// IntegrationDisconnected : (apps) Disconnected integration for member
	IntegrationDisconnected *IntegrationDisconnectedType `json:"integration_disconnected,omitempty"`
	// FileAddComment : (comments) Added file comment
	FileAddComment *FileAddCommentType `json:"file_add_comment,omitempty"`
	// FileChangeCommentSubscription : (comments) Subscribed to or unsubscribed
	// from comment notifications for file
	FileChangeCommentSubscription *FileChangeCommentSubscriptionType `json:"file_change_comment_subscription,omitempty"`
	// FileDeleteComment : (comments) Deleted file comment
	FileDeleteComment *FileDeleteCommentType `json:"file_delete_comment,omitempty"`
	// FileEditComment : (comments) Edited file comment
	FileEditComment *FileEditCommentType `json:"file_edit_comment,omitempty"`
	// FileLikeComment : (comments) Liked file comment (deprecated, no longer
	// logged)
	FileLikeComment *FileLikeCommentType `json:"file_like_comment,omitempty"`
	// FileResolveComment : (comments) Resolved file comment
	FileResolveComment *FileResolveCommentType `json:"file_resolve_comment,omitempty"`
	// FileUnlikeComment : (comments) Unliked file comment (deprecated, no
	// longer logged)
	FileUnlikeComment *FileUnlikeCommentType `json:"file_unlike_comment,omitempty"`
	// FileUnresolveComment : (comments) Unresolved file comment
	FileUnresolveComment *FileUnresolveCommentType `json:"file_unresolve_comment,omitempty"`
	// GovernancePolicyAddFolders : (data_governance) Added folders to policy
	GovernancePolicyAddFolders *GovernancePolicyAddFoldersType `json:"governance_policy_add_folders,omitempty"`
	// GovernancePolicyAddFolderFailed : (data_governance) Couldn't add a folder
	// to a policy
	GovernancePolicyAddFolderFailed *GovernancePolicyAddFolderFailedType `json:"governance_policy_add_folder_failed,omitempty"`
	// GovernancePolicyContentDisposed : (data_governance) Content disposed
	GovernancePolicyContentDisposed *GovernancePolicyContentDisposedType `json:"governance_policy_content_disposed,omitempty"`
	// GovernancePolicyCreate : (data_governance) Activated a new policy
	GovernancePolicyCreate *GovernancePolicyCreateType `json:"governance_policy_create,omitempty"`
	// GovernancePolicyDelete : (data_governance) Deleted a policy
	GovernancePolicyDelete *GovernancePolicyDeleteType `json:"governance_policy_delete,omitempty"`
	// GovernancePolicyEditDetails : (data_governance) Edited policy
	GovernancePolicyEditDetails *GovernancePolicyEditDetailsType `json:"governance_policy_edit_details,omitempty"`
	// GovernancePolicyEditDuration : (data_governance) Changed policy duration
	GovernancePolicyEditDuration *GovernancePolicyEditDurationType `json:"governance_policy_edit_duration,omitempty"`
	// GovernancePolicyExportCreated : (data_governance) Created a policy
	// download
	GovernancePolicyExportCreated *GovernancePolicyExportCreatedType `json:"governance_policy_export_created,omitempty"`
	// GovernancePolicyExportRemoved : (data_governance) Removed a policy
	// download
	GovernancePolicyExportRemoved *GovernancePolicyExportRemovedType `json:"governance_policy_export_removed,omitempty"`
	// GovernancePolicyRemoveFolders : (data_governance) Removed folders from
	// policy
	GovernancePolicyRemoveFolders *GovernancePolicyRemoveFoldersType `json:"governance_policy_remove_folders,omitempty"`
	// GovernancePolicyReportCreated : (data_governance) Created a summary
	// report for a policy
	GovernancePolicyReportCreated *GovernancePolicyReportCreatedType `json:"governance_policy_report_created,omitempty"`
	// GovernancePolicyZipPartDownloaded : (data_governance) Downloaded content
	// from a policy
	GovernancePolicyZipPartDownloaded *GovernancePolicyZipPartDownloadedType `json:"governance_policy_zip_part_downloaded,omitempty"`
	// LegalHoldsActivateAHold : (data_governance) Activated a hold
	LegalHoldsActivateAHold *LegalHoldsActivateAHoldType `json:"legal_holds_activate_a_hold,omitempty"`
	// LegalHoldsAddMembers : (data_governance) Added members to a hold
	LegalHoldsAddMembers *LegalHoldsAddMembersType `json:"legal_holds_add_members,omitempty"`
	// LegalHoldsChangeHoldDetails : (data_governance) Edited details for a hold
	LegalHoldsChangeHoldDetails *LegalHoldsChangeHoldDetailsType `json:"legal_holds_change_hold_details,omitempty"`
	// LegalHoldsChangeHoldName : (data_governance) Renamed a hold
	LegalHoldsChangeHoldName *LegalHoldsChangeHoldNameType `json:"legal_holds_change_hold_name,omitempty"`
	// LegalHoldsExportAHold : (data_governance) Exported hold
	LegalHoldsExportAHold *LegalHoldsExportAHoldType `json:"legal_holds_export_a_hold,omitempty"`
	// LegalHoldsExportCancelled : (data_governance) Canceled export for a hold
	LegalHoldsExportCancelled *LegalHoldsExportCancelledType `json:"legal_holds_export_cancelled,omitempty"`
	// LegalHoldsExportDownloaded : (data_governance) Downloaded export for a
	// hold
	LegalHoldsExportDownloaded *LegalHoldsExportDownloadedType `json:"legal_holds_export_downloaded,omitempty"`
	// LegalHoldsExportRemoved : (data_governance) Removed export for a hold
	LegalHoldsExportRemoved *LegalHoldsExportRemovedType `json:"legal_holds_export_removed,omitempty"`
	// LegalHoldsReleaseAHold : (data_governance) Released a hold
	LegalHoldsReleaseAHold *LegalHoldsReleaseAHoldType `json:"legal_holds_release_a_hold,omitempty"`
	// LegalHoldsRemoveMembers : (data_governance) Removed members from a hold
	LegalHoldsRemoveMembers *LegalHoldsRemoveMembersType `json:"legal_holds_remove_members,omitempty"`
	// LegalHoldsReportAHold : (data_governance) Created a summary report for a
	// hold
	LegalHoldsReportAHold *LegalHoldsReportAHoldType `json:"legal_holds_report_a_hold,omitempty"`
	// DeviceChangeIpDesktop : (devices) Changed IP address associated with
	// active desktop session
	DeviceChangeIpDesktop *DeviceChangeIpDesktopType `json:"device_change_ip_desktop,omitempty"`
	// DeviceChangeIpMobile : (devices) Changed IP address associated with
	// active mobile session
	DeviceChangeIpMobile *DeviceChangeIpMobileType `json:"device_change_ip_mobile,omitempty"`
	// DeviceChangeIpWeb : (devices) Changed IP address associated with active
	// web session
	DeviceChangeIpWeb *DeviceChangeIpWebType `json:"device_change_ip_web,omitempty"`
	// DeviceDeleteOnUnlinkFail : (devices) Failed to delete all files from
	// unlinked device
	DeviceDeleteOnUnlinkFail *DeviceDeleteOnUnlinkFailType `json:"device_delete_on_unlink_fail,omitempty"`
	// DeviceDeleteOnUnlinkSuccess : (devices) Deleted all files from unlinked
	// device
	DeviceDeleteOnUnlinkSuccess *DeviceDeleteOnUnlinkSuccessType `json:"device_delete_on_unlink_success,omitempty"`
	// DeviceLinkFail : (devices) Failed to link device
	DeviceLinkFail *DeviceLinkFailType `json:"device_link_fail,omitempty"`
	// DeviceLinkSuccess : (devices) Linked device
	DeviceLinkSuccess *DeviceLinkSuccessType `json:"device_link_success,omitempty"`
	// DeviceManagementDisabled : (devices) Disabled device management
	// (deprecated, no longer logged)
	DeviceManagementDisabled *DeviceManagementDisabledType `json:"device_management_disabled,omitempty"`
	// DeviceManagementEnabled : (devices) Enabled device management
	// (deprecated, no longer logged)
	DeviceManagementEnabled *DeviceManagementEnabledType `json:"device_management_enabled,omitempty"`
	// DeviceSyncBackupStatusChanged : (devices) Enabled/disabled backup for
	// computer
	DeviceSyncBackupStatusChanged *DeviceSyncBackupStatusChangedType `json:"device_sync_backup_status_changed,omitempty"`
	// DeviceUnlink : (devices) Disconnected device
	DeviceUnlink *DeviceUnlinkType `json:"device_unlink,omitempty"`
	// DropboxPasswordsExported : (devices) Exported passwords
	DropboxPasswordsExported *DropboxPasswordsExportedType `json:"dropbox_passwords_exported,omitempty"`
	// DropboxPasswordsNewDeviceEnrolled : (devices) Enrolled new Dropbox
	// Passwords device
	DropboxPasswordsNewDeviceEnrolled *DropboxPasswordsNewDeviceEnrolledType `json:"dropbox_passwords_new_device_enrolled,omitempty"`
	// EmmRefreshAuthToken : (devices) Refreshed auth token used for setting up
	// EMM
	EmmRefreshAuthToken *EmmRefreshAuthTokenType `json:"emm_refresh_auth_token,omitempty"`
	// ExternalDriveBackupEligibilityStatusChecked : (devices) Checked external
	// drive backup eligibility status
	ExternalDriveBackupEligibilityStatusChecked *ExternalDriveBackupEligibilityStatusCheckedType `json:"external_drive_backup_eligibility_status_checked,omitempty"`
	// ExternalDriveBackupStatusChanged : (devices) Modified external drive
	// backup
	ExternalDriveBackupStatusChanged *ExternalDriveBackupStatusChangedType `json:"external_drive_backup_status_changed,omitempty"`
	// AccountCaptureChangeAvailability : (domains) Granted/revoked option to
	// enable account capture on team domains
	AccountCaptureChangeAvailability *AccountCaptureChangeAvailabilityType `json:"account_capture_change_availability,omitempty"`
	// AccountCaptureMigrateAccount : (domains) Account-captured user migrated
	// account to team
	AccountCaptureMigrateAccount *AccountCaptureMigrateAccountType `json:"account_capture_migrate_account,omitempty"`
	// AccountCaptureNotificationEmailsSent : (domains) Sent account capture
	// email to all unmanaged members
	AccountCaptureNotificationEmailsSent *AccountCaptureNotificationEmailsSentType `json:"account_capture_notification_emails_sent,omitempty"`
	// AccountCaptureRelinquishAccount : (domains) Account-captured user changed
	// account email to personal email
	AccountCaptureRelinquishAccount *AccountCaptureRelinquishAccountType `json:"account_capture_relinquish_account,omitempty"`
	// DisabledDomainInvites : (domains) Disabled domain invites (deprecated, no
	// longer logged)
	DisabledDomainInvites *DisabledDomainInvitesType `json:"disabled_domain_invites,omitempty"`
	// DomainInvitesApproveRequestToJoinTeam : (domains) Approved user's request
	// to join team
	DomainInvitesApproveRequestToJoinTeam *DomainInvitesApproveRequestToJoinTeamType `json:"domain_invites_approve_request_to_join_team,omitempty"`
	// DomainInvitesDeclineRequestToJoinTeam : (domains) Declined user's request
	// to join team
	DomainInvitesDeclineRequestToJoinTeam *DomainInvitesDeclineRequestToJoinTeamType `json:"domain_invites_decline_request_to_join_team,omitempty"`
	// DomainInvitesEmailExistingUsers : (domains) Sent domain invites to
	// existing domain accounts (deprecated, no longer logged)
	DomainInvitesEmailExistingUsers *DomainInvitesEmailExistingUsersType `json:"domain_invites_email_existing_users,omitempty"`
	// DomainInvitesRequestToJoinTeam : (domains) Requested to join team
	DomainInvitesRequestToJoinTeam *DomainInvitesRequestToJoinTeamType `json:"domain_invites_request_to_join_team,omitempty"`
	// DomainInvitesSetInviteNewUserPrefToNo : (domains) Disabled "Automatically
	// invite new users" (deprecated, no longer logged)
	DomainInvitesSetInviteNewUserPrefToNo *DomainInvitesSetInviteNewUserPrefToNoType `json:"domain_invites_set_invite_new_user_pref_to_no,omitempty"`
	// DomainInvitesSetInviteNewUserPrefToYes : (domains) Enabled "Automatically
	// invite new users" (deprecated, no longer logged)
	DomainInvitesSetInviteNewUserPrefToYes *DomainInvitesSetInviteNewUserPrefToYesType `json:"domain_invites_set_invite_new_user_pref_to_yes,omitempty"`
	// DomainVerificationAddDomainFail : (domains) Failed to verify team domain
	DomainVerificationAddDomainFail *DomainVerificationAddDomainFailType `json:"domain_verification_add_domain_fail,omitempty"`
	// DomainVerificationAddDomainSuccess : (domains) Verified team domain
	DomainVerificationAddDomainSuccess *DomainVerificationAddDomainSuccessType `json:"domain_verification_add_domain_success,omitempty"`
	// DomainVerificationRemoveDomain : (domains) Removed domain from list of
	// verified team domains
	DomainVerificationRemoveDomain *DomainVerificationRemoveDomainType `json:"domain_verification_remove_domain,omitempty"`
	// EnabledDomainInvites : (domains) Enabled domain invites (deprecated, no
	// longer logged)
	EnabledDomainInvites *EnabledDomainInvitesType `json:"enabled_domain_invites,omitempty"`
	// ApplyNamingConvention : (file_operations) Applied naming convention
	ApplyNamingConvention *ApplyNamingConventionType `json:"apply_naming_convention,omitempty"`
	// CreateFolder : (file_operations) Created folders (deprecated, no longer
	// logged)
	CreateFolder *CreateFolderType `json:"create_folder,omitempty"`
	// FileAdd : (file_operations) Added files and/or folders
	FileAdd *FileAddType `json:"file_add,omitempty"`
	// FileCopy : (file_operations) Copied files and/or folders
	FileCopy *FileCopyType `json:"file_copy,omitempty"`
	// FileDelete : (file_operations) Deleted files and/or folders
	FileDelete *FileDeleteType `json:"file_delete,omitempty"`
	// FileDownload : (file_operations) Downloaded files and/or folders
	FileDownload *FileDownloadType `json:"file_download,omitempty"`
	// FileEdit : (file_operations) Edited files
	FileEdit *FileEditType `json:"file_edit,omitempty"`
	// FileGetCopyReference : (file_operations) Created copy reference to
	// file/folder
	FileGetCopyReference *FileGetCopyReferenceType `json:"file_get_copy_reference,omitempty"`
	// FileLockingLockStatusChanged : (file_operations) Locked/unlocked editing
	// for a file
	FileLockingLockStatusChanged *FileLockingLockStatusChangedType `json:"file_locking_lock_status_changed,omitempty"`
	// FileMove : (file_operations) Moved files and/or folders
	FileMove *FileMoveType `json:"file_move,omitempty"`
	// FilePermanentlyDelete : (file_operations) Permanently deleted files
	// and/or folders
	FilePermanentlyDelete *FilePermanentlyDeleteType `json:"file_permanently_delete,omitempty"`
	// FilePreview : (file_operations) Previewed files and/or folders
	FilePreview *FilePreviewType `json:"file_preview,omitempty"`
	// FileRename : (file_operations) Renamed files and/or folders
	FileRename *FileRenameType `json:"file_rename,omitempty"`
	// FileRestore : (file_operations) Restored deleted files and/or folders
	FileRestore *FileRestoreType `json:"file_restore,omitempty"`
	// FileRevert : (file_operations) Reverted files to previous version
	FileRevert *FileRevertType `json:"file_revert,omitempty"`
	// FileRollbackChanges : (file_operations) Rolled back file actions
	FileRollbackChanges *FileRollbackChangesType `json:"file_rollback_changes,omitempty"`
	// FileSaveCopyReference : (file_operations) Saved file/folder using copy
	// reference
	FileSaveCopyReference *FileSaveCopyReferenceType `json:"file_save_copy_reference,omitempty"`
	// FolderOverviewDescriptionChanged : (file_operations) Updated folder
	// overview
	FolderOverviewDescriptionChanged *FolderOverviewDescriptionChangedType `json:"folder_overview_description_changed,omitempty"`
	// FolderOverviewItemPinned : (file_operations) Pinned item to folder
	// overview
	FolderOverviewItemPinned *FolderOverviewItemPinnedType `json:"folder_overview_item_pinned,omitempty"`
	// FolderOverviewItemUnpinned : (file_operations) Unpinned item from folder
	// overview
	FolderOverviewItemUnpinned *FolderOverviewItemUnpinnedType `json:"folder_overview_item_unpinned,omitempty"`
	// ObjectLabelAdded : (file_operations) Added a label
	ObjectLabelAdded *ObjectLabelAddedType `json:"object_label_added,omitempty"`
	// ObjectLabelRemoved : (file_operations) Removed a label
	ObjectLabelRemoved *ObjectLabelRemovedType `json:"object_label_removed,omitempty"`
	// ObjectLabelUpdatedValue : (file_operations) Updated a label's value
	ObjectLabelUpdatedValue *ObjectLabelUpdatedValueType `json:"object_label_updated_value,omitempty"`
	// OrganizeFolderWithTidy : (file_operations) Organized a folder with
	// multi-file organize
	OrganizeFolderWithTidy *OrganizeFolderWithTidyType `json:"organize_folder_with_tidy,omitempty"`
	// RewindFolder : (file_operations) Rewound a folder
	RewindFolder *RewindFolderType `json:"rewind_folder,omitempty"`
	// UndoNamingConvention : (file_operations) Reverted naming convention
	UndoNamingConvention *UndoNamingConventionType `json:"undo_naming_convention,omitempty"`
	// UndoOrganizeFolderWithTidy : (file_operations) Removed multi-file
	// organize
	UndoOrganizeFolderWithTidy *UndoOrganizeFolderWithTidyType `json:"undo_organize_folder_with_tidy,omitempty"`
	// UserTagsAdded : (file_operations) Tagged a file
	UserTagsAdded *UserTagsAddedType `json:"user_tags_added,omitempty"`
	// UserTagsRemoved : (file_operations) Removed tags
	UserTagsRemoved *UserTagsRemovedType `json:"user_tags_removed,omitempty"`
	// EmailIngestReceiveFile : (file_requests) Received files via Email to
	// Dropbox
	EmailIngestReceiveFile *EmailIngestReceiveFileType `json:"email_ingest_receive_file,omitempty"`
	// FileRequestChange : (file_requests) Changed file request
	FileRequestChange *FileRequestChangeType `json:"file_request_change,omitempty"`
	// FileRequestClose : (file_requests) Closed file request
	FileRequestClose *FileRequestCloseType `json:"file_request_close,omitempty"`
	// FileRequestCreate : (file_requests) Created file request
	FileRequestCreate *FileRequestCreateType `json:"file_request_create,omitempty"`
	// FileRequestDelete : (file_requests) Delete file request
	FileRequestDelete *FileRequestDeleteType `json:"file_request_delete,omitempty"`
	// FileRequestReceiveFile : (file_requests) Received files for file request
	FileRequestReceiveFile *FileRequestReceiveFileType `json:"file_request_receive_file,omitempty"`
	// GroupAddExternalId : (groups) Added external ID for group
	GroupAddExternalId *GroupAddExternalIdType `json:"group_add_external_id,omitempty"`
	// GroupAddMember : (groups) Added team members to group
	GroupAddMember *GroupAddMemberType `json:"group_add_member,omitempty"`
	// GroupChangeExternalId : (groups) Changed external ID for group
	GroupChangeExternalId *GroupChangeExternalIdType `json:"group_change_external_id,omitempty"`
	// GroupChangeManagementType : (groups) Changed group management type
	GroupChangeManagementType *GroupChangeManagementTypeType `json:"group_change_management_type,omitempty"`
	// GroupChangeMemberRole : (groups) Changed manager permissions of group
	// member
	GroupChangeMemberRole *GroupChangeMemberRoleType `json:"group_change_member_role,omitempty"`
	// GroupCreate : (groups) Created group
	GroupCreate *GroupCreateType `json:"group_create,omitempty"`
	// GroupDelete : (groups) Deleted group
	GroupDelete *GroupDeleteType `json:"group_delete,omitempty"`
	// GroupDescriptionUpdated : (groups) Updated group (deprecated, no longer
	// logged)
	GroupDescriptionUpdated *GroupDescriptionUpdatedType `json:"group_description_updated,omitempty"`
	// GroupJoinPolicyUpdated : (groups) Updated group join policy (deprecated,
	// no longer logged)
	GroupJoinPolicyUpdated *GroupJoinPolicyUpdatedType `json:"group_join_policy_updated,omitempty"`
	// GroupMoved : (groups) Moved group (deprecated, no longer logged)
	GroupMoved *GroupMovedType `json:"group_moved,omitempty"`
	// GroupRemoveExternalId : (groups) Removed external ID for group
	GroupRemoveExternalId *GroupRemoveExternalIdType `json:"group_remove_external_id,omitempty"`
	// GroupRemoveMember : (groups) Removed team members from group
	GroupRemoveMember *GroupRemoveMemberType `json:"group_remove_member,omitempty"`
	// GroupRename : (groups) Renamed group
	GroupRename *GroupRenameType `json:"group_rename,omitempty"`
	// AccountLockOrUnlocked : (logins) Unlocked/locked account after failed
	// sign in attempts
	AccountLockOrUnlocked *AccountLockOrUnlockedType `json:"account_lock_or_unlocked,omitempty"`
	// EmmError : (logins) Failed to sign in via EMM (deprecated, replaced by
	// 'Failed to sign in')
	EmmError *EmmErrorType `json:"emm_error,omitempty"`
	// GuestAdminSignedInViaTrustedTeams : (logins) Started trusted team admin
	// session
	GuestAdminSignedInViaTrustedTeams *GuestAdminSignedInViaTrustedTeamsType `json:"guest_admin_signed_in_via_trusted_teams,omitempty"`
	// GuestAdminSignedOutViaTrustedTeams : (logins) Ended trusted team admin
	// session
	GuestAdminSignedOutViaTrustedTeams *GuestAdminSignedOutViaTrustedTeamsType `json:"guest_admin_signed_out_via_trusted_teams,omitempty"`
	// LoginFail : (logins) Failed to sign in
	LoginFail *LoginFailType `json:"login_fail,omitempty"`
	// LoginSuccess : (logins) Signed in
	LoginSuccess *LoginSuccessType `json:"login_success,omitempty"`
	// Logout : (logins) Signed out
	Logout *LogoutType `json:"logout,omitempty"`
	// ResellerSupportSessionEnd : (logins) Ended reseller support session
	ResellerSupportSessionEnd *ResellerSupportSessionEndType `json:"reseller_support_session_end,omitempty"`
	// ResellerSupportSessionStart : (logins) Started reseller support session
	ResellerSupportSessionStart *ResellerSupportSessionStartType `json:"reseller_support_session_start,omitempty"`
	// SignInAsSessionEnd : (logins) Ended admin sign-in-as session
	SignInAsSessionEnd *SignInAsSessionEndType `json:"sign_in_as_session_end,omitempty"`
	// SignInAsSessionStart : (logins) Started admin sign-in-as session
	SignInAsSessionStart *SignInAsSessionStartType `json:"sign_in_as_session_start,omitempty"`
	// SsoError : (logins) Failed to sign in via SSO (deprecated, replaced by
	// 'Failed to sign in')
	SsoError *SsoErrorType `json:"sso_error,omitempty"`
	// CreateTeamInviteLink : (members) Created team invite link
	CreateTeamInviteLink *CreateTeamInviteLinkType `json:"create_team_invite_link,omitempty"`
	// DeleteTeamInviteLink : (members) Deleted team invite link
	DeleteTeamInviteLink *DeleteTeamInviteLinkType `json:"delete_team_invite_link,omitempty"`
	// MemberAddExternalId : (members) Added an external ID for team member
	MemberAddExternalId *MemberAddExternalIdType `json:"member_add_external_id,omitempty"`
	// MemberAddName : (members) Added team member name
	MemberAddName *MemberAddNameType `json:"member_add_name,omitempty"`
	// MemberChangeAdminRole : (members) Changed team member admin role
	MemberChangeAdminRole *MemberChangeAdminRoleType `json:"member_change_admin_role,omitempty"`
	// MemberChangeEmail : (members) Changed team member email
	MemberChangeEmail *MemberChangeEmailType `json:"member_change_email,omitempty"`
	// MemberChangeExternalId : (members) Changed the external ID for team
	// member
	MemberChangeExternalId *MemberChangeExternalIdType `json:"member_change_external_id,omitempty"`
	// MemberChangeMembershipType : (members) Changed membership type
	// (limited/full) of member (deprecated, no longer logged)
	MemberChangeMembershipType *MemberChangeMembershipTypeType `json:"member_change_membership_type,omitempty"`
	// MemberChangeName : (members) Changed team member name
	MemberChangeName *MemberChangeNameType `json:"member_change_name,omitempty"`
	// MemberChangeResellerRole : (members) Changed team member reseller role
	MemberChangeResellerRole *MemberChangeResellerRoleType `json:"member_change_reseller_role,omitempty"`
	// MemberChangeStatus : (members) Changed member status (invited, joined,
	// suspended, etc.)
	MemberChangeStatus *MemberChangeStatusType `json:"member_change_status,omitempty"`
	// MemberDeleteManualContacts : (members) Cleared manually added contacts
	MemberDeleteManualContacts *MemberDeleteManualContactsType `json:"member_delete_manual_contacts,omitempty"`
	// MemberDeleteProfilePhoto : (members) Deleted team member profile photo
	MemberDeleteProfilePhoto *MemberDeleteProfilePhotoType `json:"member_delete_profile_photo,omitempty"`
	// MemberPermanentlyDeleteAccountContents : (members) Permanently deleted
	// contents of deleted team member account
	MemberPermanentlyDeleteAccountContents *MemberPermanentlyDeleteAccountContentsType `json:"member_permanently_delete_account_contents,omitempty"`
	// MemberRemoveExternalId : (members) Removed the external ID for team
	// member
	MemberRemoveExternalId *MemberRemoveExternalIdType `json:"member_remove_external_id,omitempty"`
	// MemberSetProfilePhoto : (members) Set team member profile photo
	MemberSetProfilePhoto *MemberSetProfilePhotoType `json:"member_set_profile_photo,omitempty"`
	// MemberSpaceLimitsAddCustomQuota : (members) Set custom member space limit
	MemberSpaceLimitsAddCustomQuota *MemberSpaceLimitsAddCustomQuotaType `json:"member_space_limits_add_custom_quota,omitempty"`
	// MemberSpaceLimitsChangeCustomQuota : (members) Changed custom member
	// space limit
	MemberSpaceLimitsChangeCustomQuota *MemberSpaceLimitsChangeCustomQuotaType `json:"member_space_limits_change_custom_quota,omitempty"`
	// MemberSpaceLimitsChangeStatus : (members) Changed space limit status
	MemberSpaceLimitsChangeStatus *MemberSpaceLimitsChangeStatusType `json:"member_space_limits_change_status,omitempty"`
	// MemberSpaceLimitsRemoveCustomQuota : (members) Removed custom member
	// space limit
	MemberSpaceLimitsRemoveCustomQuota *MemberSpaceLimitsRemoveCustomQuotaType `json:"member_space_limits_remove_custom_quota,omitempty"`
	// MemberSuggest : (members) Suggested person to add to team
	MemberSuggest *MemberSuggestType `json:"member_suggest,omitempty"`
	// MemberTransferAccountContents : (members) Transferred contents of deleted
	// member account to another member
	MemberTransferAccountContents *MemberTransferAccountContentsType `json:"member_transfer_account_contents,omitempty"`
	// PendingSecondaryEmailAdded : (members) Added pending secondary email
	PendingSecondaryEmailAdded *PendingSecondaryEmailAddedType `json:"pending_secondary_email_added,omitempty"`
	// SecondaryEmailDeleted : (members) Deleted secondary email
	SecondaryEmailDeleted *SecondaryEmailDeletedType `json:"secondary_email_deleted,omitempty"`
	// SecondaryEmailVerified : (members) Verified secondary email
	SecondaryEmailVerified *SecondaryEmailVerifiedType `json:"secondary_email_verified,omitempty"`
	// SecondaryMailsPolicyChanged : (members) Secondary mails policy changed
	SecondaryMailsPolicyChanged *SecondaryMailsPolicyChangedType `json:"secondary_mails_policy_changed,omitempty"`
	// BinderAddPage : (paper) Added Binder page (deprecated, replaced by
	// 'Edited files')
	BinderAddPage *BinderAddPageType `json:"binder_add_page,omitempty"`
	// BinderAddSection : (paper) Added Binder section (deprecated, replaced by
	// 'Edited files')
	BinderAddSection *BinderAddSectionType `json:"binder_add_section,omitempty"`
	// BinderRemovePage : (paper) Removed Binder page (deprecated, replaced by
	// 'Edited files')
	BinderRemovePage *BinderRemovePageType `json:"binder_remove_page,omitempty"`
	// BinderRemoveSection : (paper) Removed Binder section (deprecated,
	// replaced by 'Edited files')
	BinderRemoveSection *BinderRemoveSectionType `json:"binder_remove_section,omitempty"`
	// BinderRenamePage : (paper) Renamed Binder page (deprecated, replaced by
	// 'Edited files')
	BinderRenamePage *BinderRenamePageType `json:"binder_rename_page,omitempty"`
	// BinderRenameSection : (paper) Renamed Binder section (deprecated,
	// replaced by 'Edited files')
	BinderRenameSection *BinderRenameSectionType `json:"binder_rename_section,omitempty"`
	// BinderReorderPage : (paper) Reordered Binder page (deprecated, replaced
	// by 'Edited files')
	BinderReorderPage *BinderReorderPageType `json:"binder_reorder_page,omitempty"`
	// BinderReorderSection : (paper) Reordered Binder section (deprecated,
	// replaced by 'Edited files')
	BinderReorderSection *BinderReorderSectionType `json:"binder_reorder_section,omitempty"`
	// PaperContentAddMember : (paper) Added users and/or groups to Paper
	// doc/folder
	PaperContentAddMember *PaperContentAddMemberType `json:"paper_content_add_member,omitempty"`
	// PaperContentAddToFolder : (paper) Added Paper doc/folder to folder
	PaperContentAddToFolder *PaperContentAddToFolderType `json:"paper_content_add_to_folder,omitempty"`
	// PaperContentArchive : (paper) Archived Paper doc/folder
	PaperContentArchive *PaperContentArchiveType `json:"paper_content_archive,omitempty"`
	// PaperContentCreate : (paper) Created Paper doc/folder
	PaperContentCreate *PaperContentCreateType `json:"paper_content_create,omitempty"`
	// PaperContentPermanentlyDelete : (paper) Permanently deleted Paper
	// doc/folder
	PaperContentPermanentlyDelete *PaperContentPermanentlyDeleteType `json:"paper_content_permanently_delete,omitempty"`
	// PaperContentRemoveFromFolder : (paper) Removed Paper doc/folder from
	// folder
	PaperContentRemoveFromFolder *PaperContentRemoveFromFolderType `json:"paper_content_remove_from_folder,omitempty"`
	// PaperContentRemoveMember : (paper) Removed users and/or groups from Paper
	// doc/folder
	PaperContentRemoveMember *PaperContentRemoveMemberType `json:"paper_content_remove_member,omitempty"`
	// PaperContentRename : (paper) Renamed Paper doc/folder
	PaperContentRename *PaperContentRenameType `json:"paper_content_rename,omitempty"`
	// PaperContentRestore : (paper) Restored archived Paper doc/folder
	PaperContentRestore *PaperContentRestoreType `json:"paper_content_restore,omitempty"`
	// PaperDocAddComment : (paper) Added Paper doc comment
	PaperDocAddComment *PaperDocAddCommentType `json:"paper_doc_add_comment,omitempty"`
	// PaperDocChangeMemberRole : (paper) Changed member permissions for Paper
	// doc
	PaperDocChangeMemberRole *PaperDocChangeMemberRoleType `json:"paper_doc_change_member_role,omitempty"`
	// PaperDocChangeSharingPolicy : (paper) Changed sharing setting for Paper
	// doc
	PaperDocChangeSharingPolicy *PaperDocChangeSharingPolicyType `json:"paper_doc_change_sharing_policy,omitempty"`
	// PaperDocChangeSubscription : (paper) Followed/unfollowed Paper doc
	PaperDocChangeSubscription *PaperDocChangeSubscriptionType `json:"paper_doc_change_subscription,omitempty"`
	// PaperDocDeleted : (paper) Archived Paper doc (deprecated, no longer
	// logged)
	PaperDocDeleted *PaperDocDeletedType `json:"paper_doc_deleted,omitempty"`
	// PaperDocDeleteComment : (paper) Deleted Paper doc comment
	PaperDocDeleteComment *PaperDocDeleteCommentType `json:"paper_doc_delete_comment,omitempty"`
	// PaperDocDownload : (paper) Downloaded Paper doc in specific format
	PaperDocDownload *PaperDocDownloadType `json:"paper_doc_download,omitempty"`
	// PaperDocEdit : (paper) Edited Paper doc
	PaperDocEdit *PaperDocEditType `json:"paper_doc_edit,omitempty"`
	// PaperDocEditComment : (paper) Edited Paper doc comment
	PaperDocEditComment *PaperDocEditCommentType `json:"paper_doc_edit_comment,omitempty"`
	// PaperDocFollowed : (paper) Followed Paper doc (deprecated, replaced by
	// 'Followed/unfollowed Paper doc')
	PaperDocFollowed *PaperDocFollowedType `json:"paper_doc_followed,omitempty"`
	// PaperDocMention : (paper) Mentioned user in Paper doc
	PaperDocMention *PaperDocMentionType `json:"paper_doc_mention,omitempty"`
	// PaperDocOwnershipChanged : (paper) Transferred ownership of Paper doc
	PaperDocOwnershipChanged *PaperDocOwnershipChangedType `json:"paper_doc_ownership_changed,omitempty"`
	// PaperDocRequestAccess : (paper) Requested access to Paper doc
	PaperDocRequestAccess *PaperDocRequestAccessType `json:"paper_doc_request_access,omitempty"`
	// PaperDocResolveComment : (paper) Resolved Paper doc comment
	PaperDocResolveComment *PaperDocResolveCommentType `json:"paper_doc_resolve_comment,omitempty"`
	// PaperDocRevert : (paper) Restored Paper doc to previous version
	PaperDocRevert *PaperDocRevertType `json:"paper_doc_revert,omitempty"`
	// PaperDocSlackShare : (paper) Shared Paper doc via Slack
	PaperDocSlackShare *PaperDocSlackShareType `json:"paper_doc_slack_share,omitempty"`
	// PaperDocTeamInvite : (paper) Shared Paper doc with users and/or groups
	// (deprecated, no longer logged)
	PaperDocTeamInvite *PaperDocTeamInviteType `json:"paper_doc_team_invite,omitempty"`
	// PaperDocTrashed : (paper) Deleted Paper doc
	PaperDocTrashed *PaperDocTrashedType `json:"paper_doc_trashed,omitempty"`
	// PaperDocUnresolveComment : (paper) Unresolved Paper doc comment
	PaperDocUnresolveComment *PaperDocUnresolveCommentType `json:"paper_doc_unresolve_comment,omitempty"`
	// PaperDocUntrashed : (paper) Restored Paper doc
	PaperDocUntrashed *PaperDocUntrashedType `json:"paper_doc_untrashed,omitempty"`
	// PaperDocView : (paper) Viewed Paper doc
	PaperDocView *PaperDocViewType `json:"paper_doc_view,omitempty"`
	// PaperExternalViewAllow : (paper) Changed Paper external sharing setting
	// to anyone (deprecated, no longer logged)
	PaperExternalViewAllow *PaperExternalViewAllowType `json:"paper_external_view_allow,omitempty"`
	// PaperExternalViewDefaultTeam : (paper) Changed Paper external sharing
	// setting to default team (deprecated, no longer logged)
	PaperExternalViewDefaultTeam *PaperExternalViewDefaultTeamType `json:"paper_external_view_default_team,omitempty"`
	// PaperExternalViewForbid : (paper) Changed Paper external sharing setting
	// to team-only (deprecated, no longer logged)
	PaperExternalViewForbid *PaperExternalViewForbidType `json:"paper_external_view_forbid,omitempty"`
	// PaperFolderChangeSubscription : (paper) Followed/unfollowed Paper folder
	PaperFolderChangeSubscription *PaperFolderChangeSubscriptionType `json:"paper_folder_change_subscription,omitempty"`
	// PaperFolderDeleted : (paper) Archived Paper folder (deprecated, no longer
	// logged)
	PaperFolderDeleted *PaperFolderDeletedType `json:"paper_folder_deleted,omitempty"`
	// PaperFolderFollowed : (paper) Followed Paper folder (deprecated, replaced
	// by 'Followed/unfollowed Paper folder')
	PaperFolderFollowed *PaperFolderFollowedType `json:"paper_folder_followed,omitempty"`
	// PaperFolderTeamInvite : (paper) Shared Paper folder with users and/or
	// groups (deprecated, no longer logged)
	PaperFolderTeamInvite *PaperFolderTeamInviteType `json:"paper_folder_team_invite,omitempty"`
	// PaperPublishedLinkChangePermission : (paper) Changed permissions for
	// published doc
	PaperPublishedLinkChangePermission *PaperPublishedLinkChangePermissionType `json:"paper_published_link_change_permission,omitempty"`
	// PaperPublishedLinkCreate : (paper) Published doc
	PaperPublishedLinkCreate *PaperPublishedLinkCreateType `json:"paper_published_link_create,omitempty"`
	// PaperPublishedLinkDisabled : (paper) Unpublished doc
	PaperPublishedLinkDisabled *PaperPublishedLinkDisabledType `json:"paper_published_link_disabled,omitempty"`
	// PaperPublishedLinkView : (paper) Viewed published doc
	PaperPublishedLinkView *PaperPublishedLinkViewType `json:"paper_published_link_view,omitempty"`
	// PasswordChange : (passwords) Changed password
	PasswordChange *PasswordChangeType `json:"password_change,omitempty"`
	// PasswordReset : (passwords) Reset password
	PasswordReset *PasswordResetType `json:"password_reset,omitempty"`
	// PasswordResetAll : (passwords) Reset all team member passwords
	PasswordResetAll *PasswordResetAllType `json:"password_reset_all,omitempty"`
	// ClassificationCreateReport : (reports) Created Classification report
	ClassificationCreateReport *ClassificationCreateReportType `json:"classification_create_report,omitempty"`
	// ClassificationCreateReportFail : (reports) Couldn't create Classification
	// report
	ClassificationCreateReportFail *ClassificationCreateReportFailType `json:"classification_create_report_fail,omitempty"`
	// EmmCreateExceptionsReport : (reports) Created EMM-excluded users report
	EmmCreateExceptionsReport *EmmCreateExceptionsReportType `json:"emm_create_exceptions_report,omitempty"`
	// EmmCreateUsageReport : (reports) Created EMM mobile app usage report
	EmmCreateUsageReport *EmmCreateUsageReportType `json:"emm_create_usage_report,omitempty"`
	// ExportMembersReport : (reports) Created member data report
	ExportMembersReport *ExportMembersReportType `json:"export_members_report,omitempty"`
	// ExportMembersReportFail : (reports) Failed to create members data report
	ExportMembersReportFail *ExportMembersReportFailType `json:"export_members_report_fail,omitempty"`
	// ExternalSharingCreateReport : (reports) Created External sharing report
	ExternalSharingCreateReport *ExternalSharingCreateReportType `json:"external_sharing_create_report,omitempty"`
	// ExternalSharingReportFailed : (reports) Couldn't create External sharing
	// report
	ExternalSharingReportFailed *ExternalSharingReportFailedType `json:"external_sharing_report_failed,omitempty"`
	// NoExpirationLinkGenCreateReport : (reports) Report created: Links created
	// with no expiration
	NoExpirationLinkGenCreateReport *NoExpirationLinkGenCreateReportType `json:"no_expiration_link_gen_create_report,omitempty"`
	// NoExpirationLinkGenReportFailed : (reports) Couldn't create report: Links
	// created with no expiration
	NoExpirationLinkGenReportFailed *NoExpirationLinkGenReportFailedType `json:"no_expiration_link_gen_report_failed,omitempty"`
	// NoPasswordLinkGenCreateReport : (reports) Report created: Links created
	// without passwords
	NoPasswordLinkGenCreateReport *NoPasswordLinkGenCreateReportType `json:"no_password_link_gen_create_report,omitempty"`
	// NoPasswordLinkGenReportFailed : (reports) Couldn't create report: Links
	// created without passwords
	NoPasswordLinkGenReportFailed *NoPasswordLinkGenReportFailedType `json:"no_password_link_gen_report_failed,omitempty"`
	// NoPasswordLinkViewCreateReport : (reports) Report created: Views of links
	// without passwords
	NoPasswordLinkViewCreateReport *NoPasswordLinkViewCreateReportType `json:"no_password_link_view_create_report,omitempty"`
	// NoPasswordLinkViewReportFailed : (reports) Couldn't create report: Views
	// of links without passwords
	NoPasswordLinkViewReportFailed *NoPasswordLinkViewReportFailedType `json:"no_password_link_view_report_failed,omitempty"`
	// OutdatedLinkViewCreateReport : (reports) Report created: Views of old
	// links
	OutdatedLinkViewCreateReport *OutdatedLinkViewCreateReportType `json:"outdated_link_view_create_report,omitempty"`
	// OutdatedLinkViewReportFailed : (reports) Couldn't create report: Views of
	// old links
	OutdatedLinkViewReportFailed *OutdatedLinkViewReportFailedType `json:"outdated_link_view_report_failed,omitempty"`
	// PaperAdminExportStart : (reports) Exported all team Paper docs
	PaperAdminExportStart *PaperAdminExportStartType `json:"paper_admin_export_start,omitempty"`
	// SmartSyncCreateAdminPrivilegeReport : (reports) Created Smart Sync
	// non-admin devices report
	SmartSyncCreateAdminPrivilegeReport *SmartSyncCreateAdminPrivilegeReportType `json:"smart_sync_create_admin_privilege_report,omitempty"`
	// TeamActivityCreateReport : (reports) Created team activity report
	TeamActivityCreateReport *TeamActivityCreateReportType `json:"team_activity_create_report,omitempty"`
	// TeamActivityCreateReportFail : (reports) Couldn't generate team activity
	// report
	TeamActivityCreateReportFail *TeamActivityCreateReportFailType `json:"team_activity_create_report_fail,omitempty"`
	// CollectionShare : (sharing) Shared album
	CollectionShare *CollectionShareType `json:"collection_share,omitempty"`
	// FileTransfersFileAdd : (sharing) Transfer files added
	FileTransfersFileAdd *FileTransfersFileAddType `json:"file_transfers_file_add,omitempty"`
	// FileTransfersTransferDelete : (sharing) Deleted transfer
	FileTransfersTransferDelete *FileTransfersTransferDeleteType `json:"file_transfers_transfer_delete,omitempty"`
	// FileTransfersTransferDownload : (sharing) Transfer downloaded
	FileTransfersTransferDownload *FileTransfersTransferDownloadType `json:"file_transfers_transfer_download,omitempty"`
	// FileTransfersTransferSend : (sharing) Sent transfer
	FileTransfersTransferSend *FileTransfersTransferSendType `json:"file_transfers_transfer_send,omitempty"`
	// FileTransfersTransferView : (sharing) Viewed transfer
	FileTransfersTransferView *FileTransfersTransferViewType `json:"file_transfers_transfer_view,omitempty"`
	// NoteAclInviteOnly : (sharing) Changed Paper doc to invite-only
	// (deprecated, no longer logged)
	NoteAclInviteOnly *NoteAclInviteOnlyType `json:"note_acl_invite_only,omitempty"`
	// NoteAclLink : (sharing) Changed Paper doc to link-accessible (deprecated,
	// no longer logged)
	NoteAclLink *NoteAclLinkType `json:"note_acl_link,omitempty"`
	// NoteAclTeamLink : (sharing) Changed Paper doc to link-accessible for team
	// (deprecated, no longer logged)
	NoteAclTeamLink *NoteAclTeamLinkType `json:"note_acl_team_link,omitempty"`
	// NoteShared : (sharing) Shared Paper doc (deprecated, no longer logged)
	NoteShared *NoteSharedType `json:"note_shared,omitempty"`
	// NoteShareReceive : (sharing) Shared received Paper doc (deprecated, no
	// longer logged)
	NoteShareReceive *NoteShareReceiveType `json:"note_share_receive,omitempty"`
	// OpenNoteShared : (sharing) Opened shared Paper doc (deprecated, no longer
	// logged)
	OpenNoteShared *OpenNoteSharedType `json:"open_note_shared,omitempty"`
	// SfAddGroup : (sharing) Added team to shared folder (deprecated, no longer
	// logged)
	SfAddGroup *SfAddGroupType `json:"sf_add_group,omitempty"`
	// SfAllowNonMembersToViewSharedLinks : (sharing) Allowed non-collaborators
	// to view links to files in shared folder (deprecated, no longer logged)
	SfAllowNonMembersToViewSharedLinks *SfAllowNonMembersToViewSharedLinksType `json:"sf_allow_non_members_to_view_shared_links,omitempty"`
	// SfExternalInviteWarn : (sharing) Set team members to see warning before
	// sharing folders outside team (deprecated, no longer logged)
	SfExternalInviteWarn *SfExternalInviteWarnType `json:"sf_external_invite_warn,omitempty"`
	// SfFbInvite : (sharing) Invited Facebook users to shared folder
	// (deprecated, no longer logged)
	SfFbInvite *SfFbInviteType `json:"sf_fb_invite,omitempty"`
	// SfFbInviteChangeRole : (sharing) Changed Facebook user's role in shared
	// folder (deprecated, no longer logged)
	SfFbInviteChangeRole *SfFbInviteChangeRoleType `json:"sf_fb_invite_change_role,omitempty"`
	// SfFbUninvite : (sharing) Uninvited Facebook user from shared folder
	// (deprecated, no longer logged)
	SfFbUninvite *SfFbUninviteType `json:"sf_fb_uninvite,omitempty"`
	// SfInviteGroup : (sharing) Invited group to shared folder (deprecated, no
	// longer logged)
	SfInviteGroup *SfInviteGroupType `json:"sf_invite_group,omitempty"`
	// SfTeamGrantAccess : (sharing) Granted access to shared folder
	// (deprecated, no longer logged)
	SfTeamGrantAccess *SfTeamGrantAccessType `json:"sf_team_grant_access,omitempty"`
	// SfTeamInvite : (sharing) Invited team members to shared folder
	// (deprecated, replaced by 'Invited user to Dropbox and added them to
	// shared file/folder')
	SfTeamInvite *SfTeamInviteType `json:"sf_team_invite,omitempty"`
	// SfTeamInviteChangeRole : (sharing) Changed team member's role in shared
	// folder (deprecated, no longer logged)
	SfTeamInviteChangeRole *SfTeamInviteChangeRoleType `json:"sf_team_invite_change_role,omitempty"`
	// SfTeamJoin : (sharing) Joined team member's shared folder (deprecated, no
	// longer logged)
	SfTeamJoin *SfTeamJoinType `json:"sf_team_join,omitempty"`
	// SfTeamJoinFromOobLink : (sharing) Joined team member's shared folder from
	// link (deprecated, no longer logged)
	SfTeamJoinFromOobLink *SfTeamJoinFromOobLinkType `json:"sf_team_join_from_oob_link,omitempty"`
	// SfTeamUninvite : (sharing) Unshared folder with team member (deprecated,
	// replaced by 'Removed invitee from shared file/folder before invite was
	// accepted')
	SfTeamUninvite *SfTeamUninviteType `json:"sf_team_uninvite,omitempty"`
	// SharedContentAddInvitees : (sharing) Invited user to Dropbox and added
	// them to shared file/folder
	SharedContentAddInvitees *SharedContentAddInviteesType `json:"shared_content_add_invitees,omitempty"`
	// SharedContentAddLinkExpiry : (sharing) Added expiration date to link for
	// shared file/folder (deprecated, no longer logged)
	SharedContentAddLinkExpiry *SharedContentAddLinkExpiryType `json:"shared_content_add_link_expiry,omitempty"`
	// SharedContentAddLinkPassword : (sharing) Added password to link for
	// shared file/folder (deprecated, no longer logged)
	SharedContentAddLinkPassword *SharedContentAddLinkPasswordType `json:"shared_content_add_link_password,omitempty"`
	// SharedContentAddMember : (sharing) Added users and/or groups to shared
	// file/folder
	SharedContentAddMember *SharedContentAddMemberType `json:"shared_content_add_member,omitempty"`
	// SharedContentChangeDownloadsPolicy : (sharing) Changed whether members
	// can download shared file/folder (deprecated, no longer logged)
	SharedContentChangeDownloadsPolicy *SharedContentChangeDownloadsPolicyType `json:"shared_content_change_downloads_policy,omitempty"`
	// SharedContentChangeInviteeRole : (sharing) Changed access type of invitee
	// to shared file/folder before invite was accepted
	SharedContentChangeInviteeRole *SharedContentChangeInviteeRoleType `json:"shared_content_change_invitee_role,omitempty"`
	// SharedContentChangeLinkAudience : (sharing) Changed link audience of
	// shared file/folder (deprecated, no longer logged)
	SharedContentChangeLinkAudience *SharedContentChangeLinkAudienceType `json:"shared_content_change_link_audience,omitempty"`
	// SharedContentChangeLinkExpiry : (sharing) Changed link expiration of
	// shared file/folder (deprecated, no longer logged)
	SharedContentChangeLinkExpiry *SharedContentChangeLinkExpiryType `json:"shared_content_change_link_expiry,omitempty"`
	// SharedContentChangeLinkPassword : (sharing) Changed link password of
	// shared file/folder (deprecated, no longer logged)
	SharedContentChangeLinkPassword *SharedContentChangeLinkPasswordType `json:"shared_content_change_link_password,omitempty"`
	// SharedContentChangeMemberRole : (sharing) Changed access type of shared
	// file/folder member
	SharedContentChangeMemberRole *SharedContentChangeMemberRoleType `json:"shared_content_change_member_role,omitempty"`
	// SharedContentChangeViewerInfoPolicy : (sharing) Changed whether members
	// can see who viewed shared file/folder
	SharedContentChangeViewerInfoPolicy *SharedContentChangeViewerInfoPolicyType `json:"shared_content_change_viewer_info_policy,omitempty"`
	// SharedContentClaimInvitation : (sharing) Acquired membership of shared
	// file/folder by accepting invite
	SharedContentClaimInvitation *SharedContentClaimInvitationType `json:"shared_content_claim_invitation,omitempty"`
	// SharedContentCopy : (sharing) Copied shared file/folder to own Dropbox
	SharedContentCopy *SharedContentCopyType `json:"shared_content_copy,omitempty"`
	// SharedContentDownload : (sharing) Downloaded shared file/folder
	SharedContentDownload *SharedContentDownloadType `json:"shared_content_download,omitempty"`
	// SharedContentRelinquishMembership : (sharing) Left shared file/folder
	SharedContentRelinquishMembership *SharedContentRelinquishMembershipType `json:"shared_content_relinquish_membership,omitempty"`
	// SharedContentRemoveInvitees : (sharing) Removed invitee from shared
	// file/folder before invite was accepted
	SharedContentRemoveInvitees *SharedContentRemoveInviteesType `json:"shared_content_remove_invitees,omitempty"`
	// SharedContentRemoveLinkExpiry : (sharing) Removed link expiration date of
	// shared file/folder (deprecated, no longer logged)
	SharedContentRemoveLinkExpiry *SharedContentRemoveLinkExpiryType `json:"shared_content_remove_link_expiry,omitempty"`
	// SharedContentRemoveLinkPassword : (sharing) Removed link password of
	// shared file/folder (deprecated, no longer logged)
	SharedContentRemoveLinkPassword *SharedContentRemoveLinkPasswordType `json:"shared_content_remove_link_password,omitempty"`
	// SharedContentRemoveMember : (sharing) Removed user/group from shared
	// file/folder
	SharedContentRemoveMember *SharedContentRemoveMemberType `json:"shared_content_remove_member,omitempty"`
	// SharedContentRequestAccess : (sharing) Requested access to shared
	// file/folder
	SharedContentRequestAccess *SharedContentRequestAccessType `json:"shared_content_request_access,omitempty"`
	// SharedContentRestoreInvitees : (sharing) Restored shared file/folder
	// invitees
	SharedContentRestoreInvitees *SharedContentRestoreInviteesType `json:"shared_content_restore_invitees,omitempty"`
	// SharedContentRestoreMember : (sharing) Restored users and/or groups to
	// membership of shared file/folder
	SharedContentRestoreMember *SharedContentRestoreMemberType `json:"shared_content_restore_member,omitempty"`
	// SharedContentUnshare : (sharing) Unshared file/folder by clearing
	// membership
	SharedContentUnshare *SharedContentUnshareType `json:"shared_content_unshare,omitempty"`
	// SharedContentView : (sharing) Previewed shared file/folder
	SharedContentView *SharedContentViewType `json:"shared_content_view,omitempty"`
	// SharedFolderChangeLinkPolicy : (sharing) Changed who can access shared
	// folder via link
	SharedFolderChangeLinkPolicy *SharedFolderChangeLinkPolicyType `json:"shared_folder_change_link_policy,omitempty"`
	// SharedFolderChangeMembersInheritancePolicy : (sharing) Changed whether
	// shared folder inherits members from parent folder
	SharedFolderChangeMembersInheritancePolicy *SharedFolderChangeMembersInheritancePolicyType `json:"shared_folder_change_members_inheritance_policy,omitempty"`
	// SharedFolderChangeMembersManagementPolicy : (sharing) Changed who can
	// add/remove members of shared folder
	SharedFolderChangeMembersManagementPolicy *SharedFolderChangeMembersManagementPolicyType `json:"shared_folder_change_members_management_policy,omitempty"`
	// SharedFolderChangeMembersPolicy : (sharing) Changed who can become member
	// of shared folder
	SharedFolderChangeMembersPolicy *SharedFolderChangeMembersPolicyType `json:"shared_folder_change_members_policy,omitempty"`
	// SharedFolderCreate : (sharing) Created shared folder
	SharedFolderCreate *SharedFolderCreateType `json:"shared_folder_create,omitempty"`
	// SharedFolderDeclineInvitation : (sharing) Declined team member's invite
	// to shared folder
	SharedFolderDeclineInvitation *SharedFolderDeclineInvitationType `json:"shared_folder_decline_invitation,omitempty"`
	// SharedFolderMount : (sharing) Added shared folder to own Dropbox
	SharedFolderMount *SharedFolderMountType `json:"shared_folder_mount,omitempty"`
	// SharedFolderNest : (sharing) Changed parent of shared folder
	SharedFolderNest *SharedFolderNestType `json:"shared_folder_nest,omitempty"`
	// SharedFolderTransferOwnership : (sharing) Transferred ownership of shared
	// folder to another member
	SharedFolderTransferOwnership *SharedFolderTransferOwnershipType `json:"shared_folder_transfer_ownership,omitempty"`
	// SharedFolderUnmount : (sharing) Deleted shared folder from Dropbox
	SharedFolderUnmount *SharedFolderUnmountType `json:"shared_folder_unmount,omitempty"`
	// SharedLinkAddExpiry : (sharing) Added shared link expiration date
	SharedLinkAddExpiry *SharedLinkAddExpiryType `json:"shared_link_add_expiry,omitempty"`
	// SharedLinkChangeExpiry : (sharing) Changed shared link expiration date
	SharedLinkChangeExpiry *SharedLinkChangeExpiryType `json:"shared_link_change_expiry,omitempty"`
	// SharedLinkChangeVisibility : (sharing) Changed visibility of shared link
	SharedLinkChangeVisibility *SharedLinkChangeVisibilityType `json:"shared_link_change_visibility,omitempty"`
	// SharedLinkCopy : (sharing) Added file/folder to Dropbox from shared link
	SharedLinkCopy *SharedLinkCopyType `json:"shared_link_copy,omitempty"`
	// SharedLinkCreate : (sharing) Created shared link
	SharedLinkCreate *SharedLinkCreateType `json:"shared_link_create,omitempty"`
	// SharedLinkDisable : (sharing) Removed shared link
	SharedLinkDisable *SharedLinkDisableType `json:"shared_link_disable,omitempty"`
	// SharedLinkDownload : (sharing) Downloaded file/folder from shared link
	SharedLinkDownload *SharedLinkDownloadType `json:"shared_link_download,omitempty"`
	// SharedLinkRemoveExpiry : (sharing) Removed shared link expiration date
	SharedLinkRemoveExpiry *SharedLinkRemoveExpiryType `json:"shared_link_remove_expiry,omitempty"`
	// SharedLinkSettingsAddExpiration : (sharing) Added an expiration date to
	// the shared link
	SharedLinkSettingsAddExpiration *SharedLinkSettingsAddExpirationType `json:"shared_link_settings_add_expiration,omitempty"`
	// SharedLinkSettingsAddPassword : (sharing) Added a password to the shared
	// link
	SharedLinkSettingsAddPassword *SharedLinkSettingsAddPasswordType `json:"shared_link_settings_add_password,omitempty"`
	// SharedLinkSettingsAllowDownloadDisabled : (sharing) Disabled downloads
	SharedLinkSettingsAllowDownloadDisabled *SharedLinkSettingsAllowDownloadDisabledType `json:"shared_link_settings_allow_download_disabled,omitempty"`
	// SharedLinkSettingsAllowDownloadEnabled : (sharing) Enabled downloads
	SharedLinkSettingsAllowDownloadEnabled *SharedLinkSettingsAllowDownloadEnabledType `json:"shared_link_settings_allow_download_enabled,omitempty"`
	// SharedLinkSettingsChangeAudience : (sharing) Changed the audience of the
	// shared link
	SharedLinkSettingsChangeAudience *SharedLinkSettingsChangeAudienceType `json:"shared_link_settings_change_audience,omitempty"`
	// SharedLinkSettingsChangeExpiration : (sharing) Changed the expiration
	// date of the shared link
	SharedLinkSettingsChangeExpiration *SharedLinkSettingsChangeExpirationType `json:"shared_link_settings_change_expiration,omitempty"`
	// SharedLinkSettingsChangePassword : (sharing) Changed the password of the
	// shared link
	SharedLinkSettingsChangePassword *SharedLinkSettingsChangePasswordType `json:"shared_link_settings_change_password,omitempty"`
	// SharedLinkSettingsRemoveExpiration : (sharing) Removed the expiration
	// date from the shared link
	SharedLinkSettingsRemoveExpiration *SharedLinkSettingsRemoveExpirationType `json:"shared_link_settings_remove_expiration,omitempty"`
	// SharedLinkSettingsRemovePassword : (sharing) Removed the password from
	// the shared link
	SharedLinkSettingsRemovePassword *SharedLinkSettingsRemovePasswordType `json:"shared_link_settings_remove_password,omitempty"`
	// SharedLinkShare : (sharing) Added members as audience of shared link
	SharedLinkShare *SharedLinkShareType `json:"shared_link_share,omitempty"`
	// SharedLinkView : (sharing) Opened shared link
	SharedLinkView *SharedLinkViewType `json:"shared_link_view,omitempty"`
	// SharedNoteOpened : (sharing) Opened shared Paper doc (deprecated, no
	// longer logged)
	SharedNoteOpened *SharedNoteOpenedType `json:"shared_note_opened,omitempty"`
	// ShmodelDisableDownloads : (sharing) Disabled downloads for link
	// (deprecated, no longer logged)
	ShmodelDisableDownloads *ShmodelDisableDownloadsType `json:"shmodel_disable_downloads,omitempty"`
	// ShmodelEnableDownloads : (sharing) Enabled downloads for link
	// (deprecated, no longer logged)
	ShmodelEnableDownloads *ShmodelEnableDownloadsType `json:"shmodel_enable_downloads,omitempty"`
	// ShmodelGroupShare : (sharing) Shared link with group (deprecated, no
	// longer logged)
	ShmodelGroupShare *ShmodelGroupShareType `json:"shmodel_group_share,omitempty"`
	// ShowcaseAccessGranted : (showcase) Granted access to showcase
	ShowcaseAccessGranted *ShowcaseAccessGrantedType `json:"showcase_access_granted,omitempty"`
	// ShowcaseAddMember : (showcase) Added member to showcase
	ShowcaseAddMember *ShowcaseAddMemberType `json:"showcase_add_member,omitempty"`
	// ShowcaseArchived : (showcase) Archived showcase
	ShowcaseArchived *ShowcaseArchivedType `json:"showcase_archived,omitempty"`
	// ShowcaseCreated : (showcase) Created showcase
	ShowcaseCreated *ShowcaseCreatedType `json:"showcase_created,omitempty"`
	// ShowcaseDeleteComment : (showcase) Deleted showcase comment
	ShowcaseDeleteComment *ShowcaseDeleteCommentType `json:"showcase_delete_comment,omitempty"`
	// ShowcaseEdited : (showcase) Edited showcase
	ShowcaseEdited *ShowcaseEditedType `json:"showcase_edited,omitempty"`
	// ShowcaseEditComment : (showcase) Edited showcase comment
	ShowcaseEditComment *ShowcaseEditCommentType `json:"showcase_edit_comment,omitempty"`
	// ShowcaseFileAdded : (showcase) Added file to showcase
	ShowcaseFileAdded *ShowcaseFileAddedType `json:"showcase_file_added,omitempty"`
	// ShowcaseFileDownload : (showcase) Downloaded file from showcase
	ShowcaseFileDownload *ShowcaseFileDownloadType `json:"showcase_file_download,omitempty"`
	// ShowcaseFileRemoved : (showcase) Removed file from showcase
	ShowcaseFileRemoved *ShowcaseFileRemovedType `json:"showcase_file_removed,omitempty"`
	// ShowcaseFileView : (showcase) Viewed file in showcase
	ShowcaseFileView *ShowcaseFileViewType `json:"showcase_file_view,omitempty"`
	// ShowcasePermanentlyDeleted : (showcase) Permanently deleted showcase
	ShowcasePermanentlyDeleted *ShowcasePermanentlyDeletedType `json:"showcase_permanently_deleted,omitempty"`
	// ShowcasePostComment : (showcase) Added showcase comment
	ShowcasePostComment *ShowcasePostCommentType `json:"showcase_post_comment,omitempty"`
	// ShowcaseRemoveMember : (showcase) Removed member from showcase
	ShowcaseRemoveMember *ShowcaseRemoveMemberType `json:"showcase_remove_member,omitempty"`
	// ShowcaseRenamed : (showcase) Renamed showcase
	ShowcaseRenamed *ShowcaseRenamedType `json:"showcase_renamed,omitempty"`
	// ShowcaseRequestAccess : (showcase) Requested access to showcase
	ShowcaseRequestAccess *ShowcaseRequestAccessType `json:"showcase_request_access,omitempty"`
	// ShowcaseResolveComment : (showcase) Resolved showcase comment
	ShowcaseResolveComment *ShowcaseResolveCommentType `json:"showcase_resolve_comment,omitempty"`
	// ShowcaseRestored : (showcase) Unarchived showcase
	ShowcaseRestored *ShowcaseRestoredType `json:"showcase_restored,omitempty"`
	// ShowcaseTrashed : (showcase) Deleted showcase
	ShowcaseTrashed *ShowcaseTrashedType `json:"showcase_trashed,omitempty"`
	// ShowcaseTrashedDeprecated : (showcase) Deleted showcase (old version)
	// (deprecated, replaced by 'Deleted showcase')
	ShowcaseTrashedDeprecated *ShowcaseTrashedDeprecatedType `json:"showcase_trashed_deprecated,omitempty"`
	// ShowcaseUnresolveComment : (showcase) Unresolved showcase comment
	ShowcaseUnresolveComment *ShowcaseUnresolveCommentType `json:"showcase_unresolve_comment,omitempty"`
	// ShowcaseUntrashed : (showcase) Restored showcase
	ShowcaseUntrashed *ShowcaseUntrashedType `json:"showcase_untrashed,omitempty"`
	// ShowcaseUntrashedDeprecated : (showcase) Restored showcase (old version)
	// (deprecated, replaced by 'Restored showcase')
	ShowcaseUntrashedDeprecated *ShowcaseUntrashedDeprecatedType `json:"showcase_untrashed_deprecated,omitempty"`
	// ShowcaseView : (showcase) Viewed showcase
	ShowcaseView *ShowcaseViewType `json:"showcase_view,omitempty"`
	// SsoAddCert : (sso) Added X.509 certificate for SSO
	SsoAddCert *SsoAddCertType `json:"sso_add_cert,omitempty"`
	// SsoAddLoginUrl : (sso) Added sign-in URL for SSO
	SsoAddLoginUrl *SsoAddLoginUrlType `json:"sso_add_login_url,omitempty"`
	// SsoAddLogoutUrl : (sso) Added sign-out URL for SSO
	SsoAddLogoutUrl *SsoAddLogoutUrlType `json:"sso_add_logout_url,omitempty"`
	// SsoChangeCert : (sso) Changed X.509 certificate for SSO
	SsoChangeCert *SsoChangeCertType `json:"sso_change_cert,omitempty"`
	// SsoChangeLoginUrl : (sso) Changed sign-in URL for SSO
	SsoChangeLoginUrl *SsoChangeLoginUrlType `json:"sso_change_login_url,omitempty"`
	// SsoChangeLogoutUrl : (sso) Changed sign-out URL for SSO
	SsoChangeLogoutUrl *SsoChangeLogoutUrlType `json:"sso_change_logout_url,omitempty"`
	// SsoChangeSamlIdentityMode : (sso) Changed SAML identity mode for SSO
	SsoChangeSamlIdentityMode *SsoChangeSamlIdentityModeType `json:"sso_change_saml_identity_mode,omitempty"`
	// SsoRemoveCert : (sso) Removed X.509 certificate for SSO
	SsoRemoveCert *SsoRemoveCertType `json:"sso_remove_cert,omitempty"`
	// SsoRemoveLoginUrl : (sso) Removed sign-in URL for SSO
	SsoRemoveLoginUrl *SsoRemoveLoginUrlType `json:"sso_remove_login_url,omitempty"`
	// SsoRemoveLogoutUrl : (sso) Removed sign-out URL for SSO
	SsoRemoveLogoutUrl *SsoRemoveLogoutUrlType `json:"sso_remove_logout_url,omitempty"`
	// TeamFolderChangeStatus : (team_folders) Changed archival status of team
	// folder
	TeamFolderChangeStatus *TeamFolderChangeStatusType `json:"team_folder_change_status,omitempty"`
	// TeamFolderCreate : (team_folders) Created team folder in active status
	TeamFolderCreate *TeamFolderCreateType `json:"team_folder_create,omitempty"`
	// TeamFolderDowngrade : (team_folders) Downgraded team folder to regular
	// shared folder
	TeamFolderDowngrade *TeamFolderDowngradeType `json:"team_folder_downgrade,omitempty"`
	// TeamFolderPermanentlyDelete : (team_folders) Permanently deleted archived
	// team folder
	TeamFolderPermanentlyDelete *TeamFolderPermanentlyDeleteType `json:"team_folder_permanently_delete,omitempty"`
	// TeamFolderRename : (team_folders) Renamed active/archived team folder
	TeamFolderRename *TeamFolderRenameType `json:"team_folder_rename,omitempty"`
	// TeamSelectiveSyncSettingsChanged : (team_folders) Changed sync default
	TeamSelectiveSyncSettingsChanged *TeamSelectiveSyncSettingsChangedType `json:"team_selective_sync_settings_changed,omitempty"`
	// AccountCaptureChangePolicy : (team_policies) Changed account capture
	// setting on team domain
	AccountCaptureChangePolicy *AccountCaptureChangePolicyType `json:"account_capture_change_policy,omitempty"`
	// AdminEmailRemindersChanged : (team_policies) Changed admin reminder
	// settings for requests to join the team
	AdminEmailRemindersChanged *AdminEmailRemindersChangedType `json:"admin_email_reminders_changed,omitempty"`
	// AllowDownloadDisabled : (team_policies) Disabled downloads (deprecated,
	// no longer logged)
	AllowDownloadDisabled *AllowDownloadDisabledType `json:"allow_download_disabled,omitempty"`
	// AllowDownloadEnabled : (team_policies) Enabled downloads (deprecated, no
	// longer logged)
	AllowDownloadEnabled *AllowDownloadEnabledType `json:"allow_download_enabled,omitempty"`
	// AppPermissionsChanged : (team_policies) Changed app permissions
	AppPermissionsChanged *AppPermissionsChangedType `json:"app_permissions_changed,omitempty"`
	// CameraUploadsPolicyChanged : (team_policies) Changed camera uploads
	// setting for team
	CameraUploadsPolicyChanged *CameraUploadsPolicyChangedType `json:"camera_uploads_policy_changed,omitempty"`
	// CaptureTranscriptPolicyChanged : (team_policies) Changed Capture
	// transcription policy for team
	CaptureTranscriptPolicyChanged *CaptureTranscriptPolicyChangedType `json:"capture_transcript_policy_changed,omitempty"`
	// ClassificationChangePolicy : (team_policies) Changed classification
	// policy for team
	ClassificationChangePolicy *ClassificationChangePolicyType `json:"classification_change_policy,omitempty"`
	// ComputerBackupPolicyChanged : (team_policies) Changed computer backup
	// policy for team
	ComputerBackupPolicyChanged *ComputerBackupPolicyChangedType `json:"computer_backup_policy_changed,omitempty"`
	// ContentAdministrationPolicyChanged : (team_policies) Changed content
	// management setting
	ContentAdministrationPolicyChanged *ContentAdministrationPolicyChangedType `json:"content_administration_policy_changed,omitempty"`
	// DataPlacementRestrictionChangePolicy : (team_policies) Set restrictions
	// on data center locations where team data resides
	DataPlacementRestrictionChangePolicy *DataPlacementRestrictionChangePolicyType `json:"data_placement_restriction_change_policy,omitempty"`
	// DataPlacementRestrictionSatisfyPolicy : (team_policies) Completed
	// restrictions on data center locations where team data resides
	DataPlacementRestrictionSatisfyPolicy *DataPlacementRestrictionSatisfyPolicyType `json:"data_placement_restriction_satisfy_policy,omitempty"`
	// DeviceApprovalsAddException : (team_policies) Added members to device
	// approvals exception list
	DeviceApprovalsAddException *DeviceApprovalsAddExceptionType `json:"device_approvals_add_exception,omitempty"`
	// DeviceApprovalsChangeDesktopPolicy : (team_policies) Set/removed limit on
	// number of computers member can link to team Dropbox account
	DeviceApprovalsChangeDesktopPolicy *DeviceApprovalsChangeDesktopPolicyType `json:"device_approvals_change_desktop_policy,omitempty"`
	// DeviceApprovalsChangeMobilePolicy : (team_policies) Set/removed limit on
	// number of mobile devices member can link to team Dropbox account
	DeviceApprovalsChangeMobilePolicy *DeviceApprovalsChangeMobilePolicyType `json:"device_approvals_change_mobile_policy,omitempty"`
	// DeviceApprovalsChangeOverageAction : (team_policies) Changed device
	// approvals setting when member is over limit
	DeviceApprovalsChangeOverageAction *DeviceApprovalsChangeOverageActionType `json:"device_approvals_change_overage_action,omitempty"`
	// DeviceApprovalsChangeUnlinkAction : (team_policies) Changed device
	// approvals setting when member unlinks approved device
	DeviceApprovalsChangeUnlinkAction *DeviceApprovalsChangeUnlinkActionType `json:"device_approvals_change_unlink_action,omitempty"`
	// DeviceApprovalsRemoveException : (team_policies) Removed members from
	// device approvals exception list
	DeviceApprovalsRemoveException *DeviceApprovalsRemoveExceptionType `json:"device_approvals_remove_exception,omitempty"`
	// DirectoryRestrictionsAddMembers : (team_policies) Added members to
	// directory restrictions list
	DirectoryRestrictionsAddMembers *DirectoryRestrictionsAddMembersType `json:"directory_restrictions_add_members,omitempty"`
	// DirectoryRestrictionsRemoveMembers : (team_policies) Removed members from
	// directory restrictions list
	DirectoryRestrictionsRemoveMembers *DirectoryRestrictionsRemoveMembersType `json:"directory_restrictions_remove_members,omitempty"`
	// DropboxPasswordsPolicyChanged : (team_policies) Changed Dropbox Passwords
	// policy for team
	DropboxPasswordsPolicyChanged *DropboxPasswordsPolicyChangedType `json:"dropbox_passwords_policy_changed,omitempty"`
	// EmailIngestPolicyChanged : (team_policies) Changed email to Dropbox
	// policy for team
	EmailIngestPolicyChanged *EmailIngestPolicyChangedType `json:"email_ingest_policy_changed,omitempty"`
	// EmmAddException : (team_policies) Added members to EMM exception list
	EmmAddException *EmmAddExceptionType `json:"emm_add_exception,omitempty"`
	// EmmChangePolicy : (team_policies) Enabled/disabled enterprise mobility
	// management for members
	EmmChangePolicy *EmmChangePolicyType `json:"emm_change_policy,omitempty"`
	// EmmRemoveException : (team_policies) Removed members from EMM exception
	// list
	EmmRemoveException *EmmRemoveExceptionType `json:"emm_remove_exception,omitempty"`
	// ExtendedVersionHistoryChangePolicy : (team_policies) Accepted/opted out
	// of extended version history
	ExtendedVersionHistoryChangePolicy *ExtendedVersionHistoryChangePolicyType `json:"extended_version_history_change_policy,omitempty"`
	// ExternalDriveBackupPolicyChanged : (team_policies) Changed external drive
	// backup policy for team
	ExternalDriveBackupPolicyChanged *ExternalDriveBackupPolicyChangedType `json:"external_drive_backup_policy_changed,omitempty"`
	// FileCommentsChangePolicy : (team_policies) Enabled/disabled commenting on
	// team files
	FileCommentsChangePolicy *FileCommentsChangePolicyType `json:"file_comments_change_policy,omitempty"`
	// FileLockingPolicyChanged : (team_policies) Changed file locking policy
	// for team
	FileLockingPolicyChanged *FileLockingPolicyChangedType `json:"file_locking_policy_changed,omitempty"`
	// FileProviderMigrationPolicyChanged : (team_policies) Changed File
	// Provider Migration policy for team
	FileProviderMigrationPolicyChanged *FileProviderMigrationPolicyChangedType `json:"file_provider_migration_policy_changed,omitempty"`
	// FileRequestsChangePolicy : (team_policies) Enabled/disabled file requests
	FileRequestsChangePolicy *FileRequestsChangePolicyType `json:"file_requests_change_policy,omitempty"`
	// FileRequestsEmailsEnabled : (team_policies) Enabled file request emails
	// for everyone (deprecated, no longer logged)
	FileRequestsEmailsEnabled *FileRequestsEmailsEnabledType `json:"file_requests_emails_enabled,omitempty"`
	// FileRequestsEmailsRestrictedToTeamOnly : (team_policies) Enabled file
	// request emails for team (deprecated, no longer logged)
	FileRequestsEmailsRestrictedToTeamOnly *FileRequestsEmailsRestrictedToTeamOnlyType `json:"file_requests_emails_restricted_to_team_only,omitempty"`
	// FileTransfersPolicyChanged : (team_policies) Changed file transfers
	// policy for team
	FileTransfersPolicyChanged *FileTransfersPolicyChangedType `json:"file_transfers_policy_changed,omitempty"`
	// GoogleSsoChangePolicy : (team_policies) Enabled/disabled Google single
	// sign-on for team
	GoogleSsoChangePolicy *GoogleSsoChangePolicyType `json:"google_sso_change_policy,omitempty"`
	// GroupUserManagementChangePolicy : (team_policies) Changed who can create
	// groups
	GroupUserManagementChangePolicy *GroupUserManagementChangePolicyType `json:"group_user_management_change_policy,omitempty"`
	// IntegrationPolicyChanged : (team_policies) Changed integration policy for
	// team
	IntegrationPolicyChanged *IntegrationPolicyChangedType `json:"integration_policy_changed,omitempty"`
	// InviteAcceptanceEmailPolicyChanged : (team_policies) Changed invite
	// accept email policy for team
	InviteAcceptanceEmailPolicyChanged *InviteAcceptanceEmailPolicyChangedType `json:"invite_acceptance_email_policy_changed,omitempty"`
	// MemberRequestsChangePolicy : (team_policies) Changed whether users can
	// find team when not invited
	MemberRequestsChangePolicy *MemberRequestsChangePolicyType `json:"member_requests_change_policy,omitempty"`
	// MemberSendInvitePolicyChanged : (team_policies) Changed member send
	// invite policy for team
	MemberSendInvitePolicyChanged *MemberSendInvitePolicyChangedType `json:"member_send_invite_policy_changed,omitempty"`
	// MemberSpaceLimitsAddException : (team_policies) Added members to member
	// space limit exception list
	MemberSpaceLimitsAddException *MemberSpaceLimitsAddExceptionType `json:"member_space_limits_add_exception,omitempty"`
	// MemberSpaceLimitsChangeCapsTypePolicy : (team_policies) Changed member
	// space limit type for team
	MemberSpaceLimitsChangeCapsTypePolicy *MemberSpaceLimitsChangeCapsTypePolicyType `json:"member_space_limits_change_caps_type_policy,omitempty"`
	// MemberSpaceLimitsChangePolicy : (team_policies) Changed team default
	// member space limit
	MemberSpaceLimitsChangePolicy *MemberSpaceLimitsChangePolicyType `json:"member_space_limits_change_policy,omitempty"`
	// MemberSpaceLimitsRemoveException : (team_policies) Removed members from
	// member space limit exception list
	MemberSpaceLimitsRemoveException *MemberSpaceLimitsRemoveExceptionType `json:"member_space_limits_remove_exception,omitempty"`
	// MemberSuggestionsChangePolicy : (team_policies) Enabled/disabled option
	// for team members to suggest people to add to team
	MemberSuggestionsChangePolicy *MemberSuggestionsChangePolicyType `json:"member_suggestions_change_policy,omitempty"`
	// MicrosoftOfficeAddinChangePolicy : (team_policies) Enabled/disabled
	// Microsoft Office add-in
	MicrosoftOfficeAddinChangePolicy *MicrosoftOfficeAddinChangePolicyType `json:"microsoft_office_addin_change_policy,omitempty"`
	// NetworkControlChangePolicy : (team_policies) Enabled/disabled network
	// control
	NetworkControlChangePolicy *NetworkControlChangePolicyType `json:"network_control_change_policy,omitempty"`
	// PaperChangeDeploymentPolicy : (team_policies) Changed whether Dropbox
	// Paper, when enabled, is deployed to all members or to specific members
	PaperChangeDeploymentPolicy *PaperChangeDeploymentPolicyType `json:"paper_change_deployment_policy,omitempty"`
	// PaperChangeMemberLinkPolicy : (team_policies) Changed whether non-members
	// can view Paper docs with link (deprecated, no longer logged)
	PaperChangeMemberLinkPolicy *PaperChangeMemberLinkPolicyType `json:"paper_change_member_link_policy,omitempty"`
	// PaperChangeMemberPolicy : (team_policies) Changed whether members can
	// share Paper docs outside team, and if docs are accessible only by team
	// members or anyone by default
	PaperChangeMemberPolicy *PaperChangeMemberPolicyType `json:"paper_change_member_policy,omitempty"`
	// PaperChangePolicy : (team_policies) Enabled/disabled Dropbox Paper for
	// team
	PaperChangePolicy *PaperChangePolicyType `json:"paper_change_policy,omitempty"`
	// PaperDefaultFolderPolicyChanged : (team_policies) Changed Paper Default
	// Folder Policy setting for team
	PaperDefaultFolderPolicyChanged *PaperDefaultFolderPolicyChangedType `json:"paper_default_folder_policy_changed,omitempty"`
	// PaperDesktopPolicyChanged : (team_policies) Enabled/disabled Paper
	// Desktop for team
	PaperDesktopPolicyChanged *PaperDesktopPolicyChangedType `json:"paper_desktop_policy_changed,omitempty"`
	// PaperEnabledUsersGroupAddition : (team_policies) Added users to
	// Paper-enabled users list
	PaperEnabledUsersGroupAddition *PaperEnabledUsersGroupAdditionType `json:"paper_enabled_users_group_addition,omitempty"`
	// PaperEnabledUsersGroupRemoval : (team_policies) Removed users from
	// Paper-enabled users list
	PaperEnabledUsersGroupRemoval *PaperEnabledUsersGroupRemovalType `json:"paper_enabled_users_group_removal,omitempty"`
	// PasswordStrengthRequirementsChangePolicy : (team_policies) Changed team
	// password strength requirements
	PasswordStrengthRequirementsChangePolicy *PasswordStrengthRequirementsChangePolicyType `json:"password_strength_requirements_change_policy,omitempty"`
	// PermanentDeleteChangePolicy : (team_policies) Enabled/disabled ability of
	// team members to permanently delete content
	PermanentDeleteChangePolicy *PermanentDeleteChangePolicyType `json:"permanent_delete_change_policy,omitempty"`
	// ResellerSupportChangePolicy : (team_policies) Enabled/disabled reseller
	// support
	ResellerSupportChangePolicy *ResellerSupportChangePolicyType `json:"reseller_support_change_policy,omitempty"`
	// RewindPolicyChanged : (team_policies) Changed Rewind policy for team
	RewindPolicyChanged *RewindPolicyChangedType `json:"rewind_policy_changed,omitempty"`
	// SendForSignaturePolicyChanged : (team_policies) Changed send for
	// signature policy for team
	SendForSignaturePolicyChanged *SendForSignaturePolicyChangedType `json:"send_for_signature_policy_changed,omitempty"`
	// SharingChangeFolderJoinPolicy : (team_policies) Changed whether team
	// members can join shared folders owned outside team
	SharingChangeFolderJoinPolicy *SharingChangeFolderJoinPolicyType `json:"sharing_change_folder_join_policy,omitempty"`
	// SharingChangeLinkAllowChangeExpirationPolicy : (team_policies) Changed
	// the allow remove or change expiration policy for the links shared outside
	// of the team
	SharingChangeLinkAllowChangeExpirationPolicy *SharingChangeLinkAllowChangeExpirationPolicyType `json:"sharing_change_link_allow_change_expiration_policy,omitempty"`
	// SharingChangeLinkDefaultExpirationPolicy : (team_policies) Changed the
	// default expiration for the links shared outside of the team
	SharingChangeLinkDefaultExpirationPolicy *SharingChangeLinkDefaultExpirationPolicyType `json:"sharing_change_link_default_expiration_policy,omitempty"`
	// SharingChangeLinkEnforcePasswordPolicy : (team_policies) Changed the
	// password requirement for the links shared outside of the team
	SharingChangeLinkEnforcePasswordPolicy *SharingChangeLinkEnforcePasswordPolicyType `json:"sharing_change_link_enforce_password_policy,omitempty"`
	// SharingChangeLinkPolicy : (team_policies) Changed whether members can
	// share links outside team, and if links are accessible only by team
	// members or anyone by default
	SharingChangeLinkPolicy *SharingChangeLinkPolicyType `json:"sharing_change_link_policy,omitempty"`
	// SharingChangeMemberPolicy : (team_policies) Changed whether members can
	// share files/folders outside team
	SharingChangeMemberPolicy *SharingChangeMemberPolicyType `json:"sharing_change_member_policy,omitempty"`
	// ShowcaseChangeDownloadPolicy : (team_policies) Enabled/disabled
	// downloading files from Dropbox Showcase for team
	ShowcaseChangeDownloadPolicy *ShowcaseChangeDownloadPolicyType `json:"showcase_change_download_policy,omitempty"`
	// ShowcaseChangeEnabledPolicy : (team_policies) Enabled/disabled Dropbox
	// Showcase for team
	ShowcaseChangeEnabledPolicy *ShowcaseChangeEnabledPolicyType `json:"showcase_change_enabled_policy,omitempty"`
	// ShowcaseChangeExternalSharingPolicy : (team_policies) Enabled/disabled
	// sharing Dropbox Showcase externally for team
	ShowcaseChangeExternalSharingPolicy *ShowcaseChangeExternalSharingPolicyType `json:"showcase_change_external_sharing_policy,omitempty"`
	// SmarterSmartSyncPolicyChanged : (team_policies) Changed automatic Smart
	// Sync setting for team
	SmarterSmartSyncPolicyChanged *SmarterSmartSyncPolicyChangedType `json:"smarter_smart_sync_policy_changed,omitempty"`
	// SmartSyncChangePolicy : (team_policies) Changed default Smart Sync
	// setting for team members
	SmartSyncChangePolicy *SmartSyncChangePolicyType `json:"smart_sync_change_policy,omitempty"`
	// SmartSyncNotOptOut : (team_policies) Opted team into Smart Sync
	SmartSyncNotOptOut *SmartSyncNotOptOutType `json:"smart_sync_not_opt_out,omitempty"`
	// SmartSyncOptOut : (team_policies) Opted team out of Smart Sync
	SmartSyncOptOut *SmartSyncOptOutType `json:"smart_sync_opt_out,omitempty"`
	// SsoChangePolicy : (team_policies) Changed single sign-on setting for team
	SsoChangePolicy *SsoChangePolicyType `json:"sso_change_policy,omitempty"`
	// TeamBrandingPolicyChanged : (team_policies) Changed team branding policy
	// for team
	TeamBrandingPolicyChanged *TeamBrandingPolicyChangedType `json:"team_branding_policy_changed,omitempty"`
	// TeamExtensionsPolicyChanged : (team_policies) Changed App Integrations
	// setting for team
	TeamExtensionsPolicyChanged *TeamExtensionsPolicyChangedType `json:"team_extensions_policy_changed,omitempty"`
	// TeamSelectiveSyncPolicyChanged : (team_policies) Enabled/disabled Team
	// Selective Sync for team
	TeamSelectiveSyncPolicyChanged *TeamSelectiveSyncPolicyChangedType `json:"team_selective_sync_policy_changed,omitempty"`
	// TeamSharingWhitelistSubjectsChanged : (team_policies) Edited the approved
	// list for sharing externally
	TeamSharingWhitelistSubjectsChanged *TeamSharingWhitelistSubjectsChangedType `json:"team_sharing_whitelist_subjects_changed,omitempty"`
	// TfaAddException : (team_policies) Added members to two factor
	// authentication exception list
	TfaAddException *TfaAddExceptionType `json:"tfa_add_exception,omitempty"`
	// TfaChangePolicy : (team_policies) Changed two-step verification setting
	// for team
	TfaChangePolicy *TfaChangePolicyType `json:"tfa_change_policy,omitempty"`
	// TfaRemoveException : (team_policies) Removed members from two factor
	// authentication exception list
	TfaRemoveException *TfaRemoveExceptionType `json:"tfa_remove_exception,omitempty"`
	// TwoAccountChangePolicy : (team_policies) Enabled/disabled option for
	// members to link personal Dropbox account and team account to same
	// computer
	TwoAccountChangePolicy *TwoAccountChangePolicyType `json:"two_account_change_policy,omitempty"`
	// ViewerInfoPolicyChanged : (team_policies) Changed team policy for viewer
	// info
	ViewerInfoPolicyChanged *ViewerInfoPolicyChangedType `json:"viewer_info_policy_changed,omitempty"`
	// WatermarkingPolicyChanged : (team_policies) Changed watermarking policy
	// for team
	WatermarkingPolicyChanged *WatermarkingPolicyChangedType `json:"watermarking_policy_changed,omitempty"`
	// WebSessionsChangeActiveSessionLimit : (team_policies) Changed limit on
	// active sessions per member
	WebSessionsChangeActiveSessionLimit *WebSessionsChangeActiveSessionLimitType `json:"web_sessions_change_active_session_limit,omitempty"`
	// WebSessionsChangeFixedLengthPolicy : (team_policies) Changed how long
	// members can stay signed in to Dropbox.com
	WebSessionsChangeFixedLengthPolicy *WebSessionsChangeFixedLengthPolicyType `json:"web_sessions_change_fixed_length_policy,omitempty"`
	// WebSessionsChangeIdleLengthPolicy : (team_policies) Changed how long team
	// members can be idle while signed in to Dropbox.com
	WebSessionsChangeIdleLengthPolicy *WebSessionsChangeIdleLengthPolicyType `json:"web_sessions_change_idle_length_policy,omitempty"`
	// DataResidencyMigrationRequestSuccessful : (team_profile) Requested data
	// residency migration for team data
	DataResidencyMigrationRequestSuccessful *DataResidencyMigrationRequestSuccessfulType `json:"data_residency_migration_request_successful,omitempty"`
	// DataResidencyMigrationRequestUnsuccessful : (team_profile) Request for
	// data residency migration for team data has failed
	DataResidencyMigrationRequestUnsuccessful *DataResidencyMigrationRequestUnsuccessfulType `json:"data_residency_migration_request_unsuccessful,omitempty"`
	// TeamMergeFrom : (team_profile) Merged another team into this team
	TeamMergeFrom *TeamMergeFromType `json:"team_merge_from,omitempty"`
	// TeamMergeTo : (team_profile) Merged this team into another team
	TeamMergeTo *TeamMergeToType `json:"team_merge_to,omitempty"`
	// TeamProfileAddBackground : (team_profile) Added team background to
	// display on shared link headers
	TeamProfileAddBackground *TeamProfileAddBackgroundType `json:"team_profile_add_background,omitempty"`
	// TeamProfileAddLogo : (team_profile) Added team logo to display on shared
	// link headers
	TeamProfileAddLogo *TeamProfileAddLogoType `json:"team_profile_add_logo,omitempty"`
	// TeamProfileChangeBackground : (team_profile) Changed team background
	// displayed on shared link headers
	TeamProfileChangeBackground *TeamProfileChangeBackgroundType `json:"team_profile_change_background,omitempty"`
	// TeamProfileChangeDefaultLanguage : (team_profile) Changed default
	// language for team
	TeamProfileChangeDefaultLanguage *TeamProfileChangeDefaultLanguageType `json:"team_profile_change_default_language,omitempty"`
	// TeamProfileChangeLogo : (team_profile) Changed team logo displayed on
	// shared link headers
	TeamProfileChangeLogo *TeamProfileChangeLogoType `json:"team_profile_change_logo,omitempty"`
	// TeamProfileChangeName : (team_profile) Changed team name
	TeamProfileChangeName *TeamProfileChangeNameType `json:"team_profile_change_name,omitempty"`
	// TeamProfileRemoveBackground : (team_profile) Removed team background
	// displayed on shared link headers
	TeamProfileRemoveBackground *TeamProfileRemoveBackgroundType `json:"team_profile_remove_background,omitempty"`
	// TeamProfileRemoveLogo : (team_profile) Removed team logo displayed on
	// shared link headers
	TeamProfileRemoveLogo *TeamProfileRemoveLogoType `json:"team_profile_remove_logo,omitempty"`
	// TfaAddBackupPhone : (tfa) Added backup phone for two-step verification
	TfaAddBackupPhone *TfaAddBackupPhoneType `json:"tfa_add_backup_phone,omitempty"`
	// TfaAddSecurityKey : (tfa) Added security key for two-step verification
	TfaAddSecurityKey *TfaAddSecurityKeyType `json:"tfa_add_security_key,omitempty"`
	// TfaChangeBackupPhone : (tfa) Changed backup phone for two-step
	// verification
	TfaChangeBackupPhone *TfaChangeBackupPhoneType `json:"tfa_change_backup_phone,omitempty"`
	// TfaChangeStatus : (tfa) Enabled/disabled/changed two-step verification
	// setting
	TfaChangeStatus *TfaChangeStatusType `json:"tfa_change_status,omitempty"`
	// TfaRemoveBackupPhone : (tfa) Removed backup phone for two-step
	// verification
	TfaRemoveBackupPhone *TfaRemoveBackupPhoneType `json:"tfa_remove_backup_phone,omitempty"`
	// TfaRemoveSecurityKey : (tfa) Removed security key for two-step
	// verification
	TfaRemoveSecurityKey *TfaRemoveSecurityKeyType `json:"tfa_remove_security_key,omitempty"`
	// TfaReset : (tfa) Reset two-step verification for team member
	TfaReset *TfaResetType `json:"tfa_reset,omitempty"`
	// ChangedEnterpriseAdminRole : (trusted_teams) Changed enterprise admin
	// role
	ChangedEnterpriseAdminRole *ChangedEnterpriseAdminRoleType `json:"changed_enterprise_admin_role,omitempty"`
	// ChangedEnterpriseConnectedTeamStatus : (trusted_teams) Changed
	// enterprise-connected team status
	ChangedEnterpriseConnectedTeamStatus *ChangedEnterpriseConnectedTeamStatusType `json:"changed_enterprise_connected_team_status,omitempty"`
	// EndedEnterpriseAdminSession : (trusted_teams) Ended enterprise admin
	// session
	EndedEnterpriseAdminSession *EndedEnterpriseAdminSessionType `json:"ended_enterprise_admin_session,omitempty"`
	// EndedEnterpriseAdminSessionDeprecated : (trusted_teams) Ended enterprise
	// admin session (deprecated, replaced by 'Ended enterprise admin session')
	EndedEnterpriseAdminSessionDeprecated *EndedEnterpriseAdminSessionDeprecatedType `json:"ended_enterprise_admin_session_deprecated,omitempty"`
	// EnterpriseSettingsLocking : (trusted_teams) Changed who can update a
	// setting
	EnterpriseSettingsLocking *EnterpriseSettingsLockingType `json:"enterprise_settings_locking,omitempty"`
	// GuestAdminChangeStatus : (trusted_teams) Changed guest team admin status
	GuestAdminChangeStatus *GuestAdminChangeStatusType `json:"guest_admin_change_status,omitempty"`
	// StartedEnterpriseAdminSession : (trusted_teams) Started enterprise admin
	// session
	StartedEnterpriseAdminSession *StartedEnterpriseAdminSessionType `json:"started_enterprise_admin_session,omitempty"`
	// TeamMergeRequestAccepted : (trusted_teams) Accepted a team merge request
	TeamMergeRequestAccepted *TeamMergeRequestAcceptedType `json:"team_merge_request_accepted,omitempty"`
	// TeamMergeRequestAcceptedShownToPrimaryTeam : (trusted_teams) Accepted a
	// team merge request (deprecated, replaced by 'Accepted a team merge
	// request')
	TeamMergeRequestAcceptedShownToPrimaryTeam *TeamMergeRequestAcceptedShownToPrimaryTeamType `json:"team_merge_request_accepted_shown_to_primary_team,omitempty"`
	// TeamMergeRequestAcceptedShownToSecondaryTeam : (trusted_teams) Accepted a
	// team merge request (deprecated, replaced by 'Accepted a team merge
	// request')
	TeamMergeRequestAcceptedShownToSecondaryTeam *TeamMergeRequestAcceptedShownToSecondaryTeamType `json:"team_merge_request_accepted_shown_to_secondary_team,omitempty"`
	// TeamMergeRequestAutoCanceled : (trusted_teams) Automatically canceled
	// team merge request
	TeamMergeRequestAutoCanceled *TeamMergeRequestAutoCanceledType `json:"team_merge_request_auto_canceled,omitempty"`
	// TeamMergeRequestCanceled : (trusted_teams) Canceled a team merge request
	TeamMergeRequestCanceled *TeamMergeRequestCanceledType `json:"team_merge_request_canceled,omitempty"`
	// TeamMergeRequestCanceledShownToPrimaryTeam : (trusted_teams) Canceled a
	// team merge request (deprecated, replaced by 'Canceled a team merge
	// request')
	TeamMergeRequestCanceledShownToPrimaryTeam *TeamMergeRequestCanceledShownToPrimaryTeamType `json:"team_merge_request_canceled_shown_to_primary_team,omitempty"`
	// TeamMergeRequestCanceledShownToSecondaryTeam : (trusted_teams) Canceled a
	// team merge request (deprecated, replaced by 'Canceled a team merge
	// request')
	TeamMergeRequestCanceledShownToSecondaryTeam *TeamMergeRequestCanceledShownToSecondaryTeamType `json:"team_merge_request_canceled_shown_to_secondary_team,omitempty"`
	// TeamMergeRequestExpired : (trusted_teams) Team merge request expired
	TeamMergeRequestExpired *TeamMergeRequestExpiredType `json:"team_merge_request_expired,omitempty"`
	// TeamMergeRequestExpiredShownToPrimaryTeam : (trusted_teams) Team merge
	// request expired (deprecated, replaced by 'Team merge request expired')
	TeamMergeRequestExpiredShownToPrimaryTeam *TeamMergeRequestExpiredShownToPrimaryTeamType `json:"team_merge_request_expired_shown_to_primary_team,omitempty"`
	// TeamMergeRequestExpiredShownToSecondaryTeam : (trusted_teams) Team merge
	// request expired (deprecated, replaced by 'Team merge request expired')
	TeamMergeRequestExpiredShownToSecondaryTeam *TeamMergeRequestExpiredShownToSecondaryTeamType `json:"team_merge_request_expired_shown_to_secondary_team,omitempty"`
	// TeamMergeRequestRejectedShownToPrimaryTeam : (trusted_teams) Rejected a
	// team merge request (deprecated, no longer logged)
	TeamMergeRequestRejectedShownToPrimaryTeam *TeamMergeRequestRejectedShownToPrimaryTeamType `json:"team_merge_request_rejected_shown_to_primary_team,omitempty"`
	// TeamMergeRequestRejectedShownToSecondaryTeam : (trusted_teams) Rejected a
	// team merge request (deprecated, no longer logged)
	TeamMergeRequestRejectedShownToSecondaryTeam *TeamMergeRequestRejectedShownToSecondaryTeamType `json:"team_merge_request_rejected_shown_to_secondary_team,omitempty"`
	// TeamMergeRequestReminder : (trusted_teams) Sent a team merge request
	// reminder
	TeamMergeRequestReminder *TeamMergeRequestReminderType `json:"team_merge_request_reminder,omitempty"`
	// TeamMergeRequestReminderShownToPrimaryTeam : (trusted_teams) Sent a team
	// merge request reminder (deprecated, replaced by 'Sent a team merge
	// request reminder')
	TeamMergeRequestReminderShownToPrimaryTeam *TeamMergeRequestReminderShownToPrimaryTeamType `json:"team_merge_request_reminder_shown_to_primary_team,omitempty"`
	// TeamMergeRequestReminderShownToSecondaryTeam : (trusted_teams) Sent a
	// team merge request reminder (deprecated, replaced by 'Sent a team merge
	// request reminder')
	TeamMergeRequestReminderShownToSecondaryTeam *TeamMergeRequestReminderShownToSecondaryTeamType `json:"team_merge_request_reminder_shown_to_secondary_team,omitempty"`
	// TeamMergeRequestRevoked : (trusted_teams) Canceled the team merge
	TeamMergeRequestRevoked *TeamMergeRequestRevokedType `json:"team_merge_request_revoked,omitempty"`
	// TeamMergeRequestSentShownToPrimaryTeam : (trusted_teams) Requested to
	// merge their Dropbox team into yours
	TeamMergeRequestSentShownToPrimaryTeam *TeamMergeRequestSentShownToPrimaryTeamType `json:"team_merge_request_sent_shown_to_primary_team,omitempty"`
	// TeamMergeRequestSentShownToSecondaryTeam : (trusted_teams) Requested to
	// merge your team into another Dropbox team
	TeamMergeRequestSentShownToSecondaryTeam *TeamMergeRequestSentShownToSecondaryTeamType `json:"team_merge_request_sent_shown_to_secondary_team,omitempty"`
}

// Valid tag values for EventType
const (
	EventTypeAdminAlertingAlertStateChanged               = "admin_alerting_alert_state_changed"
	EventTypeAdminAlertingChangedAlertConfig              = "admin_alerting_changed_alert_config"
	EventTypeAdminAlertingTriggeredAlert                  = "admin_alerting_triggered_alert"
	EventTypeAppBlockedByPermissions                      = "app_blocked_by_permissions"
	EventTypeAppLinkTeam                                  = "app_link_team"
	EventTypeAppLinkUser                                  = "app_link_user"
	EventTypeAppUnlinkTeam                                = "app_unlink_team"
	EventTypeAppUnlinkUser                                = "app_unlink_user"
	EventTypeIntegrationConnected                         = "integration_connected"
	EventTypeIntegrationDisconnected                      = "integration_disconnected"
	EventTypeFileAddComment                               = "file_add_comment"
	EventTypeFileChangeCommentSubscription                = "file_change_comment_subscription"
	EventTypeFileDeleteComment                            = "file_delete_comment"
	EventTypeFileEditComment                              = "file_edit_comment"
	EventTypeFileLikeComment                              = "file_like_comment"
	EventTypeFileResolveComment                           = "file_resolve_comment"
	EventTypeFileUnlikeComment                            = "file_unlike_comment"
	EventTypeFileUnresolveComment                         = "file_unresolve_comment"
	EventTypeGovernancePolicyAddFolders                   = "governance_policy_add_folders"
	EventTypeGovernancePolicyAddFolderFailed              = "governance_policy_add_folder_failed"
	EventTypeGovernancePolicyContentDisposed              = "governance_policy_content_disposed"
	EventTypeGovernancePolicyCreate                       = "governance_policy_create"
	EventTypeGovernancePolicyDelete                       = "governance_policy_delete"
	EventTypeGovernancePolicyEditDetails                  = "governance_policy_edit_details"
	EventTypeGovernancePolicyEditDuration                 = "governance_policy_edit_duration"
	EventTypeGovernancePolicyExportCreated                = "governance_policy_export_created"
	EventTypeGovernancePolicyExportRemoved                = "governance_policy_export_removed"
	EventTypeGovernancePolicyRemoveFolders                = "governance_policy_remove_folders"
	EventTypeGovernancePolicyReportCreated                = "governance_policy_report_created"
	EventTypeGovernancePolicyZipPartDownloaded            = "governance_policy_zip_part_downloaded"
	EventTypeLegalHoldsActivateAHold                      = "legal_holds_activate_a_hold"
	EventTypeLegalHoldsAddMembers                         = "legal_holds_add_members"
	EventTypeLegalHoldsChangeHoldDetails                  = "legal_holds_change_hold_details"
	EventTypeLegalHoldsChangeHoldName                     = "legal_holds_change_hold_name"
	EventTypeLegalHoldsExportAHold                        = "legal_holds_export_a_hold"
	EventTypeLegalHoldsExportCancelled                    = "legal_holds_export_cancelled"
	EventTypeLegalHoldsExportDownloaded                   = "legal_holds_export_downloaded"
	EventTypeLegalHoldsExportRemoved                      = "legal_holds_export_removed"
	EventTypeLegalHoldsReleaseAHold                       = "legal_holds_release_a_hold"
	EventTypeLegalHoldsRemoveMembers                      = "legal_holds_remove_members"
	EventTypeLegalHoldsReportAHold                        = "legal_holds_report_a_hold"
	EventTypeDeviceChangeIpDesktop                        = "device_change_ip_desktop"
	EventTypeDeviceChangeIpMobile                         = "device_change_ip_mobile"
	EventTypeDeviceChangeIpWeb                            = "device_change_ip_web"
	EventTypeDeviceDeleteOnUnlinkFail                     = "device_delete_on_unlink_fail"
	EventTypeDeviceDeleteOnUnlinkSuccess                  = "device_delete_on_unlink_success"
	EventTypeDeviceLinkFail                               = "device_link_fail"
	EventTypeDeviceLinkSuccess                            = "device_link_success"
	EventTypeDeviceManagementDisabled                     = "device_management_disabled"
	EventTypeDeviceManagementEnabled                      = "device_management_enabled"
	EventTypeDeviceSyncBackupStatusChanged                = "device_sync_backup_status_changed"
	EventTypeDeviceUnlink                                 = "device_unlink"
	EventTypeDropboxPasswordsExported                     = "dropbox_passwords_exported"
	EventTypeDropboxPasswordsNewDeviceEnrolled            = "dropbox_passwords_new_device_enrolled"
	EventTypeEmmRefreshAuthToken                          = "emm_refresh_auth_token"
	EventTypeExternalDriveBackupEligibilityStatusChecked  = "external_drive_backup_eligibility_status_checked"
	EventTypeExternalDriveBackupStatusChanged             = "external_drive_backup_status_changed"
	EventTypeAccountCaptureChangeAvailability             = "account_capture_change_availability"
	EventTypeAccountCaptureMigrateAccount                 = "account_capture_migrate_account"
	EventTypeAccountCaptureNotificationEmailsSent         = "account_capture_notification_emails_sent"
	EventTypeAccountCaptureRelinquishAccount              = "account_capture_relinquish_account"
	EventTypeDisabledDomainInvites                        = "disabled_domain_invites"
	EventTypeDomainInvitesApproveRequestToJoinTeam        = "domain_invites_approve_request_to_join_team"
	EventTypeDomainInvitesDeclineRequestToJoinTeam        = "domain_invites_decline_request_to_join_team"
	EventTypeDomainInvitesEmailExistingUsers              = "domain_invites_email_existing_users"
	EventTypeDomainInvitesRequestToJoinTeam               = "domain_invites_request_to_join_team"
	EventTypeDomainInvitesSetInviteNewUserPrefToNo        = "domain_invites_set_invite_new_user_pref_to_no"
	EventTypeDomainInvitesSetInviteNewUserPrefToYes       = "domain_invites_set_invite_new_user_pref_to_yes"
	EventTypeDomainVerificationAddDomainFail              = "domain_verification_add_domain_fail"
	EventTypeDomainVerificationAddDomainSuccess           = "domain_verification_add_domain_success"
	EventTypeDomainVerificationRemoveDomain               = "domain_verification_remove_domain"
	EventTypeEnabledDomainInvites                         = "enabled_domain_invites"
	EventTypeApplyNamingConvention                        = "apply_naming_convention"
	EventTypeCreateFolder                                 = "create_folder"
	EventTypeFileAdd                                      = "file_add"
	EventTypeFileCopy                                     = "file_copy"
	EventTypeFileDelete                                   = "file_delete"
	EventTypeFileDownload                                 = "file_download"
	EventTypeFileEdit                                     = "file_edit"
	EventTypeFileGetCopyReference                         = "file_get_copy_reference"
	EventTypeFileLockingLockStatusChanged                 = "file_locking_lock_status_changed"
	EventTypeFileMove                                     = "file_move"
	EventTypeFilePermanentlyDelete                        = "file_permanently_delete"
	EventTypeFilePreview                                  = "file_preview"
	EventTypeFileRename                                   = "file_rename"
	EventTypeFileRestore                                  = "file_restore"
	EventTypeFileRevert                                   = "file_revert"
	EventTypeFileRollbackChanges                          = "file_rollback_changes"
	EventTypeFileSaveCopyReference                        = "file_save_copy_reference"
	EventTypeFolderOverviewDescriptionChanged             = "folder_overview_description_changed"
	EventTypeFolderOverviewItemPinned                     = "folder_overview_item_pinned"
	EventTypeFolderOverviewItemUnpinned                   = "folder_overview_item_unpinned"
	EventTypeObjectLabelAdded                             = "object_label_added"
	EventTypeObjectLabelRemoved                           = "object_label_removed"
	EventTypeObjectLabelUpdatedValue                      = "object_label_updated_value"
	EventTypeOrganizeFolderWithTidy                       = "organize_folder_with_tidy"
	EventTypeRewindFolder                                 = "rewind_folder"
	EventTypeUndoNamingConvention                         = "undo_naming_convention"
	EventTypeUndoOrganizeFolderWithTidy                   = "undo_organize_folder_with_tidy"
	EventTypeUserTagsAdded                                = "user_tags_added"
	EventTypeUserTagsRemoved                              = "user_tags_removed"
	EventTypeEmailIngestReceiveFile                       = "email_ingest_receive_file"
	EventTypeFileRequestChange                            = "file_request_change"
	EventTypeFileRequestClose                             = "file_request_close"
	EventTypeFileRequestCreate                            = "file_request_create"
	EventTypeFileRequestDelete                            = "file_request_delete"
	EventTypeFileRequestReceiveFile                       = "file_request_receive_file"
	EventTypeGroupAddExternalId                           = "group_add_external_id"
	EventTypeGroupAddMember                               = "group_add_member"
	EventTypeGroupChangeExternalId                        = "group_change_external_id"
	EventTypeGroupChangeManagementType                    = "group_change_management_type"
	EventTypeGroupChangeMemberRole                        = "group_change_member_role"
	EventTypeGroupCreate                                  = "group_create"
	EventTypeGroupDelete                                  = "group_delete"
	EventTypeGroupDescriptionUpdated                      = "group_description_updated"
	EventTypeGroupJoinPolicyUpdated                       = "group_join_policy_updated"
	EventTypeGroupMoved                                   = "group_moved"
	EventTypeGroupRemoveExternalId                        = "group_remove_external_id"
	EventTypeGroupRemoveMember                            = "group_remove_member"
	EventTypeGroupRename                                  = "group_rename"
	EventTypeAccountLockOrUnlocked                        = "account_lock_or_unlocked"
	EventTypeEmmError                                     = "emm_error"
	EventTypeGuestAdminSignedInViaTrustedTeams            = "guest_admin_signed_in_via_trusted_teams"
	EventTypeGuestAdminSignedOutViaTrustedTeams           = "guest_admin_signed_out_via_trusted_teams"
	EventTypeLoginFail                                    = "login_fail"
	EventTypeLoginSuccess                                 = "login_success"
	EventTypeLogout                                       = "logout"
	EventTypeResellerSupportSessionEnd                    = "reseller_support_session_end"
	EventTypeResellerSupportSessionStart                  = "reseller_support_session_start"
	EventTypeSignInAsSessionEnd                           = "sign_in_as_session_end"
	EventTypeSignInAsSessionStart                         = "sign_in_as_session_start"
	EventTypeSsoError                                     = "sso_error"
	EventTypeCreateTeamInviteLink                         = "create_team_invite_link"
	EventTypeDeleteTeamInviteLink                         = "delete_team_invite_link"
	EventTypeMemberAddExternalId                          = "member_add_external_id"
	EventTypeMemberAddName                                = "member_add_name"
	EventTypeMemberChangeAdminRole                        = "member_change_admin_role"
	EventTypeMemberChangeEmail                            = "member_change_email"
	EventTypeMemberChangeExternalId                       = "member_change_external_id"
	EventTypeMemberChangeMembershipType                   = "member_change_membership_type"
	EventTypeMemberChangeName                             = "member_change_name"
	EventTypeMemberChangeResellerRole                     = "member_change_reseller_role"
	EventTypeMemberChangeStatus                           = "member_change_status"
	EventTypeMemberDeleteManualContacts                   = "member_delete_manual_contacts"
	EventTypeMemberDeleteProfilePhoto                     = "member_delete_profile_photo"
	EventTypeMemberPermanentlyDeleteAccountContents       = "member_permanently_delete_account_contents"
	EventTypeMemberRemoveExternalId                       = "member_remove_external_id"
	EventTypeMemberSetProfilePhoto                        = "member_set_profile_photo"
	EventTypeMemberSpaceLimitsAddCustomQuota              = "member_space_limits_add_custom_quota"
	EventTypeMemberSpaceLimitsChangeCustomQuota           = "member_space_limits_change_custom_quota"
	EventTypeMemberSpaceLimitsChangeStatus                = "member_space_limits_change_status"
	EventTypeMemberSpaceLimitsRemoveCustomQuota           = "member_space_limits_remove_custom_quota"
	EventTypeMemberSuggest                                = "member_suggest"
	EventTypeMemberTransferAccountContents                = "member_transfer_account_contents"
	EventTypePendingSecondaryEmailAdded                   = "pending_secondary_email_added"
	EventTypeSecondaryEmailDeleted                        = "secondary_email_deleted"
	EventTypeSecondaryEmailVerified                       = "secondary_email_verified"
	EventTypeSecondaryMailsPolicyChanged                  = "secondary_mails_policy_changed"
	EventTypeBinderAddPage                                = "binder_add_page"
	EventTypeBinderAddSection                             = "binder_add_section"
	EventTypeBinderRemovePage                             = "binder_remove_page"
	EventTypeBinderRemoveSection                          = "binder_remove_section"
	EventTypeBinderRenamePage                             = "binder_rename_page"
	EventTypeBinderRenameSection                          = "binder_rename_section"
	EventTypeBinderReorderPage                            = "binder_reorder_page"
	EventTypeBinderReorderSection                         = "binder_reorder_section"
	EventTypePaperContentAddMember                        = "paper_content_add_member"
	EventTypePaperContentAddToFolder                      = "paper_content_add_to_folder"
	EventTypePaperContentArchive                          = "paper_content_archive"
	EventTypePaperContentCreate                           = "paper_content_create"
	EventTypePaperContentPermanentlyDelete                = "paper_content_permanently_delete"
	EventTypePaperContentRemoveFromFolder                 = "paper_content_remove_from_folder"
	EventTypePaperContentRemoveMember                     = "paper_content_remove_member"
	EventTypePaperContentRename                           = "paper_content_rename"
	EventTypePaperContentRestore                          = "paper_content_restore"
	EventTypePaperDocAddComment                           = "paper_doc_add_comment"
	EventTypePaperDocChangeMemberRole                     = "paper_doc_change_member_role"
	EventTypePaperDocChangeSharingPolicy                  = "paper_doc_change_sharing_policy"
	EventTypePaperDocChangeSubscription                   = "paper_doc_change_subscription"
	EventTypePaperDocDeleted                              = "paper_doc_deleted"
	EventTypePaperDocDeleteComment                        = "paper_doc_delete_comment"
	EventTypePaperDocDownload                             = "paper_doc_download"
	EventTypePaperDocEdit                                 = "paper_doc_edit"
	EventTypePaperDocEditComment                          = "paper_doc_edit_comment"
	EventTypePaperDocFollowed                             = "paper_doc_followed"
	EventTypePaperDocMention                              = "paper_doc_mention"
	EventTypePaperDocOwnershipChanged                     = "paper_doc_ownership_changed"
	EventTypePaperDocRequestAccess                        = "paper_doc_request_access"
	EventTypePaperDocResolveComment                       = "paper_doc_resolve_comment"
	EventTypePaperDocRevert                               = "paper_doc_revert"
	EventTypePaperDocSlackShare                           = "paper_doc_slack_share"
	EventTypePaperDocTeamInvite                           = "paper_doc_team_invite"
	EventTypePaperDocTrashed                              = "paper_doc_trashed"
	EventTypePaperDocUnresolveComment                     = "paper_doc_unresolve_comment"
	EventTypePaperDocUntrashed                            = "paper_doc_untrashed"
	EventTypePaperDocView                                 = "paper_doc_view"
	EventTypePaperExternalViewAllow                       = "paper_external_view_allow"
	EventTypePaperExternalViewDefaultTeam                 = "paper_external_view_default_team"
	EventTypePaperExternalViewForbid                      = "paper_external_view_forbid"
	EventTypePaperFolderChangeSubscription                = "paper_folder_change_subscription"
	EventTypePaperFolderDeleted                           = "paper_folder_deleted"
	EventTypePaperFolderFollowed                          = "paper_folder_followed"
	EventTypePaperFolderTeamInvite                        = "paper_folder_team_invite"
	EventTypePaperPublishedLinkChangePermission           = "paper_published_link_change_permission"
	EventTypePaperPublishedLinkCreate                     = "paper_published_link_create"
	EventTypePaperPublishedLinkDisabled                   = "paper_published_link_disabled"
	EventTypePaperPublishedLinkView                       = "paper_published_link_view"
	EventTypePasswordChange                               = "password_change"
	EventTypePasswordReset                                = "password_reset"
	EventTypePasswordResetAll                             = "password_reset_all"
	EventTypeClassificationCreateReport                   = "classification_create_report"
	EventTypeClassificationCreateReportFail               = "classification_create_report_fail"
	EventTypeEmmCreateExceptionsReport                    = "emm_create_exceptions_report"
	EventTypeEmmCreateUsageReport                         = "emm_create_usage_report"
	EventTypeExportMembersReport                          = "export_members_report"
	EventTypeExportMembersReportFail                      = "export_members_report_fail"
	EventTypeExternalSharingCreateReport                  = "external_sharing_create_report"
	EventTypeExternalSharingReportFailed                  = "external_sharing_report_failed"
	EventTypeNoExpirationLinkGenCreateReport              = "no_expiration_link_gen_create_report"
	EventTypeNoExpirationLinkGenReportFailed              = "no_expiration_link_gen_report_failed"
	EventTypeNoPasswordLinkGenCreateReport                = "no_password_link_gen_create_report"
	EventTypeNoPasswordLinkGenReportFailed                = "no_password_link_gen_report_failed"
	EventTypeNoPasswordLinkViewCreateReport               = "no_password_link_view_create_report"
	EventTypeNoPasswordLinkViewReportFailed               = "no_password_link_view_report_failed"
	EventTypeOutdatedLinkViewCreateReport                 = "outdated_link_view_create_report"
	EventTypeOutdatedLinkViewReportFailed                 = "outdated_link_view_report_failed"
	EventTypePaperAdminExportStart                        = "paper_admin_export_start"
	EventTypeSmartSyncCreateAdminPrivilegeReport          = "smart_sync_create_admin_privilege_report"
	EventTypeTeamActivityCreateReport                     = "team_activity_create_report"
	EventTypeTeamActivityCreateReportFail                 = "team_activity_create_report_fail"
	EventTypeCollectionShare                              = "collection_share"
	EventTypeFileTransfersFileAdd                         = "file_transfers_file_add"
	EventTypeFileTransfersTransferDelete                  = "file_transfers_transfer_delete"
	EventTypeFileTransfersTransferDownload                = "file_transfers_transfer_download"
	EventTypeFileTransfersTransferSend                    = "file_transfers_transfer_send"
	EventTypeFileTransfersTransferView                    = "file_transfers_transfer_view"
	EventTypeNoteAclInviteOnly                            = "note_acl_invite_only"
	EventTypeNoteAclLink                                  = "note_acl_link"
	EventTypeNoteAclTeamLink                              = "note_acl_team_link"
	EventTypeNoteShared                                   = "note_shared"
	EventTypeNoteShareReceive                             = "note_share_receive"
	EventTypeOpenNoteShared                               = "open_note_shared"
	EventTypeSfAddGroup                                   = "sf_add_group"
	EventTypeSfAllowNonMembersToViewSharedLinks           = "sf_allow_non_members_to_view_shared_links"
	EventTypeSfExternalInviteWarn                         = "sf_external_invite_warn"
	EventTypeSfFbInvite                                   = "sf_fb_invite"
	EventTypeSfFbInviteChangeRole                         = "sf_fb_invite_change_role"
	EventTypeSfFbUninvite                                 = "sf_fb_uninvite"
	EventTypeSfInviteGroup                                = "sf_invite_group"
	EventTypeSfTeamGrantAccess                            = "sf_team_grant_access"
	EventTypeSfTeamInvite                                 = "sf_team_invite"
	EventTypeSfTeamInviteChangeRole                       = "sf_team_invite_change_role"
	EventTypeSfTeamJoin                                   = "sf_team_join"
	EventTypeSfTeamJoinFromOobLink                        = "sf_team_join_from_oob_link"
	EventTypeSfTeamUninvite                               = "sf_team_uninvite"
	EventTypeSharedContentAddInvitees                     = "shared_content_add_invitees"
	EventTypeSharedContentAddLinkExpiry                   = "shared_content_add_link_expiry"
	EventTypeSharedContentAddLinkPassword                 = "shared_content_add_link_password"
	EventTypeSharedContentAddMember                       = "shared_content_add_member"
	EventTypeSharedContentChangeDownloadsPolicy           = "shared_content_change_downloads_policy"
	EventTypeSharedContentChangeInviteeRole               = "shared_content_change_invitee_role"
	EventTypeSharedContentChangeLinkAudience              = "shared_content_change_link_audience"
	EventTypeSharedContentChangeLinkExpiry                = "shared_content_change_link_expiry"
	EventTypeSharedContentChangeLinkPassword              = "shared_content_change_link_password"
	EventTypeSharedContentChangeMemberRole                = "shared_content_change_member_role"
	EventTypeSharedContentChangeViewerInfoPolicy          = "shared_content_change_viewer_info_policy"
	EventTypeSharedContentClaimInvitation                 = "shared_content_claim_invitation"
	EventTypeSharedContentCopy                            = "shared_content_copy"
	EventTypeSharedContentDownload                        = "shared_content_download"
	EventTypeSharedContentRelinquishMembership            = "shared_content_relinquish_membership"
	EventTypeSharedContentRemoveInvitees                  = "shared_content_remove_invitees"
	EventTypeSharedContentRemoveLinkExpiry                = "shared_content_remove_link_expiry"
	EventTypeSharedContentRemoveLinkPassword              = "shared_content_remove_link_password"
	EventTypeSharedContentRemoveMember                    = "shared_content_remove_member"
	EventTypeSharedContentRequestAccess                   = "shared_content_request_access"
	EventTypeSharedContentRestoreInvitees                 = "shared_content_restore_invitees"
	EventTypeSharedContentRestoreMember                   = "shared_content_restore_member"
	EventTypeSharedContentUnshare                         = "shared_content_unshare"
	EventTypeSharedContentView                            = "shared_content_view"
	EventTypeSharedFolderChangeLinkPolicy                 = "shared_folder_change_link_policy"
	EventTypeSharedFolderChangeMembersInheritancePolicy   = "shared_folder_change_members_inheritance_policy"
	EventTypeSharedFolderChangeMembersManagementPolicy    = "shared_folder_change_members_management_policy"
	EventTypeSharedFolderChangeMembersPolicy              = "shared_folder_change_members_policy"
	EventTypeSharedFolderCreate                           = "shared_folder_create"
	EventTypeSharedFolderDeclineInvitation                = "shared_folder_decline_invitation"
	EventTypeSharedFolderMount                            = "shared_folder_mount"
	EventTypeSharedFolderNest                             = "shared_folder_nest"
	EventTypeSharedFolderTransferOwnership                = "shared_folder_transfer_ownership"
	EventTypeSharedFolderUnmount                          = "shared_folder_unmount"
	EventTypeSharedLinkAddExpiry                          = "shared_link_add_expiry"
	EventTypeSharedLinkChangeExpiry                       = "shared_link_change_expiry"
	EventTypeSharedLinkChangeVisibility                   = "shared_link_change_visibility"
	EventTypeSharedLinkCopy                               = "shared_link_copy"
	EventTypeSharedLinkCreate                             = "shared_link_create"
	EventTypeSharedLinkDisable                            = "shared_link_disable"
	EventTypeSharedLinkDownload                           = "shared_link_download"
	EventTypeSharedLinkRemoveExpiry                       = "shared_link_remove_expiry"
	EventTypeSharedLinkSettingsAddExpiration              = "shared_link_settings_add_expiration"
	EventTypeSharedLinkSettingsAddPassword                = "shared_link_settings_add_password"
	EventTypeSharedLinkSettingsAllowDownloadDisabled      = "shared_link_settings_allow_download_disabled"
	EventTypeSharedLinkSettingsAllowDownloadEnabled       = "shared_link_settings_allow_download_enabled"
	EventTypeSharedLinkSettingsChangeAudience             = "shared_link_settings_change_audience"
	EventTypeSharedLinkSettingsChangeExpiration           = "shared_link_settings_change_expiration"
	EventTypeSharedLinkSettingsChangePassword             = "shared_link_settings_change_password"
	EventTypeSharedLinkSettingsRemoveExpiration           = "shared_link_settings_remove_expiration"
	EventTypeSharedLinkSettingsRemovePassword             = "shared_link_settings_remove_password"
	EventTypeSharedLinkShare                              = "shared_link_share"
	EventTypeSharedLinkView                               = "shared_link_view"
	EventTypeSharedNoteOpened                             = "shared_note_opened"
	EventTypeShmodelDisableDownloads                      = "shmodel_disable_downloads"
	EventTypeShmodelEnableDownloads                       = "shmodel_enable_downloads"
	EventTypeShmodelGroupShare                            = "shmodel_group_share"
	EventTypeShowcaseAccessGranted                        = "showcase_access_granted"
	EventTypeShowcaseAddMember                            = "showcase_add_member"
	EventTypeShowcaseArchived                             = "showcase_archived"
	EventTypeShowcaseCreated                              = "showcase_created"
	EventTypeShowcaseDeleteComment                        = "showcase_delete_comment"
	EventTypeShowcaseEdited                               = "showcase_edited"
	EventTypeShowcaseEditComment                          = "showcase_edit_comment"
	EventTypeShowcaseFileAdded                            = "showcase_file_added"
	EventTypeShowcaseFileDownload                         = "showcase_file_download"
	EventTypeShowcaseFileRemoved                          = "showcase_file_removed"
	EventTypeShowcaseFileView                             = "showcase_file_view"
	EventTypeShowcasePermanentlyDeleted                   = "showcase_permanently_deleted"
	EventTypeShowcasePostComment                          = "showcase_post_comment"
	EventTypeShowcaseRemoveMember                         = "showcase_remove_member"
	EventTypeShowcaseRenamed                              = "showcase_renamed"
	EventTypeShowcaseRequestAccess                        = "showcase_request_access"
	EventTypeShowcaseResolveComment                       = "showcase_resolve_comment"
	EventTypeShowcaseRestored                             = "showcase_restored"
	EventTypeShowcaseTrashed                              = "showcase_trashed"
	EventTypeShowcaseTrashedDeprecated                    = "showcase_trashed_deprecated"
	EventTypeShowcaseUnresolveComment                     = "showcase_unresolve_comment"
	EventTypeShowcaseUntrashed                            = "showcase_untrashed"
	EventTypeShowcaseUntrashedDeprecated                  = "showcase_untrashed_deprecated"
	EventTypeShowcaseView                                 = "showcase_view"
	EventTypeSsoAddCert                                   = "sso_add_cert"
	EventTypeSsoAddLoginUrl                               = "sso_add_login_url"
	EventTypeSsoAddLogoutUrl                              = "sso_add_logout_url"
	EventTypeSsoChangeCert                                = "sso_change_cert"
	EventTypeSsoChangeLoginUrl                            = "sso_change_login_url"
	EventTypeSsoChangeLogoutUrl                           = "sso_change_logout_url"
	EventTypeSsoChangeSamlIdentityMode                    = "sso_change_saml_identity_mode"
	EventTypeSsoRemoveCert                                = "sso_remove_cert"
	EventTypeSsoRemoveLoginUrl                            = "sso_remove_login_url"
	EventTypeSsoRemoveLogoutUrl                           = "sso_remove_logout_url"
	EventTypeTeamFolderChangeStatus                       = "team_folder_change_status"
	EventTypeTeamFolderCreate                             = "team_folder_create"
	EventTypeTeamFolderDowngrade                          = "team_folder_downgrade"
	EventTypeTeamFolderPermanentlyDelete                  = "team_folder_permanently_delete"
	EventTypeTeamFolderRename                             = "team_folder_rename"
	EventTypeTeamSelectiveSyncSettingsChanged             = "team_selective_sync_settings_changed"
	EventTypeAccountCaptureChangePolicy                   = "account_capture_change_policy"
	EventTypeAdminEmailRemindersChanged                   = "admin_email_reminders_changed"
	EventTypeAllowDownloadDisabled                        = "allow_download_disabled"
	EventTypeAllowDownloadEnabled                         = "allow_download_enabled"
	EventTypeAppPermissionsChanged                        = "app_permissions_changed"
	EventTypeCameraUploadsPolicyChanged                   = "camera_uploads_policy_changed"
	EventTypeCaptureTranscriptPolicyChanged               = "capture_transcript_policy_changed"
	EventTypeClassificationChangePolicy                   = "classification_change_policy"
	EventTypeComputerBackupPolicyChanged                  = "computer_backup_policy_changed"
	EventTypeContentAdministrationPolicyChanged           = "content_administration_policy_changed"
	EventTypeDataPlacementRestrictionChangePolicy         = "data_placement_restriction_change_policy"
	EventTypeDataPlacementRestrictionSatisfyPolicy        = "data_placement_restriction_satisfy_policy"
	EventTypeDeviceApprovalsAddException                  = "device_approvals_add_exception"
	EventTypeDeviceApprovalsChangeDesktopPolicy           = "device_approvals_change_desktop_policy"
	EventTypeDeviceApprovalsChangeMobilePolicy            = "device_approvals_change_mobile_policy"
	EventTypeDeviceApprovalsChangeOverageAction           = "device_approvals_change_overage_action"
	EventTypeDeviceApprovalsChangeUnlinkAction            = "device_approvals_change_unlink_action"
	EventTypeDeviceApprovalsRemoveException               = "device_approvals_remove_exception"
	EventTypeDirectoryRestrictionsAddMembers              = "directory_restrictions_add_members"
	EventTypeDirectoryRestrictionsRemoveMembers           = "directory_restrictions_remove_members"
	EventTypeDropboxPasswordsPolicyChanged                = "dropbox_passwords_policy_changed"
	EventTypeEmailIngestPolicyChanged                     = "email_ingest_policy_changed"
	EventTypeEmmAddException                              = "emm_add_exception"
	EventTypeEmmChangePolicy                              = "emm_change_policy"
	EventTypeEmmRemoveException                           = "emm_remove_exception"
	EventTypeExtendedVersionHistoryChangePolicy           = "extended_version_history_change_policy"
	EventTypeExternalDriveBackupPolicyChanged             = "external_drive_backup_policy_changed"
	EventTypeFileCommentsChangePolicy                     = "file_comments_change_policy"
	EventTypeFileLockingPolicyChanged                     = "file_locking_policy_changed"
	EventTypeFileProviderMigrationPolicyChanged           = "file_provider_migration_policy_changed"
	EventTypeFileRequestsChangePolicy                     = "file_requests_change_policy"
	EventTypeFileRequestsEmailsEnabled                    = "file_requests_emails_enabled"
	EventTypeFileRequestsEmailsRestrictedToTeamOnly       = "file_requests_emails_restricted_to_team_only"
	EventTypeFileTransfersPolicyChanged                   = "file_transfers_policy_changed"
	EventTypeGoogleSsoChangePolicy                        = "google_sso_change_policy"
	EventTypeGroupUserManagementChangePolicy              = "group_user_management_change_policy"
	EventTypeIntegrationPolicyChanged                     = "integration_policy_changed"
	EventTypeInviteAcceptanceEmailPolicyChanged           = "invite_acceptance_email_policy_changed"
	EventTypeMemberRequestsChangePolicy                   = "member_requests_change_policy"
	EventTypeMemberSendInvitePolicyChanged                = "member_send_invite_policy_changed"
	EventTypeMemberSpaceLimitsAddException                = "member_space_limits_add_exception"
	EventTypeMemberSpaceLimitsChangeCapsTypePolicy        = "member_space_limits_change_caps_type_policy"
	EventTypeMemberSpaceLimitsChangePolicy                = "member_space_limits_change_policy"
	EventTypeMemberSpaceLimitsRemoveException             = "member_space_limits_remove_exception"
	EventTypeMemberSuggestionsChangePolicy                = "member_suggestions_change_policy"
	EventTypeMicrosoftOfficeAddinChangePolicy             = "microsoft_office_addin_change_policy"
	EventTypeNetworkControlChangePolicy                   = "network_control_change_policy"
	EventTypePaperChangeDeploymentPolicy                  = "paper_change_deployment_policy"
	EventTypePaperChangeMemberLinkPolicy                  = "paper_change_member_link_policy"
	EventTypePaperChangeMemberPolicy                      = "paper_change_member_policy"
	EventTypePaperChangePolicy                            = "paper_change_policy"
	EventTypePaperDefaultFolderPolicyChanged              = "paper_default_folder_policy_changed"
	EventTypePaperDesktopPolicyChanged                    = "paper_desktop_policy_changed"
	EventTypePaperEnabledUsersGroupAddition               = "paper_enabled_users_group_addition"
	EventTypePaperEnabledUsersGroupRemoval                = "paper_enabled_users_group_removal"
	EventTypePasswordStrengthRequirementsChangePolicy     = "password_strength_requirements_change_policy"
	EventTypePermanentDeleteChangePolicy                  = "permanent_delete_change_policy"
	EventTypeResellerSupportChangePolicy                  = "reseller_support_change_policy"
	EventTypeRewindPolicyChanged                          = "rewind_policy_changed"
	EventTypeSendForSignaturePolicyChanged                = "send_for_signature_policy_changed"
	EventTypeSharingChangeFolderJoinPolicy                = "sharing_change_folder_join_policy"
	EventTypeSharingChangeLinkAllowChangeExpirationPolicy = "sharing_change_link_allow_change_expiration_policy"
	EventTypeSharingChangeLinkDefaultExpirationPolicy     = "sharing_change_link_default_expiration_policy"
	EventTypeSharingChangeLinkEnforcePasswordPolicy       = "sharing_change_link_enforce_password_policy"
	EventTypeSharingChangeLinkPolicy                      = "sharing_change_link_policy"
	EventTypeSharingChangeMemberPolicy                    = "sharing_change_member_policy"
	EventTypeShowcaseChangeDownloadPolicy                 = "showcase_change_download_policy"
	EventTypeShowcaseChangeEnabledPolicy                  = "showcase_change_enabled_policy"
	EventTypeShowcaseChangeExternalSharingPolicy          = "showcase_change_external_sharing_policy"
	EventTypeSmarterSmartSyncPolicyChanged                = "smarter_smart_sync_policy_changed"
	EventTypeSmartSyncChangePolicy                        = "smart_sync_change_policy"
	EventTypeSmartSyncNotOptOut                           = "smart_sync_not_opt_out"
	EventTypeSmartSyncOptOut                              = "smart_sync_opt_out"
	EventTypeSsoChangePolicy                              = "sso_change_policy"
	EventTypeTeamBrandingPolicyChanged                    = "team_branding_policy_changed"
	EventTypeTeamExtensionsPolicyChanged                  = "team_extensions_policy_changed"
	EventTypeTeamSelectiveSyncPolicyChanged               = "team_selective_sync_policy_changed"
	EventTypeTeamSharingWhitelistSubjectsChanged          = "team_sharing_whitelist_subjects_changed"
	EventTypeTfaAddException                              = "tfa_add_exception"
	EventTypeTfaChangePolicy                              = "tfa_change_policy"
	EventTypeTfaRemoveException                           = "tfa_remove_exception"
	EventTypeTwoAccountChangePolicy                       = "two_account_change_policy"
	EventTypeViewerInfoPolicyChanged                      = "viewer_info_policy_changed"
	EventTypeWatermarkingPolicyChanged                    = "watermarking_policy_changed"
	EventTypeWebSessionsChangeActiveSessionLimit          = "web_sessions_change_active_session_limit"
	EventTypeWebSessionsChangeFixedLengthPolicy           = "web_sessions_change_fixed_length_policy"
	EventTypeWebSessionsChangeIdleLengthPolicy            = "web_sessions_change_idle_length_policy"
	EventTypeDataResidencyMigrationRequestSuccessful      = "data_residency_migration_request_successful"
	EventTypeDataResidencyMigrationRequestUnsuccessful    = "data_residency_migration_request_unsuccessful"
	EventTypeTeamMergeFrom                                = "team_merge_from"
	EventTypeTeamMergeTo                                  = "team_merge_to"
	EventTypeTeamProfileAddBackground                     = "team_profile_add_background"
	EventTypeTeamProfileAddLogo                           = "team_profile_add_logo"
	EventTypeTeamProfileChangeBackground                  = "team_profile_change_background"
	EventTypeTeamProfileChangeDefaultLanguage             = "team_profile_change_default_language"
	EventTypeTeamProfileChangeLogo                        = "team_profile_change_logo"
	EventTypeTeamProfileChangeName                        = "team_profile_change_name"
	EventTypeTeamProfileRemoveBackground                  = "team_profile_remove_background"
	EventTypeTeamProfileRemoveLogo                        = "team_profile_remove_logo"
	EventTypeTfaAddBackupPhone                            = "tfa_add_backup_phone"
	EventTypeTfaAddSecurityKey                            = "tfa_add_security_key"
	EventTypeTfaChangeBackupPhone                         = "tfa_change_backup_phone"
	EventTypeTfaChangeStatus                              = "tfa_change_status"
	EventTypeTfaRemoveBackupPhone                         = "tfa_remove_backup_phone"
	EventTypeTfaRemoveSecurityKey                         = "tfa_remove_security_key"
	EventTypeTfaReset                                     = "tfa_reset"
	EventTypeChangedEnterpriseAdminRole                   = "changed_enterprise_admin_role"
	EventTypeChangedEnterpriseConnectedTeamStatus         = "changed_enterprise_connected_team_status"
	EventTypeEndedEnterpriseAdminSession                  = "ended_enterprise_admin_session"
	EventTypeEndedEnterpriseAdminSessionDeprecated        = "ended_enterprise_admin_session_deprecated"
	EventTypeEnterpriseSettingsLocking                    = "enterprise_settings_locking"
	EventTypeGuestAdminChangeStatus                       = "guest_admin_change_status"
	EventTypeStartedEnterpriseAdminSession                = "started_enterprise_admin_session"
	EventTypeTeamMergeRequestAccepted                     = "team_merge_request_accepted"
	EventTypeTeamMergeRequestAcceptedShownToPrimaryTeam   = "team_merge_request_accepted_shown_to_primary_team"
	EventTypeTeamMergeRequestAcceptedShownToSecondaryTeam = "team_merge_request_accepted_shown_to_secondary_team"
	EventTypeTeamMergeRequestAutoCanceled                 = "team_merge_request_auto_canceled"
	EventTypeTeamMergeRequestCanceled                     = "team_merge_request_canceled"
	EventTypeTeamMergeRequestCanceledShownToPrimaryTeam   = "team_merge_request_canceled_shown_to_primary_team"
	EventTypeTeamMergeRequestCanceledShownToSecondaryTeam = "team_merge_request_canceled_shown_to_secondary_team"
	EventTypeTeamMergeRequestExpired                      = "team_merge_request_expired"
	EventTypeTeamMergeRequestExpiredShownToPrimaryTeam    = "team_merge_request_expired_shown_to_primary_team"
	EventTypeTeamMergeRequestExpiredShownToSecondaryTeam  = "team_merge_request_expired_shown_to_secondary_team"
	EventTypeTeamMergeRequestRejectedShownToPrimaryTeam   = "team_merge_request_rejected_shown_to_primary_team"
	EventTypeTeamMergeRequestRejectedShownToSecondaryTeam = "team_merge_request_rejected_shown_to_secondary_team"
	EventTypeTeamMergeRequestReminder                     = "team_merge_request_reminder"
	EventTypeTeamMergeRequestReminderShownToPrimaryTeam   = "team_merge_request_reminder_shown_to_primary_team"
	EventTypeTeamMergeRequestReminderShownToSecondaryTeam = "team_merge_request_reminder_shown_to_secondary_team"
	EventTypeTeamMergeRequestRevoked                      = "team_merge_request_revoked"
	EventTypeTeamMergeRequestSentShownToPrimaryTeam       = "team_merge_request_sent_shown_to_primary_team"
	EventTypeTeamMergeRequestSentShownToSecondaryTeam     = "team_merge_request_sent_shown_to_secondary_team"
	EventTypeOther                                        = "other"
)

// UnmarshalJSON deserializes into a EventType instance
func (u *EventType) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "admin_alerting_alert_state_changed":
		if err = json.Unmarshal(body, &u.AdminAlertingAlertStateChanged); err != nil {
			return err
		}

	case "admin_alerting_changed_alert_config":
		if err = json.Unmarshal(body, &u.AdminAlertingChangedAlertConfig); err != nil {
			return err
		}

	case "admin_alerting_triggered_alert":
		if err = json.Unmarshal(body, &u.AdminAlertingTriggeredAlert); err != nil {
			return err
		}

	case "app_blocked_by_permissions":
		if err = json.Unmarshal(body, &u.AppBlockedByPermissions); err != nil {
			return err
		}

	case "app_link_team":
		if err = json.Unmarshal(body, &u.AppLinkTeam); err != nil {
			return err
		}

	case "app_link_user":
		if err = json.Unmarshal(body, &u.AppLinkUser); err != nil {
			return err
		}

	case "app_unlink_team":
		if err = json.Unmarshal(body, &u.AppUnlinkTeam); err != nil {
			return err
		}

	case "app_unlink_user":
		if err = json.Unmarshal(body, &u.AppUnlinkUser); err != nil {
			return err
		}

	case "integration_connected":
		if err = json.Unmarshal(body, &u.IntegrationConnected); err != nil {
			return err
		}

	case "integration_disconnected":
		if err = json.Unmarshal(body, &u.IntegrationDisconnected); err != nil {
			return err
		}

	case "file_add_comment":
		if err = json.Unmarshal(body, &u.FileAddComment); err != nil {
			return err
		}

	case "file_change_comment_subscription":
		if err = json.Unmarshal(body, &u.FileChangeCommentSubscription); err != nil {
			return err
		}

	case "file_delete_comment":
		if err = json.Unmarshal(body, &u.FileDeleteComment); err != nil {
			return err
		}

	case "file_edit_comment":
		if err = json.Unmarshal(body, &u.FileEditComment); err != nil {
			return err
		}

	case "file_like_comment":
		if err = json.Unmarshal(body, &u.FileLikeComment); err != nil {
			return err
		}

	case "file_resolve_comment":
		if err = json.Unmarshal(body, &u.FileResolveComment); err != nil {
			return err
		}

	case "file_unlike_comment":
		if err = json.Unmarshal(body, &u.FileUnlikeComment); err != nil {
			return err
		}

	case "file_unresolve_comment":
		if err = json.Unmarshal(body, &u.FileUnresolveComment); err != nil {
			return err
		}

	case "governance_policy_add_folders":
		if err = json.Unmarshal(body, &u.GovernancePolicyAddFolders); err != nil {
			return err
		}

	case "governance_policy_add_folder_failed":
		if err = json.Unmarshal(body, &u.GovernancePolicyAddFolderFailed); err != nil {
			return err
		}

	case "governance_policy_content_disposed":
		if err = json.Unmarshal(body, &u.GovernancePolicyContentDisposed); err != nil {
			return err
		}

	case "governance_policy_create":
		if err = json.Unmarshal(body, &u.GovernancePolicyCreate); err != nil {
			return err
		}

	case "governance_policy_delete":
		if err = json.Unmarshal(body, &u.GovernancePolicyDelete); err != nil {
			return err
		}

	case "governance_policy_edit_details":
		if err = json.Unmarshal(body, &u.GovernancePolicyEditDetails); err != nil {
			return err
		}

	case "governance_policy_edit_duration":
		if err = json.Unmarshal(body, &u.GovernancePolicyEditDuration); err != nil {
			return err
		}

	case "governance_policy_export_created":
		if err = json.Unmarshal(body, &u.GovernancePolicyExportCreated); err != nil {
			return err
		}

	case "governance_policy_export_removed":
		if err = json.Unmarshal(body, &u.GovernancePolicyExportRemoved); err != nil {
			return err
		}

	case "governance_policy_remove_folders":
		if err = json.Unmarshal(body, &u.GovernancePolicyRemoveFolders); err != nil {
			return err
		}

	case "governance_policy_report_created":
		if err = json.Unmarshal(body, &u.GovernancePolicyReportCreated); err != nil {
			return err
		}

	case "governance_policy_zip_part_downloaded":
		if err = json.Unmarshal(body, &u.GovernancePolicyZipPartDownloaded); err != nil {
			return err
		}

	case "legal_holds_activate_a_hold":
		if err = json.Unmarshal(body, &u.LegalHoldsActivateAHold); err != nil {
			return err
		}

	case "legal_holds_add_members":
		if err = json.Unmarshal(body, &u.LegalHoldsAddMembers); err != nil {
			return err
		}

	case "legal_holds_change_hold_details":
		if err = json.Unmarshal(body, &u.LegalHoldsChangeHoldDetails); err != nil {
			return err
		}

	case "legal_holds_change_hold_name":
		if err = json.Unmarshal(body, &u.LegalHoldsChangeHoldName); err != nil {
			return err
		}

	case "legal_holds_export_a_hold":
		if err = json.Unmarshal(body, &u.LegalHoldsExportAHold); err != nil {
			return err
		}

	case "legal_holds_export_cancelled":
		if err = json.Unmarshal(body, &u.LegalHoldsExportCancelled); err != nil {
			return err
		}

	case "legal_holds_export_downloaded":
		if err = json.Unmarshal(body, &u.LegalHoldsExportDownloaded); err != nil {
			return err
		}

	case "legal_holds_export_removed":
		if err = json.Unmarshal(body, &u.LegalHoldsExportRemoved); err != nil {
			return err
		}

	case "legal_holds_release_a_hold":
		if err = json.Unmarshal(body, &u.LegalHoldsReleaseAHold); err != nil {
			return err
		}

	case "legal_holds_remove_members":
		if err = json.Unmarshal(body, &u.LegalHoldsRemoveMembers); err != nil {
			return err
		}

	case "legal_holds_report_a_hold":
		if err = json.Unmarshal(body, &u.LegalHoldsReportAHold); err != nil {
			return err
		}

	case "device_change_ip_desktop":
		if err = json.Unmarshal(body, &u.DeviceChangeIpDesktop); err != nil {
			return err
		}

	case "device_change_ip_mobile":
		if err = json.Unmarshal(body, &u.DeviceChangeIpMobile); err != nil {
			return err
		}

	case "device_change_ip_web":
		if err = json.Unmarshal(body, &u.DeviceChangeIpWeb); err != nil {
			return err
		}

	case "device_delete_on_unlink_fail":
		if err = json.Unmarshal(body, &u.DeviceDeleteOnUnlinkFail); err != nil {
			return err
		}

	case "device_delete_on_unlink_success":
		if err = json.Unmarshal(body, &u.DeviceDeleteOnUnlinkSuccess); err != nil {
			return err
		}

	case "device_link_fail":
		if err = json.Unmarshal(body, &u.DeviceLinkFail); err != nil {
			return err
		}

	case "device_link_success":
		if err = json.Unmarshal(body, &u.DeviceLinkSuccess); err != nil {
			return err
		}

	case "device_management_disabled":
		if err = json.Unmarshal(body, &u.DeviceManagementDisabled); err != nil {
			return err
		}

	case "device_management_enabled":
		if err = json.Unmarshal(body, &u.DeviceManagementEnabled); err != nil {
			return err
		}

	case "device_sync_backup_status_changed":
		if err = json.Unmarshal(body, &u.DeviceSyncBackupStatusChanged); err != nil {
			return err
		}

	case "device_unlink":
		if err = json.Unmarshal(body, &u.DeviceUnlink); err != nil {
			return err
		}

	case "dropbox_passwords_exported":
		if err = json.Unmarshal(body, &u.DropboxPasswordsExported); err != nil {
			return err
		}

	case "dropbox_passwords_new_device_enrolled":
		if err = json.Unmarshal(body, &u.DropboxPasswordsNewDeviceEnrolled); err != nil {
			return err
		}

	case "emm_refresh_auth_token":
		if err = json.Unmarshal(body, &u.EmmRefreshAuthToken); err != nil {
			return err
		}

	case "external_drive_backup_eligibility_status_checked":
		if err = json.Unmarshal(body, &u.ExternalDriveBackupEligibilityStatusChecked); err != nil {
			return err
		}

	case "external_drive_backup_status_changed":
		if err = json.Unmarshal(body, &u.ExternalDriveBackupStatusChanged); err != nil {
			return err
		}

	case "account_capture_change_availability":
		if err = json.Unmarshal(body, &u.AccountCaptureChangeAvailability); err != nil {
			return err
		}

	case "account_capture_migrate_account":
		if err = json.Unmarshal(body, &u.AccountCaptureMigrateAccount); err != nil {
			return err
		}

	case "account_capture_notification_emails_sent":
		if err = json.Unmarshal(body, &u.AccountCaptureNotificationEmailsSent); err != nil {
			return err
		}

	case "account_capture_relinquish_account":
		if err = json.Unmarshal(body, &u.AccountCaptureRelinquishAccount); err != nil {
			return err
		}

	case "disabled_domain_invites":
		if err = json.Unmarshal(body, &u.DisabledDomainInvites); err != nil {
			return err
		}

	case "domain_invites_approve_request_to_join_team":
		if err = json.Unmarshal(body, &u.DomainInvitesApproveRequestToJoinTeam); err != nil {
			return err
		}

	case "domain_invites_decline_request_to_join_team":
		if err = json.Unmarshal(body, &u.DomainInvitesDeclineRequestToJoinTeam); err != nil {
			return err
		}

	case "domain_invites_email_existing_users":
		if err = json.Unmarshal(body, &u.DomainInvitesEmailExistingUsers); err != nil {
			return err
		}

	case "domain_invites_request_to_join_team":
		if err = json.Unmarshal(body, &u.DomainInvitesRequestToJoinTeam); err != nil {
			return err
		}

	case "domain_invites_set_invite_new_user_pref_to_no":
		if err = json.Unmarshal(body, &u.DomainInvitesSetInviteNewUserPrefToNo); err != nil {
			return err
		}

	case "domain_invites_set_invite_new_user_pref_to_yes":
		if err = json.Unmarshal(body, &u.DomainInvitesSetInviteNewUserPrefToYes); err != nil {
			return err
		}

	case "domain_verification_add_domain_fail":
		if err = json.Unmarshal(body, &u.DomainVerificationAddDomainFail); err != nil {
			return err
		}

	case "domain_verification_add_domain_success":
		if err = json.Unmarshal(body, &u.DomainVerificationAddDomainSuccess); err != nil {
			return err
		}

	case "domain_verification_remove_domain":
		if err = json.Unmarshal(body, &u.DomainVerificationRemoveDomain); err != nil {
			return err
		}

	case "enabled_domain_invites":
		if err = json.Unmarshal(body, &u.EnabledDomainInvites); err != nil {
			return err
		}

	case "apply_naming_convention":
		if err = json.Unmarshal(body, &u.ApplyNamingConvention); err != nil {
			return err
		}

	case "create_folder":
		if err = json.Unmarshal(body, &u.CreateFolder); err != nil {
			return err
		}

	case "file_add":
		if err = json.Unmarshal(body, &u.FileAdd); err != nil {
			return err
		}

	case "file_copy":
		if err = json.Unmarshal(body, &u.FileCopy); err != nil {
			return err
		}

	case "file_delete":
		if err = json.Unmarshal(body, &u.FileDelete); err != nil {
			return err
		}

	case "file_download":
		if err = json.Unmarshal(body, &u.FileDownload); err != nil {
			return err
		}

	case "file_edit":
		if err = json.Unmarshal(body, &u.FileEdit); err != nil {
			return err
		}

	case "file_get_copy_reference":
		if err = json.Unmarshal(body, &u.FileGetCopyReference); err != nil {
			return err
		}

	case "file_locking_lock_status_changed":
		if err = json.Unmarshal(body, &u.FileLockingLockStatusChanged); err != nil {
			return err
		}

	case "file_move":
		if err = json.Unmarshal(body, &u.FileMove); err != nil {
			return err
		}

	case "file_permanently_delete":
		if err = json.Unmarshal(body, &u.FilePermanentlyDelete); err != nil {
			return err
		}

	case "file_preview":
		if err = json.Unmarshal(body, &u.FilePreview); err != nil {
			return err
		}

	case "file_rename":
		if err = json.Unmarshal(body, &u.FileRename); err != nil {
			return err
		}

	case "file_restore":
		if err = json.Unmarshal(body, &u.FileRestore); err != nil {
			return err
		}

	case "file_revert":
		if err = json.Unmarshal(body, &u.FileRevert); err != nil {
			return err
		}

	case "file_rollback_changes":
		if err = json.Unmarshal(body, &u.FileRollbackChanges); err != nil {
			return err
		}

	case "file_save_copy_reference":
		if err = json.Unmarshal(body, &u.FileSaveCopyReference); err != nil {
			return err
		}

	case "folder_overview_description_changed":
		if err = json.Unmarshal(body, &u.FolderOverviewDescriptionChanged); err != nil {
			return err
		}

	case "folder_overview_item_pinned":
		if err = json.Unmarshal(body, &u.FolderOverviewItemPinned); err != nil {
			return err
		}

	case "folder_overview_item_unpinned":
		if err = json.Unmarshal(body, &u.FolderOverviewItemUnpinned); err != nil {
			return err
		}

	case "object_label_added":
		if err = json.Unmarshal(body, &u.ObjectLabelAdded); err != nil {
			return err
		}

	case "object_label_removed":
		if err = json.Unmarshal(body, &u.ObjectLabelRemoved); err != nil {
			return err
		}

	case "object_label_updated_value":
		if err = json.Unmarshal(body, &u.ObjectLabelUpdatedValue); err != nil {
			return err
		}

	case "organize_folder_with_tidy":
		if err = json.Unmarshal(body, &u.OrganizeFolderWithTidy); err != nil {
			return err
		}

	case "rewind_folder":
		if err = json.Unmarshal(body, &u.RewindFolder); err != nil {
			return err
		}

	case "undo_naming_convention":
		if err = json.Unmarshal(body, &u.UndoNamingConvention); err != nil {
			return err
		}

	case "undo_organize_folder_with_tidy":
		if err = json.Unmarshal(body, &u.UndoOrganizeFolderWithTidy); err != nil {
			return err
		}

	case "user_tags_added":
		if err = json.Unmarshal(body, &u.UserTagsAdded); err != nil {
			return err
		}

	case "user_tags_removed":
		if err = json.Unmarshal(body, &u.UserTagsRemoved); err != nil {
			return err
		}

	case "email_ingest_receive_file":
		if err = json.Unmarshal(body, &u.EmailIngestReceiveFile); err != nil {
			return err
		}

	case "file_request_change":
		if err = json.Unmarshal(body, &u.FileRequestChange); err != nil {
			return err
		}

	case "file_request_close":
		if err = json.Unmarshal(body, &u.FileRequestClose); err != nil {
			return err
		}

	case "file_request_create":
		if err = json.Unmarshal(body, &u.FileRequestCreate); err != nil {
			return err
		}

	case "file_request_delete":
		if err = json.Unmarshal(body, &u.FileRequestDelete); err != nil {
			return err
		}

	case "file_request_receive_file":
		if err = json.Unmarshal(body, &u.FileRequestReceiveFile); err != nil {
			return err
		}

	case "group_add_external_id":
		if err = json.Unmarshal(body, &u.GroupAddExternalId); err != nil {
			return err
		}

	case "group_add_member":
		if err = json.Unmarshal(body, &u.GroupAddMember); err != nil {
			return err
		}

	case "group_change_external_id":
		if err = json.Unmarshal(body, &u.GroupChangeExternalId); err != nil {
			return err
		}

	case "group_change_management_type":
		if err = json.Unmarshal(body, &u.GroupChangeManagementType); err != nil {
			return err
		}

	case "group_change_member_role":
		if err = json.Unmarshal(body, &u.GroupChangeMemberRole); err != nil {
			return err
		}

	case "group_create":
		if err = json.Unmarshal(body, &u.GroupCreate); err != nil {
			return err
		}

	case "group_delete":
		if err = json.Unmarshal(body, &u.GroupDelete); err != nil {
			return err
		}

	case "group_description_updated":
		if err = json.Unmarshal(body, &u.GroupDescriptionUpdated); err != nil {
			return err
		}

	case "group_join_policy_updated":
		if err = json.Unmarshal(body, &u.GroupJoinPolicyUpdated); err != nil {
			return err
		}

	case "group_moved":
		if err = json.Unmarshal(body, &u.GroupMoved); err != nil {
			return err
		}

	case "group_remove_external_id":
		if err = json.Unmarshal(body, &u.GroupRemoveExternalId); err != nil {
			return err
		}

	case "group_remove_member":
		if err = json.Unmarshal(body, &u.GroupRemoveMember); err != nil {
			return err
		}

	case "group_rename":
		if err = json.Unmarshal(body, &u.GroupRename); err != nil {
			return err
		}

	case "account_lock_or_unlocked":
		if err = json.Unmarshal(body, &u.AccountLockOrUnlocked); err != nil {
			return err
		}

	case "emm_error":
		if err = json.Unmarshal(body, &u.EmmError); err != nil {
			return err
		}

	case "guest_admin_signed_in_via_trusted_teams":
		if err = json.Unmarshal(body, &u.GuestAdminSignedInViaTrustedTeams); err != nil {
			return err
		}

	case "guest_admin_signed_out_via_trusted_teams":
		if err = json.Unmarshal(body, &u.GuestAdminSignedOutViaTrustedTeams); err != nil {
			return err
		}

	case "login_fail":
		if err = json.Unmarshal(body, &u.LoginFail); err != nil {
			return err
		}

	case "login_success":
		if err = json.Unmarshal(body, &u.LoginSuccess); err != nil {
			return err
		}

	case "logout":
		if err = json.Unmarshal(body, &u.Logout); err != nil {
			return err
		}

	case "reseller_support_session_end":
		if err = json.Unmarshal(body, &u.ResellerSupportSessionEnd); err != nil {
			return err
		}

	case "reseller_support_session_start":
		if err = json.Unmarshal(body, &u.ResellerSupportSessionStart); err != nil {
			return err
		}

	case "sign_in_as_session_end":
		if err = json.Unmarshal(body, &u.SignInAsSessionEnd); err != nil {
			return err
		}

	case "sign_in_as_session_start":
		if err = json.Unmarshal(body, &u.SignInAsSessionStart); err != nil {
			return err
		}

	case "sso_error":
		if err = json.Unmarshal(body, &u.SsoError); err != nil {
			return err
		}

	case "create_team_invite_link":
		if err = json.Unmarshal(body, &u.CreateTeamInviteLink); err != nil {
			return err
		}

	case "delete_team_invite_link":
		if err = json.Unmarshal(body, &u.DeleteTeamInviteLink); err != nil {
			return err
		}

	case "member_add_external_id":
		if err = json.Unmarshal(body, &u.MemberAddExternalId); err != nil {
			return err
		}

	case "member_add_name":
		if err = json.Unmarshal(body, &u.MemberAddName); err != nil {
			return err
		}

	case "member_change_admin_role":
		if err = json.Unmarshal(body, &u.MemberChangeAdminRole); err != nil {
			return err
		}

	case "member_change_email":
		if err = json.Unmarshal(body, &u.MemberChangeEmail); err != nil {
			return err
		}

	case "member_change_external_id":
		if err = json.Unmarshal(body, &u.MemberChangeExternalId); err != nil {
			return err
		}

	case "member_change_membership_type":
		if err = json.Unmarshal(body, &u.MemberChangeMembershipType); err != nil {
			return err
		}

	case "member_change_name":
		if err = json.Unmarshal(body, &u.MemberChangeName); err != nil {
			return err
		}

	case "member_change_reseller_role":
		if err = json.Unmarshal(body, &u.MemberChangeResellerRole); err != nil {
			return err
		}

	case "member_change_status":
		if err = json.Unmarshal(body, &u.MemberChangeStatus); err != nil {
			return err
		}

	case "member_delete_manual_contacts":
		if err = json.Unmarshal(body, &u.MemberDeleteManualContacts); err != nil {
			return err
		}

	case "member_delete_profile_photo":
		if err = json.Unmarshal(body, &u.MemberDeleteProfilePhoto); err != nil {
			return err
		}

	case "member_permanently_delete_account_contents":
		if err = json.Unmarshal(body, &u.MemberPermanentlyDeleteAccountContents); err != nil {
			return err
		}

	case "member_remove_external_id":
		if err = json.Unmarshal(body, &u.MemberRemoveExternalId); err != nil {
			return err
		}

	case "member_set_profile_photo":
		if err = json.Unmarshal(body, &u.MemberSetProfilePhoto); err != nil {
			return err
		}

	case "member_space_limits_add_custom_quota":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsAddCustomQuota); err != nil {
			return err
		}

	case "member_space_limits_change_custom_quota":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsChangeCustomQuota); err != nil {
			return err
		}

	case "member_space_limits_change_status":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsChangeStatus); err != nil {
			return err
		}

	case "member_space_limits_remove_custom_quota":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsRemoveCustomQuota); err != nil {
			return err
		}

	case "member_suggest":
		if err = json.Unmarshal(body, &u.MemberSuggest); err != nil {
			return err
		}

	case "member_transfer_account_contents":
		if err = json.Unmarshal(body, &u.MemberTransferAccountContents); err != nil {
			return err
		}

	case "pending_secondary_email_added":
		if err = json.Unmarshal(body, &u.PendingSecondaryEmailAdded); err != nil {
			return err
		}

	case "secondary_email_deleted":
		if err = json.Unmarshal(body, &u.SecondaryEmailDeleted); err != nil {
			return err
		}

	case "secondary_email_verified":
		if err = json.Unmarshal(body, &u.SecondaryEmailVerified); err != nil {
			return err
		}

	case "secondary_mails_policy_changed":
		if err = json.Unmarshal(body, &u.SecondaryMailsPolicyChanged); err != nil {
			return err
		}

	case "binder_add_page":
		if err = json.Unmarshal(body, &u.BinderAddPage); err != nil {
			return err
		}

	case "binder_add_section":
		if err = json.Unmarshal(body, &u.BinderAddSection); err != nil {
			return err
		}

	case "binder_remove_page":
		if err = json.Unmarshal(body, &u.BinderRemovePage); err != nil {
			return err
		}

	case "binder_remove_section":
		if err = json.Unmarshal(body, &u.BinderRemoveSection); err != nil {
			return err
		}

	case "binder_rename_page":
		if err = json.Unmarshal(body, &u.BinderRenamePage); err != nil {
			return err
		}

	case "binder_rename_section":
		if err = json.Unmarshal(body, &u.BinderRenameSection); err != nil {
			return err
		}

	case "binder_reorder_page":
		if err = json.Unmarshal(body, &u.BinderReorderPage); err != nil {
			return err
		}

	case "binder_reorder_section":
		if err = json.Unmarshal(body, &u.BinderReorderSection); err != nil {
			return err
		}

	case "paper_content_add_member":
		if err = json.Unmarshal(body, &u.PaperContentAddMember); err != nil {
			return err
		}

	case "paper_content_add_to_folder":
		if err = json.Unmarshal(body, &u.PaperContentAddToFolder); err != nil {
			return err
		}

	case "paper_content_archive":
		if err = json.Unmarshal(body, &u.PaperContentArchive); err != nil {
			return err
		}

	case "paper_content_create":
		if err = json.Unmarshal(body, &u.PaperContentCreate); err != nil {
			return err
		}

	case "paper_content_permanently_delete":
		if err = json.Unmarshal(body, &u.PaperContentPermanentlyDelete); err != nil {
			return err
		}

	case "paper_content_remove_from_folder":
		if err = json.Unmarshal(body, &u.PaperContentRemoveFromFolder); err != nil {
			return err
		}

	case "paper_content_remove_member":
		if err = json.Unmarshal(body, &u.PaperContentRemoveMember); err != nil {
			return err
		}

	case "paper_content_rename":
		if err = json.Unmarshal(body, &u.PaperContentRename); err != nil {
			return err
		}

	case "paper_content_restore":
		if err = json.Unmarshal(body, &u.PaperContentRestore); err != nil {
			return err
		}

	case "paper_doc_add_comment":
		if err = json.Unmarshal(body, &u.PaperDocAddComment); err != nil {
			return err
		}

	case "paper_doc_change_member_role":
		if err = json.Unmarshal(body, &u.PaperDocChangeMemberRole); err != nil {
			return err
		}

	case "paper_doc_change_sharing_policy":
		if err = json.Unmarshal(body, &u.PaperDocChangeSharingPolicy); err != nil {
			return err
		}

	case "paper_doc_change_subscription":
		if err = json.Unmarshal(body, &u.PaperDocChangeSubscription); err != nil {
			return err
		}

	case "paper_doc_deleted":
		if err = json.Unmarshal(body, &u.PaperDocDeleted); err != nil {
			return err
		}

	case "paper_doc_delete_comment":
		if err = json.Unmarshal(body, &u.PaperDocDeleteComment); err != nil {
			return err
		}

	case "paper_doc_download":
		if err = json.Unmarshal(body, &u.PaperDocDownload); err != nil {
			return err
		}

	case "paper_doc_edit":
		if err = json.Unmarshal(body, &u.PaperDocEdit); err != nil {
			return err
		}

	case "paper_doc_edit_comment":
		if err = json.Unmarshal(body, &u.PaperDocEditComment); err != nil {
			return err
		}

	case "paper_doc_followed":
		if err = json.Unmarshal(body, &u.PaperDocFollowed); err != nil {
			return err
		}

	case "paper_doc_mention":
		if err = json.Unmarshal(body, &u.PaperDocMention); err != nil {
			return err
		}

	case "paper_doc_ownership_changed":
		if err = json.Unmarshal(body, &u.PaperDocOwnershipChanged); err != nil {
			return err
		}

	case "paper_doc_request_access":
		if err = json.Unmarshal(body, &u.PaperDocRequestAccess); err != nil {
			return err
		}

	case "paper_doc_resolve_comment":
		if err = json.Unmarshal(body, &u.PaperDocResolveComment); err != nil {
			return err
		}

	case "paper_doc_revert":
		if err = json.Unmarshal(body, &u.PaperDocRevert); err != nil {
			return err
		}

	case "paper_doc_slack_share":
		if err = json.Unmarshal(body, &u.PaperDocSlackShare); err != nil {
			return err
		}

	case "paper_doc_team_invite":
		if err = json.Unmarshal(body, &u.PaperDocTeamInvite); err != nil {
			return err
		}

	case "paper_doc_trashed":
		if err = json.Unmarshal(body, &u.PaperDocTrashed); err != nil {
			return err
		}

	case "paper_doc_unresolve_comment":
		if err = json.Unmarshal(body, &u.PaperDocUnresolveComment); err != nil {
			return err
		}

	case "paper_doc_untrashed":
		if err = json.Unmarshal(body, &u.PaperDocUntrashed); err != nil {
			return err
		}

	case "paper_doc_view":
		if err = json.Unmarshal(body, &u.PaperDocView); err != nil {
			return err
		}

	case "paper_external_view_allow":
		if err = json.Unmarshal(body, &u.PaperExternalViewAllow); err != nil {
			return err
		}

	case "paper_external_view_default_team":
		if err = json.Unmarshal(body, &u.PaperExternalViewDefaultTeam); err != nil {
			return err
		}

	case "paper_external_view_forbid":
		if err = json.Unmarshal(body, &u.PaperExternalViewForbid); err != nil {
			return err
		}

	case "paper_folder_change_subscription":
		if err = json.Unmarshal(body, &u.PaperFolderChangeSubscription); err != nil {
			return err
		}

	case "paper_folder_deleted":
		if err = json.Unmarshal(body, &u.PaperFolderDeleted); err != nil {
			return err
		}

	case "paper_folder_followed":
		if err = json.Unmarshal(body, &u.PaperFolderFollowed); err != nil {
			return err
		}

	case "paper_folder_team_invite":
		if err = json.Unmarshal(body, &u.PaperFolderTeamInvite); err != nil {
			return err
		}

	case "paper_published_link_change_permission":
		if err = json.Unmarshal(body, &u.PaperPublishedLinkChangePermission); err != nil {
			return err
		}

	case "paper_published_link_create":
		if err = json.Unmarshal(body, &u.PaperPublishedLinkCreate); err != nil {
			return err
		}

	case "paper_published_link_disabled":
		if err = json.Unmarshal(body, &u.PaperPublishedLinkDisabled); err != nil {
			return err
		}

	case "paper_published_link_view":
		if err = json.Unmarshal(body, &u.PaperPublishedLinkView); err != nil {
			return err
		}

	case "password_change":
		if err = json.Unmarshal(body, &u.PasswordChange); err != nil {
			return err
		}

	case "password_reset":
		if err = json.Unmarshal(body, &u.PasswordReset); err != nil {
			return err
		}

	case "password_reset_all":
		if err = json.Unmarshal(body, &u.PasswordResetAll); err != nil {
			return err
		}

	case "classification_create_report":
		if err = json.Unmarshal(body, &u.ClassificationCreateReport); err != nil {
			return err
		}

	case "classification_create_report_fail":
		if err = json.Unmarshal(body, &u.ClassificationCreateReportFail); err != nil {
			return err
		}

	case "emm_create_exceptions_report":
		if err = json.Unmarshal(body, &u.EmmCreateExceptionsReport); err != nil {
			return err
		}

	case "emm_create_usage_report":
		if err = json.Unmarshal(body, &u.EmmCreateUsageReport); err != nil {
			return err
		}

	case "export_members_report":
		if err = json.Unmarshal(body, &u.ExportMembersReport); err != nil {
			return err
		}

	case "export_members_report_fail":
		if err = json.Unmarshal(body, &u.ExportMembersReportFail); err != nil {
			return err
		}

	case "external_sharing_create_report":
		if err = json.Unmarshal(body, &u.ExternalSharingCreateReport); err != nil {
			return err
		}

	case "external_sharing_report_failed":
		if err = json.Unmarshal(body, &u.ExternalSharingReportFailed); err != nil {
			return err
		}

	case "no_expiration_link_gen_create_report":
		if err = json.Unmarshal(body, &u.NoExpirationLinkGenCreateReport); err != nil {
			return err
		}

	case "no_expiration_link_gen_report_failed":
		if err = json.Unmarshal(body, &u.NoExpirationLinkGenReportFailed); err != nil {
			return err
		}

	case "no_password_link_gen_create_report":
		if err = json.Unmarshal(body, &u.NoPasswordLinkGenCreateReport); err != nil {
			return err
		}

	case "no_password_link_gen_report_failed":
		if err = json.Unmarshal(body, &u.NoPasswordLinkGenReportFailed); err != nil {
			return err
		}

	case "no_password_link_view_create_report":
		if err = json.Unmarshal(body, &u.NoPasswordLinkViewCreateReport); err != nil {
			return err
		}

	case "no_password_link_view_report_failed":
		if err = json.Unmarshal(body, &u.NoPasswordLinkViewReportFailed); err != nil {
			return err
		}

	case "outdated_link_view_create_report":
		if err = json.Unmarshal(body, &u.OutdatedLinkViewCreateReport); err != nil {
			return err
		}

	case "outdated_link_view_report_failed":
		if err = json.Unmarshal(body, &u.OutdatedLinkViewReportFailed); err != nil {
			return err
		}

	case "paper_admin_export_start":
		if err = json.Unmarshal(body, &u.PaperAdminExportStart); err != nil {
			return err
		}

	case "smart_sync_create_admin_privilege_report":
		if err = json.Unmarshal(body, &u.SmartSyncCreateAdminPrivilegeReport); err != nil {
			return err
		}

	case "team_activity_create_report":
		if err = json.Unmarshal(body, &u.TeamActivityCreateReport); err != nil {
			return err
		}

	case "team_activity_create_report_fail":
		if err = json.Unmarshal(body, &u.TeamActivityCreateReportFail); err != nil {
			return err
		}

	case "collection_share":
		if err = json.Unmarshal(body, &u.CollectionShare); err != nil {
			return err
		}

	case "file_transfers_file_add":
		if err = json.Unmarshal(body, &u.FileTransfersFileAdd); err != nil {
			return err
		}

	case "file_transfers_transfer_delete":
		if err = json.Unmarshal(body, &u.FileTransfersTransferDelete); err != nil {
			return err
		}

	case "file_transfers_transfer_download":
		if err = json.Unmarshal(body, &u.FileTransfersTransferDownload); err != nil {
			return err
		}

	case "file_transfers_transfer_send":
		if err = json.Unmarshal(body, &u.FileTransfersTransferSend); err != nil {
			return err
		}

	case "file_transfers_transfer_view":
		if err = json.Unmarshal(body, &u.FileTransfersTransferView); err != nil {
			return err
		}

	case "note_acl_invite_only":
		if err = json.Unmarshal(body, &u.NoteAclInviteOnly); err != nil {
			return err
		}

	case "note_acl_link":
		if err = json.Unmarshal(body, &u.NoteAclLink); err != nil {
			return err
		}

	case "note_acl_team_link":
		if err = json.Unmarshal(body, &u.NoteAclTeamLink); err != nil {
			return err
		}

	case "note_shared":
		if err = json.Unmarshal(body, &u.NoteShared); err != nil {
			return err
		}

	case "note_share_receive":
		if err = json.Unmarshal(body, &u.NoteShareReceive); err != nil {
			return err
		}

	case "open_note_shared":
		if err = json.Unmarshal(body, &u.OpenNoteShared); err != nil {
			return err
		}

	case "sf_add_group":
		if err = json.Unmarshal(body, &u.SfAddGroup); err != nil {
			return err
		}

	case "sf_allow_non_members_to_view_shared_links":
		if err = json.Unmarshal(body, &u.SfAllowNonMembersToViewSharedLinks); err != nil {
			return err
		}

	case "sf_external_invite_warn":
		if err = json.Unmarshal(body, &u.SfExternalInviteWarn); err != nil {
			return err
		}

	case "sf_fb_invite":
		if err = json.Unmarshal(body, &u.SfFbInvite); err != nil {
			return err
		}

	case "sf_fb_invite_change_role":
		if err = json.Unmarshal(body, &u.SfFbInviteChangeRole); err != nil {
			return err
		}

	case "sf_fb_uninvite":
		if err = json.Unmarshal(body, &u.SfFbUninvite); err != nil {
			return err
		}

	case "sf_invite_group":
		if err = json.Unmarshal(body, &u.SfInviteGroup); err != nil {
			return err
		}

	case "sf_team_grant_access":
		if err = json.Unmarshal(body, &u.SfTeamGrantAccess); err != nil {
			return err
		}

	case "sf_team_invite":
		if err = json.Unmarshal(body, &u.SfTeamInvite); err != nil {
			return err
		}

	case "sf_team_invite_change_role":
		if err = json.Unmarshal(body, &u.SfTeamInviteChangeRole); err != nil {
			return err
		}

	case "sf_team_join":
		if err = json.Unmarshal(body, &u.SfTeamJoin); err != nil {
			return err
		}

	case "sf_team_join_from_oob_link":
		if err = json.Unmarshal(body, &u.SfTeamJoinFromOobLink); err != nil {
			return err
		}

	case "sf_team_uninvite":
		if err = json.Unmarshal(body, &u.SfTeamUninvite); err != nil {
			return err
		}

	case "shared_content_add_invitees":
		if err = json.Unmarshal(body, &u.SharedContentAddInvitees); err != nil {
			return err
		}

	case "shared_content_add_link_expiry":
		if err = json.Unmarshal(body, &u.SharedContentAddLinkExpiry); err != nil {
			return err
		}

	case "shared_content_add_link_password":
		if err = json.Unmarshal(body, &u.SharedContentAddLinkPassword); err != nil {
			return err
		}

	case "shared_content_add_member":
		if err = json.Unmarshal(body, &u.SharedContentAddMember); err != nil {
			return err
		}

	case "shared_content_change_downloads_policy":
		if err = json.Unmarshal(body, &u.SharedContentChangeDownloadsPolicy); err != nil {
			return err
		}

	case "shared_content_change_invitee_role":
		if err = json.Unmarshal(body, &u.SharedContentChangeInviteeRole); err != nil {
			return err
		}

	case "shared_content_change_link_audience":
		if err = json.Unmarshal(body, &u.SharedContentChangeLinkAudience); err != nil {
			return err
		}

	case "shared_content_change_link_expiry":
		if err = json.Unmarshal(body, &u.SharedContentChangeLinkExpiry); err != nil {
			return err
		}

	case "shared_content_change_link_password":
		if err = json.Unmarshal(body, &u.SharedContentChangeLinkPassword); err != nil {
			return err
		}

	case "shared_content_change_member_role":
		if err = json.Unmarshal(body, &u.SharedContentChangeMemberRole); err != nil {
			return err
		}

	case "shared_content_change_viewer_info_policy":
		if err = json.Unmarshal(body, &u.SharedContentChangeViewerInfoPolicy); err != nil {
			return err
		}

	case "shared_content_claim_invitation":
		if err = json.Unmarshal(body, &u.SharedContentClaimInvitation); err != nil {
			return err
		}

	case "shared_content_copy":
		if err = json.Unmarshal(body, &u.SharedContentCopy); err != nil {
			return err
		}

	case "shared_content_download":
		if err = json.Unmarshal(body, &u.SharedContentDownload); err != nil {
			return err
		}

	case "shared_content_relinquish_membership":
		if err = json.Unmarshal(body, &u.SharedContentRelinquishMembership); err != nil {
			return err
		}

	case "shared_content_remove_invitees":
		if err = json.Unmarshal(body, &u.SharedContentRemoveInvitees); err != nil {
			return err
		}

	case "shared_content_remove_link_expiry":
		if err = json.Unmarshal(body, &u.SharedContentRemoveLinkExpiry); err != nil {
			return err
		}

	case "shared_content_remove_link_password":
		if err = json.Unmarshal(body, &u.SharedContentRemoveLinkPassword); err != nil {
			return err
		}

	case "shared_content_remove_member":
		if err = json.Unmarshal(body, &u.SharedContentRemoveMember); err != nil {
			return err
		}

	case "shared_content_request_access":
		if err = json.Unmarshal(body, &u.SharedContentRequestAccess); err != nil {
			return err
		}

	case "shared_content_restore_invitees":
		if err = json.Unmarshal(body, &u.SharedContentRestoreInvitees); err != nil {
			return err
		}

	case "shared_content_restore_member":
		if err = json.Unmarshal(body, &u.SharedContentRestoreMember); err != nil {
			return err
		}

	case "shared_content_unshare":
		if err = json.Unmarshal(body, &u.SharedContentUnshare); err != nil {
			return err
		}

	case "shared_content_view":
		if err = json.Unmarshal(body, &u.SharedContentView); err != nil {
			return err
		}

	case "shared_folder_change_link_policy":
		if err = json.Unmarshal(body, &u.SharedFolderChangeLinkPolicy); err != nil {
			return err
		}

	case "shared_folder_change_members_inheritance_policy":
		if err = json.Unmarshal(body, &u.SharedFolderChangeMembersInheritancePolicy); err != nil {
			return err
		}

	case "shared_folder_change_members_management_policy":
		if err = json.Unmarshal(body, &u.SharedFolderChangeMembersManagementPolicy); err != nil {
			return err
		}

	case "shared_folder_change_members_policy":
		if err = json.Unmarshal(body, &u.SharedFolderChangeMembersPolicy); err != nil {
			return err
		}

	case "shared_folder_create":
		if err = json.Unmarshal(body, &u.SharedFolderCreate); err != nil {
			return err
		}

	case "shared_folder_decline_invitation":
		if err = json.Unmarshal(body, &u.SharedFolderDeclineInvitation); err != nil {
			return err
		}

	case "shared_folder_mount":
		if err = json.Unmarshal(body, &u.SharedFolderMount); err != nil {
			return err
		}

	case "shared_folder_nest":
		if err = json.Unmarshal(body, &u.SharedFolderNest); err != nil {
			return err
		}

	case "shared_folder_transfer_ownership":
		if err = json.Unmarshal(body, &u.SharedFolderTransferOwnership); err != nil {
			return err
		}

	case "shared_folder_unmount":
		if err = json.Unmarshal(body, &u.SharedFolderUnmount); err != nil {
			return err
		}

	case "shared_link_add_expiry":
		if err = json.Unmarshal(body, &u.SharedLinkAddExpiry); err != nil {
			return err
		}

	case "shared_link_change_expiry":
		if err = json.Unmarshal(body, &u.SharedLinkChangeExpiry); err != nil {
			return err
		}

	case "shared_link_change_visibility":
		if err = json.Unmarshal(body, &u.SharedLinkChangeVisibility); err != nil {
			return err
		}

	case "shared_link_copy":
		if err = json.Unmarshal(body, &u.SharedLinkCopy); err != nil {
			return err
		}

	case "shared_link_create":
		if err = json.Unmarshal(body, &u.SharedLinkCreate); err != nil {
			return err
		}

	case "shared_link_disable":
		if err = json.Unmarshal(body, &u.SharedLinkDisable); err != nil {
			return err
		}

	case "shared_link_download":
		if err = json.Unmarshal(body, &u.SharedLinkDownload); err != nil {
			return err
		}

	case "shared_link_remove_expiry":
		if err = json.Unmarshal(body, &u.SharedLinkRemoveExpiry); err != nil {
			return err
		}

	case "shared_link_settings_add_expiration":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsAddExpiration); err != nil {
			return err
		}

	case "shared_link_settings_add_password":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsAddPassword); err != nil {
			return err
		}

	case "shared_link_settings_allow_download_disabled":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsAllowDownloadDisabled); err != nil {
			return err
		}

	case "shared_link_settings_allow_download_enabled":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsAllowDownloadEnabled); err != nil {
			return err
		}

	case "shared_link_settings_change_audience":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsChangeAudience); err != nil {
			return err
		}

	case "shared_link_settings_change_expiration":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsChangeExpiration); err != nil {
			return err
		}

	case "shared_link_settings_change_password":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsChangePassword); err != nil {
			return err
		}

	case "shared_link_settings_remove_expiration":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsRemoveExpiration); err != nil {
			return err
		}

	case "shared_link_settings_remove_password":
		if err = json.Unmarshal(body, &u.SharedLinkSettingsRemovePassword); err != nil {
			return err
		}

	case "shared_link_share":
		if err = json.Unmarshal(body, &u.SharedLinkShare); err != nil {
			return err
		}

	case "shared_link_view":
		if err = json.Unmarshal(body, &u.SharedLinkView); err != nil {
			return err
		}

	case "shared_note_opened":
		if err = json.Unmarshal(body, &u.SharedNoteOpened); err != nil {
			return err
		}

	case "shmodel_disable_downloads":
		if err = json.Unmarshal(body, &u.ShmodelDisableDownloads); err != nil {
			return err
		}

	case "shmodel_enable_downloads":
		if err = json.Unmarshal(body, &u.ShmodelEnableDownloads); err != nil {
			return err
		}

	case "shmodel_group_share":
		if err = json.Unmarshal(body, &u.ShmodelGroupShare); err != nil {
			return err
		}

	case "showcase_access_granted":
		if err = json.Unmarshal(body, &u.ShowcaseAccessGranted); err != nil {
			return err
		}

	case "showcase_add_member":
		if err = json.Unmarshal(body, &u.ShowcaseAddMember); err != nil {
			return err
		}

	case "showcase_archived":
		if err = json.Unmarshal(body, &u.ShowcaseArchived); err != nil {
			return err
		}

	case "showcase_created":
		if err = json.Unmarshal(body, &u.ShowcaseCreated); err != nil {
			return err
		}

	case "showcase_delete_comment":
		if err = json.Unmarshal(body, &u.ShowcaseDeleteComment); err != nil {
			return err
		}

	case "showcase_edited":
		if err = json.Unmarshal(body, &u.ShowcaseEdited); err != nil {
			return err
		}

	case "showcase_edit_comment":
		if err = json.Unmarshal(body, &u.ShowcaseEditComment); err != nil {
			return err
		}

	case "showcase_file_added":
		if err = json.Unmarshal(body, &u.ShowcaseFileAdded); err != nil {
			return err
		}

	case "showcase_file_download":
		if err = json.Unmarshal(body, &u.ShowcaseFileDownload); err != nil {
			return err
		}

	case "showcase_file_removed":
		if err = json.Unmarshal(body, &u.ShowcaseFileRemoved); err != nil {
			return err
		}

	case "showcase_file_view":
		if err = json.Unmarshal(body, &u.ShowcaseFileView); err != nil {
			return err
		}

	case "showcase_permanently_deleted":
		if err = json.Unmarshal(body, &u.ShowcasePermanentlyDeleted); err != nil {
			return err
		}

	case "showcase_post_comment":
		if err = json.Unmarshal(body, &u.ShowcasePostComment); err != nil {
			return err
		}

	case "showcase_remove_member":
		if err = json.Unmarshal(body, &u.ShowcaseRemoveMember); err != nil {
			return err
		}

	case "showcase_renamed":
		if err = json.Unmarshal(body, &u.ShowcaseRenamed); err != nil {
			return err
		}

	case "showcase_request_access":
		if err = json.Unmarshal(body, &u.ShowcaseRequestAccess); err != nil {
			return err
		}

	case "showcase_resolve_comment":
		if err = json.Unmarshal(body, &u.ShowcaseResolveComment); err != nil {
			return err
		}

	case "showcase_restored":
		if err = json.Unmarshal(body, &u.ShowcaseRestored); err != nil {
			return err
		}

	case "showcase_trashed":
		if err = json.Unmarshal(body, &u.ShowcaseTrashed); err != nil {
			return err
		}

	case "showcase_trashed_deprecated":
		if err = json.Unmarshal(body, &u.ShowcaseTrashedDeprecated); err != nil {
			return err
		}

	case "showcase_unresolve_comment":
		if err = json.Unmarshal(body, &u.ShowcaseUnresolveComment); err != nil {
			return err
		}

	case "showcase_untrashed":
		if err = json.Unmarshal(body, &u.ShowcaseUntrashed); err != nil {
			return err
		}

	case "showcase_untrashed_deprecated":
		if err = json.Unmarshal(body, &u.ShowcaseUntrashedDeprecated); err != nil {
			return err
		}

	case "showcase_view":
		if err = json.Unmarshal(body, &u.ShowcaseView); err != nil {
			return err
		}

	case "sso_add_cert":
		if err = json.Unmarshal(body, &u.SsoAddCert); err != nil {
			return err
		}

	case "sso_add_login_url":
		if err = json.Unmarshal(body, &u.SsoAddLoginUrl); err != nil {
			return err
		}

	case "sso_add_logout_url":
		if err = json.Unmarshal(body, &u.SsoAddLogoutUrl); err != nil {
			return err
		}

	case "sso_change_cert":
		if err = json.Unmarshal(body, &u.SsoChangeCert); err != nil {
			return err
		}

	case "sso_change_login_url":
		if err = json.Unmarshal(body, &u.SsoChangeLoginUrl); err != nil {
			return err
		}

	case "sso_change_logout_url":
		if err = json.Unmarshal(body, &u.SsoChangeLogoutUrl); err != nil {
			return err
		}

	case "sso_change_saml_identity_mode":
		if err = json.Unmarshal(body, &u.SsoChangeSamlIdentityMode); err != nil {
			return err
		}

	case "sso_remove_cert":
		if err = json.Unmarshal(body, &u.SsoRemoveCert); err != nil {
			return err
		}

	case "sso_remove_login_url":
		if err = json.Unmarshal(body, &u.SsoRemoveLoginUrl); err != nil {
			return err
		}

	case "sso_remove_logout_url":
		if err = json.Unmarshal(body, &u.SsoRemoveLogoutUrl); err != nil {
			return err
		}

	case "team_folder_change_status":
		if err = json.Unmarshal(body, &u.TeamFolderChangeStatus); err != nil {
			return err
		}

	case "team_folder_create":
		if err = json.Unmarshal(body, &u.TeamFolderCreate); err != nil {
			return err
		}

	case "team_folder_downgrade":
		if err = json.Unmarshal(body, &u.TeamFolderDowngrade); err != nil {
			return err
		}

	case "team_folder_permanently_delete":
		if err = json.Unmarshal(body, &u.TeamFolderPermanentlyDelete); err != nil {
			return err
		}

	case "team_folder_rename":
		if err = json.Unmarshal(body, &u.TeamFolderRename); err != nil {
			return err
		}

	case "team_selective_sync_settings_changed":
		if err = json.Unmarshal(body, &u.TeamSelectiveSyncSettingsChanged); err != nil {
			return err
		}

	case "account_capture_change_policy":
		if err = json.Unmarshal(body, &u.AccountCaptureChangePolicy); err != nil {
			return err
		}

	case "admin_email_reminders_changed":
		if err = json.Unmarshal(body, &u.AdminEmailRemindersChanged); err != nil {
			return err
		}

	case "allow_download_disabled":
		if err = json.Unmarshal(body, &u.AllowDownloadDisabled); err != nil {
			return err
		}

	case "allow_download_enabled":
		if err = json.Unmarshal(body, &u.AllowDownloadEnabled); err != nil {
			return err
		}

	case "app_permissions_changed":
		if err = json.Unmarshal(body, &u.AppPermissionsChanged); err != nil {
			return err
		}

	case "camera_uploads_policy_changed":
		if err = json.Unmarshal(body, &u.CameraUploadsPolicyChanged); err != nil {
			return err
		}

	case "capture_transcript_policy_changed":
		if err = json.Unmarshal(body, &u.CaptureTranscriptPolicyChanged); err != nil {
			return err
		}

	case "classification_change_policy":
		if err = json.Unmarshal(body, &u.ClassificationChangePolicy); err != nil {
			return err
		}

	case "computer_backup_policy_changed":
		if err = json.Unmarshal(body, &u.ComputerBackupPolicyChanged); err != nil {
			return err
		}

	case "content_administration_policy_changed":
		if err = json.Unmarshal(body, &u.ContentAdministrationPolicyChanged); err != nil {
			return err
		}

	case "data_placement_restriction_change_policy":
		if err = json.Unmarshal(body, &u.DataPlacementRestrictionChangePolicy); err != nil {
			return err
		}

	case "data_placement_restriction_satisfy_policy":
		if err = json.Unmarshal(body, &u.DataPlacementRestrictionSatisfyPolicy); err != nil {
			return err
		}

	case "device_approvals_add_exception":
		if err = json.Unmarshal(body, &u.DeviceApprovalsAddException); err != nil {
			return err
		}

	case "device_approvals_change_desktop_policy":
		if err = json.Unmarshal(body, &u.DeviceApprovalsChangeDesktopPolicy); err != nil {
			return err
		}

	case "device_approvals_change_mobile_policy":
		if err = json.Unmarshal(body, &u.DeviceApprovalsChangeMobilePolicy); err != nil {
			return err
		}

	case "device_approvals_change_overage_action":
		if err = json.Unmarshal(body, &u.DeviceApprovalsChangeOverageAction); err != nil {
			return err
		}

	case "device_approvals_change_unlink_action":
		if err = json.Unmarshal(body, &u.DeviceApprovalsChangeUnlinkAction); err != nil {
			return err
		}

	case "device_approvals_remove_exception":
		if err = json.Unmarshal(body, &u.DeviceApprovalsRemoveException); err != nil {
			return err
		}

	case "directory_restrictions_add_members":
		if err = json.Unmarshal(body, &u.DirectoryRestrictionsAddMembers); err != nil {
			return err
		}

	case "directory_restrictions_remove_members":
		if err = json.Unmarshal(body, &u.DirectoryRestrictionsRemoveMembers); err != nil {
			return err
		}

	case "dropbox_passwords_policy_changed":
		if err = json.Unmarshal(body, &u.DropboxPasswordsPolicyChanged); err != nil {
			return err
		}

	case "email_ingest_policy_changed":
		if err = json.Unmarshal(body, &u.EmailIngestPolicyChanged); err != nil {
			return err
		}

	case "emm_add_exception":
		if err = json.Unmarshal(body, &u.EmmAddException); err != nil {
			return err
		}

	case "emm_change_policy":
		if err = json.Unmarshal(body, &u.EmmChangePolicy); err != nil {
			return err
		}

	case "emm_remove_exception":
		if err = json.Unmarshal(body, &u.EmmRemoveException); err != nil {
			return err
		}

	case "extended_version_history_change_policy":
		if err = json.Unmarshal(body, &u.ExtendedVersionHistoryChangePolicy); err != nil {
			return err
		}

	case "external_drive_backup_policy_changed":
		if err = json.Unmarshal(body, &u.ExternalDriveBackupPolicyChanged); err != nil {
			return err
		}

	case "file_comments_change_policy":
		if err = json.Unmarshal(body, &u.FileCommentsChangePolicy); err != nil {
			return err
		}

	case "file_locking_policy_changed":
		if err = json.Unmarshal(body, &u.FileLockingPolicyChanged); err != nil {
			return err
		}

	case "file_provider_migration_policy_changed":
		if err = json.Unmarshal(body, &u.FileProviderMigrationPolicyChanged); err != nil {
			return err
		}

	case "file_requests_change_policy":
		if err = json.Unmarshal(body, &u.FileRequestsChangePolicy); err != nil {
			return err
		}

	case "file_requests_emails_enabled":
		if err = json.Unmarshal(body, &u.FileRequestsEmailsEnabled); err != nil {
			return err
		}

	case "file_requests_emails_restricted_to_team_only":
		if err = json.Unmarshal(body, &u.FileRequestsEmailsRestrictedToTeamOnly); err != nil {
			return err
		}

	case "file_transfers_policy_changed":
		if err = json.Unmarshal(body, &u.FileTransfersPolicyChanged); err != nil {
			return err
		}

	case "google_sso_change_policy":
		if err = json.Unmarshal(body, &u.GoogleSsoChangePolicy); err != nil {
			return err
		}

	case "group_user_management_change_policy":
		if err = json.Unmarshal(body, &u.GroupUserManagementChangePolicy); err != nil {
			return err
		}

	case "integration_policy_changed":
		if err = json.Unmarshal(body, &u.IntegrationPolicyChanged); err != nil {
			return err
		}

	case "invite_acceptance_email_policy_changed":
		if err = json.Unmarshal(body, &u.InviteAcceptanceEmailPolicyChanged); err != nil {
			return err
		}

	case "member_requests_change_policy":
		if err = json.Unmarshal(body, &u.MemberRequestsChangePolicy); err != nil {
			return err
		}

	case "member_send_invite_policy_changed":
		if err = json.Unmarshal(body, &u.MemberSendInvitePolicyChanged); err != nil {
			return err
		}

	case "member_space_limits_add_exception":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsAddException); err != nil {
			return err
		}

	case "member_space_limits_change_caps_type_policy":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsChangeCapsTypePolicy); err != nil {
			return err
		}

	case "member_space_limits_change_policy":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsChangePolicy); err != nil {
			return err
		}

	case "member_space_limits_remove_exception":
		if err = json.Unmarshal(body, &u.MemberSpaceLimitsRemoveException); err != nil {
			return err
		}

	case "member_suggestions_change_policy":
		if err = json.Unmarshal(body, &u.MemberSuggestionsChangePolicy); err != nil {
			return err
		}

	case "microsoft_office_addin_change_policy":
		if err = json.Unmarshal(body, &u.MicrosoftOfficeAddinChangePolicy); err != nil {
			return err
		}

	case "network_control_change_policy":
		if err = json.Unmarshal(body, &u.NetworkControlChangePolicy); err != nil {
			return err
		}

	case "paper_change_deployment_policy":
		if err = json.Unmarshal(body, &u.PaperChangeDeploymentPolicy); err != nil {
			return err
		}

	case "paper_change_member_link_policy":
		if err = json.Unmarshal(body, &u.PaperChangeMemberLinkPolicy); err != nil {
			return err
		}

	case "paper_change_member_policy":
		if err = json.Unmarshal(body, &u.PaperChangeMemberPolicy); err != nil {
			return err
		}

	case "paper_change_policy":
		if err = json.Unmarshal(body, &u.PaperChangePolicy); err != nil {
			return err
		}

	case "paper_default_folder_policy_changed":
		if err = json.Unmarshal(body, &u.PaperDefaultFolderPolicyChanged); err != nil {
			return err
		}

	case "paper_desktop_policy_changed":
		if err = json.Unmarshal(body, &u.PaperDesktopPolicyChanged); err != nil {
			return err
		}

	case "paper_enabled_users_group_addition":
		if err = json.Unmarshal(body, &u.PaperEnabledUsersGroupAddition); err != nil {
			return err
		}

	case "paper_enabled_users_group_removal":
		if err = json.Unmarshal(body, &u.PaperEnabledUsersGroupRemoval); err != nil {
			return err
		}

	case "password_strength_requirements_change_policy":
		if err = json.Unmarshal(body, &u.PasswordStrengthRequirementsChangePolicy); err != nil {
			return err
		}

	case "permanent_delete_change_policy":
		if err = json.Unmarshal(body, &u.PermanentDeleteChangePolicy); err != nil {
			return err
		}

	case "reseller_support_change_policy":
		if err = json.Unmarshal(body, &u.ResellerSupportChangePolicy); err != nil {
			return err
		}

	case "rewind_policy_changed":
		if err = json.Unmarshal(body, &u.RewindPolicyChanged); err != nil {
			return err
		}

	case "send_for_signature_policy_changed":
		if err = json.Unmarshal(body, &u.SendForSignaturePolicyChanged); err != nil {
			return err
		}

	case "sharing_change_folder_join_policy":
		if err = json.Unmarshal(body, &u.SharingChangeFolderJoinPolicy); err != nil {
			return err
		}

	case "sharing_change_link_allow_change_expiration_policy":
		if err = json.Unmarshal(body, &u.SharingChangeLinkAllowChangeExpirationPolicy); err != nil {
			return err
		}

	case "sharing_change_link_default_expiration_policy":
		if err = json.Unmarshal(body, &u.SharingChangeLinkDefaultExpirationPolicy); err != nil {
			return err
		}

	case "sharing_change_link_enforce_password_policy":
		if err = json.Unmarshal(body, &u.SharingChangeLinkEnforcePasswordPolicy); err != nil {
			return err
		}

	case "sharing_change_link_policy":
		if err = json.Unmarshal(body, &u.SharingChangeLinkPolicy); err != nil {
			return err
		}

	case "sharing_change_member_policy":
		if err = json.Unmarshal(body, &u.SharingChangeMemberPolicy); err != nil {
			return err
		}

	case "showcase_change_download_policy":
		if err = json.Unmarshal(body, &u.ShowcaseChangeDownloadPolicy); err != nil {
			return err
		}

	case "showcase_change_enabled_policy":
		if err = json.Unmarshal(body, &u.ShowcaseChangeEnabledPolicy); err != nil {
			return err
		}

	case "showcase_change_external_sharing_policy":
		if err = json.Unmarshal(body, &u.ShowcaseChangeExternalSharingPolicy); err != nil {
			return err
		}

	case "smarter_smart_sync_policy_changed":
		if err = json.Unmarshal(body, &u.SmarterSmartSyncPolicyChanged); err != nil {
			return err
		}

	case "smart_sync_change_policy":
		if err = json.Unmarshal(body, &u.SmartSyncChangePolicy); err != nil {
			return err
		}

	case "smart_sync_not_opt_out":
		if err = json.Unmarshal(body, &u.SmartSyncNotOptOut); err != nil {
			return err
		}

	case "smart_sync_opt_out":
		if err = json.Unmarshal(body, &u.SmartSyncOptOut); err != nil {
			return err
		}

	case "sso_change_policy":
		if err = json.Unmarshal(body, &u.SsoChangePolicy); err != nil {
			return err
		}

	case "team_branding_policy_changed":
		if err = json.Unmarshal(body, &u.TeamBrandingPolicyChanged); err != nil {
			return err
		}

	case "team_extensions_policy_changed":
		if err = json.Unmarshal(body, &u.TeamExtensionsPolicyChanged); err != nil {
			return err
		}

	case "team_selective_sync_policy_changed":
		if err = json.Unmarshal(body, &u.TeamSelectiveSyncPolicyChanged); err != nil {
			return err
		}

	case "team_sharing_whitelist_subjects_changed":
		if err = json.Unmarshal(body, &u.TeamSharingWhitelistSubjectsChanged); err != nil {
			return err
		}

	case "tfa_add_exception":
		if err = json.Unmarshal(body, &u.TfaAddException); err != nil {
			return err
		}

	case "tfa_change_policy":
		if err = json.Unmarshal(body, &u.TfaChangePolicy); err != nil {
			return err
		}

	case "tfa_remove_exception":
		if err = json.Unmarshal(body, &u.TfaRemoveException); err != nil {
			return err
		}

	case "two_account_change_policy":
		if err = json.Unmarshal(body, &u.TwoAccountChangePolicy); err != nil {
			return err
		}

	case "viewer_info_policy_changed":
		if err = json.Unmarshal(body, &u.ViewerInfoPolicyChanged); err != nil {
			return err
		}

	case "watermarking_policy_changed":
		if err = json.Unmarshal(body, &u.WatermarkingPolicyChanged); err != nil {
			return err
		}

	case "web_sessions_change_active_session_limit":
		if err = json.Unmarshal(body, &u.WebSessionsChangeActiveSessionLimit); err != nil {
			return err
		}

	case "web_sessions_change_fixed_length_policy":
		if err = json.Unmarshal(body, &u.WebSessionsChangeFixedLengthPolicy); err != nil {
			return err
		}

	case "web_sessions_change_idle_length_policy":
		if err = json.Unmarshal(body, &u.WebSessionsChangeIdleLengthPolicy); err != nil {
			return err
		}

	case "data_residency_migration_request_successful":
		if err = json.Unmarshal(body, &u.DataResidencyMigrationRequestSuccessful); err != nil {
			return err
		}

	case "data_residency_migration_request_unsuccessful":
		if err = json.Unmarshal(body, &u.DataResidencyMigrationRequestUnsuccessful); err != nil {
			return err
		}

	case "team_merge_from":
		if err = json.Unmarshal(body, &u.TeamMergeFrom); err != nil {
			return err
		}

	case "team_merge_to":
		if err = json.Unmarshal(body, &u.TeamMergeTo); err != nil {
			return err
		}

	case "team_profile_add_background":
		if err = json.Unmarshal(body, &u.TeamProfileAddBackground); err != nil {
			return err
		}

	case "team_profile_add_logo":
		if err = json.Unmarshal(body, &u.TeamProfileAddLogo); err != nil {
			return err
		}

	case "team_profile_change_background":
		if err = json.Unmarshal(body, &u.TeamProfileChangeBackground); err != nil {
			return err
		}

	case "team_profile_change_default_language":
		if err = json.Unmarshal(body, &u.TeamProfileChangeDefaultLanguage); err != nil {
			return err
		}

	case "team_profile_change_logo":
		if err = json.Unmarshal(body, &u.TeamProfileChangeLogo); err != nil {
			return err
		}

	case "team_profile_change_name":
		if err = json.Unmarshal(body, &u.TeamProfileChangeName); err != nil {
			return err
		}

	case "team_profile_remove_background":
		if err = json.Unmarshal(body, &u.TeamProfileRemoveBackground); err != nil {
			return err
		}

	case "team_profile_remove_logo":
		if err = json.Unmarshal(body, &u.TeamProfileRemoveLogo); err != nil {
			return err
		}

	case "tfa_add_backup_phone":
		if err = json.Unmarshal(body, &u.TfaAddBackupPhone); err != nil {
			return err
		}

	case "tfa_add_security_key":
		if err = json.Unmarshal(body, &u.TfaAddSecurityKey); err != nil {
			return err
		}

	case "tfa_change_backup_phone":
		if err = json.Unmarshal(body, &u.TfaChangeBackupPhone); err != nil {
			return err
		}

	case "tfa_change_status":
		if err = json.Unmarshal(body, &u.TfaChangeStatus); err != nil {
			return err
		}

	case "tfa_remove_backup_phone":
		if err = json.Unmarshal(body, &u.TfaRemoveBackupPhone); err != nil {
			return err
		}

	case "tfa_remove_security_key":
		if err = json.Unmarshal(body, &u.TfaRemoveSecurityKey); err != nil {
			return err
		}

	case "tfa_reset":
		if err = json.Unmarshal(body, &u.TfaReset); err != nil {
			return err
		}

	case "changed_enterprise_admin_role":
		if err = json.Unmarshal(body, &u.ChangedEnterpriseAdminRole); err != nil {
			return err
		}

	case "changed_enterprise_connected_team_status":
		if err = json.Unmarshal(body, &u.ChangedEnterpriseConnectedTeamStatus); err != nil {
			return err
		}

	case "ended_enterprise_admin_session":
		if err = json.Unmarshal(body, &u.EndedEnterpriseAdminSession); err != nil {
			return err
		}

	case "ended_enterprise_admin_session_deprecated":
		if err = json.Unmarshal(body, &u.EndedEnterpriseAdminSessionDeprecated); err != nil {
			return err
		}

	case "enterprise_settings_locking":
		if err = json.Unmarshal(body, &u.EnterpriseSettingsLocking); err != nil {
			return err
		}

	case "guest_admin_change_status":
		if err = json.Unmarshal(body, &u.GuestAdminChangeStatus); err != nil {
			return err
		}

	case "started_enterprise_admin_session":
		if err = json.Unmarshal(body, &u.StartedEnterpriseAdminSession); err != nil {
			return err
		}

	case "team_merge_request_accepted":
		if err = json.Unmarshal(body, &u.TeamMergeRequestAccepted); err != nil {
			return err
		}

	case "team_merge_request_accepted_shown_to_primary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestAcceptedShownToPrimaryTeam); err != nil {
			return err
		}

	case "team_merge_request_accepted_shown_to_secondary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestAcceptedShownToSecondaryTeam); err != nil {
			return err
		}

	case "team_merge_request_auto_canceled":
		if err = json.Unmarshal(body, &u.TeamMergeRequestAutoCanceled); err != nil {
			return err
		}

	case "team_merge_request_canceled":
		if err = json.Unmarshal(body, &u.TeamMergeRequestCanceled); err != nil {
			return err
		}

	case "team_merge_request_canceled_shown_to_primary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestCanceledShownToPrimaryTeam); err != nil {
			return err
		}

	case "team_merge_request_canceled_shown_to_secondary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestCanceledShownToSecondaryTeam); err != nil {
			return err
		}

	case "team_merge_request_expired":
		if err = json.Unmarshal(body, &u.TeamMergeRequestExpired); err != nil {
			return err
		}

	case "team_merge_request_expired_shown_to_primary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestExpiredShownToPrimaryTeam); err != nil {
			return err
		}

	case "team_merge_request_expired_shown_to_secondary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestExpiredShownToSecondaryTeam); err != nil {
			return err
		}

	case "team_merge_request_rejected_shown_to_primary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestRejectedShownToPrimaryTeam); err != nil {
			return err
		}

	case "team_merge_request_rejected_shown_to_secondary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestRejectedShownToSecondaryTeam); err != nil {
			return err
		}

	case "team_merge_request_reminder":
		if err = json.Unmarshal(body, &u.TeamMergeRequestReminder); err != nil {
			return err
		}

	case "team_merge_request_reminder_shown_to_primary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestReminderShownToPrimaryTeam); err != nil {
			return err
		}

	case "team_merge_request_reminder_shown_to_secondary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestReminderShownToSecondaryTeam); err != nil {
			return err
		}

	case "team_merge_request_revoked":
		if err = json.Unmarshal(body, &u.TeamMergeRequestRevoked); err != nil {
			return err
		}

	case "team_merge_request_sent_shown_to_primary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestSentShownToPrimaryTeam); err != nil {
			return err
		}

	case "team_merge_request_sent_shown_to_secondary_team":
		if err = json.Unmarshal(body, &u.TeamMergeRequestSentShownToSecondaryTeam); err != nil {
			return err
		}

	}
	return nil
}

// EventTypeArg : The type of the event.
type EventTypeArg struct {
	dropbox.Tagged
}

// Valid tag values for EventTypeArg
const (
	EventTypeArgAdminAlertingAlertStateChanged               = "admin_alerting_alert_state_changed"
	EventTypeArgAdminAlertingChangedAlertConfig              = "admin_alerting_changed_alert_config"
	EventTypeArgAdminAlertingTriggeredAlert                  = "admin_alerting_triggered_alert"
	EventTypeArgAppBlockedByPermissions                      = "app_blocked_by_permissions"
	EventTypeArgAppLinkTeam                                  = "app_link_team"
	EventTypeArgAppLinkUser                                  = "app_link_user"
	EventTypeArgAppUnlinkTeam                                = "app_unlink_team"
	EventTypeArgAppUnlinkUser                                = "app_unlink_user"
	EventTypeArgIntegrationConnected                         = "integration_connected"
	EventTypeArgIntegrationDisconnected                      = "integration_disconnected"
	EventTypeArgFileAddComment                               = "file_add_comment"
	EventTypeArgFileChangeCommentSubscription                = "file_change_comment_subscription"
	EventTypeArgFileDeleteComment                            = "file_delete_comment"
	EventTypeArgFileEditComment                              = "file_edit_comment"
	EventTypeArgFileLikeComment                              = "file_like_comment"
	EventTypeArgFileResolveComment                           = "file_resolve_comment"
	EventTypeArgFileUnlikeComment                            = "file_unlike_comment"
	EventTypeArgFileUnresolveComment                         = "file_unresolve_comment"
	EventTypeArgGovernancePolicyAddFolders                   = "governance_policy_add_folders"
	EventTypeArgGovernancePolicyAddFolderFailed              = "governance_policy_add_folder_failed"
	EventTypeArgGovernancePolicyContentDisposed              = "governance_policy_content_disposed"
	EventTypeArgGovernancePolicyCreate                       = "governance_policy_create"
	EventTypeArgGovernancePolicyDelete                       = "governance_policy_delete"
	EventTypeArgGovernancePolicyEditDetails                  = "governance_policy_edit_details"
	EventTypeArgGovernancePolicyEditDuration                 = "governance_policy_edit_duration"
	EventTypeArgGovernancePolicyExportCreated                = "governance_policy_export_created"
	EventTypeArgGovernancePolicyExportRemoved                = "governance_policy_export_removed"
	EventTypeArgGovernancePolicyRemoveFolders                = "governance_policy_remove_folders"
	EventTypeArgGovernancePolicyReportCreated                = "governance_policy_report_created"
	EventTypeArgGovernancePolicyZipPartDownloaded            = "governance_policy_zip_part_downloaded"
	EventTypeArgLegalHoldsActivateAHold                      = "legal_holds_activate_a_hold"
	EventTypeArgLegalHoldsAddMembers                         = "legal_holds_add_members"
	EventTypeArgLegalHoldsChangeHoldDetails                  = "legal_holds_change_hold_details"
	EventTypeArgLegalHoldsChangeHoldName                     = "legal_holds_change_hold_name"
	EventTypeArgLegalHoldsExportAHold                        = "legal_holds_export_a_hold"
	EventTypeArgLegalHoldsExportCancelled                    = "legal_holds_export_cancelled"
	EventTypeArgLegalHoldsExportDownloaded                   = "legal_holds_export_downloaded"
	EventTypeArgLegalHoldsExportRemoved                      = "legal_holds_export_removed"
	EventTypeArgLegalHoldsReleaseAHold                       = "legal_holds_release_a_hold"
	EventTypeArgLegalHoldsRemoveMembers                      = "legal_holds_remove_members"
	EventTypeArgLegalHoldsReportAHold                        = "legal_holds_report_a_hold"
	EventTypeArgDeviceChangeIpDesktop                        = "device_change_ip_desktop"
	EventTypeArgDeviceChangeIpMobile                         = "device_change_ip_mobile"
	EventTypeArgDeviceChangeIpWeb                            = "device_change_ip_web"
	EventTypeArgDeviceDeleteOnUnlinkFail                     = "device_delete_on_unlink_fail"
	EventTypeArgDeviceDeleteOnUnlinkSuccess                  = "device_delete_on_unlink_success"
	EventTypeArgDeviceLinkFail                               = "device_link_fail"
	EventTypeArgDeviceLinkSuccess                            = "device_link_success"
	EventTypeArgDeviceManagementDisabled                     = "device_management_disabled"
	EventTypeArgDeviceManagementEnabled                      = "device_management_enabled"
	EventTypeArgDeviceSyncBackupStatusChanged                = "device_sync_backup_status_changed"
	EventTypeArgDeviceUnlink                                 = "device_unlink"
	EventTypeArgDropboxPasswordsExported                     = "dropbox_passwords_exported"
	EventTypeArgDropboxPasswordsNewDeviceEnrolled            = "dropbox_passwords_new_device_enrolled"
	EventTypeArgEmmRefreshAuthToken                          = "emm_refresh_auth_token"
	EventTypeArgExternalDriveBackupEligibilityStatusChecked  = "external_drive_backup_eligibility_status_checked"
	EventTypeArgExternalDriveBackupStatusChanged             = "external_drive_backup_status_changed"
	EventTypeArgAccountCaptureChangeAvailability             = "account_capture_change_availability"
	EventTypeArgAccountCaptureMigrateAccount                 = "account_capture_migrate_account"
	EventTypeArgAccountCaptureNotificationEmailsSent         = "account_capture_notification_emails_sent"
	EventTypeArgAccountCaptureRelinquishAccount              = "account_capture_relinquish_account"
	EventTypeArgDisabledDomainInvites                        = "disabled_domain_invites"
	EventTypeArgDomainInvitesApproveRequestToJoinTeam        = "domain_invites_approve_request_to_join_team"
	EventTypeArgDomainInvitesDeclineRequestToJoinTeam        = "domain_invites_decline_request_to_join_team"
	EventTypeArgDomainInvitesEmailExistingUsers              = "domain_invites_email_existing_users"
	EventTypeArgDomainInvitesRequestToJoinTeam               = "domain_invites_request_to_join_team"
	EventTypeArgDomainInvitesSetInviteNewUserPrefToNo        = "domain_invites_set_invite_new_user_pref_to_no"
	EventTypeArgDomainInvitesSetInviteNewUserPrefToYes       = "domain_invites_set_invite_new_user_pref_to_yes"
	EventTypeArgDomainVerificationAddDomainFail              = "domain_verification_add_domain_fail"
	EventTypeArgDomainVerificationAddDomainSuccess           = "domain_verification_add_domain_success"
	EventTypeArgDomainVerificationRemoveDomain               = "domain_verification_remove_domain"
	EventTypeArgEnabledDomainInvites                         = "enabled_domain_invites"
	EventTypeArgApplyNamingConvention                        = "apply_naming_convention"
	EventTypeArgCreateFolder                                 = "create_folder"
	EventTypeArgFileAdd                                      = "file_add"
	EventTypeArgFileCopy                                     = "file_copy"
	EventTypeArgFileDelete                                   = "file_delete"
	EventTypeArgFileDownload                                 = "file_download"
	EventTypeArgFileEdit                                     = "file_edit"
	EventTypeArgFileGetCopyReference                         = "file_get_copy_reference"
	EventTypeArgFileLockingLockStatusChanged                 = "file_locking_lock_status_changed"
	EventTypeArgFileMove                                     = "file_move"
	EventTypeArgFilePermanentlyDelete                        = "file_permanently_delete"
	EventTypeArgFilePreview                                  = "file_preview"
	EventTypeArgFileRename                                   = "file_rename"
	EventTypeArgFileRestore                                  = "file_restore"
	EventTypeArgFileRevert                                   = "file_revert"
	EventTypeArgFileRollbackChanges                          = "file_rollback_changes"
	EventTypeArgFileSaveCopyReference                        = "file_save_copy_reference"
	EventTypeArgFolderOverviewDescriptionChanged             = "folder_overview_description_changed"
	EventTypeArgFolderOverviewItemPinned                     = "folder_overview_item_pinned"
	EventTypeArgFolderOverviewItemUnpinned                   = "folder_overview_item_unpinned"
	EventTypeArgObjectLabelAdded                             = "object_label_added"
	EventTypeArgObjectLabelRemoved                           = "object_label_removed"
	EventTypeArgObjectLabelUpdatedValue                      = "object_label_updated_value"
	EventTypeArgOrganizeFolderWithTidy                       = "organize_folder_with_tidy"
	EventTypeArgRewindFolder                                 = "rewind_folder"
	EventTypeArgUndoNamingConvention                         = "undo_naming_convention"
	EventTypeArgUndoOrganizeFolderWithTidy                   = "undo_organize_folder_with_tidy"
	EventTypeArgUserTagsAdded                                = "user_tags_added"
	EventTypeArgUserTagsRemoved                              = "user_tags_removed"
	EventTypeArgEmailIngestReceiveFile                       = "email_ingest_receive_file"
	EventTypeArgFileRequestChange                            = "file_request_change"
	EventTypeArgFileRequestClose                             = "file_request_close"
	EventTypeArgFileRequestCreate                            = "file_request_create"
	EventTypeArgFileRequestDelete                            = "file_request_delete"
	EventTypeArgFileRequestReceiveFile                       = "file_request_receive_file"
	EventTypeArgGroupAddExternalId                           = "group_add_external_id"
	EventTypeArgGroupAddMember                               = "group_add_member"
	EventTypeArgGroupChangeExternalId                        = "group_change_external_id"
	EventTypeArgGroupChangeManagementType                    = "group_change_management_type"
	EventTypeArgGroupChangeMemberRole                        = "group_change_member_role"
	EventTypeArgGroupCreate                                  = "group_create"
	EventTypeArgGroupDelete                                  = "group_delete"
	EventTypeArgGroupDescriptionUpdated                      = "group_description_updated"
	EventTypeArgGroupJoinPolicyUpdated                       = "group_join_policy_updated"
	EventTypeArgGroupMoved                                   = "group_moved"
	EventTypeArgGroupRemoveExternalId                        = "group_remove_external_id"
	EventTypeArgGroupRemoveMember                            = "group_remove_member"
	EventTypeArgGroupRename                                  = "group_rename"
	EventTypeArgAccountLockOrUnlocked                        = "account_lock_or_unlocked"
	EventTypeArgEmmError                                     = "emm_error"
	EventTypeArgGuestAdminSignedInViaTrustedTeams            = "guest_admin_signed_in_via_trusted_teams"
	EventTypeArgGuestAdminSignedOutViaTrustedTeams           = "guest_admin_signed_out_via_trusted_teams"
	EventTypeArgLoginFail                                    = "login_fail"
	EventTypeArgLoginSuccess                                 = "login_success"
	EventTypeArgLogout                                       = "logout"
	EventTypeArgResellerSupportSessionEnd                    = "reseller_support_session_end"
	EventTypeArgResellerSupportSessionStart                  = "reseller_support_session_start"
	EventTypeArgSignInAsSessionEnd                           = "sign_in_as_session_end"
	EventTypeArgSignInAsSessionStart                         = "sign_in_as_session_start"
	EventTypeArgSsoError                                     = "sso_error"
	EventTypeArgCreateTeamInviteLink                         = "create_team_invite_link"
	EventTypeArgDeleteTeamInviteLink                         = "delete_team_invite_link"
	EventTypeArgMemberAddExternalId                          = "member_add_external_id"
	EventTypeArgMemberAddName                                = "member_add_name"
	EventTypeArgMemberChangeAdminRole                        = "member_change_admin_role"
	EventTypeArgMemberChangeEmail                            = "member_change_email"
	EventTypeArgMemberChangeExternalId                       = "member_change_external_id"
	EventTypeArgMemberChangeMembershipType                   = "member_change_membership_type"
	EventTypeArgMemberChangeName                             = "member_change_name"
	EventTypeArgMemberChangeResellerRole                     = "member_change_reseller_role"
	EventTypeArgMemberChangeStatus                           = "member_change_status"
	EventTypeArgMemberDeleteManualContacts                   = "member_delete_manual_contacts"
	EventTypeArgMemberDeleteProfilePhoto                     = "member_delete_profile_photo"
	EventTypeArgMemberPermanentlyDeleteAccountContents       = "member_permanently_delete_account_contents"
	EventTypeArgMemberRemoveExternalId                       = "member_remove_external_id"
	EventTypeArgMemberSetProfilePhoto                        = "member_set_profile_photo"
	EventTypeArgMemberSpaceLimitsAddCustomQuota              = "member_space_limits_add_custom_quota"
	EventTypeArgMemberSpaceLimitsChangeCustomQuota           = "member_space_limits_change_custom_quota"
	EventTypeArgMemberSpaceLimitsChangeStatus                = "member_space_limits_change_status"
	EventTypeArgMemberSpaceLimitsRemoveCustomQuota           = "member_space_limits_remove_custom_quota"
	EventTypeArgMemberSuggest                                = "member_suggest"
	EventTypeArgMemberTransferAccountContents                = "member_transfer_account_contents"
	EventTypeArgPendingSecondaryEmailAdded                   = "pending_secondary_email_added"
	EventTypeArgSecondaryEmailDeleted                        = "secondary_email_deleted"
	EventTypeArgSecondaryEmailVerified                       = "secondary_email_verified"
	EventTypeArgSecondaryMailsPolicyChanged                  = "secondary_mails_policy_changed"
	EventTypeArgBinderAddPage                                = "binder_add_page"
	EventTypeArgBinderAddSection                             = "binder_add_section"
	EventTypeArgBinderRemovePage                             = "binder_remove_page"
	EventTypeArgBinderRemoveSection                          = "binder_remove_section"
	EventTypeArgBinderRenamePage                             = "binder_rename_page"
	EventTypeArgBinderRenameSection                          = "binder_rename_section"
	EventTypeArgBinderReorderPage                            = "binder_reorder_page"
	EventTypeArgBinderReorderSection                         = "binder_reorder_section"
	EventTypeArgPaperContentAddMember                        = "paper_content_add_member"
	EventTypeArgPaperContentAddToFolder                      = "paper_content_add_to_folder"
	EventTypeArgPaperContentArchive                          = "paper_content_archive"
	EventTypeArgPaperContentCreate                           = "paper_content_create"
	EventTypeArgPaperContentPermanentlyDelete                = "paper_content_permanently_delete"
	EventTypeArgPaperContentRemoveFromFolder                 = "paper_content_remove_from_folder"
	EventTypeArgPaperContentRemoveMember                     = "paper_content_remove_member"
	EventTypeArgPaperContentRename                           = "paper_content_rename"
	EventTypeArgPaperContentRestore                          = "paper_content_restore"
	EventTypeArgPaperDocAddComment                           = "paper_doc_add_comment"
	EventTypeArgPaperDocChangeMemberRole                     = "paper_doc_change_member_role"
	EventTypeArgPaperDocChangeSharingPolicy                  = "paper_doc_change_sharing_policy"
	EventTypeArgPaperDocChangeSubscription                   = "paper_doc_change_subscription"
	EventTypeArgPaperDocDeleted                              = "paper_doc_deleted"
	EventTypeArgPaperDocDeleteComment                        = "paper_doc_delete_comment"
	EventTypeArgPaperDocDownload                             = "paper_doc_download"
	EventTypeArgPaperDocEdit                                 = "paper_doc_edit"
	EventTypeArgPaperDocEditComment                          = "paper_doc_edit_comment"
	EventTypeArgPaperDocFollowed                             = "paper_doc_followed"
	EventTypeArgPaperDocMention                              = "paper_doc_mention"
	EventTypeArgPaperDocOwnershipChanged                     = "paper_doc_ownership_changed"
	EventTypeArgPaperDocRequestAccess                        = "paper_doc_request_access"
	EventTypeArgPaperDocResolveComment                       = "paper_doc_resolve_comment"
	EventTypeArgPaperDocRevert                               = "paper_doc_revert"
	EventTypeArgPaperDocSlackShare                           = "paper_doc_slack_share"
	EventTypeArgPaperDocTeamInvite                           = "paper_doc_team_invite"
	EventTypeArgPaperDocTrashed                              = "paper_doc_trashed"
	EventTypeArgPaperDocUnresolveComment                     = "paper_doc_unresolve_comment"
	EventTypeArgPaperDocUntrashed                            = "paper_doc_untrashed"
	EventTypeArgPaperDocView                                 = "paper_doc_view"
	EventTypeArgPaperExternalViewAllow                       = "paper_external_view_allow"
	EventTypeArgPaperExternalViewDefaultTeam                 = "paper_external_view_default_team"
	EventTypeArgPaperExternalViewForbid                      = "paper_external_view_forbid"
	EventTypeArgPaperFolderChangeSubscription                = "paper_folder_change_subscription"
	EventTypeArgPaperFolderDeleted                           = "paper_folder_deleted"
	EventTypeArgPaperFolderFollowed                          = "paper_folder_followed"
	EventTypeArgPaperFolderTeamInvite                        = "paper_folder_team_invite"
	EventTypeArgPaperPublishedLinkChangePermission           = "paper_published_link_change_permission"
	EventTypeArgPaperPublishedLinkCreate                     = "paper_published_link_create"
	EventTypeArgPaperPublishedLinkDisabled                   = "paper_published_link_disabled"
	EventTypeArgPaperPublishedLinkView                       = "paper_published_link_view"
	EventTypeArgPasswordChange                               = "password_change"
	EventTypeArgPasswordReset                                = "password_reset"
	EventTypeArgPasswordResetAll                             = "password_reset_all"
	EventTypeArgClassificationCreateReport                   = "classification_create_report"
	EventTypeArgClassificationCreateReportFail               = "classification_create_report_fail"
	EventTypeArgEmmCreateExceptionsReport                    = "emm_create_exceptions_report"
	EventTypeArgEmmCreateUsageReport                         = "emm_create_usage_report"
	EventTypeArgExportMembersReport                          = "export_members_report"
	EventTypeArgExportMembersReportFail                      = "export_members_report_fail"
	EventTypeArgExternalSharingCreateReport                  = "external_sharing_create_report"
	EventTypeArgExternalSharingReportFailed                  = "external_sharing_report_failed"
	EventTypeArgNoExpirationLinkGenCreateReport              = "no_expiration_link_gen_create_report"
	EventTypeArgNoExpirationLinkGenReportFailed              = "no_expiration_link_gen_report_failed"
	EventTypeArgNoPasswordLinkGenCreateReport                = "no_password_link_gen_create_report"
	EventTypeArgNoPasswordLinkGenReportFailed                = "no_password_link_gen_report_failed"
	EventTypeArgNoPasswordLinkViewCreateReport               = "no_password_link_view_create_report"
	EventTypeArgNoPasswordLinkViewReportFailed               = "no_password_link_view_report_failed"
	EventTypeArgOutdatedLinkViewCreateReport                 = "outdated_link_view_create_report"
	EventTypeArgOutdatedLinkViewReportFailed                 = "outdated_link_view_report_failed"
	EventTypeArgPaperAdminExportStart                        = "paper_admin_export_start"
	EventTypeArgSmartSyncCreateAdminPrivilegeReport          = "smart_sync_create_admin_privilege_report"
	EventTypeArgTeamActivityCreateReport                     = "team_activity_create_report"
	EventTypeArgTeamActivityCreateReportFail                 = "team_activity_create_report_fail"
	EventTypeArgCollectionShare                              = "collection_share"
	EventTypeArgFileTransfersFileAdd                         = "file_transfers_file_add"
	EventTypeArgFileTransfersTransferDelete                  = "file_transfers_transfer_delete"
	EventTypeArgFileTransfersTransferDownload                = "file_transfers_transfer_download"
	EventTypeArgFileTransfersTransferSend                    = "file_transfers_transfer_send"
	EventTypeArgFileTransfersTransferView                    = "file_transfers_transfer_view"
	EventTypeArgNoteAclInviteOnly                            = "note_acl_invite_only"
	EventTypeArgNoteAclLink                                  = "note_acl_link"
	EventTypeArgNoteAclTeamLink                              = "note_acl_team_link"
	EventTypeArgNoteShared                                   = "note_shared"
	EventTypeArgNoteShareReceive                             = "note_share_receive"
	EventTypeArgOpenNoteShared                               = "open_note_shared"
	EventTypeArgSfAddGroup                                   = "sf_add_group"
	EventTypeArgSfAllowNonMembersToViewSharedLinks           = "sf_allow_non_members_to_view_shared_links"
	EventTypeArgSfExternalInviteWarn                         = "sf_external_invite_warn"
	EventTypeArgSfFbInvite                                   = "sf_fb_invite"
	EventTypeArgSfFbInviteChangeRole                         = "sf_fb_invite_change_role"
	EventTypeArgSfFbUninvite                                 = "sf_fb_uninvite"
	EventTypeArgSfInviteGroup                                = "sf_invite_group"
	EventTypeArgSfTeamGrantAccess                            = "sf_team_grant_access"
	EventTypeArgSfTeamInvite                                 = "sf_team_invite"
	EventTypeArgSfTeamInviteChangeRole                       = "sf_team_invite_change_role"
	EventTypeArgSfTeamJoin                                   = "sf_team_join"
	EventTypeArgSfTeamJoinFromOobLink                        = "sf_team_join_from_oob_link"
	EventTypeArgSfTeamUninvite                               = "sf_team_uninvite"
	EventTypeArgSharedContentAddInvitees                     = "shared_content_add_invitees"
	EventTypeArgSharedContentAddLinkExpiry                   = "shared_content_add_link_expiry"
	EventTypeArgSharedContentAddLinkPassword                 = "shared_content_add_link_password"
	EventTypeArgSharedContentAddMember                       = "shared_content_add_member"
	EventTypeArgSharedContentChangeDownloadsPolicy           = "shared_content_change_downloads_policy"
	EventTypeArgSharedContentChangeInviteeRole               = "shared_content_change_invitee_role"
	EventTypeArgSharedContentChangeLinkAudience              = "shared_content_change_link_audience"
	EventTypeArgSharedContentChangeLinkExpiry                = "shared_content_change_link_expiry"
	EventTypeArgSharedContentChangeLinkPassword              = "shared_content_change_link_password"
	EventTypeArgSharedContentChangeMemberRole                = "shared_content_change_member_role"
	EventTypeArgSharedContentChangeViewerInfoPolicy          = "shared_content_change_viewer_info_policy"
	EventTypeArgSharedContentClaimInvitation                 = "shared_content_claim_invitation"
	EventTypeArgSharedContentCopy                            = "shared_content_copy"
	EventTypeArgSharedContentDownload                        = "shared_content_download"
	EventTypeArgSharedContentRelinquishMembership            = "shared_content_relinquish_membership"
	EventTypeArgSharedContentRemoveInvitees                  = "shared_content_remove_invitees"
	EventTypeArgSharedContentRemoveLinkExpiry                = "shared_content_remove_link_expiry"
	EventTypeArgSharedContentRemoveLinkPassword              = "shared_content_remove_link_password"
	EventTypeArgSharedContentRemoveMember                    = "shared_content_remove_member"
	EventTypeArgSharedContentRequestAccess                   = "shared_content_request_access"
	EventTypeArgSharedContentRestoreInvitees                 = "shared_content_restore_invitees"
	EventTypeArgSharedContentRestoreMember                   = "shared_content_restore_member"
	EventTypeArgSharedContentUnshare                         = "shared_content_unshare"
	EventTypeArgSharedContentView                            = "shared_content_view"
	EventTypeArgSharedFolderChangeLinkPolicy                 = "shared_folder_change_link_policy"
	EventTypeArgSharedFolderChangeMembersInheritancePolicy   = "shared_folder_change_members_inheritance_policy"
	EventTypeArgSharedFolderChangeMembersManagementPolicy    = "shared_folder_change_members_management_policy"
	EventTypeArgSharedFolderChangeMembersPolicy              = "shared_folder_change_members_policy"
	EventTypeArgSharedFolderCreate                           = "shared_folder_create"
	EventTypeArgSharedFolderDeclineInvitation                = "shared_folder_decline_invitation"
	EventTypeArgSharedFolderMount                            = "shared_folder_mount"
	EventTypeArgSharedFolderNest                             = "shared_folder_nest"
	EventTypeArgSharedFolderTransferOwnership                = "shared_folder_transfer_ownership"
	EventTypeArgSharedFolderUnmount                          = "shared_folder_unmount"
	EventTypeArgSharedLinkAddExpiry                          = "shared_link_add_expiry"
	EventTypeArgSharedLinkChangeExpiry                       = "shared_link_change_expiry"
	EventTypeArgSharedLinkChangeVisibility                   = "shared_link_change_visibility"
	EventTypeArgSharedLinkCopy                               = "shared_link_copy"
	EventTypeArgSharedLinkCreate                             = "shared_link_create"
	EventTypeArgSharedLinkDisable                            = "shared_link_disable"
	EventTypeArgSharedLinkDownload                           = "shared_link_download"
	EventTypeArgSharedLinkRemoveExpiry                       = "shared_link_remove_expiry"
	EventTypeArgSharedLinkSettingsAddExpiration              = "shared_link_settings_add_expiration"
	EventTypeArgSharedLinkSettingsAddPassword                = "shared_link_settings_add_password"
	EventTypeArgSharedLinkSettingsAllowDownloadDisabled      = "shared_link_settings_allow_download_disabled"
	EventTypeArgSharedLinkSettingsAllowDownloadEnabled       = "shared_link_settings_allow_download_enabled"
	EventTypeArgSharedLinkSettingsChangeAudience             = "shared_link_settings_change_audience"
	EventTypeArgSharedLinkSettingsChangeExpiration           = "shared_link_settings_change_expiration"
	EventTypeArgSharedLinkSettingsChangePassword             = "shared_link_settings_change_password"
	EventTypeArgSharedLinkSettingsRemoveExpiration           = "shared_link_settings_remove_expiration"
	EventTypeArgSharedLinkSettingsRemovePassword             = "shared_link_settings_remove_password"
	EventTypeArgSharedLinkShare                              = "shared_link_share"
	EventTypeArgSharedLinkView                               = "shared_link_view"
	EventTypeArgSharedNoteOpened                             = "shared_note_opened"
	EventTypeArgShmodelDisableDownloads                      = "shmodel_disable_downloads"
	EventTypeArgShmodelEnableDownloads                       = "shmodel_enable_downloads"
	EventTypeArgShmodelGroupShare                            = "shmodel_group_share"
	EventTypeArgShowcaseAccessGranted                        = "showcase_access_granted"
	EventTypeArgShowcaseAddMember                            = "showcase_add_member"
	EventTypeArgShowcaseArchived                             = "showcase_archived"
	EventTypeArgShowcaseCreated                              = "showcase_created"
	EventTypeArgShowcaseDeleteComment                        = "showcase_delete_comment"
	EventTypeArgShowcaseEdited                               = "showcase_edited"
	EventTypeArgShowcaseEditComment                          = "showcase_edit_comment"
	EventTypeArgShowcaseFileAdded                            = "showcase_file_added"
	EventTypeArgShowcaseFileDownload                         = "showcase_file_download"
	EventTypeArgShowcaseFileRemoved                          = "showcase_file_removed"
	EventTypeArgShowcaseFileView                             = "showcase_file_view"
	EventTypeArgShowcasePermanentlyDeleted                   = "showcase_permanently_deleted"
	EventTypeArgShowcasePostComment                          = "showcase_post_comment"
	EventTypeArgShowcaseRemoveMember                         = "showcase_remove_member"
	EventTypeArgShowcaseRenamed                              = "showcase_renamed"
	EventTypeArgShowcaseRequestAccess                        = "showcase_request_access"
	EventTypeArgShowcaseResolveComment                       = "showcase_resolve_comment"
	EventTypeArgShowcaseRestored                             = "showcase_restored"
	EventTypeArgShowcaseTrashed                              = "showcase_trashed"
	EventTypeArgShowcaseTrashedDeprecated                    = "showcase_trashed_deprecated"
	EventTypeArgShowcaseUnresolveComment                     = "showcase_unresolve_comment"
	EventTypeArgShowcaseUntrashed                            = "showcase_untrashed"
	EventTypeArgShowcaseUntrashedDeprecated                  = "showcase_untrashed_deprecated"
	EventTypeArgShowcaseView                                 = "showcase_view"
	EventTypeArgSsoAddCert                                   = "sso_add_cert"
	EventTypeArgSsoAddLoginUrl                               = "sso_add_login_url"
	EventTypeArgSsoAddLogoutUrl                              = "sso_add_logout_url"
	EventTypeArgSsoChangeCert                                = "sso_change_cert"
	EventTypeArgSsoChangeLoginUrl                            = "sso_change_login_url"
	EventTypeArgSsoChangeLogoutUrl                           = "sso_change_logout_url"
	EventTypeArgSsoChangeSamlIdentityMode                    = "sso_change_saml_identity_mode"
	EventTypeArgSsoRemoveCert                                = "sso_remove_cert"
	EventTypeArgSsoRemoveLoginUrl                            = "sso_remove_login_url"
	EventTypeArgSsoRemoveLogoutUrl                           = "sso_remove_logout_url"
	EventTypeArgTeamFolderChangeStatus                       = "team_folder_change_status"
	EventTypeArgTeamFolderCreate                             = "team_folder_create"
	EventTypeArgTeamFolderDowngrade                          = "team_folder_downgrade"
	EventTypeArgTeamFolderPermanentlyDelete                  = "team_folder_permanently_delete"
	EventTypeArgTeamFolderRename                             = "team_folder_rename"
	EventTypeArgTeamSelectiveSyncSettingsChanged             = "team_selective_sync_settings_changed"
	EventTypeArgAccountCaptureChangePolicy                   = "account_capture_change_policy"
	EventTypeArgAdminEmailRemindersChanged                   = "admin_email_reminders_changed"
	EventTypeArgAllowDownloadDisabled                        = "allow_download_disabled"
	EventTypeArgAllowDownloadEnabled                         = "allow_download_enabled"
	EventTypeArgAppPermissionsChanged                        = "app_permissions_changed"
	EventTypeArgCameraUploadsPolicyChanged                   = "camera_uploads_policy_changed"
	EventTypeArgCaptureTranscriptPolicyChanged               = "capture_transcript_policy_changed"
	EventTypeArgClassificationChangePolicy                   = "classification_change_policy"
	EventTypeArgComputerBackupPolicyChanged                  = "computer_backup_policy_changed"
	EventTypeArgContentAdministrationPolicyChanged           = "content_administration_policy_changed"
	EventTypeArgDataPlacementRestrictionChangePolicy         = "data_placement_restriction_change_policy"
	EventTypeArgDataPlacementRestrictionSatisfyPolicy        = "data_placement_restriction_satisfy_policy"
	EventTypeArgDeviceApprovalsAddException                  = "device_approvals_add_exception"
	EventTypeArgDeviceApprovalsChangeDesktopPolicy           = "device_approvals_change_desktop_policy"
	EventTypeArgDeviceApprovalsChangeMobilePolicy            = "device_approvals_change_mobile_policy"
	EventTypeArgDeviceApprovalsChangeOverageAction           = "device_approvals_change_overage_action"
	EventTypeArgDeviceApprovalsChangeUnlinkAction            = "device_approvals_change_unlink_action"
	EventTypeArgDeviceApprovalsRemoveException               = "device_approvals_remove_exception"
	EventTypeArgDirectoryRestrictionsAddMembers              = "directory_restrictions_add_members"
	EventTypeArgDirectoryRestrictionsRemoveMembers           = "directory_restrictions_remove_members"
	EventTypeArgDropboxPasswordsPolicyChanged                = "dropbox_passwords_policy_changed"
	EventTypeArgEmailIngestPolicyChanged                     = "email_ingest_policy_changed"
	EventTypeArgEmmAddException                              = "emm_add_exception"
	EventTypeArgEmmChangePolicy                              = "emm_change_policy"
	EventTypeArgEmmRemoveException                           = "emm_remove_exception"
	EventTypeArgExtendedVersionHistoryChangePolicy           = "extended_version_history_change_policy"
	EventTypeArgExternalDriveBackupPolicyChanged             = "external_drive_backup_policy_changed"
	EventTypeArgFileCommentsChangePolicy                     = "file_comments_change_policy"
	EventTypeArgFileLockingPolicyChanged                     = "file_locking_policy_changed"
	EventTypeArgFileProviderMigrationPolicyChanged           = "file_provider_migration_policy_changed"
	EventTypeArgFileRequestsChangePolicy                     = "file_requests_change_policy"
	EventTypeArgFileRequestsEmailsEnabled                    = "file_requests_emails_enabled"
	EventTypeArgFileRequestsEmailsRestrictedToTeamOnly       = "file_requests_emails_restricted_to_team_only"
	EventTypeArgFileTransfersPolicyChanged                   = "file_transfers_policy_changed"
	EventTypeArgGoogleSsoChangePolicy                        = "google_sso_change_policy"
	EventTypeArgGroupUserManagementChangePolicy              = "group_user_management_change_policy"
	EventTypeArgIntegrationPolicyChanged                     = "integration_policy_changed"
	EventTypeArgInviteAcceptanceEmailPolicyChanged           = "invite_acceptance_email_policy_changed"
	EventTypeArgMemberRequestsChangePolicy                   = "member_requests_change_policy"
	EventTypeArgMemberSendInvitePolicyChanged                = "member_send_invite_policy_changed"
	EventTypeArgMemberSpaceLimitsAddException                = "member_space_limits_add_exception"
	EventTypeArgMemberSpaceLimitsChangeCapsTypePolicy        = "member_space_limits_change_caps_type_policy"
	EventTypeArgMemberSpaceLimitsChangePolicy                = "member_space_limits_change_policy"
	EventTypeArgMemberSpaceLimitsRemoveException             = "member_space_limits_remove_exception"
	EventTypeArgMemberSuggestionsChangePolicy                = "member_suggestions_change_policy"
	EventTypeArgMicrosoftOfficeAddinChangePolicy             = "microsoft_office_addin_change_policy"
	EventTypeArgNetworkControlChangePolicy                   = "network_control_change_policy"
	EventTypeArgPaperChangeDeploymentPolicy                  = "paper_change_deployment_policy"
	EventTypeArgPaperChangeMemberLinkPolicy                  = "paper_change_member_link_policy"
	EventTypeArgPaperChangeMemberPolicy                      = "paper_change_member_policy"
	EventTypeArgPaperChangePolicy                            = "paper_change_policy"
	EventTypeArgPaperDefaultFolderPolicyChanged              = "paper_default_folder_policy_changed"
	EventTypeArgPaperDesktopPolicyChanged                    = "paper_desktop_policy_changed"
	EventTypeArgPaperEnabledUsersGroupAddition               = "paper_enabled_users_group_addition"
	EventTypeArgPaperEnabledUsersGroupRemoval                = "paper_enabled_users_group_removal"
	EventTypeArgPasswordStrengthRequirementsChangePolicy     = "password_strength_requirements_change_policy"
	EventTypeArgPermanentDeleteChangePolicy                  = "permanent_delete_change_policy"
	EventTypeArgResellerSupportChangePolicy                  = "reseller_support_change_policy"
	EventTypeArgRewindPolicyChanged                          = "rewind_policy_changed"
	EventTypeArgSendForSignaturePolicyChanged                = "send_for_signature_policy_changed"
	EventTypeArgSharingChangeFolderJoinPolicy                = "sharing_change_folder_join_policy"
	EventTypeArgSharingChangeLinkAllowChangeExpirationPolicy = "sharing_change_link_allow_change_expiration_policy"
	EventTypeArgSharingChangeLinkDefaultExpirationPolicy     = "sharing_change_link_default_expiration_policy"
	EventTypeArgSharingChangeLinkEnforcePasswordPolicy       = "sharing_change_link_enforce_password_policy"
	EventTypeArgSharingChangeLinkPolicy                      = "sharing_change_link_policy"
	EventTypeArgSharingChangeMemberPolicy                    = "sharing_change_member_policy"
	EventTypeArgShowcaseChangeDownloadPolicy                 = "showcase_change_download_policy"
	EventTypeArgShowcaseChangeEnabledPolicy                  = "showcase_change_enabled_policy"
	EventTypeArgShowcaseChangeExternalSharingPolicy          = "showcase_change_external_sharing_policy"
	EventTypeArgSmarterSmartSyncPolicyChanged                = "smarter_smart_sync_policy_changed"
	EventTypeArgSmartSyncChangePolicy                        = "smart_sync_change_policy"
	EventTypeArgSmartSyncNotOptOut                           = "smart_sync_not_opt_out"
	EventTypeArgSmartSyncOptOut                              = "smart_sync_opt_out"
	EventTypeArgSsoChangePolicy                              = "sso_change_policy"
	EventTypeArgTeamBrandingPolicyChanged                    = "team_branding_policy_changed"
	EventTypeArgTeamExtensionsPolicyChanged                  = "team_extensions_policy_changed"
	EventTypeArgTeamSelectiveSyncPolicyChanged               = "team_selective_sync_policy_changed"
	EventTypeArgTeamSharingWhitelistSubjectsChanged          = "team_sharing_whitelist_subjects_changed"
	EventTypeArgTfaAddException                              = "tfa_add_exception"
	EventTypeArgTfaChangePolicy                              = "tfa_change_policy"
	EventTypeArgTfaRemoveException                           = "tfa_remove_exception"
	EventTypeArgTwoAccountChangePolicy                       = "two_account_change_policy"
	EventTypeArgViewerInfoPolicyChanged                      = "viewer_info_policy_changed"
	EventTypeArgWatermarkingPolicyChanged                    = "watermarking_policy_changed"
	EventTypeArgWebSessionsChangeActiveSessionLimit          = "web_sessions_change_active_session_limit"
	EventTypeArgWebSessionsChangeFixedLengthPolicy           = "web_sessions_change_fixed_length_policy"
	EventTypeArgWebSessionsChangeIdleLengthPolicy            = "web_sessions_change_idle_length_policy"
	EventTypeArgDataResidencyMigrationRequestSuccessful      = "data_residency_migration_request_successful"
	EventTypeArgDataResidencyMigrationRequestUnsuccessful    = "data_residency_migration_request_unsuccessful"
	EventTypeArgTeamMergeFrom                                = "team_merge_from"
	EventTypeArgTeamMergeTo                                  = "team_merge_to"
	EventTypeArgTeamProfileAddBackground                     = "team_profile_add_background"
	EventTypeArgTeamProfileAddLogo                           = "team_profile_add_logo"
	EventTypeArgTeamProfileChangeBackground                  = "team_profile_change_background"
	EventTypeArgTeamProfileChangeDefaultLanguage             = "team_profile_change_default_language"
	EventTypeArgTeamProfileChangeLogo                        = "team_profile_change_logo"
	EventTypeArgTeamProfileChangeName                        = "team_profile_change_name"
	EventTypeArgTeamProfileRemoveBackground                  = "team_profile_remove_background"
	EventTypeArgTeamProfileRemoveLogo                        = "team_profile_remove_logo"
	EventTypeArgTfaAddBackupPhone                            = "tfa_add_backup_phone"
	EventTypeArgTfaAddSecurityKey                            = "tfa_add_security_key"
	EventTypeArgTfaChangeBackupPhone                         = "tfa_change_backup_phone"
	EventTypeArgTfaChangeStatus                              = "tfa_change_status"
	EventTypeArgTfaRemoveBackupPhone                         = "tfa_remove_backup_phone"
	EventTypeArgTfaRemoveSecurityKey                         = "tfa_remove_security_key"
	EventTypeArgTfaReset                                     = "tfa_reset"
	EventTypeArgChangedEnterpriseAdminRole                   = "changed_enterprise_admin_role"
	EventTypeArgChangedEnterpriseConnectedTeamStatus         = "changed_enterprise_connected_team_status"
	EventTypeArgEndedEnterpriseAdminSession                  = "ended_enterprise_admin_session"
	EventTypeArgEndedEnterpriseAdminSessionDeprecated        = "ended_enterprise_admin_session_deprecated"
	EventTypeArgEnterpriseSettingsLocking                    = "enterprise_settings_locking"
	EventTypeArgGuestAdminChangeStatus                       = "guest_admin_change_status"
	EventTypeArgStartedEnterpriseAdminSession                = "started_enterprise_admin_session"
	EventTypeArgTeamMergeRequestAccepted                     = "team_merge_request_accepted"
	EventTypeArgTeamMergeRequestAcceptedShownToPrimaryTeam   = "team_merge_request_accepted_shown_to_primary_team"
	EventTypeArgTeamMergeRequestAcceptedShownToSecondaryTeam = "team_merge_request_accepted_shown_to_secondary_team"
	EventTypeArgTeamMergeRequestAutoCanceled                 = "team_merge_request_auto_canceled"
	EventTypeArgTeamMergeRequestCanceled                     = "team_merge_request_canceled"
	EventTypeArgTeamMergeRequestCanceledShownToPrimaryTeam   = "team_merge_request_canceled_shown_to_primary_team"
	EventTypeArgTeamMergeRequestCanceledShownToSecondaryTeam = "team_merge_request_canceled_shown_to_secondary_team"
	EventTypeArgTeamMergeRequestExpired                      = "team_merge_request_expired"
	EventTypeArgTeamMergeRequestExpiredShownToPrimaryTeam    = "team_merge_request_expired_shown_to_primary_team"
	EventTypeArgTeamMergeRequestExpiredShownToSecondaryTeam  = "team_merge_request_expired_shown_to_secondary_team"
	EventTypeArgTeamMergeRequestRejectedShownToPrimaryTeam   = "team_merge_request_rejected_shown_to_primary_team"
	EventTypeArgTeamMergeRequestRejectedShownToSecondaryTeam = "team_merge_request_rejected_shown_to_secondary_team"
	EventTypeArgTeamMergeRequestReminder                     = "team_merge_request_reminder"
	EventTypeArgTeamMergeRequestReminderShownToPrimaryTeam   = "team_merge_request_reminder_shown_to_primary_team"
	EventTypeArgTeamMergeRequestReminderShownToSecondaryTeam = "team_merge_request_reminder_shown_to_secondary_team"
	EventTypeArgTeamMergeRequestRevoked                      = "team_merge_request_revoked"
	EventTypeArgTeamMergeRequestSentShownToPrimaryTeam       = "team_merge_request_sent_shown_to_primary_team"
	EventTypeArgTeamMergeRequestSentShownToSecondaryTeam     = "team_merge_request_sent_shown_to_secondary_team"
	EventTypeArgOther                                        = "other"
)

// ExportMembersReportDetails : Created member data report.
type ExportMembersReportDetails struct {
}

// NewExportMembersReportDetails returns a new ExportMembersReportDetails instance
func NewExportMembersReportDetails() *ExportMembersReportDetails {
	s := new(ExportMembersReportDetails)
	return s
}

// ExportMembersReportFailDetails : Failed to create members data report.
type ExportMembersReportFailDetails struct {
	// FailureReason : Failure reason.
	FailureReason *team.TeamReportFailureReason `json:"failure_reason"`
}

// NewExportMembersReportFailDetails returns a new ExportMembersReportFailDetails instance
func NewExportMembersReportFailDetails(FailureReason *team.TeamReportFailureReason) *ExportMembersReportFailDetails {
	s := new(ExportMembersReportFailDetails)
	s.FailureReason = FailureReason
	return s
}

// ExportMembersReportFailType : has no documentation (yet)
type ExportMembersReportFailType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewExportMembersReportFailType returns a new ExportMembersReportFailType instance
func NewExportMembersReportFailType(Description string) *ExportMembersReportFailType {
	s := new(ExportMembersReportFailType)
	s.Description = Description
	return s
}

// ExportMembersReportType : has no documentation (yet)
type ExportMembersReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewExportMembersReportType returns a new ExportMembersReportType instance
func NewExportMembersReportType(Description string) *ExportMembersReportType {
	s := new(ExportMembersReportType)
	s.Description = Description
	return s
}

// ExtendedVersionHistoryChangePolicyDetails : Accepted/opted out of extended
// version history.
type ExtendedVersionHistoryChangePolicyDetails struct {
	// NewValue : New extended version history policy.
	NewValue *ExtendedVersionHistoryPolicy `json:"new_value"`
	// PreviousValue : Previous extended version history policy. Might be
	// missing due to historical data gap.
	PreviousValue *ExtendedVersionHistoryPolicy `json:"previous_value,omitempty"`
}

// NewExtendedVersionHistoryChangePolicyDetails returns a new ExtendedVersionHistoryChangePolicyDetails instance
func NewExtendedVersionHistoryChangePolicyDetails(NewValue *ExtendedVersionHistoryPolicy) *ExtendedVersionHistoryChangePolicyDetails {
	s := new(ExtendedVersionHistoryChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// ExtendedVersionHistoryChangePolicyType : has no documentation (yet)
type ExtendedVersionHistoryChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewExtendedVersionHistoryChangePolicyType returns a new ExtendedVersionHistoryChangePolicyType instance
func NewExtendedVersionHistoryChangePolicyType(Description string) *ExtendedVersionHistoryChangePolicyType {
	s := new(ExtendedVersionHistoryChangePolicyType)
	s.Description = Description
	return s
}

// ExtendedVersionHistoryPolicy : has no documentation (yet)
type ExtendedVersionHistoryPolicy struct {
	dropbox.Tagged
}

// Valid tag values for ExtendedVersionHistoryPolicy
const (
	ExtendedVersionHistoryPolicyExplicitlyLimited   = "explicitly_limited"
	ExtendedVersionHistoryPolicyExplicitlyUnlimited = "explicitly_unlimited"
	ExtendedVersionHistoryPolicyImplicitlyLimited   = "implicitly_limited"
	ExtendedVersionHistoryPolicyImplicitlyUnlimited = "implicitly_unlimited"
	ExtendedVersionHistoryPolicyOther               = "other"
)

// ExternalDriveBackupEligibilityStatus : External Drive Backup eligibility
// status
type ExternalDriveBackupEligibilityStatus struct {
	dropbox.Tagged
}

// Valid tag values for ExternalDriveBackupEligibilityStatus
const (
	ExternalDriveBackupEligibilityStatusExceedLicenseCap = "exceed_license_cap"
	ExternalDriveBackupEligibilityStatusSuccess          = "success"
	ExternalDriveBackupEligibilityStatusOther            = "other"
)

// ExternalDriveBackupEligibilityStatusCheckedDetails : Checked external drive
// backup eligibility status.
type ExternalDriveBackupEligibilityStatusCheckedDetails struct {
	// DesktopDeviceSessionInfo : Device's session logged information.
	DesktopDeviceSessionInfo *DesktopDeviceSessionLogInfo `json:"desktop_device_session_info"`
	// Status : Current eligibility status of external drive backup.
	Status *ExternalDriveBackupEligibilityStatus `json:"status"`
	// NumberOfExternalDriveBackup : Total number of valid external drive backup
	// for all the team members.
	NumberOfExternalDriveBackup uint64 `json:"number_of_external_drive_backup"`
}

// NewExternalDriveBackupEligibilityStatusCheckedDetails returns a new ExternalDriveBackupEligibilityStatusCheckedDetails instance
func NewExternalDriveBackupEligibilityStatusCheckedDetails(DesktopDeviceSessionInfo *DesktopDeviceSessionLogInfo, Status *ExternalDriveBackupEligibilityStatus, NumberOfExternalDriveBackup uint64) *ExternalDriveBackupEligibilityStatusCheckedDetails {
	s := new(ExternalDriveBackupEligibilityStatusCheckedDetails)
	s.DesktopDeviceSessionInfo = DesktopDeviceSessionInfo
	s.Status = Status
	s.NumberOfExternalDriveBackup = NumberOfExternalDriveBackup
	return s
}

// ExternalDriveBackupEligibilityStatusCheckedType : has no documentation (yet)
type ExternalDriveBackupEligibilityStatusCheckedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewExternalDriveBackupEligibilityStatusCheckedType returns a new ExternalDriveBackupEligibilityStatusCheckedType instance
func NewExternalDriveBackupEligibilityStatusCheckedType(Description string) *ExternalDriveBackupEligibilityStatusCheckedType {
	s := new(ExternalDriveBackupEligibilityStatusCheckedType)
	s.Description = Description
	return s
}

// ExternalDriveBackupPolicy : Policy for controlling team access to external
// drive backup feature
type ExternalDriveBackupPolicy struct {
	dropbox.Tagged
}

// Valid tag values for ExternalDriveBackupPolicy
const (
	ExternalDriveBackupPolicyDefault  = "default"
	ExternalDriveBackupPolicyDisabled = "disabled"
	ExternalDriveBackupPolicyEnabled  = "enabled"
	ExternalDriveBackupPolicyOther    = "other"
)

// ExternalDriveBackupPolicyChangedDetails : Changed external drive backup
// policy for team.
type ExternalDriveBackupPolicyChangedDetails struct {
	// NewValue : New external drive backup policy.
	NewValue *ExternalDriveBackupPolicy `json:"new_value"`
	// PreviousValue : Previous external drive backup policy.
	PreviousValue *ExternalDriveBackupPolicy `json:"previous_value"`
}

// NewExternalDriveBackupPolicyChangedDetails returns a new ExternalDriveBackupPolicyChangedDetails instance
func NewExternalDriveBackupPolicyChangedDetails(NewValue *ExternalDriveBackupPolicy, PreviousValue *ExternalDriveBackupPolicy) *ExternalDriveBackupPolicyChangedDetails {
	s := new(ExternalDriveBackupPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// ExternalDriveBackupPolicyChangedType : has no documentation (yet)
type ExternalDriveBackupPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewExternalDriveBackupPolicyChangedType returns a new ExternalDriveBackupPolicyChangedType instance
func NewExternalDriveBackupPolicyChangedType(Description string) *ExternalDriveBackupPolicyChangedType {
	s := new(ExternalDriveBackupPolicyChangedType)
	s.Description = Description
	return s
}

// ExternalDriveBackupStatus : External Drive Backup status
type ExternalDriveBackupStatus struct {
	dropbox.Tagged
}

// Valid tag values for ExternalDriveBackupStatus
const (
	ExternalDriveBackupStatusBroken          = "broken"
	ExternalDriveBackupStatusCreated         = "created"
	ExternalDriveBackupStatusCreatedOrBroken = "created_or_broken"
	ExternalDriveBackupStatusDeleted         = "deleted"
	ExternalDriveBackupStatusEmpty           = "empty"
	ExternalDriveBackupStatusUnknown         = "unknown"
	ExternalDriveBackupStatusOther           = "other"
)

// ExternalDriveBackupStatusChangedDetails : Modified external drive backup.
type ExternalDriveBackupStatusChangedDetails struct {
	// DesktopDeviceSessionInfo : Device's session logged information.
	DesktopDeviceSessionInfo *DesktopDeviceSessionLogInfo `json:"desktop_device_session_info"`
	// PreviousValue : Previous status of this external drive backup.
	PreviousValue *ExternalDriveBackupStatus `json:"previous_value"`
	// NewValue : Next status of this external drive backup.
	NewValue *ExternalDriveBackupStatus `json:"new_value"`
}

// NewExternalDriveBackupStatusChangedDetails returns a new ExternalDriveBackupStatusChangedDetails instance
func NewExternalDriveBackupStatusChangedDetails(DesktopDeviceSessionInfo *DesktopDeviceSessionLogInfo, PreviousValue *ExternalDriveBackupStatus, NewValue *ExternalDriveBackupStatus) *ExternalDriveBackupStatusChangedDetails {
	s := new(ExternalDriveBackupStatusChangedDetails)
	s.DesktopDeviceSessionInfo = DesktopDeviceSessionInfo
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// ExternalDriveBackupStatusChangedType : has no documentation (yet)
type ExternalDriveBackupStatusChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewExternalDriveBackupStatusChangedType returns a new ExternalDriveBackupStatusChangedType instance
func NewExternalDriveBackupStatusChangedType(Description string) *ExternalDriveBackupStatusChangedType {
	s := new(ExternalDriveBackupStatusChangedType)
	s.Description = Description
	return s
}

// ExternalSharingCreateReportDetails : Created External sharing report.
type ExternalSharingCreateReportDetails struct {
}

// NewExternalSharingCreateReportDetails returns a new ExternalSharingCreateReportDetails instance
func NewExternalSharingCreateReportDetails() *ExternalSharingCreateReportDetails {
	s := new(ExternalSharingCreateReportDetails)
	return s
}

// ExternalSharingCreateReportType : has no documentation (yet)
type ExternalSharingCreateReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewExternalSharingCreateReportType returns a new ExternalSharingCreateReportType instance
func NewExternalSharingCreateReportType(Description string) *ExternalSharingCreateReportType {
	s := new(ExternalSharingCreateReportType)
	s.Description = Description
	return s
}

// ExternalSharingReportFailedDetails : Couldn't create External sharing report.
type ExternalSharingReportFailedDetails struct {
	// FailureReason : Failure reason.
	FailureReason *team.TeamReportFailureReason `json:"failure_reason"`
}

// NewExternalSharingReportFailedDetails returns a new ExternalSharingReportFailedDetails instance
func NewExternalSharingReportFailedDetails(FailureReason *team.TeamReportFailureReason) *ExternalSharingReportFailedDetails {
	s := new(ExternalSharingReportFailedDetails)
	s.FailureReason = FailureReason
	return s
}

// ExternalSharingReportFailedType : has no documentation (yet)
type ExternalSharingReportFailedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewExternalSharingReportFailedType returns a new ExternalSharingReportFailedType instance
func NewExternalSharingReportFailedType(Description string) *ExternalSharingReportFailedType {
	s := new(ExternalSharingReportFailedType)
	s.Description = Description
	return s
}

// ExternalUserLogInfo : A user without a Dropbox account.
type ExternalUserLogInfo struct {
	// UserIdentifier : An external user identifier.
	UserIdentifier string `json:"user_identifier"`
	// IdentifierType : Identifier type.
	IdentifierType *IdentifierType `json:"identifier_type"`
}

// NewExternalUserLogInfo returns a new ExternalUserLogInfo instance
func NewExternalUserLogInfo(UserIdentifier string, IdentifierType *IdentifierType) *ExternalUserLogInfo {
	s := new(ExternalUserLogInfo)
	s.UserIdentifier = UserIdentifier
	s.IdentifierType = IdentifierType
	return s
}

// FailureDetailsLogInfo : Provides details about a failure
type FailureDetailsLogInfo struct {
	// UserFriendlyMessage : A user friendly explanation of the error.
	UserFriendlyMessage string `json:"user_friendly_message,omitempty"`
	// TechnicalErrorMessage : A technical explanation of the error. This is
	// relevant for some errors.
	TechnicalErrorMessage string `json:"technical_error_message,omitempty"`
}

// NewFailureDetailsLogInfo returns a new FailureDetailsLogInfo instance
func NewFailureDetailsLogInfo() *FailureDetailsLogInfo {
	s := new(FailureDetailsLogInfo)
	return s
}

// FedAdminRole : has no documentation (yet)
type FedAdminRole struct {
	dropbox.Tagged
}

// Valid tag values for FedAdminRole
const (
	FedAdminRoleEnterpriseAdmin    = "enterprise_admin"
	FedAdminRoleNotEnterpriseAdmin = "not_enterprise_admin"
	FedAdminRoleOther              = "other"
)

// FedExtraDetails : More details about the organization or team.
type FedExtraDetails struct {
	dropbox.Tagged
	// Organization : More details about the organization.
	Organization *OrganizationDetails `json:"organization,omitempty"`
	// Team : More details about the team.
	Team *TeamDetails `json:"team,omitempty"`
}

// Valid tag values for FedExtraDetails
const (
	FedExtraDetailsOrganization = "organization"
	FedExtraDetailsTeam         = "team"
	FedExtraDetailsOther        = "other"
)

// UnmarshalJSON deserializes into a FedExtraDetails instance
func (u *FedExtraDetails) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "organization":
		if err = json.Unmarshal(body, &u.Organization); err != nil {
			return err
		}

	case "team":
		if err = json.Unmarshal(body, &u.Team); err != nil {
			return err
		}

	}
	return nil
}

// FedHandshakeAction : has no documentation (yet)
type FedHandshakeAction struct {
	dropbox.Tagged
}

// Valid tag values for FedHandshakeAction
const (
	FedHandshakeActionAcceptedInvite = "accepted_invite"
	FedHandshakeActionCanceledInvite = "canceled_invite"
	FedHandshakeActionInviteExpired  = "invite_expired"
	FedHandshakeActionInvited        = "invited"
	FedHandshakeActionRejectedInvite = "rejected_invite"
	FedHandshakeActionRemovedTeam    = "removed_team"
	FedHandshakeActionOther          = "other"
)

// FederationStatusChangeAdditionalInfo : Additional information about the
// organization or connected team
type FederationStatusChangeAdditionalInfo struct {
	dropbox.Tagged
	// ConnectedTeamName : The name of the team.
	ConnectedTeamName *ConnectedTeamName `json:"connected_team_name,omitempty"`
	// NonTrustedTeamDetails : The email to which the request was sent.
	NonTrustedTeamDetails *NonTrustedTeamDetails `json:"non_trusted_team_details,omitempty"`
	// OrganizationName : The name of the organization.
	OrganizationName *OrganizationName `json:"organization_name,omitempty"`
}

// Valid tag values for FederationStatusChangeAdditionalInfo
const (
	FederationStatusChangeAdditionalInfoConnectedTeamName     = "connected_team_name"
	FederationStatusChangeAdditionalInfoNonTrustedTeamDetails = "non_trusted_team_details"
	FederationStatusChangeAdditionalInfoOrganizationName      = "organization_name"
	FederationStatusChangeAdditionalInfoOther                 = "other"
)

// UnmarshalJSON deserializes into a FederationStatusChangeAdditionalInfo instance
func (u *FederationStatusChangeAdditionalInfo) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "connected_team_name":
		if err = json.Unmarshal(body, &u.ConnectedTeamName); err != nil {
			return err
		}

	case "non_trusted_team_details":
		if err = json.Unmarshal(body, &u.NonTrustedTeamDetails); err != nil {
			return err
		}

	case "organization_name":
		if err = json.Unmarshal(body, &u.OrganizationName); err != nil {
			return err
		}

	}
	return nil
}

// FileAddCommentDetails : Added file comment.
type FileAddCommentDetails struct {
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewFileAddCommentDetails returns a new FileAddCommentDetails instance
func NewFileAddCommentDetails() *FileAddCommentDetails {
	s := new(FileAddCommentDetails)
	return s
}

// FileAddCommentType : has no documentation (yet)
type FileAddCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileAddCommentType returns a new FileAddCommentType instance
func NewFileAddCommentType(Description string) *FileAddCommentType {
	s := new(FileAddCommentType)
	s.Description = Description
	return s
}

// FileAddDetails : Added files and/or folders.
type FileAddDetails struct {
}

// NewFileAddDetails returns a new FileAddDetails instance
func NewFileAddDetails() *FileAddDetails {
	s := new(FileAddDetails)
	return s
}

// FileAddType : has no documentation (yet)
type FileAddType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileAddType returns a new FileAddType instance
func NewFileAddType(Description string) *FileAddType {
	s := new(FileAddType)
	s.Description = Description
	return s
}

// FileChangeCommentSubscriptionDetails : Subscribed to or unsubscribed from
// comment notifications for file.
type FileChangeCommentSubscriptionDetails struct {
	// NewValue : New file comment subscription.
	NewValue *FileCommentNotificationPolicy `json:"new_value"`
	// PreviousValue : Previous file comment subscription. Might be missing due
	// to historical data gap.
	PreviousValue *FileCommentNotificationPolicy `json:"previous_value,omitempty"`
}

// NewFileChangeCommentSubscriptionDetails returns a new FileChangeCommentSubscriptionDetails instance
func NewFileChangeCommentSubscriptionDetails(NewValue *FileCommentNotificationPolicy) *FileChangeCommentSubscriptionDetails {
	s := new(FileChangeCommentSubscriptionDetails)
	s.NewValue = NewValue
	return s
}

// FileChangeCommentSubscriptionType : has no documentation (yet)
type FileChangeCommentSubscriptionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileChangeCommentSubscriptionType returns a new FileChangeCommentSubscriptionType instance
func NewFileChangeCommentSubscriptionType(Description string) *FileChangeCommentSubscriptionType {
	s := new(FileChangeCommentSubscriptionType)
	s.Description = Description
	return s
}

// FileCommentNotificationPolicy : Enable or disable file comments notifications
type FileCommentNotificationPolicy struct {
	dropbox.Tagged
}

// Valid tag values for FileCommentNotificationPolicy
const (
	FileCommentNotificationPolicyDisabled = "disabled"
	FileCommentNotificationPolicyEnabled  = "enabled"
	FileCommentNotificationPolicyOther    = "other"
)

// FileCommentsChangePolicyDetails : Enabled/disabled commenting on team files.
type FileCommentsChangePolicyDetails struct {
	// NewValue : New commenting on team files policy.
	NewValue *FileCommentsPolicy `json:"new_value"`
	// PreviousValue : Previous commenting on team files policy. Might be
	// missing due to historical data gap.
	PreviousValue *FileCommentsPolicy `json:"previous_value,omitempty"`
}

// NewFileCommentsChangePolicyDetails returns a new FileCommentsChangePolicyDetails instance
func NewFileCommentsChangePolicyDetails(NewValue *FileCommentsPolicy) *FileCommentsChangePolicyDetails {
	s := new(FileCommentsChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// FileCommentsChangePolicyType : has no documentation (yet)
type FileCommentsChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileCommentsChangePolicyType returns a new FileCommentsChangePolicyType instance
func NewFileCommentsChangePolicyType(Description string) *FileCommentsChangePolicyType {
	s := new(FileCommentsChangePolicyType)
	s.Description = Description
	return s
}

// FileCommentsPolicy : File comments policy
type FileCommentsPolicy struct {
	dropbox.Tagged
}

// Valid tag values for FileCommentsPolicy
const (
	FileCommentsPolicyDisabled = "disabled"
	FileCommentsPolicyEnabled  = "enabled"
	FileCommentsPolicyOther    = "other"
)

// FileCopyDetails : Copied files and/or folders.
type FileCopyDetails struct {
	// RelocateActionDetails : Relocate action details.
	RelocateActionDetails []*RelocateAssetReferencesLogInfo `json:"relocate_action_details"`
}

// NewFileCopyDetails returns a new FileCopyDetails instance
func NewFileCopyDetails(RelocateActionDetails []*RelocateAssetReferencesLogInfo) *FileCopyDetails {
	s := new(FileCopyDetails)
	s.RelocateActionDetails = RelocateActionDetails
	return s
}

// FileCopyType : has no documentation (yet)
type FileCopyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileCopyType returns a new FileCopyType instance
func NewFileCopyType(Description string) *FileCopyType {
	s := new(FileCopyType)
	s.Description = Description
	return s
}

// FileDeleteCommentDetails : Deleted file comment.
type FileDeleteCommentDetails struct {
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewFileDeleteCommentDetails returns a new FileDeleteCommentDetails instance
func NewFileDeleteCommentDetails() *FileDeleteCommentDetails {
	s := new(FileDeleteCommentDetails)
	return s
}

// FileDeleteCommentType : has no documentation (yet)
type FileDeleteCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileDeleteCommentType returns a new FileDeleteCommentType instance
func NewFileDeleteCommentType(Description string) *FileDeleteCommentType {
	s := new(FileDeleteCommentType)
	s.Description = Description
	return s
}

// FileDeleteDetails : Deleted files and/or folders.
type FileDeleteDetails struct {
}

// NewFileDeleteDetails returns a new FileDeleteDetails instance
func NewFileDeleteDetails() *FileDeleteDetails {
	s := new(FileDeleteDetails)
	return s
}

// FileDeleteType : has no documentation (yet)
type FileDeleteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileDeleteType returns a new FileDeleteType instance
func NewFileDeleteType(Description string) *FileDeleteType {
	s := new(FileDeleteType)
	s.Description = Description
	return s
}

// FileDownloadDetails : Downloaded files and/or folders.
type FileDownloadDetails struct {
}

// NewFileDownloadDetails returns a new FileDownloadDetails instance
func NewFileDownloadDetails() *FileDownloadDetails {
	s := new(FileDownloadDetails)
	return s
}

// FileDownloadType : has no documentation (yet)
type FileDownloadType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileDownloadType returns a new FileDownloadType instance
func NewFileDownloadType(Description string) *FileDownloadType {
	s := new(FileDownloadType)
	s.Description = Description
	return s
}

// FileEditCommentDetails : Edited file comment.
type FileEditCommentDetails struct {
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
	// PreviousCommentText : Previous comment text.
	PreviousCommentText string `json:"previous_comment_text"`
}

// NewFileEditCommentDetails returns a new FileEditCommentDetails instance
func NewFileEditCommentDetails(PreviousCommentText string) *FileEditCommentDetails {
	s := new(FileEditCommentDetails)
	s.PreviousCommentText = PreviousCommentText
	return s
}

// FileEditCommentType : has no documentation (yet)
type FileEditCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileEditCommentType returns a new FileEditCommentType instance
func NewFileEditCommentType(Description string) *FileEditCommentType {
	s := new(FileEditCommentType)
	s.Description = Description
	return s
}

// FileEditDetails : Edited files.
type FileEditDetails struct {
}

// NewFileEditDetails returns a new FileEditDetails instance
func NewFileEditDetails() *FileEditDetails {
	s := new(FileEditDetails)
	return s
}

// FileEditType : has no documentation (yet)
type FileEditType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileEditType returns a new FileEditType instance
func NewFileEditType(Description string) *FileEditType {
	s := new(FileEditType)
	s.Description = Description
	return s
}

// FileGetCopyReferenceDetails : Created copy reference to file/folder.
type FileGetCopyReferenceDetails struct {
}

// NewFileGetCopyReferenceDetails returns a new FileGetCopyReferenceDetails instance
func NewFileGetCopyReferenceDetails() *FileGetCopyReferenceDetails {
	s := new(FileGetCopyReferenceDetails)
	return s
}

// FileGetCopyReferenceType : has no documentation (yet)
type FileGetCopyReferenceType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileGetCopyReferenceType returns a new FileGetCopyReferenceType instance
func NewFileGetCopyReferenceType(Description string) *FileGetCopyReferenceType {
	s := new(FileGetCopyReferenceType)
	s.Description = Description
	return s
}

// FileLikeCommentDetails : Liked file comment.
type FileLikeCommentDetails struct {
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewFileLikeCommentDetails returns a new FileLikeCommentDetails instance
func NewFileLikeCommentDetails() *FileLikeCommentDetails {
	s := new(FileLikeCommentDetails)
	return s
}

// FileLikeCommentType : has no documentation (yet)
type FileLikeCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileLikeCommentType returns a new FileLikeCommentType instance
func NewFileLikeCommentType(Description string) *FileLikeCommentType {
	s := new(FileLikeCommentType)
	s.Description = Description
	return s
}

// FileLockingLockStatusChangedDetails : Locked/unlocked editing for a file.
type FileLockingLockStatusChangedDetails struct {
	// PreviousValue : Previous lock status of the file.
	PreviousValue *LockStatus `json:"previous_value"`
	// NewValue : New lock status of the file.
	NewValue *LockStatus `json:"new_value"`
}

// NewFileLockingLockStatusChangedDetails returns a new FileLockingLockStatusChangedDetails instance
func NewFileLockingLockStatusChangedDetails(PreviousValue *LockStatus, NewValue *LockStatus) *FileLockingLockStatusChangedDetails {
	s := new(FileLockingLockStatusChangedDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// FileLockingLockStatusChangedType : has no documentation (yet)
type FileLockingLockStatusChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileLockingLockStatusChangedType returns a new FileLockingLockStatusChangedType instance
func NewFileLockingLockStatusChangedType(Description string) *FileLockingLockStatusChangedType {
	s := new(FileLockingLockStatusChangedType)
	s.Description = Description
	return s
}

// FileLockingPolicyChangedDetails : Changed file locking policy for team.
type FileLockingPolicyChangedDetails struct {
	// NewValue : New file locking policy.
	NewValue *team_policies.FileLockingPolicyState `json:"new_value"`
	// PreviousValue : Previous file locking policy.
	PreviousValue *team_policies.FileLockingPolicyState `json:"previous_value"`
}

// NewFileLockingPolicyChangedDetails returns a new FileLockingPolicyChangedDetails instance
func NewFileLockingPolicyChangedDetails(NewValue *team_policies.FileLockingPolicyState, PreviousValue *team_policies.FileLockingPolicyState) *FileLockingPolicyChangedDetails {
	s := new(FileLockingPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// FileLockingPolicyChangedType : has no documentation (yet)
type FileLockingPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileLockingPolicyChangedType returns a new FileLockingPolicyChangedType instance
func NewFileLockingPolicyChangedType(Description string) *FileLockingPolicyChangedType {
	s := new(FileLockingPolicyChangedType)
	s.Description = Description
	return s
}

// FileOrFolderLogInfo : Generic information relevant both for files and folders
type FileOrFolderLogInfo struct {
	// Path : Path relative to event context.
	Path *PathLogInfo `json:"path"`
	// DisplayName : Display name.
	DisplayName string `json:"display_name,omitempty"`
	// FileId : Unique ID.
	FileId string `json:"file_id,omitempty"`
	// FileSize : File or folder size in bytes.
	FileSize uint64 `json:"file_size,omitempty"`
}

// NewFileOrFolderLogInfo returns a new FileOrFolderLogInfo instance
func NewFileOrFolderLogInfo(Path *PathLogInfo) *FileOrFolderLogInfo {
	s := new(FileOrFolderLogInfo)
	s.Path = Path
	return s
}

// FileLogInfo : File's logged information.
type FileLogInfo struct {
	FileOrFolderLogInfo
}

// NewFileLogInfo returns a new FileLogInfo instance
func NewFileLogInfo(Path *PathLogInfo) *FileLogInfo {
	s := new(FileLogInfo)
	s.Path = Path
	return s
}

// FileMoveDetails : Moved files and/or folders.
type FileMoveDetails struct {
	// RelocateActionDetails : Relocate action details.
	RelocateActionDetails []*RelocateAssetReferencesLogInfo `json:"relocate_action_details"`
}

// NewFileMoveDetails returns a new FileMoveDetails instance
func NewFileMoveDetails(RelocateActionDetails []*RelocateAssetReferencesLogInfo) *FileMoveDetails {
	s := new(FileMoveDetails)
	s.RelocateActionDetails = RelocateActionDetails
	return s
}

// FileMoveType : has no documentation (yet)
type FileMoveType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileMoveType returns a new FileMoveType instance
func NewFileMoveType(Description string) *FileMoveType {
	s := new(FileMoveType)
	s.Description = Description
	return s
}

// FilePermanentlyDeleteDetails : Permanently deleted files and/or folders.
type FilePermanentlyDeleteDetails struct {
}

// NewFilePermanentlyDeleteDetails returns a new FilePermanentlyDeleteDetails instance
func NewFilePermanentlyDeleteDetails() *FilePermanentlyDeleteDetails {
	s := new(FilePermanentlyDeleteDetails)
	return s
}

// FilePermanentlyDeleteType : has no documentation (yet)
type FilePermanentlyDeleteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFilePermanentlyDeleteType returns a new FilePermanentlyDeleteType instance
func NewFilePermanentlyDeleteType(Description string) *FilePermanentlyDeleteType {
	s := new(FilePermanentlyDeleteType)
	s.Description = Description
	return s
}

// FilePreviewDetails : Previewed files and/or folders.
type FilePreviewDetails struct {
}

// NewFilePreviewDetails returns a new FilePreviewDetails instance
func NewFilePreviewDetails() *FilePreviewDetails {
	s := new(FilePreviewDetails)
	return s
}

// FilePreviewType : has no documentation (yet)
type FilePreviewType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFilePreviewType returns a new FilePreviewType instance
func NewFilePreviewType(Description string) *FilePreviewType {
	s := new(FilePreviewType)
	s.Description = Description
	return s
}

// FileProviderMigrationPolicyChangedDetails : Changed File Provider Migration
// policy for team.
type FileProviderMigrationPolicyChangedDetails struct {
	// NewValue : To.
	NewValue *team_policies.FileProviderMigrationPolicyState `json:"new_value"`
	// PreviousValue : From.
	PreviousValue *team_policies.FileProviderMigrationPolicyState `json:"previous_value"`
}

// NewFileProviderMigrationPolicyChangedDetails returns a new FileProviderMigrationPolicyChangedDetails instance
func NewFileProviderMigrationPolicyChangedDetails(NewValue *team_policies.FileProviderMigrationPolicyState, PreviousValue *team_policies.FileProviderMigrationPolicyState) *FileProviderMigrationPolicyChangedDetails {
	s := new(FileProviderMigrationPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// FileProviderMigrationPolicyChangedType : has no documentation (yet)
type FileProviderMigrationPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileProviderMigrationPolicyChangedType returns a new FileProviderMigrationPolicyChangedType instance
func NewFileProviderMigrationPolicyChangedType(Description string) *FileProviderMigrationPolicyChangedType {
	s := new(FileProviderMigrationPolicyChangedType)
	s.Description = Description
	return s
}

// FileRenameDetails : Renamed files and/or folders.
type FileRenameDetails struct {
	// RelocateActionDetails : Relocate action details.
	RelocateActionDetails []*RelocateAssetReferencesLogInfo `json:"relocate_action_details"`
}

// NewFileRenameDetails returns a new FileRenameDetails instance
func NewFileRenameDetails(RelocateActionDetails []*RelocateAssetReferencesLogInfo) *FileRenameDetails {
	s := new(FileRenameDetails)
	s.RelocateActionDetails = RelocateActionDetails
	return s
}

// FileRenameType : has no documentation (yet)
type FileRenameType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRenameType returns a new FileRenameType instance
func NewFileRenameType(Description string) *FileRenameType {
	s := new(FileRenameType)
	s.Description = Description
	return s
}

// FileRequestChangeDetails : Changed file request.
type FileRequestChangeDetails struct {
	// FileRequestId : File request id. Might be missing due to historical data
	// gap.
	FileRequestId string `json:"file_request_id,omitempty"`
	// PreviousDetails : Previous file request details. Might be missing due to
	// historical data gap.
	PreviousDetails *FileRequestDetails `json:"previous_details,omitempty"`
	// NewDetails : New file request details.
	NewDetails *FileRequestDetails `json:"new_details"`
}

// NewFileRequestChangeDetails returns a new FileRequestChangeDetails instance
func NewFileRequestChangeDetails(NewDetails *FileRequestDetails) *FileRequestChangeDetails {
	s := new(FileRequestChangeDetails)
	s.NewDetails = NewDetails
	return s
}

// FileRequestChangeType : has no documentation (yet)
type FileRequestChangeType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRequestChangeType returns a new FileRequestChangeType instance
func NewFileRequestChangeType(Description string) *FileRequestChangeType {
	s := new(FileRequestChangeType)
	s.Description = Description
	return s
}

// FileRequestCloseDetails : Closed file request.
type FileRequestCloseDetails struct {
	// FileRequestId : File request id. Might be missing due to historical data
	// gap.
	FileRequestId string `json:"file_request_id,omitempty"`
	// PreviousDetails : Previous file request details. Might be missing due to
	// historical data gap.
	PreviousDetails *FileRequestDetails `json:"previous_details,omitempty"`
}

// NewFileRequestCloseDetails returns a new FileRequestCloseDetails instance
func NewFileRequestCloseDetails() *FileRequestCloseDetails {
	s := new(FileRequestCloseDetails)
	return s
}

// FileRequestCloseType : has no documentation (yet)
type FileRequestCloseType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRequestCloseType returns a new FileRequestCloseType instance
func NewFileRequestCloseType(Description string) *FileRequestCloseType {
	s := new(FileRequestCloseType)
	s.Description = Description
	return s
}

// FileRequestCreateDetails : Created file request.
type FileRequestCreateDetails struct {
	// FileRequestId : File request id. Might be missing due to historical data
	// gap.
	FileRequestId string `json:"file_request_id,omitempty"`
	// RequestDetails : File request details. Might be missing due to historical
	// data gap.
	RequestDetails *FileRequestDetails `json:"request_details,omitempty"`
}

// NewFileRequestCreateDetails returns a new FileRequestCreateDetails instance
func NewFileRequestCreateDetails() *FileRequestCreateDetails {
	s := new(FileRequestCreateDetails)
	return s
}

// FileRequestCreateType : has no documentation (yet)
type FileRequestCreateType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRequestCreateType returns a new FileRequestCreateType instance
func NewFileRequestCreateType(Description string) *FileRequestCreateType {
	s := new(FileRequestCreateType)
	s.Description = Description
	return s
}

// FileRequestDeadline : File request deadline
type FileRequestDeadline struct {
	// Deadline : The deadline for this file request. Might be missing due to
	// historical data gap.
	Deadline *time.Time `json:"deadline,omitempty"`
	// AllowLateUploads : If set, allow uploads after the deadline has passed.
	AllowLateUploads string `json:"allow_late_uploads,omitempty"`
}

// NewFileRequestDeadline returns a new FileRequestDeadline instance
func NewFileRequestDeadline() *FileRequestDeadline {
	s := new(FileRequestDeadline)
	return s
}

// FileRequestDeleteDetails : Delete file request.
type FileRequestDeleteDetails struct {
	// FileRequestId : File request id. Might be missing due to historical data
	// gap.
	FileRequestId string `json:"file_request_id,omitempty"`
	// PreviousDetails : Previous file request details. Might be missing due to
	// historical data gap.
	PreviousDetails *FileRequestDetails `json:"previous_details,omitempty"`
}

// NewFileRequestDeleteDetails returns a new FileRequestDeleteDetails instance
func NewFileRequestDeleteDetails() *FileRequestDeleteDetails {
	s := new(FileRequestDeleteDetails)
	return s
}

// FileRequestDeleteType : has no documentation (yet)
type FileRequestDeleteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRequestDeleteType returns a new FileRequestDeleteType instance
func NewFileRequestDeleteType(Description string) *FileRequestDeleteType {
	s := new(FileRequestDeleteType)
	s.Description = Description
	return s
}

// FileRequestDetails : File request details
type FileRequestDetails struct {
	// AssetIndex : Asset position in the Assets list.
	AssetIndex uint64 `json:"asset_index"`
	// Deadline : File request deadline.
	Deadline *FileRequestDeadline `json:"deadline,omitempty"`
}

// NewFileRequestDetails returns a new FileRequestDetails instance
func NewFileRequestDetails(AssetIndex uint64) *FileRequestDetails {
	s := new(FileRequestDetails)
	s.AssetIndex = AssetIndex
	return s
}

// FileRequestReceiveFileDetails : Received files for file request.
type FileRequestReceiveFileDetails struct {
	// FileRequestId : File request id. Might be missing due to historical data
	// gap.
	FileRequestId string `json:"file_request_id,omitempty"`
	// FileRequestDetails : File request details. Might be missing due to
	// historical data gap.
	FileRequestDetails *FileRequestDetails `json:"file_request_details,omitempty"`
	// SubmittedFileNames : Submitted file names.
	SubmittedFileNames []string `json:"submitted_file_names"`
	// SubmitterName : The name as provided by the submitter.
	SubmitterName string `json:"submitter_name,omitempty"`
	// SubmitterEmail : The email as provided by the submitter.
	SubmitterEmail string `json:"submitter_email,omitempty"`
}

// NewFileRequestReceiveFileDetails returns a new FileRequestReceiveFileDetails instance
func NewFileRequestReceiveFileDetails(SubmittedFileNames []string) *FileRequestReceiveFileDetails {
	s := new(FileRequestReceiveFileDetails)
	s.SubmittedFileNames = SubmittedFileNames
	return s
}

// FileRequestReceiveFileType : has no documentation (yet)
type FileRequestReceiveFileType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRequestReceiveFileType returns a new FileRequestReceiveFileType instance
func NewFileRequestReceiveFileType(Description string) *FileRequestReceiveFileType {
	s := new(FileRequestReceiveFileType)
	s.Description = Description
	return s
}

// FileRequestsChangePolicyDetails : Enabled/disabled file requests.
type FileRequestsChangePolicyDetails struct {
	// NewValue : New file requests policy.
	NewValue *FileRequestsPolicy `json:"new_value"`
	// PreviousValue : Previous file requests policy. Might be missing due to
	// historical data gap.
	PreviousValue *FileRequestsPolicy `json:"previous_value,omitempty"`
}

// NewFileRequestsChangePolicyDetails returns a new FileRequestsChangePolicyDetails instance
func NewFileRequestsChangePolicyDetails(NewValue *FileRequestsPolicy) *FileRequestsChangePolicyDetails {
	s := new(FileRequestsChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// FileRequestsChangePolicyType : has no documentation (yet)
type FileRequestsChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRequestsChangePolicyType returns a new FileRequestsChangePolicyType instance
func NewFileRequestsChangePolicyType(Description string) *FileRequestsChangePolicyType {
	s := new(FileRequestsChangePolicyType)
	s.Description = Description
	return s
}

// FileRequestsEmailsEnabledDetails : Enabled file request emails for everyone.
type FileRequestsEmailsEnabledDetails struct {
}

// NewFileRequestsEmailsEnabledDetails returns a new FileRequestsEmailsEnabledDetails instance
func NewFileRequestsEmailsEnabledDetails() *FileRequestsEmailsEnabledDetails {
	s := new(FileRequestsEmailsEnabledDetails)
	return s
}

// FileRequestsEmailsEnabledType : has no documentation (yet)
type FileRequestsEmailsEnabledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRequestsEmailsEnabledType returns a new FileRequestsEmailsEnabledType instance
func NewFileRequestsEmailsEnabledType(Description string) *FileRequestsEmailsEnabledType {
	s := new(FileRequestsEmailsEnabledType)
	s.Description = Description
	return s
}

// FileRequestsEmailsRestrictedToTeamOnlyDetails : Enabled file request emails
// for team.
type FileRequestsEmailsRestrictedToTeamOnlyDetails struct {
}

// NewFileRequestsEmailsRestrictedToTeamOnlyDetails returns a new FileRequestsEmailsRestrictedToTeamOnlyDetails instance
func NewFileRequestsEmailsRestrictedToTeamOnlyDetails() *FileRequestsEmailsRestrictedToTeamOnlyDetails {
	s := new(FileRequestsEmailsRestrictedToTeamOnlyDetails)
	return s
}

// FileRequestsEmailsRestrictedToTeamOnlyType : has no documentation (yet)
type FileRequestsEmailsRestrictedToTeamOnlyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRequestsEmailsRestrictedToTeamOnlyType returns a new FileRequestsEmailsRestrictedToTeamOnlyType instance
func NewFileRequestsEmailsRestrictedToTeamOnlyType(Description string) *FileRequestsEmailsRestrictedToTeamOnlyType {
	s := new(FileRequestsEmailsRestrictedToTeamOnlyType)
	s.Description = Description
	return s
}

// FileRequestsPolicy : File requests policy
type FileRequestsPolicy struct {
	dropbox.Tagged
}

// Valid tag values for FileRequestsPolicy
const (
	FileRequestsPolicyDisabled = "disabled"
	FileRequestsPolicyEnabled  = "enabled"
	FileRequestsPolicyOther    = "other"
)

// FileResolveCommentDetails : Resolved file comment.
type FileResolveCommentDetails struct {
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewFileResolveCommentDetails returns a new FileResolveCommentDetails instance
func NewFileResolveCommentDetails() *FileResolveCommentDetails {
	s := new(FileResolveCommentDetails)
	return s
}

// FileResolveCommentType : has no documentation (yet)
type FileResolveCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileResolveCommentType returns a new FileResolveCommentType instance
func NewFileResolveCommentType(Description string) *FileResolveCommentType {
	s := new(FileResolveCommentType)
	s.Description = Description
	return s
}

// FileRestoreDetails : Restored deleted files and/or folders.
type FileRestoreDetails struct {
}

// NewFileRestoreDetails returns a new FileRestoreDetails instance
func NewFileRestoreDetails() *FileRestoreDetails {
	s := new(FileRestoreDetails)
	return s
}

// FileRestoreType : has no documentation (yet)
type FileRestoreType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRestoreType returns a new FileRestoreType instance
func NewFileRestoreType(Description string) *FileRestoreType {
	s := new(FileRestoreType)
	s.Description = Description
	return s
}

// FileRevertDetails : Reverted files to previous version.
type FileRevertDetails struct {
}

// NewFileRevertDetails returns a new FileRevertDetails instance
func NewFileRevertDetails() *FileRevertDetails {
	s := new(FileRevertDetails)
	return s
}

// FileRevertType : has no documentation (yet)
type FileRevertType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRevertType returns a new FileRevertType instance
func NewFileRevertType(Description string) *FileRevertType {
	s := new(FileRevertType)
	s.Description = Description
	return s
}

// FileRollbackChangesDetails : Rolled back file actions.
type FileRollbackChangesDetails struct {
}

// NewFileRollbackChangesDetails returns a new FileRollbackChangesDetails instance
func NewFileRollbackChangesDetails() *FileRollbackChangesDetails {
	s := new(FileRollbackChangesDetails)
	return s
}

// FileRollbackChangesType : has no documentation (yet)
type FileRollbackChangesType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileRollbackChangesType returns a new FileRollbackChangesType instance
func NewFileRollbackChangesType(Description string) *FileRollbackChangesType {
	s := new(FileRollbackChangesType)
	s.Description = Description
	return s
}

// FileSaveCopyReferenceDetails : Saved file/folder using copy reference.
type FileSaveCopyReferenceDetails struct {
	// RelocateActionDetails : Relocate action details.
	RelocateActionDetails []*RelocateAssetReferencesLogInfo `json:"relocate_action_details"`
}

// NewFileSaveCopyReferenceDetails returns a new FileSaveCopyReferenceDetails instance
func NewFileSaveCopyReferenceDetails(RelocateActionDetails []*RelocateAssetReferencesLogInfo) *FileSaveCopyReferenceDetails {
	s := new(FileSaveCopyReferenceDetails)
	s.RelocateActionDetails = RelocateActionDetails
	return s
}

// FileSaveCopyReferenceType : has no documentation (yet)
type FileSaveCopyReferenceType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileSaveCopyReferenceType returns a new FileSaveCopyReferenceType instance
func NewFileSaveCopyReferenceType(Description string) *FileSaveCopyReferenceType {
	s := new(FileSaveCopyReferenceType)
	s.Description = Description
	return s
}

// FileTransfersFileAddDetails : Transfer files added.
type FileTransfersFileAddDetails struct {
	// FileTransferId : Transfer id.
	FileTransferId string `json:"file_transfer_id"`
}

// NewFileTransfersFileAddDetails returns a new FileTransfersFileAddDetails instance
func NewFileTransfersFileAddDetails(FileTransferId string) *FileTransfersFileAddDetails {
	s := new(FileTransfersFileAddDetails)
	s.FileTransferId = FileTransferId
	return s
}

// FileTransfersFileAddType : has no documentation (yet)
type FileTransfersFileAddType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileTransfersFileAddType returns a new FileTransfersFileAddType instance
func NewFileTransfersFileAddType(Description string) *FileTransfersFileAddType {
	s := new(FileTransfersFileAddType)
	s.Description = Description
	return s
}

// FileTransfersPolicy : File transfers policy
type FileTransfersPolicy struct {
	dropbox.Tagged
}

// Valid tag values for FileTransfersPolicy
const (
	FileTransfersPolicyDisabled = "disabled"
	FileTransfersPolicyEnabled  = "enabled"
	FileTransfersPolicyOther    = "other"
)

// FileTransfersPolicyChangedDetails : Changed file transfers policy for team.
type FileTransfersPolicyChangedDetails struct {
	// NewValue : New file transfers policy.
	NewValue *FileTransfersPolicy `json:"new_value"`
	// PreviousValue : Previous file transfers policy.
	PreviousValue *FileTransfersPolicy `json:"previous_value"`
}

// NewFileTransfersPolicyChangedDetails returns a new FileTransfersPolicyChangedDetails instance
func NewFileTransfersPolicyChangedDetails(NewValue *FileTransfersPolicy, PreviousValue *FileTransfersPolicy) *FileTransfersPolicyChangedDetails {
	s := new(FileTransfersPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// FileTransfersPolicyChangedType : has no documentation (yet)
type FileTransfersPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileTransfersPolicyChangedType returns a new FileTransfersPolicyChangedType instance
func NewFileTransfersPolicyChangedType(Description string) *FileTransfersPolicyChangedType {
	s := new(FileTransfersPolicyChangedType)
	s.Description = Description
	return s
}

// FileTransfersTransferDeleteDetails : Deleted transfer.
type FileTransfersTransferDeleteDetails struct {
	// FileTransferId : Transfer id.
	FileTransferId string `json:"file_transfer_id"`
}

// NewFileTransfersTransferDeleteDetails returns a new FileTransfersTransferDeleteDetails instance
func NewFileTransfersTransferDeleteDetails(FileTransferId string) *FileTransfersTransferDeleteDetails {
	s := new(FileTransfersTransferDeleteDetails)
	s.FileTransferId = FileTransferId
	return s
}

// FileTransfersTransferDeleteType : has no documentation (yet)
type FileTransfersTransferDeleteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileTransfersTransferDeleteType returns a new FileTransfersTransferDeleteType instance
func NewFileTransfersTransferDeleteType(Description string) *FileTransfersTransferDeleteType {
	s := new(FileTransfersTransferDeleteType)
	s.Description = Description
	return s
}

// FileTransfersTransferDownloadDetails : Transfer downloaded.
type FileTransfersTransferDownloadDetails struct {
	// FileTransferId : Transfer id.
	FileTransferId string `json:"file_transfer_id"`
}

// NewFileTransfersTransferDownloadDetails returns a new FileTransfersTransferDownloadDetails instance
func NewFileTransfersTransferDownloadDetails(FileTransferId string) *FileTransfersTransferDownloadDetails {
	s := new(FileTransfersTransferDownloadDetails)
	s.FileTransferId = FileTransferId
	return s
}

// FileTransfersTransferDownloadType : has no documentation (yet)
type FileTransfersTransferDownloadType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileTransfersTransferDownloadType returns a new FileTransfersTransferDownloadType instance
func NewFileTransfersTransferDownloadType(Description string) *FileTransfersTransferDownloadType {
	s := new(FileTransfersTransferDownloadType)
	s.Description = Description
	return s
}

// FileTransfersTransferSendDetails : Sent transfer.
type FileTransfersTransferSendDetails struct {
	// FileTransferId : Transfer id.
	FileTransferId string `json:"file_transfer_id"`
}

// NewFileTransfersTransferSendDetails returns a new FileTransfersTransferSendDetails instance
func NewFileTransfersTransferSendDetails(FileTransferId string) *FileTransfersTransferSendDetails {
	s := new(FileTransfersTransferSendDetails)
	s.FileTransferId = FileTransferId
	return s
}

// FileTransfersTransferSendType : has no documentation (yet)
type FileTransfersTransferSendType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileTransfersTransferSendType returns a new FileTransfersTransferSendType instance
func NewFileTransfersTransferSendType(Description string) *FileTransfersTransferSendType {
	s := new(FileTransfersTransferSendType)
	s.Description = Description
	return s
}

// FileTransfersTransferViewDetails : Viewed transfer.
type FileTransfersTransferViewDetails struct {
	// FileTransferId : Transfer id.
	FileTransferId string `json:"file_transfer_id"`
}

// NewFileTransfersTransferViewDetails returns a new FileTransfersTransferViewDetails instance
func NewFileTransfersTransferViewDetails(FileTransferId string) *FileTransfersTransferViewDetails {
	s := new(FileTransfersTransferViewDetails)
	s.FileTransferId = FileTransferId
	return s
}

// FileTransfersTransferViewType : has no documentation (yet)
type FileTransfersTransferViewType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileTransfersTransferViewType returns a new FileTransfersTransferViewType instance
func NewFileTransfersTransferViewType(Description string) *FileTransfersTransferViewType {
	s := new(FileTransfersTransferViewType)
	s.Description = Description
	return s
}

// FileUnlikeCommentDetails : Unliked file comment.
type FileUnlikeCommentDetails struct {
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewFileUnlikeCommentDetails returns a new FileUnlikeCommentDetails instance
func NewFileUnlikeCommentDetails() *FileUnlikeCommentDetails {
	s := new(FileUnlikeCommentDetails)
	return s
}

// FileUnlikeCommentType : has no documentation (yet)
type FileUnlikeCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileUnlikeCommentType returns a new FileUnlikeCommentType instance
func NewFileUnlikeCommentType(Description string) *FileUnlikeCommentType {
	s := new(FileUnlikeCommentType)
	s.Description = Description
	return s
}

// FileUnresolveCommentDetails : Unresolved file comment.
type FileUnresolveCommentDetails struct {
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewFileUnresolveCommentDetails returns a new FileUnresolveCommentDetails instance
func NewFileUnresolveCommentDetails() *FileUnresolveCommentDetails {
	s := new(FileUnresolveCommentDetails)
	return s
}

// FileUnresolveCommentType : has no documentation (yet)
type FileUnresolveCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFileUnresolveCommentType returns a new FileUnresolveCommentType instance
func NewFileUnresolveCommentType(Description string) *FileUnresolveCommentType {
	s := new(FileUnresolveCommentType)
	s.Description = Description
	return s
}

// FolderLogInfo : Folder's logged information.
type FolderLogInfo struct {
	FileOrFolderLogInfo
	// FileCount : Number of files within the folder.
	FileCount uint64 `json:"file_count,omitempty"`
}

// NewFolderLogInfo returns a new FolderLogInfo instance
func NewFolderLogInfo(Path *PathLogInfo) *FolderLogInfo {
	s := new(FolderLogInfo)
	s.Path = Path
	return s
}

// FolderOverviewDescriptionChangedDetails : Updated folder overview.
type FolderOverviewDescriptionChangedDetails struct {
	// FolderOverviewLocationAsset : Folder Overview location position in the
	// Assets list.
	FolderOverviewLocationAsset uint64 `json:"folder_overview_location_asset"`
}

// NewFolderOverviewDescriptionChangedDetails returns a new FolderOverviewDescriptionChangedDetails instance
func NewFolderOverviewDescriptionChangedDetails(FolderOverviewLocationAsset uint64) *FolderOverviewDescriptionChangedDetails {
	s := new(FolderOverviewDescriptionChangedDetails)
	s.FolderOverviewLocationAsset = FolderOverviewLocationAsset
	return s
}

// FolderOverviewDescriptionChangedType : has no documentation (yet)
type FolderOverviewDescriptionChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFolderOverviewDescriptionChangedType returns a new FolderOverviewDescriptionChangedType instance
func NewFolderOverviewDescriptionChangedType(Description string) *FolderOverviewDescriptionChangedType {
	s := new(FolderOverviewDescriptionChangedType)
	s.Description = Description
	return s
}

// FolderOverviewItemPinnedDetails : Pinned item to folder overview.
type FolderOverviewItemPinnedDetails struct {
	// FolderOverviewLocationAsset : Folder Overview location position in the
	// Assets list.
	FolderOverviewLocationAsset uint64 `json:"folder_overview_location_asset"`
	// PinnedItemsAssetIndices : Pinned items positions in the Assets list.
	PinnedItemsAssetIndices []uint64 `json:"pinned_items_asset_indices"`
}

// NewFolderOverviewItemPinnedDetails returns a new FolderOverviewItemPinnedDetails instance
func NewFolderOverviewItemPinnedDetails(FolderOverviewLocationAsset uint64, PinnedItemsAssetIndices []uint64) *FolderOverviewItemPinnedDetails {
	s := new(FolderOverviewItemPinnedDetails)
	s.FolderOverviewLocationAsset = FolderOverviewLocationAsset
	s.PinnedItemsAssetIndices = PinnedItemsAssetIndices
	return s
}

// FolderOverviewItemPinnedType : has no documentation (yet)
type FolderOverviewItemPinnedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFolderOverviewItemPinnedType returns a new FolderOverviewItemPinnedType instance
func NewFolderOverviewItemPinnedType(Description string) *FolderOverviewItemPinnedType {
	s := new(FolderOverviewItemPinnedType)
	s.Description = Description
	return s
}

// FolderOverviewItemUnpinnedDetails : Unpinned item from folder overview.
type FolderOverviewItemUnpinnedDetails struct {
	// FolderOverviewLocationAsset : Folder Overview location position in the
	// Assets list.
	FolderOverviewLocationAsset uint64 `json:"folder_overview_location_asset"`
	// PinnedItemsAssetIndices : Pinned items positions in the Assets list.
	PinnedItemsAssetIndices []uint64 `json:"pinned_items_asset_indices"`
}

// NewFolderOverviewItemUnpinnedDetails returns a new FolderOverviewItemUnpinnedDetails instance
func NewFolderOverviewItemUnpinnedDetails(FolderOverviewLocationAsset uint64, PinnedItemsAssetIndices []uint64) *FolderOverviewItemUnpinnedDetails {
	s := new(FolderOverviewItemUnpinnedDetails)
	s.FolderOverviewLocationAsset = FolderOverviewLocationAsset
	s.PinnedItemsAssetIndices = PinnedItemsAssetIndices
	return s
}

// FolderOverviewItemUnpinnedType : has no documentation (yet)
type FolderOverviewItemUnpinnedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewFolderOverviewItemUnpinnedType returns a new FolderOverviewItemUnpinnedType instance
func NewFolderOverviewItemUnpinnedType(Description string) *FolderOverviewItemUnpinnedType {
	s := new(FolderOverviewItemUnpinnedType)
	s.Description = Description
	return s
}

// GeoLocationLogInfo : Geographic location details.
type GeoLocationLogInfo struct {
	// City : City name.
	City string `json:"city,omitempty"`
	// Region : Region name.
	Region string `json:"region,omitempty"`
	// Country : Country code.
	Country string `json:"country,omitempty"`
	// IpAddress : IP address.
	IpAddress string `json:"ip_address"`
}

// NewGeoLocationLogInfo returns a new GeoLocationLogInfo instance
func NewGeoLocationLogInfo(IpAddress string) *GeoLocationLogInfo {
	s := new(GeoLocationLogInfo)
	s.IpAddress = IpAddress
	return s
}

// GetTeamEventsArg : has no documentation (yet)
type GetTeamEventsArg struct {
	// Limit : The maximal number of results to return per call. Note that some
	// calls may not return `limit` number of events, and may even return no
	// events, even with `has_more` set to true. In this case, callers should
	// fetch again using `getEventsContinue`.
	Limit uint32 `json:"limit"`
	// AccountId : Filter the events by account ID. Return only events with this
	// account_id as either Actor, Context, or Participants.
	AccountId string `json:"account_id,omitempty"`
	// Time : Filter by time range.
	Time *team_common.TimeRange `json:"time,omitempty"`
	// Category : Filter the returned events to a single category. Note that
	// category shouldn't be provided together with event_type.
	Category *EventCategory `json:"category,omitempty"`
	// EventType : Filter the returned events to a single event type. Note that
	// event_type shouldn't be provided together with category.
	EventType *EventTypeArg `json:"event_type,omitempty"`
}

// NewGetTeamEventsArg returns a new GetTeamEventsArg instance
func NewGetTeamEventsArg() *GetTeamEventsArg {
	s := new(GetTeamEventsArg)
	s.Limit = 1000
	return s
}

// GetTeamEventsContinueArg : has no documentation (yet)
type GetTeamEventsContinueArg struct {
	// Cursor : Indicates from what point to get the next set of events.
	Cursor string `json:"cursor"`
}

// NewGetTeamEventsContinueArg returns a new GetTeamEventsContinueArg instance
func NewGetTeamEventsContinueArg(Cursor string) *GetTeamEventsContinueArg {
	s := new(GetTeamEventsContinueArg)
	s.Cursor = Cursor
	return s
}

// GetTeamEventsContinueError : Errors that can be raised when calling
// `getEventsContinue`.
type GetTeamEventsContinueError struct {
	dropbox.Tagged
	// Reset : Cursors are intended to be used quickly. Individual cursor values
	// are normally valid for days, but in rare cases may be reset sooner.
	// Cursor reset errors should be handled by fetching a new cursor from
	// `getEvents`. The associated value is the approximate timestamp of the
	// most recent event returned by the cursor. This should be used as a
	// resumption point when calling `getEvents` to obtain a new cursor.
	Reset time.Time `json:"reset,omitempty"`
}

// Valid tag values for GetTeamEventsContinueError
const (
	GetTeamEventsContinueErrorBadCursor = "bad_cursor"
	GetTeamEventsContinueErrorReset     = "reset"
	GetTeamEventsContinueErrorOther     = "other"
)

// UnmarshalJSON deserializes into a GetTeamEventsContinueError instance
func (u *GetTeamEventsContinueError) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
		// Reset : Cursors are intended to be used quickly. Individual cursor
		// values are normally valid for days, but in rare cases may be reset
		// sooner. Cursor reset errors should be handled by fetching a new
		// cursor from `getEvents`. The associated value is the approximate
		// timestamp of the most recent event returned by the cursor. This
		// should be used as a resumption point when calling `getEvents` to
		// obtain a new cursor.
		Reset time.Time `json:"reset,omitempty"`
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "reset":
		u.Reset = w.Reset

	}
	return nil
}

// GetTeamEventsError : Errors that can be raised when calling `getEvents`.
type GetTeamEventsError struct {
	dropbox.Tagged
}

// Valid tag values for GetTeamEventsError
const (
	GetTeamEventsErrorAccountIdNotFound = "account_id_not_found"
	GetTeamEventsErrorInvalidTimeRange  = "invalid_time_range"
	GetTeamEventsErrorInvalidFilters    = "invalid_filters"
	GetTeamEventsErrorOther             = "other"
)

// GetTeamEventsResult : has no documentation (yet)
type GetTeamEventsResult struct {
	// Events : List of events. Note that events are not guaranteed to be sorted
	// by their timestamp value.
	Events []*TeamEvent `json:"events"`
	// Cursor : Pass the cursor into `getEventsContinue` to obtain additional
	// events. The value of `cursor` may change for each response from
	// `getEventsContinue`, regardless of the value of `has_more`; older cursor
	// strings may expire. Thus, callers should ensure that they update their
	// cursor based on the latest value of `cursor` after each call, and poll
	// regularly if they wish to poll for new events. Callers should handle
	// reset exceptions for expired cursors.
	Cursor string `json:"cursor"`
	// HasMore : Is true if there may be additional events that have not been
	// returned yet. An additional call to `getEventsContinue` can retrieve
	// them. Note that `has_more` may be true, even if `events` is empty.
	HasMore bool `json:"has_more"`
}

// NewGetTeamEventsResult returns a new GetTeamEventsResult instance
func NewGetTeamEventsResult(Events []*TeamEvent, Cursor string, HasMore bool) *GetTeamEventsResult {
	s := new(GetTeamEventsResult)
	s.Events = Events
	s.Cursor = Cursor
	s.HasMore = HasMore
	return s
}

// GoogleSsoChangePolicyDetails : Enabled/disabled Google single sign-on for
// team.
type GoogleSsoChangePolicyDetails struct {
	// NewValue : New Google single sign-on policy.
	NewValue *GoogleSsoPolicy `json:"new_value"`
	// PreviousValue : Previous Google single sign-on policy. Might be missing
	// due to historical data gap.
	PreviousValue *GoogleSsoPolicy `json:"previous_value,omitempty"`
}

// NewGoogleSsoChangePolicyDetails returns a new GoogleSsoChangePolicyDetails instance
func NewGoogleSsoChangePolicyDetails(NewValue *GoogleSsoPolicy) *GoogleSsoChangePolicyDetails {
	s := new(GoogleSsoChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// GoogleSsoChangePolicyType : has no documentation (yet)
type GoogleSsoChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGoogleSsoChangePolicyType returns a new GoogleSsoChangePolicyType instance
func NewGoogleSsoChangePolicyType(Description string) *GoogleSsoChangePolicyType {
	s := new(GoogleSsoChangePolicyType)
	s.Description = Description
	return s
}

// GoogleSsoPolicy : Google SSO policy
type GoogleSsoPolicy struct {
	dropbox.Tagged
}

// Valid tag values for GoogleSsoPolicy
const (
	GoogleSsoPolicyDisabled = "disabled"
	GoogleSsoPolicyEnabled  = "enabled"
	GoogleSsoPolicyOther    = "other"
)

// GovernancePolicyAddFolderFailedDetails : Couldn't add a folder to a policy.
type GovernancePolicyAddFolderFailedDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// Folder : Folder.
	Folder string `json:"folder"`
	// Reason : Reason.
	Reason string `json:"reason,omitempty"`
}

// NewGovernancePolicyAddFolderFailedDetails returns a new GovernancePolicyAddFolderFailedDetails instance
func NewGovernancePolicyAddFolderFailedDetails(GovernancePolicyId string, Name string, Folder string) *GovernancePolicyAddFolderFailedDetails {
	s := new(GovernancePolicyAddFolderFailedDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	s.Folder = Folder
	return s
}

// GovernancePolicyAddFolderFailedType : has no documentation (yet)
type GovernancePolicyAddFolderFailedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyAddFolderFailedType returns a new GovernancePolicyAddFolderFailedType instance
func NewGovernancePolicyAddFolderFailedType(Description string) *GovernancePolicyAddFolderFailedType {
	s := new(GovernancePolicyAddFolderFailedType)
	s.Description = Description
	return s
}

// GovernancePolicyAddFoldersDetails : Added folders to policy.
type GovernancePolicyAddFoldersDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// Folders : Folders.
	Folders []string `json:"folders,omitempty"`
}

// NewGovernancePolicyAddFoldersDetails returns a new GovernancePolicyAddFoldersDetails instance
func NewGovernancePolicyAddFoldersDetails(GovernancePolicyId string, Name string) *GovernancePolicyAddFoldersDetails {
	s := new(GovernancePolicyAddFoldersDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	return s
}

// GovernancePolicyAddFoldersType : has no documentation (yet)
type GovernancePolicyAddFoldersType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyAddFoldersType returns a new GovernancePolicyAddFoldersType instance
func NewGovernancePolicyAddFoldersType(Description string) *GovernancePolicyAddFoldersType {
	s := new(GovernancePolicyAddFoldersType)
	s.Description = Description
	return s
}

// GovernancePolicyContentDisposedDetails : Content disposed.
type GovernancePolicyContentDisposedDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// DispositionType : Disposition type.
	DispositionType *DispositionActionType `json:"disposition_type"`
}

// NewGovernancePolicyContentDisposedDetails returns a new GovernancePolicyContentDisposedDetails instance
func NewGovernancePolicyContentDisposedDetails(GovernancePolicyId string, Name string, DispositionType *DispositionActionType) *GovernancePolicyContentDisposedDetails {
	s := new(GovernancePolicyContentDisposedDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	s.DispositionType = DispositionType
	return s
}

// GovernancePolicyContentDisposedType : has no documentation (yet)
type GovernancePolicyContentDisposedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyContentDisposedType returns a new GovernancePolicyContentDisposedType instance
func NewGovernancePolicyContentDisposedType(Description string) *GovernancePolicyContentDisposedType {
	s := new(GovernancePolicyContentDisposedType)
	s.Description = Description
	return s
}

// GovernancePolicyCreateDetails : Activated a new policy.
type GovernancePolicyCreateDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// Duration : Duration in days.
	Duration *DurationLogInfo `json:"duration"`
	// Folders : Folders.
	Folders []string `json:"folders,omitempty"`
}

// NewGovernancePolicyCreateDetails returns a new GovernancePolicyCreateDetails instance
func NewGovernancePolicyCreateDetails(GovernancePolicyId string, Name string, Duration *DurationLogInfo) *GovernancePolicyCreateDetails {
	s := new(GovernancePolicyCreateDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	s.Duration = Duration
	return s
}

// GovernancePolicyCreateType : has no documentation (yet)
type GovernancePolicyCreateType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyCreateType returns a new GovernancePolicyCreateType instance
func NewGovernancePolicyCreateType(Description string) *GovernancePolicyCreateType {
	s := new(GovernancePolicyCreateType)
	s.Description = Description
	return s
}

// GovernancePolicyDeleteDetails : Deleted a policy.
type GovernancePolicyDeleteDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
}

// NewGovernancePolicyDeleteDetails returns a new GovernancePolicyDeleteDetails instance
func NewGovernancePolicyDeleteDetails(GovernancePolicyId string, Name string) *GovernancePolicyDeleteDetails {
	s := new(GovernancePolicyDeleteDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	return s
}

// GovernancePolicyDeleteType : has no documentation (yet)
type GovernancePolicyDeleteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyDeleteType returns a new GovernancePolicyDeleteType instance
func NewGovernancePolicyDeleteType(Description string) *GovernancePolicyDeleteType {
	s := new(GovernancePolicyDeleteType)
	s.Description = Description
	return s
}

// GovernancePolicyEditDetailsDetails : Edited policy.
type GovernancePolicyEditDetailsDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// Attribute : Attribute.
	Attribute string `json:"attribute"`
	// PreviousValue : From.
	PreviousValue string `json:"previous_value"`
	// NewValue : To.
	NewValue string `json:"new_value"`
}

// NewGovernancePolicyEditDetailsDetails returns a new GovernancePolicyEditDetailsDetails instance
func NewGovernancePolicyEditDetailsDetails(GovernancePolicyId string, Name string, Attribute string, PreviousValue string, NewValue string) *GovernancePolicyEditDetailsDetails {
	s := new(GovernancePolicyEditDetailsDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	s.Attribute = Attribute
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// GovernancePolicyEditDetailsType : has no documentation (yet)
type GovernancePolicyEditDetailsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyEditDetailsType returns a new GovernancePolicyEditDetailsType instance
func NewGovernancePolicyEditDetailsType(Description string) *GovernancePolicyEditDetailsType {
	s := new(GovernancePolicyEditDetailsType)
	s.Description = Description
	return s
}

// GovernancePolicyEditDurationDetails : Changed policy duration.
type GovernancePolicyEditDurationDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// PreviousValue : From.
	PreviousValue *DurationLogInfo `json:"previous_value"`
	// NewValue : To.
	NewValue *DurationLogInfo `json:"new_value"`
}

// NewGovernancePolicyEditDurationDetails returns a new GovernancePolicyEditDurationDetails instance
func NewGovernancePolicyEditDurationDetails(GovernancePolicyId string, Name string, PreviousValue *DurationLogInfo, NewValue *DurationLogInfo) *GovernancePolicyEditDurationDetails {
	s := new(GovernancePolicyEditDurationDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// GovernancePolicyEditDurationType : has no documentation (yet)
type GovernancePolicyEditDurationType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyEditDurationType returns a new GovernancePolicyEditDurationType instance
func NewGovernancePolicyEditDurationType(Description string) *GovernancePolicyEditDurationType {
	s := new(GovernancePolicyEditDurationType)
	s.Description = Description
	return s
}

// GovernancePolicyExportCreatedDetails : Created a policy download.
type GovernancePolicyExportCreatedDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// ExportName : Export name.
	ExportName string `json:"export_name"`
}

// NewGovernancePolicyExportCreatedDetails returns a new GovernancePolicyExportCreatedDetails instance
func NewGovernancePolicyExportCreatedDetails(GovernancePolicyId string, Name string, ExportName string) *GovernancePolicyExportCreatedDetails {
	s := new(GovernancePolicyExportCreatedDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	s.ExportName = ExportName
	return s
}

// GovernancePolicyExportCreatedType : has no documentation (yet)
type GovernancePolicyExportCreatedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyExportCreatedType returns a new GovernancePolicyExportCreatedType instance
func NewGovernancePolicyExportCreatedType(Description string) *GovernancePolicyExportCreatedType {
	s := new(GovernancePolicyExportCreatedType)
	s.Description = Description
	return s
}

// GovernancePolicyExportRemovedDetails : Removed a policy download.
type GovernancePolicyExportRemovedDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// ExportName : Export name.
	ExportName string `json:"export_name"`
}

// NewGovernancePolicyExportRemovedDetails returns a new GovernancePolicyExportRemovedDetails instance
func NewGovernancePolicyExportRemovedDetails(GovernancePolicyId string, Name string, ExportName string) *GovernancePolicyExportRemovedDetails {
	s := new(GovernancePolicyExportRemovedDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	s.ExportName = ExportName
	return s
}

// GovernancePolicyExportRemovedType : has no documentation (yet)
type GovernancePolicyExportRemovedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyExportRemovedType returns a new GovernancePolicyExportRemovedType instance
func NewGovernancePolicyExportRemovedType(Description string) *GovernancePolicyExportRemovedType {
	s := new(GovernancePolicyExportRemovedType)
	s.Description = Description
	return s
}

// GovernancePolicyRemoveFoldersDetails : Removed folders from policy.
type GovernancePolicyRemoveFoldersDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// Folders : Folders.
	Folders []string `json:"folders,omitempty"`
	// Reason : Reason.
	Reason string `json:"reason,omitempty"`
}

// NewGovernancePolicyRemoveFoldersDetails returns a new GovernancePolicyRemoveFoldersDetails instance
func NewGovernancePolicyRemoveFoldersDetails(GovernancePolicyId string, Name string) *GovernancePolicyRemoveFoldersDetails {
	s := new(GovernancePolicyRemoveFoldersDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	return s
}

// GovernancePolicyRemoveFoldersType : has no documentation (yet)
type GovernancePolicyRemoveFoldersType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyRemoveFoldersType returns a new GovernancePolicyRemoveFoldersType instance
func NewGovernancePolicyRemoveFoldersType(Description string) *GovernancePolicyRemoveFoldersType {
	s := new(GovernancePolicyRemoveFoldersType)
	s.Description = Description
	return s
}

// GovernancePolicyReportCreatedDetails : Created a summary report for a policy.
type GovernancePolicyReportCreatedDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
}

// NewGovernancePolicyReportCreatedDetails returns a new GovernancePolicyReportCreatedDetails instance
func NewGovernancePolicyReportCreatedDetails(GovernancePolicyId string, Name string) *GovernancePolicyReportCreatedDetails {
	s := new(GovernancePolicyReportCreatedDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	return s
}

// GovernancePolicyReportCreatedType : has no documentation (yet)
type GovernancePolicyReportCreatedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyReportCreatedType returns a new GovernancePolicyReportCreatedType instance
func NewGovernancePolicyReportCreatedType(Description string) *GovernancePolicyReportCreatedType {
	s := new(GovernancePolicyReportCreatedType)
	s.Description = Description
	return s
}

// GovernancePolicyZipPartDownloadedDetails : Downloaded content from a policy.
type GovernancePolicyZipPartDownloadedDetails struct {
	// GovernancePolicyId : Policy ID.
	GovernancePolicyId string `json:"governance_policy_id"`
	// Name : Policy name.
	Name string `json:"name"`
	// PolicyType : Policy type.
	PolicyType *PolicyType `json:"policy_type,omitempty"`
	// ExportName : Export name.
	ExportName string `json:"export_name"`
	// Part : Part.
	Part string `json:"part,omitempty"`
}

// NewGovernancePolicyZipPartDownloadedDetails returns a new GovernancePolicyZipPartDownloadedDetails instance
func NewGovernancePolicyZipPartDownloadedDetails(GovernancePolicyId string, Name string, ExportName string) *GovernancePolicyZipPartDownloadedDetails {
	s := new(GovernancePolicyZipPartDownloadedDetails)
	s.GovernancePolicyId = GovernancePolicyId
	s.Name = Name
	s.ExportName = ExportName
	return s
}

// GovernancePolicyZipPartDownloadedType : has no documentation (yet)
type GovernancePolicyZipPartDownloadedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGovernancePolicyZipPartDownloadedType returns a new GovernancePolicyZipPartDownloadedType instance
func NewGovernancePolicyZipPartDownloadedType(Description string) *GovernancePolicyZipPartDownloadedType {
	s := new(GovernancePolicyZipPartDownloadedType)
	s.Description = Description
	return s
}

// GroupAddExternalIdDetails : Added external ID for group.
type GroupAddExternalIdDetails struct {
	// NewValue : Current external id.
	NewValue string `json:"new_value"`
}

// NewGroupAddExternalIdDetails returns a new GroupAddExternalIdDetails instance
func NewGroupAddExternalIdDetails(NewValue string) *GroupAddExternalIdDetails {
	s := new(GroupAddExternalIdDetails)
	s.NewValue = NewValue
	return s
}

// GroupAddExternalIdType : has no documentation (yet)
type GroupAddExternalIdType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupAddExternalIdType returns a new GroupAddExternalIdType instance
func NewGroupAddExternalIdType(Description string) *GroupAddExternalIdType {
	s := new(GroupAddExternalIdType)
	s.Description = Description
	return s
}

// GroupAddMemberDetails : Added team members to group.
type GroupAddMemberDetails struct {
	// IsGroupOwner : Is group owner.
	IsGroupOwner bool `json:"is_group_owner"`
}

// NewGroupAddMemberDetails returns a new GroupAddMemberDetails instance
func NewGroupAddMemberDetails(IsGroupOwner bool) *GroupAddMemberDetails {
	s := new(GroupAddMemberDetails)
	s.IsGroupOwner = IsGroupOwner
	return s
}

// GroupAddMemberType : has no documentation (yet)
type GroupAddMemberType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupAddMemberType returns a new GroupAddMemberType instance
func NewGroupAddMemberType(Description string) *GroupAddMemberType {
	s := new(GroupAddMemberType)
	s.Description = Description
	return s
}

// GroupChangeExternalIdDetails : Changed external ID for group.
type GroupChangeExternalIdDetails struct {
	// NewValue : Current external id.
	NewValue string `json:"new_value"`
	// PreviousValue : Old external id.
	PreviousValue string `json:"previous_value"`
}

// NewGroupChangeExternalIdDetails returns a new GroupChangeExternalIdDetails instance
func NewGroupChangeExternalIdDetails(NewValue string, PreviousValue string) *GroupChangeExternalIdDetails {
	s := new(GroupChangeExternalIdDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// GroupChangeExternalIdType : has no documentation (yet)
type GroupChangeExternalIdType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupChangeExternalIdType returns a new GroupChangeExternalIdType instance
func NewGroupChangeExternalIdType(Description string) *GroupChangeExternalIdType {
	s := new(GroupChangeExternalIdType)
	s.Description = Description
	return s
}

// GroupChangeManagementTypeDetails : Changed group management type.
type GroupChangeManagementTypeDetails struct {
	// NewValue : New group management type.
	NewValue *team_common.GroupManagementType `json:"new_value"`
	// PreviousValue : Previous group management type. Might be missing due to
	// historical data gap.
	PreviousValue *team_common.GroupManagementType `json:"previous_value,omitempty"`
}

// NewGroupChangeManagementTypeDetails returns a new GroupChangeManagementTypeDetails instance
func NewGroupChangeManagementTypeDetails(NewValue *team_common.GroupManagementType) *GroupChangeManagementTypeDetails {
	s := new(GroupChangeManagementTypeDetails)
	s.NewValue = NewValue
	return s
}

// GroupChangeManagementTypeType : has no documentation (yet)
type GroupChangeManagementTypeType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupChangeManagementTypeType returns a new GroupChangeManagementTypeType instance
func NewGroupChangeManagementTypeType(Description string) *GroupChangeManagementTypeType {
	s := new(GroupChangeManagementTypeType)
	s.Description = Description
	return s
}

// GroupChangeMemberRoleDetails : Changed manager permissions of group member.
type GroupChangeMemberRoleDetails struct {
	// IsGroupOwner : Is group owner.
	IsGroupOwner bool `json:"is_group_owner"`
}

// NewGroupChangeMemberRoleDetails returns a new GroupChangeMemberRoleDetails instance
func NewGroupChangeMemberRoleDetails(IsGroupOwner bool) *GroupChangeMemberRoleDetails {
	s := new(GroupChangeMemberRoleDetails)
	s.IsGroupOwner = IsGroupOwner
	return s
}

// GroupChangeMemberRoleType : has no documentation (yet)
type GroupChangeMemberRoleType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupChangeMemberRoleType returns a new GroupChangeMemberRoleType instance
func NewGroupChangeMemberRoleType(Description string) *GroupChangeMemberRoleType {
	s := new(GroupChangeMemberRoleType)
	s.Description = Description
	return s
}

// GroupCreateDetails : Created group.
type GroupCreateDetails struct {
	// IsCompanyManaged : Is company managed group.
	IsCompanyManaged bool `json:"is_company_managed,omitempty"`
	// JoinPolicy : Group join policy.
	JoinPolicy *GroupJoinPolicy `json:"join_policy,omitempty"`
}

// NewGroupCreateDetails returns a new GroupCreateDetails instance
func NewGroupCreateDetails() *GroupCreateDetails {
	s := new(GroupCreateDetails)
	return s
}

// GroupCreateType : has no documentation (yet)
type GroupCreateType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupCreateType returns a new GroupCreateType instance
func NewGroupCreateType(Description string) *GroupCreateType {
	s := new(GroupCreateType)
	s.Description = Description
	return s
}

// GroupDeleteDetails : Deleted group.
type GroupDeleteDetails struct {
	// IsCompanyManaged : Is company managed group.
	IsCompanyManaged bool `json:"is_company_managed,omitempty"`
}

// NewGroupDeleteDetails returns a new GroupDeleteDetails instance
func NewGroupDeleteDetails() *GroupDeleteDetails {
	s := new(GroupDeleteDetails)
	return s
}

// GroupDeleteType : has no documentation (yet)
type GroupDeleteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupDeleteType returns a new GroupDeleteType instance
func NewGroupDeleteType(Description string) *GroupDeleteType {
	s := new(GroupDeleteType)
	s.Description = Description
	return s
}

// GroupDescriptionUpdatedDetails : Updated group.
type GroupDescriptionUpdatedDetails struct {
}

// NewGroupDescriptionUpdatedDetails returns a new GroupDescriptionUpdatedDetails instance
func NewGroupDescriptionUpdatedDetails() *GroupDescriptionUpdatedDetails {
	s := new(GroupDescriptionUpdatedDetails)
	return s
}

// GroupDescriptionUpdatedType : has no documentation (yet)
type GroupDescriptionUpdatedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupDescriptionUpdatedType returns a new GroupDescriptionUpdatedType instance
func NewGroupDescriptionUpdatedType(Description string) *GroupDescriptionUpdatedType {
	s := new(GroupDescriptionUpdatedType)
	s.Description = Description
	return s
}

// GroupJoinPolicy : has no documentation (yet)
type GroupJoinPolicy struct {
	dropbox.Tagged
}

// Valid tag values for GroupJoinPolicy
const (
	GroupJoinPolicyOpen          = "open"
	GroupJoinPolicyRequestToJoin = "request_to_join"
	GroupJoinPolicyOther         = "other"
)

// GroupJoinPolicyUpdatedDetails : Updated group join policy.
type GroupJoinPolicyUpdatedDetails struct {
	// IsCompanyManaged : Is company managed group.
	IsCompanyManaged bool `json:"is_company_managed,omitempty"`
	// JoinPolicy : Group join policy.
	JoinPolicy *GroupJoinPolicy `json:"join_policy,omitempty"`
}

// NewGroupJoinPolicyUpdatedDetails returns a new GroupJoinPolicyUpdatedDetails instance
func NewGroupJoinPolicyUpdatedDetails() *GroupJoinPolicyUpdatedDetails {
	s := new(GroupJoinPolicyUpdatedDetails)
	return s
}

// GroupJoinPolicyUpdatedType : has no documentation (yet)
type GroupJoinPolicyUpdatedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupJoinPolicyUpdatedType returns a new GroupJoinPolicyUpdatedType instance
func NewGroupJoinPolicyUpdatedType(Description string) *GroupJoinPolicyUpdatedType {
	s := new(GroupJoinPolicyUpdatedType)
	s.Description = Description
	return s
}

// GroupLogInfo : Group's logged information.
type GroupLogInfo struct {
	// GroupId : The unique id of this group.
	GroupId string `json:"group_id,omitempty"`
	// DisplayName : The name of this group.
	DisplayName string `json:"display_name"`
	// ExternalId : External group ID.
	ExternalId string `json:"external_id,omitempty"`
}

// NewGroupLogInfo returns a new GroupLogInfo instance
func NewGroupLogInfo(DisplayName string) *GroupLogInfo {
	s := new(GroupLogInfo)
	s.DisplayName = DisplayName
	return s
}

// GroupMovedDetails : Moved group.
type GroupMovedDetails struct {
}

// NewGroupMovedDetails returns a new GroupMovedDetails instance
func NewGroupMovedDetails() *GroupMovedDetails {
	s := new(GroupMovedDetails)
	return s
}

// GroupMovedType : has no documentation (yet)
type GroupMovedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupMovedType returns a new GroupMovedType instance
func NewGroupMovedType(Description string) *GroupMovedType {
	s := new(GroupMovedType)
	s.Description = Description
	return s
}

// GroupRemoveExternalIdDetails : Removed external ID for group.
type GroupRemoveExternalIdDetails struct {
	// PreviousValue : Old external id.
	PreviousValue string `json:"previous_value"`
}

// NewGroupRemoveExternalIdDetails returns a new GroupRemoveExternalIdDetails instance
func NewGroupRemoveExternalIdDetails(PreviousValue string) *GroupRemoveExternalIdDetails {
	s := new(GroupRemoveExternalIdDetails)
	s.PreviousValue = PreviousValue
	return s
}

// GroupRemoveExternalIdType : has no documentation (yet)
type GroupRemoveExternalIdType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupRemoveExternalIdType returns a new GroupRemoveExternalIdType instance
func NewGroupRemoveExternalIdType(Description string) *GroupRemoveExternalIdType {
	s := new(GroupRemoveExternalIdType)
	s.Description = Description
	return s
}

// GroupRemoveMemberDetails : Removed team members from group.
type GroupRemoveMemberDetails struct {
}

// NewGroupRemoveMemberDetails returns a new GroupRemoveMemberDetails instance
func NewGroupRemoveMemberDetails() *GroupRemoveMemberDetails {
	s := new(GroupRemoveMemberDetails)
	return s
}

// GroupRemoveMemberType : has no documentation (yet)
type GroupRemoveMemberType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupRemoveMemberType returns a new GroupRemoveMemberType instance
func NewGroupRemoveMemberType(Description string) *GroupRemoveMemberType {
	s := new(GroupRemoveMemberType)
	s.Description = Description
	return s
}

// GroupRenameDetails : Renamed group.
type GroupRenameDetails struct {
	// PreviousValue : Previous display name.
	PreviousValue string `json:"previous_value"`
	// NewValue : New display name.
	NewValue string `json:"new_value"`
}

// NewGroupRenameDetails returns a new GroupRenameDetails instance
func NewGroupRenameDetails(PreviousValue string, NewValue string) *GroupRenameDetails {
	s := new(GroupRenameDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// GroupRenameType : has no documentation (yet)
type GroupRenameType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupRenameType returns a new GroupRenameType instance
func NewGroupRenameType(Description string) *GroupRenameType {
	s := new(GroupRenameType)
	s.Description = Description
	return s
}

// GroupUserManagementChangePolicyDetails : Changed who can create groups.
type GroupUserManagementChangePolicyDetails struct {
	// NewValue : New group users management policy.
	NewValue *team_policies.GroupCreation `json:"new_value"`
	// PreviousValue : Previous group users management policy. Might be missing
	// due to historical data gap.
	PreviousValue *team_policies.GroupCreation `json:"previous_value,omitempty"`
}

// NewGroupUserManagementChangePolicyDetails returns a new GroupUserManagementChangePolicyDetails instance
func NewGroupUserManagementChangePolicyDetails(NewValue *team_policies.GroupCreation) *GroupUserManagementChangePolicyDetails {
	s := new(GroupUserManagementChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// GroupUserManagementChangePolicyType : has no documentation (yet)
type GroupUserManagementChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGroupUserManagementChangePolicyType returns a new GroupUserManagementChangePolicyType instance
func NewGroupUserManagementChangePolicyType(Description string) *GroupUserManagementChangePolicyType {
	s := new(GroupUserManagementChangePolicyType)
	s.Description = Description
	return s
}

// GuestAdminChangeStatusDetails : Changed guest team admin status.
type GuestAdminChangeStatusDetails struct {
	// IsGuest : True for guest, false for host.
	IsGuest bool `json:"is_guest"`
	// GuestTeamName : The name of the guest team.
	GuestTeamName string `json:"guest_team_name,omitempty"`
	// HostTeamName : The name of the host team.
	HostTeamName string `json:"host_team_name,omitempty"`
	// PreviousValue : Previous request state.
	PreviousValue *TrustedTeamsRequestState `json:"previous_value"`
	// NewValue : New request state.
	NewValue *TrustedTeamsRequestState `json:"new_value"`
	// ActionDetails : Action details.
	ActionDetails *TrustedTeamsRequestAction `json:"action_details"`
}

// NewGuestAdminChangeStatusDetails returns a new GuestAdminChangeStatusDetails instance
func NewGuestAdminChangeStatusDetails(IsGuest bool, PreviousValue *TrustedTeamsRequestState, NewValue *TrustedTeamsRequestState, ActionDetails *TrustedTeamsRequestAction) *GuestAdminChangeStatusDetails {
	s := new(GuestAdminChangeStatusDetails)
	s.IsGuest = IsGuest
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	s.ActionDetails = ActionDetails
	return s
}

// GuestAdminChangeStatusType : has no documentation (yet)
type GuestAdminChangeStatusType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGuestAdminChangeStatusType returns a new GuestAdminChangeStatusType instance
func NewGuestAdminChangeStatusType(Description string) *GuestAdminChangeStatusType {
	s := new(GuestAdminChangeStatusType)
	s.Description = Description
	return s
}

// GuestAdminSignedInViaTrustedTeamsDetails : Started trusted team admin
// session.
type GuestAdminSignedInViaTrustedTeamsDetails struct {
	// TeamName : Host team name.
	TeamName string `json:"team_name,omitempty"`
	// TrustedTeamName : Trusted team name.
	TrustedTeamName string `json:"trusted_team_name,omitempty"`
}

// NewGuestAdminSignedInViaTrustedTeamsDetails returns a new GuestAdminSignedInViaTrustedTeamsDetails instance
func NewGuestAdminSignedInViaTrustedTeamsDetails() *GuestAdminSignedInViaTrustedTeamsDetails {
	s := new(GuestAdminSignedInViaTrustedTeamsDetails)
	return s
}

// GuestAdminSignedInViaTrustedTeamsType : has no documentation (yet)
type GuestAdminSignedInViaTrustedTeamsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGuestAdminSignedInViaTrustedTeamsType returns a new GuestAdminSignedInViaTrustedTeamsType instance
func NewGuestAdminSignedInViaTrustedTeamsType(Description string) *GuestAdminSignedInViaTrustedTeamsType {
	s := new(GuestAdminSignedInViaTrustedTeamsType)
	s.Description = Description
	return s
}

// GuestAdminSignedOutViaTrustedTeamsDetails : Ended trusted team admin session.
type GuestAdminSignedOutViaTrustedTeamsDetails struct {
	// TeamName : Host team name.
	TeamName string `json:"team_name,omitempty"`
	// TrustedTeamName : Trusted team name.
	TrustedTeamName string `json:"trusted_team_name,omitempty"`
}

// NewGuestAdminSignedOutViaTrustedTeamsDetails returns a new GuestAdminSignedOutViaTrustedTeamsDetails instance
func NewGuestAdminSignedOutViaTrustedTeamsDetails() *GuestAdminSignedOutViaTrustedTeamsDetails {
	s := new(GuestAdminSignedOutViaTrustedTeamsDetails)
	return s
}

// GuestAdminSignedOutViaTrustedTeamsType : has no documentation (yet)
type GuestAdminSignedOutViaTrustedTeamsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewGuestAdminSignedOutViaTrustedTeamsType returns a new GuestAdminSignedOutViaTrustedTeamsType instance
func NewGuestAdminSignedOutViaTrustedTeamsType(Description string) *GuestAdminSignedOutViaTrustedTeamsType {
	s := new(GuestAdminSignedOutViaTrustedTeamsType)
	s.Description = Description
	return s
}

// IdentifierType : has no documentation (yet)
type IdentifierType struct {
	dropbox.Tagged
}

// Valid tag values for IdentifierType
const (
	IdentifierTypeEmail               = "email"
	IdentifierTypeFacebookProfileName = "facebook_profile_name"
	IdentifierTypeOther               = "other"
)

// IntegrationConnectedDetails : Connected integration for member.
type IntegrationConnectedDetails struct {
	// IntegrationName : Name of the third-party integration.
	IntegrationName string `json:"integration_name"`
}

// NewIntegrationConnectedDetails returns a new IntegrationConnectedDetails instance
func NewIntegrationConnectedDetails(IntegrationName string) *IntegrationConnectedDetails {
	s := new(IntegrationConnectedDetails)
	s.IntegrationName = IntegrationName
	return s
}

// IntegrationConnectedType : has no documentation (yet)
type IntegrationConnectedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewIntegrationConnectedType returns a new IntegrationConnectedType instance
func NewIntegrationConnectedType(Description string) *IntegrationConnectedType {
	s := new(IntegrationConnectedType)
	s.Description = Description
	return s
}

// IntegrationDisconnectedDetails : Disconnected integration for member.
type IntegrationDisconnectedDetails struct {
	// IntegrationName : Name of the third-party integration.
	IntegrationName string `json:"integration_name"`
}

// NewIntegrationDisconnectedDetails returns a new IntegrationDisconnectedDetails instance
func NewIntegrationDisconnectedDetails(IntegrationName string) *IntegrationDisconnectedDetails {
	s := new(IntegrationDisconnectedDetails)
	s.IntegrationName = IntegrationName
	return s
}

// IntegrationDisconnectedType : has no documentation (yet)
type IntegrationDisconnectedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewIntegrationDisconnectedType returns a new IntegrationDisconnectedType instance
func NewIntegrationDisconnectedType(Description string) *IntegrationDisconnectedType {
	s := new(IntegrationDisconnectedType)
	s.Description = Description
	return s
}

// IntegrationPolicy : Policy for controlling whether a service integration is
// enabled for the team.
type IntegrationPolicy struct {
	dropbox.Tagged
}

// Valid tag values for IntegrationPolicy
const (
	IntegrationPolicyDisabled = "disabled"
	IntegrationPolicyEnabled  = "enabled"
	IntegrationPolicyOther    = "other"
)

// IntegrationPolicyChangedDetails : Changed integration policy for team.
type IntegrationPolicyChangedDetails struct {
	// IntegrationName : Name of the third-party integration.
	IntegrationName string `json:"integration_name"`
	// NewValue : New integration policy.
	NewValue *IntegrationPolicy `json:"new_value"`
	// PreviousValue : Previous integration policy.
	PreviousValue *IntegrationPolicy `json:"previous_value"`
}

// NewIntegrationPolicyChangedDetails returns a new IntegrationPolicyChangedDetails instance
func NewIntegrationPolicyChangedDetails(IntegrationName string, NewValue *IntegrationPolicy, PreviousValue *IntegrationPolicy) *IntegrationPolicyChangedDetails {
	s := new(IntegrationPolicyChangedDetails)
	s.IntegrationName = IntegrationName
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// IntegrationPolicyChangedType : has no documentation (yet)
type IntegrationPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewIntegrationPolicyChangedType returns a new IntegrationPolicyChangedType instance
func NewIntegrationPolicyChangedType(Description string) *IntegrationPolicyChangedType {
	s := new(IntegrationPolicyChangedType)
	s.Description = Description
	return s
}

// InviteAcceptanceEmailPolicy : Policy for deciding whether team admins receive
// email when an invitation to join the team is accepted
type InviteAcceptanceEmailPolicy struct {
	dropbox.Tagged
}

// Valid tag values for InviteAcceptanceEmailPolicy
const (
	InviteAcceptanceEmailPolicyDisabled = "disabled"
	InviteAcceptanceEmailPolicyEnabled  = "enabled"
	InviteAcceptanceEmailPolicyOther    = "other"
)

// InviteAcceptanceEmailPolicyChangedDetails : Changed invite accept email
// policy for team.
type InviteAcceptanceEmailPolicyChangedDetails struct {
	// NewValue : To.
	NewValue *InviteAcceptanceEmailPolicy `json:"new_value"`
	// PreviousValue : From.
	PreviousValue *InviteAcceptanceEmailPolicy `json:"previous_value"`
}

// NewInviteAcceptanceEmailPolicyChangedDetails returns a new InviteAcceptanceEmailPolicyChangedDetails instance
func NewInviteAcceptanceEmailPolicyChangedDetails(NewValue *InviteAcceptanceEmailPolicy, PreviousValue *InviteAcceptanceEmailPolicy) *InviteAcceptanceEmailPolicyChangedDetails {
	s := new(InviteAcceptanceEmailPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// InviteAcceptanceEmailPolicyChangedType : has no documentation (yet)
type InviteAcceptanceEmailPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewInviteAcceptanceEmailPolicyChangedType returns a new InviteAcceptanceEmailPolicyChangedType instance
func NewInviteAcceptanceEmailPolicyChangedType(Description string) *InviteAcceptanceEmailPolicyChangedType {
	s := new(InviteAcceptanceEmailPolicyChangedType)
	s.Description = Description
	return s
}

// InviteMethod : has no documentation (yet)
type InviteMethod struct {
	dropbox.Tagged
}

// Valid tag values for InviteMethod
const (
	InviteMethodAutoApprove          = "auto_approve"
	InviteMethodInviteLink           = "invite_link"
	InviteMethodMemberInvite         = "member_invite"
	InviteMethodMovedFromAnotherTeam = "moved_from_another_team"
	InviteMethodOther                = "other"
)

// JoinTeamDetails : Additional information relevant when a new member joins the
// team.
type JoinTeamDetails struct {
	// LinkedApps : Linked applications. (Deprecated) Please use has_linked_apps
	// boolean field instead.
	LinkedApps []*UserLinkedAppLogInfo `json:"linked_apps"`
	// LinkedDevices : Linked devices. (Deprecated) Please use
	// has_linked_devices boolean field instead.
	LinkedDevices []*LinkedDeviceLogInfo `json:"linked_devices"`
	// LinkedSharedFolders : Linked shared folders. (Deprecated) Please use
	// has_linked_shared_folders boolean field instead.
	LinkedSharedFolders []*FolderLogInfo `json:"linked_shared_folders"`
	// WasLinkedAppsTruncated : (Deprecated) True if the linked_apps list was
	// truncated to the maximum supported length (50).
	WasLinkedAppsTruncated bool `json:"was_linked_apps_truncated,omitempty"`
	// WasLinkedDevicesTruncated : (Deprecated) True if the linked_devices list
	// was truncated to the maximum supported length (50).
	WasLinkedDevicesTruncated bool `json:"was_linked_devices_truncated,omitempty"`
	// WasLinkedSharedFoldersTruncated : (Deprecated) True if the
	// linked_shared_folders list was truncated to the maximum supported length
	// (50).
	WasLinkedSharedFoldersTruncated bool `json:"was_linked_shared_folders_truncated,omitempty"`
	// HasLinkedApps : True if the user had linked apps at event time.
	HasLinkedApps bool `json:"has_linked_apps,omitempty"`
	// HasLinkedDevices : True if the user had linked apps at event time.
	HasLinkedDevices bool `json:"has_linked_devices,omitempty"`
	// HasLinkedSharedFolders : True if the user had linked shared folders at
	// event time.
	HasLinkedSharedFolders bool `json:"has_linked_shared_folders,omitempty"`
}

// NewJoinTeamDetails returns a new JoinTeamDetails instance
func NewJoinTeamDetails(LinkedApps []*UserLinkedAppLogInfo, LinkedDevices []*LinkedDeviceLogInfo, LinkedSharedFolders []*FolderLogInfo) *JoinTeamDetails {
	s := new(JoinTeamDetails)
	s.LinkedApps = LinkedApps
	s.LinkedDevices = LinkedDevices
	s.LinkedSharedFolders = LinkedSharedFolders
	return s
}

// LabelType : Label type
type LabelType struct {
	dropbox.Tagged
}

// Valid tag values for LabelType
const (
	LabelTypePersonalInformation = "personal_information"
	LabelTypeTestOnly            = "test_only"
	LabelTypeUserDefinedTag      = "user_defined_tag"
	LabelTypeOther               = "other"
)

// LegacyDeviceSessionLogInfo : Information on sessions, in legacy format
type LegacyDeviceSessionLogInfo struct {
	DeviceSessionLogInfo
	// SessionInfo : Session unique id.
	SessionInfo IsSessionLogInfo `json:"session_info,omitempty"`
	// DisplayName : The device name. Might be missing due to historical data
	// gap.
	DisplayName string `json:"display_name,omitempty"`
	// IsEmmManaged : Is device managed by emm. Might be missing due to
	// historical data gap.
	IsEmmManaged bool `json:"is_emm_managed,omitempty"`
	// Platform : Information on the hosting platform. Might be missing due to
	// historical data gap.
	Platform string `json:"platform,omitempty"`
	// MacAddress : The mac address of the last activity from this session.
	// Might be missing due to historical data gap.
	MacAddress string `json:"mac_address,omitempty"`
	// OsVersion : The hosting OS version. Might be missing due to historical
	// data gap.
	OsVersion string `json:"os_version,omitempty"`
	// DeviceType : Information on the hosting device type. Might be missing due
	// to historical data gap.
	DeviceType string `json:"device_type,omitempty"`
	// ClientVersion : The Dropbox client version. Might be missing due to
	// historical data gap.
	ClientVersion string `json:"client_version,omitempty"`
	// LegacyUniqId : Alternative unique device session id, instead of session
	// id field. Might be missing due to historical data gap.
	LegacyUniqId string `json:"legacy_uniq_id,omitempty"`
}

// NewLegacyDeviceSessionLogInfo returns a new LegacyDeviceSessionLogInfo instance
func NewLegacyDeviceSessionLogInfo() *LegacyDeviceSessionLogInfo {
	s := new(LegacyDeviceSessionLogInfo)
	return s
}

// UnmarshalJSON deserializes into a LegacyDeviceSessionLogInfo instance
func (u *LegacyDeviceSessionLogInfo) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// IpAddress : The IP address of the last activity from this session.
		IpAddress string `json:"ip_address,omitempty"`
		// Created : The time this session was created.
		Created *time.Time `json:"created,omitempty"`
		// Updated : The time of the last activity from this session.
		Updated *time.Time `json:"updated,omitempty"`
		// SessionInfo : Session unique id.
		SessionInfo json.RawMessage `json:"session_info,omitempty"`
		// DisplayName : The device name. Might be missing due to historical
		// data gap.
		DisplayName string `json:"display_name,omitempty"`
		// IsEmmManaged : Is device managed by emm. Might be missing due to
		// historical data gap.
		IsEmmManaged bool `json:"is_emm_managed,omitempty"`
		// Platform : Information on the hosting platform. Might be missing due
		// to historical data gap.
		Platform string `json:"platform,omitempty"`
		// MacAddress : The mac address of the last activity from this session.
		// Might be missing due to historical data gap.
		MacAddress string `json:"mac_address,omitempty"`
		// OsVersion : The hosting OS version. Might be missing due to
		// historical data gap.
		OsVersion string `json:"os_version,omitempty"`
		// DeviceType : Information on the hosting device type. Might be missing
		// due to historical data gap.
		DeviceType string `json:"device_type,omitempty"`
		// ClientVersion : The Dropbox client version. Might be missing due to
		// historical data gap.
		ClientVersion string `json:"client_version,omitempty"`
		// LegacyUniqId : Alternative unique device session id, instead of
		// session id field. Might be missing due to historical data gap.
		LegacyUniqId string `json:"legacy_uniq_id,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	u.IpAddress = w.IpAddress
	u.Created = w.Created
	u.Updated = w.Updated
	SessionInfo, err := IsSessionLogInfoFromJSON(w.SessionInfo)
	if err != nil {
		return err
	}
	u.SessionInfo = SessionInfo
	u.DisplayName = w.DisplayName
	u.IsEmmManaged = w.IsEmmManaged
	u.Platform = w.Platform
	u.MacAddress = w.MacAddress
	u.OsVersion = w.OsVersion
	u.DeviceType = w.DeviceType
	u.ClientVersion = w.ClientVersion
	u.LegacyUniqId = w.LegacyUniqId
	return nil
}

// LegalHoldsActivateAHoldDetails : Activated a hold.
type LegalHoldsActivateAHoldDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
	// StartDate : Hold start date.
	StartDate time.Time `json:"start_date"`
	// EndDate : Hold end date.
	EndDate *time.Time `json:"end_date,omitempty"`
}

// NewLegalHoldsActivateAHoldDetails returns a new LegalHoldsActivateAHoldDetails instance
func NewLegalHoldsActivateAHoldDetails(LegalHoldId string, Name string, StartDate time.Time) *LegalHoldsActivateAHoldDetails {
	s := new(LegalHoldsActivateAHoldDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	s.StartDate = StartDate
	return s
}

// LegalHoldsActivateAHoldType : has no documentation (yet)
type LegalHoldsActivateAHoldType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsActivateAHoldType returns a new LegalHoldsActivateAHoldType instance
func NewLegalHoldsActivateAHoldType(Description string) *LegalHoldsActivateAHoldType {
	s := new(LegalHoldsActivateAHoldType)
	s.Description = Description
	return s
}

// LegalHoldsAddMembersDetails : Added members to a hold.
type LegalHoldsAddMembersDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
}

// NewLegalHoldsAddMembersDetails returns a new LegalHoldsAddMembersDetails instance
func NewLegalHoldsAddMembersDetails(LegalHoldId string, Name string) *LegalHoldsAddMembersDetails {
	s := new(LegalHoldsAddMembersDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	return s
}

// LegalHoldsAddMembersType : has no documentation (yet)
type LegalHoldsAddMembersType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsAddMembersType returns a new LegalHoldsAddMembersType instance
func NewLegalHoldsAddMembersType(Description string) *LegalHoldsAddMembersType {
	s := new(LegalHoldsAddMembersType)
	s.Description = Description
	return s
}

// LegalHoldsChangeHoldDetailsDetails : Edited details for a hold.
type LegalHoldsChangeHoldDetailsDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
	// PreviousValue : Previous details.
	PreviousValue string `json:"previous_value"`
	// NewValue : New details.
	NewValue string `json:"new_value"`
}

// NewLegalHoldsChangeHoldDetailsDetails returns a new LegalHoldsChangeHoldDetailsDetails instance
func NewLegalHoldsChangeHoldDetailsDetails(LegalHoldId string, Name string, PreviousValue string, NewValue string) *LegalHoldsChangeHoldDetailsDetails {
	s := new(LegalHoldsChangeHoldDetailsDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// LegalHoldsChangeHoldDetailsType : has no documentation (yet)
type LegalHoldsChangeHoldDetailsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsChangeHoldDetailsType returns a new LegalHoldsChangeHoldDetailsType instance
func NewLegalHoldsChangeHoldDetailsType(Description string) *LegalHoldsChangeHoldDetailsType {
	s := new(LegalHoldsChangeHoldDetailsType)
	s.Description = Description
	return s
}

// LegalHoldsChangeHoldNameDetails : Renamed a hold.
type LegalHoldsChangeHoldNameDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// PreviousValue : Previous Name.
	PreviousValue string `json:"previous_value"`
	// NewValue : New Name.
	NewValue string `json:"new_value"`
}

// NewLegalHoldsChangeHoldNameDetails returns a new LegalHoldsChangeHoldNameDetails instance
func NewLegalHoldsChangeHoldNameDetails(LegalHoldId string, PreviousValue string, NewValue string) *LegalHoldsChangeHoldNameDetails {
	s := new(LegalHoldsChangeHoldNameDetails)
	s.LegalHoldId = LegalHoldId
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// LegalHoldsChangeHoldNameType : has no documentation (yet)
type LegalHoldsChangeHoldNameType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsChangeHoldNameType returns a new LegalHoldsChangeHoldNameType instance
func NewLegalHoldsChangeHoldNameType(Description string) *LegalHoldsChangeHoldNameType {
	s := new(LegalHoldsChangeHoldNameType)
	s.Description = Description
	return s
}

// LegalHoldsExportAHoldDetails : Exported hold.
type LegalHoldsExportAHoldDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
	// ExportName : Export name.
	ExportName string `json:"export_name,omitempty"`
}

// NewLegalHoldsExportAHoldDetails returns a new LegalHoldsExportAHoldDetails instance
func NewLegalHoldsExportAHoldDetails(LegalHoldId string, Name string) *LegalHoldsExportAHoldDetails {
	s := new(LegalHoldsExportAHoldDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	return s
}

// LegalHoldsExportAHoldType : has no documentation (yet)
type LegalHoldsExportAHoldType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsExportAHoldType returns a new LegalHoldsExportAHoldType instance
func NewLegalHoldsExportAHoldType(Description string) *LegalHoldsExportAHoldType {
	s := new(LegalHoldsExportAHoldType)
	s.Description = Description
	return s
}

// LegalHoldsExportCancelledDetails : Canceled export for a hold.
type LegalHoldsExportCancelledDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
	// ExportName : Export name.
	ExportName string `json:"export_name"`
}

// NewLegalHoldsExportCancelledDetails returns a new LegalHoldsExportCancelledDetails instance
func NewLegalHoldsExportCancelledDetails(LegalHoldId string, Name string, ExportName string) *LegalHoldsExportCancelledDetails {
	s := new(LegalHoldsExportCancelledDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	s.ExportName = ExportName
	return s
}

// LegalHoldsExportCancelledType : has no documentation (yet)
type LegalHoldsExportCancelledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsExportCancelledType returns a new LegalHoldsExportCancelledType instance
func NewLegalHoldsExportCancelledType(Description string) *LegalHoldsExportCancelledType {
	s := new(LegalHoldsExportCancelledType)
	s.Description = Description
	return s
}

// LegalHoldsExportDownloadedDetails : Downloaded export for a hold.
type LegalHoldsExportDownloadedDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
	// ExportName : Export name.
	ExportName string `json:"export_name"`
	// Part : Part.
	Part string `json:"part,omitempty"`
	// FileName : Filename.
	FileName string `json:"file_name,omitempty"`
}

// NewLegalHoldsExportDownloadedDetails returns a new LegalHoldsExportDownloadedDetails instance
func NewLegalHoldsExportDownloadedDetails(LegalHoldId string, Name string, ExportName string) *LegalHoldsExportDownloadedDetails {
	s := new(LegalHoldsExportDownloadedDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	s.ExportName = ExportName
	return s
}

// LegalHoldsExportDownloadedType : has no documentation (yet)
type LegalHoldsExportDownloadedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsExportDownloadedType returns a new LegalHoldsExportDownloadedType instance
func NewLegalHoldsExportDownloadedType(Description string) *LegalHoldsExportDownloadedType {
	s := new(LegalHoldsExportDownloadedType)
	s.Description = Description
	return s
}

// LegalHoldsExportRemovedDetails : Removed export for a hold.
type LegalHoldsExportRemovedDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
	// ExportName : Export name.
	ExportName string `json:"export_name"`
}

// NewLegalHoldsExportRemovedDetails returns a new LegalHoldsExportRemovedDetails instance
func NewLegalHoldsExportRemovedDetails(LegalHoldId string, Name string, ExportName string) *LegalHoldsExportRemovedDetails {
	s := new(LegalHoldsExportRemovedDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	s.ExportName = ExportName
	return s
}

// LegalHoldsExportRemovedType : has no documentation (yet)
type LegalHoldsExportRemovedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsExportRemovedType returns a new LegalHoldsExportRemovedType instance
func NewLegalHoldsExportRemovedType(Description string) *LegalHoldsExportRemovedType {
	s := new(LegalHoldsExportRemovedType)
	s.Description = Description
	return s
}

// LegalHoldsReleaseAHoldDetails : Released a hold.
type LegalHoldsReleaseAHoldDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
}

// NewLegalHoldsReleaseAHoldDetails returns a new LegalHoldsReleaseAHoldDetails instance
func NewLegalHoldsReleaseAHoldDetails(LegalHoldId string, Name string) *LegalHoldsReleaseAHoldDetails {
	s := new(LegalHoldsReleaseAHoldDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	return s
}

// LegalHoldsReleaseAHoldType : has no documentation (yet)
type LegalHoldsReleaseAHoldType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsReleaseAHoldType returns a new LegalHoldsReleaseAHoldType instance
func NewLegalHoldsReleaseAHoldType(Description string) *LegalHoldsReleaseAHoldType {
	s := new(LegalHoldsReleaseAHoldType)
	s.Description = Description
	return s
}

// LegalHoldsRemoveMembersDetails : Removed members from a hold.
type LegalHoldsRemoveMembersDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
}

// NewLegalHoldsRemoveMembersDetails returns a new LegalHoldsRemoveMembersDetails instance
func NewLegalHoldsRemoveMembersDetails(LegalHoldId string, Name string) *LegalHoldsRemoveMembersDetails {
	s := new(LegalHoldsRemoveMembersDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	return s
}

// LegalHoldsRemoveMembersType : has no documentation (yet)
type LegalHoldsRemoveMembersType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsRemoveMembersType returns a new LegalHoldsRemoveMembersType instance
func NewLegalHoldsRemoveMembersType(Description string) *LegalHoldsRemoveMembersType {
	s := new(LegalHoldsRemoveMembersType)
	s.Description = Description
	return s
}

// LegalHoldsReportAHoldDetails : Created a summary report for a hold.
type LegalHoldsReportAHoldDetails struct {
	// LegalHoldId : Hold ID.
	LegalHoldId string `json:"legal_hold_id"`
	// Name : Hold name.
	Name string `json:"name"`
}

// NewLegalHoldsReportAHoldDetails returns a new LegalHoldsReportAHoldDetails instance
func NewLegalHoldsReportAHoldDetails(LegalHoldId string, Name string) *LegalHoldsReportAHoldDetails {
	s := new(LegalHoldsReportAHoldDetails)
	s.LegalHoldId = LegalHoldId
	s.Name = Name
	return s
}

// LegalHoldsReportAHoldType : has no documentation (yet)
type LegalHoldsReportAHoldType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLegalHoldsReportAHoldType returns a new LegalHoldsReportAHoldType instance
func NewLegalHoldsReportAHoldType(Description string) *LegalHoldsReportAHoldType {
	s := new(LegalHoldsReportAHoldType)
	s.Description = Description
	return s
}

// LinkedDeviceLogInfo : The device sessions that user is linked to.
type LinkedDeviceLogInfo struct {
	dropbox.Tagged
	// DesktopDeviceSession : desktop device session's details.
	DesktopDeviceSession *DesktopDeviceSessionLogInfo `json:"desktop_device_session,omitempty"`
	// LegacyDeviceSession : legacy device session's details.
	LegacyDeviceSession *LegacyDeviceSessionLogInfo `json:"legacy_device_session,omitempty"`
	// MobileDeviceSession : mobile device session's details.
	MobileDeviceSession *MobileDeviceSessionLogInfo `json:"mobile_device_session,omitempty"`
	// WebDeviceSession : web device session's details.
	WebDeviceSession *WebDeviceSessionLogInfo `json:"web_device_session,omitempty"`
}

// Valid tag values for LinkedDeviceLogInfo
const (
	LinkedDeviceLogInfoDesktopDeviceSession = "desktop_device_session"
	LinkedDeviceLogInfoLegacyDeviceSession  = "legacy_device_session"
	LinkedDeviceLogInfoMobileDeviceSession  = "mobile_device_session"
	LinkedDeviceLogInfoWebDeviceSession     = "web_device_session"
	LinkedDeviceLogInfoOther                = "other"
)

// UnmarshalJSON deserializes into a LinkedDeviceLogInfo instance
func (u *LinkedDeviceLogInfo) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "desktop_device_session":
		if err = json.Unmarshal(body, &u.DesktopDeviceSession); err != nil {
			return err
		}

	case "legacy_device_session":
		if err = json.Unmarshal(body, &u.LegacyDeviceSession); err != nil {
			return err
		}

	case "mobile_device_session":
		if err = json.Unmarshal(body, &u.MobileDeviceSession); err != nil {
			return err
		}

	case "web_device_session":
		if err = json.Unmarshal(body, &u.WebDeviceSession); err != nil {
			return err
		}

	}
	return nil
}

// LockStatus : File lock status
type LockStatus struct {
	dropbox.Tagged
}

// Valid tag values for LockStatus
const (
	LockStatusLocked   = "locked"
	LockStatusUnlocked = "unlocked"
	LockStatusOther    = "other"
)

// LoginFailDetails : Failed to sign in.
type LoginFailDetails struct {
	// IsEmmManaged : Tells if the login device is EMM managed. Might be missing
	// due to historical data gap.
	IsEmmManaged bool `json:"is_emm_managed,omitempty"`
	// LoginMethod : Login method.
	LoginMethod *LoginMethod `json:"login_method"`
	// ErrorDetails : Error details.
	ErrorDetails *FailureDetailsLogInfo `json:"error_details"`
}

// NewLoginFailDetails returns a new LoginFailDetails instance
func NewLoginFailDetails(LoginMethod *LoginMethod, ErrorDetails *FailureDetailsLogInfo) *LoginFailDetails {
	s := new(LoginFailDetails)
	s.LoginMethod = LoginMethod
	s.ErrorDetails = ErrorDetails
	return s
}

// LoginFailType : has no documentation (yet)
type LoginFailType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLoginFailType returns a new LoginFailType instance
func NewLoginFailType(Description string) *LoginFailType {
	s := new(LoginFailType)
	s.Description = Description
	return s
}

// LoginMethod : has no documentation (yet)
type LoginMethod struct {
	dropbox.Tagged
}

// Valid tag values for LoginMethod
const (
	LoginMethodAppleOauth              = "apple_oauth"
	LoginMethodFirstPartyTokenExchange = "first_party_token_exchange"
	LoginMethodGoogleOauth             = "google_oauth"
	LoginMethodPassword                = "password"
	LoginMethodQrCode                  = "qr_code"
	LoginMethodSaml                    = "saml"
	LoginMethodTwoFactorAuthentication = "two_factor_authentication"
	LoginMethodWebSession              = "web_session"
	LoginMethodOther                   = "other"
)

// LoginSuccessDetails : Signed in.
type LoginSuccessDetails struct {
	// IsEmmManaged : Tells if the login device is EMM managed. Might be missing
	// due to historical data gap.
	IsEmmManaged bool `json:"is_emm_managed,omitempty"`
	// LoginMethod : Login method.
	LoginMethod *LoginMethod `json:"login_method"`
}

// NewLoginSuccessDetails returns a new LoginSuccessDetails instance
func NewLoginSuccessDetails(LoginMethod *LoginMethod) *LoginSuccessDetails {
	s := new(LoginSuccessDetails)
	s.LoginMethod = LoginMethod
	return s
}

// LoginSuccessType : has no documentation (yet)
type LoginSuccessType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLoginSuccessType returns a new LoginSuccessType instance
func NewLoginSuccessType(Description string) *LoginSuccessType {
	s := new(LoginSuccessType)
	s.Description = Description
	return s
}

// LogoutDetails : Signed out.
type LogoutDetails struct {
	// LoginId : Login session id.
	LoginId string `json:"login_id,omitempty"`
}

// NewLogoutDetails returns a new LogoutDetails instance
func NewLogoutDetails() *LogoutDetails {
	s := new(LogoutDetails)
	return s
}

// LogoutType : has no documentation (yet)
type LogoutType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewLogoutType returns a new LogoutType instance
func NewLogoutType(Description string) *LogoutType {
	s := new(LogoutType)
	s.Description = Description
	return s
}

// MemberAddExternalIdDetails : Added an external ID for team member.
type MemberAddExternalIdDetails struct {
	// NewValue : Current external id.
	NewValue string `json:"new_value"`
}

// NewMemberAddExternalIdDetails returns a new MemberAddExternalIdDetails instance
func NewMemberAddExternalIdDetails(NewValue string) *MemberAddExternalIdDetails {
	s := new(MemberAddExternalIdDetails)
	s.NewValue = NewValue
	return s
}

// MemberAddExternalIdType : has no documentation (yet)
type MemberAddExternalIdType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberAddExternalIdType returns a new MemberAddExternalIdType instance
func NewMemberAddExternalIdType(Description string) *MemberAddExternalIdType {
	s := new(MemberAddExternalIdType)
	s.Description = Description
	return s
}

// MemberAddNameDetails : Added team member name.
type MemberAddNameDetails struct {
	// NewValue : New user's name.
	NewValue *UserNameLogInfo `json:"new_value"`
}

// NewMemberAddNameDetails returns a new MemberAddNameDetails instance
func NewMemberAddNameDetails(NewValue *UserNameLogInfo) *MemberAddNameDetails {
	s := new(MemberAddNameDetails)
	s.NewValue = NewValue
	return s
}

// MemberAddNameType : has no documentation (yet)
type MemberAddNameType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberAddNameType returns a new MemberAddNameType instance
func NewMemberAddNameType(Description string) *MemberAddNameType {
	s := new(MemberAddNameType)
	s.Description = Description
	return s
}

// MemberChangeAdminRoleDetails : Changed team member admin role.
type MemberChangeAdminRoleDetails struct {
	// NewValue : New admin role. This field is relevant when the admin role is
	// changed or whenthe user role changes from no admin rights to with admin
	// rights.
	NewValue *AdminRole `json:"new_value,omitempty"`
	// PreviousValue : Previous admin role. This field is relevant when the
	// admin role is changed or when the admin role is removed.
	PreviousValue *AdminRole `json:"previous_value,omitempty"`
}

// NewMemberChangeAdminRoleDetails returns a new MemberChangeAdminRoleDetails instance
func NewMemberChangeAdminRoleDetails() *MemberChangeAdminRoleDetails {
	s := new(MemberChangeAdminRoleDetails)
	return s
}

// MemberChangeAdminRoleType : has no documentation (yet)
type MemberChangeAdminRoleType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberChangeAdminRoleType returns a new MemberChangeAdminRoleType instance
func NewMemberChangeAdminRoleType(Description string) *MemberChangeAdminRoleType {
	s := new(MemberChangeAdminRoleType)
	s.Description = Description
	return s
}

// MemberChangeEmailDetails : Changed team member email.
type MemberChangeEmailDetails struct {
	// NewValue : New email.
	NewValue string `json:"new_value"`
	// PreviousValue : Previous email. Might be missing due to historical data
	// gap.
	PreviousValue string `json:"previous_value,omitempty"`
}

// NewMemberChangeEmailDetails returns a new MemberChangeEmailDetails instance
func NewMemberChangeEmailDetails(NewValue string) *MemberChangeEmailDetails {
	s := new(MemberChangeEmailDetails)
	s.NewValue = NewValue
	return s
}

// MemberChangeEmailType : has no documentation (yet)
type MemberChangeEmailType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberChangeEmailType returns a new MemberChangeEmailType instance
func NewMemberChangeEmailType(Description string) *MemberChangeEmailType {
	s := new(MemberChangeEmailType)
	s.Description = Description
	return s
}

// MemberChangeExternalIdDetails : Changed the external ID for team member.
type MemberChangeExternalIdDetails struct {
	// NewValue : Current external id.
	NewValue string `json:"new_value"`
	// PreviousValue : Old external id.
	PreviousValue string `json:"previous_value"`
}

// NewMemberChangeExternalIdDetails returns a new MemberChangeExternalIdDetails instance
func NewMemberChangeExternalIdDetails(NewValue string, PreviousValue string) *MemberChangeExternalIdDetails {
	s := new(MemberChangeExternalIdDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// MemberChangeExternalIdType : has no documentation (yet)
type MemberChangeExternalIdType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberChangeExternalIdType returns a new MemberChangeExternalIdType instance
func NewMemberChangeExternalIdType(Description string) *MemberChangeExternalIdType {
	s := new(MemberChangeExternalIdType)
	s.Description = Description
	return s
}

// MemberChangeMembershipTypeDetails : Changed membership type (limited/full) of
// member.
type MemberChangeMembershipTypeDetails struct {
	// PrevValue : Previous membership type.
	PrevValue *TeamMembershipType `json:"prev_value"`
	// NewValue : New membership type.
	NewValue *TeamMembershipType `json:"new_value"`
}

// NewMemberChangeMembershipTypeDetails returns a new MemberChangeMembershipTypeDetails instance
func NewMemberChangeMembershipTypeDetails(PrevValue *TeamMembershipType, NewValue *TeamMembershipType) *MemberChangeMembershipTypeDetails {
	s := new(MemberChangeMembershipTypeDetails)
	s.PrevValue = PrevValue
	s.NewValue = NewValue
	return s
}

// MemberChangeMembershipTypeType : has no documentation (yet)
type MemberChangeMembershipTypeType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberChangeMembershipTypeType returns a new MemberChangeMembershipTypeType instance
func NewMemberChangeMembershipTypeType(Description string) *MemberChangeMembershipTypeType {
	s := new(MemberChangeMembershipTypeType)
	s.Description = Description
	return s
}

// MemberChangeNameDetails : Changed team member name.
type MemberChangeNameDetails struct {
	// NewValue : New user's name.
	NewValue *UserNameLogInfo `json:"new_value"`
	// PreviousValue : Previous user's name. Might be missing due to historical
	// data gap.
	PreviousValue *UserNameLogInfo `json:"previous_value,omitempty"`
}

// NewMemberChangeNameDetails returns a new MemberChangeNameDetails instance
func NewMemberChangeNameDetails(NewValue *UserNameLogInfo) *MemberChangeNameDetails {
	s := new(MemberChangeNameDetails)
	s.NewValue = NewValue
	return s
}

// MemberChangeNameType : has no documentation (yet)
type MemberChangeNameType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberChangeNameType returns a new MemberChangeNameType instance
func NewMemberChangeNameType(Description string) *MemberChangeNameType {
	s := new(MemberChangeNameType)
	s.Description = Description
	return s
}

// MemberChangeResellerRoleDetails : Changed team member reseller role.
type MemberChangeResellerRoleDetails struct {
	// NewValue : New reseller role. This field is relevant when the reseller
	// role is changed.
	NewValue *ResellerRole `json:"new_value"`
	// PreviousValue : Previous reseller role. This field is relevant when the
	// reseller role is changed or when the reseller role is removed.
	PreviousValue *ResellerRole `json:"previous_value"`
}

// NewMemberChangeResellerRoleDetails returns a new MemberChangeResellerRoleDetails instance
func NewMemberChangeResellerRoleDetails(NewValue *ResellerRole, PreviousValue *ResellerRole) *MemberChangeResellerRoleDetails {
	s := new(MemberChangeResellerRoleDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// MemberChangeResellerRoleType : has no documentation (yet)
type MemberChangeResellerRoleType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberChangeResellerRoleType returns a new MemberChangeResellerRoleType instance
func NewMemberChangeResellerRoleType(Description string) *MemberChangeResellerRoleType {
	s := new(MemberChangeResellerRoleType)
	s.Description = Description
	return s
}

// MemberChangeStatusDetails : Changed member status (invited, joined,
// suspended, etc.).
type MemberChangeStatusDetails struct {
	// PreviousValue : Previous member status. Might be missing due to
	// historical data gap.
	PreviousValue *MemberStatus `json:"previous_value,omitempty"`
	// NewValue : New member status.
	NewValue *MemberStatus `json:"new_value"`
	// Action : Additional information indicating the action taken that caused
	// status change.
	Action *ActionDetails `json:"action,omitempty"`
	// NewTeam : The user's new team name. This field is relevant when the user
	// is transferred off the team.
	NewTeam string `json:"new_team,omitempty"`
	// PreviousTeam : The user's previous team name. This field is relevant when
	// the user is transferred onto the team.
	PreviousTeam string `json:"previous_team,omitempty"`
}

// NewMemberChangeStatusDetails returns a new MemberChangeStatusDetails instance
func NewMemberChangeStatusDetails(NewValue *MemberStatus) *MemberChangeStatusDetails {
	s := new(MemberChangeStatusDetails)
	s.NewValue = NewValue
	return s
}

// MemberChangeStatusType : has no documentation (yet)
type MemberChangeStatusType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberChangeStatusType returns a new MemberChangeStatusType instance
func NewMemberChangeStatusType(Description string) *MemberChangeStatusType {
	s := new(MemberChangeStatusType)
	s.Description = Description
	return s
}

// MemberDeleteManualContactsDetails : Cleared manually added contacts.
type MemberDeleteManualContactsDetails struct {
}

// NewMemberDeleteManualContactsDetails returns a new MemberDeleteManualContactsDetails instance
func NewMemberDeleteManualContactsDetails() *MemberDeleteManualContactsDetails {
	s := new(MemberDeleteManualContactsDetails)
	return s
}

// MemberDeleteManualContactsType : has no documentation (yet)
type MemberDeleteManualContactsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberDeleteManualContactsType returns a new MemberDeleteManualContactsType instance
func NewMemberDeleteManualContactsType(Description string) *MemberDeleteManualContactsType {
	s := new(MemberDeleteManualContactsType)
	s.Description = Description
	return s
}

// MemberDeleteProfilePhotoDetails : Deleted team member profile photo.
type MemberDeleteProfilePhotoDetails struct {
}

// NewMemberDeleteProfilePhotoDetails returns a new MemberDeleteProfilePhotoDetails instance
func NewMemberDeleteProfilePhotoDetails() *MemberDeleteProfilePhotoDetails {
	s := new(MemberDeleteProfilePhotoDetails)
	return s
}

// MemberDeleteProfilePhotoType : has no documentation (yet)
type MemberDeleteProfilePhotoType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberDeleteProfilePhotoType returns a new MemberDeleteProfilePhotoType instance
func NewMemberDeleteProfilePhotoType(Description string) *MemberDeleteProfilePhotoType {
	s := new(MemberDeleteProfilePhotoType)
	s.Description = Description
	return s
}

// MemberPermanentlyDeleteAccountContentsDetails : Permanently deleted contents
// of deleted team member account.
type MemberPermanentlyDeleteAccountContentsDetails struct {
}

// NewMemberPermanentlyDeleteAccountContentsDetails returns a new MemberPermanentlyDeleteAccountContentsDetails instance
func NewMemberPermanentlyDeleteAccountContentsDetails() *MemberPermanentlyDeleteAccountContentsDetails {
	s := new(MemberPermanentlyDeleteAccountContentsDetails)
	return s
}

// MemberPermanentlyDeleteAccountContentsType : has no documentation (yet)
type MemberPermanentlyDeleteAccountContentsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberPermanentlyDeleteAccountContentsType returns a new MemberPermanentlyDeleteAccountContentsType instance
func NewMemberPermanentlyDeleteAccountContentsType(Description string) *MemberPermanentlyDeleteAccountContentsType {
	s := new(MemberPermanentlyDeleteAccountContentsType)
	s.Description = Description
	return s
}

// MemberRemoveActionType : has no documentation (yet)
type MemberRemoveActionType struct {
	dropbox.Tagged
}

// Valid tag values for MemberRemoveActionType
const (
	MemberRemoveActionTypeDelete                       = "delete"
	MemberRemoveActionTypeLeave                        = "leave"
	MemberRemoveActionTypeOffboard                     = "offboard"
	MemberRemoveActionTypeOffboardAndRetainTeamFolders = "offboard_and_retain_team_folders"
	MemberRemoveActionTypeOther                        = "other"
)

// MemberRemoveExternalIdDetails : Removed the external ID for team member.
type MemberRemoveExternalIdDetails struct {
	// PreviousValue : Old external id.
	PreviousValue string `json:"previous_value"`
}

// NewMemberRemoveExternalIdDetails returns a new MemberRemoveExternalIdDetails instance
func NewMemberRemoveExternalIdDetails(PreviousValue string) *MemberRemoveExternalIdDetails {
	s := new(MemberRemoveExternalIdDetails)
	s.PreviousValue = PreviousValue
	return s
}

// MemberRemoveExternalIdType : has no documentation (yet)
type MemberRemoveExternalIdType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberRemoveExternalIdType returns a new MemberRemoveExternalIdType instance
func NewMemberRemoveExternalIdType(Description string) *MemberRemoveExternalIdType {
	s := new(MemberRemoveExternalIdType)
	s.Description = Description
	return s
}

// MemberRequestsChangePolicyDetails : Changed whether users can find team when
// not invited.
type MemberRequestsChangePolicyDetails struct {
	// NewValue : New member change requests policy.
	NewValue *MemberRequestsPolicy `json:"new_value"`
	// PreviousValue : Previous member change requests policy. Might be missing
	// due to historical data gap.
	PreviousValue *MemberRequestsPolicy `json:"previous_value,omitempty"`
}

// NewMemberRequestsChangePolicyDetails returns a new MemberRequestsChangePolicyDetails instance
func NewMemberRequestsChangePolicyDetails(NewValue *MemberRequestsPolicy) *MemberRequestsChangePolicyDetails {
	s := new(MemberRequestsChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// MemberRequestsChangePolicyType : has no documentation (yet)
type MemberRequestsChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberRequestsChangePolicyType returns a new MemberRequestsChangePolicyType instance
func NewMemberRequestsChangePolicyType(Description string) *MemberRequestsChangePolicyType {
	s := new(MemberRequestsChangePolicyType)
	s.Description = Description
	return s
}

// MemberRequestsPolicy : has no documentation (yet)
type MemberRequestsPolicy struct {
	dropbox.Tagged
}

// Valid tag values for MemberRequestsPolicy
const (
	MemberRequestsPolicyAutoAccept      = "auto_accept"
	MemberRequestsPolicyDisabled        = "disabled"
	MemberRequestsPolicyRequireApproval = "require_approval"
	MemberRequestsPolicyOther           = "other"
)

// MemberSendInvitePolicy : Policy for controlling whether team members can send
// team invites
type MemberSendInvitePolicy struct {
	dropbox.Tagged
}

// Valid tag values for MemberSendInvitePolicy
const (
	MemberSendInvitePolicyDisabled        = "disabled"
	MemberSendInvitePolicyEveryone        = "everyone"
	MemberSendInvitePolicySpecificMembers = "specific_members"
	MemberSendInvitePolicyOther           = "other"
)

// MemberSendInvitePolicyChangedDetails : Changed member send invite policy for
// team.
type MemberSendInvitePolicyChangedDetails struct {
	// NewValue : New team member send invite policy.
	NewValue *MemberSendInvitePolicy `json:"new_value"`
	// PreviousValue : Previous team member send invite policy.
	PreviousValue *MemberSendInvitePolicy `json:"previous_value"`
}

// NewMemberSendInvitePolicyChangedDetails returns a new MemberSendInvitePolicyChangedDetails instance
func NewMemberSendInvitePolicyChangedDetails(NewValue *MemberSendInvitePolicy, PreviousValue *MemberSendInvitePolicy) *MemberSendInvitePolicyChangedDetails {
	s := new(MemberSendInvitePolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// MemberSendInvitePolicyChangedType : has no documentation (yet)
type MemberSendInvitePolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSendInvitePolicyChangedType returns a new MemberSendInvitePolicyChangedType instance
func NewMemberSendInvitePolicyChangedType(Description string) *MemberSendInvitePolicyChangedType {
	s := new(MemberSendInvitePolicyChangedType)
	s.Description = Description
	return s
}

// MemberSetProfilePhotoDetails : Set team member profile photo.
type MemberSetProfilePhotoDetails struct {
}

// NewMemberSetProfilePhotoDetails returns a new MemberSetProfilePhotoDetails instance
func NewMemberSetProfilePhotoDetails() *MemberSetProfilePhotoDetails {
	s := new(MemberSetProfilePhotoDetails)
	return s
}

// MemberSetProfilePhotoType : has no documentation (yet)
type MemberSetProfilePhotoType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSetProfilePhotoType returns a new MemberSetProfilePhotoType instance
func NewMemberSetProfilePhotoType(Description string) *MemberSetProfilePhotoType {
	s := new(MemberSetProfilePhotoType)
	s.Description = Description
	return s
}

// MemberSpaceLimitsAddCustomQuotaDetails : Set custom member space limit.
type MemberSpaceLimitsAddCustomQuotaDetails struct {
	// NewValue : New custom quota value in bytes.
	NewValue uint64 `json:"new_value"`
}

// NewMemberSpaceLimitsAddCustomQuotaDetails returns a new MemberSpaceLimitsAddCustomQuotaDetails instance
func NewMemberSpaceLimitsAddCustomQuotaDetails(NewValue uint64) *MemberSpaceLimitsAddCustomQuotaDetails {
	s := new(MemberSpaceLimitsAddCustomQuotaDetails)
	s.NewValue = NewValue
	return s
}

// MemberSpaceLimitsAddCustomQuotaType : has no documentation (yet)
type MemberSpaceLimitsAddCustomQuotaType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSpaceLimitsAddCustomQuotaType returns a new MemberSpaceLimitsAddCustomQuotaType instance
func NewMemberSpaceLimitsAddCustomQuotaType(Description string) *MemberSpaceLimitsAddCustomQuotaType {
	s := new(MemberSpaceLimitsAddCustomQuotaType)
	s.Description = Description
	return s
}

// MemberSpaceLimitsAddExceptionDetails : Added members to member space limit
// exception list.
type MemberSpaceLimitsAddExceptionDetails struct {
}

// NewMemberSpaceLimitsAddExceptionDetails returns a new MemberSpaceLimitsAddExceptionDetails instance
func NewMemberSpaceLimitsAddExceptionDetails() *MemberSpaceLimitsAddExceptionDetails {
	s := new(MemberSpaceLimitsAddExceptionDetails)
	return s
}

// MemberSpaceLimitsAddExceptionType : has no documentation (yet)
type MemberSpaceLimitsAddExceptionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSpaceLimitsAddExceptionType returns a new MemberSpaceLimitsAddExceptionType instance
func NewMemberSpaceLimitsAddExceptionType(Description string) *MemberSpaceLimitsAddExceptionType {
	s := new(MemberSpaceLimitsAddExceptionType)
	s.Description = Description
	return s
}

// MemberSpaceLimitsChangeCapsTypePolicyDetails : Changed member space limit
// type for team.
type MemberSpaceLimitsChangeCapsTypePolicyDetails struct {
	// PreviousValue : Previous space limit type.
	PreviousValue *SpaceCapsType `json:"previous_value"`
	// NewValue : New space limit type.
	NewValue *SpaceCapsType `json:"new_value"`
}

// NewMemberSpaceLimitsChangeCapsTypePolicyDetails returns a new MemberSpaceLimitsChangeCapsTypePolicyDetails instance
func NewMemberSpaceLimitsChangeCapsTypePolicyDetails(PreviousValue *SpaceCapsType, NewValue *SpaceCapsType) *MemberSpaceLimitsChangeCapsTypePolicyDetails {
	s := new(MemberSpaceLimitsChangeCapsTypePolicyDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// MemberSpaceLimitsChangeCapsTypePolicyType : has no documentation (yet)
type MemberSpaceLimitsChangeCapsTypePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSpaceLimitsChangeCapsTypePolicyType returns a new MemberSpaceLimitsChangeCapsTypePolicyType instance
func NewMemberSpaceLimitsChangeCapsTypePolicyType(Description string) *MemberSpaceLimitsChangeCapsTypePolicyType {
	s := new(MemberSpaceLimitsChangeCapsTypePolicyType)
	s.Description = Description
	return s
}

// MemberSpaceLimitsChangeCustomQuotaDetails : Changed custom member space
// limit.
type MemberSpaceLimitsChangeCustomQuotaDetails struct {
	// PreviousValue : Previous custom quota value in bytes.
	PreviousValue uint64 `json:"previous_value"`
	// NewValue : New custom quota value in bytes.
	NewValue uint64 `json:"new_value"`
}

// NewMemberSpaceLimitsChangeCustomQuotaDetails returns a new MemberSpaceLimitsChangeCustomQuotaDetails instance
func NewMemberSpaceLimitsChangeCustomQuotaDetails(PreviousValue uint64, NewValue uint64) *MemberSpaceLimitsChangeCustomQuotaDetails {
	s := new(MemberSpaceLimitsChangeCustomQuotaDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// MemberSpaceLimitsChangeCustomQuotaType : has no documentation (yet)
type MemberSpaceLimitsChangeCustomQuotaType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSpaceLimitsChangeCustomQuotaType returns a new MemberSpaceLimitsChangeCustomQuotaType instance
func NewMemberSpaceLimitsChangeCustomQuotaType(Description string) *MemberSpaceLimitsChangeCustomQuotaType {
	s := new(MemberSpaceLimitsChangeCustomQuotaType)
	s.Description = Description
	return s
}

// MemberSpaceLimitsChangePolicyDetails : Changed team default member space
// limit.
type MemberSpaceLimitsChangePolicyDetails struct {
	// PreviousValue : Previous team default limit value in bytes. Might be
	// missing due to historical data gap.
	PreviousValue uint64 `json:"previous_value,omitempty"`
	// NewValue : New team default limit value in bytes. Might be missing due to
	// historical data gap.
	NewValue uint64 `json:"new_value,omitempty"`
}

// NewMemberSpaceLimitsChangePolicyDetails returns a new MemberSpaceLimitsChangePolicyDetails instance
func NewMemberSpaceLimitsChangePolicyDetails() *MemberSpaceLimitsChangePolicyDetails {
	s := new(MemberSpaceLimitsChangePolicyDetails)
	return s
}

// MemberSpaceLimitsChangePolicyType : has no documentation (yet)
type MemberSpaceLimitsChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSpaceLimitsChangePolicyType returns a new MemberSpaceLimitsChangePolicyType instance
func NewMemberSpaceLimitsChangePolicyType(Description string) *MemberSpaceLimitsChangePolicyType {
	s := new(MemberSpaceLimitsChangePolicyType)
	s.Description = Description
	return s
}

// MemberSpaceLimitsChangeStatusDetails : Changed space limit status.
type MemberSpaceLimitsChangeStatusDetails struct {
	// PreviousValue : Previous storage quota status.
	PreviousValue *SpaceLimitsStatus `json:"previous_value"`
	// NewValue : New storage quota status.
	NewValue *SpaceLimitsStatus `json:"new_value"`
}

// NewMemberSpaceLimitsChangeStatusDetails returns a new MemberSpaceLimitsChangeStatusDetails instance
func NewMemberSpaceLimitsChangeStatusDetails(PreviousValue *SpaceLimitsStatus, NewValue *SpaceLimitsStatus) *MemberSpaceLimitsChangeStatusDetails {
	s := new(MemberSpaceLimitsChangeStatusDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// MemberSpaceLimitsChangeStatusType : has no documentation (yet)
type MemberSpaceLimitsChangeStatusType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSpaceLimitsChangeStatusType returns a new MemberSpaceLimitsChangeStatusType instance
func NewMemberSpaceLimitsChangeStatusType(Description string) *MemberSpaceLimitsChangeStatusType {
	s := new(MemberSpaceLimitsChangeStatusType)
	s.Description = Description
	return s
}

// MemberSpaceLimitsRemoveCustomQuotaDetails : Removed custom member space
// limit.
type MemberSpaceLimitsRemoveCustomQuotaDetails struct {
}

// NewMemberSpaceLimitsRemoveCustomQuotaDetails returns a new MemberSpaceLimitsRemoveCustomQuotaDetails instance
func NewMemberSpaceLimitsRemoveCustomQuotaDetails() *MemberSpaceLimitsRemoveCustomQuotaDetails {
	s := new(MemberSpaceLimitsRemoveCustomQuotaDetails)
	return s
}

// MemberSpaceLimitsRemoveCustomQuotaType : has no documentation (yet)
type MemberSpaceLimitsRemoveCustomQuotaType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSpaceLimitsRemoveCustomQuotaType returns a new MemberSpaceLimitsRemoveCustomQuotaType instance
func NewMemberSpaceLimitsRemoveCustomQuotaType(Description string) *MemberSpaceLimitsRemoveCustomQuotaType {
	s := new(MemberSpaceLimitsRemoveCustomQuotaType)
	s.Description = Description
	return s
}

// MemberSpaceLimitsRemoveExceptionDetails : Removed members from member space
// limit exception list.
type MemberSpaceLimitsRemoveExceptionDetails struct {
}

// NewMemberSpaceLimitsRemoveExceptionDetails returns a new MemberSpaceLimitsRemoveExceptionDetails instance
func NewMemberSpaceLimitsRemoveExceptionDetails() *MemberSpaceLimitsRemoveExceptionDetails {
	s := new(MemberSpaceLimitsRemoveExceptionDetails)
	return s
}

// MemberSpaceLimitsRemoveExceptionType : has no documentation (yet)
type MemberSpaceLimitsRemoveExceptionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSpaceLimitsRemoveExceptionType returns a new MemberSpaceLimitsRemoveExceptionType instance
func NewMemberSpaceLimitsRemoveExceptionType(Description string) *MemberSpaceLimitsRemoveExceptionType {
	s := new(MemberSpaceLimitsRemoveExceptionType)
	s.Description = Description
	return s
}

// MemberStatus : has no documentation (yet)
type MemberStatus struct {
	dropbox.Tagged
}

// Valid tag values for MemberStatus
const (
	MemberStatusActive             = "active"
	MemberStatusInvited            = "invited"
	MemberStatusMovedToAnotherTeam = "moved_to_another_team"
	MemberStatusNotJoined          = "not_joined"
	MemberStatusRemoved            = "removed"
	MemberStatusSuspended          = "suspended"
	MemberStatusOther              = "other"
)

// MemberSuggestDetails : Suggested person to add to team.
type MemberSuggestDetails struct {
	// SuggestedMembers : suggested users emails.
	SuggestedMembers []string `json:"suggested_members"`
}

// NewMemberSuggestDetails returns a new MemberSuggestDetails instance
func NewMemberSuggestDetails(SuggestedMembers []string) *MemberSuggestDetails {
	s := new(MemberSuggestDetails)
	s.SuggestedMembers = SuggestedMembers
	return s
}

// MemberSuggestType : has no documentation (yet)
type MemberSuggestType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSuggestType returns a new MemberSuggestType instance
func NewMemberSuggestType(Description string) *MemberSuggestType {
	s := new(MemberSuggestType)
	s.Description = Description
	return s
}

// MemberSuggestionsChangePolicyDetails : Enabled/disabled option for team
// members to suggest people to add to team.
type MemberSuggestionsChangePolicyDetails struct {
	// NewValue : New team member suggestions policy.
	NewValue *MemberSuggestionsPolicy `json:"new_value"`
	// PreviousValue : Previous team member suggestions policy. Might be missing
	// due to historical data gap.
	PreviousValue *MemberSuggestionsPolicy `json:"previous_value,omitempty"`
}

// NewMemberSuggestionsChangePolicyDetails returns a new MemberSuggestionsChangePolicyDetails instance
func NewMemberSuggestionsChangePolicyDetails(NewValue *MemberSuggestionsPolicy) *MemberSuggestionsChangePolicyDetails {
	s := new(MemberSuggestionsChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// MemberSuggestionsChangePolicyType : has no documentation (yet)
type MemberSuggestionsChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberSuggestionsChangePolicyType returns a new MemberSuggestionsChangePolicyType instance
func NewMemberSuggestionsChangePolicyType(Description string) *MemberSuggestionsChangePolicyType {
	s := new(MemberSuggestionsChangePolicyType)
	s.Description = Description
	return s
}

// MemberSuggestionsPolicy : Member suggestions policy
type MemberSuggestionsPolicy struct {
	dropbox.Tagged
}

// Valid tag values for MemberSuggestionsPolicy
const (
	MemberSuggestionsPolicyDisabled = "disabled"
	MemberSuggestionsPolicyEnabled  = "enabled"
	MemberSuggestionsPolicyOther    = "other"
)

// MemberTransferAccountContentsDetails : Transferred contents of deleted member
// account to another member.
type MemberTransferAccountContentsDetails struct {
}

// NewMemberTransferAccountContentsDetails returns a new MemberTransferAccountContentsDetails instance
func NewMemberTransferAccountContentsDetails() *MemberTransferAccountContentsDetails {
	s := new(MemberTransferAccountContentsDetails)
	return s
}

// MemberTransferAccountContentsType : has no documentation (yet)
type MemberTransferAccountContentsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMemberTransferAccountContentsType returns a new MemberTransferAccountContentsType instance
func NewMemberTransferAccountContentsType(Description string) *MemberTransferAccountContentsType {
	s := new(MemberTransferAccountContentsType)
	s.Description = Description
	return s
}

// MemberTransferredInternalFields : Internal only - fields for target team
// computations
type MemberTransferredInternalFields struct {
	// SourceTeamId : Internal only - team user was moved from.
	SourceTeamId string `json:"source_team_id"`
	// TargetTeamId : Internal only - team user was moved to.
	TargetTeamId string `json:"target_team_id"`
}

// NewMemberTransferredInternalFields returns a new MemberTransferredInternalFields instance
func NewMemberTransferredInternalFields(SourceTeamId string, TargetTeamId string) *MemberTransferredInternalFields {
	s := new(MemberTransferredInternalFields)
	s.SourceTeamId = SourceTeamId
	s.TargetTeamId = TargetTeamId
	return s
}

// MicrosoftOfficeAddinChangePolicyDetails : Enabled/disabled Microsoft Office
// add-in.
type MicrosoftOfficeAddinChangePolicyDetails struct {
	// NewValue : New Microsoft Office addin policy.
	NewValue *MicrosoftOfficeAddinPolicy `json:"new_value"`
	// PreviousValue : Previous Microsoft Office addin policy. Might be missing
	// due to historical data gap.
	PreviousValue *MicrosoftOfficeAddinPolicy `json:"previous_value,omitempty"`
}

// NewMicrosoftOfficeAddinChangePolicyDetails returns a new MicrosoftOfficeAddinChangePolicyDetails instance
func NewMicrosoftOfficeAddinChangePolicyDetails(NewValue *MicrosoftOfficeAddinPolicy) *MicrosoftOfficeAddinChangePolicyDetails {
	s := new(MicrosoftOfficeAddinChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// MicrosoftOfficeAddinChangePolicyType : has no documentation (yet)
type MicrosoftOfficeAddinChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewMicrosoftOfficeAddinChangePolicyType returns a new MicrosoftOfficeAddinChangePolicyType instance
func NewMicrosoftOfficeAddinChangePolicyType(Description string) *MicrosoftOfficeAddinChangePolicyType {
	s := new(MicrosoftOfficeAddinChangePolicyType)
	s.Description = Description
	return s
}

// MicrosoftOfficeAddinPolicy : Microsoft Office addin policy
type MicrosoftOfficeAddinPolicy struct {
	dropbox.Tagged
}

// Valid tag values for MicrosoftOfficeAddinPolicy
const (
	MicrosoftOfficeAddinPolicyDisabled = "disabled"
	MicrosoftOfficeAddinPolicyEnabled  = "enabled"
	MicrosoftOfficeAddinPolicyOther    = "other"
)

// MissingDetails : An indication that an error occurred while retrieving the
// event. Some attributes of the event may be omitted as a result.
type MissingDetails struct {
	// SourceEventFields : All the data that could be retrieved and converted
	// from the source event.
	SourceEventFields string `json:"source_event_fields,omitempty"`
}

// NewMissingDetails returns a new MissingDetails instance
func NewMissingDetails() *MissingDetails {
	s := new(MissingDetails)
	return s
}

// MobileDeviceSessionLogInfo : Information about linked Dropbox mobile client
// sessions
type MobileDeviceSessionLogInfo struct {
	DeviceSessionLogInfo
	// SessionInfo : Mobile session unique id.
	SessionInfo *MobileSessionLogInfo `json:"session_info,omitempty"`
	// DeviceName : The device name.
	DeviceName string `json:"device_name"`
	// ClientType : The mobile application type.
	ClientType *team.MobileClientPlatform `json:"client_type"`
	// ClientVersion : The Dropbox client version.
	ClientVersion string `json:"client_version,omitempty"`
	// OsVersion : The hosting OS version.
	OsVersion string `json:"os_version,omitempty"`
	// LastCarrier : last carrier used by the device.
	LastCarrier string `json:"last_carrier,omitempty"`
}

// NewMobileDeviceSessionLogInfo returns a new MobileDeviceSessionLogInfo instance
func NewMobileDeviceSessionLogInfo(DeviceName string, ClientType *team.MobileClientPlatform) *MobileDeviceSessionLogInfo {
	s := new(MobileDeviceSessionLogInfo)
	s.DeviceName = DeviceName
	s.ClientType = ClientType
	return s
}

// MobileSessionLogInfo : Mobile session.
type MobileSessionLogInfo struct {
	SessionLogInfo
}

// NewMobileSessionLogInfo returns a new MobileSessionLogInfo instance
func NewMobileSessionLogInfo() *MobileSessionLogInfo {
	s := new(MobileSessionLogInfo)
	return s
}

// NamespaceRelativePathLogInfo : Namespace relative path details.
type NamespaceRelativePathLogInfo struct {
	// NsId : Namespace ID.
	NsId string `json:"ns_id,omitempty"`
	// RelativePath : A path relative to the specified namespace ID.
	RelativePath string `json:"relative_path,omitempty"`
	// IsSharedNamespace : True if the namespace is shared.
	IsSharedNamespace bool `json:"is_shared_namespace,omitempty"`
}

// NewNamespaceRelativePathLogInfo returns a new NamespaceRelativePathLogInfo instance
func NewNamespaceRelativePathLogInfo() *NamespaceRelativePathLogInfo {
	s := new(NamespaceRelativePathLogInfo)
	return s
}

// NetworkControlChangePolicyDetails : Enabled/disabled network control.
type NetworkControlChangePolicyDetails struct {
	// NewValue : New network control policy.
	NewValue *NetworkControlPolicy `json:"new_value"`
	// PreviousValue : Previous network control policy. Might be missing due to
	// historical data gap.
	PreviousValue *NetworkControlPolicy `json:"previous_value,omitempty"`
}

// NewNetworkControlChangePolicyDetails returns a new NetworkControlChangePolicyDetails instance
func NewNetworkControlChangePolicyDetails(NewValue *NetworkControlPolicy) *NetworkControlChangePolicyDetails {
	s := new(NetworkControlChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// NetworkControlChangePolicyType : has no documentation (yet)
type NetworkControlChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNetworkControlChangePolicyType returns a new NetworkControlChangePolicyType instance
func NewNetworkControlChangePolicyType(Description string) *NetworkControlChangePolicyType {
	s := new(NetworkControlChangePolicyType)
	s.Description = Description
	return s
}

// NetworkControlPolicy : Network control policy
type NetworkControlPolicy struct {
	dropbox.Tagged
}

// Valid tag values for NetworkControlPolicy
const (
	NetworkControlPolicyDisabled = "disabled"
	NetworkControlPolicyEnabled  = "enabled"
	NetworkControlPolicyOther    = "other"
)

// NoExpirationLinkGenCreateReportDetails : Report created: Links created with
// no expiration.
type NoExpirationLinkGenCreateReportDetails struct {
	// StartDate : Report start date.
	StartDate time.Time `json:"start_date"`
	// EndDate : Report end date.
	EndDate time.Time `json:"end_date"`
}

// NewNoExpirationLinkGenCreateReportDetails returns a new NoExpirationLinkGenCreateReportDetails instance
func NewNoExpirationLinkGenCreateReportDetails(StartDate time.Time, EndDate time.Time) *NoExpirationLinkGenCreateReportDetails {
	s := new(NoExpirationLinkGenCreateReportDetails)
	s.StartDate = StartDate
	s.EndDate = EndDate
	return s
}

// NoExpirationLinkGenCreateReportType : has no documentation (yet)
type NoExpirationLinkGenCreateReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoExpirationLinkGenCreateReportType returns a new NoExpirationLinkGenCreateReportType instance
func NewNoExpirationLinkGenCreateReportType(Description string) *NoExpirationLinkGenCreateReportType {
	s := new(NoExpirationLinkGenCreateReportType)
	s.Description = Description
	return s
}

// NoExpirationLinkGenReportFailedDetails : Couldn't create report: Links
// created with no expiration.
type NoExpirationLinkGenReportFailedDetails struct {
	// FailureReason : Failure reason.
	FailureReason *team.TeamReportFailureReason `json:"failure_reason"`
}

// NewNoExpirationLinkGenReportFailedDetails returns a new NoExpirationLinkGenReportFailedDetails instance
func NewNoExpirationLinkGenReportFailedDetails(FailureReason *team.TeamReportFailureReason) *NoExpirationLinkGenReportFailedDetails {
	s := new(NoExpirationLinkGenReportFailedDetails)
	s.FailureReason = FailureReason
	return s
}

// NoExpirationLinkGenReportFailedType : has no documentation (yet)
type NoExpirationLinkGenReportFailedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoExpirationLinkGenReportFailedType returns a new NoExpirationLinkGenReportFailedType instance
func NewNoExpirationLinkGenReportFailedType(Description string) *NoExpirationLinkGenReportFailedType {
	s := new(NoExpirationLinkGenReportFailedType)
	s.Description = Description
	return s
}

// NoPasswordLinkGenCreateReportDetails : Report created: Links created without
// passwords.
type NoPasswordLinkGenCreateReportDetails struct {
	// StartDate : Report start date.
	StartDate time.Time `json:"start_date"`
	// EndDate : Report end date.
	EndDate time.Time `json:"end_date"`
}

// NewNoPasswordLinkGenCreateReportDetails returns a new NoPasswordLinkGenCreateReportDetails instance
func NewNoPasswordLinkGenCreateReportDetails(StartDate time.Time, EndDate time.Time) *NoPasswordLinkGenCreateReportDetails {
	s := new(NoPasswordLinkGenCreateReportDetails)
	s.StartDate = StartDate
	s.EndDate = EndDate
	return s
}

// NoPasswordLinkGenCreateReportType : has no documentation (yet)
type NoPasswordLinkGenCreateReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoPasswordLinkGenCreateReportType returns a new NoPasswordLinkGenCreateReportType instance
func NewNoPasswordLinkGenCreateReportType(Description string) *NoPasswordLinkGenCreateReportType {
	s := new(NoPasswordLinkGenCreateReportType)
	s.Description = Description
	return s
}

// NoPasswordLinkGenReportFailedDetails : Couldn't create report: Links created
// without passwords.
type NoPasswordLinkGenReportFailedDetails struct {
	// FailureReason : Failure reason.
	FailureReason *team.TeamReportFailureReason `json:"failure_reason"`
}

// NewNoPasswordLinkGenReportFailedDetails returns a new NoPasswordLinkGenReportFailedDetails instance
func NewNoPasswordLinkGenReportFailedDetails(FailureReason *team.TeamReportFailureReason) *NoPasswordLinkGenReportFailedDetails {
	s := new(NoPasswordLinkGenReportFailedDetails)
	s.FailureReason = FailureReason
	return s
}

// NoPasswordLinkGenReportFailedType : has no documentation (yet)
type NoPasswordLinkGenReportFailedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoPasswordLinkGenReportFailedType returns a new NoPasswordLinkGenReportFailedType instance
func NewNoPasswordLinkGenReportFailedType(Description string) *NoPasswordLinkGenReportFailedType {
	s := new(NoPasswordLinkGenReportFailedType)
	s.Description = Description
	return s
}

// NoPasswordLinkViewCreateReportDetails : Report created: Views of links
// without passwords.
type NoPasswordLinkViewCreateReportDetails struct {
	// StartDate : Report start date.
	StartDate time.Time `json:"start_date"`
	// EndDate : Report end date.
	EndDate time.Time `json:"end_date"`
}

// NewNoPasswordLinkViewCreateReportDetails returns a new NoPasswordLinkViewCreateReportDetails instance
func NewNoPasswordLinkViewCreateReportDetails(StartDate time.Time, EndDate time.Time) *NoPasswordLinkViewCreateReportDetails {
	s := new(NoPasswordLinkViewCreateReportDetails)
	s.StartDate = StartDate
	s.EndDate = EndDate
	return s
}

// NoPasswordLinkViewCreateReportType : has no documentation (yet)
type NoPasswordLinkViewCreateReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoPasswordLinkViewCreateReportType returns a new NoPasswordLinkViewCreateReportType instance
func NewNoPasswordLinkViewCreateReportType(Description string) *NoPasswordLinkViewCreateReportType {
	s := new(NoPasswordLinkViewCreateReportType)
	s.Description = Description
	return s
}

// NoPasswordLinkViewReportFailedDetails : Couldn't create report: Views of
// links without passwords.
type NoPasswordLinkViewReportFailedDetails struct {
	// FailureReason : Failure reason.
	FailureReason *team.TeamReportFailureReason `json:"failure_reason"`
}

// NewNoPasswordLinkViewReportFailedDetails returns a new NoPasswordLinkViewReportFailedDetails instance
func NewNoPasswordLinkViewReportFailedDetails(FailureReason *team.TeamReportFailureReason) *NoPasswordLinkViewReportFailedDetails {
	s := new(NoPasswordLinkViewReportFailedDetails)
	s.FailureReason = FailureReason
	return s
}

// NoPasswordLinkViewReportFailedType : has no documentation (yet)
type NoPasswordLinkViewReportFailedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoPasswordLinkViewReportFailedType returns a new NoPasswordLinkViewReportFailedType instance
func NewNoPasswordLinkViewReportFailedType(Description string) *NoPasswordLinkViewReportFailedType {
	s := new(NoPasswordLinkViewReportFailedType)
	s.Description = Description
	return s
}

// UserLogInfo : User's logged information.
type UserLogInfo struct {
	// AccountId : User unique ID.
	AccountId string `json:"account_id,omitempty"`
	// DisplayName : User display name.
	DisplayName string `json:"display_name,omitempty"`
	// Email : User email address.
	Email string `json:"email,omitempty"`
}

// NewUserLogInfo returns a new UserLogInfo instance
func NewUserLogInfo() *UserLogInfo {
	s := new(UserLogInfo)
	return s
}

// IsUserLogInfo is the interface type for UserLogInfo and its subtypes
type IsUserLogInfo interface {
	IsUserLogInfo()
}

// IsUserLogInfo implements the IsUserLogInfo interface
func (u *UserLogInfo) IsUserLogInfo() {}

type userLogInfoUnion struct {
	dropbox.Tagged
	// TeamMember : has no documentation (yet)
	TeamMember *TeamMemberLogInfo `json:"team_member,omitempty"`
	// TrustedNonTeamMember : has no documentation (yet)
	TrustedNonTeamMember *TrustedNonTeamMemberLogInfo `json:"trusted_non_team_member,omitempty"`
	// NonTeamMember : has no documentation (yet)
	NonTeamMember *NonTeamMemberLogInfo `json:"non_team_member,omitempty"`
}

// Valid tag values for UserLogInfo
const (
	UserLogInfoTeamMember           = "team_member"
	UserLogInfoTrustedNonTeamMember = "trusted_non_team_member"
	UserLogInfoNonTeamMember        = "non_team_member"
)

// UnmarshalJSON deserializes into a userLogInfoUnion instance
func (u *userLogInfoUnion) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "team_member":
		if err = json.Unmarshal(body, &u.TeamMember); err != nil {
			return err
		}

	case "trusted_non_team_member":
		if err = json.Unmarshal(body, &u.TrustedNonTeamMember); err != nil {
			return err
		}

	case "non_team_member":
		if err = json.Unmarshal(body, &u.NonTeamMember); err != nil {
			return err
		}

	}
	return nil
}

// IsUserLogInfoFromJSON converts JSON to a concrete IsUserLogInfo instance
func IsUserLogInfoFromJSON(data []byte) (IsUserLogInfo, error) {
	var t userLogInfoUnion
	if err := json.Unmarshal(data, &t); err != nil {
		return nil, err
	}
	switch t.Tag {
	case "team_member":
		return t.TeamMember, nil

	case "trusted_non_team_member":
		return t.TrustedNonTeamMember, nil

	case "non_team_member":
		return t.NonTeamMember, nil

	}
	return nil, nil
}

// NonTeamMemberLogInfo : Non team member's logged information.
type NonTeamMemberLogInfo struct {
	UserLogInfo
}

// NewNonTeamMemberLogInfo returns a new NonTeamMemberLogInfo instance
func NewNonTeamMemberLogInfo() *NonTeamMemberLogInfo {
	s := new(NonTeamMemberLogInfo)
	return s
}

// NonTrustedTeamDetails : The email to which the request was sent
type NonTrustedTeamDetails struct {
	// Team : The email to which the request was sent.
	Team string `json:"team"`
}

// NewNonTrustedTeamDetails returns a new NonTrustedTeamDetails instance
func NewNonTrustedTeamDetails(Team string) *NonTrustedTeamDetails {
	s := new(NonTrustedTeamDetails)
	s.Team = Team
	return s
}

// NoteAclInviteOnlyDetails : Changed Paper doc to invite-only.
type NoteAclInviteOnlyDetails struct {
}

// NewNoteAclInviteOnlyDetails returns a new NoteAclInviteOnlyDetails instance
func NewNoteAclInviteOnlyDetails() *NoteAclInviteOnlyDetails {
	s := new(NoteAclInviteOnlyDetails)
	return s
}

// NoteAclInviteOnlyType : has no documentation (yet)
type NoteAclInviteOnlyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoteAclInviteOnlyType returns a new NoteAclInviteOnlyType instance
func NewNoteAclInviteOnlyType(Description string) *NoteAclInviteOnlyType {
	s := new(NoteAclInviteOnlyType)
	s.Description = Description
	return s
}

// NoteAclLinkDetails : Changed Paper doc to link-accessible.
type NoteAclLinkDetails struct {
}

// NewNoteAclLinkDetails returns a new NoteAclLinkDetails instance
func NewNoteAclLinkDetails() *NoteAclLinkDetails {
	s := new(NoteAclLinkDetails)
	return s
}

// NoteAclLinkType : has no documentation (yet)
type NoteAclLinkType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoteAclLinkType returns a new NoteAclLinkType instance
func NewNoteAclLinkType(Description string) *NoteAclLinkType {
	s := new(NoteAclLinkType)
	s.Description = Description
	return s
}

// NoteAclTeamLinkDetails : Changed Paper doc to link-accessible for team.
type NoteAclTeamLinkDetails struct {
}

// NewNoteAclTeamLinkDetails returns a new NoteAclTeamLinkDetails instance
func NewNoteAclTeamLinkDetails() *NoteAclTeamLinkDetails {
	s := new(NoteAclTeamLinkDetails)
	return s
}

// NoteAclTeamLinkType : has no documentation (yet)
type NoteAclTeamLinkType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoteAclTeamLinkType returns a new NoteAclTeamLinkType instance
func NewNoteAclTeamLinkType(Description string) *NoteAclTeamLinkType {
	s := new(NoteAclTeamLinkType)
	s.Description = Description
	return s
}

// NoteShareReceiveDetails : Shared received Paper doc.
type NoteShareReceiveDetails struct {
}

// NewNoteShareReceiveDetails returns a new NoteShareReceiveDetails instance
func NewNoteShareReceiveDetails() *NoteShareReceiveDetails {
	s := new(NoteShareReceiveDetails)
	return s
}

// NoteShareReceiveType : has no documentation (yet)
type NoteShareReceiveType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoteShareReceiveType returns a new NoteShareReceiveType instance
func NewNoteShareReceiveType(Description string) *NoteShareReceiveType {
	s := new(NoteShareReceiveType)
	s.Description = Description
	return s
}

// NoteSharedDetails : Shared Paper doc.
type NoteSharedDetails struct {
}

// NewNoteSharedDetails returns a new NoteSharedDetails instance
func NewNoteSharedDetails() *NoteSharedDetails {
	s := new(NoteSharedDetails)
	return s
}

// NoteSharedType : has no documentation (yet)
type NoteSharedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewNoteSharedType returns a new NoteSharedType instance
func NewNoteSharedType(Description string) *NoteSharedType {
	s := new(NoteSharedType)
	s.Description = Description
	return s
}

// ObjectLabelAddedDetails : Added a label.
type ObjectLabelAddedDetails struct {
	// LabelType : Labels mark a file or folder.
	LabelType *LabelType `json:"label_type"`
}

// NewObjectLabelAddedDetails returns a new ObjectLabelAddedDetails instance
func NewObjectLabelAddedDetails(LabelType *LabelType) *ObjectLabelAddedDetails {
	s := new(ObjectLabelAddedDetails)
	s.LabelType = LabelType
	return s
}

// ObjectLabelAddedType : has no documentation (yet)
type ObjectLabelAddedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewObjectLabelAddedType returns a new ObjectLabelAddedType instance
func NewObjectLabelAddedType(Description string) *ObjectLabelAddedType {
	s := new(ObjectLabelAddedType)
	s.Description = Description
	return s
}

// ObjectLabelRemovedDetails : Removed a label.
type ObjectLabelRemovedDetails struct {
	// LabelType : Labels mark a file or folder.
	LabelType *LabelType `json:"label_type"`
}

// NewObjectLabelRemovedDetails returns a new ObjectLabelRemovedDetails instance
func NewObjectLabelRemovedDetails(LabelType *LabelType) *ObjectLabelRemovedDetails {
	s := new(ObjectLabelRemovedDetails)
	s.LabelType = LabelType
	return s
}

// ObjectLabelRemovedType : has no documentation (yet)
type ObjectLabelRemovedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewObjectLabelRemovedType returns a new ObjectLabelRemovedType instance
func NewObjectLabelRemovedType(Description string) *ObjectLabelRemovedType {
	s := new(ObjectLabelRemovedType)
	s.Description = Description
	return s
}

// ObjectLabelUpdatedValueDetails : Updated a label's value.
type ObjectLabelUpdatedValueDetails struct {
	// LabelType : Labels mark a file or folder.
	LabelType *LabelType `json:"label_type"`
}

// NewObjectLabelUpdatedValueDetails returns a new ObjectLabelUpdatedValueDetails instance
func NewObjectLabelUpdatedValueDetails(LabelType *LabelType) *ObjectLabelUpdatedValueDetails {
	s := new(ObjectLabelUpdatedValueDetails)
	s.LabelType = LabelType
	return s
}

// ObjectLabelUpdatedValueType : has no documentation (yet)
type ObjectLabelUpdatedValueType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewObjectLabelUpdatedValueType returns a new ObjectLabelUpdatedValueType instance
func NewObjectLabelUpdatedValueType(Description string) *ObjectLabelUpdatedValueType {
	s := new(ObjectLabelUpdatedValueType)
	s.Description = Description
	return s
}

// OpenNoteSharedDetails : Opened shared Paper doc.
type OpenNoteSharedDetails struct {
}

// NewOpenNoteSharedDetails returns a new OpenNoteSharedDetails instance
func NewOpenNoteSharedDetails() *OpenNoteSharedDetails {
	s := new(OpenNoteSharedDetails)
	return s
}

// OpenNoteSharedType : has no documentation (yet)
type OpenNoteSharedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewOpenNoteSharedType returns a new OpenNoteSharedType instance
func NewOpenNoteSharedType(Description string) *OpenNoteSharedType {
	s := new(OpenNoteSharedType)
	s.Description = Description
	return s
}

// OrganizationDetails : More details about the organization.
type OrganizationDetails struct {
	// Organization : The name of the organization.
	Organization string `json:"organization"`
}

// NewOrganizationDetails returns a new OrganizationDetails instance
func NewOrganizationDetails(Organization string) *OrganizationDetails {
	s := new(OrganizationDetails)
	s.Organization = Organization
	return s
}

// OrganizationName : The name of the organization
type OrganizationName struct {
	// Organization : The name of the organization.
	Organization string `json:"organization"`
}

// NewOrganizationName returns a new OrganizationName instance
func NewOrganizationName(Organization string) *OrganizationName {
	s := new(OrganizationName)
	s.Organization = Organization
	return s
}

// OrganizeFolderWithTidyDetails : Organized a folder with multi-file organize.
type OrganizeFolderWithTidyDetails struct {
}

// NewOrganizeFolderWithTidyDetails returns a new OrganizeFolderWithTidyDetails instance
func NewOrganizeFolderWithTidyDetails() *OrganizeFolderWithTidyDetails {
	s := new(OrganizeFolderWithTidyDetails)
	return s
}

// OrganizeFolderWithTidyType : has no documentation (yet)
type OrganizeFolderWithTidyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewOrganizeFolderWithTidyType returns a new OrganizeFolderWithTidyType instance
func NewOrganizeFolderWithTidyType(Description string) *OrganizeFolderWithTidyType {
	s := new(OrganizeFolderWithTidyType)
	s.Description = Description
	return s
}

// OriginLogInfo : The origin from which the actor performed the action.
type OriginLogInfo struct {
	// GeoLocation : Geographic location details.
	GeoLocation *GeoLocationLogInfo `json:"geo_location,omitempty"`
	// AccessMethod : The method that was used to perform the action.
	AccessMethod *AccessMethodLogInfo `json:"access_method"`
}

// NewOriginLogInfo returns a new OriginLogInfo instance
func NewOriginLogInfo(AccessMethod *AccessMethodLogInfo) *OriginLogInfo {
	s := new(OriginLogInfo)
	s.AccessMethod = AccessMethod
	return s
}

// OutdatedLinkViewCreateReportDetails : Report created: Views of old links.
type OutdatedLinkViewCreateReportDetails struct {
	// StartDate : Report start date.
	StartDate time.Time `json:"start_date"`
	// EndDate : Report end date.
	EndDate time.Time `json:"end_date"`
}

// NewOutdatedLinkViewCreateReportDetails returns a new OutdatedLinkViewCreateReportDetails instance
func NewOutdatedLinkViewCreateReportDetails(StartDate time.Time, EndDate time.Time) *OutdatedLinkViewCreateReportDetails {
	s := new(OutdatedLinkViewCreateReportDetails)
	s.StartDate = StartDate
	s.EndDate = EndDate
	return s
}

// OutdatedLinkViewCreateReportType : has no documentation (yet)
type OutdatedLinkViewCreateReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewOutdatedLinkViewCreateReportType returns a new OutdatedLinkViewCreateReportType instance
func NewOutdatedLinkViewCreateReportType(Description string) *OutdatedLinkViewCreateReportType {
	s := new(OutdatedLinkViewCreateReportType)
	s.Description = Description
	return s
}

// OutdatedLinkViewReportFailedDetails : Couldn't create report: Views of old
// links.
type OutdatedLinkViewReportFailedDetails struct {
	// FailureReason : Failure reason.
	FailureReason *team.TeamReportFailureReason `json:"failure_reason"`
}

// NewOutdatedLinkViewReportFailedDetails returns a new OutdatedLinkViewReportFailedDetails instance
func NewOutdatedLinkViewReportFailedDetails(FailureReason *team.TeamReportFailureReason) *OutdatedLinkViewReportFailedDetails {
	s := new(OutdatedLinkViewReportFailedDetails)
	s.FailureReason = FailureReason
	return s
}

// OutdatedLinkViewReportFailedType : has no documentation (yet)
type OutdatedLinkViewReportFailedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewOutdatedLinkViewReportFailedType returns a new OutdatedLinkViewReportFailedType instance
func NewOutdatedLinkViewReportFailedType(Description string) *OutdatedLinkViewReportFailedType {
	s := new(OutdatedLinkViewReportFailedType)
	s.Description = Description
	return s
}

// PaperAccessType : has no documentation (yet)
type PaperAccessType struct {
	dropbox.Tagged
}

// Valid tag values for PaperAccessType
const (
	PaperAccessTypeCommenter = "commenter"
	PaperAccessTypeEditor    = "editor"
	PaperAccessTypeViewer    = "viewer"
	PaperAccessTypeOther     = "other"
)

// PaperAdminExportStartDetails : Exported all team Paper docs.
type PaperAdminExportStartDetails struct {
}

// NewPaperAdminExportStartDetails returns a new PaperAdminExportStartDetails instance
func NewPaperAdminExportStartDetails() *PaperAdminExportStartDetails {
	s := new(PaperAdminExportStartDetails)
	return s
}

// PaperAdminExportStartType : has no documentation (yet)
type PaperAdminExportStartType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperAdminExportStartType returns a new PaperAdminExportStartType instance
func NewPaperAdminExportStartType(Description string) *PaperAdminExportStartType {
	s := new(PaperAdminExportStartType)
	s.Description = Description
	return s
}

// PaperChangeDeploymentPolicyDetails : Changed whether Dropbox Paper, when
// enabled, is deployed to all members or to specific members.
type PaperChangeDeploymentPolicyDetails struct {
	// NewValue : New Dropbox Paper deployment policy.
	NewValue *team_policies.PaperDeploymentPolicy `json:"new_value"`
	// PreviousValue : Previous Dropbox Paper deployment policy. Might be
	// missing due to historical data gap.
	PreviousValue *team_policies.PaperDeploymentPolicy `json:"previous_value,omitempty"`
}

// NewPaperChangeDeploymentPolicyDetails returns a new PaperChangeDeploymentPolicyDetails instance
func NewPaperChangeDeploymentPolicyDetails(NewValue *team_policies.PaperDeploymentPolicy) *PaperChangeDeploymentPolicyDetails {
	s := new(PaperChangeDeploymentPolicyDetails)
	s.NewValue = NewValue
	return s
}

// PaperChangeDeploymentPolicyType : has no documentation (yet)
type PaperChangeDeploymentPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperChangeDeploymentPolicyType returns a new PaperChangeDeploymentPolicyType instance
func NewPaperChangeDeploymentPolicyType(Description string) *PaperChangeDeploymentPolicyType {
	s := new(PaperChangeDeploymentPolicyType)
	s.Description = Description
	return s
}

// PaperChangeMemberLinkPolicyDetails : Changed whether non-members can view
// Paper docs with link.
type PaperChangeMemberLinkPolicyDetails struct {
	// NewValue : New paper external link accessibility policy.
	NewValue *PaperMemberPolicy `json:"new_value"`
}

// NewPaperChangeMemberLinkPolicyDetails returns a new PaperChangeMemberLinkPolicyDetails instance
func NewPaperChangeMemberLinkPolicyDetails(NewValue *PaperMemberPolicy) *PaperChangeMemberLinkPolicyDetails {
	s := new(PaperChangeMemberLinkPolicyDetails)
	s.NewValue = NewValue
	return s
}

// PaperChangeMemberLinkPolicyType : has no documentation (yet)
type PaperChangeMemberLinkPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperChangeMemberLinkPolicyType returns a new PaperChangeMemberLinkPolicyType instance
func NewPaperChangeMemberLinkPolicyType(Description string) *PaperChangeMemberLinkPolicyType {
	s := new(PaperChangeMemberLinkPolicyType)
	s.Description = Description
	return s
}

// PaperChangeMemberPolicyDetails : Changed whether members can share Paper docs
// outside team, and if docs are accessible only by team members or anyone by
// default.
type PaperChangeMemberPolicyDetails struct {
	// NewValue : New paper external accessibility policy.
	NewValue *PaperMemberPolicy `json:"new_value"`
	// PreviousValue : Previous paper external accessibility policy. Might be
	// missing due to historical data gap.
	PreviousValue *PaperMemberPolicy `json:"previous_value,omitempty"`
}

// NewPaperChangeMemberPolicyDetails returns a new PaperChangeMemberPolicyDetails instance
func NewPaperChangeMemberPolicyDetails(NewValue *PaperMemberPolicy) *PaperChangeMemberPolicyDetails {
	s := new(PaperChangeMemberPolicyDetails)
	s.NewValue = NewValue
	return s
}

// PaperChangeMemberPolicyType : has no documentation (yet)
type PaperChangeMemberPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperChangeMemberPolicyType returns a new PaperChangeMemberPolicyType instance
func NewPaperChangeMemberPolicyType(Description string) *PaperChangeMemberPolicyType {
	s := new(PaperChangeMemberPolicyType)
	s.Description = Description
	return s
}

// PaperChangePolicyDetails : Enabled/disabled Dropbox Paper for team.
type PaperChangePolicyDetails struct {
	// NewValue : New Dropbox Paper policy.
	NewValue *team_policies.PaperEnabledPolicy `json:"new_value"`
	// PreviousValue : Previous Dropbox Paper policy. Might be missing due to
	// historical data gap.
	PreviousValue *team_policies.PaperEnabledPolicy `json:"previous_value,omitempty"`
}

// NewPaperChangePolicyDetails returns a new PaperChangePolicyDetails instance
func NewPaperChangePolicyDetails(NewValue *team_policies.PaperEnabledPolicy) *PaperChangePolicyDetails {
	s := new(PaperChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// PaperChangePolicyType : has no documentation (yet)
type PaperChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperChangePolicyType returns a new PaperChangePolicyType instance
func NewPaperChangePolicyType(Description string) *PaperChangePolicyType {
	s := new(PaperChangePolicyType)
	s.Description = Description
	return s
}

// PaperContentAddMemberDetails : Added users and/or groups to Paper doc/folder.
type PaperContentAddMemberDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperContentAddMemberDetails returns a new PaperContentAddMemberDetails instance
func NewPaperContentAddMemberDetails(EventUuid string) *PaperContentAddMemberDetails {
	s := new(PaperContentAddMemberDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperContentAddMemberType : has no documentation (yet)
type PaperContentAddMemberType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperContentAddMemberType returns a new PaperContentAddMemberType instance
func NewPaperContentAddMemberType(Description string) *PaperContentAddMemberType {
	s := new(PaperContentAddMemberType)
	s.Description = Description
	return s
}

// PaperContentAddToFolderDetails : Added Paper doc/folder to folder.
type PaperContentAddToFolderDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// ParentAssetIndex : Parent asset position in the Assets list.
	ParentAssetIndex uint64 `json:"parent_asset_index"`
}

// NewPaperContentAddToFolderDetails returns a new PaperContentAddToFolderDetails instance
func NewPaperContentAddToFolderDetails(EventUuid string, TargetAssetIndex uint64, ParentAssetIndex uint64) *PaperContentAddToFolderDetails {
	s := new(PaperContentAddToFolderDetails)
	s.EventUuid = EventUuid
	s.TargetAssetIndex = TargetAssetIndex
	s.ParentAssetIndex = ParentAssetIndex
	return s
}

// PaperContentAddToFolderType : has no documentation (yet)
type PaperContentAddToFolderType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperContentAddToFolderType returns a new PaperContentAddToFolderType instance
func NewPaperContentAddToFolderType(Description string) *PaperContentAddToFolderType {
	s := new(PaperContentAddToFolderType)
	s.Description = Description
	return s
}

// PaperContentArchiveDetails : Archived Paper doc/folder.
type PaperContentArchiveDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperContentArchiveDetails returns a new PaperContentArchiveDetails instance
func NewPaperContentArchiveDetails(EventUuid string) *PaperContentArchiveDetails {
	s := new(PaperContentArchiveDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperContentArchiveType : has no documentation (yet)
type PaperContentArchiveType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperContentArchiveType returns a new PaperContentArchiveType instance
func NewPaperContentArchiveType(Description string) *PaperContentArchiveType {
	s := new(PaperContentArchiveType)
	s.Description = Description
	return s
}

// PaperContentCreateDetails : Created Paper doc/folder.
type PaperContentCreateDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperContentCreateDetails returns a new PaperContentCreateDetails instance
func NewPaperContentCreateDetails(EventUuid string) *PaperContentCreateDetails {
	s := new(PaperContentCreateDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperContentCreateType : has no documentation (yet)
type PaperContentCreateType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperContentCreateType returns a new PaperContentCreateType instance
func NewPaperContentCreateType(Description string) *PaperContentCreateType {
	s := new(PaperContentCreateType)
	s.Description = Description
	return s
}

// PaperContentPermanentlyDeleteDetails : Permanently deleted Paper doc/folder.
type PaperContentPermanentlyDeleteDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperContentPermanentlyDeleteDetails returns a new PaperContentPermanentlyDeleteDetails instance
func NewPaperContentPermanentlyDeleteDetails(EventUuid string) *PaperContentPermanentlyDeleteDetails {
	s := new(PaperContentPermanentlyDeleteDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperContentPermanentlyDeleteType : has no documentation (yet)
type PaperContentPermanentlyDeleteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperContentPermanentlyDeleteType returns a new PaperContentPermanentlyDeleteType instance
func NewPaperContentPermanentlyDeleteType(Description string) *PaperContentPermanentlyDeleteType {
	s := new(PaperContentPermanentlyDeleteType)
	s.Description = Description
	return s
}

// PaperContentRemoveFromFolderDetails : Removed Paper doc/folder from folder.
type PaperContentRemoveFromFolderDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index,omitempty"`
	// ParentAssetIndex : Parent asset position in the Assets list.
	ParentAssetIndex uint64 `json:"parent_asset_index,omitempty"`
}

// NewPaperContentRemoveFromFolderDetails returns a new PaperContentRemoveFromFolderDetails instance
func NewPaperContentRemoveFromFolderDetails(EventUuid string) *PaperContentRemoveFromFolderDetails {
	s := new(PaperContentRemoveFromFolderDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperContentRemoveFromFolderType : has no documentation (yet)
type PaperContentRemoveFromFolderType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperContentRemoveFromFolderType returns a new PaperContentRemoveFromFolderType instance
func NewPaperContentRemoveFromFolderType(Description string) *PaperContentRemoveFromFolderType {
	s := new(PaperContentRemoveFromFolderType)
	s.Description = Description
	return s
}

// PaperContentRemoveMemberDetails : Removed users and/or groups from Paper
// doc/folder.
type PaperContentRemoveMemberDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperContentRemoveMemberDetails returns a new PaperContentRemoveMemberDetails instance
func NewPaperContentRemoveMemberDetails(EventUuid string) *PaperContentRemoveMemberDetails {
	s := new(PaperContentRemoveMemberDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperContentRemoveMemberType : has no documentation (yet)
type PaperContentRemoveMemberType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperContentRemoveMemberType returns a new PaperContentRemoveMemberType instance
func NewPaperContentRemoveMemberType(Description string) *PaperContentRemoveMemberType {
	s := new(PaperContentRemoveMemberType)
	s.Description = Description
	return s
}

// PaperContentRenameDetails : Renamed Paper doc/folder.
type PaperContentRenameDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperContentRenameDetails returns a new PaperContentRenameDetails instance
func NewPaperContentRenameDetails(EventUuid string) *PaperContentRenameDetails {
	s := new(PaperContentRenameDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperContentRenameType : has no documentation (yet)
type PaperContentRenameType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperContentRenameType returns a new PaperContentRenameType instance
func NewPaperContentRenameType(Description string) *PaperContentRenameType {
	s := new(PaperContentRenameType)
	s.Description = Description
	return s
}

// PaperContentRestoreDetails : Restored archived Paper doc/folder.
type PaperContentRestoreDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperContentRestoreDetails returns a new PaperContentRestoreDetails instance
func NewPaperContentRestoreDetails(EventUuid string) *PaperContentRestoreDetails {
	s := new(PaperContentRestoreDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperContentRestoreType : has no documentation (yet)
type PaperContentRestoreType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperContentRestoreType returns a new PaperContentRestoreType instance
func NewPaperContentRestoreType(Description string) *PaperContentRestoreType {
	s := new(PaperContentRestoreType)
	s.Description = Description
	return s
}

// PaperDefaultFolderPolicy : Policy to set default access for newly created
// Paper folders.
type PaperDefaultFolderPolicy struct {
	dropbox.Tagged
}

// Valid tag values for PaperDefaultFolderPolicy
const (
	PaperDefaultFolderPolicyEveryoneInTeam = "everyone_in_team"
	PaperDefaultFolderPolicyInviteOnly     = "invite_only"
	PaperDefaultFolderPolicyOther          = "other"
)

// PaperDefaultFolderPolicyChangedDetails : Changed Paper Default Folder Policy
// setting for team.
type PaperDefaultFolderPolicyChangedDetails struct {
	// NewValue : New Paper Default Folder Policy.
	NewValue *PaperDefaultFolderPolicy `json:"new_value"`
	// PreviousValue : Previous Paper Default Folder Policy.
	PreviousValue *PaperDefaultFolderPolicy `json:"previous_value"`
}

// NewPaperDefaultFolderPolicyChangedDetails returns a new PaperDefaultFolderPolicyChangedDetails instance
func NewPaperDefaultFolderPolicyChangedDetails(NewValue *PaperDefaultFolderPolicy, PreviousValue *PaperDefaultFolderPolicy) *PaperDefaultFolderPolicyChangedDetails {
	s := new(PaperDefaultFolderPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// PaperDefaultFolderPolicyChangedType : has no documentation (yet)
type PaperDefaultFolderPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDefaultFolderPolicyChangedType returns a new PaperDefaultFolderPolicyChangedType instance
func NewPaperDefaultFolderPolicyChangedType(Description string) *PaperDefaultFolderPolicyChangedType {
	s := new(PaperDefaultFolderPolicyChangedType)
	s.Description = Description
	return s
}

// PaperDesktopPolicy : Policy for controlling if team members can use Paper
// Desktop
type PaperDesktopPolicy struct {
	dropbox.Tagged
}

// Valid tag values for PaperDesktopPolicy
const (
	PaperDesktopPolicyDisabled = "disabled"
	PaperDesktopPolicyEnabled  = "enabled"
	PaperDesktopPolicyOther    = "other"
)

// PaperDesktopPolicyChangedDetails : Enabled/disabled Paper Desktop for team.
type PaperDesktopPolicyChangedDetails struct {
	// NewValue : New Paper Desktop policy.
	NewValue *PaperDesktopPolicy `json:"new_value"`
	// PreviousValue : Previous Paper Desktop policy.
	PreviousValue *PaperDesktopPolicy `json:"previous_value"`
}

// NewPaperDesktopPolicyChangedDetails returns a new PaperDesktopPolicyChangedDetails instance
func NewPaperDesktopPolicyChangedDetails(NewValue *PaperDesktopPolicy, PreviousValue *PaperDesktopPolicy) *PaperDesktopPolicyChangedDetails {
	s := new(PaperDesktopPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// PaperDesktopPolicyChangedType : has no documentation (yet)
type PaperDesktopPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDesktopPolicyChangedType returns a new PaperDesktopPolicyChangedType instance
func NewPaperDesktopPolicyChangedType(Description string) *PaperDesktopPolicyChangedType {
	s := new(PaperDesktopPolicyChangedType)
	s.Description = Description
	return s
}

// PaperDocAddCommentDetails : Added Paper doc comment.
type PaperDocAddCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewPaperDocAddCommentDetails returns a new PaperDocAddCommentDetails instance
func NewPaperDocAddCommentDetails(EventUuid string) *PaperDocAddCommentDetails {
	s := new(PaperDocAddCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocAddCommentType : has no documentation (yet)
type PaperDocAddCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocAddCommentType returns a new PaperDocAddCommentType instance
func NewPaperDocAddCommentType(Description string) *PaperDocAddCommentType {
	s := new(PaperDocAddCommentType)
	s.Description = Description
	return s
}

// PaperDocChangeMemberRoleDetails : Changed member permissions for Paper doc.
type PaperDocChangeMemberRoleDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// AccessType : Paper doc access type.
	AccessType *PaperAccessType `json:"access_type"`
}

// NewPaperDocChangeMemberRoleDetails returns a new PaperDocChangeMemberRoleDetails instance
func NewPaperDocChangeMemberRoleDetails(EventUuid string, AccessType *PaperAccessType) *PaperDocChangeMemberRoleDetails {
	s := new(PaperDocChangeMemberRoleDetails)
	s.EventUuid = EventUuid
	s.AccessType = AccessType
	return s
}

// PaperDocChangeMemberRoleType : has no documentation (yet)
type PaperDocChangeMemberRoleType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocChangeMemberRoleType returns a new PaperDocChangeMemberRoleType instance
func NewPaperDocChangeMemberRoleType(Description string) *PaperDocChangeMemberRoleType {
	s := new(PaperDocChangeMemberRoleType)
	s.Description = Description
	return s
}

// PaperDocChangeSharingPolicyDetails : Changed sharing setting for Paper doc.
type PaperDocChangeSharingPolicyDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// PublicSharingPolicy : Sharing policy with external users.
	PublicSharingPolicy string `json:"public_sharing_policy,omitempty"`
	// TeamSharingPolicy : Sharing policy with team.
	TeamSharingPolicy string `json:"team_sharing_policy,omitempty"`
}

// NewPaperDocChangeSharingPolicyDetails returns a new PaperDocChangeSharingPolicyDetails instance
func NewPaperDocChangeSharingPolicyDetails(EventUuid string) *PaperDocChangeSharingPolicyDetails {
	s := new(PaperDocChangeSharingPolicyDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocChangeSharingPolicyType : has no documentation (yet)
type PaperDocChangeSharingPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocChangeSharingPolicyType returns a new PaperDocChangeSharingPolicyType instance
func NewPaperDocChangeSharingPolicyType(Description string) *PaperDocChangeSharingPolicyType {
	s := new(PaperDocChangeSharingPolicyType)
	s.Description = Description
	return s
}

// PaperDocChangeSubscriptionDetails : Followed/unfollowed Paper doc.
type PaperDocChangeSubscriptionDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// NewSubscriptionLevel : New doc subscription level.
	NewSubscriptionLevel string `json:"new_subscription_level"`
	// PreviousSubscriptionLevel : Previous doc subscription level. Might be
	// missing due to historical data gap.
	PreviousSubscriptionLevel string `json:"previous_subscription_level,omitempty"`
}

// NewPaperDocChangeSubscriptionDetails returns a new PaperDocChangeSubscriptionDetails instance
func NewPaperDocChangeSubscriptionDetails(EventUuid string, NewSubscriptionLevel string) *PaperDocChangeSubscriptionDetails {
	s := new(PaperDocChangeSubscriptionDetails)
	s.EventUuid = EventUuid
	s.NewSubscriptionLevel = NewSubscriptionLevel
	return s
}

// PaperDocChangeSubscriptionType : has no documentation (yet)
type PaperDocChangeSubscriptionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocChangeSubscriptionType returns a new PaperDocChangeSubscriptionType instance
func NewPaperDocChangeSubscriptionType(Description string) *PaperDocChangeSubscriptionType {
	s := new(PaperDocChangeSubscriptionType)
	s.Description = Description
	return s
}

// PaperDocDeleteCommentDetails : Deleted Paper doc comment.
type PaperDocDeleteCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewPaperDocDeleteCommentDetails returns a new PaperDocDeleteCommentDetails instance
func NewPaperDocDeleteCommentDetails(EventUuid string) *PaperDocDeleteCommentDetails {
	s := new(PaperDocDeleteCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocDeleteCommentType : has no documentation (yet)
type PaperDocDeleteCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocDeleteCommentType returns a new PaperDocDeleteCommentType instance
func NewPaperDocDeleteCommentType(Description string) *PaperDocDeleteCommentType {
	s := new(PaperDocDeleteCommentType)
	s.Description = Description
	return s
}

// PaperDocDeletedDetails : Archived Paper doc.
type PaperDocDeletedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocDeletedDetails returns a new PaperDocDeletedDetails instance
func NewPaperDocDeletedDetails(EventUuid string) *PaperDocDeletedDetails {
	s := new(PaperDocDeletedDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocDeletedType : has no documentation (yet)
type PaperDocDeletedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocDeletedType returns a new PaperDocDeletedType instance
func NewPaperDocDeletedType(Description string) *PaperDocDeletedType {
	s := new(PaperDocDeletedType)
	s.Description = Description
	return s
}

// PaperDocDownloadDetails : Downloaded Paper doc in specific format.
type PaperDocDownloadDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// ExportFileFormat : Export file format.
	ExportFileFormat *PaperDownloadFormat `json:"export_file_format"`
}

// NewPaperDocDownloadDetails returns a new PaperDocDownloadDetails instance
func NewPaperDocDownloadDetails(EventUuid string, ExportFileFormat *PaperDownloadFormat) *PaperDocDownloadDetails {
	s := new(PaperDocDownloadDetails)
	s.EventUuid = EventUuid
	s.ExportFileFormat = ExportFileFormat
	return s
}

// PaperDocDownloadType : has no documentation (yet)
type PaperDocDownloadType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocDownloadType returns a new PaperDocDownloadType instance
func NewPaperDocDownloadType(Description string) *PaperDocDownloadType {
	s := new(PaperDocDownloadType)
	s.Description = Description
	return s
}

// PaperDocEditCommentDetails : Edited Paper doc comment.
type PaperDocEditCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewPaperDocEditCommentDetails returns a new PaperDocEditCommentDetails instance
func NewPaperDocEditCommentDetails(EventUuid string) *PaperDocEditCommentDetails {
	s := new(PaperDocEditCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocEditCommentType : has no documentation (yet)
type PaperDocEditCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocEditCommentType returns a new PaperDocEditCommentType instance
func NewPaperDocEditCommentType(Description string) *PaperDocEditCommentType {
	s := new(PaperDocEditCommentType)
	s.Description = Description
	return s
}

// PaperDocEditDetails : Edited Paper doc.
type PaperDocEditDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocEditDetails returns a new PaperDocEditDetails instance
func NewPaperDocEditDetails(EventUuid string) *PaperDocEditDetails {
	s := new(PaperDocEditDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocEditType : has no documentation (yet)
type PaperDocEditType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocEditType returns a new PaperDocEditType instance
func NewPaperDocEditType(Description string) *PaperDocEditType {
	s := new(PaperDocEditType)
	s.Description = Description
	return s
}

// PaperDocFollowedDetails : Followed Paper doc.
type PaperDocFollowedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocFollowedDetails returns a new PaperDocFollowedDetails instance
func NewPaperDocFollowedDetails(EventUuid string) *PaperDocFollowedDetails {
	s := new(PaperDocFollowedDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocFollowedType : has no documentation (yet)
type PaperDocFollowedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocFollowedType returns a new PaperDocFollowedType instance
func NewPaperDocFollowedType(Description string) *PaperDocFollowedType {
	s := new(PaperDocFollowedType)
	s.Description = Description
	return s
}

// PaperDocMentionDetails : Mentioned user in Paper doc.
type PaperDocMentionDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocMentionDetails returns a new PaperDocMentionDetails instance
func NewPaperDocMentionDetails(EventUuid string) *PaperDocMentionDetails {
	s := new(PaperDocMentionDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocMentionType : has no documentation (yet)
type PaperDocMentionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocMentionType returns a new PaperDocMentionType instance
func NewPaperDocMentionType(Description string) *PaperDocMentionType {
	s := new(PaperDocMentionType)
	s.Description = Description
	return s
}

// PaperDocOwnershipChangedDetails : Transferred ownership of Paper doc.
type PaperDocOwnershipChangedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// OldOwnerUserId : Previous owner.
	OldOwnerUserId string `json:"old_owner_user_id,omitempty"`
	// NewOwnerUserId : New owner.
	NewOwnerUserId string `json:"new_owner_user_id"`
}

// NewPaperDocOwnershipChangedDetails returns a new PaperDocOwnershipChangedDetails instance
func NewPaperDocOwnershipChangedDetails(EventUuid string, NewOwnerUserId string) *PaperDocOwnershipChangedDetails {
	s := new(PaperDocOwnershipChangedDetails)
	s.EventUuid = EventUuid
	s.NewOwnerUserId = NewOwnerUserId
	return s
}

// PaperDocOwnershipChangedType : has no documentation (yet)
type PaperDocOwnershipChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocOwnershipChangedType returns a new PaperDocOwnershipChangedType instance
func NewPaperDocOwnershipChangedType(Description string) *PaperDocOwnershipChangedType {
	s := new(PaperDocOwnershipChangedType)
	s.Description = Description
	return s
}

// PaperDocRequestAccessDetails : Requested access to Paper doc.
type PaperDocRequestAccessDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocRequestAccessDetails returns a new PaperDocRequestAccessDetails instance
func NewPaperDocRequestAccessDetails(EventUuid string) *PaperDocRequestAccessDetails {
	s := new(PaperDocRequestAccessDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocRequestAccessType : has no documentation (yet)
type PaperDocRequestAccessType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocRequestAccessType returns a new PaperDocRequestAccessType instance
func NewPaperDocRequestAccessType(Description string) *PaperDocRequestAccessType {
	s := new(PaperDocRequestAccessType)
	s.Description = Description
	return s
}

// PaperDocResolveCommentDetails : Resolved Paper doc comment.
type PaperDocResolveCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewPaperDocResolveCommentDetails returns a new PaperDocResolveCommentDetails instance
func NewPaperDocResolveCommentDetails(EventUuid string) *PaperDocResolveCommentDetails {
	s := new(PaperDocResolveCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocResolveCommentType : has no documentation (yet)
type PaperDocResolveCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocResolveCommentType returns a new PaperDocResolveCommentType instance
func NewPaperDocResolveCommentType(Description string) *PaperDocResolveCommentType {
	s := new(PaperDocResolveCommentType)
	s.Description = Description
	return s
}

// PaperDocRevertDetails : Restored Paper doc to previous version.
type PaperDocRevertDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocRevertDetails returns a new PaperDocRevertDetails instance
func NewPaperDocRevertDetails(EventUuid string) *PaperDocRevertDetails {
	s := new(PaperDocRevertDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocRevertType : has no documentation (yet)
type PaperDocRevertType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocRevertType returns a new PaperDocRevertType instance
func NewPaperDocRevertType(Description string) *PaperDocRevertType {
	s := new(PaperDocRevertType)
	s.Description = Description
	return s
}

// PaperDocSlackShareDetails : Shared Paper doc via Slack.
type PaperDocSlackShareDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocSlackShareDetails returns a new PaperDocSlackShareDetails instance
func NewPaperDocSlackShareDetails(EventUuid string) *PaperDocSlackShareDetails {
	s := new(PaperDocSlackShareDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocSlackShareType : has no documentation (yet)
type PaperDocSlackShareType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocSlackShareType returns a new PaperDocSlackShareType instance
func NewPaperDocSlackShareType(Description string) *PaperDocSlackShareType {
	s := new(PaperDocSlackShareType)
	s.Description = Description
	return s
}

// PaperDocTeamInviteDetails : Shared Paper doc with users and/or groups.
type PaperDocTeamInviteDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocTeamInviteDetails returns a new PaperDocTeamInviteDetails instance
func NewPaperDocTeamInviteDetails(EventUuid string) *PaperDocTeamInviteDetails {
	s := new(PaperDocTeamInviteDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocTeamInviteType : has no documentation (yet)
type PaperDocTeamInviteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocTeamInviteType returns a new PaperDocTeamInviteType instance
func NewPaperDocTeamInviteType(Description string) *PaperDocTeamInviteType {
	s := new(PaperDocTeamInviteType)
	s.Description = Description
	return s
}

// PaperDocTrashedDetails : Deleted Paper doc.
type PaperDocTrashedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocTrashedDetails returns a new PaperDocTrashedDetails instance
func NewPaperDocTrashedDetails(EventUuid string) *PaperDocTrashedDetails {
	s := new(PaperDocTrashedDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocTrashedType : has no documentation (yet)
type PaperDocTrashedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocTrashedType returns a new PaperDocTrashedType instance
func NewPaperDocTrashedType(Description string) *PaperDocTrashedType {
	s := new(PaperDocTrashedType)
	s.Description = Description
	return s
}

// PaperDocUnresolveCommentDetails : Unresolved Paper doc comment.
type PaperDocUnresolveCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewPaperDocUnresolveCommentDetails returns a new PaperDocUnresolveCommentDetails instance
func NewPaperDocUnresolveCommentDetails(EventUuid string) *PaperDocUnresolveCommentDetails {
	s := new(PaperDocUnresolveCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocUnresolveCommentType : has no documentation (yet)
type PaperDocUnresolveCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocUnresolveCommentType returns a new PaperDocUnresolveCommentType instance
func NewPaperDocUnresolveCommentType(Description string) *PaperDocUnresolveCommentType {
	s := new(PaperDocUnresolveCommentType)
	s.Description = Description
	return s
}

// PaperDocUntrashedDetails : Restored Paper doc.
type PaperDocUntrashedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocUntrashedDetails returns a new PaperDocUntrashedDetails instance
func NewPaperDocUntrashedDetails(EventUuid string) *PaperDocUntrashedDetails {
	s := new(PaperDocUntrashedDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocUntrashedType : has no documentation (yet)
type PaperDocUntrashedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocUntrashedType returns a new PaperDocUntrashedType instance
func NewPaperDocUntrashedType(Description string) *PaperDocUntrashedType {
	s := new(PaperDocUntrashedType)
	s.Description = Description
	return s
}

// PaperDocViewDetails : Viewed Paper doc.
type PaperDocViewDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperDocViewDetails returns a new PaperDocViewDetails instance
func NewPaperDocViewDetails(EventUuid string) *PaperDocViewDetails {
	s := new(PaperDocViewDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperDocViewType : has no documentation (yet)
type PaperDocViewType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperDocViewType returns a new PaperDocViewType instance
func NewPaperDocViewType(Description string) *PaperDocViewType {
	s := new(PaperDocViewType)
	s.Description = Description
	return s
}

// PaperDocumentLogInfo : Paper document's logged information.
type PaperDocumentLogInfo struct {
	// DocId : Papers document Id.
	DocId string `json:"doc_id"`
	// DocTitle : Paper document title.
	DocTitle string `json:"doc_title"`
}

// NewPaperDocumentLogInfo returns a new PaperDocumentLogInfo instance
func NewPaperDocumentLogInfo(DocId string, DocTitle string) *PaperDocumentLogInfo {
	s := new(PaperDocumentLogInfo)
	s.DocId = DocId
	s.DocTitle = DocTitle
	return s
}

// PaperDownloadFormat : has no documentation (yet)
type PaperDownloadFormat struct {
	dropbox.Tagged
}

// Valid tag values for PaperDownloadFormat
const (
	PaperDownloadFormatDocx     = "docx"
	PaperDownloadFormatHtml     = "html"
	PaperDownloadFormatMarkdown = "markdown"
	PaperDownloadFormatPdf      = "pdf"
	PaperDownloadFormatOther    = "other"
)

// PaperEnabledUsersGroupAdditionDetails : Added users to Paper-enabled users
// list.
type PaperEnabledUsersGroupAdditionDetails struct {
}

// NewPaperEnabledUsersGroupAdditionDetails returns a new PaperEnabledUsersGroupAdditionDetails instance
func NewPaperEnabledUsersGroupAdditionDetails() *PaperEnabledUsersGroupAdditionDetails {
	s := new(PaperEnabledUsersGroupAdditionDetails)
	return s
}

// PaperEnabledUsersGroupAdditionType : has no documentation (yet)
type PaperEnabledUsersGroupAdditionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperEnabledUsersGroupAdditionType returns a new PaperEnabledUsersGroupAdditionType instance
func NewPaperEnabledUsersGroupAdditionType(Description string) *PaperEnabledUsersGroupAdditionType {
	s := new(PaperEnabledUsersGroupAdditionType)
	s.Description = Description
	return s
}

// PaperEnabledUsersGroupRemovalDetails : Removed users from Paper-enabled users
// list.
type PaperEnabledUsersGroupRemovalDetails struct {
}

// NewPaperEnabledUsersGroupRemovalDetails returns a new PaperEnabledUsersGroupRemovalDetails instance
func NewPaperEnabledUsersGroupRemovalDetails() *PaperEnabledUsersGroupRemovalDetails {
	s := new(PaperEnabledUsersGroupRemovalDetails)
	return s
}

// PaperEnabledUsersGroupRemovalType : has no documentation (yet)
type PaperEnabledUsersGroupRemovalType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperEnabledUsersGroupRemovalType returns a new PaperEnabledUsersGroupRemovalType instance
func NewPaperEnabledUsersGroupRemovalType(Description string) *PaperEnabledUsersGroupRemovalType {
	s := new(PaperEnabledUsersGroupRemovalType)
	s.Description = Description
	return s
}

// PaperExternalViewAllowDetails : Changed Paper external sharing setting to
// anyone.
type PaperExternalViewAllowDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperExternalViewAllowDetails returns a new PaperExternalViewAllowDetails instance
func NewPaperExternalViewAllowDetails(EventUuid string) *PaperExternalViewAllowDetails {
	s := new(PaperExternalViewAllowDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperExternalViewAllowType : has no documentation (yet)
type PaperExternalViewAllowType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperExternalViewAllowType returns a new PaperExternalViewAllowType instance
func NewPaperExternalViewAllowType(Description string) *PaperExternalViewAllowType {
	s := new(PaperExternalViewAllowType)
	s.Description = Description
	return s
}

// PaperExternalViewDefaultTeamDetails : Changed Paper external sharing setting
// to default team.
type PaperExternalViewDefaultTeamDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperExternalViewDefaultTeamDetails returns a new PaperExternalViewDefaultTeamDetails instance
func NewPaperExternalViewDefaultTeamDetails(EventUuid string) *PaperExternalViewDefaultTeamDetails {
	s := new(PaperExternalViewDefaultTeamDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperExternalViewDefaultTeamType : has no documentation (yet)
type PaperExternalViewDefaultTeamType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperExternalViewDefaultTeamType returns a new PaperExternalViewDefaultTeamType instance
func NewPaperExternalViewDefaultTeamType(Description string) *PaperExternalViewDefaultTeamType {
	s := new(PaperExternalViewDefaultTeamType)
	s.Description = Description
	return s
}

// PaperExternalViewForbidDetails : Changed Paper external sharing setting to
// team-only.
type PaperExternalViewForbidDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperExternalViewForbidDetails returns a new PaperExternalViewForbidDetails instance
func NewPaperExternalViewForbidDetails(EventUuid string) *PaperExternalViewForbidDetails {
	s := new(PaperExternalViewForbidDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperExternalViewForbidType : has no documentation (yet)
type PaperExternalViewForbidType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperExternalViewForbidType returns a new PaperExternalViewForbidType instance
func NewPaperExternalViewForbidType(Description string) *PaperExternalViewForbidType {
	s := new(PaperExternalViewForbidType)
	s.Description = Description
	return s
}

// PaperFolderChangeSubscriptionDetails : Followed/unfollowed Paper folder.
type PaperFolderChangeSubscriptionDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// NewSubscriptionLevel : New folder subscription level.
	NewSubscriptionLevel string `json:"new_subscription_level"`
	// PreviousSubscriptionLevel : Previous folder subscription level. Might be
	// missing due to historical data gap.
	PreviousSubscriptionLevel string `json:"previous_subscription_level,omitempty"`
}

// NewPaperFolderChangeSubscriptionDetails returns a new PaperFolderChangeSubscriptionDetails instance
func NewPaperFolderChangeSubscriptionDetails(EventUuid string, NewSubscriptionLevel string) *PaperFolderChangeSubscriptionDetails {
	s := new(PaperFolderChangeSubscriptionDetails)
	s.EventUuid = EventUuid
	s.NewSubscriptionLevel = NewSubscriptionLevel
	return s
}

// PaperFolderChangeSubscriptionType : has no documentation (yet)
type PaperFolderChangeSubscriptionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperFolderChangeSubscriptionType returns a new PaperFolderChangeSubscriptionType instance
func NewPaperFolderChangeSubscriptionType(Description string) *PaperFolderChangeSubscriptionType {
	s := new(PaperFolderChangeSubscriptionType)
	s.Description = Description
	return s
}

// PaperFolderDeletedDetails : Archived Paper folder.
type PaperFolderDeletedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperFolderDeletedDetails returns a new PaperFolderDeletedDetails instance
func NewPaperFolderDeletedDetails(EventUuid string) *PaperFolderDeletedDetails {
	s := new(PaperFolderDeletedDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperFolderDeletedType : has no documentation (yet)
type PaperFolderDeletedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperFolderDeletedType returns a new PaperFolderDeletedType instance
func NewPaperFolderDeletedType(Description string) *PaperFolderDeletedType {
	s := new(PaperFolderDeletedType)
	s.Description = Description
	return s
}

// PaperFolderFollowedDetails : Followed Paper folder.
type PaperFolderFollowedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperFolderFollowedDetails returns a new PaperFolderFollowedDetails instance
func NewPaperFolderFollowedDetails(EventUuid string) *PaperFolderFollowedDetails {
	s := new(PaperFolderFollowedDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperFolderFollowedType : has no documentation (yet)
type PaperFolderFollowedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperFolderFollowedType returns a new PaperFolderFollowedType instance
func NewPaperFolderFollowedType(Description string) *PaperFolderFollowedType {
	s := new(PaperFolderFollowedType)
	s.Description = Description
	return s
}

// PaperFolderLogInfo : Paper folder's logged information.
type PaperFolderLogInfo struct {
	// FolderId : Papers folder Id.
	FolderId string `json:"folder_id"`
	// FolderName : Paper folder name.
	FolderName string `json:"folder_name"`
}

// NewPaperFolderLogInfo returns a new PaperFolderLogInfo instance
func NewPaperFolderLogInfo(FolderId string, FolderName string) *PaperFolderLogInfo {
	s := new(PaperFolderLogInfo)
	s.FolderId = FolderId
	s.FolderName = FolderName
	return s
}

// PaperFolderTeamInviteDetails : Shared Paper folder with users and/or groups.
type PaperFolderTeamInviteDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperFolderTeamInviteDetails returns a new PaperFolderTeamInviteDetails instance
func NewPaperFolderTeamInviteDetails(EventUuid string) *PaperFolderTeamInviteDetails {
	s := new(PaperFolderTeamInviteDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperFolderTeamInviteType : has no documentation (yet)
type PaperFolderTeamInviteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperFolderTeamInviteType returns a new PaperFolderTeamInviteType instance
func NewPaperFolderTeamInviteType(Description string) *PaperFolderTeamInviteType {
	s := new(PaperFolderTeamInviteType)
	s.Description = Description
	return s
}

// PaperMemberPolicy : Policy for controlling if team members can share Paper
// documents externally.
type PaperMemberPolicy struct {
	dropbox.Tagged
}

// Valid tag values for PaperMemberPolicy
const (
	PaperMemberPolicyAnyoneWithLink          = "anyone_with_link"
	PaperMemberPolicyOnlyTeam                = "only_team"
	PaperMemberPolicyTeamAndExplicitlyShared = "team_and_explicitly_shared"
	PaperMemberPolicyOther                   = "other"
)

// PaperPublishedLinkChangePermissionDetails : Changed permissions for published
// doc.
type PaperPublishedLinkChangePermissionDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// NewPermissionLevel : New permission level.
	NewPermissionLevel string `json:"new_permission_level"`
	// PreviousPermissionLevel : Previous permission level.
	PreviousPermissionLevel string `json:"previous_permission_level"`
}

// NewPaperPublishedLinkChangePermissionDetails returns a new PaperPublishedLinkChangePermissionDetails instance
func NewPaperPublishedLinkChangePermissionDetails(EventUuid string, NewPermissionLevel string, PreviousPermissionLevel string) *PaperPublishedLinkChangePermissionDetails {
	s := new(PaperPublishedLinkChangePermissionDetails)
	s.EventUuid = EventUuid
	s.NewPermissionLevel = NewPermissionLevel
	s.PreviousPermissionLevel = PreviousPermissionLevel
	return s
}

// PaperPublishedLinkChangePermissionType : has no documentation (yet)
type PaperPublishedLinkChangePermissionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperPublishedLinkChangePermissionType returns a new PaperPublishedLinkChangePermissionType instance
func NewPaperPublishedLinkChangePermissionType(Description string) *PaperPublishedLinkChangePermissionType {
	s := new(PaperPublishedLinkChangePermissionType)
	s.Description = Description
	return s
}

// PaperPublishedLinkCreateDetails : Published doc.
type PaperPublishedLinkCreateDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperPublishedLinkCreateDetails returns a new PaperPublishedLinkCreateDetails instance
func NewPaperPublishedLinkCreateDetails(EventUuid string) *PaperPublishedLinkCreateDetails {
	s := new(PaperPublishedLinkCreateDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperPublishedLinkCreateType : has no documentation (yet)
type PaperPublishedLinkCreateType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperPublishedLinkCreateType returns a new PaperPublishedLinkCreateType instance
func NewPaperPublishedLinkCreateType(Description string) *PaperPublishedLinkCreateType {
	s := new(PaperPublishedLinkCreateType)
	s.Description = Description
	return s
}

// PaperPublishedLinkDisabledDetails : Unpublished doc.
type PaperPublishedLinkDisabledDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperPublishedLinkDisabledDetails returns a new PaperPublishedLinkDisabledDetails instance
func NewPaperPublishedLinkDisabledDetails(EventUuid string) *PaperPublishedLinkDisabledDetails {
	s := new(PaperPublishedLinkDisabledDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperPublishedLinkDisabledType : has no documentation (yet)
type PaperPublishedLinkDisabledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperPublishedLinkDisabledType returns a new PaperPublishedLinkDisabledType instance
func NewPaperPublishedLinkDisabledType(Description string) *PaperPublishedLinkDisabledType {
	s := new(PaperPublishedLinkDisabledType)
	s.Description = Description
	return s
}

// PaperPublishedLinkViewDetails : Viewed published doc.
type PaperPublishedLinkViewDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewPaperPublishedLinkViewDetails returns a new PaperPublishedLinkViewDetails instance
func NewPaperPublishedLinkViewDetails(EventUuid string) *PaperPublishedLinkViewDetails {
	s := new(PaperPublishedLinkViewDetails)
	s.EventUuid = EventUuid
	return s
}

// PaperPublishedLinkViewType : has no documentation (yet)
type PaperPublishedLinkViewType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPaperPublishedLinkViewType returns a new PaperPublishedLinkViewType instance
func NewPaperPublishedLinkViewType(Description string) *PaperPublishedLinkViewType {
	s := new(PaperPublishedLinkViewType)
	s.Description = Description
	return s
}

// ParticipantLogInfo : A user or group
type ParticipantLogInfo struct {
	dropbox.Tagged
	// Group : Group details.
	Group *GroupLogInfo `json:"group,omitempty"`
	// User : A user with a Dropbox account.
	User IsUserLogInfo `json:"user,omitempty"`
}

// Valid tag values for ParticipantLogInfo
const (
	ParticipantLogInfoGroup = "group"
	ParticipantLogInfoUser  = "user"
	ParticipantLogInfoOther = "other"
)

// UnmarshalJSON deserializes into a ParticipantLogInfo instance
func (u *ParticipantLogInfo) UnmarshalJSON(body []byte) error {
	type wrap struct {
		dropbox.Tagged
		// User : A user with a Dropbox account.
		User json.RawMessage `json:"user,omitempty"`
	}
	var w wrap
	var err error
	if err = json.Unmarshal(body, &w); err != nil {
		return err
	}
	u.Tag = w.Tag
	switch u.Tag {
	case "group":
		if err = json.Unmarshal(body, &u.Group); err != nil {
			return err
		}

	case "user":
		if u.User, err = IsUserLogInfoFromJSON(w.User); err != nil {
			return err
		}

	}
	return nil
}

// PassPolicy : has no documentation (yet)
type PassPolicy struct {
	dropbox.Tagged
}

// Valid tag values for PassPolicy
const (
	PassPolicyAllow    = "allow"
	PassPolicyDisabled = "disabled"
	PassPolicyEnabled  = "enabled"
	PassPolicyOther    = "other"
)

// PasswordChangeDetails : Changed password.
type PasswordChangeDetails struct {
}

// NewPasswordChangeDetails returns a new PasswordChangeDetails instance
func NewPasswordChangeDetails() *PasswordChangeDetails {
	s := new(PasswordChangeDetails)
	return s
}

// PasswordChangeType : has no documentation (yet)
type PasswordChangeType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPasswordChangeType returns a new PasswordChangeType instance
func NewPasswordChangeType(Description string) *PasswordChangeType {
	s := new(PasswordChangeType)
	s.Description = Description
	return s
}

// PasswordResetAllDetails : Reset all team member passwords.
type PasswordResetAllDetails struct {
}

// NewPasswordResetAllDetails returns a new PasswordResetAllDetails instance
func NewPasswordResetAllDetails() *PasswordResetAllDetails {
	s := new(PasswordResetAllDetails)
	return s
}

// PasswordResetAllType : has no documentation (yet)
type PasswordResetAllType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPasswordResetAllType returns a new PasswordResetAllType instance
func NewPasswordResetAllType(Description string) *PasswordResetAllType {
	s := new(PasswordResetAllType)
	s.Description = Description
	return s
}

// PasswordResetDetails : Reset password.
type PasswordResetDetails struct {
}

// NewPasswordResetDetails returns a new PasswordResetDetails instance
func NewPasswordResetDetails() *PasswordResetDetails {
	s := new(PasswordResetDetails)
	return s
}

// PasswordResetType : has no documentation (yet)
type PasswordResetType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPasswordResetType returns a new PasswordResetType instance
func NewPasswordResetType(Description string) *PasswordResetType {
	s := new(PasswordResetType)
	s.Description = Description
	return s
}

// PasswordStrengthRequirementsChangePolicyDetails : Changed team password
// strength requirements.
type PasswordStrengthRequirementsChangePolicyDetails struct {
	// PreviousValue : Old password strength policy.
	PreviousValue *team_policies.PasswordStrengthPolicy `json:"previous_value"`
	// NewValue : New password strength policy.
	NewValue *team_policies.PasswordStrengthPolicy `json:"new_value"`
}

// NewPasswordStrengthRequirementsChangePolicyDetails returns a new PasswordStrengthRequirementsChangePolicyDetails instance
func NewPasswordStrengthRequirementsChangePolicyDetails(PreviousValue *team_policies.PasswordStrengthPolicy, NewValue *team_policies.PasswordStrengthPolicy) *PasswordStrengthRequirementsChangePolicyDetails {
	s := new(PasswordStrengthRequirementsChangePolicyDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// PasswordStrengthRequirementsChangePolicyType : has no documentation (yet)
type PasswordStrengthRequirementsChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPasswordStrengthRequirementsChangePolicyType returns a new PasswordStrengthRequirementsChangePolicyType instance
func NewPasswordStrengthRequirementsChangePolicyType(Description string) *PasswordStrengthRequirementsChangePolicyType {
	s := new(PasswordStrengthRequirementsChangePolicyType)
	s.Description = Description
	return s
}

// PathLogInfo : Path's details.
type PathLogInfo struct {
	// Contextual : Fully qualified path relative to event's context.
	Contextual string `json:"contextual,omitempty"`
	// NamespaceRelative : Path relative to the namespace containing the
	// content.
	NamespaceRelative *NamespaceRelativePathLogInfo `json:"namespace_relative"`
}

// NewPathLogInfo returns a new PathLogInfo instance
func NewPathLogInfo(NamespaceRelative *NamespaceRelativePathLogInfo) *PathLogInfo {
	s := new(PathLogInfo)
	s.NamespaceRelative = NamespaceRelative
	return s
}

// PendingSecondaryEmailAddedDetails : Added pending secondary email.
type PendingSecondaryEmailAddedDetails struct {
	// SecondaryEmail : New pending secondary email.
	SecondaryEmail string `json:"secondary_email"`
}

// NewPendingSecondaryEmailAddedDetails returns a new PendingSecondaryEmailAddedDetails instance
func NewPendingSecondaryEmailAddedDetails(SecondaryEmail string) *PendingSecondaryEmailAddedDetails {
	s := new(PendingSecondaryEmailAddedDetails)
	s.SecondaryEmail = SecondaryEmail
	return s
}

// PendingSecondaryEmailAddedType : has no documentation (yet)
type PendingSecondaryEmailAddedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPendingSecondaryEmailAddedType returns a new PendingSecondaryEmailAddedType instance
func NewPendingSecondaryEmailAddedType(Description string) *PendingSecondaryEmailAddedType {
	s := new(PendingSecondaryEmailAddedType)
	s.Description = Description
	return s
}

// PermanentDeleteChangePolicyDetails : Enabled/disabled ability of team members
// to permanently delete content.
type PermanentDeleteChangePolicyDetails struct {
	// NewValue : New permanent delete content policy.
	NewValue *ContentPermanentDeletePolicy `json:"new_value"`
	// PreviousValue : Previous permanent delete content policy. Might be
	// missing due to historical data gap.
	PreviousValue *ContentPermanentDeletePolicy `json:"previous_value,omitempty"`
}

// NewPermanentDeleteChangePolicyDetails returns a new PermanentDeleteChangePolicyDetails instance
func NewPermanentDeleteChangePolicyDetails(NewValue *ContentPermanentDeletePolicy) *PermanentDeleteChangePolicyDetails {
	s := new(PermanentDeleteChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// PermanentDeleteChangePolicyType : has no documentation (yet)
type PermanentDeleteChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewPermanentDeleteChangePolicyType returns a new PermanentDeleteChangePolicyType instance
func NewPermanentDeleteChangePolicyType(Description string) *PermanentDeleteChangePolicyType {
	s := new(PermanentDeleteChangePolicyType)
	s.Description = Description
	return s
}

// PlacementRestriction : has no documentation (yet)
type PlacementRestriction struct {
	dropbox.Tagged
}

// Valid tag values for PlacementRestriction
const (
	PlacementRestrictionAustraliaOnly = "australia_only"
	PlacementRestrictionEuropeOnly    = "europe_only"
	PlacementRestrictionJapanOnly     = "japan_only"
	PlacementRestrictionNone          = "none"
	PlacementRestrictionUkOnly        = "uk_only"
	PlacementRestrictionUsS3Only      = "us_s3_only"
	PlacementRestrictionOther         = "other"
)

// PolicyType : has no documentation (yet)
type PolicyType struct {
	dropbox.Tagged
}

// Valid tag values for PolicyType
const (
	PolicyTypeDisposition = "disposition"
	PolicyTypeRetention   = "retention"
	PolicyTypeOther       = "other"
)

// PrimaryTeamRequestAcceptedDetails : Team merge request acceptance details
// shown to the primary team
type PrimaryTeamRequestAcceptedDetails struct {
	// SecondaryTeam : The secondary team name.
	SecondaryTeam string `json:"secondary_team"`
	// SentBy : The name of the secondary team admin who sent the request
	// originally.
	SentBy string `json:"sent_by"`
}

// NewPrimaryTeamRequestAcceptedDetails returns a new PrimaryTeamRequestAcceptedDetails instance
func NewPrimaryTeamRequestAcceptedDetails(SecondaryTeam string, SentBy string) *PrimaryTeamRequestAcceptedDetails {
	s := new(PrimaryTeamRequestAcceptedDetails)
	s.SecondaryTeam = SecondaryTeam
	s.SentBy = SentBy
	return s
}

// PrimaryTeamRequestCanceledDetails : Team merge request cancellation details
// shown to the primary team
type PrimaryTeamRequestCanceledDetails struct {
	// SecondaryTeam : The secondary team name.
	SecondaryTeam string `json:"secondary_team"`
	// SentBy : The name of the secondary team admin who sent the request
	// originally.
	SentBy string `json:"sent_by"`
}

// NewPrimaryTeamRequestCanceledDetails returns a new PrimaryTeamRequestCanceledDetails instance
func NewPrimaryTeamRequestCanceledDetails(SecondaryTeam string, SentBy string) *PrimaryTeamRequestCanceledDetails {
	s := new(PrimaryTeamRequestCanceledDetails)
	s.SecondaryTeam = SecondaryTeam
	s.SentBy = SentBy
	return s
}

// PrimaryTeamRequestExpiredDetails : Team merge request expiration details
// shown to the primary team
type PrimaryTeamRequestExpiredDetails struct {
	// SecondaryTeam : The secondary team name.
	SecondaryTeam string `json:"secondary_team"`
	// SentBy : The name of the secondary team admin who sent the request
	// originally.
	SentBy string `json:"sent_by"`
}

// NewPrimaryTeamRequestExpiredDetails returns a new PrimaryTeamRequestExpiredDetails instance
func NewPrimaryTeamRequestExpiredDetails(SecondaryTeam string, SentBy string) *PrimaryTeamRequestExpiredDetails {
	s := new(PrimaryTeamRequestExpiredDetails)
	s.SecondaryTeam = SecondaryTeam
	s.SentBy = SentBy
	return s
}

// PrimaryTeamRequestReminderDetails : Team merge request reminder details shown
// to the primary team
type PrimaryTeamRequestReminderDetails struct {
	// SecondaryTeam : The secondary team name.
	SecondaryTeam string `json:"secondary_team"`
	// SentTo : The name of the primary team admin the request was sent to.
	SentTo string `json:"sent_to"`
}

// NewPrimaryTeamRequestReminderDetails returns a new PrimaryTeamRequestReminderDetails instance
func NewPrimaryTeamRequestReminderDetails(SecondaryTeam string, SentTo string) *PrimaryTeamRequestReminderDetails {
	s := new(PrimaryTeamRequestReminderDetails)
	s.SecondaryTeam = SecondaryTeam
	s.SentTo = SentTo
	return s
}

// QuickActionType : Quick action type.
type QuickActionType struct {
	dropbox.Tagged
}

// Valid tag values for QuickActionType
const (
	QuickActionTypeDeleteSharedLink    = "delete_shared_link"
	QuickActionTypeResetPassword       = "reset_password"
	QuickActionTypeRestoreFileOrFolder = "restore_file_or_folder"
	QuickActionTypeUnlinkApp           = "unlink_app"
	QuickActionTypeUnlinkDevice        = "unlink_device"
	QuickActionTypeUnlinkSession       = "unlink_session"
	QuickActionTypeOther               = "other"
)

// RecipientsConfiguration : Recipients Configuration
type RecipientsConfiguration struct {
	// RecipientSettingType : Recipients setting type.
	RecipientSettingType *AlertRecipientsSettingType `json:"recipient_setting_type,omitempty"`
	// Emails : A list of user emails to notify.
	Emails []string `json:"emails,omitempty"`
	// Groups : A list of groups to notify.
	Groups []string `json:"groups,omitempty"`
}

// NewRecipientsConfiguration returns a new RecipientsConfiguration instance
func NewRecipientsConfiguration() *RecipientsConfiguration {
	s := new(RecipientsConfiguration)
	return s
}

// RelocateAssetReferencesLogInfo : Provides the indices of the source asset and
// the destination asset for a relocate action.
type RelocateAssetReferencesLogInfo struct {
	// SrcAssetIndex : Source asset position in the Assets list.
	SrcAssetIndex uint64 `json:"src_asset_index"`
	// DestAssetIndex : Destination asset position in the Assets list.
	DestAssetIndex uint64 `json:"dest_asset_index"`
}

// NewRelocateAssetReferencesLogInfo returns a new RelocateAssetReferencesLogInfo instance
func NewRelocateAssetReferencesLogInfo(SrcAssetIndex uint64, DestAssetIndex uint64) *RelocateAssetReferencesLogInfo {
	s := new(RelocateAssetReferencesLogInfo)
	s.SrcAssetIndex = SrcAssetIndex
	s.DestAssetIndex = DestAssetIndex
	return s
}

// ResellerLogInfo : Reseller information.
type ResellerLogInfo struct {
	// ResellerName : Reseller name.
	ResellerName string `json:"reseller_name"`
	// ResellerEmail : Reseller email.
	ResellerEmail string `json:"reseller_email"`
}

// NewResellerLogInfo returns a new ResellerLogInfo instance
func NewResellerLogInfo(ResellerName string, ResellerEmail string) *ResellerLogInfo {
	s := new(ResellerLogInfo)
	s.ResellerName = ResellerName
	s.ResellerEmail = ResellerEmail
	return s
}

// ResellerRole : has no documentation (yet)
type ResellerRole struct {
	dropbox.Tagged
}

// Valid tag values for ResellerRole
const (
	ResellerRoleNotReseller   = "not_reseller"
	ResellerRoleResellerAdmin = "reseller_admin"
	ResellerRoleOther         = "other"
)

// ResellerSupportChangePolicyDetails : Enabled/disabled reseller support.
type ResellerSupportChangePolicyDetails struct {
	// NewValue : New Reseller support policy.
	NewValue *ResellerSupportPolicy `json:"new_value"`
	// PreviousValue : Previous Reseller support policy.
	PreviousValue *ResellerSupportPolicy `json:"previous_value"`
}

// NewResellerSupportChangePolicyDetails returns a new ResellerSupportChangePolicyDetails instance
func NewResellerSupportChangePolicyDetails(NewValue *ResellerSupportPolicy, PreviousValue *ResellerSupportPolicy) *ResellerSupportChangePolicyDetails {
	s := new(ResellerSupportChangePolicyDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// ResellerSupportChangePolicyType : has no documentation (yet)
type ResellerSupportChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewResellerSupportChangePolicyType returns a new ResellerSupportChangePolicyType instance
func NewResellerSupportChangePolicyType(Description string) *ResellerSupportChangePolicyType {
	s := new(ResellerSupportChangePolicyType)
	s.Description = Description
	return s
}

// ResellerSupportPolicy : Policy for controlling if reseller can access the
// admin console as administrator
type ResellerSupportPolicy struct {
	dropbox.Tagged
}

// Valid tag values for ResellerSupportPolicy
const (
	ResellerSupportPolicyDisabled = "disabled"
	ResellerSupportPolicyEnabled  = "enabled"
	ResellerSupportPolicyOther    = "other"
)

// ResellerSupportSessionEndDetails : Ended reseller support session.
type ResellerSupportSessionEndDetails struct {
}

// NewResellerSupportSessionEndDetails returns a new ResellerSupportSessionEndDetails instance
func NewResellerSupportSessionEndDetails() *ResellerSupportSessionEndDetails {
	s := new(ResellerSupportSessionEndDetails)
	return s
}

// ResellerSupportSessionEndType : has no documentation (yet)
type ResellerSupportSessionEndType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewResellerSupportSessionEndType returns a new ResellerSupportSessionEndType instance
func NewResellerSupportSessionEndType(Description string) *ResellerSupportSessionEndType {
	s := new(ResellerSupportSessionEndType)
	s.Description = Description
	return s
}

// ResellerSupportSessionStartDetails : Started reseller support session.
type ResellerSupportSessionStartDetails struct {
}

// NewResellerSupportSessionStartDetails returns a new ResellerSupportSessionStartDetails instance
func NewResellerSupportSessionStartDetails() *ResellerSupportSessionStartDetails {
	s := new(ResellerSupportSessionStartDetails)
	return s
}

// ResellerSupportSessionStartType : has no documentation (yet)
type ResellerSupportSessionStartType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewResellerSupportSessionStartType returns a new ResellerSupportSessionStartType instance
func NewResellerSupportSessionStartType(Description string) *ResellerSupportSessionStartType {
	s := new(ResellerSupportSessionStartType)
	s.Description = Description
	return s
}

// RewindFolderDetails : Rewound a folder.
type RewindFolderDetails struct {
	// RewindFolderTargetTsMs : Folder was Rewound to this date.
	RewindFolderTargetTsMs time.Time `json:"rewind_folder_target_ts_ms"`
}

// NewRewindFolderDetails returns a new RewindFolderDetails instance
func NewRewindFolderDetails(RewindFolderTargetTsMs time.Time) *RewindFolderDetails {
	s := new(RewindFolderDetails)
	s.RewindFolderTargetTsMs = RewindFolderTargetTsMs
	return s
}

// RewindFolderType : has no documentation (yet)
type RewindFolderType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewRewindFolderType returns a new RewindFolderType instance
func NewRewindFolderType(Description string) *RewindFolderType {
	s := new(RewindFolderType)
	s.Description = Description
	return s
}

// RewindPolicy : Policy for controlling whether team members can rewind
type RewindPolicy struct {
	dropbox.Tagged
}

// Valid tag values for RewindPolicy
const (
	RewindPolicyAdminsOnly = "admins_only"
	RewindPolicyEveryone   = "everyone"
	RewindPolicyOther      = "other"
)

// RewindPolicyChangedDetails : Changed Rewind policy for team.
type RewindPolicyChangedDetails struct {
	// NewValue : New Dropbox Rewind policy.
	NewValue *RewindPolicy `json:"new_value"`
	// PreviousValue : Previous Dropbox Rewind policy.
	PreviousValue *RewindPolicy `json:"previous_value"`
}

// NewRewindPolicyChangedDetails returns a new RewindPolicyChangedDetails instance
func NewRewindPolicyChangedDetails(NewValue *RewindPolicy, PreviousValue *RewindPolicy) *RewindPolicyChangedDetails {
	s := new(RewindPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// RewindPolicyChangedType : has no documentation (yet)
type RewindPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewRewindPolicyChangedType returns a new RewindPolicyChangedType instance
func NewRewindPolicyChangedType(Description string) *RewindPolicyChangedType {
	s := new(RewindPolicyChangedType)
	s.Description = Description
	return s
}

// SecondaryEmailDeletedDetails : Deleted secondary email.
type SecondaryEmailDeletedDetails struct {
	// SecondaryEmail : Deleted secondary email.
	SecondaryEmail string `json:"secondary_email"`
}

// NewSecondaryEmailDeletedDetails returns a new SecondaryEmailDeletedDetails instance
func NewSecondaryEmailDeletedDetails(SecondaryEmail string) *SecondaryEmailDeletedDetails {
	s := new(SecondaryEmailDeletedDetails)
	s.SecondaryEmail = SecondaryEmail
	return s
}

// SecondaryEmailDeletedType : has no documentation (yet)
type SecondaryEmailDeletedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSecondaryEmailDeletedType returns a new SecondaryEmailDeletedType instance
func NewSecondaryEmailDeletedType(Description string) *SecondaryEmailDeletedType {
	s := new(SecondaryEmailDeletedType)
	s.Description = Description
	return s
}

// SecondaryEmailVerifiedDetails : Verified secondary email.
type SecondaryEmailVerifiedDetails struct {
	// SecondaryEmail : Verified secondary email.
	SecondaryEmail string `json:"secondary_email"`
}

// NewSecondaryEmailVerifiedDetails returns a new SecondaryEmailVerifiedDetails instance
func NewSecondaryEmailVerifiedDetails(SecondaryEmail string) *SecondaryEmailVerifiedDetails {
	s := new(SecondaryEmailVerifiedDetails)
	s.SecondaryEmail = SecondaryEmail
	return s
}

// SecondaryEmailVerifiedType : has no documentation (yet)
type SecondaryEmailVerifiedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSecondaryEmailVerifiedType returns a new SecondaryEmailVerifiedType instance
func NewSecondaryEmailVerifiedType(Description string) *SecondaryEmailVerifiedType {
	s := new(SecondaryEmailVerifiedType)
	s.Description = Description
	return s
}

// SecondaryMailsPolicy : has no documentation (yet)
type SecondaryMailsPolicy struct {
	dropbox.Tagged
}

// Valid tag values for SecondaryMailsPolicy
const (
	SecondaryMailsPolicyDisabled = "disabled"
	SecondaryMailsPolicyEnabled  = "enabled"
	SecondaryMailsPolicyOther    = "other"
)

// SecondaryMailsPolicyChangedDetails : Secondary mails policy changed.
type SecondaryMailsPolicyChangedDetails struct {
	// PreviousValue : Previous secondary mails policy.
	PreviousValue *SecondaryMailsPolicy `json:"previous_value"`
	// NewValue : New secondary mails policy.
	NewValue *SecondaryMailsPolicy `json:"new_value"`
}

// NewSecondaryMailsPolicyChangedDetails returns a new SecondaryMailsPolicyChangedDetails instance
func NewSecondaryMailsPolicyChangedDetails(PreviousValue *SecondaryMailsPolicy, NewValue *SecondaryMailsPolicy) *SecondaryMailsPolicyChangedDetails {
	s := new(SecondaryMailsPolicyChangedDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// SecondaryMailsPolicyChangedType : has no documentation (yet)
type SecondaryMailsPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSecondaryMailsPolicyChangedType returns a new SecondaryMailsPolicyChangedType instance
func NewSecondaryMailsPolicyChangedType(Description string) *SecondaryMailsPolicyChangedType {
	s := new(SecondaryMailsPolicyChangedType)
	s.Description = Description
	return s
}

// SecondaryTeamRequestAcceptedDetails : Team merge request acceptance details
// shown to the secondary team
type SecondaryTeamRequestAcceptedDetails struct {
	// PrimaryTeam : The primary team name.
	PrimaryTeam string `json:"primary_team"`
	// SentBy : The name of the secondary team admin who sent the request
	// originally.
	SentBy string `json:"sent_by"`
}

// NewSecondaryTeamRequestAcceptedDetails returns a new SecondaryTeamRequestAcceptedDetails instance
func NewSecondaryTeamRequestAcceptedDetails(PrimaryTeam string, SentBy string) *SecondaryTeamRequestAcceptedDetails {
	s := new(SecondaryTeamRequestAcceptedDetails)
	s.PrimaryTeam = PrimaryTeam
	s.SentBy = SentBy
	return s
}

// SecondaryTeamRequestCanceledDetails : Team merge request cancellation details
// shown to the secondary team
type SecondaryTeamRequestCanceledDetails struct {
	// SentTo : The email of the primary team admin that the request was sent
	// to.
	SentTo string `json:"sent_to"`
	// SentBy : The name of the secondary team admin who sent the request
	// originally.
	SentBy string `json:"sent_by"`
}

// NewSecondaryTeamRequestCanceledDetails returns a new SecondaryTeamRequestCanceledDetails instance
func NewSecondaryTeamRequestCanceledDetails(SentTo string, SentBy string) *SecondaryTeamRequestCanceledDetails {
	s := new(SecondaryTeamRequestCanceledDetails)
	s.SentTo = SentTo
	s.SentBy = SentBy
	return s
}

// SecondaryTeamRequestExpiredDetails : Team merge request expiration details
// shown to the secondary team
type SecondaryTeamRequestExpiredDetails struct {
	// SentTo : The email of the primary team admin the request was sent to.
	SentTo string `json:"sent_to"`
}

// NewSecondaryTeamRequestExpiredDetails returns a new SecondaryTeamRequestExpiredDetails instance
func NewSecondaryTeamRequestExpiredDetails(SentTo string) *SecondaryTeamRequestExpiredDetails {
	s := new(SecondaryTeamRequestExpiredDetails)
	s.SentTo = SentTo
	return s
}

// SecondaryTeamRequestReminderDetails : Team merge request reminder details
// shown to the secondary team
type SecondaryTeamRequestReminderDetails struct {
	// SentTo : The email of the primary team admin the request was sent to.
	SentTo string `json:"sent_to"`
}

// NewSecondaryTeamRequestReminderDetails returns a new SecondaryTeamRequestReminderDetails instance
func NewSecondaryTeamRequestReminderDetails(SentTo string) *SecondaryTeamRequestReminderDetails {
	s := new(SecondaryTeamRequestReminderDetails)
	s.SentTo = SentTo
	return s
}

// SendForSignaturePolicy : Policy for controlling team access to send for
// signature feature
type SendForSignaturePolicy struct {
	dropbox.Tagged
}

// Valid tag values for SendForSignaturePolicy
const (
	SendForSignaturePolicyDisabled = "disabled"
	SendForSignaturePolicyEnabled  = "enabled"
	SendForSignaturePolicyOther    = "other"
)

// SendForSignaturePolicyChangedDetails : Changed send for signature policy for
// team.
type SendForSignaturePolicyChangedDetails struct {
	// NewValue : New send for signature policy.
	NewValue *SendForSignaturePolicy `json:"new_value"`
	// PreviousValue : Previous send for signature policy.
	PreviousValue *SendForSignaturePolicy `json:"previous_value"`
}

// NewSendForSignaturePolicyChangedDetails returns a new SendForSignaturePolicyChangedDetails instance
func NewSendForSignaturePolicyChangedDetails(NewValue *SendForSignaturePolicy, PreviousValue *SendForSignaturePolicy) *SendForSignaturePolicyChangedDetails {
	s := new(SendForSignaturePolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// SendForSignaturePolicyChangedType : has no documentation (yet)
type SendForSignaturePolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSendForSignaturePolicyChangedType returns a new SendForSignaturePolicyChangedType instance
func NewSendForSignaturePolicyChangedType(Description string) *SendForSignaturePolicyChangedType {
	s := new(SendForSignaturePolicyChangedType)
	s.Description = Description
	return s
}

// SfAddGroupDetails : Added team to shared folder.
type SfAddGroupDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
	// SharingPermission : Sharing permission.
	SharingPermission string `json:"sharing_permission,omitempty"`
	// TeamName : Team name.
	TeamName string `json:"team_name"`
}

// NewSfAddGroupDetails returns a new SfAddGroupDetails instance
func NewSfAddGroupDetails(TargetAssetIndex uint64, OriginalFolderName string, TeamName string) *SfAddGroupDetails {
	s := new(SfAddGroupDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	s.TeamName = TeamName
	return s
}

// SfAddGroupType : has no documentation (yet)
type SfAddGroupType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfAddGroupType returns a new SfAddGroupType instance
func NewSfAddGroupType(Description string) *SfAddGroupType {
	s := new(SfAddGroupType)
	s.Description = Description
	return s
}

// SfAllowNonMembersToViewSharedLinksDetails : Allowed non-collaborators to view
// links to files in shared folder.
type SfAllowNonMembersToViewSharedLinksDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
	// SharedFolderType : Shared folder type.
	SharedFolderType string `json:"shared_folder_type,omitempty"`
}

// NewSfAllowNonMembersToViewSharedLinksDetails returns a new SfAllowNonMembersToViewSharedLinksDetails instance
func NewSfAllowNonMembersToViewSharedLinksDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfAllowNonMembersToViewSharedLinksDetails {
	s := new(SfAllowNonMembersToViewSharedLinksDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfAllowNonMembersToViewSharedLinksType : has no documentation (yet)
type SfAllowNonMembersToViewSharedLinksType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfAllowNonMembersToViewSharedLinksType returns a new SfAllowNonMembersToViewSharedLinksType instance
func NewSfAllowNonMembersToViewSharedLinksType(Description string) *SfAllowNonMembersToViewSharedLinksType {
	s := new(SfAllowNonMembersToViewSharedLinksType)
	s.Description = Description
	return s
}

// SfExternalInviteWarnDetails : Set team members to see warning before sharing
// folders outside team.
type SfExternalInviteWarnDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
	// NewSharingPermission : New sharing permission.
	NewSharingPermission string `json:"new_sharing_permission,omitempty"`
	// PreviousSharingPermission : Previous sharing permission.
	PreviousSharingPermission string `json:"previous_sharing_permission,omitempty"`
}

// NewSfExternalInviteWarnDetails returns a new SfExternalInviteWarnDetails instance
func NewSfExternalInviteWarnDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfExternalInviteWarnDetails {
	s := new(SfExternalInviteWarnDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfExternalInviteWarnType : has no documentation (yet)
type SfExternalInviteWarnType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfExternalInviteWarnType returns a new SfExternalInviteWarnType instance
func NewSfExternalInviteWarnType(Description string) *SfExternalInviteWarnType {
	s := new(SfExternalInviteWarnType)
	s.Description = Description
	return s
}

// SfFbInviteChangeRoleDetails : Changed Facebook user's role in shared folder.
type SfFbInviteChangeRoleDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
	// PreviousSharingPermission : Previous sharing permission.
	PreviousSharingPermission string `json:"previous_sharing_permission,omitempty"`
	// NewSharingPermission : New sharing permission.
	NewSharingPermission string `json:"new_sharing_permission,omitempty"`
}

// NewSfFbInviteChangeRoleDetails returns a new SfFbInviteChangeRoleDetails instance
func NewSfFbInviteChangeRoleDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfFbInviteChangeRoleDetails {
	s := new(SfFbInviteChangeRoleDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfFbInviteChangeRoleType : has no documentation (yet)
type SfFbInviteChangeRoleType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfFbInviteChangeRoleType returns a new SfFbInviteChangeRoleType instance
func NewSfFbInviteChangeRoleType(Description string) *SfFbInviteChangeRoleType {
	s := new(SfFbInviteChangeRoleType)
	s.Description = Description
	return s
}

// SfFbInviteDetails : Invited Facebook users to shared folder.
type SfFbInviteDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
	// SharingPermission : Sharing permission.
	SharingPermission string `json:"sharing_permission,omitempty"`
}

// NewSfFbInviteDetails returns a new SfFbInviteDetails instance
func NewSfFbInviteDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfFbInviteDetails {
	s := new(SfFbInviteDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfFbInviteType : has no documentation (yet)
type SfFbInviteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfFbInviteType returns a new SfFbInviteType instance
func NewSfFbInviteType(Description string) *SfFbInviteType {
	s := new(SfFbInviteType)
	s.Description = Description
	return s
}

// SfFbUninviteDetails : Uninvited Facebook user from shared folder.
type SfFbUninviteDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
}

// NewSfFbUninviteDetails returns a new SfFbUninviteDetails instance
func NewSfFbUninviteDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfFbUninviteDetails {
	s := new(SfFbUninviteDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfFbUninviteType : has no documentation (yet)
type SfFbUninviteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfFbUninviteType returns a new SfFbUninviteType instance
func NewSfFbUninviteType(Description string) *SfFbUninviteType {
	s := new(SfFbUninviteType)
	s.Description = Description
	return s
}

// SfInviteGroupDetails : Invited group to shared folder.
type SfInviteGroupDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
}

// NewSfInviteGroupDetails returns a new SfInviteGroupDetails instance
func NewSfInviteGroupDetails(TargetAssetIndex uint64) *SfInviteGroupDetails {
	s := new(SfInviteGroupDetails)
	s.TargetAssetIndex = TargetAssetIndex
	return s
}

// SfInviteGroupType : has no documentation (yet)
type SfInviteGroupType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfInviteGroupType returns a new SfInviteGroupType instance
func NewSfInviteGroupType(Description string) *SfInviteGroupType {
	s := new(SfInviteGroupType)
	s.Description = Description
	return s
}

// SfTeamGrantAccessDetails : Granted access to shared folder.
type SfTeamGrantAccessDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
}

// NewSfTeamGrantAccessDetails returns a new SfTeamGrantAccessDetails instance
func NewSfTeamGrantAccessDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfTeamGrantAccessDetails {
	s := new(SfTeamGrantAccessDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfTeamGrantAccessType : has no documentation (yet)
type SfTeamGrantAccessType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfTeamGrantAccessType returns a new SfTeamGrantAccessType instance
func NewSfTeamGrantAccessType(Description string) *SfTeamGrantAccessType {
	s := new(SfTeamGrantAccessType)
	s.Description = Description
	return s
}

// SfTeamInviteChangeRoleDetails : Changed team member's role in shared folder.
type SfTeamInviteChangeRoleDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
	// NewSharingPermission : New sharing permission.
	NewSharingPermission string `json:"new_sharing_permission,omitempty"`
	// PreviousSharingPermission : Previous sharing permission.
	PreviousSharingPermission string `json:"previous_sharing_permission,omitempty"`
}

// NewSfTeamInviteChangeRoleDetails returns a new SfTeamInviteChangeRoleDetails instance
func NewSfTeamInviteChangeRoleDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfTeamInviteChangeRoleDetails {
	s := new(SfTeamInviteChangeRoleDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfTeamInviteChangeRoleType : has no documentation (yet)
type SfTeamInviteChangeRoleType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfTeamInviteChangeRoleType returns a new SfTeamInviteChangeRoleType instance
func NewSfTeamInviteChangeRoleType(Description string) *SfTeamInviteChangeRoleType {
	s := new(SfTeamInviteChangeRoleType)
	s.Description = Description
	return s
}

// SfTeamInviteDetails : Invited team members to shared folder.
type SfTeamInviteDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
	// SharingPermission : Sharing permission.
	SharingPermission string `json:"sharing_permission,omitempty"`
}

// NewSfTeamInviteDetails returns a new SfTeamInviteDetails instance
func NewSfTeamInviteDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfTeamInviteDetails {
	s := new(SfTeamInviteDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfTeamInviteType : has no documentation (yet)
type SfTeamInviteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfTeamInviteType returns a new SfTeamInviteType instance
func NewSfTeamInviteType(Description string) *SfTeamInviteType {
	s := new(SfTeamInviteType)
	s.Description = Description
	return s
}

// SfTeamJoinDetails : Joined team member's shared folder.
type SfTeamJoinDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
}

// NewSfTeamJoinDetails returns a new SfTeamJoinDetails instance
func NewSfTeamJoinDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfTeamJoinDetails {
	s := new(SfTeamJoinDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfTeamJoinFromOobLinkDetails : Joined team member's shared folder from link.
type SfTeamJoinFromOobLinkDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
	// TokenKey : Shared link token key.
	TokenKey string `json:"token_key,omitempty"`
	// SharingPermission : Sharing permission.
	SharingPermission string `json:"sharing_permission,omitempty"`
}

// NewSfTeamJoinFromOobLinkDetails returns a new SfTeamJoinFromOobLinkDetails instance
func NewSfTeamJoinFromOobLinkDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfTeamJoinFromOobLinkDetails {
	s := new(SfTeamJoinFromOobLinkDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfTeamJoinFromOobLinkType : has no documentation (yet)
type SfTeamJoinFromOobLinkType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfTeamJoinFromOobLinkType returns a new SfTeamJoinFromOobLinkType instance
func NewSfTeamJoinFromOobLinkType(Description string) *SfTeamJoinFromOobLinkType {
	s := new(SfTeamJoinFromOobLinkType)
	s.Description = Description
	return s
}

// SfTeamJoinType : has no documentation (yet)
type SfTeamJoinType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfTeamJoinType returns a new SfTeamJoinType instance
func NewSfTeamJoinType(Description string) *SfTeamJoinType {
	s := new(SfTeamJoinType)
	s.Description = Description
	return s
}

// SfTeamUninviteDetails : Unshared folder with team member.
type SfTeamUninviteDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
	// OriginalFolderName : Original shared folder name.
	OriginalFolderName string `json:"original_folder_name"`
}

// NewSfTeamUninviteDetails returns a new SfTeamUninviteDetails instance
func NewSfTeamUninviteDetails(TargetAssetIndex uint64, OriginalFolderName string) *SfTeamUninviteDetails {
	s := new(SfTeamUninviteDetails)
	s.TargetAssetIndex = TargetAssetIndex
	s.OriginalFolderName = OriginalFolderName
	return s
}

// SfTeamUninviteType : has no documentation (yet)
type SfTeamUninviteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSfTeamUninviteType returns a new SfTeamUninviteType instance
func NewSfTeamUninviteType(Description string) *SfTeamUninviteType {
	s := new(SfTeamUninviteType)
	s.Description = Description
	return s
}

// SharedContentAddInviteesDetails : Invited user to Dropbox and added them to
// shared file/folder.
type SharedContentAddInviteesDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// Invitees : A list of invitees.
	Invitees []string `json:"invitees"`
}

// NewSharedContentAddInviteesDetails returns a new SharedContentAddInviteesDetails instance
func NewSharedContentAddInviteesDetails(SharedContentAccessLevel *sharing.AccessLevel, Invitees []string) *SharedContentAddInviteesDetails {
	s := new(SharedContentAddInviteesDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	s.Invitees = Invitees
	return s
}

// SharedContentAddInviteesType : has no documentation (yet)
type SharedContentAddInviteesType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentAddInviteesType returns a new SharedContentAddInviteesType instance
func NewSharedContentAddInviteesType(Description string) *SharedContentAddInviteesType {
	s := new(SharedContentAddInviteesType)
	s.Description = Description
	return s
}

// SharedContentAddLinkExpiryDetails : Added expiration date to link for shared
// file/folder.
type SharedContentAddLinkExpiryDetails struct {
	// NewValue : New shared content link expiration date. Might be missing due
	// to historical data gap.
	NewValue *time.Time `json:"new_value,omitempty"`
}

// NewSharedContentAddLinkExpiryDetails returns a new SharedContentAddLinkExpiryDetails instance
func NewSharedContentAddLinkExpiryDetails() *SharedContentAddLinkExpiryDetails {
	s := new(SharedContentAddLinkExpiryDetails)
	return s
}

// SharedContentAddLinkExpiryType : has no documentation (yet)
type SharedContentAddLinkExpiryType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentAddLinkExpiryType returns a new SharedContentAddLinkExpiryType instance
func NewSharedContentAddLinkExpiryType(Description string) *SharedContentAddLinkExpiryType {
	s := new(SharedContentAddLinkExpiryType)
	s.Description = Description
	return s
}

// SharedContentAddLinkPasswordDetails : Added password to link for shared
// file/folder.
type SharedContentAddLinkPasswordDetails struct {
}

// NewSharedContentAddLinkPasswordDetails returns a new SharedContentAddLinkPasswordDetails instance
func NewSharedContentAddLinkPasswordDetails() *SharedContentAddLinkPasswordDetails {
	s := new(SharedContentAddLinkPasswordDetails)
	return s
}

// SharedContentAddLinkPasswordType : has no documentation (yet)
type SharedContentAddLinkPasswordType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentAddLinkPasswordType returns a new SharedContentAddLinkPasswordType instance
func NewSharedContentAddLinkPasswordType(Description string) *SharedContentAddLinkPasswordType {
	s := new(SharedContentAddLinkPasswordType)
	s.Description = Description
	return s
}

// SharedContentAddMemberDetails : Added users and/or groups to shared
// file/folder.
type SharedContentAddMemberDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
}

// NewSharedContentAddMemberDetails returns a new SharedContentAddMemberDetails instance
func NewSharedContentAddMemberDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedContentAddMemberDetails {
	s := new(SharedContentAddMemberDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedContentAddMemberType : has no documentation (yet)
type SharedContentAddMemberType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentAddMemberType returns a new SharedContentAddMemberType instance
func NewSharedContentAddMemberType(Description string) *SharedContentAddMemberType {
	s := new(SharedContentAddMemberType)
	s.Description = Description
	return s
}

// SharedContentChangeDownloadsPolicyDetails : Changed whether members can
// download shared file/folder.
type SharedContentChangeDownloadsPolicyDetails struct {
	// NewValue : New downloads policy.
	NewValue *DownloadPolicyType `json:"new_value"`
	// PreviousValue : Previous downloads policy. Might be missing due to
	// historical data gap.
	PreviousValue *DownloadPolicyType `json:"previous_value,omitempty"`
}

// NewSharedContentChangeDownloadsPolicyDetails returns a new SharedContentChangeDownloadsPolicyDetails instance
func NewSharedContentChangeDownloadsPolicyDetails(NewValue *DownloadPolicyType) *SharedContentChangeDownloadsPolicyDetails {
	s := new(SharedContentChangeDownloadsPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharedContentChangeDownloadsPolicyType : has no documentation (yet)
type SharedContentChangeDownloadsPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentChangeDownloadsPolicyType returns a new SharedContentChangeDownloadsPolicyType instance
func NewSharedContentChangeDownloadsPolicyType(Description string) *SharedContentChangeDownloadsPolicyType {
	s := new(SharedContentChangeDownloadsPolicyType)
	s.Description = Description
	return s
}

// SharedContentChangeInviteeRoleDetails : Changed access type of invitee to
// shared file/folder before invite was accepted.
type SharedContentChangeInviteeRoleDetails struct {
	// PreviousAccessLevel : Previous access level. Might be missing due to
	// historical data gap.
	PreviousAccessLevel *sharing.AccessLevel `json:"previous_access_level,omitempty"`
	// NewAccessLevel : New access level.
	NewAccessLevel *sharing.AccessLevel `json:"new_access_level"`
	// Invitee : The invitee whose role was changed.
	Invitee string `json:"invitee"`
}

// NewSharedContentChangeInviteeRoleDetails returns a new SharedContentChangeInviteeRoleDetails instance
func NewSharedContentChangeInviteeRoleDetails(NewAccessLevel *sharing.AccessLevel, Invitee string) *SharedContentChangeInviteeRoleDetails {
	s := new(SharedContentChangeInviteeRoleDetails)
	s.NewAccessLevel = NewAccessLevel
	s.Invitee = Invitee
	return s
}

// SharedContentChangeInviteeRoleType : has no documentation (yet)
type SharedContentChangeInviteeRoleType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentChangeInviteeRoleType returns a new SharedContentChangeInviteeRoleType instance
func NewSharedContentChangeInviteeRoleType(Description string) *SharedContentChangeInviteeRoleType {
	s := new(SharedContentChangeInviteeRoleType)
	s.Description = Description
	return s
}

// SharedContentChangeLinkAudienceDetails : Changed link audience of shared
// file/folder.
type SharedContentChangeLinkAudienceDetails struct {
	// NewValue : New link audience value.
	NewValue *sharing.LinkAudience `json:"new_value"`
	// PreviousValue : Previous link audience value.
	PreviousValue *sharing.LinkAudience `json:"previous_value,omitempty"`
}

// NewSharedContentChangeLinkAudienceDetails returns a new SharedContentChangeLinkAudienceDetails instance
func NewSharedContentChangeLinkAudienceDetails(NewValue *sharing.LinkAudience) *SharedContentChangeLinkAudienceDetails {
	s := new(SharedContentChangeLinkAudienceDetails)
	s.NewValue = NewValue
	return s
}

// SharedContentChangeLinkAudienceType : has no documentation (yet)
type SharedContentChangeLinkAudienceType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentChangeLinkAudienceType returns a new SharedContentChangeLinkAudienceType instance
func NewSharedContentChangeLinkAudienceType(Description string) *SharedContentChangeLinkAudienceType {
	s := new(SharedContentChangeLinkAudienceType)
	s.Description = Description
	return s
}

// SharedContentChangeLinkExpiryDetails : Changed link expiration of shared
// file/folder.
type SharedContentChangeLinkExpiryDetails struct {
	// NewValue : New shared content link expiration date. Might be missing due
	// to historical data gap.
	NewValue *time.Time `json:"new_value,omitempty"`
	// PreviousValue : Previous shared content link expiration date. Might be
	// missing due to historical data gap.
	PreviousValue *time.Time `json:"previous_value,omitempty"`
}

// NewSharedContentChangeLinkExpiryDetails returns a new SharedContentChangeLinkExpiryDetails instance
func NewSharedContentChangeLinkExpiryDetails() *SharedContentChangeLinkExpiryDetails {
	s := new(SharedContentChangeLinkExpiryDetails)
	return s
}

// SharedContentChangeLinkExpiryType : has no documentation (yet)
type SharedContentChangeLinkExpiryType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentChangeLinkExpiryType returns a new SharedContentChangeLinkExpiryType instance
func NewSharedContentChangeLinkExpiryType(Description string) *SharedContentChangeLinkExpiryType {
	s := new(SharedContentChangeLinkExpiryType)
	s.Description = Description
	return s
}

// SharedContentChangeLinkPasswordDetails : Changed link password of shared
// file/folder.
type SharedContentChangeLinkPasswordDetails struct {
}

// NewSharedContentChangeLinkPasswordDetails returns a new SharedContentChangeLinkPasswordDetails instance
func NewSharedContentChangeLinkPasswordDetails() *SharedContentChangeLinkPasswordDetails {
	s := new(SharedContentChangeLinkPasswordDetails)
	return s
}

// SharedContentChangeLinkPasswordType : has no documentation (yet)
type SharedContentChangeLinkPasswordType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentChangeLinkPasswordType returns a new SharedContentChangeLinkPasswordType instance
func NewSharedContentChangeLinkPasswordType(Description string) *SharedContentChangeLinkPasswordType {
	s := new(SharedContentChangeLinkPasswordType)
	s.Description = Description
	return s
}

// SharedContentChangeMemberRoleDetails : Changed access type of shared
// file/folder member.
type SharedContentChangeMemberRoleDetails struct {
	// PreviousAccessLevel : Previous access level. Might be missing due to
	// historical data gap.
	PreviousAccessLevel *sharing.AccessLevel `json:"previous_access_level,omitempty"`
	// NewAccessLevel : New access level.
	NewAccessLevel *sharing.AccessLevel `json:"new_access_level"`
}

// NewSharedContentChangeMemberRoleDetails returns a new SharedContentChangeMemberRoleDetails instance
func NewSharedContentChangeMemberRoleDetails(NewAccessLevel *sharing.AccessLevel) *SharedContentChangeMemberRoleDetails {
	s := new(SharedContentChangeMemberRoleDetails)
	s.NewAccessLevel = NewAccessLevel
	return s
}

// SharedContentChangeMemberRoleType : has no documentation (yet)
type SharedContentChangeMemberRoleType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentChangeMemberRoleType returns a new SharedContentChangeMemberRoleType instance
func NewSharedContentChangeMemberRoleType(Description string) *SharedContentChangeMemberRoleType {
	s := new(SharedContentChangeMemberRoleType)
	s.Description = Description
	return s
}

// SharedContentChangeViewerInfoPolicyDetails : Changed whether members can see
// who viewed shared file/folder.
type SharedContentChangeViewerInfoPolicyDetails struct {
	// NewValue : New viewer info policy.
	NewValue *sharing.ViewerInfoPolicy `json:"new_value"`
	// PreviousValue : Previous view info policy.
	PreviousValue *sharing.ViewerInfoPolicy `json:"previous_value,omitempty"`
}

// NewSharedContentChangeViewerInfoPolicyDetails returns a new SharedContentChangeViewerInfoPolicyDetails instance
func NewSharedContentChangeViewerInfoPolicyDetails(NewValue *sharing.ViewerInfoPolicy) *SharedContentChangeViewerInfoPolicyDetails {
	s := new(SharedContentChangeViewerInfoPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharedContentChangeViewerInfoPolicyType : has no documentation (yet)
type SharedContentChangeViewerInfoPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentChangeViewerInfoPolicyType returns a new SharedContentChangeViewerInfoPolicyType instance
func NewSharedContentChangeViewerInfoPolicyType(Description string) *SharedContentChangeViewerInfoPolicyType {
	s := new(SharedContentChangeViewerInfoPolicyType)
	s.Description = Description
	return s
}

// SharedContentClaimInvitationDetails : Acquired membership of shared
// file/folder by accepting invite.
type SharedContentClaimInvitationDetails struct {
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
}

// NewSharedContentClaimInvitationDetails returns a new SharedContentClaimInvitationDetails instance
func NewSharedContentClaimInvitationDetails() *SharedContentClaimInvitationDetails {
	s := new(SharedContentClaimInvitationDetails)
	return s
}

// SharedContentClaimInvitationType : has no documentation (yet)
type SharedContentClaimInvitationType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentClaimInvitationType returns a new SharedContentClaimInvitationType instance
func NewSharedContentClaimInvitationType(Description string) *SharedContentClaimInvitationType {
	s := new(SharedContentClaimInvitationType)
	s.Description = Description
	return s
}

// SharedContentCopyDetails : Copied shared file/folder to own Dropbox.
type SharedContentCopyDetails struct {
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link"`
	// SharedContentOwner : The shared content owner.
	SharedContentOwner IsUserLogInfo `json:"shared_content_owner,omitempty"`
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// DestinationPath : The path where the member saved the content.
	DestinationPath string `json:"destination_path"`
}

// NewSharedContentCopyDetails returns a new SharedContentCopyDetails instance
func NewSharedContentCopyDetails(SharedContentLink string, SharedContentAccessLevel *sharing.AccessLevel, DestinationPath string) *SharedContentCopyDetails {
	s := new(SharedContentCopyDetails)
	s.SharedContentLink = SharedContentLink
	s.SharedContentAccessLevel = SharedContentAccessLevel
	s.DestinationPath = DestinationPath
	return s
}

// UnmarshalJSON deserializes into a SharedContentCopyDetails instance
func (u *SharedContentCopyDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedContentLink : Shared content link.
		SharedContentLink string `json:"shared_content_link"`
		// SharedContentAccessLevel : Shared content access level.
		SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
		// DestinationPath : The path where the member saved the content.
		DestinationPath string `json:"destination_path"`
		// SharedContentOwner : The shared content owner.
		SharedContentOwner json.RawMessage `json:"shared_content_owner,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	u.SharedContentLink = w.SharedContentLink
	u.SharedContentAccessLevel = w.SharedContentAccessLevel
	u.DestinationPath = w.DestinationPath
	SharedContentOwner, err := IsUserLogInfoFromJSON(w.SharedContentOwner)
	if err != nil {
		return err
	}
	u.SharedContentOwner = SharedContentOwner
	return nil
}

// SharedContentCopyType : has no documentation (yet)
type SharedContentCopyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentCopyType returns a new SharedContentCopyType instance
func NewSharedContentCopyType(Description string) *SharedContentCopyType {
	s := new(SharedContentCopyType)
	s.Description = Description
	return s
}

// SharedContentDownloadDetails : Downloaded shared file/folder.
type SharedContentDownloadDetails struct {
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link"`
	// SharedContentOwner : The shared content owner.
	SharedContentOwner IsUserLogInfo `json:"shared_content_owner,omitempty"`
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
}

// NewSharedContentDownloadDetails returns a new SharedContentDownloadDetails instance
func NewSharedContentDownloadDetails(SharedContentLink string, SharedContentAccessLevel *sharing.AccessLevel) *SharedContentDownloadDetails {
	s := new(SharedContentDownloadDetails)
	s.SharedContentLink = SharedContentLink
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// UnmarshalJSON deserializes into a SharedContentDownloadDetails instance
func (u *SharedContentDownloadDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedContentLink : Shared content link.
		SharedContentLink string `json:"shared_content_link"`
		// SharedContentAccessLevel : Shared content access level.
		SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
		// SharedContentOwner : The shared content owner.
		SharedContentOwner json.RawMessage `json:"shared_content_owner,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	u.SharedContentLink = w.SharedContentLink
	u.SharedContentAccessLevel = w.SharedContentAccessLevel
	SharedContentOwner, err := IsUserLogInfoFromJSON(w.SharedContentOwner)
	if err != nil {
		return err
	}
	u.SharedContentOwner = SharedContentOwner
	return nil
}

// SharedContentDownloadType : has no documentation (yet)
type SharedContentDownloadType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentDownloadType returns a new SharedContentDownloadType instance
func NewSharedContentDownloadType(Description string) *SharedContentDownloadType {
	s := new(SharedContentDownloadType)
	s.Description = Description
	return s
}

// SharedContentRelinquishMembershipDetails : Left shared file/folder.
type SharedContentRelinquishMembershipDetails struct {
}

// NewSharedContentRelinquishMembershipDetails returns a new SharedContentRelinquishMembershipDetails instance
func NewSharedContentRelinquishMembershipDetails() *SharedContentRelinquishMembershipDetails {
	s := new(SharedContentRelinquishMembershipDetails)
	return s
}

// SharedContentRelinquishMembershipType : has no documentation (yet)
type SharedContentRelinquishMembershipType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentRelinquishMembershipType returns a new SharedContentRelinquishMembershipType instance
func NewSharedContentRelinquishMembershipType(Description string) *SharedContentRelinquishMembershipType {
	s := new(SharedContentRelinquishMembershipType)
	s.Description = Description
	return s
}

// SharedContentRemoveInviteesDetails : Removed invitee from shared file/folder
// before invite was accepted.
type SharedContentRemoveInviteesDetails struct {
	// Invitees : A list of invitees.
	Invitees []string `json:"invitees"`
}

// NewSharedContentRemoveInviteesDetails returns a new SharedContentRemoveInviteesDetails instance
func NewSharedContentRemoveInviteesDetails(Invitees []string) *SharedContentRemoveInviteesDetails {
	s := new(SharedContentRemoveInviteesDetails)
	s.Invitees = Invitees
	return s
}

// SharedContentRemoveInviteesType : has no documentation (yet)
type SharedContentRemoveInviteesType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentRemoveInviteesType returns a new SharedContentRemoveInviteesType instance
func NewSharedContentRemoveInviteesType(Description string) *SharedContentRemoveInviteesType {
	s := new(SharedContentRemoveInviteesType)
	s.Description = Description
	return s
}

// SharedContentRemoveLinkExpiryDetails : Removed link expiration date of shared
// file/folder.
type SharedContentRemoveLinkExpiryDetails struct {
	// PreviousValue : Previous shared content link expiration date. Might be
	// missing due to historical data gap.
	PreviousValue *time.Time `json:"previous_value,omitempty"`
}

// NewSharedContentRemoveLinkExpiryDetails returns a new SharedContentRemoveLinkExpiryDetails instance
func NewSharedContentRemoveLinkExpiryDetails() *SharedContentRemoveLinkExpiryDetails {
	s := new(SharedContentRemoveLinkExpiryDetails)
	return s
}

// SharedContentRemoveLinkExpiryType : has no documentation (yet)
type SharedContentRemoveLinkExpiryType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentRemoveLinkExpiryType returns a new SharedContentRemoveLinkExpiryType instance
func NewSharedContentRemoveLinkExpiryType(Description string) *SharedContentRemoveLinkExpiryType {
	s := new(SharedContentRemoveLinkExpiryType)
	s.Description = Description
	return s
}

// SharedContentRemoveLinkPasswordDetails : Removed link password of shared
// file/folder.
type SharedContentRemoveLinkPasswordDetails struct {
}

// NewSharedContentRemoveLinkPasswordDetails returns a new SharedContentRemoveLinkPasswordDetails instance
func NewSharedContentRemoveLinkPasswordDetails() *SharedContentRemoveLinkPasswordDetails {
	s := new(SharedContentRemoveLinkPasswordDetails)
	return s
}

// SharedContentRemoveLinkPasswordType : has no documentation (yet)
type SharedContentRemoveLinkPasswordType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentRemoveLinkPasswordType returns a new SharedContentRemoveLinkPasswordType instance
func NewSharedContentRemoveLinkPasswordType(Description string) *SharedContentRemoveLinkPasswordType {
	s := new(SharedContentRemoveLinkPasswordType)
	s.Description = Description
	return s
}

// SharedContentRemoveMemberDetails : Removed user/group from shared
// file/folder.
type SharedContentRemoveMemberDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level,omitempty"`
}

// NewSharedContentRemoveMemberDetails returns a new SharedContentRemoveMemberDetails instance
func NewSharedContentRemoveMemberDetails() *SharedContentRemoveMemberDetails {
	s := new(SharedContentRemoveMemberDetails)
	return s
}

// SharedContentRemoveMemberType : has no documentation (yet)
type SharedContentRemoveMemberType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentRemoveMemberType returns a new SharedContentRemoveMemberType instance
func NewSharedContentRemoveMemberType(Description string) *SharedContentRemoveMemberType {
	s := new(SharedContentRemoveMemberType)
	s.Description = Description
	return s
}

// SharedContentRequestAccessDetails : Requested access to shared file/folder.
type SharedContentRequestAccessDetails struct {
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
}

// NewSharedContentRequestAccessDetails returns a new SharedContentRequestAccessDetails instance
func NewSharedContentRequestAccessDetails() *SharedContentRequestAccessDetails {
	s := new(SharedContentRequestAccessDetails)
	return s
}

// SharedContentRequestAccessType : has no documentation (yet)
type SharedContentRequestAccessType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentRequestAccessType returns a new SharedContentRequestAccessType instance
func NewSharedContentRequestAccessType(Description string) *SharedContentRequestAccessType {
	s := new(SharedContentRequestAccessType)
	s.Description = Description
	return s
}

// SharedContentRestoreInviteesDetails : Restored shared file/folder invitees.
type SharedContentRestoreInviteesDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// Invitees : A list of invitees.
	Invitees []string `json:"invitees"`
}

// NewSharedContentRestoreInviteesDetails returns a new SharedContentRestoreInviteesDetails instance
func NewSharedContentRestoreInviteesDetails(SharedContentAccessLevel *sharing.AccessLevel, Invitees []string) *SharedContentRestoreInviteesDetails {
	s := new(SharedContentRestoreInviteesDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	s.Invitees = Invitees
	return s
}

// SharedContentRestoreInviteesType : has no documentation (yet)
type SharedContentRestoreInviteesType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentRestoreInviteesType returns a new SharedContentRestoreInviteesType instance
func NewSharedContentRestoreInviteesType(Description string) *SharedContentRestoreInviteesType {
	s := new(SharedContentRestoreInviteesType)
	s.Description = Description
	return s
}

// SharedContentRestoreMemberDetails : Restored users and/or groups to
// membership of shared file/folder.
type SharedContentRestoreMemberDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
}

// NewSharedContentRestoreMemberDetails returns a new SharedContentRestoreMemberDetails instance
func NewSharedContentRestoreMemberDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedContentRestoreMemberDetails {
	s := new(SharedContentRestoreMemberDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedContentRestoreMemberType : has no documentation (yet)
type SharedContentRestoreMemberType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentRestoreMemberType returns a new SharedContentRestoreMemberType instance
func NewSharedContentRestoreMemberType(Description string) *SharedContentRestoreMemberType {
	s := new(SharedContentRestoreMemberType)
	s.Description = Description
	return s
}

// SharedContentUnshareDetails : Unshared file/folder by clearing membership.
type SharedContentUnshareDetails struct {
}

// NewSharedContentUnshareDetails returns a new SharedContentUnshareDetails instance
func NewSharedContentUnshareDetails() *SharedContentUnshareDetails {
	s := new(SharedContentUnshareDetails)
	return s
}

// SharedContentUnshareType : has no documentation (yet)
type SharedContentUnshareType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentUnshareType returns a new SharedContentUnshareType instance
func NewSharedContentUnshareType(Description string) *SharedContentUnshareType {
	s := new(SharedContentUnshareType)
	s.Description = Description
	return s
}

// SharedContentViewDetails : Previewed shared file/folder.
type SharedContentViewDetails struct {
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link"`
	// SharedContentOwner : The shared content owner.
	SharedContentOwner IsUserLogInfo `json:"shared_content_owner,omitempty"`
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
}

// NewSharedContentViewDetails returns a new SharedContentViewDetails instance
func NewSharedContentViewDetails(SharedContentLink string, SharedContentAccessLevel *sharing.AccessLevel) *SharedContentViewDetails {
	s := new(SharedContentViewDetails)
	s.SharedContentLink = SharedContentLink
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// UnmarshalJSON deserializes into a SharedContentViewDetails instance
func (u *SharedContentViewDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedContentLink : Shared content link.
		SharedContentLink string `json:"shared_content_link"`
		// SharedContentAccessLevel : Shared content access level.
		SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
		// SharedContentOwner : The shared content owner.
		SharedContentOwner json.RawMessage `json:"shared_content_owner,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	u.SharedContentLink = w.SharedContentLink
	u.SharedContentAccessLevel = w.SharedContentAccessLevel
	SharedContentOwner, err := IsUserLogInfoFromJSON(w.SharedContentOwner)
	if err != nil {
		return err
	}
	u.SharedContentOwner = SharedContentOwner
	return nil
}

// SharedContentViewType : has no documentation (yet)
type SharedContentViewType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedContentViewType returns a new SharedContentViewType instance
func NewSharedContentViewType(Description string) *SharedContentViewType {
	s := new(SharedContentViewType)
	s.Description = Description
	return s
}

// SharedFolderChangeLinkPolicyDetails : Changed who can access shared folder
// via link.
type SharedFolderChangeLinkPolicyDetails struct {
	// NewValue : New shared folder link policy.
	NewValue *sharing.SharedLinkPolicy `json:"new_value"`
	// PreviousValue : Previous shared folder link policy. Might be missing due
	// to historical data gap.
	PreviousValue *sharing.SharedLinkPolicy `json:"previous_value,omitempty"`
}

// NewSharedFolderChangeLinkPolicyDetails returns a new SharedFolderChangeLinkPolicyDetails instance
func NewSharedFolderChangeLinkPolicyDetails(NewValue *sharing.SharedLinkPolicy) *SharedFolderChangeLinkPolicyDetails {
	s := new(SharedFolderChangeLinkPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharedFolderChangeLinkPolicyType : has no documentation (yet)
type SharedFolderChangeLinkPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderChangeLinkPolicyType returns a new SharedFolderChangeLinkPolicyType instance
func NewSharedFolderChangeLinkPolicyType(Description string) *SharedFolderChangeLinkPolicyType {
	s := new(SharedFolderChangeLinkPolicyType)
	s.Description = Description
	return s
}

// SharedFolderChangeMembersInheritancePolicyDetails : Changed whether shared
// folder inherits members from parent folder.
type SharedFolderChangeMembersInheritancePolicyDetails struct {
	// NewValue : New member inheritance policy.
	NewValue *SharedFolderMembersInheritancePolicy `json:"new_value"`
	// PreviousValue : Previous member inheritance policy. Might be missing due
	// to historical data gap.
	PreviousValue *SharedFolderMembersInheritancePolicy `json:"previous_value,omitempty"`
}

// NewSharedFolderChangeMembersInheritancePolicyDetails returns a new SharedFolderChangeMembersInheritancePolicyDetails instance
func NewSharedFolderChangeMembersInheritancePolicyDetails(NewValue *SharedFolderMembersInheritancePolicy) *SharedFolderChangeMembersInheritancePolicyDetails {
	s := new(SharedFolderChangeMembersInheritancePolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharedFolderChangeMembersInheritancePolicyType : has no documentation (yet)
type SharedFolderChangeMembersInheritancePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderChangeMembersInheritancePolicyType returns a new SharedFolderChangeMembersInheritancePolicyType instance
func NewSharedFolderChangeMembersInheritancePolicyType(Description string) *SharedFolderChangeMembersInheritancePolicyType {
	s := new(SharedFolderChangeMembersInheritancePolicyType)
	s.Description = Description
	return s
}

// SharedFolderChangeMembersManagementPolicyDetails : Changed who can add/remove
// members of shared folder.
type SharedFolderChangeMembersManagementPolicyDetails struct {
	// NewValue : New members management policy.
	NewValue *sharing.AclUpdatePolicy `json:"new_value"`
	// PreviousValue : Previous members management policy. Might be missing due
	// to historical data gap.
	PreviousValue *sharing.AclUpdatePolicy `json:"previous_value,omitempty"`
}

// NewSharedFolderChangeMembersManagementPolicyDetails returns a new SharedFolderChangeMembersManagementPolicyDetails instance
func NewSharedFolderChangeMembersManagementPolicyDetails(NewValue *sharing.AclUpdatePolicy) *SharedFolderChangeMembersManagementPolicyDetails {
	s := new(SharedFolderChangeMembersManagementPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharedFolderChangeMembersManagementPolicyType : has no documentation (yet)
type SharedFolderChangeMembersManagementPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderChangeMembersManagementPolicyType returns a new SharedFolderChangeMembersManagementPolicyType instance
func NewSharedFolderChangeMembersManagementPolicyType(Description string) *SharedFolderChangeMembersManagementPolicyType {
	s := new(SharedFolderChangeMembersManagementPolicyType)
	s.Description = Description
	return s
}

// SharedFolderChangeMembersPolicyDetails : Changed who can become member of
// shared folder.
type SharedFolderChangeMembersPolicyDetails struct {
	// NewValue : New external invite policy.
	NewValue *sharing.MemberPolicy `json:"new_value"`
	// PreviousValue : Previous external invite policy. Might be missing due to
	// historical data gap.
	PreviousValue *sharing.MemberPolicy `json:"previous_value,omitempty"`
}

// NewSharedFolderChangeMembersPolicyDetails returns a new SharedFolderChangeMembersPolicyDetails instance
func NewSharedFolderChangeMembersPolicyDetails(NewValue *sharing.MemberPolicy) *SharedFolderChangeMembersPolicyDetails {
	s := new(SharedFolderChangeMembersPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharedFolderChangeMembersPolicyType : has no documentation (yet)
type SharedFolderChangeMembersPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderChangeMembersPolicyType returns a new SharedFolderChangeMembersPolicyType instance
func NewSharedFolderChangeMembersPolicyType(Description string) *SharedFolderChangeMembersPolicyType {
	s := new(SharedFolderChangeMembersPolicyType)
	s.Description = Description
	return s
}

// SharedFolderCreateDetails : Created shared folder.
type SharedFolderCreateDetails struct {
	// TargetNsId : Target namespace ID.
	TargetNsId string `json:"target_ns_id,omitempty"`
}

// NewSharedFolderCreateDetails returns a new SharedFolderCreateDetails instance
func NewSharedFolderCreateDetails() *SharedFolderCreateDetails {
	s := new(SharedFolderCreateDetails)
	return s
}

// SharedFolderCreateType : has no documentation (yet)
type SharedFolderCreateType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderCreateType returns a new SharedFolderCreateType instance
func NewSharedFolderCreateType(Description string) *SharedFolderCreateType {
	s := new(SharedFolderCreateType)
	s.Description = Description
	return s
}

// SharedFolderDeclineInvitationDetails : Declined team member's invite to
// shared folder.
type SharedFolderDeclineInvitationDetails struct {
}

// NewSharedFolderDeclineInvitationDetails returns a new SharedFolderDeclineInvitationDetails instance
func NewSharedFolderDeclineInvitationDetails() *SharedFolderDeclineInvitationDetails {
	s := new(SharedFolderDeclineInvitationDetails)
	return s
}

// SharedFolderDeclineInvitationType : has no documentation (yet)
type SharedFolderDeclineInvitationType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderDeclineInvitationType returns a new SharedFolderDeclineInvitationType instance
func NewSharedFolderDeclineInvitationType(Description string) *SharedFolderDeclineInvitationType {
	s := new(SharedFolderDeclineInvitationType)
	s.Description = Description
	return s
}

// SharedFolderMembersInheritancePolicy : Specifies if a shared folder inherits
// its members from the parent folder.
type SharedFolderMembersInheritancePolicy struct {
	dropbox.Tagged
}

// Valid tag values for SharedFolderMembersInheritancePolicy
const (
	SharedFolderMembersInheritancePolicyDontInheritMembers = "dont_inherit_members"
	SharedFolderMembersInheritancePolicyInheritMembers     = "inherit_members"
	SharedFolderMembersInheritancePolicyOther              = "other"
)

// SharedFolderMountDetails : Added shared folder to own Dropbox.
type SharedFolderMountDetails struct {
}

// NewSharedFolderMountDetails returns a new SharedFolderMountDetails instance
func NewSharedFolderMountDetails() *SharedFolderMountDetails {
	s := new(SharedFolderMountDetails)
	return s
}

// SharedFolderMountType : has no documentation (yet)
type SharedFolderMountType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderMountType returns a new SharedFolderMountType instance
func NewSharedFolderMountType(Description string) *SharedFolderMountType {
	s := new(SharedFolderMountType)
	s.Description = Description
	return s
}

// SharedFolderNestDetails : Changed parent of shared folder.
type SharedFolderNestDetails struct {
	// PreviousParentNsId : Previous parent namespace ID.
	PreviousParentNsId string `json:"previous_parent_ns_id,omitempty"`
	// NewParentNsId : New parent namespace ID.
	NewParentNsId string `json:"new_parent_ns_id,omitempty"`
	// PreviousNsPath : Previous namespace path.
	PreviousNsPath string `json:"previous_ns_path,omitempty"`
	// NewNsPath : New namespace path.
	NewNsPath string `json:"new_ns_path,omitempty"`
}

// NewSharedFolderNestDetails returns a new SharedFolderNestDetails instance
func NewSharedFolderNestDetails() *SharedFolderNestDetails {
	s := new(SharedFolderNestDetails)
	return s
}

// SharedFolderNestType : has no documentation (yet)
type SharedFolderNestType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderNestType returns a new SharedFolderNestType instance
func NewSharedFolderNestType(Description string) *SharedFolderNestType {
	s := new(SharedFolderNestType)
	s.Description = Description
	return s
}

// SharedFolderTransferOwnershipDetails : Transferred ownership of shared folder
// to another member.
type SharedFolderTransferOwnershipDetails struct {
	// PreviousOwnerEmail : The email address of the previous shared folder
	// owner.
	PreviousOwnerEmail string `json:"previous_owner_email,omitempty"`
	// NewOwnerEmail : The email address of the new shared folder owner.
	NewOwnerEmail string `json:"new_owner_email"`
}

// NewSharedFolderTransferOwnershipDetails returns a new SharedFolderTransferOwnershipDetails instance
func NewSharedFolderTransferOwnershipDetails(NewOwnerEmail string) *SharedFolderTransferOwnershipDetails {
	s := new(SharedFolderTransferOwnershipDetails)
	s.NewOwnerEmail = NewOwnerEmail
	return s
}

// SharedFolderTransferOwnershipType : has no documentation (yet)
type SharedFolderTransferOwnershipType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderTransferOwnershipType returns a new SharedFolderTransferOwnershipType instance
func NewSharedFolderTransferOwnershipType(Description string) *SharedFolderTransferOwnershipType {
	s := new(SharedFolderTransferOwnershipType)
	s.Description = Description
	return s
}

// SharedFolderUnmountDetails : Deleted shared folder from Dropbox.
type SharedFolderUnmountDetails struct {
}

// NewSharedFolderUnmountDetails returns a new SharedFolderUnmountDetails instance
func NewSharedFolderUnmountDetails() *SharedFolderUnmountDetails {
	s := new(SharedFolderUnmountDetails)
	return s
}

// SharedFolderUnmountType : has no documentation (yet)
type SharedFolderUnmountType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedFolderUnmountType returns a new SharedFolderUnmountType instance
func NewSharedFolderUnmountType(Description string) *SharedFolderUnmountType {
	s := new(SharedFolderUnmountType)
	s.Description = Description
	return s
}

// SharedLinkAccessLevel : Shared link access level.
type SharedLinkAccessLevel struct {
	dropbox.Tagged
}

// Valid tag values for SharedLinkAccessLevel
const (
	SharedLinkAccessLevelNone   = "none"
	SharedLinkAccessLevelReader = "reader"
	SharedLinkAccessLevelWriter = "writer"
	SharedLinkAccessLevelOther  = "other"
)

// SharedLinkAddExpiryDetails : Added shared link expiration date.
type SharedLinkAddExpiryDetails struct {
	// NewValue : New shared link expiration date.
	NewValue time.Time `json:"new_value"`
}

// NewSharedLinkAddExpiryDetails returns a new SharedLinkAddExpiryDetails instance
func NewSharedLinkAddExpiryDetails(NewValue time.Time) *SharedLinkAddExpiryDetails {
	s := new(SharedLinkAddExpiryDetails)
	s.NewValue = NewValue
	return s
}

// SharedLinkAddExpiryType : has no documentation (yet)
type SharedLinkAddExpiryType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkAddExpiryType returns a new SharedLinkAddExpiryType instance
func NewSharedLinkAddExpiryType(Description string) *SharedLinkAddExpiryType {
	s := new(SharedLinkAddExpiryType)
	s.Description = Description
	return s
}

// SharedLinkChangeExpiryDetails : Changed shared link expiration date.
type SharedLinkChangeExpiryDetails struct {
	// NewValue : New shared link expiration date. Might be missing due to
	// historical data gap.
	NewValue *time.Time `json:"new_value,omitempty"`
	// PreviousValue : Previous shared link expiration date. Might be missing
	// due to historical data gap.
	PreviousValue *time.Time `json:"previous_value,omitempty"`
}

// NewSharedLinkChangeExpiryDetails returns a new SharedLinkChangeExpiryDetails instance
func NewSharedLinkChangeExpiryDetails() *SharedLinkChangeExpiryDetails {
	s := new(SharedLinkChangeExpiryDetails)
	return s
}

// SharedLinkChangeExpiryType : has no documentation (yet)
type SharedLinkChangeExpiryType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkChangeExpiryType returns a new SharedLinkChangeExpiryType instance
func NewSharedLinkChangeExpiryType(Description string) *SharedLinkChangeExpiryType {
	s := new(SharedLinkChangeExpiryType)
	s.Description = Description
	return s
}

// SharedLinkChangeVisibilityDetails : Changed visibility of shared link.
type SharedLinkChangeVisibilityDetails struct {
	// NewValue : New shared link visibility.
	NewValue *SharedLinkVisibility `json:"new_value"`
	// PreviousValue : Previous shared link visibility. Might be missing due to
	// historical data gap.
	PreviousValue *SharedLinkVisibility `json:"previous_value,omitempty"`
}

// NewSharedLinkChangeVisibilityDetails returns a new SharedLinkChangeVisibilityDetails instance
func NewSharedLinkChangeVisibilityDetails(NewValue *SharedLinkVisibility) *SharedLinkChangeVisibilityDetails {
	s := new(SharedLinkChangeVisibilityDetails)
	s.NewValue = NewValue
	return s
}

// SharedLinkChangeVisibilityType : has no documentation (yet)
type SharedLinkChangeVisibilityType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkChangeVisibilityType returns a new SharedLinkChangeVisibilityType instance
func NewSharedLinkChangeVisibilityType(Description string) *SharedLinkChangeVisibilityType {
	s := new(SharedLinkChangeVisibilityType)
	s.Description = Description
	return s
}

// SharedLinkCopyDetails : Added file/folder to Dropbox from shared link.
type SharedLinkCopyDetails struct {
	// SharedLinkOwner : Shared link owner details. Might be missing due to
	// historical data gap.
	SharedLinkOwner IsUserLogInfo `json:"shared_link_owner,omitempty"`
}

// NewSharedLinkCopyDetails returns a new SharedLinkCopyDetails instance
func NewSharedLinkCopyDetails() *SharedLinkCopyDetails {
	s := new(SharedLinkCopyDetails)
	return s
}

// UnmarshalJSON deserializes into a SharedLinkCopyDetails instance
func (u *SharedLinkCopyDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedLinkOwner : Shared link owner details. Might be missing due to
		// historical data gap.
		SharedLinkOwner json.RawMessage `json:"shared_link_owner,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	SharedLinkOwner, err := IsUserLogInfoFromJSON(w.SharedLinkOwner)
	if err != nil {
		return err
	}
	u.SharedLinkOwner = SharedLinkOwner
	return nil
}

// SharedLinkCopyType : has no documentation (yet)
type SharedLinkCopyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkCopyType returns a new SharedLinkCopyType instance
func NewSharedLinkCopyType(Description string) *SharedLinkCopyType {
	s := new(SharedLinkCopyType)
	s.Description = Description
	return s
}

// SharedLinkCreateDetails : Created shared link.
type SharedLinkCreateDetails struct {
	// SharedLinkAccessLevel : Defines who can access the shared link. Might be
	// missing due to historical data gap.
	SharedLinkAccessLevel *SharedLinkAccessLevel `json:"shared_link_access_level,omitempty"`
}

// NewSharedLinkCreateDetails returns a new SharedLinkCreateDetails instance
func NewSharedLinkCreateDetails() *SharedLinkCreateDetails {
	s := new(SharedLinkCreateDetails)
	return s
}

// SharedLinkCreateType : has no documentation (yet)
type SharedLinkCreateType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkCreateType returns a new SharedLinkCreateType instance
func NewSharedLinkCreateType(Description string) *SharedLinkCreateType {
	s := new(SharedLinkCreateType)
	s.Description = Description
	return s
}

// SharedLinkDisableDetails : Removed shared link.
type SharedLinkDisableDetails struct {
	// SharedLinkOwner : Shared link owner details. Might be missing due to
	// historical data gap.
	SharedLinkOwner IsUserLogInfo `json:"shared_link_owner,omitempty"`
}

// NewSharedLinkDisableDetails returns a new SharedLinkDisableDetails instance
func NewSharedLinkDisableDetails() *SharedLinkDisableDetails {
	s := new(SharedLinkDisableDetails)
	return s
}

// UnmarshalJSON deserializes into a SharedLinkDisableDetails instance
func (u *SharedLinkDisableDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedLinkOwner : Shared link owner details. Might be missing due to
		// historical data gap.
		SharedLinkOwner json.RawMessage `json:"shared_link_owner,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	SharedLinkOwner, err := IsUserLogInfoFromJSON(w.SharedLinkOwner)
	if err != nil {
		return err
	}
	u.SharedLinkOwner = SharedLinkOwner
	return nil
}

// SharedLinkDisableType : has no documentation (yet)
type SharedLinkDisableType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkDisableType returns a new SharedLinkDisableType instance
func NewSharedLinkDisableType(Description string) *SharedLinkDisableType {
	s := new(SharedLinkDisableType)
	s.Description = Description
	return s
}

// SharedLinkDownloadDetails : Downloaded file/folder from shared link.
type SharedLinkDownloadDetails struct {
	// SharedLinkOwner : Shared link owner details. Might be missing due to
	// historical data gap.
	SharedLinkOwner IsUserLogInfo `json:"shared_link_owner,omitempty"`
}

// NewSharedLinkDownloadDetails returns a new SharedLinkDownloadDetails instance
func NewSharedLinkDownloadDetails() *SharedLinkDownloadDetails {
	s := new(SharedLinkDownloadDetails)
	return s
}

// UnmarshalJSON deserializes into a SharedLinkDownloadDetails instance
func (u *SharedLinkDownloadDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedLinkOwner : Shared link owner details. Might be missing due to
		// historical data gap.
		SharedLinkOwner json.RawMessage `json:"shared_link_owner,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	SharedLinkOwner, err := IsUserLogInfoFromJSON(w.SharedLinkOwner)
	if err != nil {
		return err
	}
	u.SharedLinkOwner = SharedLinkOwner
	return nil
}

// SharedLinkDownloadType : has no documentation (yet)
type SharedLinkDownloadType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkDownloadType returns a new SharedLinkDownloadType instance
func NewSharedLinkDownloadType(Description string) *SharedLinkDownloadType {
	s := new(SharedLinkDownloadType)
	s.Description = Description
	return s
}

// SharedLinkRemoveExpiryDetails : Removed shared link expiration date.
type SharedLinkRemoveExpiryDetails struct {
	// PreviousValue : Previous shared link expiration date. Might be missing
	// due to historical data gap.
	PreviousValue *time.Time `json:"previous_value,omitempty"`
}

// NewSharedLinkRemoveExpiryDetails returns a new SharedLinkRemoveExpiryDetails instance
func NewSharedLinkRemoveExpiryDetails() *SharedLinkRemoveExpiryDetails {
	s := new(SharedLinkRemoveExpiryDetails)
	return s
}

// SharedLinkRemoveExpiryType : has no documentation (yet)
type SharedLinkRemoveExpiryType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkRemoveExpiryType returns a new SharedLinkRemoveExpiryType instance
func NewSharedLinkRemoveExpiryType(Description string) *SharedLinkRemoveExpiryType {
	s := new(SharedLinkRemoveExpiryType)
	s.Description = Description
	return s
}

// SharedLinkSettingsAddExpirationDetails : Added an expiration date to the
// shared link.
type SharedLinkSettingsAddExpirationDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
	// NewValue : New shared content link expiration date. Might be missing due
	// to historical data gap.
	NewValue *time.Time `json:"new_value,omitempty"`
}

// NewSharedLinkSettingsAddExpirationDetails returns a new SharedLinkSettingsAddExpirationDetails instance
func NewSharedLinkSettingsAddExpirationDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedLinkSettingsAddExpirationDetails {
	s := new(SharedLinkSettingsAddExpirationDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedLinkSettingsAddExpirationType : has no documentation (yet)
type SharedLinkSettingsAddExpirationType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkSettingsAddExpirationType returns a new SharedLinkSettingsAddExpirationType instance
func NewSharedLinkSettingsAddExpirationType(Description string) *SharedLinkSettingsAddExpirationType {
	s := new(SharedLinkSettingsAddExpirationType)
	s.Description = Description
	return s
}

// SharedLinkSettingsAddPasswordDetails : Added a password to the shared link.
type SharedLinkSettingsAddPasswordDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
}

// NewSharedLinkSettingsAddPasswordDetails returns a new SharedLinkSettingsAddPasswordDetails instance
func NewSharedLinkSettingsAddPasswordDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedLinkSettingsAddPasswordDetails {
	s := new(SharedLinkSettingsAddPasswordDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedLinkSettingsAddPasswordType : has no documentation (yet)
type SharedLinkSettingsAddPasswordType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkSettingsAddPasswordType returns a new SharedLinkSettingsAddPasswordType instance
func NewSharedLinkSettingsAddPasswordType(Description string) *SharedLinkSettingsAddPasswordType {
	s := new(SharedLinkSettingsAddPasswordType)
	s.Description = Description
	return s
}

// SharedLinkSettingsAllowDownloadDisabledDetails : Disabled downloads.
type SharedLinkSettingsAllowDownloadDisabledDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
}

// NewSharedLinkSettingsAllowDownloadDisabledDetails returns a new SharedLinkSettingsAllowDownloadDisabledDetails instance
func NewSharedLinkSettingsAllowDownloadDisabledDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedLinkSettingsAllowDownloadDisabledDetails {
	s := new(SharedLinkSettingsAllowDownloadDisabledDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedLinkSettingsAllowDownloadDisabledType : has no documentation (yet)
type SharedLinkSettingsAllowDownloadDisabledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkSettingsAllowDownloadDisabledType returns a new SharedLinkSettingsAllowDownloadDisabledType instance
func NewSharedLinkSettingsAllowDownloadDisabledType(Description string) *SharedLinkSettingsAllowDownloadDisabledType {
	s := new(SharedLinkSettingsAllowDownloadDisabledType)
	s.Description = Description
	return s
}

// SharedLinkSettingsAllowDownloadEnabledDetails : Enabled downloads.
type SharedLinkSettingsAllowDownloadEnabledDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
}

// NewSharedLinkSettingsAllowDownloadEnabledDetails returns a new SharedLinkSettingsAllowDownloadEnabledDetails instance
func NewSharedLinkSettingsAllowDownloadEnabledDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedLinkSettingsAllowDownloadEnabledDetails {
	s := new(SharedLinkSettingsAllowDownloadEnabledDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedLinkSettingsAllowDownloadEnabledType : has no documentation (yet)
type SharedLinkSettingsAllowDownloadEnabledType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkSettingsAllowDownloadEnabledType returns a new SharedLinkSettingsAllowDownloadEnabledType instance
func NewSharedLinkSettingsAllowDownloadEnabledType(Description string) *SharedLinkSettingsAllowDownloadEnabledType {
	s := new(SharedLinkSettingsAllowDownloadEnabledType)
	s.Description = Description
	return s
}

// SharedLinkSettingsChangeAudienceDetails : Changed the audience of the shared
// link.
type SharedLinkSettingsChangeAudienceDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
	// NewValue : New link audience value.
	NewValue *sharing.LinkAudience `json:"new_value"`
	// PreviousValue : Previous link audience value.
	PreviousValue *sharing.LinkAudience `json:"previous_value,omitempty"`
}

// NewSharedLinkSettingsChangeAudienceDetails returns a new SharedLinkSettingsChangeAudienceDetails instance
func NewSharedLinkSettingsChangeAudienceDetails(SharedContentAccessLevel *sharing.AccessLevel, NewValue *sharing.LinkAudience) *SharedLinkSettingsChangeAudienceDetails {
	s := new(SharedLinkSettingsChangeAudienceDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	s.NewValue = NewValue
	return s
}

// SharedLinkSettingsChangeAudienceType : has no documentation (yet)
type SharedLinkSettingsChangeAudienceType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkSettingsChangeAudienceType returns a new SharedLinkSettingsChangeAudienceType instance
func NewSharedLinkSettingsChangeAudienceType(Description string) *SharedLinkSettingsChangeAudienceType {
	s := new(SharedLinkSettingsChangeAudienceType)
	s.Description = Description
	return s
}

// SharedLinkSettingsChangeExpirationDetails : Changed the expiration date of
// the shared link.
type SharedLinkSettingsChangeExpirationDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
	// NewValue : New shared content link expiration date. Might be missing due
	// to historical data gap.
	NewValue *time.Time `json:"new_value,omitempty"`
	// PreviousValue : Previous shared content link expiration date. Might be
	// missing due to historical data gap.
	PreviousValue *time.Time `json:"previous_value,omitempty"`
}

// NewSharedLinkSettingsChangeExpirationDetails returns a new SharedLinkSettingsChangeExpirationDetails instance
func NewSharedLinkSettingsChangeExpirationDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedLinkSettingsChangeExpirationDetails {
	s := new(SharedLinkSettingsChangeExpirationDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedLinkSettingsChangeExpirationType : has no documentation (yet)
type SharedLinkSettingsChangeExpirationType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkSettingsChangeExpirationType returns a new SharedLinkSettingsChangeExpirationType instance
func NewSharedLinkSettingsChangeExpirationType(Description string) *SharedLinkSettingsChangeExpirationType {
	s := new(SharedLinkSettingsChangeExpirationType)
	s.Description = Description
	return s
}

// SharedLinkSettingsChangePasswordDetails : Changed the password of the shared
// link.
type SharedLinkSettingsChangePasswordDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
}

// NewSharedLinkSettingsChangePasswordDetails returns a new SharedLinkSettingsChangePasswordDetails instance
func NewSharedLinkSettingsChangePasswordDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedLinkSettingsChangePasswordDetails {
	s := new(SharedLinkSettingsChangePasswordDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedLinkSettingsChangePasswordType : has no documentation (yet)
type SharedLinkSettingsChangePasswordType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkSettingsChangePasswordType returns a new SharedLinkSettingsChangePasswordType instance
func NewSharedLinkSettingsChangePasswordType(Description string) *SharedLinkSettingsChangePasswordType {
	s := new(SharedLinkSettingsChangePasswordType)
	s.Description = Description
	return s
}

// SharedLinkSettingsRemoveExpirationDetails : Removed the expiration date from
// the shared link.
type SharedLinkSettingsRemoveExpirationDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
	// PreviousValue : Previous shared link expiration date. Might be missing
	// due to historical data gap.
	PreviousValue *time.Time `json:"previous_value,omitempty"`
}

// NewSharedLinkSettingsRemoveExpirationDetails returns a new SharedLinkSettingsRemoveExpirationDetails instance
func NewSharedLinkSettingsRemoveExpirationDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedLinkSettingsRemoveExpirationDetails {
	s := new(SharedLinkSettingsRemoveExpirationDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedLinkSettingsRemoveExpirationType : has no documentation (yet)
type SharedLinkSettingsRemoveExpirationType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkSettingsRemoveExpirationType returns a new SharedLinkSettingsRemoveExpirationType instance
func NewSharedLinkSettingsRemoveExpirationType(Description string) *SharedLinkSettingsRemoveExpirationType {
	s := new(SharedLinkSettingsRemoveExpirationType)
	s.Description = Description
	return s
}

// SharedLinkSettingsRemovePasswordDetails : Removed the password from the
// shared link.
type SharedLinkSettingsRemovePasswordDetails struct {
	// SharedContentAccessLevel : Shared content access level.
	SharedContentAccessLevel *sharing.AccessLevel `json:"shared_content_access_level"`
	// SharedContentLink : Shared content link.
	SharedContentLink string `json:"shared_content_link,omitempty"`
}

// NewSharedLinkSettingsRemovePasswordDetails returns a new SharedLinkSettingsRemovePasswordDetails instance
func NewSharedLinkSettingsRemovePasswordDetails(SharedContentAccessLevel *sharing.AccessLevel) *SharedLinkSettingsRemovePasswordDetails {
	s := new(SharedLinkSettingsRemovePasswordDetails)
	s.SharedContentAccessLevel = SharedContentAccessLevel
	return s
}

// SharedLinkSettingsRemovePasswordType : has no documentation (yet)
type SharedLinkSettingsRemovePasswordType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkSettingsRemovePasswordType returns a new SharedLinkSettingsRemovePasswordType instance
func NewSharedLinkSettingsRemovePasswordType(Description string) *SharedLinkSettingsRemovePasswordType {
	s := new(SharedLinkSettingsRemovePasswordType)
	s.Description = Description
	return s
}

// SharedLinkShareDetails : Added members as audience of shared link.
type SharedLinkShareDetails struct {
	// SharedLinkOwner : Shared link owner details. Might be missing due to
	// historical data gap.
	SharedLinkOwner IsUserLogInfo `json:"shared_link_owner,omitempty"`
	// ExternalUsers : Users without a Dropbox account that were added as shared
	// link audience.
	ExternalUsers []*ExternalUserLogInfo `json:"external_users,omitempty"`
}

// NewSharedLinkShareDetails returns a new SharedLinkShareDetails instance
func NewSharedLinkShareDetails() *SharedLinkShareDetails {
	s := new(SharedLinkShareDetails)
	return s
}

// UnmarshalJSON deserializes into a SharedLinkShareDetails instance
func (u *SharedLinkShareDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedLinkOwner : Shared link owner details. Might be missing due to
		// historical data gap.
		SharedLinkOwner json.RawMessage `json:"shared_link_owner,omitempty"`
		// ExternalUsers : Users without a Dropbox account that were added as
		// shared link audience.
		ExternalUsers []*ExternalUserLogInfo `json:"external_users,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	SharedLinkOwner, err := IsUserLogInfoFromJSON(w.SharedLinkOwner)
	if err != nil {
		return err
	}
	u.SharedLinkOwner = SharedLinkOwner
	u.ExternalUsers = w.ExternalUsers
	return nil
}

// SharedLinkShareType : has no documentation (yet)
type SharedLinkShareType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkShareType returns a new SharedLinkShareType instance
func NewSharedLinkShareType(Description string) *SharedLinkShareType {
	s := new(SharedLinkShareType)
	s.Description = Description
	return s
}

// SharedLinkViewDetails : Opened shared link.
type SharedLinkViewDetails struct {
	// SharedLinkOwner : Shared link owner details. Might be missing due to
	// historical data gap.
	SharedLinkOwner IsUserLogInfo `json:"shared_link_owner,omitempty"`
}

// NewSharedLinkViewDetails returns a new SharedLinkViewDetails instance
func NewSharedLinkViewDetails() *SharedLinkViewDetails {
	s := new(SharedLinkViewDetails)
	return s
}

// UnmarshalJSON deserializes into a SharedLinkViewDetails instance
func (u *SharedLinkViewDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedLinkOwner : Shared link owner details. Might be missing due to
		// historical data gap.
		SharedLinkOwner json.RawMessage `json:"shared_link_owner,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	SharedLinkOwner, err := IsUserLogInfoFromJSON(w.SharedLinkOwner)
	if err != nil {
		return err
	}
	u.SharedLinkOwner = SharedLinkOwner
	return nil
}

// SharedLinkViewType : has no documentation (yet)
type SharedLinkViewType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedLinkViewType returns a new SharedLinkViewType instance
func NewSharedLinkViewType(Description string) *SharedLinkViewType {
	s := new(SharedLinkViewType)
	s.Description = Description
	return s
}

// SharedLinkVisibility : Defines who has access to a shared link.
type SharedLinkVisibility struct {
	dropbox.Tagged
}

// Valid tag values for SharedLinkVisibility
const (
	SharedLinkVisibilityNoOne    = "no_one"
	SharedLinkVisibilityPassword = "password"
	SharedLinkVisibilityPublic   = "public"
	SharedLinkVisibilityTeamOnly = "team_only"
	SharedLinkVisibilityOther    = "other"
)

// SharedNoteOpenedDetails : Opened shared Paper doc.
type SharedNoteOpenedDetails struct {
}

// NewSharedNoteOpenedDetails returns a new SharedNoteOpenedDetails instance
func NewSharedNoteOpenedDetails() *SharedNoteOpenedDetails {
	s := new(SharedNoteOpenedDetails)
	return s
}

// SharedNoteOpenedType : has no documentation (yet)
type SharedNoteOpenedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharedNoteOpenedType returns a new SharedNoteOpenedType instance
func NewSharedNoteOpenedType(Description string) *SharedNoteOpenedType {
	s := new(SharedNoteOpenedType)
	s.Description = Description
	return s
}

// SharingChangeFolderJoinPolicyDetails : Changed whether team members can join
// shared folders owned outside team.
type SharingChangeFolderJoinPolicyDetails struct {
	// NewValue : New external join policy.
	NewValue *SharingFolderJoinPolicy `json:"new_value"`
	// PreviousValue : Previous external join policy. Might be missing due to
	// historical data gap.
	PreviousValue *SharingFolderJoinPolicy `json:"previous_value,omitempty"`
}

// NewSharingChangeFolderJoinPolicyDetails returns a new SharingChangeFolderJoinPolicyDetails instance
func NewSharingChangeFolderJoinPolicyDetails(NewValue *SharingFolderJoinPolicy) *SharingChangeFolderJoinPolicyDetails {
	s := new(SharingChangeFolderJoinPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharingChangeFolderJoinPolicyType : has no documentation (yet)
type SharingChangeFolderJoinPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharingChangeFolderJoinPolicyType returns a new SharingChangeFolderJoinPolicyType instance
func NewSharingChangeFolderJoinPolicyType(Description string) *SharingChangeFolderJoinPolicyType {
	s := new(SharingChangeFolderJoinPolicyType)
	s.Description = Description
	return s
}

// SharingChangeLinkAllowChangeExpirationPolicyDetails : Changed the allow
// remove or change expiration policy for the links shared outside of the team.
type SharingChangeLinkAllowChangeExpirationPolicyDetails struct {
	// NewValue : To.
	NewValue *EnforceLinkPasswordPolicy `json:"new_value"`
	// PreviousValue : From.
	PreviousValue *EnforceLinkPasswordPolicy `json:"previous_value,omitempty"`
}

// NewSharingChangeLinkAllowChangeExpirationPolicyDetails returns a new SharingChangeLinkAllowChangeExpirationPolicyDetails instance
func NewSharingChangeLinkAllowChangeExpirationPolicyDetails(NewValue *EnforceLinkPasswordPolicy) *SharingChangeLinkAllowChangeExpirationPolicyDetails {
	s := new(SharingChangeLinkAllowChangeExpirationPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharingChangeLinkAllowChangeExpirationPolicyType : has no documentation (yet)
type SharingChangeLinkAllowChangeExpirationPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharingChangeLinkAllowChangeExpirationPolicyType returns a new SharingChangeLinkAllowChangeExpirationPolicyType instance
func NewSharingChangeLinkAllowChangeExpirationPolicyType(Description string) *SharingChangeLinkAllowChangeExpirationPolicyType {
	s := new(SharingChangeLinkAllowChangeExpirationPolicyType)
	s.Description = Description
	return s
}

// SharingChangeLinkDefaultExpirationPolicyDetails : Changed the default
// expiration for the links shared outside of the team.
type SharingChangeLinkDefaultExpirationPolicyDetails struct {
	// NewValue : To.
	NewValue *DefaultLinkExpirationDaysPolicy `json:"new_value"`
	// PreviousValue : From.
	PreviousValue *DefaultLinkExpirationDaysPolicy `json:"previous_value,omitempty"`
}

// NewSharingChangeLinkDefaultExpirationPolicyDetails returns a new SharingChangeLinkDefaultExpirationPolicyDetails instance
func NewSharingChangeLinkDefaultExpirationPolicyDetails(NewValue *DefaultLinkExpirationDaysPolicy) *SharingChangeLinkDefaultExpirationPolicyDetails {
	s := new(SharingChangeLinkDefaultExpirationPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharingChangeLinkDefaultExpirationPolicyType : has no documentation (yet)
type SharingChangeLinkDefaultExpirationPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharingChangeLinkDefaultExpirationPolicyType returns a new SharingChangeLinkDefaultExpirationPolicyType instance
func NewSharingChangeLinkDefaultExpirationPolicyType(Description string) *SharingChangeLinkDefaultExpirationPolicyType {
	s := new(SharingChangeLinkDefaultExpirationPolicyType)
	s.Description = Description
	return s
}

// SharingChangeLinkEnforcePasswordPolicyDetails : Changed the password
// requirement for the links shared outside of the team.
type SharingChangeLinkEnforcePasswordPolicyDetails struct {
	// NewValue : To.
	NewValue *ChangeLinkExpirationPolicy `json:"new_value"`
	// PreviousValue : From.
	PreviousValue *ChangeLinkExpirationPolicy `json:"previous_value,omitempty"`
}

// NewSharingChangeLinkEnforcePasswordPolicyDetails returns a new SharingChangeLinkEnforcePasswordPolicyDetails instance
func NewSharingChangeLinkEnforcePasswordPolicyDetails(NewValue *ChangeLinkExpirationPolicy) *SharingChangeLinkEnforcePasswordPolicyDetails {
	s := new(SharingChangeLinkEnforcePasswordPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharingChangeLinkEnforcePasswordPolicyType : has no documentation (yet)
type SharingChangeLinkEnforcePasswordPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharingChangeLinkEnforcePasswordPolicyType returns a new SharingChangeLinkEnforcePasswordPolicyType instance
func NewSharingChangeLinkEnforcePasswordPolicyType(Description string) *SharingChangeLinkEnforcePasswordPolicyType {
	s := new(SharingChangeLinkEnforcePasswordPolicyType)
	s.Description = Description
	return s
}

// SharingChangeLinkPolicyDetails : Changed whether members can share links
// outside team, and if links are accessible only by team members or anyone by
// default.
type SharingChangeLinkPolicyDetails struct {
	// NewValue : New external link accessibility policy.
	NewValue *SharingLinkPolicy `json:"new_value"`
	// PreviousValue : Previous external link accessibility policy. Might be
	// missing due to historical data gap.
	PreviousValue *SharingLinkPolicy `json:"previous_value,omitempty"`
}

// NewSharingChangeLinkPolicyDetails returns a new SharingChangeLinkPolicyDetails instance
func NewSharingChangeLinkPolicyDetails(NewValue *SharingLinkPolicy) *SharingChangeLinkPolicyDetails {
	s := new(SharingChangeLinkPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharingChangeLinkPolicyType : has no documentation (yet)
type SharingChangeLinkPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharingChangeLinkPolicyType returns a new SharingChangeLinkPolicyType instance
func NewSharingChangeLinkPolicyType(Description string) *SharingChangeLinkPolicyType {
	s := new(SharingChangeLinkPolicyType)
	s.Description = Description
	return s
}

// SharingChangeMemberPolicyDetails : Changed whether members can share
// files/folders outside team.
type SharingChangeMemberPolicyDetails struct {
	// NewValue : New external invite policy.
	NewValue *SharingMemberPolicy `json:"new_value"`
	// PreviousValue : Previous external invite policy. Might be missing due to
	// historical data gap.
	PreviousValue *SharingMemberPolicy `json:"previous_value,omitempty"`
}

// NewSharingChangeMemberPolicyDetails returns a new SharingChangeMemberPolicyDetails instance
func NewSharingChangeMemberPolicyDetails(NewValue *SharingMemberPolicy) *SharingChangeMemberPolicyDetails {
	s := new(SharingChangeMemberPolicyDetails)
	s.NewValue = NewValue
	return s
}

// SharingChangeMemberPolicyType : has no documentation (yet)
type SharingChangeMemberPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSharingChangeMemberPolicyType returns a new SharingChangeMemberPolicyType instance
func NewSharingChangeMemberPolicyType(Description string) *SharingChangeMemberPolicyType {
	s := new(SharingChangeMemberPolicyType)
	s.Description = Description
	return s
}

// SharingFolderJoinPolicy : Policy for controlling if team members can join
// shared folders owned by non team members.
type SharingFolderJoinPolicy struct {
	dropbox.Tagged
}

// Valid tag values for SharingFolderJoinPolicy
const (
	SharingFolderJoinPolicyFromAnyone   = "from_anyone"
	SharingFolderJoinPolicyFromTeamOnly = "from_team_only"
	SharingFolderJoinPolicyOther        = "other"
)

// SharingLinkPolicy : Policy for controlling if team members can share links
// externally
type SharingLinkPolicy struct {
	dropbox.Tagged
}

// Valid tag values for SharingLinkPolicy
const (
	SharingLinkPolicyDefaultNoOne   = "default_no_one"
	SharingLinkPolicyDefaultPrivate = "default_private"
	SharingLinkPolicyDefaultPublic  = "default_public"
	SharingLinkPolicyOnlyPrivate    = "only_private"
	SharingLinkPolicyOther          = "other"
)

// SharingMemberPolicy : External sharing policy
type SharingMemberPolicy struct {
	dropbox.Tagged
}

// Valid tag values for SharingMemberPolicy
const (
	SharingMemberPolicyAllow                = "allow"
	SharingMemberPolicyForbid               = "forbid"
	SharingMemberPolicyForbidWithExclusions = "forbid_with_exclusions"
	SharingMemberPolicyOther                = "other"
)

// ShmodelDisableDownloadsDetails : Disabled downloads for link.
type ShmodelDisableDownloadsDetails struct {
	// SharedLinkOwner : Shared link owner details. Might be missing due to
	// historical data gap.
	SharedLinkOwner IsUserLogInfo `json:"shared_link_owner,omitempty"`
}

// NewShmodelDisableDownloadsDetails returns a new ShmodelDisableDownloadsDetails instance
func NewShmodelDisableDownloadsDetails() *ShmodelDisableDownloadsDetails {
	s := new(ShmodelDisableDownloadsDetails)
	return s
}

// UnmarshalJSON deserializes into a ShmodelDisableDownloadsDetails instance
func (u *ShmodelDisableDownloadsDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedLinkOwner : Shared link owner details. Might be missing due to
		// historical data gap.
		SharedLinkOwner json.RawMessage `json:"shared_link_owner,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	SharedLinkOwner, err := IsUserLogInfoFromJSON(w.SharedLinkOwner)
	if err != nil {
		return err
	}
	u.SharedLinkOwner = SharedLinkOwner
	return nil
}

// ShmodelDisableDownloadsType : has no documentation (yet)
type ShmodelDisableDownloadsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShmodelDisableDownloadsType returns a new ShmodelDisableDownloadsType instance
func NewShmodelDisableDownloadsType(Description string) *ShmodelDisableDownloadsType {
	s := new(ShmodelDisableDownloadsType)
	s.Description = Description
	return s
}

// ShmodelEnableDownloadsDetails : Enabled downloads for link.
type ShmodelEnableDownloadsDetails struct {
	// SharedLinkOwner : Shared link owner details. Might be missing due to
	// historical data gap.
	SharedLinkOwner IsUserLogInfo `json:"shared_link_owner,omitempty"`
}

// NewShmodelEnableDownloadsDetails returns a new ShmodelEnableDownloadsDetails instance
func NewShmodelEnableDownloadsDetails() *ShmodelEnableDownloadsDetails {
	s := new(ShmodelEnableDownloadsDetails)
	return s
}

// UnmarshalJSON deserializes into a ShmodelEnableDownloadsDetails instance
func (u *ShmodelEnableDownloadsDetails) UnmarshalJSON(b []byte) error {
	type wrap struct {
		// SharedLinkOwner : Shared link owner details. Might be missing due to
		// historical data gap.
		SharedLinkOwner json.RawMessage `json:"shared_link_owner,omitempty"`
	}
	var w wrap
	if err := json.Unmarshal(b, &w); err != nil {
		return err
	}
	SharedLinkOwner, err := IsUserLogInfoFromJSON(w.SharedLinkOwner)
	if err != nil {
		return err
	}
	u.SharedLinkOwner = SharedLinkOwner
	return nil
}

// ShmodelEnableDownloadsType : has no documentation (yet)
type ShmodelEnableDownloadsType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShmodelEnableDownloadsType returns a new ShmodelEnableDownloadsType instance
func NewShmodelEnableDownloadsType(Description string) *ShmodelEnableDownloadsType {
	s := new(ShmodelEnableDownloadsType)
	s.Description = Description
	return s
}

// ShmodelGroupShareDetails : Shared link with group.
type ShmodelGroupShareDetails struct {
}

// NewShmodelGroupShareDetails returns a new ShmodelGroupShareDetails instance
func NewShmodelGroupShareDetails() *ShmodelGroupShareDetails {
	s := new(ShmodelGroupShareDetails)
	return s
}

// ShmodelGroupShareType : has no documentation (yet)
type ShmodelGroupShareType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShmodelGroupShareType returns a new ShmodelGroupShareType instance
func NewShmodelGroupShareType(Description string) *ShmodelGroupShareType {
	s := new(ShmodelGroupShareType)
	s.Description = Description
	return s
}

// ShowcaseAccessGrantedDetails : Granted access to showcase.
type ShowcaseAccessGrantedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseAccessGrantedDetails returns a new ShowcaseAccessGrantedDetails instance
func NewShowcaseAccessGrantedDetails(EventUuid string) *ShowcaseAccessGrantedDetails {
	s := new(ShowcaseAccessGrantedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseAccessGrantedType : has no documentation (yet)
type ShowcaseAccessGrantedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseAccessGrantedType returns a new ShowcaseAccessGrantedType instance
func NewShowcaseAccessGrantedType(Description string) *ShowcaseAccessGrantedType {
	s := new(ShowcaseAccessGrantedType)
	s.Description = Description
	return s
}

// ShowcaseAddMemberDetails : Added member to showcase.
type ShowcaseAddMemberDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseAddMemberDetails returns a new ShowcaseAddMemberDetails instance
func NewShowcaseAddMemberDetails(EventUuid string) *ShowcaseAddMemberDetails {
	s := new(ShowcaseAddMemberDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseAddMemberType : has no documentation (yet)
type ShowcaseAddMemberType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseAddMemberType returns a new ShowcaseAddMemberType instance
func NewShowcaseAddMemberType(Description string) *ShowcaseAddMemberType {
	s := new(ShowcaseAddMemberType)
	s.Description = Description
	return s
}

// ShowcaseArchivedDetails : Archived showcase.
type ShowcaseArchivedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseArchivedDetails returns a new ShowcaseArchivedDetails instance
func NewShowcaseArchivedDetails(EventUuid string) *ShowcaseArchivedDetails {
	s := new(ShowcaseArchivedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseArchivedType : has no documentation (yet)
type ShowcaseArchivedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseArchivedType returns a new ShowcaseArchivedType instance
func NewShowcaseArchivedType(Description string) *ShowcaseArchivedType {
	s := new(ShowcaseArchivedType)
	s.Description = Description
	return s
}

// ShowcaseChangeDownloadPolicyDetails : Enabled/disabled downloading files from
// Dropbox Showcase for team.
type ShowcaseChangeDownloadPolicyDetails struct {
	// NewValue : New Dropbox Showcase download policy.
	NewValue *ShowcaseDownloadPolicy `json:"new_value"`
	// PreviousValue : Previous Dropbox Showcase download policy.
	PreviousValue *ShowcaseDownloadPolicy `json:"previous_value"`
}

// NewShowcaseChangeDownloadPolicyDetails returns a new ShowcaseChangeDownloadPolicyDetails instance
func NewShowcaseChangeDownloadPolicyDetails(NewValue *ShowcaseDownloadPolicy, PreviousValue *ShowcaseDownloadPolicy) *ShowcaseChangeDownloadPolicyDetails {
	s := new(ShowcaseChangeDownloadPolicyDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// ShowcaseChangeDownloadPolicyType : has no documentation (yet)
type ShowcaseChangeDownloadPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseChangeDownloadPolicyType returns a new ShowcaseChangeDownloadPolicyType instance
func NewShowcaseChangeDownloadPolicyType(Description string) *ShowcaseChangeDownloadPolicyType {
	s := new(ShowcaseChangeDownloadPolicyType)
	s.Description = Description
	return s
}

// ShowcaseChangeEnabledPolicyDetails : Enabled/disabled Dropbox Showcase for
// team.
type ShowcaseChangeEnabledPolicyDetails struct {
	// NewValue : New Dropbox Showcase policy.
	NewValue *ShowcaseEnabledPolicy `json:"new_value"`
	// PreviousValue : Previous Dropbox Showcase policy.
	PreviousValue *ShowcaseEnabledPolicy `json:"previous_value"`
}

// NewShowcaseChangeEnabledPolicyDetails returns a new ShowcaseChangeEnabledPolicyDetails instance
func NewShowcaseChangeEnabledPolicyDetails(NewValue *ShowcaseEnabledPolicy, PreviousValue *ShowcaseEnabledPolicy) *ShowcaseChangeEnabledPolicyDetails {
	s := new(ShowcaseChangeEnabledPolicyDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// ShowcaseChangeEnabledPolicyType : has no documentation (yet)
type ShowcaseChangeEnabledPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseChangeEnabledPolicyType returns a new ShowcaseChangeEnabledPolicyType instance
func NewShowcaseChangeEnabledPolicyType(Description string) *ShowcaseChangeEnabledPolicyType {
	s := new(ShowcaseChangeEnabledPolicyType)
	s.Description = Description
	return s
}

// ShowcaseChangeExternalSharingPolicyDetails : Enabled/disabled sharing Dropbox
// Showcase externally for team.
type ShowcaseChangeExternalSharingPolicyDetails struct {
	// NewValue : New Dropbox Showcase external sharing policy.
	NewValue *ShowcaseExternalSharingPolicy `json:"new_value"`
	// PreviousValue : Previous Dropbox Showcase external sharing policy.
	PreviousValue *ShowcaseExternalSharingPolicy `json:"previous_value"`
}

// NewShowcaseChangeExternalSharingPolicyDetails returns a new ShowcaseChangeExternalSharingPolicyDetails instance
func NewShowcaseChangeExternalSharingPolicyDetails(NewValue *ShowcaseExternalSharingPolicy, PreviousValue *ShowcaseExternalSharingPolicy) *ShowcaseChangeExternalSharingPolicyDetails {
	s := new(ShowcaseChangeExternalSharingPolicyDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// ShowcaseChangeExternalSharingPolicyType : has no documentation (yet)
type ShowcaseChangeExternalSharingPolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseChangeExternalSharingPolicyType returns a new ShowcaseChangeExternalSharingPolicyType instance
func NewShowcaseChangeExternalSharingPolicyType(Description string) *ShowcaseChangeExternalSharingPolicyType {
	s := new(ShowcaseChangeExternalSharingPolicyType)
	s.Description = Description
	return s
}

// ShowcaseCreatedDetails : Created showcase.
type ShowcaseCreatedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseCreatedDetails returns a new ShowcaseCreatedDetails instance
func NewShowcaseCreatedDetails(EventUuid string) *ShowcaseCreatedDetails {
	s := new(ShowcaseCreatedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseCreatedType : has no documentation (yet)
type ShowcaseCreatedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseCreatedType returns a new ShowcaseCreatedType instance
func NewShowcaseCreatedType(Description string) *ShowcaseCreatedType {
	s := new(ShowcaseCreatedType)
	s.Description = Description
	return s
}

// ShowcaseDeleteCommentDetails : Deleted showcase comment.
type ShowcaseDeleteCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewShowcaseDeleteCommentDetails returns a new ShowcaseDeleteCommentDetails instance
func NewShowcaseDeleteCommentDetails(EventUuid string) *ShowcaseDeleteCommentDetails {
	s := new(ShowcaseDeleteCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseDeleteCommentType : has no documentation (yet)
type ShowcaseDeleteCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseDeleteCommentType returns a new ShowcaseDeleteCommentType instance
func NewShowcaseDeleteCommentType(Description string) *ShowcaseDeleteCommentType {
	s := new(ShowcaseDeleteCommentType)
	s.Description = Description
	return s
}

// ShowcaseDocumentLogInfo : Showcase document's logged information.
type ShowcaseDocumentLogInfo struct {
	// ShowcaseId : Showcase document Id.
	ShowcaseId string `json:"showcase_id"`
	// ShowcaseTitle : Showcase document title.
	ShowcaseTitle string `json:"showcase_title"`
}

// NewShowcaseDocumentLogInfo returns a new ShowcaseDocumentLogInfo instance
func NewShowcaseDocumentLogInfo(ShowcaseId string, ShowcaseTitle string) *ShowcaseDocumentLogInfo {
	s := new(ShowcaseDocumentLogInfo)
	s.ShowcaseId = ShowcaseId
	s.ShowcaseTitle = ShowcaseTitle
	return s
}

// ShowcaseDownloadPolicy : Policy for controlling if files can be downloaded
// from Showcases by team members
type ShowcaseDownloadPolicy struct {
	dropbox.Tagged
}

// Valid tag values for ShowcaseDownloadPolicy
const (
	ShowcaseDownloadPolicyDisabled = "disabled"
	ShowcaseDownloadPolicyEnabled  = "enabled"
	ShowcaseDownloadPolicyOther    = "other"
)

// ShowcaseEditCommentDetails : Edited showcase comment.
type ShowcaseEditCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewShowcaseEditCommentDetails returns a new ShowcaseEditCommentDetails instance
func NewShowcaseEditCommentDetails(EventUuid string) *ShowcaseEditCommentDetails {
	s := new(ShowcaseEditCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseEditCommentType : has no documentation (yet)
type ShowcaseEditCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseEditCommentType returns a new ShowcaseEditCommentType instance
func NewShowcaseEditCommentType(Description string) *ShowcaseEditCommentType {
	s := new(ShowcaseEditCommentType)
	s.Description = Description
	return s
}

// ShowcaseEditedDetails : Edited showcase.
type ShowcaseEditedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseEditedDetails returns a new ShowcaseEditedDetails instance
func NewShowcaseEditedDetails(EventUuid string) *ShowcaseEditedDetails {
	s := new(ShowcaseEditedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseEditedType : has no documentation (yet)
type ShowcaseEditedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseEditedType returns a new ShowcaseEditedType instance
func NewShowcaseEditedType(Description string) *ShowcaseEditedType {
	s := new(ShowcaseEditedType)
	s.Description = Description
	return s
}

// ShowcaseEnabledPolicy : Policy for controlling whether Showcase is enabled.
type ShowcaseEnabledPolicy struct {
	dropbox.Tagged
}

// Valid tag values for ShowcaseEnabledPolicy
const (
	ShowcaseEnabledPolicyDisabled = "disabled"
	ShowcaseEnabledPolicyEnabled  = "enabled"
	ShowcaseEnabledPolicyOther    = "other"
)

// ShowcaseExternalSharingPolicy : Policy for controlling if team members can
// share Showcases externally.
type ShowcaseExternalSharingPolicy struct {
	dropbox.Tagged
}

// Valid tag values for ShowcaseExternalSharingPolicy
const (
	ShowcaseExternalSharingPolicyDisabled = "disabled"
	ShowcaseExternalSharingPolicyEnabled  = "enabled"
	ShowcaseExternalSharingPolicyOther    = "other"
)

// ShowcaseFileAddedDetails : Added file to showcase.
type ShowcaseFileAddedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseFileAddedDetails returns a new ShowcaseFileAddedDetails instance
func NewShowcaseFileAddedDetails(EventUuid string) *ShowcaseFileAddedDetails {
	s := new(ShowcaseFileAddedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseFileAddedType : has no documentation (yet)
type ShowcaseFileAddedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseFileAddedType returns a new ShowcaseFileAddedType instance
func NewShowcaseFileAddedType(Description string) *ShowcaseFileAddedType {
	s := new(ShowcaseFileAddedType)
	s.Description = Description
	return s
}

// ShowcaseFileDownloadDetails : Downloaded file from showcase.
type ShowcaseFileDownloadDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// DownloadType : Showcase download type.
	DownloadType string `json:"download_type"`
}

// NewShowcaseFileDownloadDetails returns a new ShowcaseFileDownloadDetails instance
func NewShowcaseFileDownloadDetails(EventUuid string, DownloadType string) *ShowcaseFileDownloadDetails {
	s := new(ShowcaseFileDownloadDetails)
	s.EventUuid = EventUuid
	s.DownloadType = DownloadType
	return s
}

// ShowcaseFileDownloadType : has no documentation (yet)
type ShowcaseFileDownloadType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseFileDownloadType returns a new ShowcaseFileDownloadType instance
func NewShowcaseFileDownloadType(Description string) *ShowcaseFileDownloadType {
	s := new(ShowcaseFileDownloadType)
	s.Description = Description
	return s
}

// ShowcaseFileRemovedDetails : Removed file from showcase.
type ShowcaseFileRemovedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseFileRemovedDetails returns a new ShowcaseFileRemovedDetails instance
func NewShowcaseFileRemovedDetails(EventUuid string) *ShowcaseFileRemovedDetails {
	s := new(ShowcaseFileRemovedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseFileRemovedType : has no documentation (yet)
type ShowcaseFileRemovedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseFileRemovedType returns a new ShowcaseFileRemovedType instance
func NewShowcaseFileRemovedType(Description string) *ShowcaseFileRemovedType {
	s := new(ShowcaseFileRemovedType)
	s.Description = Description
	return s
}

// ShowcaseFileViewDetails : Viewed file in showcase.
type ShowcaseFileViewDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseFileViewDetails returns a new ShowcaseFileViewDetails instance
func NewShowcaseFileViewDetails(EventUuid string) *ShowcaseFileViewDetails {
	s := new(ShowcaseFileViewDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseFileViewType : has no documentation (yet)
type ShowcaseFileViewType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseFileViewType returns a new ShowcaseFileViewType instance
func NewShowcaseFileViewType(Description string) *ShowcaseFileViewType {
	s := new(ShowcaseFileViewType)
	s.Description = Description
	return s
}

// ShowcasePermanentlyDeletedDetails : Permanently deleted showcase.
type ShowcasePermanentlyDeletedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcasePermanentlyDeletedDetails returns a new ShowcasePermanentlyDeletedDetails instance
func NewShowcasePermanentlyDeletedDetails(EventUuid string) *ShowcasePermanentlyDeletedDetails {
	s := new(ShowcasePermanentlyDeletedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcasePermanentlyDeletedType : has no documentation (yet)
type ShowcasePermanentlyDeletedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcasePermanentlyDeletedType returns a new ShowcasePermanentlyDeletedType instance
func NewShowcasePermanentlyDeletedType(Description string) *ShowcasePermanentlyDeletedType {
	s := new(ShowcasePermanentlyDeletedType)
	s.Description = Description
	return s
}

// ShowcasePostCommentDetails : Added showcase comment.
type ShowcasePostCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewShowcasePostCommentDetails returns a new ShowcasePostCommentDetails instance
func NewShowcasePostCommentDetails(EventUuid string) *ShowcasePostCommentDetails {
	s := new(ShowcasePostCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcasePostCommentType : has no documentation (yet)
type ShowcasePostCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcasePostCommentType returns a new ShowcasePostCommentType instance
func NewShowcasePostCommentType(Description string) *ShowcasePostCommentType {
	s := new(ShowcasePostCommentType)
	s.Description = Description
	return s
}

// ShowcaseRemoveMemberDetails : Removed member from showcase.
type ShowcaseRemoveMemberDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseRemoveMemberDetails returns a new ShowcaseRemoveMemberDetails instance
func NewShowcaseRemoveMemberDetails(EventUuid string) *ShowcaseRemoveMemberDetails {
	s := new(ShowcaseRemoveMemberDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseRemoveMemberType : has no documentation (yet)
type ShowcaseRemoveMemberType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseRemoveMemberType returns a new ShowcaseRemoveMemberType instance
func NewShowcaseRemoveMemberType(Description string) *ShowcaseRemoveMemberType {
	s := new(ShowcaseRemoveMemberType)
	s.Description = Description
	return s
}

// ShowcaseRenamedDetails : Renamed showcase.
type ShowcaseRenamedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseRenamedDetails returns a new ShowcaseRenamedDetails instance
func NewShowcaseRenamedDetails(EventUuid string) *ShowcaseRenamedDetails {
	s := new(ShowcaseRenamedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseRenamedType : has no documentation (yet)
type ShowcaseRenamedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseRenamedType returns a new ShowcaseRenamedType instance
func NewShowcaseRenamedType(Description string) *ShowcaseRenamedType {
	s := new(ShowcaseRenamedType)
	s.Description = Description
	return s
}

// ShowcaseRequestAccessDetails : Requested access to showcase.
type ShowcaseRequestAccessDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseRequestAccessDetails returns a new ShowcaseRequestAccessDetails instance
func NewShowcaseRequestAccessDetails(EventUuid string) *ShowcaseRequestAccessDetails {
	s := new(ShowcaseRequestAccessDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseRequestAccessType : has no documentation (yet)
type ShowcaseRequestAccessType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseRequestAccessType returns a new ShowcaseRequestAccessType instance
func NewShowcaseRequestAccessType(Description string) *ShowcaseRequestAccessType {
	s := new(ShowcaseRequestAccessType)
	s.Description = Description
	return s
}

// ShowcaseResolveCommentDetails : Resolved showcase comment.
type ShowcaseResolveCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewShowcaseResolveCommentDetails returns a new ShowcaseResolveCommentDetails instance
func NewShowcaseResolveCommentDetails(EventUuid string) *ShowcaseResolveCommentDetails {
	s := new(ShowcaseResolveCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseResolveCommentType : has no documentation (yet)
type ShowcaseResolveCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseResolveCommentType returns a new ShowcaseResolveCommentType instance
func NewShowcaseResolveCommentType(Description string) *ShowcaseResolveCommentType {
	s := new(ShowcaseResolveCommentType)
	s.Description = Description
	return s
}

// ShowcaseRestoredDetails : Unarchived showcase.
type ShowcaseRestoredDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseRestoredDetails returns a new ShowcaseRestoredDetails instance
func NewShowcaseRestoredDetails(EventUuid string) *ShowcaseRestoredDetails {
	s := new(ShowcaseRestoredDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseRestoredType : has no documentation (yet)
type ShowcaseRestoredType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseRestoredType returns a new ShowcaseRestoredType instance
func NewShowcaseRestoredType(Description string) *ShowcaseRestoredType {
	s := new(ShowcaseRestoredType)
	s.Description = Description
	return s
}

// ShowcaseTrashedDeprecatedDetails : Deleted showcase (old version).
type ShowcaseTrashedDeprecatedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseTrashedDeprecatedDetails returns a new ShowcaseTrashedDeprecatedDetails instance
func NewShowcaseTrashedDeprecatedDetails(EventUuid string) *ShowcaseTrashedDeprecatedDetails {
	s := new(ShowcaseTrashedDeprecatedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseTrashedDeprecatedType : has no documentation (yet)
type ShowcaseTrashedDeprecatedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseTrashedDeprecatedType returns a new ShowcaseTrashedDeprecatedType instance
func NewShowcaseTrashedDeprecatedType(Description string) *ShowcaseTrashedDeprecatedType {
	s := new(ShowcaseTrashedDeprecatedType)
	s.Description = Description
	return s
}

// ShowcaseTrashedDetails : Deleted showcase.
type ShowcaseTrashedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseTrashedDetails returns a new ShowcaseTrashedDetails instance
func NewShowcaseTrashedDetails(EventUuid string) *ShowcaseTrashedDetails {
	s := new(ShowcaseTrashedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseTrashedType : has no documentation (yet)
type ShowcaseTrashedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseTrashedType returns a new ShowcaseTrashedType instance
func NewShowcaseTrashedType(Description string) *ShowcaseTrashedType {
	s := new(ShowcaseTrashedType)
	s.Description = Description
	return s
}

// ShowcaseUnresolveCommentDetails : Unresolved showcase comment.
type ShowcaseUnresolveCommentDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
	// CommentText : Comment text.
	CommentText string `json:"comment_text,omitempty"`
}

// NewShowcaseUnresolveCommentDetails returns a new ShowcaseUnresolveCommentDetails instance
func NewShowcaseUnresolveCommentDetails(EventUuid string) *ShowcaseUnresolveCommentDetails {
	s := new(ShowcaseUnresolveCommentDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseUnresolveCommentType : has no documentation (yet)
type ShowcaseUnresolveCommentType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseUnresolveCommentType returns a new ShowcaseUnresolveCommentType instance
func NewShowcaseUnresolveCommentType(Description string) *ShowcaseUnresolveCommentType {
	s := new(ShowcaseUnresolveCommentType)
	s.Description = Description
	return s
}

// ShowcaseUntrashedDeprecatedDetails : Restored showcase (old version).
type ShowcaseUntrashedDeprecatedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseUntrashedDeprecatedDetails returns a new ShowcaseUntrashedDeprecatedDetails instance
func NewShowcaseUntrashedDeprecatedDetails(EventUuid string) *ShowcaseUntrashedDeprecatedDetails {
	s := new(ShowcaseUntrashedDeprecatedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseUntrashedDeprecatedType : has no documentation (yet)
type ShowcaseUntrashedDeprecatedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseUntrashedDeprecatedType returns a new ShowcaseUntrashedDeprecatedType instance
func NewShowcaseUntrashedDeprecatedType(Description string) *ShowcaseUntrashedDeprecatedType {
	s := new(ShowcaseUntrashedDeprecatedType)
	s.Description = Description
	return s
}

// ShowcaseUntrashedDetails : Restored showcase.
type ShowcaseUntrashedDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseUntrashedDetails returns a new ShowcaseUntrashedDetails instance
func NewShowcaseUntrashedDetails(EventUuid string) *ShowcaseUntrashedDetails {
	s := new(ShowcaseUntrashedDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseUntrashedType : has no documentation (yet)
type ShowcaseUntrashedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseUntrashedType returns a new ShowcaseUntrashedType instance
func NewShowcaseUntrashedType(Description string) *ShowcaseUntrashedType {
	s := new(ShowcaseUntrashedType)
	s.Description = Description
	return s
}

// ShowcaseViewDetails : Viewed showcase.
type ShowcaseViewDetails struct {
	// EventUuid : Event unique identifier.
	EventUuid string `json:"event_uuid"`
}

// NewShowcaseViewDetails returns a new ShowcaseViewDetails instance
func NewShowcaseViewDetails(EventUuid string) *ShowcaseViewDetails {
	s := new(ShowcaseViewDetails)
	s.EventUuid = EventUuid
	return s
}

// ShowcaseViewType : has no documentation (yet)
type ShowcaseViewType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewShowcaseViewType returns a new ShowcaseViewType instance
func NewShowcaseViewType(Description string) *ShowcaseViewType {
	s := new(ShowcaseViewType)
	s.Description = Description
	return s
}

// SignInAsSessionEndDetails : Ended admin sign-in-as session.
type SignInAsSessionEndDetails struct {
}

// NewSignInAsSessionEndDetails returns a new SignInAsSessionEndDetails instance
func NewSignInAsSessionEndDetails() *SignInAsSessionEndDetails {
	s := new(SignInAsSessionEndDetails)
	return s
}

// SignInAsSessionEndType : has no documentation (yet)
type SignInAsSessionEndType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSignInAsSessionEndType returns a new SignInAsSessionEndType instance
func NewSignInAsSessionEndType(Description string) *SignInAsSessionEndType {
	s := new(SignInAsSessionEndType)
	s.Description = Description
	return s
}

// SignInAsSessionStartDetails : Started admin sign-in-as session.
type SignInAsSessionStartDetails struct {
}

// NewSignInAsSessionStartDetails returns a new SignInAsSessionStartDetails instance
func NewSignInAsSessionStartDetails() *SignInAsSessionStartDetails {
	s := new(SignInAsSessionStartDetails)
	return s
}

// SignInAsSessionStartType : has no documentation (yet)
type SignInAsSessionStartType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSignInAsSessionStartType returns a new SignInAsSessionStartType instance
func NewSignInAsSessionStartType(Description string) *SignInAsSessionStartType {
	s := new(SignInAsSessionStartType)
	s.Description = Description
	return s
}

// SmartSyncChangePolicyDetails : Changed default Smart Sync setting for team
// members.
type SmartSyncChangePolicyDetails struct {
	// NewValue : New smart sync policy.
	NewValue *team_policies.SmartSyncPolicy `json:"new_value,omitempty"`
	// PreviousValue : Previous smart sync policy.
	PreviousValue *team_policies.SmartSyncPolicy `json:"previous_value,omitempty"`
}

// NewSmartSyncChangePolicyDetails returns a new SmartSyncChangePolicyDetails instance
func NewSmartSyncChangePolicyDetails() *SmartSyncChangePolicyDetails {
	s := new(SmartSyncChangePolicyDetails)
	return s
}

// SmartSyncChangePolicyType : has no documentation (yet)
type SmartSyncChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSmartSyncChangePolicyType returns a new SmartSyncChangePolicyType instance
func NewSmartSyncChangePolicyType(Description string) *SmartSyncChangePolicyType {
	s := new(SmartSyncChangePolicyType)
	s.Description = Description
	return s
}

// SmartSyncCreateAdminPrivilegeReportDetails : Created Smart Sync non-admin
// devices report.
type SmartSyncCreateAdminPrivilegeReportDetails struct {
}

// NewSmartSyncCreateAdminPrivilegeReportDetails returns a new SmartSyncCreateAdminPrivilegeReportDetails instance
func NewSmartSyncCreateAdminPrivilegeReportDetails() *SmartSyncCreateAdminPrivilegeReportDetails {
	s := new(SmartSyncCreateAdminPrivilegeReportDetails)
	return s
}

// SmartSyncCreateAdminPrivilegeReportType : has no documentation (yet)
type SmartSyncCreateAdminPrivilegeReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSmartSyncCreateAdminPrivilegeReportType returns a new SmartSyncCreateAdminPrivilegeReportType instance
func NewSmartSyncCreateAdminPrivilegeReportType(Description string) *SmartSyncCreateAdminPrivilegeReportType {
	s := new(SmartSyncCreateAdminPrivilegeReportType)
	s.Description = Description
	return s
}

// SmartSyncNotOptOutDetails : Opted team into Smart Sync.
type SmartSyncNotOptOutDetails struct {
	// PreviousValue : Previous Smart Sync opt out policy.
	PreviousValue *SmartSyncOptOutPolicy `json:"previous_value"`
	// NewValue : New Smart Sync opt out policy.
	NewValue *SmartSyncOptOutPolicy `json:"new_value"`
}

// NewSmartSyncNotOptOutDetails returns a new SmartSyncNotOptOutDetails instance
func NewSmartSyncNotOptOutDetails(PreviousValue *SmartSyncOptOutPolicy, NewValue *SmartSyncOptOutPolicy) *SmartSyncNotOptOutDetails {
	s := new(SmartSyncNotOptOutDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// SmartSyncNotOptOutType : has no documentation (yet)
type SmartSyncNotOptOutType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSmartSyncNotOptOutType returns a new SmartSyncNotOptOutType instance
func NewSmartSyncNotOptOutType(Description string) *SmartSyncNotOptOutType {
	s := new(SmartSyncNotOptOutType)
	s.Description = Description
	return s
}

// SmartSyncOptOutDetails : Opted team out of Smart Sync.
type SmartSyncOptOutDetails struct {
	// PreviousValue : Previous Smart Sync opt out policy.
	PreviousValue *SmartSyncOptOutPolicy `json:"previous_value"`
	// NewValue : New Smart Sync opt out policy.
	NewValue *SmartSyncOptOutPolicy `json:"new_value"`
}

// NewSmartSyncOptOutDetails returns a new SmartSyncOptOutDetails instance
func NewSmartSyncOptOutDetails(PreviousValue *SmartSyncOptOutPolicy, NewValue *SmartSyncOptOutPolicy) *SmartSyncOptOutDetails {
	s := new(SmartSyncOptOutDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// SmartSyncOptOutPolicy : has no documentation (yet)
type SmartSyncOptOutPolicy struct {
	dropbox.Tagged
}

// Valid tag values for SmartSyncOptOutPolicy
const (
	SmartSyncOptOutPolicyDefault  = "default"
	SmartSyncOptOutPolicyOptedOut = "opted_out"
	SmartSyncOptOutPolicyOther    = "other"
)

// SmartSyncOptOutType : has no documentation (yet)
type SmartSyncOptOutType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSmartSyncOptOutType returns a new SmartSyncOptOutType instance
func NewSmartSyncOptOutType(Description string) *SmartSyncOptOutType {
	s := new(SmartSyncOptOutType)
	s.Description = Description
	return s
}

// SmarterSmartSyncPolicyChangedDetails : Changed automatic Smart Sync setting
// for team.
type SmarterSmartSyncPolicyChangedDetails struct {
	// PreviousValue : Previous automatic Smart Sync setting.
	PreviousValue *team_policies.SmarterSmartSyncPolicyState `json:"previous_value"`
	// NewValue : New automatic Smart Sync setting.
	NewValue *team_policies.SmarterSmartSyncPolicyState `json:"new_value"`
}

// NewSmarterSmartSyncPolicyChangedDetails returns a new SmarterSmartSyncPolicyChangedDetails instance
func NewSmarterSmartSyncPolicyChangedDetails(PreviousValue *team_policies.SmarterSmartSyncPolicyState, NewValue *team_policies.SmarterSmartSyncPolicyState) *SmarterSmartSyncPolicyChangedDetails {
	s := new(SmarterSmartSyncPolicyChangedDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// SmarterSmartSyncPolicyChangedType : has no documentation (yet)
type SmarterSmartSyncPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSmarterSmartSyncPolicyChangedType returns a new SmarterSmartSyncPolicyChangedType instance
func NewSmarterSmartSyncPolicyChangedType(Description string) *SmarterSmartSyncPolicyChangedType {
	s := new(SmarterSmartSyncPolicyChangedType)
	s.Description = Description
	return s
}

// SpaceCapsType : Space limit alert policy
type SpaceCapsType struct {
	dropbox.Tagged
}

// Valid tag values for SpaceCapsType
const (
	SpaceCapsTypeHard  = "hard"
	SpaceCapsTypeOff   = "off"
	SpaceCapsTypeSoft  = "soft"
	SpaceCapsTypeOther = "other"
)

// SpaceLimitsStatus : has no documentation (yet)
type SpaceLimitsStatus struct {
	dropbox.Tagged
}

// Valid tag values for SpaceLimitsStatus
const (
	SpaceLimitsStatusNearQuota   = "near_quota"
	SpaceLimitsStatusOverQuota   = "over_quota"
	SpaceLimitsStatusWithinQuota = "within_quota"
	SpaceLimitsStatusOther       = "other"
)

// SsoAddCertDetails : Added X.509 certificate for SSO.
type SsoAddCertDetails struct {
	// CertificateDetails : SSO certificate details.
	CertificateDetails *Certificate `json:"certificate_details"`
}

// NewSsoAddCertDetails returns a new SsoAddCertDetails instance
func NewSsoAddCertDetails(CertificateDetails *Certificate) *SsoAddCertDetails {
	s := new(SsoAddCertDetails)
	s.CertificateDetails = CertificateDetails
	return s
}

// SsoAddCertType : has no documentation (yet)
type SsoAddCertType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoAddCertType returns a new SsoAddCertType instance
func NewSsoAddCertType(Description string) *SsoAddCertType {
	s := new(SsoAddCertType)
	s.Description = Description
	return s
}

// SsoAddLoginUrlDetails : Added sign-in URL for SSO.
type SsoAddLoginUrlDetails struct {
	// NewValue : New single sign-on login URL.
	NewValue string `json:"new_value"`
}

// NewSsoAddLoginUrlDetails returns a new SsoAddLoginUrlDetails instance
func NewSsoAddLoginUrlDetails(NewValue string) *SsoAddLoginUrlDetails {
	s := new(SsoAddLoginUrlDetails)
	s.NewValue = NewValue
	return s
}

// SsoAddLoginUrlType : has no documentation (yet)
type SsoAddLoginUrlType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoAddLoginUrlType returns a new SsoAddLoginUrlType instance
func NewSsoAddLoginUrlType(Description string) *SsoAddLoginUrlType {
	s := new(SsoAddLoginUrlType)
	s.Description = Description
	return s
}

// SsoAddLogoutUrlDetails : Added sign-out URL for SSO.
type SsoAddLogoutUrlDetails struct {
	// NewValue : New single sign-on logout URL.
	NewValue string `json:"new_value,omitempty"`
}

// NewSsoAddLogoutUrlDetails returns a new SsoAddLogoutUrlDetails instance
func NewSsoAddLogoutUrlDetails() *SsoAddLogoutUrlDetails {
	s := new(SsoAddLogoutUrlDetails)
	return s
}

// SsoAddLogoutUrlType : has no documentation (yet)
type SsoAddLogoutUrlType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoAddLogoutUrlType returns a new SsoAddLogoutUrlType instance
func NewSsoAddLogoutUrlType(Description string) *SsoAddLogoutUrlType {
	s := new(SsoAddLogoutUrlType)
	s.Description = Description
	return s
}

// SsoChangeCertDetails : Changed X.509 certificate for SSO.
type SsoChangeCertDetails struct {
	// PreviousCertificateDetails : Previous SSO certificate details. Might be
	// missing due to historical data gap.
	PreviousCertificateDetails *Certificate `json:"previous_certificate_details,omitempty"`
	// NewCertificateDetails : New SSO certificate details.
	NewCertificateDetails *Certificate `json:"new_certificate_details"`
}

// NewSsoChangeCertDetails returns a new SsoChangeCertDetails instance
func NewSsoChangeCertDetails(NewCertificateDetails *Certificate) *SsoChangeCertDetails {
	s := new(SsoChangeCertDetails)
	s.NewCertificateDetails = NewCertificateDetails
	return s
}

// SsoChangeCertType : has no documentation (yet)
type SsoChangeCertType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoChangeCertType returns a new SsoChangeCertType instance
func NewSsoChangeCertType(Description string) *SsoChangeCertType {
	s := new(SsoChangeCertType)
	s.Description = Description
	return s
}

// SsoChangeLoginUrlDetails : Changed sign-in URL for SSO.
type SsoChangeLoginUrlDetails struct {
	// PreviousValue : Previous single sign-on login URL.
	PreviousValue string `json:"previous_value"`
	// NewValue : New single sign-on login URL.
	NewValue string `json:"new_value"`
}

// NewSsoChangeLoginUrlDetails returns a new SsoChangeLoginUrlDetails instance
func NewSsoChangeLoginUrlDetails(PreviousValue string, NewValue string) *SsoChangeLoginUrlDetails {
	s := new(SsoChangeLoginUrlDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// SsoChangeLoginUrlType : has no documentation (yet)
type SsoChangeLoginUrlType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoChangeLoginUrlType returns a new SsoChangeLoginUrlType instance
func NewSsoChangeLoginUrlType(Description string) *SsoChangeLoginUrlType {
	s := new(SsoChangeLoginUrlType)
	s.Description = Description
	return s
}

// SsoChangeLogoutUrlDetails : Changed sign-out URL for SSO.
type SsoChangeLogoutUrlDetails struct {
	// PreviousValue : Previous single sign-on logout URL. Might be missing due
	// to historical data gap.
	PreviousValue string `json:"previous_value,omitempty"`
	// NewValue : New single sign-on logout URL.
	NewValue string `json:"new_value,omitempty"`
}

// NewSsoChangeLogoutUrlDetails returns a new SsoChangeLogoutUrlDetails instance
func NewSsoChangeLogoutUrlDetails() *SsoChangeLogoutUrlDetails {
	s := new(SsoChangeLogoutUrlDetails)
	return s
}

// SsoChangeLogoutUrlType : has no documentation (yet)
type SsoChangeLogoutUrlType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoChangeLogoutUrlType returns a new SsoChangeLogoutUrlType instance
func NewSsoChangeLogoutUrlType(Description string) *SsoChangeLogoutUrlType {
	s := new(SsoChangeLogoutUrlType)
	s.Description = Description
	return s
}

// SsoChangePolicyDetails : Changed single sign-on setting for team.
type SsoChangePolicyDetails struct {
	// NewValue : New single sign-on policy.
	NewValue *team_policies.SsoPolicy `json:"new_value"`
	// PreviousValue : Previous single sign-on policy. Might be missing due to
	// historical data gap.
	PreviousValue *team_policies.SsoPolicy `json:"previous_value,omitempty"`
}

// NewSsoChangePolicyDetails returns a new SsoChangePolicyDetails instance
func NewSsoChangePolicyDetails(NewValue *team_policies.SsoPolicy) *SsoChangePolicyDetails {
	s := new(SsoChangePolicyDetails)
	s.NewValue = NewValue
	return s
}

// SsoChangePolicyType : has no documentation (yet)
type SsoChangePolicyType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoChangePolicyType returns a new SsoChangePolicyType instance
func NewSsoChangePolicyType(Description string) *SsoChangePolicyType {
	s := new(SsoChangePolicyType)
	s.Description = Description
	return s
}

// SsoChangeSamlIdentityModeDetails : Changed SAML identity mode for SSO.
type SsoChangeSamlIdentityModeDetails struct {
	// PreviousValue : Previous single sign-on identity mode.
	PreviousValue int64 `json:"previous_value"`
	// NewValue : New single sign-on identity mode.
	NewValue int64 `json:"new_value"`
}

// NewSsoChangeSamlIdentityModeDetails returns a new SsoChangeSamlIdentityModeDetails instance
func NewSsoChangeSamlIdentityModeDetails(PreviousValue int64, NewValue int64) *SsoChangeSamlIdentityModeDetails {
	s := new(SsoChangeSamlIdentityModeDetails)
	s.PreviousValue = PreviousValue
	s.NewValue = NewValue
	return s
}

// SsoChangeSamlIdentityModeType : has no documentation (yet)
type SsoChangeSamlIdentityModeType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoChangeSamlIdentityModeType returns a new SsoChangeSamlIdentityModeType instance
func NewSsoChangeSamlIdentityModeType(Description string) *SsoChangeSamlIdentityModeType {
	s := new(SsoChangeSamlIdentityModeType)
	s.Description = Description
	return s
}

// SsoErrorDetails : Failed to sign in via SSO.
type SsoErrorDetails struct {
	// ErrorDetails : Error details.
	ErrorDetails *FailureDetailsLogInfo `json:"error_details"`
}

// NewSsoErrorDetails returns a new SsoErrorDetails instance
func NewSsoErrorDetails(ErrorDetails *FailureDetailsLogInfo) *SsoErrorDetails {
	s := new(SsoErrorDetails)
	s.ErrorDetails = ErrorDetails
	return s
}

// SsoErrorType : has no documentation (yet)
type SsoErrorType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoErrorType returns a new SsoErrorType instance
func NewSsoErrorType(Description string) *SsoErrorType {
	s := new(SsoErrorType)
	s.Description = Description
	return s
}

// SsoRemoveCertDetails : Removed X.509 certificate for SSO.
type SsoRemoveCertDetails struct {
}

// NewSsoRemoveCertDetails returns a new SsoRemoveCertDetails instance
func NewSsoRemoveCertDetails() *SsoRemoveCertDetails {
	s := new(SsoRemoveCertDetails)
	return s
}

// SsoRemoveCertType : has no documentation (yet)
type SsoRemoveCertType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoRemoveCertType returns a new SsoRemoveCertType instance
func NewSsoRemoveCertType(Description string) *SsoRemoveCertType {
	s := new(SsoRemoveCertType)
	s.Description = Description
	return s
}

// SsoRemoveLoginUrlDetails : Removed sign-in URL for SSO.
type SsoRemoveLoginUrlDetails struct {
	// PreviousValue : Previous single sign-on login URL.
	PreviousValue string `json:"previous_value"`
}

// NewSsoRemoveLoginUrlDetails returns a new SsoRemoveLoginUrlDetails instance
func NewSsoRemoveLoginUrlDetails(PreviousValue string) *SsoRemoveLoginUrlDetails {
	s := new(SsoRemoveLoginUrlDetails)
	s.PreviousValue = PreviousValue
	return s
}

// SsoRemoveLoginUrlType : has no documentation (yet)
type SsoRemoveLoginUrlType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoRemoveLoginUrlType returns a new SsoRemoveLoginUrlType instance
func NewSsoRemoveLoginUrlType(Description string) *SsoRemoveLoginUrlType {
	s := new(SsoRemoveLoginUrlType)
	s.Description = Description
	return s
}

// SsoRemoveLogoutUrlDetails : Removed sign-out URL for SSO.
type SsoRemoveLogoutUrlDetails struct {
	// PreviousValue : Previous single sign-on logout URL.
	PreviousValue string `json:"previous_value"`
}

// NewSsoRemoveLogoutUrlDetails returns a new SsoRemoveLogoutUrlDetails instance
func NewSsoRemoveLogoutUrlDetails(PreviousValue string) *SsoRemoveLogoutUrlDetails {
	s := new(SsoRemoveLogoutUrlDetails)
	s.PreviousValue = PreviousValue
	return s
}

// SsoRemoveLogoutUrlType : has no documentation (yet)
type SsoRemoveLogoutUrlType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewSsoRemoveLogoutUrlType returns a new SsoRemoveLogoutUrlType instance
func NewSsoRemoveLogoutUrlType(Description string) *SsoRemoveLogoutUrlType {
	s := new(SsoRemoveLogoutUrlType)
	s.Description = Description
	return s
}

// StartedEnterpriseAdminSessionDetails : Started enterprise admin session.
type StartedEnterpriseAdminSessionDetails struct {
	// FederationExtraDetails : More information about the organization or team.
	FederationExtraDetails *FedExtraDetails `json:"federation_extra_details"`
}

// NewStartedEnterpriseAdminSessionDetails returns a new StartedEnterpriseAdminSessionDetails instance
func NewStartedEnterpriseAdminSessionDetails(FederationExtraDetails *FedExtraDetails) *StartedEnterpriseAdminSessionDetails {
	s := new(StartedEnterpriseAdminSessionDetails)
	s.FederationExtraDetails = FederationExtraDetails
	return s
}

// StartedEnterpriseAdminSessionType : has no documentation (yet)
type StartedEnterpriseAdminSessionType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewStartedEnterpriseAdminSessionType returns a new StartedEnterpriseAdminSessionType instance
func NewStartedEnterpriseAdminSessionType(Description string) *StartedEnterpriseAdminSessionType {
	s := new(StartedEnterpriseAdminSessionType)
	s.Description = Description
	return s
}

// TeamActivityCreateReportDetails : Created team activity report.
type TeamActivityCreateReportDetails struct {
	// StartDate : Report start date.
	StartDate time.Time `json:"start_date"`
	// EndDate : Report end date.
	EndDate time.Time `json:"end_date"`
}

// NewTeamActivityCreateReportDetails returns a new TeamActivityCreateReportDetails instance
func NewTeamActivityCreateReportDetails(StartDate time.Time, EndDate time.Time) *TeamActivityCreateReportDetails {
	s := new(TeamActivityCreateReportDetails)
	s.StartDate = StartDate
	s.EndDate = EndDate
	return s
}

// TeamActivityCreateReportFailDetails : Couldn't generate team activity report.
type TeamActivityCreateReportFailDetails struct {
	// FailureReason : Failure reason.
	FailureReason *team.TeamReportFailureReason `json:"failure_reason"`
}

// NewTeamActivityCreateReportFailDetails returns a new TeamActivityCreateReportFailDetails instance
func NewTeamActivityCreateReportFailDetails(FailureReason *team.TeamReportFailureReason) *TeamActivityCreateReportFailDetails {
	s := new(TeamActivityCreateReportFailDetails)
	s.FailureReason = FailureReason
	return s
}

// TeamActivityCreateReportFailType : has no documentation (yet)
type TeamActivityCreateReportFailType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewTeamActivityCreateReportFailType returns a new TeamActivityCreateReportFailType instance
func NewTeamActivityCreateReportFailType(Description string) *TeamActivityCreateReportFailType {
	s := new(TeamActivityCreateReportFailType)
	s.Description = Description
	return s
}

// TeamActivityCreateReportType : has no documentation (yet)
type TeamActivityCreateReportType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewTeamActivityCreateReportType returns a new TeamActivityCreateReportType instance
func NewTeamActivityCreateReportType(Description string) *TeamActivityCreateReportType {
	s := new(TeamActivityCreateReportType)
	s.Description = Description
	return s
}

// TeamBrandingPolicy : Policy for controlling team access to setting up
// branding feature
type TeamBrandingPolicy struct {
	dropbox.Tagged
}

// Valid tag values for TeamBrandingPolicy
const (
	TeamBrandingPolicyDisabled = "disabled"
	TeamBrandingPolicyEnabled  = "enabled"
	TeamBrandingPolicyOther    = "other"
)

// TeamBrandingPolicyChangedDetails : Changed team branding policy for team.
type TeamBrandingPolicyChangedDetails struct {
	// NewValue : New team branding policy.
	NewValue *TeamBrandingPolicy `json:"new_value"`
	// PreviousValue : Previous team branding policy.
	PreviousValue *TeamBrandingPolicy `json:"previous_value"`
}

// NewTeamBrandingPolicyChangedDetails returns a new TeamBrandingPolicyChangedDetails instance
func NewTeamBrandingPolicyChangedDetails(NewValue *TeamBrandingPolicy, PreviousValue *TeamBrandingPolicy) *TeamBrandingPolicyChangedDetails {
	s := new(TeamBrandingPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// TeamBrandingPolicyChangedType : has no documentation (yet)
type TeamBrandingPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewTeamBrandingPolicyChangedType returns a new TeamBrandingPolicyChangedType instance
func NewTeamBrandingPolicyChangedType(Description string) *TeamBrandingPolicyChangedType {
	s := new(TeamBrandingPolicyChangedType)
	s.Description = Description
	return s
}

// TeamDetails : More details about the team.
type TeamDetails struct {
	// Team : The name of the team.
	Team string `json:"team"`
}

// NewTeamDetails returns a new TeamDetails instance
func NewTeamDetails(Team string) *TeamDetails {
	s := new(TeamDetails)
	s.Team = Team
	return s
}

// TeamEvent : An audit log event.
type TeamEvent struct {
	// Timestamp : The Dropbox timestamp representing when the action was taken.
	Timestamp time.Time `json:"timestamp"`
	// EventCategory : The category that this type of action belongs to.
	EventCategory *EventCategory `json:"event_category"`
	// Actor : The entity who actually performed the action. Might be missing
	// due to historical data gap.
	Actor *ActorLogInfo `json:"actor,omitempty"`
	// Origin : The origin from which the actor performed the action including
	// information about host, ip address, location, session, etc. If the action
	// was performed programmatically via the API the origin represents the API
	// client.
	Origin *OriginLogInfo `json:"origin,omitempty"`
	// InvolveNonTeamMember : True if the action involved a non team member
	// either as the actor or as one of the affected users. Might be missing due
	// to historical data gap.
	InvolveNonTeamMember bool `json:"involve_non_team_member,omitempty"`
	// Context : The user or team on whose behalf the actor performed the
	// action. Might be missing due to historical data gap.
	Context *ContextLogInfo `json:"context,omitempty"`
	// Participants : Zero or more users and/or groups that are affected by the
	// action. Note that this list doesn't include any actors or users in
	// context.
	Participants []*ParticipantLogInfo `json:"participants,omitempty"`
	// Assets : Zero or more content assets involved in the action. Currently
	// these include Dropbox files and folders but in the future we might add
	// other asset types such as Paper documents, folders, projects, etc.
	Assets []*AssetLogInfo `json:"assets,omitempty"`
	// EventType : The particular type of action taken.
	EventType *EventType `json:"event_type"`
	// Details : The variable event schema applicable to this type of action,
	// instantiated with respect to this particular action.
	Details *EventDetails `json:"details"`
}

// NewTeamEvent returns a new TeamEvent instance
func NewTeamEvent(Timestamp time.Time, EventCategory *EventCategory, EventType *EventType, Details *EventDetails) *TeamEvent {
	s := new(TeamEvent)
	s.Timestamp = Timestamp
	s.EventCategory = EventCategory
	s.EventType = EventType
	s.Details = Details
	return s
}

// TeamExtensionsPolicy : Policy for controlling whether App Integrations are
// enabled for the team.
type TeamExtensionsPolicy struct {
	dropbox.Tagged
}

// Valid tag values for TeamExtensionsPolicy
const (
	TeamExtensionsPolicyDisabled = "disabled"
	TeamExtensionsPolicyEnabled  = "enabled"
	TeamExtensionsPolicyOther    = "other"
)

// TeamExtensionsPolicyChangedDetails : Changed App Integrations setting for
// team.
type TeamExtensionsPolicyChangedDetails struct {
	// NewValue : New Extensions policy.
	NewValue *TeamExtensionsPolicy `json:"new_value"`
	// PreviousValue : Previous Extensions policy.
	PreviousValue *TeamExtensionsPolicy `json:"previous_value"`
}

// NewTeamExtensionsPolicyChangedDetails returns a new TeamExtensionsPolicyChangedDetails instance
func NewTeamExtensionsPolicyChangedDetails(NewValue *TeamExtensionsPolicy, PreviousValue *TeamExtensionsPolicy) *TeamExtensionsPolicyChangedDetails {
	s := new(TeamExtensionsPolicyChangedDetails)
	s.NewValue = NewValue
	s.PreviousValue = PreviousValue
	return s
}

// TeamExtensionsPolicyChangedType : has no documentation (yet)
type TeamExtensionsPolicyChangedType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewTeamExtensionsPolicyChangedType returns a new TeamExtensionsPolicyChangedType instance
func NewTeamExtensionsPolicyChangedType(Description string) *TeamExtensionsPolicyChangedType {
	s := new(TeamExtensionsPolicyChangedType)
	s.Description = Description
	return s
}

// TeamFolderChangeStatusDetails : Changed archival status of team folder.
type TeamFolderChangeStatusDetails struct {
	// NewValue : New team folder status.
	NewValue *team.TeamFolderStatus `json:"new_value"`
	// PreviousValue : Previous team folder status. Might be missing due to
	// historical data gap.
	PreviousValue *team.TeamFolderStatus `json:"previous_value,omitempty"`
}

// NewTeamFolderChangeStatusDetails returns a new TeamFolderChangeStatusDetails instance
func NewTeamFolderChangeStatusDetails(NewValue *team.TeamFolderStatus) *TeamFolderChangeStatusDetails {
	s := new(TeamFolderChangeStatusDetails)
	s.NewValue = NewValue
	return s
}

// TeamFolderChangeStatusType : has no documentation (yet)
type TeamFolderChangeStatusType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewTeamFolderChangeStatusType returns a new TeamFolderChangeStatusType instance
func NewTeamFolderChangeStatusType(Description string) *TeamFolderChangeStatusType {
	s := new(TeamFolderChangeStatusType)
	s.Description = Description
	return s
}

// TeamFolderCreateDetails : Created team folder in active status.
type TeamFolderCreateDetails struct {
}

// NewTeamFolderCreateDetails returns a new TeamFolderCreateDetails instance
func NewTeamFolderCreateDetails() *TeamFolderCreateDetails {
	s := new(TeamFolderCreateDetails)
	return s
}

// TeamFolderCreateType : has no documentation (yet)
type TeamFolderCreateType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewTeamFolderCreateType returns a new TeamFolderCreateType instance
func NewTeamFolderCreateType(Description string) *TeamFolderCreateType {
	s := new(TeamFolderCreateType)
	s.Description = Description
	return s
}

// TeamFolderDowngradeDetails : Downgraded team folder to regular shared folder.
type TeamFolderDowngradeDetails struct {
	// TargetAssetIndex : Target asset position in the Assets list.
	TargetAssetIndex uint64 `json:"target_asset_index"`
}

// NewTeamFolderDowngradeDetails returns a new TeamFolderDowngradeDetails instance
func NewTeamFolderDowngradeDetails(TargetAssetIndex uint64) *TeamFolderDowngradeDetails {
	s := new(TeamFolderDowngradeDetails)
	s.TargetAssetIndex = TargetAssetIndex
	return s
}

// TeamFolderDowngradeType : has no documentation (yet)
type TeamFolderDowngradeType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewTeamFolderDowngradeType returns a new TeamFolderDowngradeType instance
func NewTeamFolderDowngradeType(Description string) *TeamFolderDowngradeType {
	s := new(TeamFolderDowngradeType)
	s.Description = Description
	return s
}

// TeamFolderPermanentlyDeleteDetails : Permanently deleted archived team
// folder.
type TeamFolderPermanentlyDeleteDetails struct {
}

// NewTeamFolderPermanentlyDeleteDetails returns a new TeamFolderPermanentlyDeleteDetails instance
func NewTeamFolderPermanentlyDeleteDetails() *TeamFolderPermanentlyDeleteDetails {
	s := new(TeamFolderPermanentlyDeleteDetails)
	return s
}

// TeamFolderPermanentlyDeleteType : has no documentation (yet)
type TeamFolderPermanentlyDeleteType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewTeamFolderPermanentlyDeleteType returns a new TeamFolderPermanentlyDeleteType instance
func NewTeamFolderPermanentlyDeleteType(Description string) *TeamFolderPermanentlyDeleteType {
	s := new(TeamFolderPermanentlyDeleteType)
	s.Description = Description
	return s
}

// TeamFolderRenameDetails : Renamed active/archived team folder.
type TeamFolderRenameDetails struct {
	// PreviousFolderName : Previous folder name.
	PreviousFolderName string `json:"previous_folder_name"`
	// NewFolderName : New folder name.
	NewFolderName string `json:"new_folder_name"`
}

// NewTeamFolderRenameDetails returns a new TeamFolderRenameDetails instance
func NewTeamFolderRenameDetails(PreviousFolderName string, NewFolderName string) *TeamFolderRenameDetails {
	s := new(TeamFolderRenameDetails)
	s.PreviousFolderName = PreviousFolderName
	s.NewFolderName = NewFolderName
	return s
}

// TeamFolderRenameType : has no documentation (yet)
type TeamFolderRenameType struct {
	// Description : has no documentation (yet)
	Description string `json:"description"`
}

// NewTeamFolderRenameType returns a new TeamFolderRenameType instance
func NewTeamFolderRenameType(Description string) *TeamFolderRenameType {
	s := new(TeamFolderRenameType)
	s.Description = Description
	return s
}

// TeamInviteDetails : Details about team invites
type TeamInviteDetails struct {
	// InviteMethod : How the user was invited to the team.
	InviteMethod *InviteMethod `json:"invite_method"`
	// AdditionalLicensePurchase : True if the invitation incurred an additional
	// license purchase.
	AdditionalLicensePurchase bool `json:"additional_license_purchase,omitempty"`
}

// NewTeamInviteDetails returns a new TeamInviteDetails instance
func NewTeamInviteDetails(InviteMethod *InviteMethod) *TeamInviteDetails {
	s := new(TeamInviteDetails)
	s.InviteMethod = InviteMethod
	return s
