﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace APIGateway {
namespace Model {

/**
 * <p>Request to list information about a resource.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/GetResourceRequest">AWS
 * API Reference</a></p>
 */
class GetResourceRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API GetResourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetResource"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  AWS_APIGATEWAY_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The string identifier of the associated RestApi.</p>
   */
  inline const Aws::String& GetRestApiId() const { return m_restApiId; }
  inline bool RestApiIdHasBeenSet() const { return m_restApiIdHasBeenSet; }
  template <typename RestApiIdT = Aws::String>
  void SetRestApiId(RestApiIdT&& value) {
    m_restApiIdHasBeenSet = true;
    m_restApiId = std::forward<RestApiIdT>(value);
  }
  template <typename RestApiIdT = Aws::String>
  GetResourceRequest& WithRestApiId(RestApiIdT&& value) {
    SetRestApiId(std::forward<RestApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier for the Resource resource.</p>
   */
  inline const Aws::String& GetResourceId() const { return m_resourceId; }
  inline bool ResourceIdHasBeenSet() const { return m_resourceIdHasBeenSet; }
  template <typename ResourceIdT = Aws::String>
  void SetResourceId(ResourceIdT&& value) {
    m_resourceIdHasBeenSet = true;
    m_resourceId = std::forward<ResourceIdT>(value);
  }
  template <typename ResourceIdT = Aws::String>
  GetResourceRequest& WithResourceId(ResourceIdT&& value) {
    SetResourceId(std::forward<ResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A query parameter to retrieve the specified resources embedded in the
   * returned Resource representation in the response. This <code>embed</code>
   * parameter value is a list of comma-separated strings. Currently, the request
   * supports only retrieval of the embedded Method resources this way. The query
   * parameter value must be a single-valued list and contain the
   * <code>"methods"</code> string. For example, <code>GET
   * /restapis/{restapi_id}/resources/{resource_id}?embed=methods</code>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetEmbed() const { return m_embed; }
  inline bool EmbedHasBeenSet() const { return m_embedHasBeenSet; }
  template <typename EmbedT = Aws::Vector<Aws::String>>
  void SetEmbed(EmbedT&& value) {
    m_embedHasBeenSet = true;
    m_embed = std::forward<EmbedT>(value);
  }
  template <typename EmbedT = Aws::Vector<Aws::String>>
  GetResourceRequest& WithEmbed(EmbedT&& value) {
    SetEmbed(std::forward<EmbedT>(value));
    return *this;
  }
  template <typename EmbedT = Aws::String>
  GetResourceRequest& AddEmbed(EmbedT&& value) {
    m_embedHasBeenSet = true;
    m_embed.emplace_back(std::forward<EmbedT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_restApiId;
  bool m_restApiIdHasBeenSet = false;

  Aws::String m_resourceId;
  bool m_resourceIdHasBeenSet = false;

  Aws::Vector<Aws::String> m_embed;
  bool m_embedHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
