/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.content.pm;

import android.annotation.NonNull;
import android.annotation.Nullable;
import android.os.Parcel;
import android.os.Parcelable;

import com.android.internal.util.DataClass;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;

/**
 * A typed checksum of an APK.
 *
 * @see PackageManager#requestChecksums
 */
@DataClass(genHiddenConstructor = true)
@DataClass.Suppress({"getChecksum"})
public final class ApkChecksum implements Parcelable {
    /**
     * Checksum for which split. Null indicates base.apk.
     */
    private final @Nullable String mSplitName;
    /**
     * Checksum.
     */
    private final @NonNull Checksum mChecksum;
    /**
     * For Installer-provided checksums, package name of the Installer.
     */
    private final @Nullable String mInstallerPackageName;
    /**
     * For Installer-provided checksums, certificate of the Installer.
     */
    private final @Nullable byte[] mInstallerCertificate;

    /**
     * Constructor, internal use only.
     *
     * @hide
     */
    public ApkChecksum(@Nullable String splitName, @Checksum.Type int type,
            @NonNull byte[] value) {
        this(splitName, new Checksum(type, value), (String) null, (byte[]) null);
    }

    /**
     * Constructor, internal use only.
     *
     * @hide
     */
    public ApkChecksum(@Nullable String splitName, @Checksum.Type int type, @NonNull byte[] value,
            @Nullable String sourcePackageName, @Nullable Certificate sourceCertificate)
            throws CertificateEncodingException {
        this(splitName, new Checksum(type, value), sourcePackageName,
                (sourceCertificate != null) ? sourceCertificate.getEncoded() : null);
    }


    /**
     * Checksum type.
     */
    public @Checksum.Type int getType() {
        return mChecksum.getType();
    }

    /**
     * Checksum value.
     */
    public @NonNull byte[] getValue() {
        return mChecksum.getValue();
    }

    /**
     * Returns raw bytes representing encoded certificate of the Installer.
     * @hide
     */
    public @Nullable byte[] getInstallerCertificateBytes() {
        return mInstallerCertificate;
    }

    /**
     * For Installer-provided checksums, certificate of the Installer.
     * @throws CertificateException in case when certificate can't be re-created from serialized
     * data.
     */
    public @Nullable Certificate getInstallerCertificate() throws CertificateException {
        if (mInstallerCertificate == null) {
            return null;
        }
        final CertificateFactory cf = CertificateFactory.getInstance("X.509");
        final InputStream is = new ByteArrayInputStream(mInstallerCertificate);
        final X509Certificate cert = (X509Certificate) cf.generateCertificate(is);
        return cert;
    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/content/pm/ApkChecksum.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /**
     * Creates a new ApkChecksum.
     *
     * @param splitName
     *   Checksum for which split. Null indicates base.apk.
     * @param checksum
     *   Checksum.
     * @param installerPackageName
     *   For Installer-provided checksums, package name of the Installer.
     * @param installerCertificate
     *   For Installer-provided checksums, certificate of the Installer.
     * @hide
     */
    @DataClass.Generated.Member
    public ApkChecksum(
            @Nullable String splitName,
            @NonNull Checksum checksum,
            @Nullable String installerPackageName,
            @Nullable byte[] installerCertificate) {
        this.mSplitName = splitName;
        this.mChecksum = checksum;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mChecksum);
        this.mInstallerPackageName = installerPackageName;
        this.mInstallerCertificate = installerCertificate;

        // onConstructed(); // You can define this method to get a callback
    }

    /**
     * Checksum for which split. Null indicates base.apk.
     */
    @DataClass.Generated.Member
    public @Nullable String getSplitName() {
        return mSplitName;
    }

    /**
     * For Installer-provided checksums, package name of the Installer.
     */
    @DataClass.Generated.Member
    public @Nullable String getInstallerPackageName() {
        return mInstallerPackageName;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        byte flg = 0;
        if (mSplitName != null) flg |= 0x1;
        if (mInstallerPackageName != null) flg |= 0x4;
        if (mInstallerCertificate != null) flg |= 0x8;
        dest.writeByte(flg);
        if (mSplitName != null) dest.writeString(mSplitName);
        dest.writeTypedObject(mChecksum, flags);
        if (mInstallerPackageName != null) dest.writeString(mInstallerPackageName);
        if (mInstallerCertificate != null) dest.writeByteArray(mInstallerCertificate);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ ApkChecksum(@NonNull Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        byte flg = in.readByte();
        String splitName = (flg & 0x1) == 0 ? null : in.readString();
        Checksum checksum = (Checksum) in.readTypedObject(Checksum.CREATOR);
        String installerPackageName = (flg & 0x4) == 0 ? null : in.readString();
        byte[] installerCertificate = (flg & 0x8) == 0 ? null : in.createByteArray();

        this.mSplitName = splitName;
        this.mChecksum = checksum;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mChecksum);
        this.mInstallerPackageName = installerPackageName;
        this.mInstallerCertificate = installerCertificate;

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<ApkChecksum> CREATOR
            = new Parcelable.Creator<ApkChecksum>() {
        @Override
        public ApkChecksum[] newArray(int size) {
            return new ApkChecksum[size];
        }

        @Override
        public ApkChecksum createFromParcel(@NonNull Parcel in) {
            return new ApkChecksum(in);
        }
    };

    @DataClass.Generated(
            time = 1619810171079L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/content/pm/ApkChecksum.java",
            inputSignatures = "private final @android.annotation.Nullable java.lang.String mSplitName\nprivate final @android.annotation.NonNull android.content.pm.Checksum mChecksum\nprivate final @android.annotation.Nullable java.lang.String mInstallerPackageName\nprivate final @android.annotation.Nullable byte[] mInstallerCertificate\npublic @android.content.pm.Checksum.Type int getType()\npublic @android.annotation.NonNull byte[] getValue()\npublic @android.annotation.Nullable byte[] getInstallerCertificateBytes()\npublic @android.annotation.Nullable java.security.cert.Certificate getInstallerCertificate()\nclass ApkChecksum extends java.lang.Object implements [android.os.Parcelable]\n@com.android.internal.util.DataClass(genHiddenConstructor=true)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
