#!/usr/bin/env python
# ruff: noqa: N802
"""Convert async code in the project to sync code.

Note: the version of Python used to run this script affects the output.

Hint: in order to explore the AST of a module you can run:

    python -m ast path/to/module.py

"""

from __future__ import annotations

import logging
import subprocess as sp
import sys
from argparse import ArgumentParser, Namespace, RawDescriptionHelpFormatter
from collections.abc import Sequence
from copy import deepcopy
from dataclasses import dataclass, field
from pathlib import Path
from typing import TYPE_CHECKING, Any, ClassVar, Literal

import ast_comments as ast  # type: ignore
from typing_extensions import override

if TYPE_CHECKING:
    from collections.abc import Sequence

    from ast_comments import AST

# The version of Python officially used for the conversion.
# Output may differ in other versions.
# Should be consistent with the Python version used in lint.yml
PYVER = "3.10"

PROJECT_ROOT = Path(__file__).parent.parent
ASYNC_PROJECT_DIR = PROJECT_ROOT / "key-value" / "key-value-aio"
ASYNC_PROJECT_MODULE_DIR = ASYNC_PROJECT_DIR / "src" / "key_value" / "aio"
ASYNC_PROJECT_TESTS_DIR = ASYNC_PROJECT_DIR / "tests"

SYNC_PROJECT_DIR = PROJECT_ROOT / "key-value" / "key-value-sync"
SYNC_PROJECT_MODULE_DIR = SYNC_PROJECT_DIR / "src" / "key_value" / "sync"
SYNC_PROJECT_CODE_GEN_DIR = SYNC_PROJECT_MODULE_DIR / "code_gen"
SYNC_PROJECT_TESTS_DIR = SYNC_PROJECT_DIR / "tests"
SYNC_PROJECT_TESTS_CODE_GEN_DIR = SYNC_PROJECT_TESTS_DIR / "code_gen"

PATHS_TO_LINT = [SYNC_PROJECT_MODULE_DIR, SYNC_PROJECT_TESTS_DIR]
EXCLUDE_FILES = [
    "key-value/key-value-aio/src/key_value/aio/__init__.py",
    "key-value/key-value-aio/tests/stores/wrappers/test_timeout.py",
]
EXCLUDE_DIRECTORIES = [
    "key-value/key-value-aio/src/key_value/aio/stores/dynamodb",
    "key-value/key-value-aio/tests/stores/dynamodb",
    "key-value/key-value-aio/src/key_value/aio/stores/memcached",
    "key-value/key-value-aio/tests/stores/memcached",
    "key-value/key-value-aio/src/key_value/aio/stores/filetree",
    "key-value/key-value-aio/tests/stores/filetree",
    "key-value/key-value-aio/src/key_value/aio/wrappers/timeout",
    "key-value/key-value-aio/tests/wrappers/timeout",
]

SCRIPT_NAME = Path(sys.argv[0]).name

logger = logging.getLogger()


def main() -> int:
    options: ConversionOptions = parse_cmdline()

    logging.basicConfig(level=logging.INFO, format="%(levelname)s %(message)s")

    if not check_python_version(version=options.python_version):
        msg = f"Expecting output generated by Python {options.python_version} but are running {get_python_version()}."
        raise RuntimeError(msg)

    if options.wipe_code_gen_dir:
        logger.info("Wiping code gen directory: %s", SYNC_PROJECT_CODE_GEN_DIR)
        # if SYNC_PROJECT_CODE_GEN_DIR.exists():
        #     shutil.rmtree(SYNC_PROJECT_CODE_GEN_DIR)

        for file in SYNC_PROJECT_MODULE_DIR.rglob("*.py"):
            if file_has_header(filepath=file):
                file.unlink()

        for file in SYNC_PROJECT_TESTS_DIR.rglob("*.py"):
            if file_has_header(filepath=file):
                file.unlink()

    for file_to_convert in options.files_to_convert:
        for output_path in file_to_convert.output_path:
            logger.info("Starting to convert: %s", file_to_convert.input_path)
            convert(source_path=file_to_convert.input_path, output_path=output_path)
            logger.info("Finished converting: %s", output_path)

    for path in options.paths_to_lint:
        logger.info("Starting to lint: %s", path)
        lint(path=path)
        logger.info("Finished linting: %s", path)

    lint(path=SYNC_PROJECT_DIR)
    lint(path=SYNC_PROJECT_TESTS_DIR)

    return 0


def lint(path: Path) -> None:
    """Call the linting tool on the given path."""
    _ = sp.check_call(["uv", "run", "ruff", "format", "-q", str(path)])  # noqa: S603, S607
    _ = sp.check_call(["uv", "run", "ruff", "check", "--fix", str(path)])  # noqa: S603, S607


def convert(source_path: Path, output_path: Path) -> None:
    """Convert the given source path to the given output path."""
    with source_path.open() as f:
        source = f.read()

    tree: AST = ast.parse(source=source, filename=str(object=source_path))  # pyright: ignore[reportUnknownMemberType]
    tree = async_to_sync(tree=tree, filepath=source_path)
    output: str = tree_to_str(tree=tree, filepath=source_path)

    output_path.parent.mkdir(parents=True, exist_ok=True)

    with output_path.open("w") as f:
        print(output, file=f)


def async_to_sync(tree: ast.AST, filepath: Path | None = None) -> ast.AST:  # noqa: ARG001
    tree = BlanksInserter().visit(tree)
    tree = RenameAsyncToSync().visit(tree)
    tree = AsyncToSync().visit(tree)
    return tree  # noqa: RET504


def file_has_header(filepath: Path) -> bool:
    return filepath.read_text().startswith("# WARNING: this file is auto-generated by")


def tree_to_str(tree: ast.AST, filepath: Path) -> str:
    rv = f"""\
# WARNING: this file is auto-generated by '{SCRIPT_NAME}'
# from the original file '{filepath.name}'
# DO NOT CHANGE! Change the original file instead.
"""
    rv += unparse(tree)
    return rv


class AsyncToSync(ast.NodeTransformer):  # type: ignore
    def visit_AsyncFunctionDef(self, node: ast.AsyncFunctionDef) -> ast.AST:
        new_node: ast.FunctionDef = ast.FunctionDef(**node.__dict__)
        ast.copy_location(new_node, old_node=node)
        self.visit(node=new_node)
        return new_node

    def visit_AsyncFor(self, node: ast.AsyncFor) -> ast.AST:
        new_node: ast.For = ast.For(**node.__dict__)
        ast.copy_location(new_node, old_node=node)
        self.visit(node=new_node)
        return new_node

    def visit_AsyncWith(self, node: ast.AsyncWith) -> ast.AST:
        new_node: ast.With = ast.With(**node.__dict__)
        ast.copy_location(new_node, old_node=node)
        self.visit(node=new_node)
        return new_node

    def visit_Await(self, node: ast.Await) -> ast.AST:
        new_node: ast.expr = node.value
        self.visit(new_node)
        return new_node

    def visit_If(self, node: ast.If) -> ast.AST:
        # Drop `if is_async()` branch.
        #
        # Assume that the test guards an async object becoming sync and remove
        # the async side, because it will likely contain `await` constructs
        # illegal into a sync function.
        value: bool
        comment: str
        match node:
            # manage `is_async()`
            case ast.If(test=ast.Call(func=ast.Name(id="is_async"))):
                for child in node.orelse:
                    self.visit(child)
                return node.orelse

            # Manage `if True|False:  # ASYNC`
            # drop the unneeded branch
            case ast.If(  # pyright: ignore[reportUnnecessaryComparison]
                test=ast.Constant(value=bool(value)),
                body=[ast.Comment(value=comment), *_],  # pyright: ignore[reportUnknownVariableType]
            ) if comment.startswith("# ASYNC"):
                stmts: list[ast.AST]
                # body[0] is the ASYNC comment, drop it
                stmts = node.orelse if value else node.body[1:]
                for child in stmts:
                    self.visit(child)
                return stmts
            case _:
                pass

        self.generic_visit(node)
        return node


class RenameAsyncToSync(ast.NodeTransformer):  # type: ignore
    module_map: ClassVar[dict[str, str]] = {
        "key_value.aio": "key_value.sync.code_gen",
        "pymongo.asynchronous": "pymongo",
        "tests.stores.conftest": "tests.code_gen.stores.conftest",
        "tests.conftest": "tests.code_gen.conftest",
        "tests.cases": "tests.code_gen.cases",
        "tests.stores.base": "tests.code_gen.stores.base",
    }
    names_map: ClassVar[dict[str, str]] = {
        "__aenter__": "__enter__",
        "__aexit__": "__exit__",
        "__aiter__": "__iter__",
        "asyncio.locks": "threading",
        "AsyncElasticsearch": "Elasticsearch",
        "AsyncDatabase": "Database",
        "AsyncCollection": "Collection",
        "AsyncMongoClient": "MongoClient",
        "AsyncExitStack": "ExitStack",
        "redis.asyncio": "redis",
        "redis.asyncio.client": "redis.client",
        "glide_shared.config": "glide_sync.config",
        "glide.glide_client": "glide_sync.glide_client",
        "asynccontextmanager": "contextmanager",
        "AsyncKeyValueProtocol": "KeyValueProtocol",
        "AsyncCullProtocol": "CullProtocol",
        "AsyncDestroyCollectionProtocol": "DestroyCollectionProtocol",
        "AsyncDestroyStoreProtocol": "DestroyStoreProtocol",
        "AsyncEnumerateCollectionsProtocol": "EnumerateCollectionsProtocol",
        "AsyncEnumerateKeysProtocol": "EnumerateKeysProtocol",
        "AsyncKeyValue": "KeyValue",
        "AsyncGenerator": "Generator",
        "aclose": "close",
        "enter_async_context": "enter_context",
        "asyncio.sleep": "time.sleep",
        "async_running_in_event_loop": "running_in_event_loop",
        "asleep": "sleep",
        "async_wait_for_true": "wait_for_true",
        "async_retry_operation": "retry_operation",
        "async_gather": "gather",
        "push_async_callback": "callback",
    }
    _skip_imports: ClassVar[dict[str, set[str]]] = {
        "acompat": {"alist", "anext"},
    }

    def visit_Module(self, node: ast.Module) -> ast.AST:
        self._fix_docstring(node.body)
        self.generic_visit(node)
        return node

    def visit_AsyncFunctionDef(self, node: ast.AsyncFunctionDef) -> ast.AST:
        self._fix_docstring(node.body)
        node.name = self.names_map.get(node.name, node.name)
        for arg in node.args.args:
            arg.arg = self.names_map.get(arg.arg, arg.arg)
        for arg in node.args.args:
            attr: str
            match arg.annotation:
                case ast.arg(annotation=ast.Attribute(attr=attr)):
                    arg.annotation.attr = self.names_map.get(attr, attr)
                case ast.arg(annotation=ast.Subscript(value=ast.Attribute(attr=attr))):
                    arg.annotation.value.attr = self.names_map.get(attr, attr)

        self.generic_visit(node)
        return node

    @override
    def visit_Call(self, node: ast.Call) -> ast.AST:
        # match node:
        #     case ast.Call(func=ast.Name(id="cast")):
        #         node.args[0] = self._convert_if_literal_string(node.args[0])

        _ = self.generic_visit(node)
        return node

    @override
    def visit_FunctionDef(self, node: ast.FunctionDef) -> ast.AST:
        self._fix_docstring(node.body)
        if node.decorator_list:
            self._fix_decorator(node.decorator_list)
        _ = self.generic_visit(node)
        return node

    def _fix_docstring(self, body: Sequence[ast.AST]) -> None:
        doc: str
        match body and body[0]:
            case ast.Expr(value=ast.Constant(value=str(doc))):
                doc = doc.replace("Async", "")
                doc = doc.replace("(async", "(sync")
                body[0].value.value = doc
            case _:
                pass

    def _fix_decorator(self, decorator_list: Sequence[ast.AST]) -> None:
        for dec in decorator_list:
            match dec:
                case ast.Call(
                    func=ast.Attribute(value=ast.Name(id="pytest"), attr="fixture"),
                    keywords=[ast.keyword(arg="params", value=ast.List())],
                ):
                    elts = dec.keywords[0].value.elts
                    for i, elt in enumerate(elts):
                        elts[i] = self._convert_if_literal_string(elt)
                case _:
                    pass

    def _convert_if_literal_string(self, node: ast.AST) -> ast.AST:
        value: str
        match node:
            case ast.Constant(value=str(value)):
                node.value = self._visit_type_string(value)
            case _:
                pass

        return node

    def _visit_type_string(self, source: str) -> str:
        # Convert the string to tree, visit, and convert it back to string
        tree = ast.parse(source, type_comments=False)  # pyright: ignore[reportUnknownMemberType]
        tree = async_to_sync(tree)
        return unparse(tree)

    def visit_ClassDef(self, node: ast.ClassDef) -> ast.AST:
        self._fix_docstring(node.body)
        node.name = self.names_map.get(node.name, node.name)
        node = self._fix_base_params(node)
        self.generic_visit(node)
        return node

    def _fix_base_params(self, node: ast.ClassDef) -> ast.AST:
        # Handle :
        #   class AsyncCursor(BaseCursor["AsyncConnection[Any]", Row]):
        # the base cannot be a token, even with __future__ annotation.
        elts: list[ast.AST]
        for base in node.bases:
            match base:
                case ast.Name():  # handle myClass(AsyncBaseClass) -> myClass(BaseClass)
                    base.id = self.names_map.get(base.id, base.id)
                case ast.Subscript(slice=ast.Tuple(elts=elts)):
                    for i, elt in enumerate(elts):
                        elts[i] = self._convert_if_literal_string(elt)
                case ast.Subscript(slice=ast.Constant()):
                    base.slice = self._convert_if_literal_string(base.slice)

        return node

    def visit_ImportFrom(self, node: ast.ImportFrom) -> ast.AST | None:
        if node.module:
            # Remove import of async utils eclypsing builtins
            if skips := self._skip_imports.get(node.module):
                node.names = [n for n in node.names if n.name not in skips]
                if not node.names:
                    return None

            # Handle any custom module mappings: key_value.aio.utils -> key_value.sync.utils
            node.module = self.names_map.get(node.module) or node.module
            for find_module, replace_module in self.module_map.items():
                if node.module.startswith(find_module):
                    node.module = node.module.replace(find_module, replace_module)
                    break

        # Handle any import names: from ... import AsyncKeyValue -> from ... import KeyValue
        for n in node.names:
            n.name = self.names_map.get(n.name, n.name)

            # Handle any import asliases:
            # from: ... import ... as AsyncThing
            # to:   ... import ... as Thing
            if n.asname:
                n.asname = self.names_map.get(n.asname, n.asname)

        return node

    def visit_Name(self, node: ast.Name) -> ast.AST:
        if node.id in self.names_map:
            node.id = self.names_map[node.id]
        return node

    def visit_Attribute(self, node: ast.Attribute) -> ast.AST:
        if node.attr in self.names_map:
            node.attr = self.names_map[node.attr]
        self.generic_visit(node)
        return node

    def visit_Subscript(self, node: ast.Subscript) -> ast.AST:
        # Manage AsyncGenerator[X, Y] -> Generator[X, None, Y]
        self._manage_async_generator(node)
        # # Won't result in a recursion because we change the args number
        # self.visit(node)
        # return node

        self.generic_visit(node)
        return node

    def _manage_async_generator(self, node: ast.Subscript) -> ast.AST | None:
        match node:
            case ast.Subscript(value=ast.Name(id="AsyncGenerator"), slice=ast.Tuple(elts=[_, _])):
                node.slice.elts.insert(1, deepcopy(node.slice.elts[1]))
                self.generic_visit(node)
                return node
            case _:
                pass
        return None


class BlanksInserter(ast.NodeTransformer):  # type: ignore
    """
    Restore the missing spaces in the source (or something similar)
    """

    def generic_visit(self, node: ast.AST) -> ast.AST:
        if isinstance(getattr(node, "body", None), list):
            node.body = self._inject_blanks(node.body)
        super().generic_visit(node)
        return node

    def _inject_blanks(self, body: list[ast.Node]) -> list[ast.AST]:
        if not body:
            return body

        new_body = []
        before = body[0]
        new_body.append(before)
        for i in range(1, len(body)):
            after = body[i]
            nblanks = after.lineno - before.end_lineno - 1
            if nblanks > 0:
                # Inserting one blank is enough.
                blank = ast.Comment(
                    value="",
                    inline=False,
                    lineno=before.end_lineno + 1,
                    end_lineno=before.end_lineno + 1,
                    col_offset=0,
                    end_col_offset=0,
                )
                new_body.append(blank)
            new_body.append(after)
            before = after

        return new_body


def unparse(tree: ast.AST) -> str:
    rv: str = Unparser().visit(tree)
    return _fix_comment_on_decorators(source=rv)


def _fix_comment_on_decorators(source: str) -> str:
    """
    Re-associate comments to decorators.

    In a case like:

        1  @deco  # comment
        2  def func(x):
        3     pass

    it seems that Function lineno is 2 instead of 1 (Python 3.10). Because
    the Comment lineno is 1, it ends up printed above the function, instead
    of inline. This is a problem for '# type: ignore' comments.

    Maybe the problem could be fixed in the tree, but this solution is a
    simpler way to start.
    """
    lines = source.splitlines()

    comment_at = None
    for i, line in enumerate(lines):
        if line.lstrip().startswith("#"):
            comment_at = i
        elif not line.strip():
            pass
        elif line.lstrip().startswith("@classmethod"):
            if comment_at is not None:
                lines[i] = lines[i] + "  " + lines[comment_at].lstrip()
                lines[comment_at] = ""
        else:
            comment_at = None

    return "\n".join(lines)


class Unparser(ast._Unparser):  # type: ignore
    """
    Try to emit long strings as multiline.

    The normal class only tries to emit docstrings as multiline,
    but the resulting source doesn't pass flake8.
    """

    # Beware: private method. Tested with in Python 3.10, 3.11.
    def _write_constant(self, value: Any) -> None:
        if isinstance(value, str) and len(value) > 50:  # noqa: PLR2004
            self._write_str_avoiding_backslashes(value)
        else:
            super()._write_constant(value)


def swap_key_value_aio_to_key_value_sync(path: Path) -> Path:
    """Swap the key-value-aio to key-value-sync."""
    return Path(*[part.replace("key-value-aio", "key-value-sync") for part in path.parts])


def swap_aio_to_sync(path: Path) -> Path:
    """Swap the aio to sync."""
    return Path(*[part.replace("aio", "sync") for part in path.parts])


def swap_sync_to_sync_code_gen(path: Path) -> Path:
    """Swap the code_gen to sync_code_gen."""
    new_parts: list[str] = []
    for part in path.parts:
        if part == "sync":
            new_parts.extend(["sync", "code_gen"])
        else:
            new_parts.append(part)
    return Path(*new_parts)


def get_sync_path_for_file(path: Path) -> Path:
    """Use the current file structure to redirect to the sync tree.

    i.e.
    this:     key-value/key-value-aio/src/key_value/aio/stores/elasticsearch/store.py
    to this: key-value/key-value-sync/src/key_value/sync/stores/elasticsearch/store.py
    """
    path = swap_key_value_aio_to_key_value_sync(path)
    return swap_aio_to_sync(path)


def get_sync_codegen_path_for_file(path: Path) -> Path:
    """Use the current file structure to redirect to the sync tree.

    i.e.
    this:     key-value/key-value-aio/src/key_value/aio/stores/elasticsearch/store.py
    to this: key-value/key-value-sync/src/key_value/sync/code_gen/stores/elasticsearch/store.py

    if the file is an __init__.py, then we don't need/want to put it under code_gen
    """
    path = swap_key_value_aio_to_key_value_sync(path)
    path = swap_aio_to_sync(path)
    return swap_sync_to_sync_code_gen(path)


@dataclass
class FileToConvert:
    input_path: Path
    output_path: list[Path]

    def __post_init__(self) -> None:
        if not self.input_path.exists() and not self.input_path.is_file():
            msg = f"Input path {self.input_path} does not exist or is not a file"
            raise ValueError(msg)


@dataclass
class ConversionOptions:
    files_to_convert: list[FileToConvert] = field(default_factory=list)
    paths_to_lint: list[Path] = field(default_factory=list)
    log_level: Literal["INFO", "DEBUG"] = field(default="INFO")
    python_version: str = field(default=PYVER)
    wipe_code_gen_dir: bool = field(default=False)


def get_python_version() -> str:
    """Get the Python version."""
    return ".".join(map(str, sys.version_info[:2]))


def check_python_version(version: str) -> bool:
    """Get the Python version."""
    return get_python_version() == version


def get_inputs_from_opt(opt: Namespace) -> list[Path]:
    """Type-safe way to get the inputs from the options."""
    return opt.inputs  # pyright: ignore[reportAny]


def path_is_excluded(path: Path, exclude_files: list[Path], exclude_directories: list[Path]) -> bool:
    """Check if the path is excluded."""
    resolved_path = path.resolve()
    for exclude_file in exclude_files:
        if resolved_path == exclude_file.resolve():
            return True

    for exclude_directory in exclude_directories:  # noqa: SIM110
        if resolved_path.is_relative_to(exclude_directory.resolve()):
            return True

    return False


def parse_cmdline() -> ConversionOptions:
    parser = ArgumentParser(description=__doc__, formatter_class=RawDescriptionHelpFormatter)
    _ = parser.add_argument(
        "inputs",
        metavar="FILE",
        nargs="*",
        type=Path,
        help="the files to process (process all files if not specified)",
    )

    opt: Namespace = parser.parse_args()

    files_to_convert: list[FileToConvert] = []

    if opt.inputs:
        files_to_convert = [FileToConvert(input_path=Path(input_path).resolve(), output_path=[]) for input_path in opt.inputs]
        return ConversionOptions(files_to_convert=files_to_convert, paths_to_lint=PATHS_TO_LINT, wipe_code_gen_dir=False)

    excluded_paths: list[Path] = [Path(excluded_file).resolve() for excluded_file in EXCLUDE_FILES]
    excluded_directories: list[Path] = [Path(excluded_directory).resolve() for excluded_directory in EXCLUDE_DIRECTORIES]

    for file_path in ASYNC_PROJECT_MODULE_DIR.rglob(pattern="*.py"):
        if path_is_excluded(path=file_path, exclude_files=excluded_paths, exclude_directories=excluded_directories):
            continue

        output_paths: list[Path] = []
        # Convert
        # From: key-value-aio/src/key_value/aio/**.py
        # To:   key-value-sync/src/key_value/sync/code_gen/**.py
        relative_path = file_path.relative_to(ASYNC_PROJECT_MODULE_DIR)
        output_paths.append(SYNC_PROJECT_CODE_GEN_DIR / relative_path)
        if file_path.name == "__init__.py":
            # We also want to copy the __init__.py files to match the structure of the aio package
            # From: key-value-aio/src/key_value/aio/**/__init__.py
            # To:   key-value-sync/src/key_value/sync/**/__init__.py
            output_paths.append(SYNC_PROJECT_MODULE_DIR / relative_path)

        files_to_convert.append(FileToConvert(input_path=file_path, output_path=output_paths))

    for file_path in ASYNC_PROJECT_TESTS_DIR.rglob(pattern="*.py"):
        if path_is_excluded(path=file_path, exclude_files=excluded_paths, exclude_directories=excluded_directories):
            continue

        # Convert
        # From: key-value-aio/tests/**.py
        # To:   key-value-sync/tests/code_gen/**.py
        relative_path = file_path.relative_to(ASYNC_PROJECT_TESTS_DIR)
        output_path: Path = SYNC_PROJECT_TESTS_CODE_GEN_DIR / relative_path
        files_to_convert.append(FileToConvert(input_path=file_path, output_path=[output_path]))

    return ConversionOptions(files_to_convert=files_to_convert, paths_to_lint=PATHS_TO_LINT, wipe_code_gen_dir=True)


if __name__ == "__main__":
    sys.exit(main())
