/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Spicebird code.
 *
 * The Initial Developer of the Original Code is
 * Synovel Software Technologies
 * Portions created by the Initial Developer are Copyright (C) 2008
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Prasad Sunkari <prasad@synovel.com> (Original Author)
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "csTPTextChannel.h"
#include "csITelepathyCallbacks.h"
#include "nsIMutableArray.h"
#include "nsComponentManagerUtils.h"
#include "nsArrayUtils.h"

NS_IMPL_ISUPPORTS1(csTPTextChannelPendingMessage, csITPTextChannelPendingMessage)
NS_IMPL_ISUPPORTS3(csTPTextChannel, csITPTextChannel, csITPChannel, csITPProxy)

#include "csTelepathyMacros.h"

csTPTextChannel::csTPTextChannel()
{
}

csTPTextChannel::~csTPTextChannel()
{
}

static void
GotAcknowledgePendingMessageResponse(TpChannel *proxy, const GError *error,
                                     gpointer user_data, GObject *weak_object)
{
  csITPEmptyCB *callback = (csITPEmptyCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  callback->DoAction();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPTextChannel::AcknowledgePendingMessages(PRUint32 count, 
                                                          PRUint32 *aMsgIDs, 
                                                          csITPEmptyCB *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  GArray *array = g_array_new(false, false, sizeof(guint));
  for (PRUint32 i=0; i<count; i++) 
    g_array_append_val(array, aMsgIDs[i]);

  NS_IF_ADDREF(cb);
  tp_cli_channel_type_text_call_acknowledge_pending_messages (m_Channel, -1,
                                array, GotAcknowledgePendingMessageResponse,
                                cb, NULL, NULL);
  return NS_OK;
}

static void
GotMessageTypes(TpChannel *proxy, const GArray *handles, 
                const GError *error, gpointer user_data, GObject *weak_object)
{
  csITPIntegerListCB *callback = (csITPIntegerListCB*)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  unsigned int len = handles->len;
  for (unsigned int i=0; i<len; i++)
    callback->OnAddItem(g_array_index(handles, PRUint32, i));

  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPTextChannel::GetMessageTypes(csITPIntegerListCB *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_channel_type_text_call_get_message_types(m_Channel, -1,
                                                  GotMessageTypes,
                                                  cb, NULL, NULL);
  return NS_OK;
}

static void
GotPendingMessages(TpChannel *proxy, const GPtrArray *messages,
                   const GError *error, gpointer user_data,
                   GObject *weak_object)
{
  csITPInterfaceListCB *callback = (csITPInterfaceListCB*)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  for (guint i = 0; i < messages->len; i++) {
    GValueArray *item = (GValueArray *)g_ptr_array_index(messages, i);

    PRUint32 id = g_value_get_uint(&(item->values[0]));
    PRUint32 timestamp = g_value_get_uint(&(item->values[1]));
    PRUint32 sender = g_value_get_uint(&(item->values[2]));
    PRUint32 type = g_value_get_uint(&(item->values[3]));
    PRUint32 flags = g_value_get_uint(&(item->values[4]));
    nsString text = NS_ConvertUTF8toUTF16(g_value_get_string(&(item->values[5])));

    nsCOMPtr<csITPTextChannelPendingMessage> message;
    nsresult rv = csTPTextChannelPendingMessage::Create(id, timestamp, sender,
                                                        type, flags, text,
                                                        getter_AddRefs(message));
    if (NS_FAILED(rv)) {
      NS_WARNING("Oops! Could not create an instance of PendingMessage\n");
      continue;
    }
    callback->OnAddItem(message);
  }

  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPTextChannel::ListPendingMessages(PRBool aClearMsgs, 
                                                   csITPInterfaceListCB *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_channel_type_text_call_list_pending_messages(m_Channel, -1,
                                                      aClearMsgs,
                                                      GotPendingMessages,
                                                      cb, NULL, NULL);

  return NS_OK;
}

static void
GotSendMessageResponse(TpChannel *proxy, const GError *error,
                       gpointer user_data, GObject *weak_object)
{
  csITPEmptyCB *callback = (csITPEmptyCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  callback->DoAction();
  NS_IF_RELEASE(callback);
}


NS_IMETHODIMP csTPTextChannel::Send(PRUint32 aMsgType, 
                                    const nsAString & aText, csITPEmptyCB *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_channel_type_text_call_send(m_Channel, -1, aMsgType,
                                     NS_ConvertUTF16toUTF8(aText).get(),
                                     GotSendMessageResponse, cb, NULL, NULL);
  return NS_OK;
}

// Lost Message Signal

void csTPTextChannel::LostMessage()
{
  if (m_LostMessageObservers) {
    PRUint32 length;
    m_LostMessageObservers->GetLength(&length);
    nsCOMPtr<csITPEmptyCB> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_LostMessageObservers, i);
      observer->DoAction();
    }
  }
}

static void
GotLostMessageSignal(TpChannel *proxy, gpointer user_data, GObject *unused)
{
  csTPTextChannel *channel = (csTPTextChannel *)user_data;
  if (!channel)
    return;

  channel->LostMessage();
}

NS_IMETHODIMP csTPTextChannel::AddLostMessageObserver(csITPEmptyCB *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_LostMessageObservers) {
    m_LostMessageObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_type_text_connect_to_lost_message(m_Channel,
                                                     GotLostMessageSignal,
                                                     this, NULL, NULL, NULL);
  }

  m_LostMessageObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPTextChannel, RemoveLostMessageObserver,
                             csITPEmptyCB, m_LostMessageObservers);

// Message Received Signal

void csTPTextChannel::MessageReceived(PRUint32 aId, PRUint32 aTimeStamp,
                                      PRUint32 aSender, PRUint32 aType, 
                                      PRUint32 aFlags, const gchar *aMessage)
{
  if (m_ReceivedObservers) {
    PRUint32 length;
    m_ReceivedObservers->GetLength(&length);
    nsCOMPtr<csITPTextReceivedObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_ReceivedObservers, i);
      observer->OnTextReceived(aId, aTimeStamp, aSender,
                               aType, aFlags,
                               NS_ConvertUTF8toUTF16(nsDependentCString(aMessage)));
    }
  }
}

static void
GotMessageReceivedSignal(TpChannel *proxy, guint aId,
                         guint aTimeStamp, guint aSender, guint aType, 
                         guint aFlags, const gchar *aMessage,
                         gpointer user_data, GObject *unused)
{
  csTPTextChannel *channel = (csTPTextChannel *)user_data;
  if (!channel)
    return;

  channel->MessageReceived(aId, aTimeStamp, aSender, aType, aFlags, aMessage);
}

NS_IMETHODIMP csTPTextChannel::AddReceivedObserver(csITPTextReceivedObserver *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_ReceivedObservers) {
    m_ReceivedObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_type_text_connect_to_received(m_Channel,
                                                 GotMessageReceivedSignal,
                                                 this, NULL, NULL, NULL);
  }

  m_ReceivedObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPTextChannel, RemoveReceivedObserver,
                             csITPTextReceivedObserver, m_ReceivedObservers);

// Send Error Signal

void csTPTextChannel::SendError(PRUint32 aError, PRUint32 aTimeStamp,
                                PRUint32 aType, const gchar *aMessage)
{
  if (m_SendErrorObservers) {
    PRUint32 length;
    m_SendErrorObservers->GetLength(&length);
    nsCOMPtr<csITPTextSendErrorObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_SendErrorObservers, i);
      observer->OnTextSendError(aError, aTimeStamp, aType,
                                NS_ConvertUTF8toUTF16(nsDependentCString(aMessage)));
    }
  }
}

static void
GotSendErrorSignal(TpChannel *proxy, guint aError,
                   guint aTimeStamp, guint aType, const gchar *aMessage,
                   gpointer user_data, GObject *unused)
{
  csTPTextChannel *channel = (csTPTextChannel *)user_data;
  if (!channel)
    return;

  channel->SendError(aError, aTimeStamp, aType, aMessage);
}

NS_IMETHODIMP csTPTextChannel::AddSendErrorObserver(csITPTextSendErrorObserver *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_SendErrorObservers) {
    m_SendErrorObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_type_text_connect_to_send_error(m_Channel,
                                                   GotSendErrorSignal,
                                                   this, NULL, NULL, NULL);
  }

  m_SendErrorObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPTextChannel, RemoveSendErrorObserver,
                             csITPTextSendErrorObserver, m_SendErrorObservers);

// Message Sent Signal

void csTPTextChannel::MessageSent(PRUint32 aTimeStamp,
                                  PRUint32 aType, const gchar *aMessage)
{
  if (m_SentObservers) {
    PRUint32 length;
    m_SentObservers->GetLength(&length);
    nsCOMPtr<csITPTextSentObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_SentObservers, i);
      observer->OnTextSent(aTimeStamp, aType,
                           NS_ConvertUTF8toUTF16(nsDependentCString(aMessage)));
    }
  }
}

static void
GotMessageSentSignal(TpChannel *proxy, guint aTimeStamp,
                     guint aType, const gchar *aMessage,
                     gpointer user_data, GObject *unused)
{
  csTPTextChannel *channel = (csTPTextChannel *)user_data;
  if (!channel)
    return;

  channel->MessageSent(aTimeStamp, aType, aMessage);
}

NS_IMETHODIMP csTPTextChannel::AddSentObserver(csITPTextSentObserver *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_SentObservers) {
    m_SentObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_type_text_connect_to_sent(m_Channel,
                                             GotMessageSentSignal,
                                             this, NULL, NULL, NULL);
  }

  m_SentObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPTextChannel, RemoveSentObserver,
                             csITPTextSentObserver, m_SentObservers);

/******************************************************************************
 * Implementation of csITPTextChannelPendingMessage
 */

NS_METHOD
csTPTextChannelPendingMessage::Create(PRUint32 aId, PRUint32 aTimeStamp,
                                      PRUint32 aSenderHandle, PRUint32 aMessageType,
                                      PRUint32 aMessageFlags,
                                      nsString aMessageString, void **aResult)
{
  nsCOMPtr<csITPTextChannelPendingMessage> it = 
        new csTPTextChannelPendingMessage(aId, aTimeStamp, aSenderHandle,
                                          aMessageType, aMessageFlags,
                                          aMessageString);

  if (!it)
    return NS_ERROR_OUT_OF_MEMORY;

  return it->QueryInterface(NS_GET_IID(csITPTextChannelPendingMessage), aResult);
}

csTPTextChannelPendingMessage::csTPTextChannelPendingMessage(PRUint32 aId,
                                              PRUint32 aTimeStamp,
                                              PRUint32 aSenderHandle,
                                              PRUint32 aMessageType,
                                              PRUint32 aMessageFlags,
                                              nsString aMessageString)
{
  m_Id = aId;
  m_TimeStamp = aTimeStamp;
  m_SenderHandle = aSenderHandle;
  m_MessageType = aMessageType;
  m_MessageFlags = aMessageFlags;
  m_MessageString.Assign(aMessageString);
}

csTPTextChannelPendingMessage::~csTPTextChannelPendingMessage()
{
}

NS_IMETHODIMP csTPTextChannelPendingMessage::GetId(PRUint32 *aId)
{
  *aId = m_Id;
  return NS_OK;
}

NS_IMETHODIMP csTPTextChannelPendingMessage::GetTimestamp(PRUint32 *aTimestamp)
{
  *aTimestamp = m_TimeStamp;
  return NS_OK;
}

NS_IMETHODIMP csTPTextChannelPendingMessage::GetSender(PRUint32 *aSender)
{
  *aSender = m_SenderHandle;
  return NS_OK;
}

NS_IMETHODIMP csTPTextChannelPendingMessage::GetType(PRUint32 *aType)
{
  *aType = m_MessageType;
  return NS_OK;
}

NS_IMETHODIMP csTPTextChannelPendingMessage::GetFlags(PRUint32 *aFlags)
{
  *aFlags = m_MessageFlags;
  return NS_OK;
}

NS_IMETHODIMP csTPTextChannelPendingMessage::GetText(nsAString & aText)
{
  aText.Assign(m_MessageString);
  return NS_OK;
}

