package Bencher::Scenario::StringFunctions::CommonPrefix;

use strict;
use warnings;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2022-03-27'; # DATE
our $DIST = 'Bencher-Scenarios-StringFunctions'; # DIST
our $VERSION = '0.006'; # VERSION

our $scenario = {
    summary => "Benchmark calculating common prefix",
    participants => [
        {fcall_template=>'String::CommonPrefix::common_prefix(@{<strings>})'},
    ],
    datasets => [
        {name=>'elems0'          , args=>{strings=>[]}},
        {name=>'elems1'          , args=>{strings=>['x']}},
        {name=>'elems10prefix0'  , args=>{strings=>[map{sprintf "%02d", $_} 1..10]}},
        {name=>'elems10prefix1'  , args=>{strings=>[map{sprintf "%02d", $_} 0..9]}},
        {name=>'elems100prefix0' , args=>{strings=>[map{sprintf "%03d", $_} 1..100]}},
        {name=>'elems100prefix1' , args=>{strings=>[map{sprintf "%03d", $_} 0..99]}},
        {name=>'elems1000prefix0', args=>{strings=>[map{sprintf "%04d", $_} 1..1000]}},
        {name=>'elems1000prefix1', args=>{strings=>[map{sprintf "%04d", $_} 0..999]}},
    ],
};

1;
# ABSTRACT: Benchmark calculating common prefix

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::StringFunctions::CommonPrefix - Benchmark calculating common prefix

=head1 VERSION

This document describes version 0.006 of Bencher::Scenario::StringFunctions::CommonPrefix (from Perl distribution Bencher-Scenarios-StringFunctions), released on 2022-03-27.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m StringFunctions::CommonPrefix

To run module startup overhead benchmark:

 % bencher --module-startup -m StringFunctions::CommonPrefix

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<String::CommonPrefix> 0.01

=head1 BENCHMARK PARTICIPANTS

=over

=item * String::CommonPrefix::common_prefix (perl_code)

Function call template:

 String::CommonPrefix::common_prefix(@{<strings>})



=back

=head1 BENCHMARK DATASETS

=over

=item * elems0

=item * elems1

=item * elems10prefix0

=item * elems10prefix1

=item * elems100prefix0

=item * elems100prefix1

=item * elems1000prefix0

=item * elems1000prefix1

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.34.0 >>, CPU: I<< Intel(R) Core(TM) i5-7200U CPU @ 2.50GHz (2 cores) >>, OS: I<< GNU/Linux Ubuntu version 20.04 >>, OS kernel: I<< Linux version 5.4.0-91-generic >>.

Benchmark with default options (C<< bencher -m StringFunctions::CommonPrefix >>):

 #table1#
 {dataset=>"elems0"}
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | participant                         | dataset | ds_tags | p_tags | perl | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest | errors | samples |
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | String::CommonPrefix::common_prefix | elems0  |         |        | perl |   7697760 |   129.908 |                 0.00% |                 0.00% |      0 |      20 |
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

         Rate     
    7697760/s  -- 
 
 Legends:
   : dataset=elems0 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

=for html <img src="data:image/png;base64,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" />


 #table2#
 {dataset=>"elems1"}
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | participant                         | dataset | ds_tags | p_tags | perl | rate (/s) | time (ns) | pct_faster_vs_slowest | pct_slower_vs_fastest | errors | samples |
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | String::CommonPrefix::common_prefix | elems1  |         |        | perl |   2105200 |   475.013 |                 0.00% |                 0.00% |      0 |      24 |
 +-------------------------------------+---------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

         Rate     
    2105200/s  -- 
 
 Legends:
   : dataset=elems1 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

=for html <img src="data:image/png;base64,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" />


 #table3#
 {dataset=>"elems1000prefix0"}
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset          | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems1000prefix0 |         |        | perl |      4440 |       225 |                 0.00% |                 0.00% | 2.1e-07 |      20 |
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

      Rate     
    4440/s  -- 
 
 Legends:
   : dataset=elems1000prefix0 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

=for html <img src="data:image/png;base64,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" />


 #table4#
 {dataset=>"elems1000prefix1"}
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                         | dataset          | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix::common_prefix | elems1000prefix1 |         |        | perl |      3850 |       260 |                 0.00% |                 0.00% | 2.1e-07 |      20 |
 +-------------------------------------+------------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

      Rate     
    3850/s  -- 
 
 Legends:
   : dataset=elems1000prefix1 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

=for html <img src="data:image/png;base64,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" />


 #table5#
 {dataset=>"elems100prefix0"}
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | participant                         | dataset         | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest | errors | samples |
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | String::CommonPrefix::common_prefix | elems100prefix0 |         |        | perl |   43348.8 |   23.0687 |                 0.00% |                 0.00% |      0 |      22 |
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

         Rate     
    43348.8/s  -- 
 
 Legends:
   : dataset=elems100prefix0 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

=for html <img src="data:image/png;base64,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" />


 #table6#
 {dataset=>"elems100prefix1"}
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | participant                         | dataset         | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest | errors | samples |
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | String::CommonPrefix::common_prefix | elems100prefix1 |         |        | perl |   37104.3 |   26.9511 |                 0.00% |                 0.00% |      0 |      20 |
 +-------------------------------------+-----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

         Rate     
    37104.3/s  -- 
 
 Legends:
   : dataset=elems100prefix1 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

=for html <img src="data:image/png;base64,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" />


 #table7#
 {dataset=>"elems10prefix0"}
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | participant                         | dataset        | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest | errors | samples |
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | String::CommonPrefix::common_prefix | elems10prefix0 |         |        | perl |    365948 |   2.73263 |                 0.00% |                 0.00% |      0 |      22 |
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

        Rate     
    365948/s  -- 
 
 Legends:
   : dataset=elems10prefix0 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

=for html <img src="data:image/png;base64,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" />


 #table8#
 {dataset=>"elems10prefix1"}
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | participant                         | dataset        | ds_tags | p_tags | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest | errors | samples |
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+
 | String::CommonPrefix::common_prefix | elems10prefix1 |         |        | perl |    315581 |   3.16876 |                 0.00% |                 0.00% |      0 |      20 |
 +-------------------------------------+----------------+---------+--------+------+-----------+-----------+-----------------------+-----------------------+--------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

        Rate     
    315581/s  -- 
 
 Legends:
   : dataset=elems10prefix1 ds_tags= p_tags= participant=String::CommonPrefix::common_prefix perl=perl

=for html <img src="data:image/png;base64,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" />


Benchmark module startup overhead (C<< bencher -m StringFunctions::CommonPrefix --module-startup >>):

 #table9#
 +----------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+
 | participant          | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +----------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+
 | String::CommonPrefix |      8.82 |              2.62 |                 0.00% |                42.80% | 8.5e-06 |      21 |
 | perl -e1 (baseline)  |      6.2  |              0    |                42.80% |                 0.00% | 1.9e-05 |      20 |
 +----------------------+-----------+-------------------+-----------------------+-----------------------+---------+---------+


Formatted as L<Benchmark.pm|Benchmark> result:

                          Rate  S:C  perl -e1 (baseline) 
  S:C                  113.4/s   --                 -29% 
  perl -e1 (baseline)  161.3/s  42%                   -- 
 
 Legends:
   S:C: mod_overhead_time=2.62 participant=String::CommonPrefix
   perl -e1 (baseline): mod_overhead_time=0 participant=perl -e1 (baseline)

=for html <img src="data:image/png;base64,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" />


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-StringFunctions>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-StringFunctions>.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla plugin and/or Pod::Weaver::Plugin. Any additional steps required
beyond that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2022, 2021, 2018 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-StringFunctions>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut
