package OpenInteract::Handler::News;

# $Id: News.pm,v 1.18 2003/01/09 15:07:26 lachoy Exp $

use strict;
use base qw( OpenInteract::Handler::GenericDispatcher  SPOPS::Secure );
use SPOPS::Secure qw( :level );
use Data::Dumper  qw( Dumper );
use SPOPS::Utility;

$OpenInteract::Handler::News::VERSION = sprintf("%d.%02d", q$Revision: 1.18 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::News::author            = 'chris@cwinters.com';
$OpenInteract::Handler::News::default_method    = 'home';
@OpenInteract::Handler::News::forbidden_methods = qw( _process_listing );
%OpenInteract::Handler::News::security          = (
 home    => SEC_LEVEL_READ,
 latest  => SEC_LEVEL_READ,
 listing => SEC_LEVEL_READ,
 show    => SEC_LEVEL_READ,
 search  => SEC_LEVEL_READ,
 edit    => SEC_LEVEL_WRITE,
 remove  => SEC_LEVEL_WRITE,
 notify  => SEC_LEVEL_READ,
 show_summary => SEC_LEVEL_WRITE,
 edit_summary => SEC_LEVEL_WRITE,
);

# 12 weeks -- default expiration for news item
use constant DEFAULT_EXPIRE => 60 * 60 * 24 * 7 * 12;

sub home {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my %params = ( error_msg  => $p->{error_msg},
                   status_msg => $p->{status_msg} );
    $params{section_list} = eval{ $R->news_section->fetch_group({
                                             order => 'section' }) };
    return $class->generate_content( $p,
                                     {},
                                     \%params,
                                     { name => 'news::news_home' } );
}

sub latest {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $apr = $R->apache;
    my $num_items = $p->{num_items} || $apr->param( 'num_items' ) || 5;
    my $cached = $class->check_cache( $p, { num_items => $num_items } );
    return $cached if ( $cached );

    my %params = ( error_msg   => $p->{error_msg},
                   show_box    => $p->{show_box},
                   status_msg  => $p->{status_msg},
                   num_items   => $num_items );
    my $where = "active_on <= ? AND active = ? ";
    my @values = ( SPOPS::Utility->now, 'yes' );
    $params{news_list} = $class->_process_listing({
                                        where => $where,
                                        value => \@values,
                                        limit => $params{num_items} });
    return $class->generate_content( $p,
                                     { num_items => $num_items },
                                     \%params,
                                     { name => 'news::news_listing' } );
}


# Just provide a listing of news titles, optionally filtered by
# section

sub listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $apr = $R->apache;
    my $option  = $p->{option}  || $apr->param( 'option' );
    my $section = $p->{section} || $apr->param( 'section' );
    my %date    = ( month => $apr->param( 'news_month' ),
                    day   => $apr->param( 'news_day' ),
                    year  => $apr->param( 'news_year' ) );
    my $use_date = ( $date{month} and $date{year} );

    my %cache_params = ( option  => $option, section => $section );
    if ( $use_date ) {
        $cache_params{date} = join( '-', map { $date{ $_ } } sort keys %date );
    }
    my $cached = $class->check_cache( $p, \%cache_params );
    return $cached if ( $cached );

    my %params = ( error_msg   => $p->{error_msg},
                   status_msg  => $p->{status_msg},
                   section     => $section );
    my $where = " active_on <= ? AND active = ? ";
    my @values = ( SPOPS::Utility->now, 'yes' );

    $where .= ( $option eq 'older' )
                ? ' AND ( expires_on < ? ) '
                : ' AND ( expires_on IS NULL OR expires_on > ? ) ';
    push @values, SPOPS::Utility->now;

    if ( $section ) {
        $where .= ' AND ( section = ? )';
        push @values, $section;
    }

    # Feh - I should use a date object for this...

    if ( $use_date ) {
        $where .= 'AND ( posted_on >= ? AND posted_on < ? )';
        if ( $date{day} > 0 and $date{day} < 32 ) {
            push @values, join( '-', $date{year}, $date{month}, $date{day} ),
                          join( '-', $date{year}, $date{month}, $date{day} + 1 );
            $params{date} = join( '-', $date{year}, $date{month}, $date{day} );
        }
        else {
            push @values, join( '-', $date{year}, $date{month}, 1 ),
                          join( '-', $date{year}, $date{month} + 1, 1 );
            $params{date} = join( '-', $date{year}, $date{month} );
        }
    }

    $params{news_list} = $class->_process_listing({ where        => $where,
                                                    value        => \@values,
                                                    column_group => 'listing' });
    $params{section_list} = eval{ $R->news_section->fetch_group({
                                             order => 'section' }) };
    $R->{page}{title} = 'News Entries';
    return $class->generate_content( $p, \%cache_params, \%params,
                                     { name => 'news::news_list' } );
}


sub _process_listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "Processing news listing with ($p->{where})" );

    # Note: The date-formatting stuff previously here has been
    # deleted. Use your database-specific date formatting functions in the
    # $WEBSITE_DIR/conf/override_spops.ini file.

    my $order = $p->{order} || 'posted_on DESC';
    my $news_list = eval { $R->news->fetch_group({
                                      where        => $p->{where},
                                      value        => $p->{value},
                                      limit        => $p->{limit},
                                      column_group => $p->{column_group},
                                      order        => $order  }) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 403 } );
        $news_list = [];
    }

    # Only grab the first 'chunk' of the news item, as split up by the
    # separator; also create information about the user who posted the
    # story

    my %posters = ();
    foreach my $news ( @{ $news_list } ) {
        if ( $news->{news_item} ) {
            ( $news->{tmp_content} ) = split '<!--BREAK-->', $news->{news_item};

            # If there is template content in the news item, process it

            if ( $news->{tmp_content} =~ m|\[\%| ) {
                $R->DEBUG && $R->scrib( 1, "Processing template directive content ",
                                           "for news item ($news->{title})" );
                $news->{tmp_content} = $R->template->handler( {}, {},
                                                              { text => $news->{tmp_content} } );
            }
        }

        # Now grab relevant user information, caching as we go (since
        # the same people tend to post news)

        unless ( $posters{ $news->{posted_by} } ) {
            my $user = eval { $news->posted_by_user() };
            my $poster_info = {};
            if ( $@ ) {
                $poster_info->{login_name} = 'admin';
                $poster_info->{user_id}    = undef;
            }
            else {
                $poster_info->{login_name} = $user->{login_name};
                $poster_info->{user_id}    = $user->{user_id};
            }
            $posters{ $user->{user_id} } = $poster_info;
        }
        $news->{tmp_poster_info} = $posters{ $news->{posted_by} };
    }
    return $news_list;
}


# Display detail of a news item -- displays it in editable form for
# those who have permission to edit the object.

sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $apr = $R->apache;
    my %params = ( error_msg   => $p->{error_msg},
                   status_msg  => $p->{status_msg}  );
    my $do_edit = ( $R->apache->param( 'edit' ) and
                    $p->{level} >= SEC_LEVEL_WRITE );

    $params{news} = $p->{news};
    unless ( $params{news} ) {
        my $news_id = $apr->param( 'news_id' );
        if ( $news_id ) {
            $params{news} = eval { $R->news->fetch( $news_id ) };
            return $class->listing( { error_msg => $@ } ) if ( $@ );
        }
    }

    # If the user has less than WRITE access to module and there is no
    # news_id or news object, there's no reason to continue

    unless ( $params{news} or $do_edit ) {
        my $error_msg = 'Sorry, the news object you requested does not ' .
                        'exist. Returning to listing.';
        return $class->listing({ error_msg => $error_msg });
    }

    # Object security defaults to WRITE: if there is no object,
    # then you're creating it and security then depends on the application

    my $obj_level = ( $params{news} )
                      ? $params{news}{tmp_security_level}
                      : SEC_LEVEL_WRITE;

    $params{news}  ||= $R->news->new;
    $params{section_list} = eval { $R->news_section->fetch_group({
                                                       order => 'section' }) };
    my ( $tmpl_name );
    if ( $do_edit and $obj_level == SEC_LEVEL_WRITE ) {
        $tmpl_name = 'news_form';
        $R->{page}{title} = ( $params{news}->is_saved )
                              ? "Editing News: $params{news}->{title}"
                              : "Create News";
    }
    else {
         $tmpl_name = 'news_detail';
         $R->{page}{title} = "News: $params{news}->{title}";
    }
    return $R->template->handler( {}, \%params,
                                  { name => "news::$tmpl_name" } );
}



# Make changes to a news item -- either create it or update it

sub edit {
    my ( $class, $p ) = @_; 
    my $R = OpenInteract::Request->instance;
    my $apr = $R->apache;
    $R->{page}{return_url} = '/News/';

    my $news = eval { $class->_create_object( { _class => $R->news,
                                                _id_field => 'news_id' } ) };
    return $class->listing( { error_msg => $@ } ) if ( $@ );

    # If we didn't retrieve an object, assume it's new and
    # that our default security for this object is WRITE; if the
    # user isn't supposed to be creating news objects at all,
    # he/she should have a READ permission only for the module

    my $obj_level = ( $news ) ? $news->{tmp_security_level} : SEC_LEVEL_WRITE;
    if ( $obj_level < SEC_LEVEL_WRITE ) {
        my $user_msg = 'Sorry, you do not have access to modify this ' .
                       'news object. Returning to listing.';
        return $class->show({ news => $news, error_msg => $user_msg });
    }

    # Again, create a new object if one wasn't created before.
    $news ||= $R->news->new;

    my %no_read = map { $_ => 1 }
                     qw( posted_by expires_on active_on news_id next_id previous_id );
    foreach my $field ( @{ $R->news->field_list } ) {
        next if ( $no_read{ $field } );
        $R->DEBUG && $R->scrib( 1, "Find value for field <<$field>>" );
        $news->{ $field } = $apr->param( $field );
    }

    # Set our date fields

    $news->{expires_on} = $class->date_read( 'expires_on' );
    $news->{active_on}  = $class->date_read( 'active_on' ) || SPOPS::Utility->now;
    $news->{posted_on}  = $class->date_read( 'posted_on' ) || SPOPS::Utility->now;

    # substitute <p> for hard returns where needed

    $news->{news_item} =~ s/(\r\n\r\n)(?!<(p|pre|blockquote)>)/$1<p>/g;

    # Set the expiration date if it wasn't set

    unless ( $news->{expires_on} ) {
        my $expire_time = $R->{time} + DEFAULT_EXPIRE;
        my @time_info = localtime( $expire_time );
        $news->{expires_on} = join '-', $time_info[5] + 1900, $time_info[4] + 1, $time_info[3];
    }

    # If the image URL wasn't set to something real, clear it

    $news->{image_url} = undef if ( $news->{image_url} =~ m|^http://\s*$| );

    # Set other defaults

    $news->{posted_by}   ||= $R->{auth}{user}->id;
    $news->{image_align} ||= 'left';

    eval { $news->save };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 407 } );
        $p->{error_msg} = 'Cannot save news object! See error log for details.';
    }
    else {
        $p->{status_msg} = 'News object updated ok';
        $class->clear_cache();
    }
    $p->{news} = $news;
    return $class->show( $p );
}



sub remove {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $news = eval { $class->_create_object( { _class => $R->news,
                                                _id_field => 'news_id' } ) };
    if ( $@ ) {
        return $class->listing({ error_msg => $@ });
    }
    unless ( $news ) {
        return $class->listing({ error_msg => 'Cannot locate news object for removal.' }) ;
    }
    if ( $news->{tmp_security_level} < SEC_LEVEL_WRITE ) {
        my $error_msg = 'Sorry, you do not have access to remove this news object. Returning to listing.';
        return $class->listing( { error_msg => $error_msg } );
    }
    eval { $news->remove };
    if ( $@ ) {
        my $ei = OpenInteract::Error->set( SPOPS::Error->get );
        $R->DEBUG && $R->scrib( 1, "Cannot remove News: $@\n", Dumper( $ei ) );
        $R->throw( { code => 405 } );
        $p->{error_msg} = "Cannot remove news object! See error log.";
    }
    else {
        $p->{status_msg} = 'Removed news object ok';
        $class->clear_cache();
    }
    return $class->listing( $p );
}



sub notify {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    my @news_id  = $R->apache->param( 'news_id' );
    my $email    = $R->apache->param( 'email' );
    if ( ! $email or ! scalar @news_id ) {
        return '<h2 align="center">Error</h2>' .
               '<p>Error: Cannot notify anyone about an object when no ID/email is given.</p>';
    }
    my @news_list = ();
    foreach my $nid ( @news_id ) {
        my $news = $R->news->fetch( $nid );
        push @news_list, $news   if ( $news );
    }
    my $rv = $R->news->notify({ email   => $email,
                                subject => 'News notification',
                                object  => \@news_list,
                                type    => 'news' });
    if ( $rv ) {
        return '<h2 align="center">Success!</h2>' .
               '<p>Notification sent properly!</p>';
    }
    return '<h2 align="center">Error</h2>' .
           '<p>Error sending email. Please check error logs!</p>';
}

# (IS THIS USED ANYMORE?)

sub search {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $search = $R->apache->param( 'search' );

    my $params = { error_msg   => $p->{error_msg} };

    my $where = " active_on <= NOW() AND active = ? ";
    if ( $search ) {
        $search =~ s/ /\', \'/goi;
        my $in_clause = "IN ( '$search' )";
        $where .= " AND ( title $in_clause OR news_item $in_clause ) ";
    }
    my @values = ( 'yes' );
    my $option = $R->apache->param( 'option' );
    if ( $option eq 'older' ) {
        $where .= ' AND ( expires_on < ? ) ';
    }
    else {
        $where .= ' AND ( expires_on IS NULL OR expires_on > ? ) ';
    }
    push @values, SPOPS::Utility->now;
    $params->{news_list} = $class->_process_listing({ where => $where,
                                                      value => \@values });
    $R->{page}{title} = 'News Search';
    $params->{search} = $search;
    return $R->template->handler( {}, $params,
                                  { name => 'news::news_list' } );
}

# Fetch all the news items for display
sub show_summary {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $section_list = $R->news_section->fetch_group({ order => 'section' });
    my %params = ( error_msg    => $p->{error_msg},
                   status_msg   => $p->{status_msg},
                   section_list => $section_list );
    $params{news_list} = $R->news->fetch_group({ order => 'posted_on DESC' });
    $R->{page}{title} = 'Edit Many News Items Simultaneously';
    return $R->template->handler( {}, \%params,
                                  { name => 'news::news_list_editable' } );
}


sub edit_summary {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $apr = $R->apache;
    my @do_edit = $apr->param( 'do_edit' );
    my ( $success, $attempt );
    my @errors = ();
    foreach my $news_id ( @do_edit ) {
        next unless ( $news_id );
        $attempt++;
        my $news = $R->news->fetch( $news_id );
        $news->{title} = $apr->param( "title_$news_id" );
        $news->{section} = $apr->param( "section_$news_id" );
        $news->{active}  = $apr->param( "active_$news_id" );
        eval { $news->save };
        if ( $@ ) {
            push @errors, "$@";
        }
        else {
            $success++;
        }
    }
    $class->clear_cache();
    my %params = ( status_msg => "Attempted: $attempt; successful updates $success",
                   error_msg  => join( "<br><br>\n", @errors ) );
    return $class->listing( \%params );
}

1;
