package OpenInteract::Page::Http;

# $Id: Http.pm,v 1.5 2002/12/13 21:41:14 lachoy Exp $

use strict;
use HTTP::Request;
use LWP::UserAgent;

my ( $AGENT );

BEGIN {
    $AGENT = LWP::UserAgent->new();
    $AGENT->agent( "OpenInteract Requester " . $AGENT->agent );
}

sub load {
    my ( $class, $page ) = @_;
    my $R = OpenInteract::Request->instance;
    unless ( $page->{content_location} ) {
        $R->scrib( 0, "No content location available -- cannot display" );
        return "Cannot retrieve content -- no URL specified.";
    }
    unless ( $page->{content_location} =~ /^http/ ) {
        $R->scrib( 0, "Content location invalid ($page->{content_location})" );
        return "Cannot retrieve content -- invalid URL specified.";
    }

    $R->scrib( 1, "Trying to fetch URL ($page->{content_location})" );
    my $request  = HTTP::Request->new(
                         GET => $page->{content_location} );
    my $response = $AGENT->request( $request );
    if ( $response->is_success ) {
        my $content = $response->content;
        $content =~ s|^.*<body.*?>||ism;
        $content =~ s|</body>.*$||ism;
        return $class->rewrite_content( $content );
    }
    return 'Cannot retrieve content -- code ' . $response->code . ' returned.';
}

sub rewrite_content { return $_[1] }

sub save {
    my ( $class, $page ) = @_;
    warn "--Location $page->{location} cannot be saved, since it's ",
         "using HTTP storage.\n";
    return 1;
}


sub rename_content {
    my ( $class, $page, $old_location ) = @_;
    warn "--Cannot rename content from [$old_location] to [$page->{location}]",
         "since it's using HTTP storage.\n";
    return 1;
}


sub remove {
    my ( $class, $page ) = @_;
    warn "--Location $page->{location} cannot be removed, since it's ",
         "using HTTP storage.\n";
    return 1;
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Page::Http - Fetch page content from a URL

=head1 SYNOPSIS

 my $page = $R->page->new({ storage          => 'http',
                            content_location => 'http://slashdot.org/' });
 print $page->content

=head1 DESCRIPTION

Retrieves content from a URL rather than the filesystem or database.
The URL is specified in the 'content_location' property of the page
object, and the 'storage' property is set to 'http'.

We strip everything before and including the <body> tag and everything
after and including the </body> tag.

=head1 METHODS

B<load( $page )>

Returns the content from the URL stored in the 'content_location'
property.

B<save( $page )>

Not implemented.

B<remove( $page )>

Not implemented.

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 SEE ALSO

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
