#!/usr/bin/perl -w

###############################################################################
#
# Generate function names for different locales.
#
# This program can be used to generate the hash of known functions for use in
# the Formula.pm module. By default the function names are in English but you
# can also choose to support the following languages: German, French, Spanish,
# Portuguese, Dutch, Finnish, Italian and Swedish.
#
# This would allow you to do something like the following:
#
#       $worksheet->write(0, 0, '=SUM(C1:C3)'  );
#       $worksheet->write(1, 0, '=SUMME(C1:C3)');
#       $worksheet->write(2, 0, '=SOMME(C1:C3)');
#       $worksheet->write(3, 0, '=SUMA(C1:C3)' );
#       $worksheet->write(4, 0, '=SOMA(C1:C3)' );
#       $worksheet->write(5, 0, '=SOM(C1:C3)'  );
#       $worksheet->write(6, 0, '=SUMMA(C1:C3)');
#       $worksheet->write(7, 0, '=SOMMA(C1:C3)');
#
# Unfortunately, if you wish to support more than one language there are some
# conflicts between function names:
#
#       Function        Language 1              Language 2
#       ========        ==========              ==========
#       NB              French                  Dutch
#       NA              English/French          Finnish
#       TRIM            French                  English
#       DIA             Spanish/Portuguese      German
#
# Therefore, if you try to generate a hash of function names to support both
# French and English then the function TRIM will be assigned the meaning of the
# first language that defines it, which in this case is French. You can get
# around this by renaming the function for one of the languages and documenting
# the change, for example: TRIM.EN or TRIM.FR.
#
# Please note that this only partially solves the problem of localisation.
# There are also number formats to consider (1.5 == 1,5) and the fact that the
# list separator "," and the array separator ";" are interchanged in different
# locales.
#
# The longest function name is LOI.NORMALE.STANDARD.INVERSE (29 chars) followed
# by NORM.JAKAUMA.NORMIT.KNT (25 chars).
# The shortest function name in all languages is T.
#
#
# reverse(''); John McNamara, March 2001, jmcnamara@cpan.org
#



use strict;


#
# Modify the following variables to add the language to the function name hash
#
my $english    = 1;
my $german     = 0;
my $french     = 0;
my $spanish    = 0;
my $portuguese = 0;
my $dutch      = 0;
my $finnish    = 0;
my $italian    = 0;
my $swedish    = 0;

my %funcs;

# Ignore the headings
<DATA>;

# Print the beginning of the hash definition
print "    %functions  = (\n";
print "        #" . " " x 37 ;
print "ptg  args  class  vol\n";

while (<DATA>){
    my @F = split " ";
    my $value = $F[0];
    my $args  = $F[1];
    my $ref   = $F[2];
    my $vol   = $F[3];

    print_function($F[4],  $value, $args, $ref, $vol) if $english;
    print_function($F[5],  $value, $args, $ref, $vol) if $german;
    print_function($F[6],  $value, $args, $ref, $vol) if $french;
    print_function($F[7],  $value, $args, $ref, $vol) if $spanish;
    print_function($F[8],  $value, $args, $ref, $vol) if $portuguese;
    print_function($F[9],  $value, $args, $ref, $vol) if $dutch;
    print_function($F[10], $value, $args, $ref, $vol) if $finnish;
    print_function($F[11], $value, $args, $ref, $vol) if $italian;
    print_function($F[12], $value, $args, $ref, $vol) if $swedish;
}
# Print the end of the hash definition
print "    );\n";


###############################################################################
#
# Function to print the function names. It prints a warning if there is a
# clash.
#
sub print_function {

    my $func  = shift;
    my $value = shift;
    my $args  = shift;
    my $ref   = shift;
    my $vol   = shift;

    $func = "'$func'";

    if (not exists $funcs{$func}) {
        printf("        %-31s => [%4d, %4d, %4d, %4d ],\n",
                                   $func, $value, $args, $ref, $vol);
        $funcs{$func} = $value;
    }
    else {
        if ($funcs{$func} != $value) {
            print "        # Warning ";
            print $func, " is already defined in another language\n";
            printf("        #%-31s => [%4d, %4d, %4d, %4d ],\n",
                                        $func, $value, $args, $ref, $vol);
        }
    }
}


# Note: The following data contains the function names in the various
# languages. These lines are LONG.

__DATA__
Value   Args    Refclass   Volatile   English                            German                             French                             Spanish                            Portuguese                         Dutch                        Finnish                            Italian                            Swedish
0       -1      0          0          COUNT                              ANZAHL                             NB                                 CONTAR                             CONT.NM                           AANTAL                       LASKE                              CONTA.NUMERI                       ANTAL
1       -1      1          0          IF                                 WENN                               SI                                 SI                                 SE                                 ALS                          JOS                                SE                                 OM
2        1      1          0          ISNA                               ISTNV                              ESTNA                              ESNOD                              .NO.DISP                         ISNB                         ONPUUTTUU                          VAL.NON.DISP                       RSAKNAD
3        1      1          0          ISERROR                            ISTFEHLER                          ESTERREUR                          ESERROR                            ERROS                             ISFOUT                       ONVIRHE                            VAL.ERRORE                         RFEL
4       -1      0          0          SUM                                SUMME                              SOMME                              SUMA                               SOMA                               SOM                          SUMMA                              SOMMA                              SUMMA
5       -1      0          0          AVERAGE                            MITTELWERT                         MOYENNE                            PROMEDIO                           MDIA                              GEMIDDELDE                   KESKIARVO                          MEDIA                              MEDEL
6       -1      0          0          MIN                                MIN                                MIN                                MIN                                MNIMO                             MIN                          MIN                                MIN                                MIN
7       -1      0          0          MAX                                MAX                                MAX                                MAX                                MXIMO                             MAX                          MAKS                               MAX                                MAX
8       -1      0          0          ROW                                ZEILE                              LIGNE                              FILA                               LIN                                RIJ                          RIVI                               RIF.RIGA                           RAD
9       -1      0          0          COLUMN                             SPALTE                             COLONNE                            COLUMNA                            COL                                KOLOM                        SARAKE                             RIF.COLONNA                        KOLUMN
10       0      0          0          NA                                 NV                                 NA                                 NOD                                NO.DISP                           NB                           PUUTTUU                            NON.DISP                           SAKNAS
11      -1      1          0          NPV                                NBW                                VAN                                VNA                                VPL                                NHW                          NNA                                VAN                                NETNUVRDE
12      -1      0          0          STDEV                              STABW                              ECARTYPE                           DESVEST                            DESVPAD                            STDEV                        KESKIHAJONTA                       DEV.ST                             STDAV
13      -1      1          0          DOLLAR                             DM                                 FRANC                              MONEDA                             MOEDA                              GULDEN                       VALUUTTA                           VALUTA                             VALUTA
14      -1      1          0          FIXED                              FEST                               CTXT                               DECIMAL                            DEF.NM.DEC                        VAST                         KIINTE                            FISSO                              FASTTAL
15       1      1          0          SIN                                SIN                                SIN                                SENO                               SEN                                SIN                          SIN                                SEN                                SIN
16       1      1          0          COS                                COS                                COS                                COS                                COS                                COS                          COS                                COS                                COS
17       1      1          0          TAN                                TAN                                TAN                                TAN                                TAN                                TAN                          TAN                                TAN                                TAN
18       1      1          0          ATAN                               ARCTAN                             ATAN                               ATAN                               ATAN                               BOOGTAN                      ATAN                               ARCTAN                             ARCTAN
19       0      1          0          PI                                 PI                                 PI                                 PI                                 PI                                 PI                           PII                                PI.GRECO                           PI
20       1      1          0          SQRT                               WURZEL                             RACINE                             RAIZ                               RAIZ                               WORTEL                       NELIJUURI                         RADQ                               ROT
21       1      1          0          EXP                                EXP                                EXP                                EXP                                EXP                                EXP                          EKSPONENTTI                        EXP                                EXP
22       1      1          0          LN                                 LN                                 LN                                 LN                                 LN                                 LN                           LUONNLOG                           LN                                 LN
23       1      1          0          LOG10                              LOG10                              LOG10                              LOG10                              LOG10                              LOG10                        LOG10                              LOG10                              LOG10
24       1      1          0          ABS                                ABS                                ABS                                ABS                                ABS                                ABS                          ITSEISARVO                         ASS                                ABS
25       1      1          0          INT                                GANZZAHL                           ENT                                ENTERO                             INT                                INTEGER                      KOKONAISLUKU                       INT                                HELTAL
26       1      1          0          SIGN                               VORZEICHEN                         SIGNE                              SIGNO                              SINAL                              POS.NEG                      ETUMERKKI                          SEGNO                              TECKEN
27       2      1          0          ROUND                              RUNDEN                             ARRONDI                            REDONDEAR                          ARRED                              AFRONDEN                     PYRIST                           ARROTONDA                          AVRUNDA
28      -1      0          0          LOOKUP                             VERWEIS                            RECHERCHE                          BUSCAR                             PROC                               ZOEKEN                       HAKU                               CERCA                              LETAUPP
29      -1      0          1          INDEX                              INDEX                              INDEX                              INDICE                             NDICE                             INDEX                        INDEKSI                            INDICE                             INDEX
30       2      1          0          REPT                               WIEDERHOLEN                        REPT                               REPETIR                            REPETIR                            HERHALING                    TOISTA                             RIPETI                             REP
31       3      1          0          MID                                TEIL                               STXT                               EXTRAE                             EXT.TEXTO                          DEEL                         POIMI.TEKSTI                       STRINGA.ESTRAI                     EXTEXT
32       1      1          0          LEN                                LNGE                              NBCAR                              LARGO                              NM.CARACT                         LENGTE                       PITUUS                             LUNGHEZZA                          LNGD
33       1      1          0          VALUE                              WERT                               CNUM                               VALOR                              VALOR                              WAARDE                       ARVO                               VALORE                             TEXTNUM
34       0      1          0          TRUE                               WAHR                               VRAI                               VERDADERO                          VERDADEIRO                         WAAR                         TOSI                               VERO                               SANT
35       0      1          0          FALSE                              FALSCH                             FAUX                               FALSO                              FALSO                              ONWAAR                       EPTOSI                            FALSO                              FALSKT
36      -1      0          0          AND                                UND                                ET                                 Y                                  E                                  EN                           JA                                 E                                  OCH
37      -1      0          0          OR                                 ODER                               OU                                 O                                  OU                                 OF                           TAI                                O                                  ELLER
38       1      1          0          NOT                                NICHT                              NON                                NO                                 NO                                NIET                         EI                                 NON                                ICKE
39       2      1          0          MOD                                REST                               MOD                                RESIDUO                            MOD                                REST                         JAKOJ                              RESTO                              REST
40       3      0          0          DCOUNT                             DBANZAHL                           BDNB                               BDCONTAR                           BDCONTAR                           DBAANTAL                     TLASKE                             DB.CONTA.NUMERI                    DANTAL
41       3      0          0          DSUM                               DBSUMME                            BDSOMME                            BDSUMA                             BDSOMA                             DBSOM                        TSUMMA                             DB.SOMMA                           DSUMMA
42       3      0          0          DAVERAGE                           DBMITTELWERT                       BDMOYENNE                          BDPROMEDIO                         BDMDIA                            DBGEMIDDELDE                 TKESKIARVO                         DB.MEDIA                           DMEDEL
43       3      0          0          DMIN                               DBMIN                              BDMIN                              BDMIN                              BDMN                              DBMIN                        TMIN                               DB.MIN                             DMIN
44       3      0          0          DMAX                               DBMAX                              BDMAX                              BDMAX                              BDMX                              DBMAX                        TMAKS                              DB.MAX                             DMAX
45       3      0          0          DSTDEV                             DBSTDABW                           BDECARTYPE                         BDDESVEST                          BDEST                              DBSTDEV                      TKESKIHAJONTA                      DB.DEV.ST                          DSTDAV
46      -1      0          0          VAR                                VARIANZ                            VAR                                VAR                                VAR                                VAR                          VAR                                VAR                                VARIANS
47       3      0          0          DVAR                               DBVARIANZ                          BDVAR                              BDVAR                              BDVAREST                           DBVAR                        TVARIANSSI                         DB.VAR                             DVARIANS
48       2      1          0          TEXT                               TEXT                               TEXTE                              TEXTO                              TEXTO                              TEKST                        TEKSTI                             TESTO                              TEXT
49      -1      0          0          LINEST                             RGP                                DROITEREG                          ESTIMACION.LINEAL                  PROJ.LIN                           LIJNSCH                      LINREGR                            REGR.LIN                           REGR
50      -1      0          0          TREND                              TREND                              TENDANCE                           TENDENCIA                          TENDNCIA                          TREND                        SUUNTAUS                           TENDENZA                           TREND
51      -1      0          0          LOGEST                             RKP                                LOGREG                             ESTIMACION.LOGARITMICA             PROJ.LOG                           LOGSCH                       LOGREGR                            REGR.LOG                           EXPREGR
52      -1      0          0          GROWTH                             VARIATION                          CROISSANCE                         CRECIMIENTO                        CRESCIMENTO                        GROEI                        KASVU                              CRESCITA                           EXPTREND
56      -1      1          0          PV                                 BW                                 VA                                 VA                                 VP                                 HW                           NA                                 VA                                 NUVRDE
57      -1      1          0          FV                                 ZW                                 VC                                 VF                                 VF                                 TW                           TULEVA.ARVO                        VAL.FUT                            SLUTVRDE
58      -1      1          0          NPER                               ZZR                                NPM                                NPER                               NPER                               NPER                         NJAKSO                             NUM.RATE                           PERIODER
59      -1      1          0          PMT                                RMZ                                VPM                                PAGO                               PGTO                               BET                          MAKSU                              RATA                               BETALNING
60      -1      1          0          RATE                               ZINS                               TAUX                               TASA                               TAXA                               RENTE                        KORKO                              TASSO                              RNTA
61       3      0          0          MIRR                               QIKV                               TRIM                               TIRM                               MTIR                               GIR                          MSISINEN                          TIR.VAR                            MODIR
62      -1      0          0          IRR                                IKV                                TRI                                TIR                                TIR                                IR                           SISINEN.KORKO                     TIR.COST                           IR
63       0      1          1          RAND                               ZUFALLSZAHL                        ALEA                               ALEATORIO                          ALEATRIO                          ASELECT                      SATUNNAISLUKU                      CASUALE                            SLUMP
64      -1      0          0          MATCH                              VERGLEICH                          EQUIV                              COINCIDIR                          CORRESP                            VERGELIJKEN                  VASTINE                            CONFRONTA                          PASSA
65       3      1          0          DATE                               DATUM                              DATE                               FECHA                              DATA                               DATUM                        PIVYS                            DATA                               DATUM
66       3      1          0          TIME                               ZEIT                               TEMPS                              NSHORA                             TEMPO                              TIJD                         AIKA                               ORARIO                             KLOCKSLAG
67       1      1          0          DAY                                TAG                                JOUR                               DIA                                DIA                                DAG                          PIV                              GIORNO                             DAG
68       1      1          0          MONTH                              MONAT                              MOIS                               MES                                MS                                MAAND                        KUUKAUSI                           MESE                               MNAD
69       1      1          0          YEAR                               JAHR                               ANNEE                              AO                                ANO                                JAAR                         VUOSI                              ANNO                               R
70      -1      1          0          WEEKDAY                            WOCHENTAG                          JOURSEM                            DIASEM                             DIA.DA.SEMANA                      WEEKDAG                      VIIKONPIV                        GIORNO.SETTIMANA                   VECKODAG
71       1      1          0          HOUR                               STUNDE                             HEURE                              HORA                               HORA                               UUR                          TUNNIT                             ORA                                TIMME
72       1      1          0          MINUTE                             MINUTE                             MINUTE                             MINUTO                             MINUTO                             MINUUT                       MINUUTIT                           MINUTO                             MINUT
73       1      1          0          SECOND                             SEKUNDE                            SECONDE                            SEGUNDO                            SEGUNDO                            SECONDE                      SEKUNNIT                           SECONDO                            SEKUND
74       0      1          1          NOW                                JETZT                              MAINTENANT                         AHORA                              AGORA                              NU                           NYT                                ADESSO                             NU
75       1      0          1          AREAS                              BEREICHE                           ZONES                              AREAS                              REAS                              BEREIKEN                     ALUEET                             AREE                               OMRDEN
76       1      0          1          ROWS                               ZEILEN                             LIGNES                             FILAS                              LINS                               RIJEN                        RIVIT                              RIGHE                              RADER
77       1      0          1          COLUMNS                            SPALTEN                            COLONNES                           COLUMNAS                           COLS                               KOLOMMEN                     SARAKKEET                          COLONNE                            KOLUMNER
78      -1      0          1          OFFSET                             BEREICH.VERSCHIEBEN                DECALER                            DESREF                             DESLOC                             VERSCHUIVING                 SIIRTYM                           SCARTO                             FRSKJUTNING
82      -1      1          0          SEARCH                             SUCHEN                             CHERCHE                            HALLAR                             LOCALIZAR                          VIND.SPEC                    KY.LPI                           RICERCA                            SK
83       1      1          0          TRANSPOSE                          MTRANS                             TRANSPOSE                          TRANSPONER                         TRANSPOR                           TRANSPONEREN                 TRANSPONOI                         MATR.TRASPOSTA                     TRANSPONERA
86       1      1          0          TYPE                               TYP                                TYPE                               TIPO                               TIPO                               TYPE                         TYYPPI                             TIPO                               VRDETYP
97       2      1          0          ATAN2                              ARCTAN2                            ATAN2                              ATAN2                              ATAN2                              BOOGTAN2                     ATAN2                              ARCTAN.2                           ARCTAN2
98       1      1          0          ASIN                               ARCSIN                             ASIN                               ASENO                              ASEN                               BOOGSIN                      ASIN                               ARCSEN                             ARCSIN
99       1      1          0          ACOS                               ARCCOS                             ACOS                               ACOS                               ACOS                               BOOGCOS                      ACOS                               ARCCOS                             ARCCOS
100     -1      1          0          CHOOSE                             WAHL                               CHOISIR                            ELEGIR                             ESCOLHER                           KIEZEN                       VALITSE.INDEKSI                    SCEGLI                             VLJ
101     -1      0          0          HLOOKUP                            WVERWEIS                           RECHERCHEH                         BUSCARH                            PROCH                              HORIZ.ZOEKEN                 VHAKU                              CERCA.ORIZZ                        LETAKOLUMN
102     -1      0          0          VLOOKUP                            SVERWEIS                           RECHERCHEV                         BUSCARV                            PROCV                              VERT.ZOEKEN                  PHAKU                              CERCA.VERT                         LETARAD
105      1      0          0          ISREF                              ISTBEZUG                           ESTREF                             ESREF                              REF                               ISVERWIJZING                 ONVIITT                            VAL.RIF                            RREF
109     -1      1          0          LOG                                LOG                                LOG                                LOG                                LOG                                LOG                          LOG                                LOG                                LOG
111      1      1          0          CHAR                               ZEICHEN                            CAR                                CARACTER                           CARACT                             TEKEN                        MERKKI                             CODICE.CARATT                      TECKENKOD
112      1      1          0          LOWER                              KLEIN                              MINUSCULE                          MINUSC                             MINSCULA                          KLEINE.LETTERS               PIENET                             MINUSC                             GEMENER
113      1      1          0          UPPER                              GROSS                              MAJUSCULE                          MAYUSC                             MAISCULA                          HOOFDLETTERS                 ISOT                               MAIUSC                             VERSALER
114      1      1          0          PROPER                             GROSS2                             NOMPROPRE                          NOMPROPIO                          PRI.MAISCULA                      BEGINLETTERS                 ERISNIMI                           MAIUSC.INIZ                        INITIAL
115     -1      1          0          LEFT                               LINKS                              GAUCHE                             IZQUIERDA                          ESQUERDA                           LINKS                        VASEN                              SINISTRA                           VNSTER
116     -1      1          0          RIGHT                              RECHTS                             DROITE                             DERECHA                            DIREITA                            RECHTS                       OIKEA                              DESTRA                             HGER
117      2      1          0          EXACT                              IDENTISCH                          EXACT                              IGUAL                              EXATO                              GELIJK                       VERTAA                             IDENTICO                           EXAKT
118      1      1          0          TRIM                               GLTTEN                            SUPPRESPACE                        ESPACIOS                           ARRUMAR                            SPATIES.WISSEN               POISTA.VLIT                       ANNULLA.SPAZI                      RENSA
119      4      1          0          REPLACE                            ERSETZEN                           REMPLACER                          REEMPLAZAR                         MUDAR                              VERVANGEN                    KORVAA                             RIMPIAZZA                          ERSTT
120     -1      1          0          SUBSTITUTE                         WECHSELN                           SUBSTITUE                          SUSTITUIR                          SUBSTITUIR                         SUBSTITUEREN                 VAIHDA                             SOSTITUISCI                        BYT.UT
121      1      1          0          CODE                               CODE                               CODE                               CODIGO                             CDIGO                             CODE                         KOODI                              CODICE                             KOD
124     -1      1          0          FIND                               FINDEN                             TROUVE                             ENCONTRAR                          PROCURAR                           VIND.ALLES                   ETSI                               TROVA                              HITTA
125     -1      0          1          CELL                               ZELLE                              CELLULE                            CELDA                              CL                                CEL                          SOLU                               CELLA                              CELL
126      1      1          0          ISERR                              ISTFEHL                            ESTERR                             ESERR                              ERRO                              ISFOUT2                      ONVIRH                             VAL.ERR                            RF
127      1      1          0          ISTEXT                             ISTTEXT                            ESTTEXTE                           ESTEXTO                            TEXTO                             ISTEKST                      ONTEKSTI                           VAL.TESTO                          RTEXT
128      1      1          0          ISNUMBER                           ISTZAHL                            ESTNUM                             ESNUMERO                           NM                               ISGETAL                      ONLUKU                             VAL.NUMERO                         RTAL
129      1      1          0          ISBLANK                            ISTLEER                            ESTVIDE                            ESBLANCO                           CL.VAZIA                         ISLEEG                       ONTYHJ                            VAL.VUOTO                          RTOM
130      1      0          0          T                                  T                                  T                                  T                                  T                                  T                            T                                  T                                  T
131      1      0          0          N                                  N                                  N                                  N                                  N                                  N                            N                                  NUM                                N
140      1      1          0          DATEVALUE                          DATWERT                            DATEVAL                            FECHANUMERO                        DATA.VALOR                         DATUMWAARDE                  PIVYSARVO                        DATA.VALORE                        DATUMVRDE
141      1      1          0          TIMEVALUE                          ZEITWERT                           TEMPSVAL                           HORANUMERO                         VALOR.TEMPO                        TIJDWAARDE                   AIKA_ARVO                          ORARIO.VALORE                      TIDVRDE
142      3      1          0          SLN                                LIA                                AMORLIN                            SLN                                DPD                                LIN.AFSCHR                   STP                                AMMORT.COST                        LINAVSKR
143      4      1          0          SYD                                DIA                                SYD                                SYD                                SDA                                SYD                          VUOSIPOISTO                        AMMORT.ANNUO                       RSAVSKR
144     -1      1          0          DDB                                GDA                                DDB                                DDB                                BDD                                DDB                          DDB                                AMMORT                             DEGAVSKR
148     -1      1          1          INDIRECT                           INDIREKT                           INDIRECT                           INDIRECTO                          INDIRETO                           INDIRECT                     EPSUORA                           INDIRETTO                          INDIREKT
150     -1      1          0          CALL                               AUFRUFEN                           FONCTION.APPELANTE                 LLAMAR                             CHAMAR                             ROEPEN                       KUTSU                              RICHIAMA                           ANROPA
162      1      1          0          CLEAN                              SUBERN                            EPURAGE                            LIMPIAR                            TIRAR                              WISSEN.CONTROL               SIIVOA                             LIBERA                             STDA
163      1      2          0          MDETERM                            MDET                               DETERMAT                           MDETERM                            MATRIZ.DETERM                      DETERMINANTMAT               MDETERM                            MATR.DETERM                        MDETERM
164      1      2          0          MINVERSE                           MINV                               INVERSEMAT                         MINVERSA                           MATRIZ.INVERSO                     INVERSEMAT                   MKNTEINEN                        MATR.INVERSA                       MINVERT
165      2      2          0          MMULT                              MMULT                              PRODUITMAT                         MMULT                              MATRIZ.MULT                        PRODUKTMAT                   MKERRO                             MATR.PRODOTTO                      MMULT
167     -1      1          0          IPMT                               ZINSZ                              INTPER                             PAGOINT                            IPGTO                              IBET                         IPMT                               INTERESSI                          RBETALNING
168     -1      1          0          PPMT                               KAPZ                               PRINCPER                           PAGOPRIN                           PPGTO                              PBET                         PPMT                               P.RATA                             AMORT
169     -1      0          0          COUNTA                             ANZAHL2                            NBVAL                              CONTARA                            CONT.VALORES                       AANTALARG                    LASKE.A                            CONTA.VALORI                       ANTALV
183     -1      0          0          PRODUCT                            PRODUKT                            PRODUIT                            PRODUCTO                           MULT                               PRODUKT                      TULO                               PRODOTTO                           PRODUKT
184      1      1          0          FACT                               FAKULTT                           FACT                               FACT                               FATORIAL                           FACULTEIT                    KERTOMA                            FATTORIALE                         FAKULTET
189      3      0          0          DPRODUCT                           DBPRODUKT                          BDPRODUIT                          BDPRODUCTO                         BDMULTIPL                          DBPRODUKT                    TTULO                              DB.PRODOTTO                        DPRODUKT
190      1      1          0          ISNONTEXT                          ISTKTEXT                           ESTNONTEXTE                        ESNOTEXTO                          .NO.TEXTO                        ISGEENTEKST                  ONEI_TEKSTI                        VAL.NON.TESTO                      REJTEXT
193     -1      0          0          STDEVP                             STABWN                             ECARTYPEP                          DESVESTP                           DESVPADP                           STDEVP                       KESKIHAJONTAP                      DEV.ST.POP                         STDAVP
194     -1      0          0          VARP                               VARIANZEN                          VAR.P                              VARP                               VARP                               VARP                         VARP                               VAR.POP                            VARIANSP
195      3      0          0          DSTDEVP                            DBSTDABWN                          BDECARTYPEP                        BDDESVESTP                         BDDESVPA                           DBSTDEVP                     TKESKIHAJONTAP                     DB.DEV.ST.POP                      DSTDAVP
196      3      0          0          DVARP                              DBVARIANZEN                        BDVARP                             BDVARP                             BDVARP                             DBVARP                       TVARIANSSIP                        DB.VAR.POP                         DVARIANSP
197     -1      1          0          TRUNC                              KRZEN                             TRONQUE                            TRUNCAR                            TRUNCAR                            GEHEEL                       KATKAISE                           TRONCA                             AVKORTA
198      1      1          0          ISLOGICAL                          ISTLOG                             ESTLOGIQUE                         ESLOGICO                           LGICO                            ISLOGISCH                    ONTOTUUS                           VAL.LOGICO                         RLOGISK
199      3      0          0          DCOUNTA                            DBANZAHL2                          BDNBVAL                            BDCONTARA                          BDCONTARA                          DBAANTALC                    TLASKEA                            DB.CONTA.VALORI                    DANTALV
212      2      1          0          ROUNDUP                            AUFRUNDEN                          ARRONDI.SUP                        REDONDEAR.MAS                      ARREDONDAR.PARA.CIMA               AFRONDEN.NAAR.BOVEN          PYRIST.DES.YLS                  ARROTONDA.PER.ECC                  AVRUNDA.UPPT
213      2      1          0          ROUNDDOWN                          ABRUNDEN                           ARRONDI.INF                        REDONDEAR.MENOS                    ARREDONDAR.PARA.BAIXO              AFRONDEN.NAAR.BENEDEN        PYRIST.DES.ALAS                  ARROTONDA.PER.DIF                  AVRUNDA.NEDT
216     -1      0          0          RANK                               RANG                               RANG                               JERARQUIA                          ORDEM                              RANG                         ARVON.MUKAAN                       RANGO                              RANG
219     -1      1          0          ADDRESS                            ADRESSE                            ADRESSE                            DIRECCION                          ENDEREO                           ADRES                        OSOITE                             INDIRIZZO                          ADRESS
220     -1      1          0          DAYS360                            TAGE360                            JOURS360                           DIAS360                            DIAS360                            DAGEN360                     PIVT360                          GIORNO360                          DAGAR360
221      0      1          1          TODAY                              HEUTE                              AUJOURDHUI                         HOY                                HOJE                               VANDAAG                      TM.PIV                         OGGI                               IDAG
222     -1      1          0          VDB                                VDB                                VDB                                DVS                                BDV                                VDB                          VDB                                AMMORT.VAR                         VDEGRAVSKR
227     -1      0          0          MEDIAN                             MEDIAN                             MEDIANE                            MEDIANA                            MED                                MEDIAAN                      MEDIAANI                           MEDIANA                            MEDIAN
228     -1      2          0          SUMPRODUCT                         SUMMENPRODUKT                      SOMMEPROD                          SUMAPRODUCTO                       SOMARPRODUTO                       SOMPRODUKT                   TULOJEN.SUMMA                      MATR.SOMMA.PRODOTTO                PRODUKTSUMMA
229      1      1          0          SINH                               SINHYP                             SINH                               SENOH                              SENH                               SINH                         SINH                               SENH                               SINH
230      1      1          0          COSH                               COSHYP                             COSH                               COSH                               COSH                               COSH                         COSH                               COSH                               COSH
231      1      1          0          TANH                               TANHYP                             TANH                               TANH                               TANH                               TANH                         TANH                               TANH                               TANH
232      1      1          0          ASINH                              ARCSINHYP                          ASINH                              ASENOH                             ASENH                              BOOGSINH                     ASINH                              ARCSENH                            ARCSINH
233      1      1          0          ACOSH                              ARCCOSHYP                          ACOSH                              ACOSH                              ACOSH                              BOOGCOSH                     ACOSH                              ARCCOSH                            ARCCOSH
234      1      1          0          ATANH                              ARCTANHYP                          ATANH                              ATANH                              ATANH                              BOOGTANH                     ATANH                              ARCTANH                            ARCTANH
235      3      0          0          DGET                               DBAUSZUG                           BDLIRE                             BDEXTRAER                          BDEXTRAIR                          DBLEZEN                      TNOUDA                             DB.VALORI                          DHMTA
244      1      1          1          INFO                               INFO                               INFO                               INFO                               INFORMAO                         INFO                         KUVAUS                             AMBIENTE.INFO                      INFO
247     -1      1          0          DB                                 GDA2                               DB                                 DB                                 BD                                 DB                           DB                                 AMMORT.FISSO                       DB
252      2      0          0          FREQUENCY                          HUFIGKEIT                         FREQUENCE                          FRECUENCIA                         FREQNCIA                         INTERVAL                     TAAJUUS                            FREQUENZA                          FREKVENS
261      1      1          0          ERROR.TYPE                         FEHLER.TYP                         TYPE.ERREUR                        TIPO.DE.ERROR                      TIPO.ERRO                          TYPE.FOUT                    VIRHEEN.LAJI                       ERRORE.TIPO                        FEL.TYP
267     -1      1          0          REGISTER.ID                        REGISTER.KENNUMMER                 REGISTRE.NUMERO                    ID.REGISTRO                        IDENT.REGISTRO                     REGISTRATIE.ID               REKISTERI.TUNNUS                   IDENTIFICATORE.REGISTRO            REGISTRERA.ID
269     -1      0          0          AVEDEV                             MITTELABW                          ECART.MOYEN                        DESVPROM                           DESV.MDIO                         GEM.DEVIATIE                 KESKIPOIKKEAMA                     MEDIA.DEV                          MEDELAVV
270     -1      1          0          BETADIST                           BETAVERT                           LOI.BETA                           DISTR.BETA                         DISTBETA                           BETA.VERD                    BEETAJAKAUMA                       DISTRIB.BETA                       BETAFRD
271      1      1          0          GAMMALN                            GAMMALN                            LNGAMMA                            GAMMA.LN                           LNGAMA                             GAMMA.LN                     GAMMALN                            LN.GAMMA                           GAMMALN
272     -1      1          0          BETAINV                            BETAINV                            BETA.INVERSE                       DISTR.BETA.INV                     BETA.ACUM.INV                      BETA.INV                     BEETAJAKAUMA.KNT                 INV.BETA                           BETAINV
273      4      1          0          BINOMDIST                          BINOMVERT                          LOI.BINOMIALE                      DISTR.BINOM                        DISTRBINOM                         BINOMIALE.VERD               BINOMIJAKAUMA                      DISTRIB.BINOM                      BINOMFRD
274      2      1          0          CHIDIST                            CHIVERT                            LOI.KHIDEUX                        DISTR.CHI                          DIST.QUI                           CHI.KWADRAAT                 CHIJAKAUMA                         DISTRIB.CHI                        CHI2FRD
275      2      1          0          CHIINV                             CHIINV                             KHIDEUX.INVERSE                    PRUEBA.CHI.INV                     INV.QUI                            CHI.KWADRAAT.INV             CHIJAKAUMA.KNT                   INV.CHI                            CHI2INV
276      2      1          0          COMBIN                             KOMBINATIONEN                      COMBIN                             COMBINAT                           COMBIN                             COMBINATIES                  KOMBINAATIO                        COMBINAZIONE                       KOMBIN
277      3      1          0          CONFIDENCE                         KONFIDENZ                          INTERVALLE.CONFIANCE               INTERVALO.CONFIANZA                INT.CONFIANA                      BETROUWBAARHEID              LUOTTAMUSVLI                      CONFIDENZA                         KONFIDENS
278      3      1          0          CRITBINOM                          KRITBINOM                          CRITERE.LOI.BINOMIALE              BINOM.CRIT                         CRIT.BINOM                         CRIT.BINOM                   BINOMIJAKAUMA.KRIT                 CRIT.BINOM                         KRITBINOM
279      1      1          0          EVEN                               GERADE                             PAIR                               REDONDEA.PAR                       PAR                                EVEN                         PARILLINEN                         PARI                               JMN
280      3      1          0          EXPONDIST                          EXPONVERT                          LOI.EXPONENTIELLE                  DISTR.EXP                          DISTEXPON                          EXPON.VERD                   EKSPONENTIAALIJAKAUMA              DISTRIB.EXP                        EXPONFRD
281      3      1          0          FDIST                              FVERT                              LOI.F                              DISTR.F                            DISTF                              F.VERDELING                  FJAKAUMA                           DISTRIB.F                          FFRD
282      3      1          0          FINV                               FINV                               INVERSE.LOI.F                      DISTR.F.INV                        INVF                               F.INVERSE                    FJAKAUMA.KNT                     INV.F                              FINV
283      1      1          0          FISHER                             FISHER                             FISHER                             FISHER                             FISHER                             FISHER                       FISHER                             FISHER                             FISHER
284      1      1          0          FISHERINV                          FISHERINV                          FISHER.INVERSE                     PRUEBA.FISHER.INV                  FISHERINV                          FISHER.INV                   FISHER.KNT                       INV.FISHER                         FISHERINV
285      2      1          0          FLOOR                              UNTERGRENZE                        PLANCHER                           MULTIPLO.INFERIOR                  ARREDMULTB                         AFRONDEN.BENEDEN             PYRIST.KERR.ALAS                 ARROTONDA.DIFETTO                  RUNDA.NER
286      4      1          0          GAMMADIST                          GAMMAVERT                          LOI.GAMMA                          DISTR.GAMMA                        DISTGAMA                           GAMMA.VERD                   GAMMAJAKAUMA                       DISTRIB.GAMMA                      GAMMAFRD
287      3      1          0          GAMMAINV                           GAMMAINV                           LOI.GAMMA.INVERSE                  DISTR.GAMMA.INV                    INVGAMA                            GAMMA.INV                    GAMMAJAKAUMA.KNT                 INV.GAMMA                          GAMMAINV
288      2      1          0          CEILING                            OBERGRENZE                         PLAFOND                            MULTIPLO.SUPERIOR                  TETO                               AFRONDEN.BOVEN               PYRIST.KERR.YLS                 ARROTONDA.ECCESSO                  RUNDA.UPP
289      4      1          0          HYPGEOMDIST                        HYPGEOMVERT                        LOI.HYPERGEOMETRIQUE               DISTR.HIPERGEOM                    DIST.HIPERGEOM                     HYPERGEO.VERD                HYPERGEOM.JAKAUMA                  DISTRIB.IPERGEOM                   HYPGEOMFRD
290      3      1          0          LOGNORMDIST                        LOGNORMVERT                        LOI.LOGNORMALE                     DISTR.LOG.NORM                     DIST.LOGNORMAL                     LOG.NORM.VERD                LOGNORM.JAKAUMA                    DISTRIB.LOGNORM                    LOGNORMFRD
291      3      1          0          LOGINV                             LOGINV                             LOI.LOGNORMALE.INVERSE             DISTR.LOG.INV                      INVLOG                             LOG.NORM.INV                 LOGNORM.JAKAUMA.KNT              INV.LOGNORM                        LOGINV
292      3      1          0          NEGBINOMDIST                       NEGBINOMVERT                       LOI.BINOMIALE.NEG                  NEGBINOMDIST                       DIST.BIN.NEG                       NEG.BINOM.VERD               BINOMIJAKAUMA.NEG                  DISTRIB.BINOM.NEG                  NEGBINOMFRD
293      4      1          0          NORMDIST                           NORMVERT                           LOI.NORMALE                        DISTR.NORM                         DIST.NORM                          NORM.VERD                    NORM.JAKAUMA                       DISTRIB.NORM                       NORMFRD
294      1      1          0          NORMSDIST                          STANDNORMVERT                      LOI.NORMALE.STANDARD               DISTR.NORM.ESTAND                  DIST.NORMP                         STAND.NORM.VERD              NORM.JAKAUMA.NORMIT                DISTRIB.NORM.ST                    NORMSFRD
295      3      1          0          NORMINV                            NORMINV                            LOI.NORMALE.INVERSE                DISTR.NORM.INV                     INV.NORM                           NORM.INV                     NORM.JAKAUMA.KNT                 INV.NORM                           NORMINV
296      1      1          0          NORMSINV                           STANDNORMINV                       LOI.NORMALE.STANDARD.INVERSE       DISTR.NORM.ESTAND.INV              INV.NORMP                          STAND.NORM.INV               NORM.JAKAUMA.NORMIT.KNT          INV.NORM.ST                        NORMSINV
297      3      1          0          STANDARDIZE                        STANDARDISIERUNG                   CENTREE.REDUITE                    NORMALIZACION                      PADRONIZAR                         NORMALISEREN                 NORMITA                            NORMALIZZA                         STANDARDISERA
298      1      1          0          ODD                                UNGERADE                           IMPAIR                             REDONDEA.IMPAR                     MPAR                              ONEVEN                       PARITON                            DISPARI                            UDDA
299      2      1          0          PERMUT                             VARIATIONEN                        PERMUTATION                        PERMUTACIONES                      PERMUT                             PERMUTATIES                  PERMUTAATIO                        PERMUTAZIONE                       PERMUT
300      3      1          0          POISSON                            POISSON                            LOI.POISSON                        POISSON                            POISSON                            POISSON                      POISSON                            POISSON                            POISSON
301      3      1          0          TDIST                              TVERT                              LOI.STUDENT                        DISTR.T                            DISTT                              T.VERD                       TJAKAUMA                           DISTRIB.T                          TFRD
302      4      1          0          WEIBULL                            WEIBULL                            LOI.WEIBULL                        DIST.WEIBULL                       WEIBULL                            WEIBULL                      WEIBULL                            WEIBULL                            WEIBULL
303      2      2          0          SUMXMY2                            SUMMEXMY2                          SOMME.XMY2                         SUMAXMENOSY2                       SOMAXMY2                           SOM.XMINY.2                  EROTUSTEN.NELISUMMA               SOMMA.Q.DIFF                       SUMMAXMY2
304      2      2          0          SUMX2MY2                           SUMMEX2MY2                         SOMME.X2MY2                        SUMAX2MENOSY2                      SOMAX2DY2                          SOM.X2MINY2                  NELISUMMIEN.EROTUS                SOMMA.DIFF.Q                       SUMMAX2MY2
305      2      2          0          SUMX2PY2                           SUMMEX2PY2                         SOMME.X2PY2                        SUMAX2MASY2                        SOMAX2SY2                          SOM.X2PLUSY2                 NELISUMMIEN.SUMMA                 SOMMA.SOMMA.Q                      SUMMAX2PY2
306      2      2          0          CHITEST                            CHITEST                            TEST.KHIDEUX                       PRUEBA.CHI                         TESTE.QUI                          CHI.TOETS                    CHITESTI                           TEST.CHI                           CHI2TEST
307      2      2          0          CORREL                             KORREL                             COEFFICIENT.CORRELATION            COEF.DE.CORREL                     CORREL                             CORRELATIE                   KORRELAATIO                        CORRELAZIONE                       KORREL
308      2      2          0          COVAR                              KOVAR                              COVARIANCE                         COVAR                              COVAR                              COVARIANTIE                  KOVARIANSSI                        COVARIANZA                         KOVAR
309      3      2          0          FORECAST                           SCHTZER                           PREVISION                          PRONOSTICO                         PREVISO                           VOORSPELLEN                  ENNUSTE                            PREVISIONE                         PREDIKTION
310      2      2          0          FTEST                              FTEST                              TEST.F                             PRUEBA.F                           TESTEF                             F.TOETS                      FTESTI                             TEST.F                             FTEST
311      2      2          0          INTERCEPT                          ACHSENABSCHNITT                    ORDONNEE.ORIGINE                   INTERSECCION                       INTERCEPO                        SNIJPUNT                     LEIKKAUSPISTE                      INTERCETTA                         SKRNINGSPUNKT
312      2      2          0          PEARSON                            PEARSON                            PEARSON                            PEARSON                            PEARSON                            PEARSON                      PEARSON                            PEARSON                            PEARSON
313      2      2          0          RSQ                                BESTIMMTHEITSMASS                  COEFFICIENT.DETERMINATION          COEFICIENTE.R2                     RQUAD                              R.KWADRAAT                   PEARSON.NELI                      RQ                                 RKV
314      2      2          0          STEYX                              STFEHLERYX                         ERREUR.TYPE.XY                     ERROR.TIPICO.XY                    EPADYX                             STAND.FOUT.YX                KESKIVIRHE                         ERR.STD.YX                         STDFELYX
315      2      2          0          SLOPE                              STEIGUNG                           PENTE                              PENDIENTE                          INCLINAO                         RICHTING                     KULMAKERROIN                       PENDENZA                           LUTNING
316      4      2          0          TTEST                              TTEST                              TEST.STUDENT                       PRUEBA.T                           TESTET                             T.TOETS                      TTESTI                             TEST.T                             TTEST
317     -1      2          0          PROB                               WAHRSCHBEREICH                     PROBABILITE                        PROBABILIDAD                       PROB                               KANS                         TODENNKISYYS                     PROBABILIT                        SANNOLIKHET
318     -1      0          0          DEVSQ                              SUMQUADABW                         SOMME.CARRES.ECARTS                DESVIA2                            DESVQ                              DEV.KWAD                     OIKAISTU.NELISUMMA                DEV.Q                              KVADAVV
319     -1      0          0          GEOMEAN                            GEOMITTEL                          MOYENNE.GEOMETRIQUE                MEDIA.GEOM                         MDIA.GEOMTRICA                   MEETK.GEM                    KESKIARVO.GEOM                     MEDIA.GEOMETRICA                   GEOMEDEL
320     -1      0          0          HARMEAN                            HARMITTEL                          MOYENNE.HARMONIQUE                 MEDIA.ARMO                         MDIA.HARMNICA                    HARM.GEM                     KESKIARVO.HARM                     MEDIA.ARMONICA                     HARMMEDEL
321     -1      0          0          SUMSQ                              QUADRATESUMME                      SOMME.CARRES                       SUMA.CUADRADOS                     SOMAQUAD                           KWADRATENSOM                 NELISUMMA                         SOMMA.Q                            KVADRATSUMMA
322     -1      0          0          KURT                               KURT                               KURTOSIS                           CURTOSIS                           CURT                               KURTOSIS                     KURT                               CURTOSI                            TOPPIGHET
323     -1      0          0          SKEW                               SCHIEFE                            COEFFICIENT.ASYMETRIE              COEFICIENTE.ASIMETRIA              DISTORO                          SCHEEFHEID                   JAKAUMAN.VINOUS                    ASIMMETRIA                         SNEDHET
324     -1      0          0          ZTEST                              GTEST                              TEST.Z                             PRUEBA.Z                           TESTEZ                             Z.TOETS                      ZTESTI                             TEST.Z                             ZTEST
325      2      0          0          LARGE                              KGRSSTE                           GRANDE.VALEUR                      K.ESIMO.MAYOR                      MAIOR                              GROOTSTE                     SUURI                              GRANDE                             STRSTA
326      2      0          0          SMALL                              KKLEINSTE                          PETITE.VALEUR                      K.ESIMO.MENOR                      MENOR                              KLEINSTE                     PIENI                              PICCOLO                            MINSTA
327      2      0          0          QUARTILE                           QUARTILE                           QUARTILE                           CUARTIL                            QUARTIL                            KWARTIEL                     NELJNNES                          QUARTILE                           KVARTIL
328      2      0          0          PERCENTILE                         QUANTIL                            CENTILE                            PERCENTIL                          PERCENTIL                          PERCENTIEL                   PROSENTTIPISTE                     PERCENTILE                         PERCENTIL
329     -1      0          0          PERCENTRANK                        QUANTILSRANG                       RANG.POURCENTAGE                   RANGO.PERCENTIL                    ORDEM.PORCENTUAL                   PERCENT.RANG                 PROSENTTIJRJESTYS                 PERCENT.RANGO                      PROCENTRANG
330     -1      2          0          MODE                               MODALWERT                          MODE                               MODA                               MODO                               MODUS                        MOODI                              MODA                               TYPVRDE
331      2      0          0          TRIMMEAN                           GESTUTZTMITTEL                     MOYENNE.REDUITE                    MEDIA.ACOTADA                      MDIA.INTERNA                      GETRIMD.GEM                  KESKIARVO.TASATTU                  MEDIA.TRONCATA                     TRIMMEDEL
332      2      1          0          TINV                               TINV                               LOI.STUDENT.INVERSE                DISTR.T.INV                        INVT                               T.INV                        TJAKAUMA.KNT                     INV.T                              TINV
336     -1      1          0          CONCATENATE                        VERKETTEN                          CONCATENER                         CONCATENAR                         CONCATENAR                         TEKST.SAMENVOEGEN            KETJUTA                            CONCATENA                          SAMMANFOGA
337      2      1          0          POWER                              POTENZ                             PUISSANCE                          POTENCIA                           POTNCIA                           MACHT                        POTENSSI                           POTENZA                            UPPHJT.TILL
342      1      1          0          RADIANS                            RADIANT                            RADIANS                            RADIANES                           RADIANOS                           RADIALEN                     RADIAANIT                          RADIANTI                           RADIANER
343      1      1          0          DEGREES                            GRAD                               DEGRES                             GRADOS                             GRAUS                              GRADEN                       ASTEET                             GRADI                              GRADER
344     -1      0          0          SUBTOTAL                           TEILERGEBNIS                       SOUS.TOTAL                         SUBTOTALES                         SUBTOTAL                           SUBTOTAAL                    VLISUMMA                          SUBTOTALE                          DELSUMMA
345     -1      0          0          SUMIF                              SUMMEWENN                          SOMME.SI                           SUMAR.SI                           SOMASE                             SOM.ALS                      SUMMA.JOS                          SOMMA.SE                           SUMMA.OM
346      2      0          0          COUNTIF                            ZHLENWENN                         NB.SI                              CONTAR.SI                          CONT.SE                            AANTAL.ALS                   LASKE.JOS                          CONTA.SE                           ANTAL.OM
347      1      0          0          COUNTBLANK                         ANZAHLLEEREZELLEN                  NB.VIDE                            CONTAR.BLANCO                      CONTAR.VAZIO                       AANTAL.LEGE.CELLEN           LASKE.TYHJT                       CONTA.VUOTE                        ANTAL.TOMMA
354     -1      1          0          ROMAN                              RMISCH                            ROMAIN                             NUMERO.ROMANO                      ROMANO                             ROMEINS                      ROMAN                              ROMANO                             ROMERSK
