﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codedeploy/CodeDeployRequest.h>
#include <aws/codedeploy/CodeDeploy_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeDeploy {
namespace Model {

/**
 * <p> Represents the input of a <code>StopDeployment</code> operation.
 * </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/codedeploy-2014-10-06/StopDeploymentInput">AWS
 * API Reference</a></p>
 */
class StopDeploymentRequest : public CodeDeployRequest {
 public:
  AWS_CODEDEPLOY_API StopDeploymentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StopDeployment"; }

  AWS_CODEDEPLOY_API Aws::String SerializePayload() const override;

  AWS_CODEDEPLOY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p> The unique ID of a deployment. </p>
   */
  inline const Aws::String& GetDeploymentId() const { return m_deploymentId; }
  inline bool DeploymentIdHasBeenSet() const { return m_deploymentIdHasBeenSet; }
  template <typename DeploymentIdT = Aws::String>
  void SetDeploymentId(DeploymentIdT&& value) {
    m_deploymentIdHasBeenSet = true;
    m_deploymentId = std::forward<DeploymentIdT>(value);
  }
  template <typename DeploymentIdT = Aws::String>
  StopDeploymentRequest& WithDeploymentId(DeploymentIdT&& value) {
    SetDeploymentId(std::forward<DeploymentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Indicates, when a deployment is stopped, whether instances that have been
   * updated should be rolled back to the previous version of the application
   * revision. </p>
   */
  inline bool GetAutoRollbackEnabled() const { return m_autoRollbackEnabled; }
  inline bool AutoRollbackEnabledHasBeenSet() const { return m_autoRollbackEnabledHasBeenSet; }
  inline void SetAutoRollbackEnabled(bool value) {
    m_autoRollbackEnabledHasBeenSet = true;
    m_autoRollbackEnabled = value;
  }
  inline StopDeploymentRequest& WithAutoRollbackEnabled(bool value) {
    SetAutoRollbackEnabled(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_deploymentId;
  bool m_deploymentIdHasBeenSet = false;

  bool m_autoRollbackEnabled{false};
  bool m_autoRollbackEnabledHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeDeploy
}  // namespace Aws
