#!/usr/bin/env python

# Qt tutorial 11.

import sys
import math
import qt


class LCDRange(qt.QVBox):
    def __init__(self, parent=None, name=None):
        qt.QVBox.__init__(self, parent, name)

        lcd = qt.QLCDNumber(2, self, "lcd")
        self.slider = qt.QSlider(qt.Qt.Horizontal, self, "slider")
        self.slider.setRange(0, 99)
        self.slider.setValue(0)
        self.connect(self.slider, qt.SIGNAL("valueChanged(int)"), lcd, qt.SLOT("display(int)"))
        self.connect(self.slider, qt.SIGNAL("valueChanged(int)"), self, qt.PYSIGNAL("valueChanged(int)"))

        self.setFocusProxy(self.slider)

    def value(self):
        return self.slider.value()

    def setValue(self, value):
        self.slider.setValue(value)

    def setRange(self, minVal, maxVal):
        if minVal < 0 or maxVal > 99 or minVal > maxVal:
            raise ValueError,  "LCDRange.setRange(): invalid range"
        self.slider.setRange(minVal, maxVal)


class CannonField(qt.QWidget):
    def __init__(self, parent=None, name=None):
        qt.QWidget.__init__(self, parent, name)

        self.ang = 45
        self.f = 0
        self.timerCount = 0

        self.autoShootTimer = qt.QTimer(self, "movement handler")
        self.connect(self.autoShootTimer, qt.SIGNAL("timeout()"), self.moveShot)

        self.shoot_ang = 0
        self.shoot_f = 0

        self.setPalette(qt.QPalette(qt.QColor(250, 250, 200)))

        self.barrelRect = qt.QRect(33, -4, 15, 8)

    def angle(self):
        return self.ang

    def setAngle(self, degrees):
        if degrees < 5:
            degrees = 5
        if degrees > 70:
            degrees = 70
        if self.ang == degrees:
            return
        self.ang = degrees
        self.repaint(self.cannonRect(), 0)
        self.emit(qt.PYSIGNAL("angleChanged(int)"), (self.ang, ))

    def force(self):
        return self.f

    def setForce(self, newton):
        if newton < 0:
            newton = 0
        if self.f == newton:
            return
        self.f = newton
        self.emit(qt.PYSIGNAL("forceChanged(int)"), (self.f, ))

    def shoot(self):
        if self.autoShootTimer.isActive():
            return

        self.timerCount = 0
        self.shoot_ang = self.ang
        self.shoot_f = self.f
        self.autoShootTimer.start(50)

    def moveShot(self):
        r = qt.QRegion(self.shotRect())
        self.timerCount = self.timerCount + 1

        shotR = self.shotRect()

        if shotR.x() > self.width() or shotR.y() > self.height():
            self.autoShootTimer.stop()
        else:
            r = r.unite(qt.QRegion(shotR))

        self.repaint(r)

    def paintEvent(self, ev):
        updateR = ev.rect()
        p = qt.QPainter(self)

        if updateR.intersects(self.cannonRect()):
            self.paintCannon(p)

        if self.autoShootTimer.isActive() and updateR.intersects(self.shotRect()):
            self.paintShot(p)

    def paintShot(self, p):
        p.setBrush(qt.Qt.black)
        p.setPen(qt.Qt.NoPen)
        p.drawRect(self.shotRect())

    def paintCannon(self, p):
        cr = self.cannonRect()
        pix = qt.QPixmap(cr.size())
        pix.fill(self, cr.topLeft())

        tmp = qt.QPainter(pix)
        tmp.setBrush(qt.Qt.blue)
        tmp.setPen(qt.Qt.NoPen)

        tmp.translate(0, pix.height() - 1)
        tmp.drawPie(qt.QRect(-35, -35, 70, 70), 0, 90 * 16)
        tmp.rotate(-self.ang)
        tmp.drawRect(self.barrelRect)
        tmp.end()

        p.drawPixmap(cr.topLeft(), pix)

    def cannonRect(self):
        r = qt.QRect(0, 0, 50, 50)
        r.moveBottomLeft(self.rect().bottomLeft())
        return r

    def shotRect(self):
        gravity = 4.0

        time = self.timerCount / 4.0
        velocity = self.shoot_f
        radians = self.shoot_ang * 3.14159265 / 180

        velx = velocity * math.cos(radians)
        vely = velocity * math.sin(radians)
        x0 = (self.barrelRect.right() + 5) * math.cos(radians)
        y0 = (self.barrelRect.right() + 5) * math.sin(radians)
        x = x0 + velx * time
        y = y0 + vely * time - 0.5 * gravity * time * time

        r = qt.QRect(0, 0, 6, 6)
        r.moveCenter(qt.QPoint(x, self.height() - 1 - y))
        return r

    def sizePolicy(self):
        return qt.QSizePolicy(qt.QSizePolicy.Expanding, qt.QSizePolicy.Expanding)


class MyWidget(qt.QWidget):
    def __init__(self, parent=None, name=None):
        qt.QWidget.__init__(self, parent, name)

        quit = qt.QPushButton("&Quit", self, "quit")
        quit.setFont(qt.QFont("Times", 18, qt.QFont.Bold))
        self.connect(quit, qt.SIGNAL("clicked()"), qt.qApp, qt.SLOT("quit()"))

        self.angle = LCDRange(self, "angle")
        self.angle.setRange(5, 70)

        self.force = LCDRange(self, "force")
        self.force.setRange(10, 50)

        self.cannonField = CannonField(self, "cannonField")

        self.connect(self.angle, qt.PYSIGNAL("valueChanged(int)"), self.cannonField.setAngle)
        self.connect(self.cannonField, qt.PYSIGNAL("angleChanged(int)"), self.angle.setValue)

        self.connect(self.force, qt.PYSIGNAL("valueChanged(int)"), self.cannonField.setForce)
        self.connect(self.cannonField, qt.PYSIGNAL("forceChanged(int)"), self.force.setValue)

        shoot = qt.QPushButton("&Shoot", self, "shoot")
        shoot.setFont(qt.QFont("Times", 18, qt.QFont.Bold))
        self.connect(shoot, qt.SIGNAL("clicked()"), self.cannonField.shoot)

        grid = qt.QGridLayout(self, 2, 2, 10)

        grid.addWidget(quit, 0, 0)
        grid.addWidget(self.cannonField, 1, 1)
        grid.setColStretch(1, 10)

        leftBox = qt.QVBoxLayout()
        grid.addLayout(leftBox, 1, 0)
        leftBox.addWidget(self.angle)
        leftBox.addWidget(self.force)

        topBox = qt.QHBoxLayout()
        grid.addLayout(topBox, 0, 1)
        topBox.addWidget(shoot)
        topBox.addStretch(1)

        self.angle.setValue(60)
        self.force.setValue(25)
        self.angle.setFocus()


qt.QApplication.setColorSpec(qt.QApplication.CustomColor)
a = qt.QApplication(sys.argv)

w = MyWidget()
w.setGeometry(100, 100, 500, 355)
a.setMainWidget(w)
w.show()
sys.exit(a.exec_loop())
