/*
  'OpenLook' kwin client

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to
  deal in the Software without restriction, including without limitation the
  rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
  sell copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:
 
  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
  WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <unistd.h> // for usleep
#include <math.h>

#include <netwm.h>
#undef Bool

#include <qlayout.h>
#include <qpainter.h>
#include <qdrawutil.h>
#include <qtoolbutton.h>
#include <qimage.h>
#include <qpixmap.h>

#include <kapplication.h>

#include <kwin/workspace.h>
#include <kwin/options.h>

#include "OpenLook.h"

using namespace KWinInternal;

const uint openLookCornerSize = 11;
const uint openLookMargin = 5;
const uint openLookTextVMargin = 1;

extern "C"
{
  Client * allocate(Workspace * ws, WId w, int tool)
  {
    return new OpenLook(ws, w, tool != 0);
  }

  void init()
  {
    // Empty.
  }

  void reset()
  {
    Workspace::self()->slotResetAllClientsDelayed();
    // Empty.
  }

  void deinit()
  {
    // Empty.
  }
}

OpenLook::OpenLook
(
 Workspace    * ws,
 WId            w,
 bool           tool,
 QWidget      * parent,
 const char   * name
)
  : Client          (ws, w, parent, name),
    tool_           (tool),
    buttonDown_     (false),
    titleSpacer_    (0),
    mousePressPoint_(0, 0) // Anywhere outside button rect.
{
  setBackgroundMode(NoBackground);
  doLayout();
}

OpenLook::~OpenLook()
{
  // Empty.
}

  void
OpenLook::captionChange(const QString &)
{
  update(titleRect());
}

  void
OpenLook::paintEvent(QPaintEvent * pe)
{
  QRect tr(titleRect());

  QPainter p(this);

  QRegion clipRegion(pe->region());

  p.setClipRegion(clipRegion);

  paintBorder(p);

  paintTopLeftRect(p);
  paintTopRightRect(p);
  paintBottomLeftRect(p);
  paintBottomRightRect(p);

  p.setClipRegion(clipRegion + buttonRect());

  QBrush titleBackground(options->color(Options::TitleBar, true));

  if (isActive())
    qDrawShadePanel(&p, tr, colorGroup(), true, 1, &titleBackground);
  else
    p.fillRect(tr, colorGroup().brush(QColorGroup::Background));

  p.setClipRegion(clipRegion);

  paintButton(p);

  p.setFont(options->font(isActive(), tool_));

  p.setPen(options->color(Options::Font, isActive()));

  tr.setLeft(openLookCornerSize + 3 + buttonRect().width() + 2);

  p.drawText(tr, AlignCenter, caption());
}

  void
OpenLook::mouseDoubleClickEvent(QMouseEvent * e)
{
  if (titleRect().contains(e->pos()))
  {
    workspace()
      ->performWindowOperation(this, options->operationTitlebarDblClick());
  }
}

  void
OpenLook::activeChange(bool)
{
  repaint();
}

  Client::MousePosition
OpenLook::mousePosition(const QPoint & p) const
{
  if (topLeftRect().contains(p))
    return TopLeft;
  else if (topRightRect().contains(p))
    return TopRight;
  else if (bottomLeftRect().contains(p))
    return BottomLeft;
  else if (bottomRightRect().contains(p))
    return BottomRight;
  else
    return Center;
}

  void
OpenLook::slotReset()
{
  repaint();
}

  void
OpenLook::doLayout()
{
  QFontMetrics fm(options->font(isActive(), tool_));

  uint titleHeight = fm.height() + openLookTextVMargin * 2;

  QVBoxLayout * layout = new QVBoxLayout(this, openLookMargin);

  titleSpacer_ =
    new QSpacerItem
    (
     0,
     titleHeight,
     QSizePolicy::Expanding,
     QSizePolicy::Fixed
    );

  layout->addItem(titleSpacer_);

  layout->addSpacing(2);

  QHBoxLayout * midLayout   = new QHBoxLayout(layout, 0, 0);
 
  midLayout->addWidget(windowWrapper(), 1);
}

  void
OpenLook::animateIconifyOrDeiconify(bool iconify)
{
  NETRect r = netWinInfo()->iconGeometry();

  QRect icongeom(r.pos.x, r.pos.y, r.size.width, r.size.height);

  if (!icongeom.isValid())
    return;

  QRect wingeom(x(), y(), width(), height());

  QPainter p(workspace()->desktopWidget());

  p.setRasterOp(Qt::NotROP);

  if (iconify)
    p.setClipRegion(
      QRegion(workspace()->desktopWidget()->rect()) - wingeom
      );

  for (uint count = 0; count < 4; count++)
  {
    XGrabServer(qt_xdisplay());

    p.drawLine(wingeom.bottomRight(), icongeom.bottomRight());
    p.drawLine(wingeom.bottomLeft(), icongeom.bottomLeft());
    p.drawLine(wingeom.topLeft(), icongeom.topLeft());
    p.drawLine(wingeom.topRight(), icongeom.topRight());

    p.flush();

    kapp->syncX();

    usleep(10000);

    p.drawLine(wingeom.bottomRight(), icongeom.bottomRight());
    p.drawLine(wingeom.bottomLeft(), icongeom.bottomLeft());
    p.drawLine(wingeom.topLeft(), icongeom.topLeft());
    p.drawLine(wingeom.topRight(), icongeom.topRight());

    p.flush();

    kapp->syncX();

    usleep(10000);

    XUngrabServer(qt_xdisplay());
  }
}

  QRect
OpenLook::topLeftRect() const
{
  return QRect
    (
     0,
     0,
     openLookCornerSize,
     openLookCornerSize
    );
}

  QRect
OpenLook::topRightRect() const
{
  return QRect
    (
     width() - openLookCornerSize,
     0,
     openLookCornerSize,
     openLookCornerSize
    );
}

  QRect
OpenLook::bottomLeftRect() const
{
  return QRect
    (
     0,
     height() - openLookCornerSize,
     openLookCornerSize,
     openLookCornerSize
    );
}

  QRect
OpenLook::bottomRightRect() const
{
  return QRect
    (
     width() - openLookCornerSize,
     height() - openLookCornerSize,
     openLookCornerSize,
     openLookCornerSize
    );
}

  void
OpenLook::paintTopLeftRect(QPainter & p) const
{
  QColor handleColour(options->color(Options::Handle, isActive()));

  QRect r(topLeftRect());

  int x1(r.left());
  int y1(r.top());
  int x2(r.right());
  int y2(r.bottom());

  p.setPen(colorGroup().light());

  p.drawLine(x1, y1, x2, y1);
  p.drawLine(x1, y1 + 1, x1, y2);

  p.setPen(handleColour);

  p.drawLine(x1 + 1, y1 + 1, x2 - 1, y1 + 1);
  p.drawLine(x1 + 1, y1 + 2, x2 - 1, y1 + 2);
  p.drawLine(x1 + 1, y1 + 3, x2 - 1, y1 + 3);

  p.drawLine(x1 + 1, y1 + 4, x1 + 1, y2 - 1);
  p.drawLine(x1 + 2, y1 + 4, x1 + 2, y2 - 1);
  p.drawLine(x1 + 3, y1 + 4, x1 + 3, y2 - 1);

  p.setPen(colorGroup().dark());

  p.drawLine(x2, y1 + 1, x2, y1 + 4);

  p.drawLine(x1 + 4, y1 + 4, x2 - 1, y1 + 4);

  p.drawLine(x1 + 4, y1 + 5, x1 + 4, y2 - 1);

  p.drawLine(x1 + 1, y2, x1 + 4, y2);
}

  void
OpenLook::paintTopRightRect(QPainter & p) const
{
  QColor handleColour(options->color(Options::Handle, isActive()));

  QRect r(topRightRect());

  int x1(r.left());
  int y1(r.top());
  int x2(r.right());
  int y2(r.bottom());

  p.setPen(colorGroup().light());

  p.drawLine(x1, y1, x2, y1);
  p.drawLine(x1, y1 + 1, x1, y1 + 4);
  p.drawLine(x2 - 4, y1 + 5, x2 - 4, y2);

  p.setPen(handleColour);

  p.drawLine(x1 + 1, y1 + 1, x2 - 1, y1 + 1);
  p.drawLine(x1 + 1, y1 + 2, x2 - 1, y1 + 2);
  p.drawLine(x1 + 1, y1 + 3, x2 - 1, y1 + 3);

  p.drawLine(x2 - 3, y1 + 4, x2 - 3, y2 - 1);
  p.drawLine(x2 - 2, y1 + 4, x2 - 2, y2 - 1);
  p.drawLine(x2 - 1, y1 + 4, x2 - 1, y2 - 1);

  p.setPen(colorGroup().dark());

  p.drawLine(x1 + 1, y1 + 4, x2 - 4, y1 + 4);
  p.drawLine(x2, y1 + 1, x2, y2);
  p.drawLine(x2 - 4, y2, x2 - 1, y2);
}

  void
OpenLook::paintBottomLeftRect(QPainter & p) const
{
  QColor handleColour(options->color(Options::Handle, isActive()));

  QRect r(bottomLeftRect());

  int x1(r.left());
  int y1(r.top());
  int x2(r.right());
  int y2(r.bottom());

  p.setPen(colorGroup().light());

  p.drawLine(x1, y1, x1 + 4, y1);
  p.drawLine(x1, y1 + 1, x1, y2);
  p.drawLine(x1 + 5, y2 - 4, x2, y2 - 4);

  p.setPen(handleColour);

  p.drawLine(x1 + 1, y1 + 1, x1 + 1, y2 - 4);
  p.drawLine(x1 + 2, y1 + 1, x1 + 2, y2 - 4);
  p.drawLine(x1 + 3, y1 + 1, x1 + 3, y2 - 4);

  p.drawLine(x1 + 1, y2 - 3, x2 - 1, y2 - 3);
  p.drawLine(x1 + 1, y2 - 2, x2 - 1, y2 - 2);
  p.drawLine(x1 + 1, y2 - 1, x2 - 1, y2 - 1);

  p.setPen(colorGroup().dark());

  p.drawLine(x1 + 4, y1 + 1, x1 + 4, y2 - 5);
  p.drawLine(x1 + 1, y2, x2, y2);
  p.drawLine(x2, y2 - 3, x2, y2 - 1);
}

  void
OpenLook::paintBottomRightRect(QPainter & p) const
{
  QColor handleColour(options->color(Options::Handle, isActive()));

  QRect r(bottomRightRect());

  int x1(r.left());
  int y1(r.top());
  int x2(r.right());
  int y2(r.bottom());

  p.setPen(colorGroup().light());

  p.drawLine(x1, y2 - 4, x1, y2);
  p.drawLine(x1 + 1, y2 - 4, x2 - 4, y2 - 4);
  p.drawLine(x2 - 4, y1 + 1, x2 - 4, y2 - 5);
  p.drawLine(x2 - 4, y1, x2, y1);

  p.setPen(handleColour);

  p.drawLine(x1 + 1, y2 - 3, x2 - 1, y2 - 3);
  p.drawLine(x1 + 1, y2 - 2, x2 - 1, y2 - 2);
  p.drawLine(x1 + 1, y2 - 1, x2 - 1, y2 - 1);

  p.drawLine(x2 - 3, y1 + 1, x2 - 3, y2 - 4);
  p.drawLine(x2 - 2, y1 + 1, x2 - 2, y2 - 4);
  p.drawLine(x2 - 1, y1 + 1, x2 - 1, y2 - 4);

  p.setPen(colorGroup().dark());

  p.drawLine(x1 + 1, y2, x2, y2);
  p.drawLine(x2, y1 + 1, x2, y2 - 1);
}

  QRect
OpenLook::buttonRect() const
{
  return QRect
    (
      openLookCornerSize + 3,
      titleRect().top(),
      titleRect().height(),
      titleRect().height()
    );
}

  void
OpenLook::paintButton(QPainter & p) const
{
  QRect r(buttonRect());

  p.fillRect
    (
     r.left() + 1,
     r.top() + 1,
     r.width() - 2,
     r.height() - 2,
     buttonDown_
     ? colorGroup().dark()
     : options->color(Options::ButtonBg, isActive())
    );

  p.setPen(buttonDown_ ? colorGroup().dark() : colorGroup().light());

  p.drawLine(r.left() + 1, r.top(), r.right() - 1, r.top());
  p.drawLine(r.left(), r.top() + 1, r.left(), r.bottom() - 1);

  p.setPen(buttonDown_ ? colorGroup().light() : colorGroup().dark());

  p.drawLine(r.right(), r.top() + 1, r.right(), r.bottom() - 1);
  p.drawLine(r.left() + 1, r.bottom(), r.right() - 1, r.bottom());

  paintArrow(p);
}

  void
OpenLook::paintArrow(QPainter & p) const
{
  QRect br(buttonRect());

  int x = br.left()   + 5;
  int y = br.top()    + 5;
  int w = br.width()  - 10;
  int h = br.height() - 10;

  QPointArray poly(3);

  p.setBrush(colorGroup().mid());

  poly.setPoint(0, x, y);
  poly.setPoint(1, x + w - 1, y);
  poly.setPoint(2, x + (w / 2), y + h - 1);

  p.drawPolygon(poly);

  p.setPen(colorGroup().dark());

  p.drawLine(x, y, x + w - 1, y);

  p.drawLine(x, y, x + (w / 2), y + h - 1);

  p.setPen(colorGroup().light());

  p.drawLine(x + (w / 2), y + h - 1, x + w - 1, y);
}

  void
OpenLook::paintBorder(QPainter & p) const
{
  const uint cs(openLookCornerSize);

  uint x = rect().left();
  uint y = rect().top();
  uint w = rect().width();
  uint h = rect().height();
  uint r = rect().right();
  uint b = rect().bottom();

  p.fillRect(x + cs, y, w - cs - cs, 2,     colorGroup().shadow());
  p.fillRect(x + cs, b - 1, w - cs - cs, 2, colorGroup().shadow());
  p.fillRect(x, y + cs, 2, h - cs - cs,     colorGroup().shadow());
  p.fillRect(r - 1, y + cs, 2, h - cs - cs, colorGroup().shadow());

  QColor frameColour(options->color(Options::Frame, isActive()));

  p.fillRect(x + cs, y + 2, w - cs - cs, 3, frameColour);
  p.fillRect(x + cs, b - 4, w - cs - cs, 3, frameColour);
  p.fillRect(x + 2, y + cs, 3, h - cs - cs, frameColour);
  p.fillRect(r - 4, y + cs, 3, h - cs - cs, frameColour);

  p.fillRect
    (
     openLookMargin,
     titleRect().bottom() + 1,
     width() - 2 * openLookMargin,
     2,
     colorGroup().background()
    );
}

  QRect
OpenLook::titleRect() const
{
  return titleSpacer_->geometry();
}

  void
OpenLook::mousePressEvent(QMouseEvent * e)
{
  mousePressPoint_ = e->pos();

  buttonDown_ = buttonRect().contains(mousePressPoint_);

  repaint(buttonRect());

  if (!buttonDown_)
    Client::mousePressEvent(e);
}

  void
OpenLook::mouseReleaseEvent(QMouseEvent * e)
{
  if (buttonDown_ && buttonRect().contains(e->pos()))
  {
    iconify();
  }

  buttonDown_ = false;

  repaint(buttonRect());

  Client::mouseReleaseEvent(e);
}

#include "OpenLook.moc"
// vim:ts=2:sw=2:tw=78:set et:
