/*
**************************************************************************
                                 description
                             --------------------
    copyright            : (C) 2002 by Andreas Zehender
    email                : zehender@kde.org
**************************************************************************

**************************************************************************
*                                                                        *
*  This program is free software; you can redistribute it and/or modify  *
*  it under the terms of the GNU General Public License as published by  *
*  the Free Software Foundation; either version 2 of the License, or     *
*  (at your option) any later version.                                   *
*                                                                        *
**************************************************************************/


#include "pmpolynom.h"
#include "pmpolynomedit.h"

#include "pmoutputdevice.h"
#include "pmxmlhelper.h"
#include "pmmemento.h"

#include <kdebug.h>
#include "pmglobals.h"

#include <klocale.h>

const double c_defaultCoefficients[10] =
{  // Hyperboloid_Y
   1.0,  0.0, 0.0,
   0.0, -1.0, 0.0,
   0.0, 1.0, 0.0,
   -1.0
};
const int c_defaultOrder = 2;
const bool c_defaultSturm = true;
const int c_polynomSize[8] = { 0, 0, 10, 20, 35, 56, 84, 120 };

PMPolynom::PMPolynom( )
      : Base( )
{
   int i;
   m_order = c_defaultOrder;
   m_coefficients = PMVector( 10 );
   for( i = 0; i < 10; i++ )
      m_coefficients[i] = c_defaultCoefficients[i];
   m_sturm = c_defaultSturm;
}

PMPolynom::~PMPolynom( )
{
}

QString PMPolynom::description( ) const
{
   if( m_order == 2 )
      return i18n( "quadric" );
   else if( m_order == 3 )
      return i18n( "cubic" );
   else if( m_order == 4 )
      return i18n( "quartic" );
   return i18n( "polynom" );
}

void PMPolynom::serialize( PMOutputDevice& dev ) const
{
   if( m_order == 2 )
      dev.objectBegin( "quadric" );
   else if( m_order == 3 )
      dev.objectBegin( "cubic" );
   else if( m_order == 4 )
      dev.objectBegin( "quartic" );
   else
      dev.objectBegin( "poly" );
   
   serializeName( dev );

   if( m_order == 2 )
   {
      dev.writeLine( QString( "<%1, %2, %3>," ).arg( m_coefficients[0] )
                     .arg( m_coefficients[4] ).arg( m_coefficients[7] ) );
      dev.writeLine( QString( "<%1, %2, %3>," ).arg( m_coefficients[1] )
                     .arg( m_coefficients[2] ).arg( m_coefficients[5] ) );
      dev.writeLine( QString( "<%1, %2, %3>, %4" ).arg( m_coefficients[3] )
                     .arg( m_coefficients[6] ).arg( m_coefficients[8] )
                     .arg( m_coefficients[9] ) );
   }
   else
   {
      if( m_order > 4 )
         dev.writeLine( QString( "%1," ).arg( m_order ) );
      
      int size = m_coefficients.size( );
      
      int i;
      QString hlp;
      
      dev.write( "<" );
      for( i = 0; i < size; i++ )
      {
         hlp.setNum( m_coefficients[i] );
         dev.write( hlp );
         
         if( i != ( size - 1 ) )
         {
            dev.write( ", " );
            if( ( ( i + 1 ) % 5 ) == 0  )
               dev.writeLine( "" );
         }
      }
      dev.writeLine( ">" );
      if( m_sturm )
         dev.writeLine( "sturm" );
   }
   
   Base::serialize( dev );
   dev.objectEnd( );
}

void PMPolynom::serialize( QDomElement& e, QDomDocument& doc ) const
{
   e.setAttribute( "order", m_order );
   e.setAttribute( "coefficients", m_coefficients.serializeXML( ) );
   e.setAttribute( "sturm", m_sturm );
   Base::serialize( e, doc );
}

void PMPolynom::readAttributes( const PMXMLHelper& h )
{
   m_order = h.intAttribute( "order", c_defaultOrder );
   m_coefficients = h.vectorAttribute( "coefficients", m_coefficients );
   m_sturm = h.boolAttribute( "sturm", c_defaultSturm );
   Base::readAttributes( h );
}

bool PMPolynom::isA( PMObjectType t ) const
{
   if( t == PMTPolynom )
      return true;
   return Base::isA( t );
}

PMDialogEditBase* PMPolynom::editWidget( QWidget* parent ) const
{
   return new PMPolynomEdit( parent );
}

void PMPolynom::setPolynomOrder( int o )
{
   if( ( o < 2 ) || ( o > 7 ) )
   {
      kdError( PMArea ) << "Invalid order in PMPolynom::setPolynomOrder\n";
      o = 2;
   }
   if( o != m_order )
   {
      if( m_pMemento )
      {
         m_pMemento->addData( PMTPolynom, PMOrderID, m_order );
         if( ( o <= 4 ) || ( m_order <= 4 ) )
            m_pMemento->setDescriptionChanged( );
      }
      m_order = o;
   }
}

void PMPolynom::setCoefficients( const PMVector& c )
{
   if( c.size( ) != ( unsigned ) c_polynomSize[m_order] )
      kdError( PMArea ) << "Wrong vector size in PMPolynom::setCoefficients\n";
   
   if( c != m_coefficients )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTPolynom, PMCoefficientsID, m_coefficients );
      m_coefficients = c;
      m_coefficients.resize( c_polynomSize[m_order] );
      //setViewStructureChanged( );
   }
}

void PMPolynom::setSturm( bool s )
{
   if( m_sturm != s )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTPolynom, PMSturmID, m_sturm );
      m_sturm = s;
   }
}

void PMPolynom::restoreMemento( PMMemento* s )
{
   PMMementoDataIterator it( s );
   PMMementoData* data;

   for( ; it.current( ); ++it )
   {
      data = it.current( );
      if( data->objectType( ) == PMTPolynom )
      {
         switch( data->valueID( ) )
         {
            case PMOrderID:
               setPolynomOrder( data->intData( ) );
               break;
            case PMCoefficientsID:
               setCoefficients( data->vectorData( ) );
               break;
            case PMSturmID:
               setSturm( data->boolData( ) );
               break;
            default:
               kdError( PMArea ) << "Wrong ID in PMPolynom::restoreMemento\n";
               break;
         }
      }
   }
   Base::restoreMemento( s );
}
